/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.b2bi.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the details for a transformer object. A transformer can take an EDI file as input and transform it into a
 * JSON-or XML-formatted document. Alternatively, a transformer can take a JSON-or XML-formatted document as input and
 * transform it into an EDI file.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TransformerSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<TransformerSummary.Builder, TransformerSummary> {
    private static final SdkField<String> TRANSFORMER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("transformerId").getter(getter(TransformerSummary::transformerId)).setter(setter(Builder::transformerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("transformerId").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(TransformerSummary::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(TransformerSummary::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt")
            .getter(getter(TransformerSummary::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Instant> MODIFIED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("modifiedAt")
            .getter(getter(TransformerSummary::modifiedAt))
            .setter(setter(Builder::modifiedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("modifiedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> FILE_FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("fileFormat").getter(getter(TransformerSummary::fileFormatAsString)).setter(setter(Builder::fileFormat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("fileFormat").build()).build();

    private static final SdkField<String> MAPPING_TEMPLATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("mappingTemplate").getter(getter(TransformerSummary::mappingTemplate))
            .setter(setter(Builder::mappingTemplate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("mappingTemplate").build()).build();

    private static final SdkField<EdiType> EDI_TYPE_FIELD = SdkField.<EdiType> builder(MarshallingType.SDK_POJO)
            .memberName("ediType").getter(getter(TransformerSummary::ediType)).setter(setter(Builder::ediType))
            .constructor(EdiType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ediType").build()).build();

    private static final SdkField<String> SAMPLE_DOCUMENT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("sampleDocument").getter(getter(TransformerSummary::sampleDocument))
            .setter(setter(Builder::sampleDocument))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sampleDocument").build()).build();

    private static final SdkField<InputConversion> INPUT_CONVERSION_FIELD = SdkField
            .<InputConversion> builder(MarshallingType.SDK_POJO).memberName("inputConversion")
            .getter(getter(TransformerSummary::inputConversion)).setter(setter(Builder::inputConversion))
            .constructor(InputConversion::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("inputConversion").build()).build();

    private static final SdkField<Mapping> MAPPING_FIELD = SdkField.<Mapping> builder(MarshallingType.SDK_POJO)
            .memberName("mapping").getter(getter(TransformerSummary::mapping)).setter(setter(Builder::mapping))
            .constructor(Mapping::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("mapping").build()).build();

    private static final SdkField<OutputConversion> OUTPUT_CONVERSION_FIELD = SdkField
            .<OutputConversion> builder(MarshallingType.SDK_POJO).memberName("outputConversion")
            .getter(getter(TransformerSummary::outputConversion)).setter(setter(Builder::outputConversion))
            .constructor(OutputConversion::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("outputConversion").build()).build();

    private static final SdkField<SampleDocuments> SAMPLE_DOCUMENTS_FIELD = SdkField
            .<SampleDocuments> builder(MarshallingType.SDK_POJO).memberName("sampleDocuments")
            .getter(getter(TransformerSummary::sampleDocuments)).setter(setter(Builder::sampleDocuments))
            .constructor(SampleDocuments::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sampleDocuments").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TRANSFORMER_ID_FIELD,
            NAME_FIELD, STATUS_FIELD, CREATED_AT_FIELD, MODIFIED_AT_FIELD, FILE_FORMAT_FIELD, MAPPING_TEMPLATE_FIELD,
            EDI_TYPE_FIELD, SAMPLE_DOCUMENT_FIELD, INPUT_CONVERSION_FIELD, MAPPING_FIELD, OUTPUT_CONVERSION_FIELD,
            SAMPLE_DOCUMENTS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String transformerId;

    private final String name;

    private final String status;

    private final Instant createdAt;

    private final Instant modifiedAt;

    private final String fileFormat;

    private final String mappingTemplate;

    private final EdiType ediType;

    private final String sampleDocument;

    private final InputConversion inputConversion;

    private final Mapping mapping;

    private final OutputConversion outputConversion;

    private final SampleDocuments sampleDocuments;

    private TransformerSummary(BuilderImpl builder) {
        this.transformerId = builder.transformerId;
        this.name = builder.name;
        this.status = builder.status;
        this.createdAt = builder.createdAt;
        this.modifiedAt = builder.modifiedAt;
        this.fileFormat = builder.fileFormat;
        this.mappingTemplate = builder.mappingTemplate;
        this.ediType = builder.ediType;
        this.sampleDocument = builder.sampleDocument;
        this.inputConversion = builder.inputConversion;
        this.mapping = builder.mapping;
        this.outputConversion = builder.outputConversion;
        this.sampleDocuments = builder.sampleDocuments;
    }

    /**
     * <p>
     * Returns the system-assigned unique identifier for the transformer.
     * </p>
     * 
     * @return Returns the system-assigned unique identifier for the transformer.
     */
    public final String transformerId() {
        return transformerId;
    }

    /**
     * <p>
     * Returns the descriptive name for the transformer.
     * </p>
     * 
     * @return Returns the descriptive name for the transformer.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * Returns the state of the newly created transformer. The transformer can be either <code>active</code> or
     * <code>inactive</code>. For the transformer to be used in a capability, its status must <code>active</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link TransformerStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return Returns the state of the newly created transformer. The transformer can be either <code>active</code> or
     *         <code>inactive</code>. For the transformer to be used in a capability, its status must
     *         <code>active</code>.
     * @see TransformerStatus
     */
    public final TransformerStatus status() {
        return TransformerStatus.fromValue(status);
    }

    /**
     * <p>
     * Returns the state of the newly created transformer. The transformer can be either <code>active</code> or
     * <code>inactive</code>. For the transformer to be used in a capability, its status must <code>active</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link TransformerStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return Returns the state of the newly created transformer. The transformer can be either <code>active</code> or
     *         <code>inactive</code>. For the transformer to be used in a capability, its status must
     *         <code>active</code>.
     * @see TransformerStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * Returns a timestamp indicating when the transformer was created. For example,
     * <code>2023-07-20T19:58:44.624Z</code>.
     * </p>
     * 
     * @return Returns a timestamp indicating when the transformer was created. For example,
     *         <code>2023-07-20T19:58:44.624Z</code>.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * Returns a timestamp representing the date and time for the most recent change for the transformer object.
     * </p>
     * 
     * @return Returns a timestamp representing the date and time for the most recent change for the transformer object.
     */
    public final Instant modifiedAt() {
        return modifiedAt;
    }

    /**
     * <p>
     * Returns that the currently supported file formats for EDI transformations are <code>JSON</code> and
     * <code>XML</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #fileFormat} will
     * return {@link FileFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #fileFormatAsString}.
     * </p>
     * 
     * @return Returns that the currently supported file formats for EDI transformations are <code>JSON</code> and
     *         <code>XML</code>.
     * @see FileFormat
     * @deprecated This is a legacy trait. Please use input-conversion or output-conversion.
     */
    @Deprecated
    public final FileFormat fileFormat() {
        return FileFormat.fromValue(fileFormat);
    }

    /**
     * <p>
     * Returns that the currently supported file formats for EDI transformations are <code>JSON</code> and
     * <code>XML</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #fileFormat} will
     * return {@link FileFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #fileFormatAsString}.
     * </p>
     * 
     * @return Returns that the currently supported file formats for EDI transformations are <code>JSON</code> and
     *         <code>XML</code>.
     * @see FileFormat
     * @deprecated This is a legacy trait. Please use input-conversion or output-conversion.
     */
    @Deprecated
    public final String fileFormatAsString() {
        return fileFormat;
    }

    /**
     * <p>
     * Returns the mapping template for the transformer. This template is used to map the parsed EDI file using JSONata
     * or XSLT.
     * </p>
     * 
     * @return Returns the mapping template for the transformer. This template is used to map the parsed EDI file using
     *         JSONata or XSLT.
     * @deprecated This is a legacy trait. Please use input-conversion or output-conversion.
     */
    @Deprecated
    public final String mappingTemplate() {
        return mappingTemplate;
    }

    /**
     * <p>
     * Returns the details for the EDI standard that is being used for the transformer. Currently, only X12 is
     * supported. X12 is a set of standards and corresponding messages that define specific business documents.
     * </p>
     * 
     * @return Returns the details for the EDI standard that is being used for the transformer. Currently, only X12 is
     *         supported. X12 is a set of standards and corresponding messages that define specific business documents.
     * @deprecated This is a legacy trait. Please use input-conversion or output-conversion.
     */
    @Deprecated
    public final EdiType ediType() {
        return ediType;
    }

    /**
     * <p>
     * Returns a sample EDI document that is used by a transformer as a guide for processing the EDI data.
     * </p>
     * 
     * @return Returns a sample EDI document that is used by a transformer as a guide for processing the EDI data.
     * @deprecated This is a legacy trait. Please use input-conversion or output-conversion.
     */
    @Deprecated
    public final String sampleDocument() {
        return sampleDocument;
    }

    /**
     * <p>
     * Returns a structure that contains the format options for the transformation.
     * </p>
     * 
     * @return Returns a structure that contains the format options for the transformation.
     */
    public final InputConversion inputConversion() {
        return inputConversion;
    }

    /**
     * <p>
     * Returns the structure that contains the mapping template and its language (either XSLT or JSONATA).
     * </p>
     * 
     * @return Returns the structure that contains the mapping template and its language (either XSLT or JSONATA).
     */
    public final Mapping mapping() {
        return mapping;
    }

    /**
     * <p>
     * Returns the <code>OutputConversion</code> object, which contains the format options for the outbound
     * transformation.
     * </p>
     * 
     * @return Returns the <code>OutputConversion</code> object, which contains the format options for the outbound
     *         transformation.
     */
    public final OutputConversion outputConversion() {
        return outputConversion;
    }

    /**
     * <p>
     * Returns a structure that contains the Amazon S3 bucket and an array of the corresponding keys used to identify
     * the location for your sample documents.
     * </p>
     * 
     * @return Returns a structure that contains the Amazon S3 bucket and an array of the corresponding keys used to
     *         identify the location for your sample documents.
     */
    public final SampleDocuments sampleDocuments() {
        return sampleDocuments;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(transformerId());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(modifiedAt());
        hashCode = 31 * hashCode + Objects.hashCode(fileFormatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(mappingTemplate());
        hashCode = 31 * hashCode + Objects.hashCode(ediType());
        hashCode = 31 * hashCode + Objects.hashCode(sampleDocument());
        hashCode = 31 * hashCode + Objects.hashCode(inputConversion());
        hashCode = 31 * hashCode + Objects.hashCode(mapping());
        hashCode = 31 * hashCode + Objects.hashCode(outputConversion());
        hashCode = 31 * hashCode + Objects.hashCode(sampleDocuments());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TransformerSummary)) {
            return false;
        }
        TransformerSummary other = (TransformerSummary) obj;
        return Objects.equals(transformerId(), other.transformerId()) && Objects.equals(name(), other.name())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(modifiedAt(), other.modifiedAt())
                && Objects.equals(fileFormatAsString(), other.fileFormatAsString())
                && Objects.equals(mappingTemplate(), other.mappingTemplate()) && Objects.equals(ediType(), other.ediType())
                && Objects.equals(sampleDocument(), other.sampleDocument())
                && Objects.equals(inputConversion(), other.inputConversion()) && Objects.equals(mapping(), other.mapping())
                && Objects.equals(outputConversion(), other.outputConversion())
                && Objects.equals(sampleDocuments(), other.sampleDocuments());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TransformerSummary").add("TransformerId", transformerId()).add("Name", name())
                .add("Status", statusAsString()).add("CreatedAt", createdAt()).add("ModifiedAt", modifiedAt())
                .add("FileFormat", fileFormatAsString()).add("MappingTemplate", mappingTemplate()).add("EdiType", ediType())
                .add("SampleDocument", sampleDocument()).add("InputConversion", inputConversion()).add("Mapping", mapping())
                .add("OutputConversion", outputConversion()).add("SampleDocuments", sampleDocuments()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "transformerId":
            return Optional.ofNullable(clazz.cast(transformerId()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "modifiedAt":
            return Optional.ofNullable(clazz.cast(modifiedAt()));
        case "fileFormat":
            return Optional.ofNullable(clazz.cast(fileFormatAsString()));
        case "mappingTemplate":
            return Optional.ofNullable(clazz.cast(mappingTemplate()));
        case "ediType":
            return Optional.ofNullable(clazz.cast(ediType()));
        case "sampleDocument":
            return Optional.ofNullable(clazz.cast(sampleDocument()));
        case "inputConversion":
            return Optional.ofNullable(clazz.cast(inputConversion()));
        case "mapping":
            return Optional.ofNullable(clazz.cast(mapping()));
        case "outputConversion":
            return Optional.ofNullable(clazz.cast(outputConversion()));
        case "sampleDocuments":
            return Optional.ofNullable(clazz.cast(sampleDocuments()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<TransformerSummary, T> g) {
        return obj -> g.apply((TransformerSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TransformerSummary> {
        /**
         * <p>
         * Returns the system-assigned unique identifier for the transformer.
         * </p>
         * 
         * @param transformerId
         *        Returns the system-assigned unique identifier for the transformer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transformerId(String transformerId);

        /**
         * <p>
         * Returns the descriptive name for the transformer.
         * </p>
         * 
         * @param name
         *        Returns the descriptive name for the transformer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Returns the state of the newly created transformer. The transformer can be either <code>active</code> or
         * <code>inactive</code>. For the transformer to be used in a capability, its status must <code>active</code>.
         * </p>
         * 
         * @param status
         *        Returns the state of the newly created transformer. The transformer can be either <code>active</code>
         *        or <code>inactive</code>. For the transformer to be used in a capability, its status must
         *        <code>active</code>.
         * @see TransformerStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TransformerStatus
         */
        Builder status(String status);

        /**
         * <p>
         * Returns the state of the newly created transformer. The transformer can be either <code>active</code> or
         * <code>inactive</code>. For the transformer to be used in a capability, its status must <code>active</code>.
         * </p>
         * 
         * @param status
         *        Returns the state of the newly created transformer. The transformer can be either <code>active</code>
         *        or <code>inactive</code>. For the transformer to be used in a capability, its status must
         *        <code>active</code>.
         * @see TransformerStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TransformerStatus
         */
        Builder status(TransformerStatus status);

        /**
         * <p>
         * Returns a timestamp indicating when the transformer was created. For example,
         * <code>2023-07-20T19:58:44.624Z</code>.
         * </p>
         * 
         * @param createdAt
         *        Returns a timestamp indicating when the transformer was created. For example,
         *        <code>2023-07-20T19:58:44.624Z</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * Returns a timestamp representing the date and time for the most recent change for the transformer object.
         * </p>
         * 
         * @param modifiedAt
         *        Returns a timestamp representing the date and time for the most recent change for the transformer
         *        object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modifiedAt(Instant modifiedAt);

        /**
         * <p>
         * Returns that the currently supported file formats for EDI transformations are <code>JSON</code> and
         * <code>XML</code>.
         * </p>
         * 
         * @param fileFormat
         *        Returns that the currently supported file formats for EDI transformations are <code>JSON</code> and
         *        <code>XML</code>.
         * @see FileFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FileFormat
         * @deprecated This is a legacy trait. Please use input-conversion or output-conversion.
         */
        @Deprecated
        Builder fileFormat(String fileFormat);

        /**
         * <p>
         * Returns that the currently supported file formats for EDI transformations are <code>JSON</code> and
         * <code>XML</code>.
         * </p>
         * 
         * @param fileFormat
         *        Returns that the currently supported file formats for EDI transformations are <code>JSON</code> and
         *        <code>XML</code>.
         * @see FileFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FileFormat
         * @deprecated This is a legacy trait. Please use input-conversion or output-conversion.
         */
        @Deprecated
        Builder fileFormat(FileFormat fileFormat);

        /**
         * <p>
         * Returns the mapping template for the transformer. This template is used to map the parsed EDI file using
         * JSONata or XSLT.
         * </p>
         * 
         * @param mappingTemplate
         *        Returns the mapping template for the transformer. This template is used to map the parsed EDI file
         *        using JSONata or XSLT.
         * @return Returns a reference to this object so that method calls can be chained together.
         * @deprecated This is a legacy trait. Please use input-conversion or output-conversion.
         */
        @Deprecated
        Builder mappingTemplate(String mappingTemplate);

        /**
         * <p>
         * Returns the details for the EDI standard that is being used for the transformer. Currently, only X12 is
         * supported. X12 is a set of standards and corresponding messages that define specific business documents.
         * </p>
         * 
         * @param ediType
         *        Returns the details for the EDI standard that is being used for the transformer. Currently, only X12
         *        is supported. X12 is a set of standards and corresponding messages that define specific business
         *        documents.
         * @return Returns a reference to this object so that method calls can be chained together.
         * @deprecated This is a legacy trait. Please use input-conversion or output-conversion.
         */
        @Deprecated
        Builder ediType(EdiType ediType);

        /**
         * <p>
         * Returns the details for the EDI standard that is being used for the transformer. Currently, only X12 is
         * supported. X12 is a set of standards and corresponding messages that define specific business documents.
         * </p>
         * This is a convenience method that creates an instance of the {@link EdiType.Builder} avoiding the need to
         * create one manually via {@link EdiType#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EdiType.Builder#build()} is called immediately and its result is
         * passed to {@link #ediType(EdiType)}.
         * 
         * @param ediType
         *        a consumer that will call methods on {@link EdiType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ediType(EdiType)
         * @deprecated This is a legacy trait. Please use input-conversion or output-conversion.
         */
        @Deprecated
        default Builder ediType(Consumer<EdiType.Builder> ediType) {
            return ediType(EdiType.builder().applyMutation(ediType).build());
        }

        /**
         * <p>
         * Returns a sample EDI document that is used by a transformer as a guide for processing the EDI data.
         * </p>
         * 
         * @param sampleDocument
         *        Returns a sample EDI document that is used by a transformer as a guide for processing the EDI data.
         * @return Returns a reference to this object so that method calls can be chained together.
         * @deprecated This is a legacy trait. Please use input-conversion or output-conversion.
         */
        @Deprecated
        Builder sampleDocument(String sampleDocument);

        /**
         * <p>
         * Returns a structure that contains the format options for the transformation.
         * </p>
         * 
         * @param inputConversion
         *        Returns a structure that contains the format options for the transformation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputConversion(InputConversion inputConversion);

        /**
         * <p>
         * Returns a structure that contains the format options for the transformation.
         * </p>
         * This is a convenience method that creates an instance of the {@link InputConversion.Builder} avoiding the
         * need to create one manually via {@link InputConversion#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link InputConversion.Builder#build()} is called immediately and its
         * result is passed to {@link #inputConversion(InputConversion)}.
         * 
         * @param inputConversion
         *        a consumer that will call methods on {@link InputConversion.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #inputConversion(InputConversion)
         */
        default Builder inputConversion(Consumer<InputConversion.Builder> inputConversion) {
            return inputConversion(InputConversion.builder().applyMutation(inputConversion).build());
        }

        /**
         * <p>
         * Returns the structure that contains the mapping template and its language (either XSLT or JSONATA).
         * </p>
         * 
         * @param mapping
         *        Returns the structure that contains the mapping template and its language (either XSLT or JSONATA).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mapping(Mapping mapping);

        /**
         * <p>
         * Returns the structure that contains the mapping template and its language (either XSLT or JSONATA).
         * </p>
         * This is a convenience method that creates an instance of the {@link Mapping.Builder} avoiding the need to
         * create one manually via {@link Mapping#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Mapping.Builder#build()} is called immediately and its result is
         * passed to {@link #mapping(Mapping)}.
         * 
         * @param mapping
         *        a consumer that will call methods on {@link Mapping.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #mapping(Mapping)
         */
        default Builder mapping(Consumer<Mapping.Builder> mapping) {
            return mapping(Mapping.builder().applyMutation(mapping).build());
        }

        /**
         * <p>
         * Returns the <code>OutputConversion</code> object, which contains the format options for the outbound
         * transformation.
         * </p>
         * 
         * @param outputConversion
         *        Returns the <code>OutputConversion</code> object, which contains the format options for the outbound
         *        transformation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputConversion(OutputConversion outputConversion);

        /**
         * <p>
         * Returns the <code>OutputConversion</code> object, which contains the format options for the outbound
         * transformation.
         * </p>
         * This is a convenience method that creates an instance of the {@link OutputConversion.Builder} avoiding the
         * need to create one manually via {@link OutputConversion#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link OutputConversion.Builder#build()} is called immediately and its
         * result is passed to {@link #outputConversion(OutputConversion)}.
         * 
         * @param outputConversion
         *        a consumer that will call methods on {@link OutputConversion.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #outputConversion(OutputConversion)
         */
        default Builder outputConversion(Consumer<OutputConversion.Builder> outputConversion) {
            return outputConversion(OutputConversion.builder().applyMutation(outputConversion).build());
        }

        /**
         * <p>
         * Returns a structure that contains the Amazon S3 bucket and an array of the corresponding keys used to
         * identify the location for your sample documents.
         * </p>
         * 
         * @param sampleDocuments
         *        Returns a structure that contains the Amazon S3 bucket and an array of the corresponding keys used to
         *        identify the location for your sample documents.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sampleDocuments(SampleDocuments sampleDocuments);

        /**
         * <p>
         * Returns a structure that contains the Amazon S3 bucket and an array of the corresponding keys used to
         * identify the location for your sample documents.
         * </p>
         * This is a convenience method that creates an instance of the {@link SampleDocuments.Builder} avoiding the
         * need to create one manually via {@link SampleDocuments#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link SampleDocuments.Builder#build()} is called immediately and its
         * result is passed to {@link #sampleDocuments(SampleDocuments)}.
         * 
         * @param sampleDocuments
         *        a consumer that will call methods on {@link SampleDocuments.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sampleDocuments(SampleDocuments)
         */
        default Builder sampleDocuments(Consumer<SampleDocuments.Builder> sampleDocuments) {
            return sampleDocuments(SampleDocuments.builder().applyMutation(sampleDocuments).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String transformerId;

        private String name;

        private String status;

        private Instant createdAt;

        private Instant modifiedAt;

        private String fileFormat;

        private String mappingTemplate;

        private EdiType ediType;

        private String sampleDocument;

        private InputConversion inputConversion;

        private Mapping mapping;

        private OutputConversion outputConversion;

        private SampleDocuments sampleDocuments;

        private BuilderImpl() {
        }

        private BuilderImpl(TransformerSummary model) {
            transformerId(model.transformerId);
            name(model.name);
            status(model.status);
            createdAt(model.createdAt);
            modifiedAt(model.modifiedAt);
            fileFormat(model.fileFormat);
            mappingTemplate(model.mappingTemplate);
            ediType(model.ediType);
            sampleDocument(model.sampleDocument);
            inputConversion(model.inputConversion);
            mapping(model.mapping);
            outputConversion(model.outputConversion);
            sampleDocuments(model.sampleDocuments);
        }

        public final String getTransformerId() {
            return transformerId;
        }

        public final void setTransformerId(String transformerId) {
            this.transformerId = transformerId;
        }

        @Override
        public final Builder transformerId(String transformerId) {
            this.transformerId = transformerId;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(TransformerStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final Instant getModifiedAt() {
            return modifiedAt;
        }

        public final void setModifiedAt(Instant modifiedAt) {
            this.modifiedAt = modifiedAt;
        }

        @Override
        public final Builder modifiedAt(Instant modifiedAt) {
            this.modifiedAt = modifiedAt;
            return this;
        }

        @Deprecated
        public final String getFileFormat() {
            return fileFormat;
        }

        @Deprecated
        public final void setFileFormat(String fileFormat) {
            this.fileFormat = fileFormat;
        }

        @Override
        @Deprecated
        public final Builder fileFormat(String fileFormat) {
            this.fileFormat = fileFormat;
            return this;
        }

        @Override
        @Deprecated
        public final Builder fileFormat(FileFormat fileFormat) {
            this.fileFormat(fileFormat == null ? null : fileFormat.toString());
            return this;
        }

        @Deprecated
        public final String getMappingTemplate() {
            return mappingTemplate;
        }

        @Deprecated
        public final void setMappingTemplate(String mappingTemplate) {
            this.mappingTemplate = mappingTemplate;
        }

        @Override
        @Deprecated
        public final Builder mappingTemplate(String mappingTemplate) {
            this.mappingTemplate = mappingTemplate;
            return this;
        }

        @Deprecated
        public final EdiType.Builder getEdiType() {
            return ediType != null ? ediType.toBuilder() : null;
        }

        @Deprecated
        public final void setEdiType(EdiType.BuilderImpl ediType) {
            this.ediType = ediType != null ? ediType.build() : null;
        }

        @Override
        @Deprecated
        public final Builder ediType(EdiType ediType) {
            this.ediType = ediType;
            return this;
        }

        @Deprecated
        public final String getSampleDocument() {
            return sampleDocument;
        }

        @Deprecated
        public final void setSampleDocument(String sampleDocument) {
            this.sampleDocument = sampleDocument;
        }

        @Override
        @Deprecated
        public final Builder sampleDocument(String sampleDocument) {
            this.sampleDocument = sampleDocument;
            return this;
        }

        public final InputConversion.Builder getInputConversion() {
            return inputConversion != null ? inputConversion.toBuilder() : null;
        }

        public final void setInputConversion(InputConversion.BuilderImpl inputConversion) {
            this.inputConversion = inputConversion != null ? inputConversion.build() : null;
        }

        @Override
        public final Builder inputConversion(InputConversion inputConversion) {
            this.inputConversion = inputConversion;
            return this;
        }

        public final Mapping.Builder getMapping() {
            return mapping != null ? mapping.toBuilder() : null;
        }

        public final void setMapping(Mapping.BuilderImpl mapping) {
            this.mapping = mapping != null ? mapping.build() : null;
        }

        @Override
        public final Builder mapping(Mapping mapping) {
            this.mapping = mapping;
            return this;
        }

        public final OutputConversion.Builder getOutputConversion() {
            return outputConversion != null ? outputConversion.toBuilder() : null;
        }

        public final void setOutputConversion(OutputConversion.BuilderImpl outputConversion) {
            this.outputConversion = outputConversion != null ? outputConversion.build() : null;
        }

        @Override
        public final Builder outputConversion(OutputConversion outputConversion) {
            this.outputConversion = outputConversion;
            return this;
        }

        public final SampleDocuments.Builder getSampleDocuments() {
            return sampleDocuments != null ? sampleDocuments.toBuilder() : null;
        }

        public final void setSampleDocuments(SampleDocuments.BuilderImpl sampleDocuments) {
            this.sampleDocuments = sampleDocuments != null ? sampleDocuments.build() : null;
        }

        @Override
        public final Builder sampleDocuments(SampleDocuments sampleDocuments) {
            this.sampleDocuments = sampleDocuments;
            return this;
        }

        @Override
        public TransformerSummary build() {
            return new TransformerSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
