/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.b2bi.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A structure that contains the details for a partnership. A partnership represents the connection between you and your
 * trading partner. It ties together a profile and one or more trading capabilities.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PartnershipSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<PartnershipSummary.Builder, PartnershipSummary> {
    private static final SdkField<String> PROFILE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("profileId").getter(getter(PartnershipSummary::profileId)).setter(setter(Builder::profileId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("profileId").build()).build();

    private static final SdkField<String> PARTNERSHIP_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("partnershipId").getter(getter(PartnershipSummary::partnershipId)).setter(setter(Builder::partnershipId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("partnershipId").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(PartnershipSummary::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<List<String>> CAPABILITIES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("capabilities")
            .getter(getter(PartnershipSummary::capabilities))
            .setter(setter(Builder::capabilities))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("capabilities").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<CapabilityOptions> CAPABILITY_OPTIONS_FIELD = SdkField
            .<CapabilityOptions> builder(MarshallingType.SDK_POJO).memberName("capabilityOptions")
            .getter(getter(PartnershipSummary::capabilityOptions)).setter(setter(Builder::capabilityOptions))
            .constructor(CapabilityOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("capabilityOptions").build()).build();

    private static final SdkField<String> TRADING_PARTNER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("tradingPartnerId").getter(getter(PartnershipSummary::tradingPartnerId))
            .setter(setter(Builder::tradingPartnerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tradingPartnerId").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt")
            .getter(getter(PartnershipSummary::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Instant> MODIFIED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("modifiedAt")
            .getter(getter(PartnershipSummary::modifiedAt))
            .setter(setter(Builder::modifiedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("modifiedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PROFILE_ID_FIELD,
            PARTNERSHIP_ID_FIELD, NAME_FIELD, CAPABILITIES_FIELD, CAPABILITY_OPTIONS_FIELD, TRADING_PARTNER_ID_FIELD,
            CREATED_AT_FIELD, MODIFIED_AT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String profileId;

    private final String partnershipId;

    private final String name;

    private final List<String> capabilities;

    private final CapabilityOptions capabilityOptions;

    private final String tradingPartnerId;

    private final Instant createdAt;

    private final Instant modifiedAt;

    private PartnershipSummary(BuilderImpl builder) {
        this.profileId = builder.profileId;
        this.partnershipId = builder.partnershipId;
        this.name = builder.name;
        this.capabilities = builder.capabilities;
        this.capabilityOptions = builder.capabilityOptions;
        this.tradingPartnerId = builder.tradingPartnerId;
        this.createdAt = builder.createdAt;
        this.modifiedAt = builder.modifiedAt;
    }

    /**
     * <p>
     * Returns the unique, system-generated identifier for the profile connected to this partnership.
     * </p>
     * 
     * @return Returns the unique, system-generated identifier for the profile connected to this partnership.
     */
    public final String profileId() {
        return profileId;
    }

    /**
     * <p>
     * Returns the unique, system-generated identifier for a partnership.
     * </p>
     * 
     * @return Returns the unique, system-generated identifier for a partnership.
     */
    public final String partnershipId() {
        return partnershipId;
    }

    /**
     * <p>
     * Returns the name of the partnership.
     * </p>
     * 
     * @return Returns the name of the partnership.
     */
    public final String name() {
        return name;
    }

    /**
     * For responses, this returns true if the service returned a value for the Capabilities property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasCapabilities() {
        return capabilities != null && !(capabilities instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Returns one or more capabilities associated with this partnership.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCapabilities} method.
     * </p>
     * 
     * @return Returns one or more capabilities associated with this partnership.
     */
    public final List<String> capabilities() {
        return capabilities;
    }

    /**
     * Returns the value of the CapabilityOptions property for this object.
     * 
     * @return The value of the CapabilityOptions property for this object.
     */
    public final CapabilityOptions capabilityOptions() {
        return capabilityOptions;
    }

    /**
     * <p>
     * Returns the unique, system-generated identifier for a trading partner.
     * </p>
     * 
     * @return Returns the unique, system-generated identifier for a trading partner.
     */
    public final String tradingPartnerId() {
        return tradingPartnerId;
    }

    /**
     * <p>
     * Returns a timestamp for creation date and time of the partnership.
     * </p>
     * 
     * @return Returns a timestamp for creation date and time of the partnership.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * Returns a timestamp that identifies the most recent date and time that the partnership was modified.
     * </p>
     * 
     * @return Returns a timestamp that identifies the most recent date and time that the partnership was modified.
     */
    public final Instant modifiedAt() {
        return modifiedAt;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(profileId());
        hashCode = 31 * hashCode + Objects.hashCode(partnershipId());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(hasCapabilities() ? capabilities() : null);
        hashCode = 31 * hashCode + Objects.hashCode(capabilityOptions());
        hashCode = 31 * hashCode + Objects.hashCode(tradingPartnerId());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(modifiedAt());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PartnershipSummary)) {
            return false;
        }
        PartnershipSummary other = (PartnershipSummary) obj;
        return Objects.equals(profileId(), other.profileId()) && Objects.equals(partnershipId(), other.partnershipId())
                && Objects.equals(name(), other.name()) && hasCapabilities() == other.hasCapabilities()
                && Objects.equals(capabilities(), other.capabilities())
                && Objects.equals(capabilityOptions(), other.capabilityOptions())
                && Objects.equals(tradingPartnerId(), other.tradingPartnerId()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(modifiedAt(), other.modifiedAt());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PartnershipSummary").add("ProfileId", profileId()).add("PartnershipId", partnershipId())
                .add("Name", name()).add("Capabilities", hasCapabilities() ? capabilities() : null)
                .add("CapabilityOptions", capabilityOptions()).add("TradingPartnerId", tradingPartnerId())
                .add("CreatedAt", createdAt()).add("ModifiedAt", modifiedAt()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "profileId":
            return Optional.ofNullable(clazz.cast(profileId()));
        case "partnershipId":
            return Optional.ofNullable(clazz.cast(partnershipId()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "capabilities":
            return Optional.ofNullable(clazz.cast(capabilities()));
        case "capabilityOptions":
            return Optional.ofNullable(clazz.cast(capabilityOptions()));
        case "tradingPartnerId":
            return Optional.ofNullable(clazz.cast(tradingPartnerId()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "modifiedAt":
            return Optional.ofNullable(clazz.cast(modifiedAt()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PartnershipSummary, T> g) {
        return obj -> g.apply((PartnershipSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PartnershipSummary> {
        /**
         * <p>
         * Returns the unique, system-generated identifier for the profile connected to this partnership.
         * </p>
         * 
         * @param profileId
         *        Returns the unique, system-generated identifier for the profile connected to this partnership.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder profileId(String profileId);

        /**
         * <p>
         * Returns the unique, system-generated identifier for a partnership.
         * </p>
         * 
         * @param partnershipId
         *        Returns the unique, system-generated identifier for a partnership.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder partnershipId(String partnershipId);

        /**
         * <p>
         * Returns the name of the partnership.
         * </p>
         * 
         * @param name
         *        Returns the name of the partnership.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Returns one or more capabilities associated with this partnership.
         * </p>
         * 
         * @param capabilities
         *        Returns one or more capabilities associated with this partnership.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capabilities(Collection<String> capabilities);

        /**
         * <p>
         * Returns one or more capabilities associated with this partnership.
         * </p>
         * 
         * @param capabilities
         *        Returns one or more capabilities associated with this partnership.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capabilities(String... capabilities);

        /**
         * Sets the value of the CapabilityOptions property for this object.
         *
         * @param capabilityOptions
         *        The new value for the CapabilityOptions property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capabilityOptions(CapabilityOptions capabilityOptions);

        /**
         * Sets the value of the CapabilityOptions property for this object.
         *
         * This is a convenience method that creates an instance of the {@link CapabilityOptions.Builder} avoiding the
         * need to create one manually via {@link CapabilityOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CapabilityOptions.Builder#build()} is called immediately and its
         * result is passed to {@link #capabilityOptions(CapabilityOptions)}.
         * 
         * @param capabilityOptions
         *        a consumer that will call methods on {@link CapabilityOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #capabilityOptions(CapabilityOptions)
         */
        default Builder capabilityOptions(Consumer<CapabilityOptions.Builder> capabilityOptions) {
            return capabilityOptions(CapabilityOptions.builder().applyMutation(capabilityOptions).build());
        }

        /**
         * <p>
         * Returns the unique, system-generated identifier for a trading partner.
         * </p>
         * 
         * @param tradingPartnerId
         *        Returns the unique, system-generated identifier for a trading partner.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tradingPartnerId(String tradingPartnerId);

        /**
         * <p>
         * Returns a timestamp for creation date and time of the partnership.
         * </p>
         * 
         * @param createdAt
         *        Returns a timestamp for creation date and time of the partnership.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * Returns a timestamp that identifies the most recent date and time that the partnership was modified.
         * </p>
         * 
         * @param modifiedAt
         *        Returns a timestamp that identifies the most recent date and time that the partnership was modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modifiedAt(Instant modifiedAt);
    }

    static final class BuilderImpl implements Builder {
        private String profileId;

        private String partnershipId;

        private String name;

        private List<String> capabilities = DefaultSdkAutoConstructList.getInstance();

        private CapabilityOptions capabilityOptions;

        private String tradingPartnerId;

        private Instant createdAt;

        private Instant modifiedAt;

        private BuilderImpl() {
        }

        private BuilderImpl(PartnershipSummary model) {
            profileId(model.profileId);
            partnershipId(model.partnershipId);
            name(model.name);
            capabilities(model.capabilities);
            capabilityOptions(model.capabilityOptions);
            tradingPartnerId(model.tradingPartnerId);
            createdAt(model.createdAt);
            modifiedAt(model.modifiedAt);
        }

        public final String getProfileId() {
            return profileId;
        }

        public final void setProfileId(String profileId) {
            this.profileId = profileId;
        }

        @Override
        public final Builder profileId(String profileId) {
            this.profileId = profileId;
            return this;
        }

        public final String getPartnershipId() {
            return partnershipId;
        }

        public final void setPartnershipId(String partnershipId) {
            this.partnershipId = partnershipId;
        }

        @Override
        public final Builder partnershipId(String partnershipId) {
            this.partnershipId = partnershipId;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Collection<String> getCapabilities() {
            if (capabilities instanceof SdkAutoConstructList) {
                return null;
            }
            return capabilities;
        }

        public final void setCapabilities(Collection<String> capabilities) {
            this.capabilities = PartnershipCapabilitiesCopier.copy(capabilities);
        }

        @Override
        public final Builder capabilities(Collection<String> capabilities) {
            this.capabilities = PartnershipCapabilitiesCopier.copy(capabilities);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder capabilities(String... capabilities) {
            capabilities(Arrays.asList(capabilities));
            return this;
        }

        public final CapabilityOptions.Builder getCapabilityOptions() {
            return capabilityOptions != null ? capabilityOptions.toBuilder() : null;
        }

        public final void setCapabilityOptions(CapabilityOptions.BuilderImpl capabilityOptions) {
            this.capabilityOptions = capabilityOptions != null ? capabilityOptions.build() : null;
        }

        @Override
        public final Builder capabilityOptions(CapabilityOptions capabilityOptions) {
            this.capabilityOptions = capabilityOptions;
            return this;
        }

        public final String getTradingPartnerId() {
            return tradingPartnerId;
        }

        public final void setTradingPartnerId(String tradingPartnerId) {
            this.tradingPartnerId = tradingPartnerId;
        }

        @Override
        public final Builder tradingPartnerId(String tradingPartnerId) {
            this.tradingPartnerId = tradingPartnerId;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final Instant getModifiedAt() {
            return modifiedAt;
        }

        public final void setModifiedAt(Instant modifiedAt) {
            this.modifiedAt = modifiedAt;
        }

        @Override
        public final Builder modifiedAt(Instant modifiedAt) {
            this.modifiedAt = modifiedAt;
            return this;
        }

        @Override
        public PartnershipSummary build() {
            return new PartnershipSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
