/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.b2bi.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains configuration options for X12 EDI validation. This structure allows you to specify custom validation rules
 * that will be applied during EDI document processing, including element length constraints, code list modifications,
 * and element requirement changes. These validation options provide flexibility to accommodate trading partner-specific
 * requirements while maintaining EDI compliance. The validation rules are applied in addition to standard X12
 * validation to ensure documents meet both standard and custom requirements.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class X12ValidationOptions implements SdkPojo, Serializable,
        ToCopyableBuilder<X12ValidationOptions.Builder, X12ValidationOptions> {
    private static final SdkField<List<X12ValidationRule>> VALIDATION_RULES_FIELD = SdkField
            .<List<X12ValidationRule>> builder(MarshallingType.LIST)
            .memberName("validationRules")
            .getter(getter(X12ValidationOptions::validationRules))
            .setter(setter(Builder::validationRules))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("validationRules").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<X12ValidationRule> builder(MarshallingType.SDK_POJO)
                                            .constructor(X12ValidationRule::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VALIDATION_RULES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<X12ValidationRule> validationRules;

    private X12ValidationOptions(BuilderImpl builder) {
        this.validationRules = builder.validationRules;
    }

    /**
     * For responses, this returns true if the service returned a value for the ValidationRules property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasValidationRules() {
        return validationRules != null && !(validationRules instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies a list of validation rules to apply during EDI document processing. These rules can include code list
     * modifications, element length constraints, and element requirement changes.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasValidationRules} method.
     * </p>
     * 
     * @return Specifies a list of validation rules to apply during EDI document processing. These rules can include
     *         code list modifications, element length constraints, and element requirement changes.
     */
    public final List<X12ValidationRule> validationRules() {
        return validationRules;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasValidationRules() ? validationRules() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof X12ValidationOptions)) {
            return false;
        }
        X12ValidationOptions other = (X12ValidationOptions) obj;
        return hasValidationRules() == other.hasValidationRules() && Objects.equals(validationRules(), other.validationRules());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("X12ValidationOptions").add("ValidationRules", hasValidationRules() ? validationRules() : null)
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "validationRules":
            return Optional.ofNullable(clazz.cast(validationRules()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("validationRules", VALIDATION_RULES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<X12ValidationOptions, T> g) {
        return obj -> g.apply((X12ValidationOptions) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, X12ValidationOptions> {
        /**
         * <p>
         * Specifies a list of validation rules to apply during EDI document processing. These rules can include code
         * list modifications, element length constraints, and element requirement changes.
         * </p>
         * 
         * @param validationRules
         *        Specifies a list of validation rules to apply during EDI document processing. These rules can include
         *        code list modifications, element length constraints, and element requirement changes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validationRules(Collection<X12ValidationRule> validationRules);

        /**
         * <p>
         * Specifies a list of validation rules to apply during EDI document processing. These rules can include code
         * list modifications, element length constraints, and element requirement changes.
         * </p>
         * 
         * @param validationRules
         *        Specifies a list of validation rules to apply during EDI document processing. These rules can include
         *        code list modifications, element length constraints, and element requirement changes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validationRules(X12ValidationRule... validationRules);

        /**
         * <p>
         * Specifies a list of validation rules to apply during EDI document processing. These rules can include code
         * list modifications, element length constraints, and element requirement changes.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.b2bi.model.X12ValidationRule.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.b2bi.model.X12ValidationRule#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.b2bi.model.X12ValidationRule.Builder#build()} is called immediately
         * and its result is passed to {@link #validationRules(List<X12ValidationRule>)}.
         * 
         * @param validationRules
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.b2bi.model.X12ValidationRule.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #validationRules(java.util.Collection<X12ValidationRule>)
         */
        Builder validationRules(Consumer<X12ValidationRule.Builder>... validationRules);
    }

    static final class BuilderImpl implements Builder {
        private List<X12ValidationRule> validationRules = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(X12ValidationOptions model) {
            validationRules(model.validationRules);
        }

        public final List<X12ValidationRule.Builder> getValidationRules() {
            List<X12ValidationRule.Builder> result = X12ValidationRulesCopier.copyToBuilder(this.validationRules);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setValidationRules(Collection<X12ValidationRule.BuilderImpl> validationRules) {
            this.validationRules = X12ValidationRulesCopier.copyFromBuilder(validationRules);
        }

        @Override
        public final Builder validationRules(Collection<X12ValidationRule> validationRules) {
            this.validationRules = X12ValidationRulesCopier.copy(validationRules);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder validationRules(X12ValidationRule... validationRules) {
            validationRules(Arrays.asList(validationRules));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder validationRules(Consumer<X12ValidationRule.Builder>... validationRules) {
            validationRules(Stream.of(validationRules).map(c -> X12ValidationRule.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public X12ValidationOptions build() {
            return new X12ValidationOptions(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
