/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.backup.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Used to specify a set of resources to a backup plan.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class BackupSelection implements SdkPojo, Serializable, ToCopyableBuilder<BackupSelection.Builder, BackupSelection> {
    private static final SdkField<String> SELECTION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(BackupSelection::selectionName)).setter(setter(Builder::selectionName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SelectionName").build()).build();

    private static final SdkField<String> IAM_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(BackupSelection::iamRoleArn)).setter(setter(Builder::iamRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IamRoleArn").build()).build();

    private static final SdkField<List<String>> RESOURCES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(BackupSelection::resources))
            .setter(setter(Builder::resources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Resources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Condition>> LIST_OF_TAGS_FIELD = SdkField
            .<List<Condition>> builder(MarshallingType.LIST)
            .getter(getter(BackupSelection::listOfTags))
            .setter(setter(Builder::listOfTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ListOfTags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Condition> builder(MarshallingType.SDK_POJO)
                                            .constructor(Condition::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SELECTION_NAME_FIELD,
            IAM_ROLE_ARN_FIELD, RESOURCES_FIELD, LIST_OF_TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String selectionName;

    private final String iamRoleArn;

    private final List<String> resources;

    private final List<Condition> listOfTags;

    private BackupSelection(BuilderImpl builder) {
        this.selectionName = builder.selectionName;
        this.iamRoleArn = builder.iamRoleArn;
        this.resources = builder.resources;
        this.listOfTags = builder.listOfTags;
    }

    /**
     * <p>
     * The display name of a resource selection document.
     * </p>
     * 
     * @return The display name of a resource selection document.
     */
    public String selectionName() {
        return selectionName;
    }

    /**
     * <p>
     * The ARN of the IAM role that AWS Backup uses to authenticate when restoring the target resource; for example,
     * <code>arn:aws:iam::123456789012:role/S3Access</code>.
     * </p>
     * 
     * @return The ARN of the IAM role that AWS Backup uses to authenticate when restoring the target resource; for
     *         example, <code>arn:aws:iam::123456789012:role/S3Access</code>.
     */
    public String iamRoleArn() {
        return iamRoleArn;
    }

    /**
     * Returns true if the Resources property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasResources() {
        return resources != null && !(resources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of strings that either contain Amazon Resource Names (ARNs) or match patterns such as "
     * <code>arn:aws:ec2:us-east-1:123456789012:volume/*</code>" of resources to assign to a backup plan.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasResources()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of strings that either contain Amazon Resource Names (ARNs) or match patterns such as "
     *         <code>arn:aws:ec2:us-east-1:123456789012:volume/*</code>" of resources to assign to a backup plan.
     */
    public List<String> resources() {
        return resources;
    }

    /**
     * Returns true if the ListOfTags property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasListOfTags() {
        return listOfTags != null && !(listOfTags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of conditions used to specify a set of resources to assign to a backup plan; for example,
     * <code>"StringEquals": {"ec2:ResourceTag/Department": "accounting"</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasListOfTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of conditions used to specify a set of resources to assign to a backup plan; for example,
     *         <code>"StringEquals": {"ec2:ResourceTag/Department": "accounting"</code>.
     */
    public List<Condition> listOfTags() {
        return listOfTags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(selectionName());
        hashCode = 31 * hashCode + Objects.hashCode(iamRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(resources());
        hashCode = 31 * hashCode + Objects.hashCode(listOfTags());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof BackupSelection)) {
            return false;
        }
        BackupSelection other = (BackupSelection) obj;
        return Objects.equals(selectionName(), other.selectionName()) && Objects.equals(iamRoleArn(), other.iamRoleArn())
                && Objects.equals(resources(), other.resources()) && Objects.equals(listOfTags(), other.listOfTags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("BackupSelection").add("SelectionName", selectionName()).add("IamRoleArn", iamRoleArn())
                .add("Resources", resources()).add("ListOfTags", listOfTags()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SelectionName":
            return Optional.ofNullable(clazz.cast(selectionName()));
        case "IamRoleArn":
            return Optional.ofNullable(clazz.cast(iamRoleArn()));
        case "Resources":
            return Optional.ofNullable(clazz.cast(resources()));
        case "ListOfTags":
            return Optional.ofNullable(clazz.cast(listOfTags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<BackupSelection, T> g) {
        return obj -> g.apply((BackupSelection) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, BackupSelection> {
        /**
         * <p>
         * The display name of a resource selection document.
         * </p>
         * 
         * @param selectionName
         *        The display name of a resource selection document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder selectionName(String selectionName);

        /**
         * <p>
         * The ARN of the IAM role that AWS Backup uses to authenticate when restoring the target resource; for example,
         * <code>arn:aws:iam::123456789012:role/S3Access</code>.
         * </p>
         * 
         * @param iamRoleArn
         *        The ARN of the IAM role that AWS Backup uses to authenticate when restoring the target resource; for
         *        example, <code>arn:aws:iam::123456789012:role/S3Access</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iamRoleArn(String iamRoleArn);

        /**
         * <p>
         * An array of strings that either contain Amazon Resource Names (ARNs) or match patterns such as "
         * <code>arn:aws:ec2:us-east-1:123456789012:volume/*</code>" of resources to assign to a backup plan.
         * </p>
         * 
         * @param resources
         *        An array of strings that either contain Amazon Resource Names (ARNs) or match patterns such as "
         *        <code>arn:aws:ec2:us-east-1:123456789012:volume/*</code>" of resources to assign to a backup plan.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resources(Collection<String> resources);

        /**
         * <p>
         * An array of strings that either contain Amazon Resource Names (ARNs) or match patterns such as "
         * <code>arn:aws:ec2:us-east-1:123456789012:volume/*</code>" of resources to assign to a backup plan.
         * </p>
         * 
         * @param resources
         *        An array of strings that either contain Amazon Resource Names (ARNs) or match patterns such as "
         *        <code>arn:aws:ec2:us-east-1:123456789012:volume/*</code>" of resources to assign to a backup plan.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resources(String... resources);

        /**
         * <p>
         * An array of conditions used to specify a set of resources to assign to a backup plan; for example,
         * <code>"StringEquals": {"ec2:ResourceTag/Department": "accounting"</code>.
         * </p>
         * 
         * @param listOfTags
         *        An array of conditions used to specify a set of resources to assign to a backup plan; for example,
         *        <code>"StringEquals": {"ec2:ResourceTag/Department": "accounting"</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder listOfTags(Collection<Condition> listOfTags);

        /**
         * <p>
         * An array of conditions used to specify a set of resources to assign to a backup plan; for example,
         * <code>"StringEquals": {"ec2:ResourceTag/Department": "accounting"</code>.
         * </p>
         * 
         * @param listOfTags
         *        An array of conditions used to specify a set of resources to assign to a backup plan; for example,
         *        <code>"StringEquals": {"ec2:ResourceTag/Department": "accounting"</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder listOfTags(Condition... listOfTags);

        /**
         * <p>
         * An array of conditions used to specify a set of resources to assign to a backup plan; for example,
         * <code>"StringEquals": {"ec2:ResourceTag/Department": "accounting"</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Condition>.Builder} avoiding the need to
         * create one manually via {@link List<Condition>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Condition>.Builder#build()} is called immediately and its
         * result is passed to {@link #listOfTags(List<Condition>)}.
         * 
         * @param listOfTags
         *        a consumer that will call methods on {@link List<Condition>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #listOfTags(List<Condition>)
         */
        Builder listOfTags(Consumer<Condition.Builder>... listOfTags);
    }

    static final class BuilderImpl implements Builder {
        private String selectionName;

        private String iamRoleArn;

        private List<String> resources = DefaultSdkAutoConstructList.getInstance();

        private List<Condition> listOfTags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(BackupSelection model) {
            selectionName(model.selectionName);
            iamRoleArn(model.iamRoleArn);
            resources(model.resources);
            listOfTags(model.listOfTags);
        }

        public final String getSelectionName() {
            return selectionName;
        }

        @Override
        public final Builder selectionName(String selectionName) {
            this.selectionName = selectionName;
            return this;
        }

        public final void setSelectionName(String selectionName) {
            this.selectionName = selectionName;
        }

        public final String getIamRoleArn() {
            return iamRoleArn;
        }

        @Override
        public final Builder iamRoleArn(String iamRoleArn) {
            this.iamRoleArn = iamRoleArn;
            return this;
        }

        public final void setIamRoleArn(String iamRoleArn) {
            this.iamRoleArn = iamRoleArn;
        }

        public final Collection<String> getResources() {
            return resources;
        }

        @Override
        public final Builder resources(Collection<String> resources) {
            this.resources = ResourceArnsCopier.copy(resources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resources(String... resources) {
            resources(Arrays.asList(resources));
            return this;
        }

        public final void setResources(Collection<String> resources) {
            this.resources = ResourceArnsCopier.copy(resources);
        }

        public final Collection<Condition.Builder> getListOfTags() {
            return listOfTags != null ? listOfTags.stream().map(Condition::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder listOfTags(Collection<Condition> listOfTags) {
            this.listOfTags = ListOfTagsCopier.copy(listOfTags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder listOfTags(Condition... listOfTags) {
            listOfTags(Arrays.asList(listOfTags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder listOfTags(Consumer<Condition.Builder>... listOfTags) {
            listOfTags(Stream.of(listOfTags).map(c -> Condition.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setListOfTags(Collection<Condition.BuilderImpl> listOfTags) {
            this.listOfTags = ListOfTagsCopier.copyFromBuilder(listOfTags);
        }

        @Override
        public BackupSelection build() {
            return new BackupSelection(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
