/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.backup.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A list of backup options for each resource type.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AdvancedBackupSetting implements SdkPojo, Serializable,
        ToCopyableBuilder<AdvancedBackupSetting.Builder, AdvancedBackupSetting> {
    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceType").getter(getter(AdvancedBackupSetting::resourceType)).setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceType").build()).build();

    private static final SdkField<Map<String, String>> BACKUP_OPTIONS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("BackupOptions")
            .getter(getter(AdvancedBackupSetting::backupOptions))
            .setter(setter(Builder::backupOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BackupOptions").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RESOURCE_TYPE_FIELD,
            BACKUP_OPTIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String resourceType;

    private final Map<String, String> backupOptions;

    private AdvancedBackupSetting(BuilderImpl builder) {
        this.resourceType = builder.resourceType;
        this.backupOptions = builder.backupOptions;
    }

    /**
     * <p>
     * The type of AWS resource to be backed up. For VSS Windows backups, the only supported resource type is Amazon
     * EC2.
     * </p>
     * <p>
     * Valid values: <code>EC2</code>.
     * </p>
     * 
     * @return The type of AWS resource to be backed up. For VSS Windows backups, the only supported resource type is
     *         Amazon EC2.</p>
     *         <p>
     *         Valid values: <code>EC2</code>.
     */
    public String resourceType() {
        return resourceType;
    }

    /**
     * Returns true if the BackupOptions property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasBackupOptions() {
        return backupOptions != null && !(backupOptions instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Specifies the backup option for a selected resource. This option is only available for Windows VSS backup jobs.
     * </p>
     * <p>
     * Valid values: Set to <code>"WindowsVSS”:“enabled"</code> to enable WindowsVSS backup option and create a VSS
     * Windows backup. Set to “WindowsVSS”:”disabled” to create a regular backup. The WindowsVSS option is not enabled
     * by default.
     * </p>
     * <p>
     * If you specify an invalid option, you get an <code>InvalidParameterValueException</code> exception.
     * </p>
     * <p>
     * For more information about Windows VSS backups, see <a
     * href="https://docs.aws.amazon.com/aws-backup/latest/devguide/windows-backups.html">Creating a VSS-Enabled Windows
     * Backup</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasBackupOptions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Specifies the backup option for a selected resource. This option is only available for Windows VSS backup
     *         jobs.</p>
     *         <p>
     *         Valid values: Set to <code>"WindowsVSS”:“enabled"</code> to enable WindowsVSS backup option and create a
     *         VSS Windows backup. Set to “WindowsVSS”:”disabled” to create a regular backup. The WindowsVSS option is
     *         not enabled by default.
     *         </p>
     *         <p>
     *         If you specify an invalid option, you get an <code>InvalidParameterValueException</code> exception.
     *         </p>
     *         <p>
     *         For more information about Windows VSS backups, see <a
     *         href="https://docs.aws.amazon.com/aws-backup/latest/devguide/windows-backups.html">Creating a VSS-Enabled
     *         Windows Backup</a>.
     */
    public Map<String, String> backupOptions() {
        return backupOptions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(resourceType());
        hashCode = 31 * hashCode + Objects.hashCode(hasBackupOptions() ? backupOptions() : null);
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AdvancedBackupSetting)) {
            return false;
        }
        AdvancedBackupSetting other = (AdvancedBackupSetting) obj;
        return Objects.equals(resourceType(), other.resourceType()) && hasBackupOptions() == other.hasBackupOptions()
                && Objects.equals(backupOptions(), other.backupOptions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("AdvancedBackupSetting").add("ResourceType", resourceType())
                .add("BackupOptions", hasBackupOptions() ? backupOptions() : null).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ResourceType":
            return Optional.ofNullable(clazz.cast(resourceType()));
        case "BackupOptions":
            return Optional.ofNullable(clazz.cast(backupOptions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AdvancedBackupSetting, T> g) {
        return obj -> g.apply((AdvancedBackupSetting) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AdvancedBackupSetting> {
        /**
         * <p>
         * The type of AWS resource to be backed up. For VSS Windows backups, the only supported resource type is Amazon
         * EC2.
         * </p>
         * <p>
         * Valid values: <code>EC2</code>.
         * </p>
         * 
         * @param resourceType
         *        The type of AWS resource to be backed up. For VSS Windows backups, the only supported resource type is
         *        Amazon EC2.</p>
         *        <p>
         *        Valid values: <code>EC2</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * Specifies the backup option for a selected resource. This option is only available for Windows VSS backup
         * jobs.
         * </p>
         * <p>
         * Valid values: Set to <code>"WindowsVSS”:“enabled"</code> to enable WindowsVSS backup option and create a VSS
         * Windows backup. Set to “WindowsVSS”:”disabled” to create a regular backup. The WindowsVSS option is not
         * enabled by default.
         * </p>
         * <p>
         * If you specify an invalid option, you get an <code>InvalidParameterValueException</code> exception.
         * </p>
         * <p>
         * For more information about Windows VSS backups, see <a
         * href="https://docs.aws.amazon.com/aws-backup/latest/devguide/windows-backups.html">Creating a VSS-Enabled
         * Windows Backup</a>.
         * </p>
         * 
         * @param backupOptions
         *        Specifies the backup option for a selected resource. This option is only available for Windows VSS
         *        backup jobs.</p>
         *        <p>
         *        Valid values: Set to <code>"WindowsVSS”:“enabled"</code> to enable WindowsVSS backup option and create
         *        a VSS Windows backup. Set to “WindowsVSS”:”disabled” to create a regular backup. The WindowsVSS option
         *        is not enabled by default.
         *        </p>
         *        <p>
         *        If you specify an invalid option, you get an <code>InvalidParameterValueException</code> exception.
         *        </p>
         *        <p>
         *        For more information about Windows VSS backups, see <a
         *        href="https://docs.aws.amazon.com/aws-backup/latest/devguide/windows-backups.html">Creating a
         *        VSS-Enabled Windows Backup</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backupOptions(Map<String, String> backupOptions);
    }

    static final class BuilderImpl implements Builder {
        private String resourceType;

        private Map<String, String> backupOptions = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(AdvancedBackupSetting model) {
            resourceType(model.resourceType);
            backupOptions(model.backupOptions);
        }

        public final String getResourceType() {
            return resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        public final Map<String, String> getBackupOptions() {
            if (backupOptions instanceof SdkAutoConstructMap) {
                return null;
            }
            return backupOptions;
        }

        @Override
        public final Builder backupOptions(Map<String, String> backupOptions) {
            this.backupOptions = BackupOptionsCopier.copy(backupOptions);
            return this;
        }

        public final void setBackupOptions(Map<String, String> backupOptions) {
            this.backupOptions = BackupOptionsCopier.copy(backupOptions);
        }

        @Override
        public AdvancedBackupSetting build() {
            return new AdvancedBackupSetting(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
