/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.backup;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.backup.model.AlreadyExistsException;
import software.amazon.awssdk.services.backup.model.BackupException;
import software.amazon.awssdk.services.backup.model.BackupRequest;
import software.amazon.awssdk.services.backup.model.CreateBackupPlanRequest;
import software.amazon.awssdk.services.backup.model.CreateBackupPlanResponse;
import software.amazon.awssdk.services.backup.model.CreateBackupSelectionRequest;
import software.amazon.awssdk.services.backup.model.CreateBackupSelectionResponse;
import software.amazon.awssdk.services.backup.model.CreateBackupVaultRequest;
import software.amazon.awssdk.services.backup.model.CreateBackupVaultResponse;
import software.amazon.awssdk.services.backup.model.DeleteBackupPlanRequest;
import software.amazon.awssdk.services.backup.model.DeleteBackupPlanResponse;
import software.amazon.awssdk.services.backup.model.DeleteBackupSelectionRequest;
import software.amazon.awssdk.services.backup.model.DeleteBackupSelectionResponse;
import software.amazon.awssdk.services.backup.model.DeleteBackupVaultAccessPolicyRequest;
import software.amazon.awssdk.services.backup.model.DeleteBackupVaultAccessPolicyResponse;
import software.amazon.awssdk.services.backup.model.DeleteBackupVaultNotificationsRequest;
import software.amazon.awssdk.services.backup.model.DeleteBackupVaultNotificationsResponse;
import software.amazon.awssdk.services.backup.model.DeleteBackupVaultRequest;
import software.amazon.awssdk.services.backup.model.DeleteBackupVaultResponse;
import software.amazon.awssdk.services.backup.model.DeleteRecoveryPointRequest;
import software.amazon.awssdk.services.backup.model.DeleteRecoveryPointResponse;
import software.amazon.awssdk.services.backup.model.DependencyFailureException;
import software.amazon.awssdk.services.backup.model.DescribeBackupJobRequest;
import software.amazon.awssdk.services.backup.model.DescribeBackupJobResponse;
import software.amazon.awssdk.services.backup.model.DescribeBackupVaultRequest;
import software.amazon.awssdk.services.backup.model.DescribeBackupVaultResponse;
import software.amazon.awssdk.services.backup.model.DescribeCopyJobRequest;
import software.amazon.awssdk.services.backup.model.DescribeCopyJobResponse;
import software.amazon.awssdk.services.backup.model.DescribeGlobalSettingsRequest;
import software.amazon.awssdk.services.backup.model.DescribeGlobalSettingsResponse;
import software.amazon.awssdk.services.backup.model.DescribeProtectedResourceRequest;
import software.amazon.awssdk.services.backup.model.DescribeProtectedResourceResponse;
import software.amazon.awssdk.services.backup.model.DescribeRecoveryPointRequest;
import software.amazon.awssdk.services.backup.model.DescribeRecoveryPointResponse;
import software.amazon.awssdk.services.backup.model.DescribeRegionSettingsRequest;
import software.amazon.awssdk.services.backup.model.DescribeRegionSettingsResponse;
import software.amazon.awssdk.services.backup.model.DescribeRestoreJobRequest;
import software.amazon.awssdk.services.backup.model.DescribeRestoreJobResponse;
import software.amazon.awssdk.services.backup.model.DisassociateRecoveryPointRequest;
import software.amazon.awssdk.services.backup.model.DisassociateRecoveryPointResponse;
import software.amazon.awssdk.services.backup.model.ExportBackupPlanTemplateRequest;
import software.amazon.awssdk.services.backup.model.ExportBackupPlanTemplateResponse;
import software.amazon.awssdk.services.backup.model.GetBackupPlanFromJsonRequest;
import software.amazon.awssdk.services.backup.model.GetBackupPlanFromJsonResponse;
import software.amazon.awssdk.services.backup.model.GetBackupPlanFromTemplateRequest;
import software.amazon.awssdk.services.backup.model.GetBackupPlanFromTemplateResponse;
import software.amazon.awssdk.services.backup.model.GetBackupPlanRequest;
import software.amazon.awssdk.services.backup.model.GetBackupPlanResponse;
import software.amazon.awssdk.services.backup.model.GetBackupSelectionRequest;
import software.amazon.awssdk.services.backup.model.GetBackupSelectionResponse;
import software.amazon.awssdk.services.backup.model.GetBackupVaultAccessPolicyRequest;
import software.amazon.awssdk.services.backup.model.GetBackupVaultAccessPolicyResponse;
import software.amazon.awssdk.services.backup.model.GetBackupVaultNotificationsRequest;
import software.amazon.awssdk.services.backup.model.GetBackupVaultNotificationsResponse;
import software.amazon.awssdk.services.backup.model.GetRecoveryPointRestoreMetadataRequest;
import software.amazon.awssdk.services.backup.model.GetRecoveryPointRestoreMetadataResponse;
import software.amazon.awssdk.services.backup.model.GetSupportedResourceTypesRequest;
import software.amazon.awssdk.services.backup.model.GetSupportedResourceTypesResponse;
import software.amazon.awssdk.services.backup.model.InvalidParameterValueException;
import software.amazon.awssdk.services.backup.model.InvalidRequestException;
import software.amazon.awssdk.services.backup.model.InvalidResourceStateException;
import software.amazon.awssdk.services.backup.model.LimitExceededException;
import software.amazon.awssdk.services.backup.model.ListBackupJobsRequest;
import software.amazon.awssdk.services.backup.model.ListBackupJobsResponse;
import software.amazon.awssdk.services.backup.model.ListBackupPlanTemplatesRequest;
import software.amazon.awssdk.services.backup.model.ListBackupPlanTemplatesResponse;
import software.amazon.awssdk.services.backup.model.ListBackupPlanVersionsRequest;
import software.amazon.awssdk.services.backup.model.ListBackupPlanVersionsResponse;
import software.amazon.awssdk.services.backup.model.ListBackupPlansRequest;
import software.amazon.awssdk.services.backup.model.ListBackupPlansResponse;
import software.amazon.awssdk.services.backup.model.ListBackupSelectionsRequest;
import software.amazon.awssdk.services.backup.model.ListBackupSelectionsResponse;
import software.amazon.awssdk.services.backup.model.ListBackupVaultsRequest;
import software.amazon.awssdk.services.backup.model.ListBackupVaultsResponse;
import software.amazon.awssdk.services.backup.model.ListCopyJobsRequest;
import software.amazon.awssdk.services.backup.model.ListCopyJobsResponse;
import software.amazon.awssdk.services.backup.model.ListProtectedResourcesRequest;
import software.amazon.awssdk.services.backup.model.ListProtectedResourcesResponse;
import software.amazon.awssdk.services.backup.model.ListRecoveryPointsByBackupVaultRequest;
import software.amazon.awssdk.services.backup.model.ListRecoveryPointsByBackupVaultResponse;
import software.amazon.awssdk.services.backup.model.ListRecoveryPointsByResourceRequest;
import software.amazon.awssdk.services.backup.model.ListRecoveryPointsByResourceResponse;
import software.amazon.awssdk.services.backup.model.ListRestoreJobsRequest;
import software.amazon.awssdk.services.backup.model.ListRestoreJobsResponse;
import software.amazon.awssdk.services.backup.model.ListTagsRequest;
import software.amazon.awssdk.services.backup.model.ListTagsResponse;
import software.amazon.awssdk.services.backup.model.MissingParameterValueException;
import software.amazon.awssdk.services.backup.model.PutBackupVaultAccessPolicyRequest;
import software.amazon.awssdk.services.backup.model.PutBackupVaultAccessPolicyResponse;
import software.amazon.awssdk.services.backup.model.PutBackupVaultNotificationsRequest;
import software.amazon.awssdk.services.backup.model.PutBackupVaultNotificationsResponse;
import software.amazon.awssdk.services.backup.model.ResourceNotFoundException;
import software.amazon.awssdk.services.backup.model.ServiceUnavailableException;
import software.amazon.awssdk.services.backup.model.StartBackupJobRequest;
import software.amazon.awssdk.services.backup.model.StartBackupJobResponse;
import software.amazon.awssdk.services.backup.model.StartCopyJobRequest;
import software.amazon.awssdk.services.backup.model.StartCopyJobResponse;
import software.amazon.awssdk.services.backup.model.StartRestoreJobRequest;
import software.amazon.awssdk.services.backup.model.StartRestoreJobResponse;
import software.amazon.awssdk.services.backup.model.StopBackupJobRequest;
import software.amazon.awssdk.services.backup.model.StopBackupJobResponse;
import software.amazon.awssdk.services.backup.model.TagResourceRequest;
import software.amazon.awssdk.services.backup.model.TagResourceResponse;
import software.amazon.awssdk.services.backup.model.UntagResourceRequest;
import software.amazon.awssdk.services.backup.model.UntagResourceResponse;
import software.amazon.awssdk.services.backup.model.UpdateBackupPlanRequest;
import software.amazon.awssdk.services.backup.model.UpdateBackupPlanResponse;
import software.amazon.awssdk.services.backup.model.UpdateGlobalSettingsRequest;
import software.amazon.awssdk.services.backup.model.UpdateGlobalSettingsResponse;
import software.amazon.awssdk.services.backup.model.UpdateRecoveryPointLifecycleRequest;
import software.amazon.awssdk.services.backup.model.UpdateRecoveryPointLifecycleResponse;
import software.amazon.awssdk.services.backup.model.UpdateRegionSettingsRequest;
import software.amazon.awssdk.services.backup.model.UpdateRegionSettingsResponse;
import software.amazon.awssdk.services.backup.paginators.ListBackupJobsIterable;
import software.amazon.awssdk.services.backup.paginators.ListBackupPlanTemplatesIterable;
import software.amazon.awssdk.services.backup.paginators.ListBackupPlanVersionsIterable;
import software.amazon.awssdk.services.backup.paginators.ListBackupPlansIterable;
import software.amazon.awssdk.services.backup.paginators.ListBackupSelectionsIterable;
import software.amazon.awssdk.services.backup.paginators.ListBackupVaultsIterable;
import software.amazon.awssdk.services.backup.paginators.ListCopyJobsIterable;
import software.amazon.awssdk.services.backup.paginators.ListProtectedResourcesIterable;
import software.amazon.awssdk.services.backup.paginators.ListRecoveryPointsByBackupVaultIterable;
import software.amazon.awssdk.services.backup.paginators.ListRecoveryPointsByResourceIterable;
import software.amazon.awssdk.services.backup.paginators.ListRestoreJobsIterable;
import software.amazon.awssdk.services.backup.paginators.ListTagsIterable;
import software.amazon.awssdk.services.backup.transform.CreateBackupPlanRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.CreateBackupSelectionRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.CreateBackupVaultRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.DeleteBackupPlanRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.DeleteBackupSelectionRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.DeleteBackupVaultAccessPolicyRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.DeleteBackupVaultNotificationsRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.DeleteBackupVaultRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.DeleteRecoveryPointRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.DescribeBackupJobRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.DescribeBackupVaultRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.DescribeCopyJobRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.DescribeGlobalSettingsRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.DescribeProtectedResourceRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.DescribeRecoveryPointRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.DescribeRegionSettingsRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.DescribeRestoreJobRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.DisassociateRecoveryPointRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.ExportBackupPlanTemplateRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.GetBackupPlanFromJsonRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.GetBackupPlanFromTemplateRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.GetBackupPlanRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.GetBackupSelectionRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.GetBackupVaultAccessPolicyRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.GetBackupVaultNotificationsRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.GetRecoveryPointRestoreMetadataRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.GetSupportedResourceTypesRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.ListBackupJobsRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.ListBackupPlanTemplatesRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.ListBackupPlanVersionsRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.ListBackupPlansRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.ListBackupSelectionsRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.ListBackupVaultsRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.ListCopyJobsRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.ListProtectedResourcesRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.ListRecoveryPointsByBackupVaultRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.ListRecoveryPointsByResourceRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.ListRestoreJobsRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.ListTagsRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.PutBackupVaultAccessPolicyRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.PutBackupVaultNotificationsRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.StartBackupJobRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.StartCopyJobRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.StartRestoreJobRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.StopBackupJobRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.UpdateBackupPlanRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.UpdateGlobalSettingsRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.UpdateRecoveryPointLifecycleRequestMarshaller;
import software.amazon.awssdk.services.backup.transform.UpdateRegionSettingsRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link BackupClient}.
 *
 * @see BackupClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultBackupClient implements BackupClient {
    private static final Logger log = Logger.loggerFor(DefaultBackupClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultBackupClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a backup plan using a backup plan name and backup rules. A backup plan is a document that contains
     * information that AWS Backup uses to schedule tasks that create recovery points for resources.
     * </p>
     * <p>
     * If you call <code>CreateBackupPlan</code> with a plan that already exists, an <code>AlreadyExistsException</code>
     * is returned.
     * </p>
     *
     * @param createBackupPlanRequest
     * @return Result of the CreateBackupPlan operation returned by the service.
     * @throws LimitExceededException
     *         A limit in the request has been exceeded; for example, a maximum number of items allowed in a request.
     * @throws AlreadyExistsException
     *         The required resource already exists.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.CreateBackupPlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/CreateBackupPlan" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateBackupPlanResponse createBackupPlan(CreateBackupPlanRequest createBackupPlanRequest)
            throws LimitExceededException, AlreadyExistsException, InvalidParameterValueException,
            MissingParameterValueException, ServiceUnavailableException, AwsServiceException, SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateBackupPlanResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateBackupPlanResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBackupPlanRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBackupPlan");

            return clientHandler.execute(new ClientExecutionParams<CreateBackupPlanRequest, CreateBackupPlanResponse>()
                    .withOperationName("CreateBackupPlan").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createBackupPlanRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateBackupPlanRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a JSON document that specifies a set of resources to assign to a backup plan. Resources can be included
     * by specifying patterns for a <code>ListOfTags</code> and selected <code>Resources</code>.
     * </p>
     * <p>
     * For example, consider the following patterns:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Resources: "arn:aws:ec2:region:account-id:volume/volume-id"</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ConditionKey:"department"</code>
     * </p>
     * <p>
     * <code>ConditionValue:"finance"</code>
     * </p>
     * <p>
     * <code>ConditionType:"StringEquals"</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ConditionKey:"importance"</code>
     * </p>
     * <p>
     * <code>ConditionValue:"critical"</code>
     * </p>
     * <p>
     * <code>ConditionType:"StringEquals"</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * Using these patterns would back up all Amazon Elastic Block Store (Amazon EBS) volumes that are tagged as
     * <code>"department=finance"</code>, <code>"importance=critical"</code>, in addition to an EBS volume with the
     * specified volume ID.
     * </p>
     * <p>
     * Resources and conditions are additive in that all resources that match the pattern are selected. This shouldn't
     * be confused with a logical AND, where all conditions must match. The matching patterns are logically put together
     * using the OR operator. In other words, all patterns that match are selected for backup.
     * </p>
     *
     * @param createBackupSelectionRequest
     * @return Result of the CreateBackupSelection operation returned by the service.
     * @throws LimitExceededException
     *         A limit in the request has been exceeded; for example, a maximum number of items allowed in a request.
     * @throws AlreadyExistsException
     *         The required resource already exists.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.CreateBackupSelection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/CreateBackupSelection" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateBackupSelectionResponse createBackupSelection(CreateBackupSelectionRequest createBackupSelectionRequest)
            throws LimitExceededException, AlreadyExistsException, InvalidParameterValueException,
            MissingParameterValueException, ServiceUnavailableException, AwsServiceException, SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateBackupSelectionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateBackupSelectionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBackupSelectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBackupSelection");

            return clientHandler.execute(new ClientExecutionParams<CreateBackupSelectionRequest, CreateBackupSelectionResponse>()
                    .withOperationName("CreateBackupSelection").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createBackupSelectionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateBackupSelectionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a logical container where backups are stored. A <code>CreateBackupVault</code> request includes a name,
     * optionally one or more resource tags, an encryption key, and a request ID.
     * </p>
     * <note>
     * <p>
     * Sensitive data, such as passport numbers, should not be included the name of a backup vault.
     * </p>
     * </note>
     *
     * @param createBackupVaultRequest
     * @return Result of the CreateBackupVault operation returned by the service.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws LimitExceededException
     *         A limit in the request has been exceeded; for example, a maximum number of items allowed in a request.
     * @throws AlreadyExistsException
     *         The required resource already exists.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.CreateBackupVault
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/CreateBackupVault" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateBackupVaultResponse createBackupVault(CreateBackupVaultRequest createBackupVaultRequest)
            throws InvalidParameterValueException, MissingParameterValueException, ServiceUnavailableException,
            LimitExceededException, AlreadyExistsException, AwsServiceException, SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateBackupVaultResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateBackupVaultResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBackupVaultRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBackupVault");

            return clientHandler.execute(new ClientExecutionParams<CreateBackupVaultRequest, CreateBackupVaultResponse>()
                    .withOperationName("CreateBackupVault").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createBackupVaultRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateBackupVaultRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a backup plan. A backup plan can only be deleted after all associated selections of resources have been
     * deleted. Deleting a backup plan deletes the current version of a backup plan. Previous versions, if any, will
     * still exist.
     * </p>
     *
     * @param deleteBackupPlanRequest
     * @return Result of the DeleteBackupPlan operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws InvalidRequestException
     *         Indicates that something is wrong with the input to the request. For example, a parameter is of the wrong
     *         type.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.DeleteBackupPlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/DeleteBackupPlan" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteBackupPlanResponse deleteBackupPlan(DeleteBackupPlanRequest deleteBackupPlanRequest)
            throws ResourceNotFoundException, InvalidParameterValueException, MissingParameterValueException,
            ServiceUnavailableException, InvalidRequestException, AwsServiceException, SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteBackupPlanResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteBackupPlanResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBackupPlanRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBackupPlan");

            return clientHandler.execute(new ClientExecutionParams<DeleteBackupPlanRequest, DeleteBackupPlanResponse>()
                    .withOperationName("DeleteBackupPlan").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteBackupPlanRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteBackupPlanRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the resource selection associated with a backup plan that is specified by the <code>SelectionId</code>.
     * </p>
     *
     * @param deleteBackupSelectionRequest
     * @return Result of the DeleteBackupSelection operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.DeleteBackupSelection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/DeleteBackupSelection" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteBackupSelectionResponse deleteBackupSelection(DeleteBackupSelectionRequest deleteBackupSelectionRequest)
            throws ResourceNotFoundException, InvalidParameterValueException, MissingParameterValueException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteBackupSelectionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteBackupSelectionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBackupSelectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBackupSelection");

            return clientHandler.execute(new ClientExecutionParams<DeleteBackupSelectionRequest, DeleteBackupSelectionResponse>()
                    .withOperationName("DeleteBackupSelection").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteBackupSelectionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteBackupSelectionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the backup vault identified by its name. A vault can be deleted only if it is empty.
     * </p>
     *
     * @param deleteBackupVaultRequest
     * @return Result of the DeleteBackupVault operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws InvalidRequestException
     *         Indicates that something is wrong with the input to the request. For example, a parameter is of the wrong
     *         type.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.DeleteBackupVault
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/DeleteBackupVault" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteBackupVaultResponse deleteBackupVault(DeleteBackupVaultRequest deleteBackupVaultRequest)
            throws ResourceNotFoundException, InvalidParameterValueException, MissingParameterValueException,
            ServiceUnavailableException, InvalidRequestException, AwsServiceException, SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteBackupVaultResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteBackupVaultResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBackupVaultRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBackupVault");

            return clientHandler.execute(new ClientExecutionParams<DeleteBackupVaultRequest, DeleteBackupVaultResponse>()
                    .withOperationName("DeleteBackupVault").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteBackupVaultRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteBackupVaultRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the policy document that manages permissions on a backup vault.
     * </p>
     *
     * @param deleteBackupVaultAccessPolicyRequest
     * @return Result of the DeleteBackupVaultAccessPolicy operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.DeleteBackupVaultAccessPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/DeleteBackupVaultAccessPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteBackupVaultAccessPolicyResponse deleteBackupVaultAccessPolicy(
            DeleteBackupVaultAccessPolicyRequest deleteBackupVaultAccessPolicyRequest) throws ResourceNotFoundException,
            InvalidParameterValueException, MissingParameterValueException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteBackupVaultAccessPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteBackupVaultAccessPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteBackupVaultAccessPolicyRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBackupVaultAccessPolicy");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteBackupVaultAccessPolicyRequest, DeleteBackupVaultAccessPolicyResponse>()
                            .withOperationName("DeleteBackupVaultAccessPolicy").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteBackupVaultAccessPolicyRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteBackupVaultAccessPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes event notifications for the specified backup vault.
     * </p>
     *
     * @param deleteBackupVaultNotificationsRequest
     * @return Result of the DeleteBackupVaultNotifications operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.DeleteBackupVaultNotifications
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/DeleteBackupVaultNotifications"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteBackupVaultNotificationsResponse deleteBackupVaultNotifications(
            DeleteBackupVaultNotificationsRequest deleteBackupVaultNotificationsRequest) throws ResourceNotFoundException,
            InvalidParameterValueException, MissingParameterValueException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteBackupVaultNotificationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteBackupVaultNotificationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteBackupVaultNotificationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBackupVaultNotifications");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteBackupVaultNotificationsRequest, DeleteBackupVaultNotificationsResponse>()
                            .withOperationName("DeleteBackupVaultNotifications").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteBackupVaultNotificationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteBackupVaultNotificationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the recovery point specified by a recovery point ID.
     * </p>
     * <p>
     * If the recovery point ID belongs to a continuous backup, calling this endpoint deletes the existing continuous
     * backup and stops future continuous backup.
     * </p>
     *
     * @param deleteRecoveryPointRequest
     * @return Result of the DeleteRecoveryPoint operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws InvalidResourceStateException
     *         AWS Backup is already performing an action on this recovery point. It can't perform the action you
     *         requested until the first action finishes. Try again later.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws InvalidRequestException
     *         Indicates that something is wrong with the input to the request. For example, a parameter is of the wrong
     *         type.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.DeleteRecoveryPoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/DeleteRecoveryPoint" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteRecoveryPointResponse deleteRecoveryPoint(DeleteRecoveryPointRequest deleteRecoveryPointRequest)
            throws ResourceNotFoundException, InvalidParameterValueException, MissingParameterValueException,
            InvalidResourceStateException, ServiceUnavailableException, InvalidRequestException, AwsServiceException,
            SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteRecoveryPointResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteRecoveryPointResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRecoveryPointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRecoveryPoint");

            return clientHandler.execute(new ClientExecutionParams<DeleteRecoveryPointRequest, DeleteRecoveryPointResponse>()
                    .withOperationName("DeleteRecoveryPoint").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteRecoveryPointRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteRecoveryPointRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns backup job details for the specified <code>BackupJobId</code>.
     * </p>
     *
     * @param describeBackupJobRequest
     * @return Result of the DescribeBackupJob operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws DependencyFailureException
     *         A dependent AWS service or resource returned an error to the AWS Backup service, and the action cannot be
     *         completed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.DescribeBackupJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/DescribeBackupJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeBackupJobResponse describeBackupJob(DescribeBackupJobRequest describeBackupJobRequest)
            throws ResourceNotFoundException, InvalidParameterValueException, MissingParameterValueException,
            ServiceUnavailableException, DependencyFailureException, AwsServiceException, SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeBackupJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeBackupJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeBackupJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBackupJob");

            return clientHandler.execute(new ClientExecutionParams<DescribeBackupJobRequest, DescribeBackupJobResponse>()
                    .withOperationName("DescribeBackupJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeBackupJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeBackupJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns metadata about a backup vault specified by its name.
     * </p>
     *
     * @param describeBackupVaultRequest
     * @return Result of the DescribeBackupVault operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.DescribeBackupVault
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/DescribeBackupVault" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeBackupVaultResponse describeBackupVault(DescribeBackupVaultRequest describeBackupVaultRequest)
            throws ResourceNotFoundException, InvalidParameterValueException, MissingParameterValueException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeBackupVaultResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeBackupVaultResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeBackupVaultRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBackupVault");

            return clientHandler.execute(new ClientExecutionParams<DescribeBackupVaultRequest, DescribeBackupVaultResponse>()
                    .withOperationName("DescribeBackupVault").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeBackupVaultRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeBackupVaultRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns metadata associated with creating a copy of a resource.
     * </p>
     *
     * @param describeCopyJobRequest
     * @return Result of the DescribeCopyJob operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.DescribeCopyJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/DescribeCopyJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeCopyJobResponse describeCopyJob(DescribeCopyJobRequest describeCopyJobRequest)
            throws ResourceNotFoundException, InvalidParameterValueException, MissingParameterValueException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeCopyJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeCopyJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeCopyJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeCopyJob");

            return clientHandler.execute(new ClientExecutionParams<DescribeCopyJobRequest, DescribeCopyJobResponse>()
                    .withOperationName("DescribeCopyJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeCopyJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeCopyJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the global settings of the AWS account, including whether it is opted in to cross-account backup.
     * </p>
     *
     * @param describeGlobalSettingsRequest
     * @return Result of the DescribeGlobalSettings operation returned by the service.
     * @throws InvalidRequestException
     *         Indicates that something is wrong with the input to the request. For example, a parameter is of the wrong
     *         type.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.DescribeGlobalSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/DescribeGlobalSettings" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeGlobalSettingsResponse describeGlobalSettings(DescribeGlobalSettingsRequest describeGlobalSettingsRequest)
            throws InvalidRequestException, ServiceUnavailableException, AwsServiceException, SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeGlobalSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeGlobalSettingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeGlobalSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeGlobalSettings");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeGlobalSettingsRequest, DescribeGlobalSettingsResponse>()
                            .withOperationName("DescribeGlobalSettings").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeGlobalSettingsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeGlobalSettingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about a saved resource, including the last time it was backed up, its Amazon Resource Name
     * (ARN), and the AWS service type of the saved resource.
     * </p>
     *
     * @param describeProtectedResourceRequest
     * @return Result of the DescribeProtectedResource operation returned by the service.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.DescribeProtectedResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/DescribeProtectedResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeProtectedResourceResponse describeProtectedResource(
            DescribeProtectedResourceRequest describeProtectedResourceRequest) throws MissingParameterValueException,
            InvalidParameterValueException, ServiceUnavailableException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeProtectedResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeProtectedResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeProtectedResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeProtectedResource");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeProtectedResourceRequest, DescribeProtectedResourceResponse>()
                            .withOperationName("DescribeProtectedResource").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeProtectedResourceRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeProtectedResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns metadata associated with a recovery point, including ID, status, encryption, and lifecycle.
     * </p>
     *
     * @param describeRecoveryPointRequest
     * @return Result of the DescribeRecoveryPoint operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.DescribeRecoveryPoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/DescribeRecoveryPoint" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeRecoveryPointResponse describeRecoveryPoint(DescribeRecoveryPointRequest describeRecoveryPointRequest)
            throws ResourceNotFoundException, InvalidParameterValueException, MissingParameterValueException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeRecoveryPointResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeRecoveryPointResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeRecoveryPointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeRecoveryPoint");

            return clientHandler.execute(new ClientExecutionParams<DescribeRecoveryPointRequest, DescribeRecoveryPointResponse>()
                    .withOperationName("DescribeRecoveryPoint").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeRecoveryPointRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeRecoveryPointRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the current service opt-in settings for the Region. If service-opt-in is enabled for a service, AWS
     * Backup tries to protect that service's resources in this Region, when the resource is included in an on-demand
     * backup or scheduled backup plan. Otherwise, AWS Backup does not try to protect that service's resources in this
     * Region, AWS Backup does not try to protect that service's resources in this Region.
     * </p>
     *
     * @param describeRegionSettingsRequest
     * @return Result of the DescribeRegionSettings operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.DescribeRegionSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/DescribeRegionSettings" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeRegionSettingsResponse describeRegionSettings(DescribeRegionSettingsRequest describeRegionSettingsRequest)
            throws ServiceUnavailableException, AwsServiceException, SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeRegionSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeRegionSettingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeRegionSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeRegionSettings");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeRegionSettingsRequest, DescribeRegionSettingsResponse>()
                            .withOperationName("DescribeRegionSettings").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeRegionSettingsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeRegionSettingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns metadata associated with a restore job that is specified by a job ID.
     * </p>
     *
     * @param describeRestoreJobRequest
     * @return Result of the DescribeRestoreJob operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws DependencyFailureException
     *         A dependent AWS service or resource returned an error to the AWS Backup service, and the action cannot be
     *         completed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.DescribeRestoreJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/DescribeRestoreJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeRestoreJobResponse describeRestoreJob(DescribeRestoreJobRequest describeRestoreJobRequest)
            throws ResourceNotFoundException, InvalidParameterValueException, MissingParameterValueException,
            ServiceUnavailableException, DependencyFailureException, AwsServiceException, SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeRestoreJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeRestoreJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeRestoreJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeRestoreJob");

            return clientHandler.execute(new ClientExecutionParams<DescribeRestoreJobRequest, DescribeRestoreJobResponse>()
                    .withOperationName("DescribeRestoreJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeRestoreJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeRestoreJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified continuous backup recovery point from AWS Backup and releases control of that continuous
     * backup to the source service, such as Amazon RDS. The source service will continue to create and retain
     * continuous backups using the lifecycle that you specified in your original backup plan.
     * </p>
     * <p>
     * Does not support snapshot backup recovery points.
     * </p>
     *
     * @param disassociateRecoveryPointRequest
     * @return Result of the DisassociateRecoveryPoint operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws InvalidResourceStateException
     *         AWS Backup is already performing an action on this recovery point. It can't perform the action you
     *         requested until the first action finishes. Try again later.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws InvalidRequestException
     *         Indicates that something is wrong with the input to the request. For example, a parameter is of the wrong
     *         type.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.DisassociateRecoveryPoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/DisassociateRecoveryPoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateRecoveryPointResponse disassociateRecoveryPoint(
            DisassociateRecoveryPointRequest disassociateRecoveryPointRequest) throws ResourceNotFoundException,
            InvalidParameterValueException, MissingParameterValueException, InvalidResourceStateException,
            ServiceUnavailableException, InvalidRequestException, AwsServiceException, SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateRecoveryPointResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateRecoveryPointResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateRecoveryPointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateRecoveryPoint");

            return clientHandler
                    .execute(new ClientExecutionParams<DisassociateRecoveryPointRequest, DisassociateRecoveryPointResponse>()
                            .withOperationName("DisassociateRecoveryPoint").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(disassociateRecoveryPointRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisassociateRecoveryPointRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the backup plan that is specified by the plan ID as a backup template.
     * </p>
     *
     * @param exportBackupPlanTemplateRequest
     * @return Result of the ExportBackupPlanTemplate operation returned by the service.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.ExportBackupPlanTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/ExportBackupPlanTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ExportBackupPlanTemplateResponse exportBackupPlanTemplate(
            ExportBackupPlanTemplateRequest exportBackupPlanTemplateRequest) throws InvalidParameterValueException,
            MissingParameterValueException, ServiceUnavailableException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ExportBackupPlanTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ExportBackupPlanTemplateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, exportBackupPlanTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportBackupPlanTemplate");

            return clientHandler
                    .execute(new ClientExecutionParams<ExportBackupPlanTemplateRequest, ExportBackupPlanTemplateResponse>()
                            .withOperationName("ExportBackupPlanTemplate").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(exportBackupPlanTemplateRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ExportBackupPlanTemplateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns <code>BackupPlan</code> details for the specified <code>BackupPlanId</code>. The details are the body of
     * a backup plan in JSON format, in addition to plan metadata.
     * </p>
     *
     * @param getBackupPlanRequest
     * @return Result of the GetBackupPlan operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.GetBackupPlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/GetBackupPlan" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetBackupPlanResponse getBackupPlan(GetBackupPlanRequest getBackupPlanRequest) throws ResourceNotFoundException,
            InvalidParameterValueException, MissingParameterValueException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetBackupPlanResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetBackupPlanResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBackupPlanRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBackupPlan");

            return clientHandler.execute(new ClientExecutionParams<GetBackupPlanRequest, GetBackupPlanResponse>()
                    .withOperationName("GetBackupPlan").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getBackupPlanRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetBackupPlanRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a valid JSON document specifying a backup plan or an error.
     * </p>
     *
     * @param getBackupPlanFromJsonRequest
     * @return Result of the GetBackupPlanFromJSON operation returned by the service.
     * @throws LimitExceededException
     *         A limit in the request has been exceeded; for example, a maximum number of items allowed in a request.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws InvalidRequestException
     *         Indicates that something is wrong with the input to the request. For example, a parameter is of the wrong
     *         type.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.GetBackupPlanFromJSON
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/GetBackupPlanFromJSON" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetBackupPlanFromJsonResponse getBackupPlanFromJSON(GetBackupPlanFromJsonRequest getBackupPlanFromJsonRequest)
            throws LimitExceededException, InvalidParameterValueException, MissingParameterValueException,
            ServiceUnavailableException, InvalidRequestException, AwsServiceException, SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetBackupPlanFromJsonResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetBackupPlanFromJsonResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBackupPlanFromJsonRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBackupPlanFromJSON");

            return clientHandler.execute(new ClientExecutionParams<GetBackupPlanFromJsonRequest, GetBackupPlanFromJsonResponse>()
                    .withOperationName("GetBackupPlanFromJSON").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getBackupPlanFromJsonRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetBackupPlanFromJsonRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the template specified by its <code>templateId</code> as a backup plan.
     * </p>
     *
     * @param getBackupPlanFromTemplateRequest
     * @return Result of the GetBackupPlanFromTemplate operation returned by the service.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.GetBackupPlanFromTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/GetBackupPlanFromTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetBackupPlanFromTemplateResponse getBackupPlanFromTemplate(
            GetBackupPlanFromTemplateRequest getBackupPlanFromTemplateRequest) throws InvalidParameterValueException,
            MissingParameterValueException, ServiceUnavailableException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetBackupPlanFromTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetBackupPlanFromTemplateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBackupPlanFromTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBackupPlanFromTemplate");

            return clientHandler
                    .execute(new ClientExecutionParams<GetBackupPlanFromTemplateRequest, GetBackupPlanFromTemplateResponse>()
                            .withOperationName("GetBackupPlanFromTemplate").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getBackupPlanFromTemplateRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetBackupPlanFromTemplateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns selection metadata and a document in JSON format that specifies a list of resources that are associated
     * with a backup plan.
     * </p>
     *
     * @param getBackupSelectionRequest
     * @return Result of the GetBackupSelection operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.GetBackupSelection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/GetBackupSelection" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetBackupSelectionResponse getBackupSelection(GetBackupSelectionRequest getBackupSelectionRequest)
            throws ResourceNotFoundException, InvalidParameterValueException, MissingParameterValueException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetBackupSelectionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetBackupSelectionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBackupSelectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBackupSelection");

            return clientHandler.execute(new ClientExecutionParams<GetBackupSelectionRequest, GetBackupSelectionResponse>()
                    .withOperationName("GetBackupSelection").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getBackupSelectionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetBackupSelectionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the access policy document that is associated with the named backup vault.
     * </p>
     *
     * @param getBackupVaultAccessPolicyRequest
     * @return Result of the GetBackupVaultAccessPolicy operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.GetBackupVaultAccessPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/GetBackupVaultAccessPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetBackupVaultAccessPolicyResponse getBackupVaultAccessPolicy(
            GetBackupVaultAccessPolicyRequest getBackupVaultAccessPolicyRequest) throws ResourceNotFoundException,
            InvalidParameterValueException, MissingParameterValueException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetBackupVaultAccessPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetBackupVaultAccessPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBackupVaultAccessPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBackupVaultAccessPolicy");

            return clientHandler
                    .execute(new ClientExecutionParams<GetBackupVaultAccessPolicyRequest, GetBackupVaultAccessPolicyResponse>()
                            .withOperationName("GetBackupVaultAccessPolicy").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getBackupVaultAccessPolicyRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetBackupVaultAccessPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns event notifications for the specified backup vault.
     * </p>
     *
     * @param getBackupVaultNotificationsRequest
     * @return Result of the GetBackupVaultNotifications operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.GetBackupVaultNotifications
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/GetBackupVaultNotifications"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetBackupVaultNotificationsResponse getBackupVaultNotifications(
            GetBackupVaultNotificationsRequest getBackupVaultNotificationsRequest) throws ResourceNotFoundException,
            InvalidParameterValueException, MissingParameterValueException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetBackupVaultNotificationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetBackupVaultNotificationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBackupVaultNotificationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBackupVaultNotifications");

            return clientHandler
                    .execute(new ClientExecutionParams<GetBackupVaultNotificationsRequest, GetBackupVaultNotificationsResponse>()
                            .withOperationName("GetBackupVaultNotifications").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getBackupVaultNotificationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetBackupVaultNotificationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a set of metadata key-value pairs that were used to create the backup.
     * </p>
     *
     * @param getRecoveryPointRestoreMetadataRequest
     * @return Result of the GetRecoveryPointRestoreMetadata operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.GetRecoveryPointRestoreMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/GetRecoveryPointRestoreMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetRecoveryPointRestoreMetadataResponse getRecoveryPointRestoreMetadata(
            GetRecoveryPointRestoreMetadataRequest getRecoveryPointRestoreMetadataRequest) throws ResourceNotFoundException,
            InvalidParameterValueException, MissingParameterValueException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetRecoveryPointRestoreMetadataResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetRecoveryPointRestoreMetadataResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getRecoveryPointRestoreMetadataRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRecoveryPointRestoreMetadata");

            return clientHandler
                    .execute(new ClientExecutionParams<GetRecoveryPointRestoreMetadataRequest, GetRecoveryPointRestoreMetadataResponse>()
                            .withOperationName("GetRecoveryPointRestoreMetadata").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getRecoveryPointRestoreMetadataRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetRecoveryPointRestoreMetadataRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the AWS resource types supported by AWS Backup.
     * </p>
     *
     * @param getSupportedResourceTypesRequest
     * @return Result of the GetSupportedResourceTypes operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.GetSupportedResourceTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/GetSupportedResourceTypes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetSupportedResourceTypesResponse getSupportedResourceTypes(
            GetSupportedResourceTypesRequest getSupportedResourceTypesRequest) throws ServiceUnavailableException,
            AwsServiceException, SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSupportedResourceTypesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetSupportedResourceTypesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSupportedResourceTypesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSupportedResourceTypes");

            return clientHandler
                    .execute(new ClientExecutionParams<GetSupportedResourceTypesRequest, GetSupportedResourceTypesResponse>()
                            .withOperationName("GetSupportedResourceTypes").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getSupportedResourceTypesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetSupportedResourceTypesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of existing backup jobs for an authenticated account for the last 30 days. For a longer period of
     * time, consider using these <a
     * href="https://docs.aws.amazon.com/aws-backup/latest/devguide/monitoring.html">monitoring tools</a>.
     * </p>
     *
     * @param listBackupJobsRequest
     * @return Result of the ListBackupJobs operation returned by the service.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.ListBackupJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/ListBackupJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListBackupJobsResponse listBackupJobs(ListBackupJobsRequest listBackupJobsRequest)
            throws InvalidParameterValueException, ServiceUnavailableException, AwsServiceException, SdkClientException,
            BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListBackupJobsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListBackupJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listBackupJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListBackupJobs");

            return clientHandler.execute(new ClientExecutionParams<ListBackupJobsRequest, ListBackupJobsResponse>()
                    .withOperationName("ListBackupJobs").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listBackupJobsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListBackupJobsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of existing backup jobs for an authenticated account for the last 30 days. For a longer period of
     * time, consider using these <a
     * href="https://docs.aws.amazon.com/aws-backup/latest/devguide/monitoring.html">monitoring tools</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listBackupJobs(software.amazon.awssdk.services.backup.model.ListBackupJobsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backup.paginators.ListBackupJobsIterable responses = client.listBackupJobsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.backup.paginators.ListBackupJobsIterable responses = client.listBackupJobsPaginator(request);
     *     for (software.amazon.awssdk.services.backup.model.ListBackupJobsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backup.paginators.ListBackupJobsIterable responses = client.listBackupJobsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listBackupJobs(software.amazon.awssdk.services.backup.model.ListBackupJobsRequest)} operation.</b>
     * </p>
     *
     * @param listBackupJobsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.ListBackupJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/ListBackupJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListBackupJobsIterable listBackupJobsPaginator(ListBackupJobsRequest listBackupJobsRequest)
            throws InvalidParameterValueException, ServiceUnavailableException, AwsServiceException, SdkClientException,
            BackupException {
        return new ListBackupJobsIterable(this, applyPaginatorUserAgent(listBackupJobsRequest));
    }

    /**
     * <p>
     * Returns metadata of your saved backup plan templates, including the template ID, name, and the creation and
     * deletion dates.
     * </p>
     *
     * @param listBackupPlanTemplatesRequest
     * @return Result of the ListBackupPlanTemplates operation returned by the service.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.ListBackupPlanTemplates
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/ListBackupPlanTemplates"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListBackupPlanTemplatesResponse listBackupPlanTemplates(ListBackupPlanTemplatesRequest listBackupPlanTemplatesRequest)
            throws InvalidParameterValueException, MissingParameterValueException, ServiceUnavailableException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListBackupPlanTemplatesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListBackupPlanTemplatesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listBackupPlanTemplatesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListBackupPlanTemplates");

            return clientHandler
                    .execute(new ClientExecutionParams<ListBackupPlanTemplatesRequest, ListBackupPlanTemplatesResponse>()
                            .withOperationName("ListBackupPlanTemplates").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listBackupPlanTemplatesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListBackupPlanTemplatesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns metadata of your saved backup plan templates, including the template ID, name, and the creation and
     * deletion dates.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listBackupPlanTemplates(software.amazon.awssdk.services.backup.model.ListBackupPlanTemplatesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backup.paginators.ListBackupPlanTemplatesIterable responses = client.listBackupPlanTemplatesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.backup.paginators.ListBackupPlanTemplatesIterable responses = client
     *             .listBackupPlanTemplatesPaginator(request);
     *     for (software.amazon.awssdk.services.backup.model.ListBackupPlanTemplatesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backup.paginators.ListBackupPlanTemplatesIterable responses = client.listBackupPlanTemplatesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listBackupPlanTemplates(software.amazon.awssdk.services.backup.model.ListBackupPlanTemplatesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listBackupPlanTemplatesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.ListBackupPlanTemplates
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/ListBackupPlanTemplates"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListBackupPlanTemplatesIterable listBackupPlanTemplatesPaginator(
            ListBackupPlanTemplatesRequest listBackupPlanTemplatesRequest) throws InvalidParameterValueException,
            MissingParameterValueException, ServiceUnavailableException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, BackupException {
        return new ListBackupPlanTemplatesIterable(this, applyPaginatorUserAgent(listBackupPlanTemplatesRequest));
    }

    /**
     * <p>
     * Returns version metadata of your backup plans, including Amazon Resource Names (ARNs), backup plan IDs, creation
     * and deletion dates, plan names, and version IDs.
     * </p>
     *
     * @param listBackupPlanVersionsRequest
     * @return Result of the ListBackupPlanVersions operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.ListBackupPlanVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/ListBackupPlanVersions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListBackupPlanVersionsResponse listBackupPlanVersions(ListBackupPlanVersionsRequest listBackupPlanVersionsRequest)
            throws ResourceNotFoundException, InvalidParameterValueException, MissingParameterValueException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListBackupPlanVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListBackupPlanVersionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listBackupPlanVersionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListBackupPlanVersions");

            return clientHandler
                    .execute(new ClientExecutionParams<ListBackupPlanVersionsRequest, ListBackupPlanVersionsResponse>()
                            .withOperationName("ListBackupPlanVersions").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listBackupPlanVersionsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListBackupPlanVersionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns version metadata of your backup plans, including Amazon Resource Names (ARNs), backup plan IDs, creation
     * and deletion dates, plan names, and version IDs.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listBackupPlanVersions(software.amazon.awssdk.services.backup.model.ListBackupPlanVersionsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backup.paginators.ListBackupPlanVersionsIterable responses = client.listBackupPlanVersionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.backup.paginators.ListBackupPlanVersionsIterable responses = client
     *             .listBackupPlanVersionsPaginator(request);
     *     for (software.amazon.awssdk.services.backup.model.ListBackupPlanVersionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backup.paginators.ListBackupPlanVersionsIterable responses = client.listBackupPlanVersionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listBackupPlanVersions(software.amazon.awssdk.services.backup.model.ListBackupPlanVersionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listBackupPlanVersionsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.ListBackupPlanVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/ListBackupPlanVersions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListBackupPlanVersionsIterable listBackupPlanVersionsPaginator(
            ListBackupPlanVersionsRequest listBackupPlanVersionsRequest) throws ResourceNotFoundException,
            InvalidParameterValueException, MissingParameterValueException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, BackupException {
        return new ListBackupPlanVersionsIterable(this, applyPaginatorUserAgent(listBackupPlanVersionsRequest));
    }

    /**
     * <p>
     * Returns a list of existing backup plans for an authenticated account. The list is populated only if the advanced
     * option is set for the backup plan. The list contains information such as Amazon Resource Names (ARNs), plan IDs,
     * creation and deletion dates, version IDs, plan names, and creator request IDs.
     * </p>
     *
     * @param listBackupPlansRequest
     * @return Result of the ListBackupPlans operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.ListBackupPlans
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/ListBackupPlans" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListBackupPlansResponse listBackupPlans(ListBackupPlansRequest listBackupPlansRequest)
            throws ResourceNotFoundException, InvalidParameterValueException, MissingParameterValueException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListBackupPlansResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListBackupPlansResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listBackupPlansRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListBackupPlans");

            return clientHandler.execute(new ClientExecutionParams<ListBackupPlansRequest, ListBackupPlansResponse>()
                    .withOperationName("ListBackupPlans").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listBackupPlansRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListBackupPlansRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of existing backup plans for an authenticated account. The list is populated only if the advanced
     * option is set for the backup plan. The list contains information such as Amazon Resource Names (ARNs), plan IDs,
     * creation and deletion dates, version IDs, plan names, and creator request IDs.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listBackupPlans(software.amazon.awssdk.services.backup.model.ListBackupPlansRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backup.paginators.ListBackupPlansIterable responses = client.listBackupPlansPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.backup.paginators.ListBackupPlansIterable responses = client
     *             .listBackupPlansPaginator(request);
     *     for (software.amazon.awssdk.services.backup.model.ListBackupPlansResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backup.paginators.ListBackupPlansIterable responses = client.listBackupPlansPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listBackupPlans(software.amazon.awssdk.services.backup.model.ListBackupPlansRequest)} operation.</b>
     * </p>
     *
     * @param listBackupPlansRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.ListBackupPlans
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/ListBackupPlans" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListBackupPlansIterable listBackupPlansPaginator(ListBackupPlansRequest listBackupPlansRequest)
            throws ResourceNotFoundException, InvalidParameterValueException, MissingParameterValueException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, BackupException {
        return new ListBackupPlansIterable(this, applyPaginatorUserAgent(listBackupPlansRequest));
    }

    /**
     * <p>
     * Returns an array containing metadata of the resources associated with the target backup plan.
     * </p>
     *
     * @param listBackupSelectionsRequest
     * @return Result of the ListBackupSelections operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.ListBackupSelections
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/ListBackupSelections" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListBackupSelectionsResponse listBackupSelections(ListBackupSelectionsRequest listBackupSelectionsRequest)
            throws ResourceNotFoundException, InvalidParameterValueException, MissingParameterValueException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListBackupSelectionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListBackupSelectionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listBackupSelectionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListBackupSelections");

            return clientHandler.execute(new ClientExecutionParams<ListBackupSelectionsRequest, ListBackupSelectionsResponse>()
                    .withOperationName("ListBackupSelections").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listBackupSelectionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListBackupSelectionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an array containing metadata of the resources associated with the target backup plan.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listBackupSelections(software.amazon.awssdk.services.backup.model.ListBackupSelectionsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backup.paginators.ListBackupSelectionsIterable responses = client.listBackupSelectionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.backup.paginators.ListBackupSelectionsIterable responses = client
     *             .listBackupSelectionsPaginator(request);
     *     for (software.amazon.awssdk.services.backup.model.ListBackupSelectionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backup.paginators.ListBackupSelectionsIterable responses = client.listBackupSelectionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listBackupSelections(software.amazon.awssdk.services.backup.model.ListBackupSelectionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listBackupSelectionsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.ListBackupSelections
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/ListBackupSelections" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListBackupSelectionsIterable listBackupSelectionsPaginator(ListBackupSelectionsRequest listBackupSelectionsRequest)
            throws ResourceNotFoundException, InvalidParameterValueException, MissingParameterValueException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, BackupException {
        return new ListBackupSelectionsIterable(this, applyPaginatorUserAgent(listBackupSelectionsRequest));
    }

    /**
     * <p>
     * Returns a list of recovery point storage containers along with information about them.
     * </p>
     *
     * @param listBackupVaultsRequest
     * @return Result of the ListBackupVaults operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.ListBackupVaults
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/ListBackupVaults" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListBackupVaultsResponse listBackupVaults(ListBackupVaultsRequest listBackupVaultsRequest)
            throws ResourceNotFoundException, InvalidParameterValueException, MissingParameterValueException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListBackupVaultsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListBackupVaultsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listBackupVaultsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListBackupVaults");

            return clientHandler.execute(new ClientExecutionParams<ListBackupVaultsRequest, ListBackupVaultsResponse>()
                    .withOperationName("ListBackupVaults").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listBackupVaultsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListBackupVaultsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of recovery point storage containers along with information about them.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listBackupVaults(software.amazon.awssdk.services.backup.model.ListBackupVaultsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backup.paginators.ListBackupVaultsIterable responses = client.listBackupVaultsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.backup.paginators.ListBackupVaultsIterable responses = client
     *             .listBackupVaultsPaginator(request);
     *     for (software.amazon.awssdk.services.backup.model.ListBackupVaultsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backup.paginators.ListBackupVaultsIterable responses = client.listBackupVaultsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listBackupVaults(software.amazon.awssdk.services.backup.model.ListBackupVaultsRequest)} operation.</b>
     * </p>
     *
     * @param listBackupVaultsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.ListBackupVaults
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/ListBackupVaults" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListBackupVaultsIterable listBackupVaultsPaginator(ListBackupVaultsRequest listBackupVaultsRequest)
            throws ResourceNotFoundException, InvalidParameterValueException, MissingParameterValueException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, BackupException {
        return new ListBackupVaultsIterable(this, applyPaginatorUserAgent(listBackupVaultsRequest));
    }

    /**
     * <p>
     * Returns metadata about your copy jobs.
     * </p>
     *
     * @param listCopyJobsRequest
     * @return Result of the ListCopyJobs operation returned by the service.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.ListCopyJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/ListCopyJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListCopyJobsResponse listCopyJobs(ListCopyJobsRequest listCopyJobsRequest) throws InvalidParameterValueException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListCopyJobsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListCopyJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCopyJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCopyJobs");

            return clientHandler.execute(new ClientExecutionParams<ListCopyJobsRequest, ListCopyJobsResponse>()
                    .withOperationName("ListCopyJobs").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listCopyJobsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListCopyJobsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns metadata about your copy jobs.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listCopyJobs(software.amazon.awssdk.services.backup.model.ListCopyJobsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backup.paginators.ListCopyJobsIterable responses = client.listCopyJobsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.backup.paginators.ListCopyJobsIterable responses = client.listCopyJobsPaginator(request);
     *     for (software.amazon.awssdk.services.backup.model.ListCopyJobsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backup.paginators.ListCopyJobsIterable responses = client.listCopyJobsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listCopyJobs(software.amazon.awssdk.services.backup.model.ListCopyJobsRequest)} operation.</b>
     * </p>
     *
     * @param listCopyJobsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.ListCopyJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/ListCopyJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListCopyJobsIterable listCopyJobsPaginator(ListCopyJobsRequest listCopyJobsRequest)
            throws InvalidParameterValueException, ServiceUnavailableException, AwsServiceException, SdkClientException,
            BackupException {
        return new ListCopyJobsIterable(this, applyPaginatorUserAgent(listCopyJobsRequest));
    }

    /**
     * <p>
     * Returns an array of resources successfully backed up by AWS Backup, including the time the resource was saved, an
     * Amazon Resource Name (ARN) of the resource, and a resource type.
     * </p>
     *
     * @param listProtectedResourcesRequest
     * @return Result of the ListProtectedResources operation returned by the service.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.ListProtectedResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/ListProtectedResources" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListProtectedResourcesResponse listProtectedResources(ListProtectedResourcesRequest listProtectedResourcesRequest)
            throws InvalidParameterValueException, ServiceUnavailableException, AwsServiceException, SdkClientException,
            BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListProtectedResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListProtectedResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProtectedResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProtectedResources");

            return clientHandler
                    .execute(new ClientExecutionParams<ListProtectedResourcesRequest, ListProtectedResourcesResponse>()
                            .withOperationName("ListProtectedResources").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listProtectedResourcesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListProtectedResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an array of resources successfully backed up by AWS Backup, including the time the resource was saved, an
     * Amazon Resource Name (ARN) of the resource, and a resource type.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listProtectedResources(software.amazon.awssdk.services.backup.model.ListProtectedResourcesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backup.paginators.ListProtectedResourcesIterable responses = client.listProtectedResourcesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.backup.paginators.ListProtectedResourcesIterable responses = client
     *             .listProtectedResourcesPaginator(request);
     *     for (software.amazon.awssdk.services.backup.model.ListProtectedResourcesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backup.paginators.ListProtectedResourcesIterable responses = client.listProtectedResourcesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listProtectedResources(software.amazon.awssdk.services.backup.model.ListProtectedResourcesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listProtectedResourcesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.ListProtectedResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/ListProtectedResources" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListProtectedResourcesIterable listProtectedResourcesPaginator(
            ListProtectedResourcesRequest listProtectedResourcesRequest) throws InvalidParameterValueException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, BackupException {
        return new ListProtectedResourcesIterable(this, applyPaginatorUserAgent(listProtectedResourcesRequest));
    }

    /**
     * <p>
     * Returns detailed information about the recovery points stored in a backup vault.
     * </p>
     *
     * @param listRecoveryPointsByBackupVaultRequest
     * @return Result of the ListRecoveryPointsByBackupVault operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.ListRecoveryPointsByBackupVault
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/ListRecoveryPointsByBackupVault"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRecoveryPointsByBackupVaultResponse listRecoveryPointsByBackupVault(
            ListRecoveryPointsByBackupVaultRequest listRecoveryPointsByBackupVaultRequest) throws ResourceNotFoundException,
            InvalidParameterValueException, MissingParameterValueException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRecoveryPointsByBackupVaultResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListRecoveryPointsByBackupVaultResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listRecoveryPointsByBackupVaultRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRecoveryPointsByBackupVault");

            return clientHandler
                    .execute(new ClientExecutionParams<ListRecoveryPointsByBackupVaultRequest, ListRecoveryPointsByBackupVaultResponse>()
                            .withOperationName("ListRecoveryPointsByBackupVault").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listRecoveryPointsByBackupVaultRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListRecoveryPointsByBackupVaultRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns detailed information about the recovery points stored in a backup vault.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listRecoveryPointsByBackupVault(software.amazon.awssdk.services.backup.model.ListRecoveryPointsByBackupVaultRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backup.paginators.ListRecoveryPointsByBackupVaultIterable responses = client.listRecoveryPointsByBackupVaultPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.backup.paginators.ListRecoveryPointsByBackupVaultIterable responses = client
     *             .listRecoveryPointsByBackupVaultPaginator(request);
     *     for (software.amazon.awssdk.services.backup.model.ListRecoveryPointsByBackupVaultResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backup.paginators.ListRecoveryPointsByBackupVaultIterable responses = client.listRecoveryPointsByBackupVaultPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRecoveryPointsByBackupVault(software.amazon.awssdk.services.backup.model.ListRecoveryPointsByBackupVaultRequest)}
     * operation.</b>
     * </p>
     *
     * @param listRecoveryPointsByBackupVaultRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.ListRecoveryPointsByBackupVault
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/ListRecoveryPointsByBackupVault"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRecoveryPointsByBackupVaultIterable listRecoveryPointsByBackupVaultPaginator(
            ListRecoveryPointsByBackupVaultRequest listRecoveryPointsByBackupVaultRequest) throws ResourceNotFoundException,
            InvalidParameterValueException, MissingParameterValueException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, BackupException {
        return new ListRecoveryPointsByBackupVaultIterable(this, applyPaginatorUserAgent(listRecoveryPointsByBackupVaultRequest));
    }

    /**
     * <p>
     * Returns detailed information about recovery points of the type specified by a resource Amazon Resource Name
     * (ARN).
     * </p>
     *
     * @param listRecoveryPointsByResourceRequest
     * @return Result of the ListRecoveryPointsByResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.ListRecoveryPointsByResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/ListRecoveryPointsByResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRecoveryPointsByResourceResponse listRecoveryPointsByResource(
            ListRecoveryPointsByResourceRequest listRecoveryPointsByResourceRequest) throws ResourceNotFoundException,
            InvalidParameterValueException, MissingParameterValueException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRecoveryPointsByResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListRecoveryPointsByResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRecoveryPointsByResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRecoveryPointsByResource");

            return clientHandler
                    .execute(new ClientExecutionParams<ListRecoveryPointsByResourceRequest, ListRecoveryPointsByResourceResponse>()
                            .withOperationName("ListRecoveryPointsByResource").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listRecoveryPointsByResourceRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListRecoveryPointsByResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns detailed information about recovery points of the type specified by a resource Amazon Resource Name
     * (ARN).
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listRecoveryPointsByResource(software.amazon.awssdk.services.backup.model.ListRecoveryPointsByResourceRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backup.paginators.ListRecoveryPointsByResourceIterable responses = client.listRecoveryPointsByResourcePaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.backup.paginators.ListRecoveryPointsByResourceIterable responses = client
     *             .listRecoveryPointsByResourcePaginator(request);
     *     for (software.amazon.awssdk.services.backup.model.ListRecoveryPointsByResourceResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backup.paginators.ListRecoveryPointsByResourceIterable responses = client.listRecoveryPointsByResourcePaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRecoveryPointsByResource(software.amazon.awssdk.services.backup.model.ListRecoveryPointsByResourceRequest)}
     * operation.</b>
     * </p>
     *
     * @param listRecoveryPointsByResourceRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.ListRecoveryPointsByResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/ListRecoveryPointsByResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRecoveryPointsByResourceIterable listRecoveryPointsByResourcePaginator(
            ListRecoveryPointsByResourceRequest listRecoveryPointsByResourceRequest) throws ResourceNotFoundException,
            InvalidParameterValueException, MissingParameterValueException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, BackupException {
        return new ListRecoveryPointsByResourceIterable(this, applyPaginatorUserAgent(listRecoveryPointsByResourceRequest));
    }

    /**
     * <p>
     * Returns a list of jobs that AWS Backup initiated to restore a saved resource, including metadata about the
     * recovery process.
     * </p>
     *
     * @param listRestoreJobsRequest
     * @return Result of the ListRestoreJobs operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.ListRestoreJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/ListRestoreJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListRestoreJobsResponse listRestoreJobs(ListRestoreJobsRequest listRestoreJobsRequest)
            throws ResourceNotFoundException, InvalidParameterValueException, MissingParameterValueException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRestoreJobsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListRestoreJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRestoreJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRestoreJobs");

            return clientHandler.execute(new ClientExecutionParams<ListRestoreJobsRequest, ListRestoreJobsResponse>()
                    .withOperationName("ListRestoreJobs").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listRestoreJobsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListRestoreJobsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of jobs that AWS Backup initiated to restore a saved resource, including metadata about the
     * recovery process.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listRestoreJobs(software.amazon.awssdk.services.backup.model.ListRestoreJobsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backup.paginators.ListRestoreJobsIterable responses = client.listRestoreJobsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.backup.paginators.ListRestoreJobsIterable responses = client
     *             .listRestoreJobsPaginator(request);
     *     for (software.amazon.awssdk.services.backup.model.ListRestoreJobsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backup.paginators.ListRestoreJobsIterable responses = client.listRestoreJobsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRestoreJobs(software.amazon.awssdk.services.backup.model.ListRestoreJobsRequest)} operation.</b>
     * </p>
     *
     * @param listRestoreJobsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.ListRestoreJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/ListRestoreJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListRestoreJobsIterable listRestoreJobsPaginator(ListRestoreJobsRequest listRestoreJobsRequest)
            throws ResourceNotFoundException, InvalidParameterValueException, MissingParameterValueException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, BackupException {
        return new ListRestoreJobsIterable(this, applyPaginatorUserAgent(listRestoreJobsRequest));
    }

    /**
     * <p>
     * Returns a list of key-value pairs assigned to a target recovery point, backup plan, or backup vault.
     * </p>
     * <note>
     * <p>
     * <code>ListTags</code> are currently only supported with Amazon EFS backups.
     * </p>
     * </note>
     *
     * @param listTagsRequest
     * @return Result of the ListTags operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.ListTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/ListTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsResponse listTags(ListTagsRequest listTagsRequest) throws ResourceNotFoundException,
            InvalidParameterValueException, MissingParameterValueException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTags");

            return clientHandler.execute(new ClientExecutionParams<ListTagsRequest, ListTagsResponse>()
                    .withOperationName("ListTags").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new ListTagsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of key-value pairs assigned to a target recovery point, backup plan, or backup vault.
     * </p>
     * <note>
     * <p>
     * <code>ListTags</code> are currently only supported with Amazon EFS backups.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of {@link #listTags(software.amazon.awssdk.services.backup.model.ListTagsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backup.paginators.ListTagsIterable responses = client.listTagsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.backup.paginators.ListTagsIterable responses = client.listTagsPaginator(request);
     *     for (software.amazon.awssdk.services.backup.model.ListTagsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backup.paginators.ListTagsIterable responses = client.listTagsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTags(software.amazon.awssdk.services.backup.model.ListTagsRequest)} operation.</b>
     * </p>
     *
     * @param listTagsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.ListTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/ListTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsIterable listTagsPaginator(ListTagsRequest listTagsRequest) throws ResourceNotFoundException,
            InvalidParameterValueException, MissingParameterValueException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, BackupException {
        return new ListTagsIterable(this, applyPaginatorUserAgent(listTagsRequest));
    }

    /**
     * <p>
     * Sets a resource-based policy that is used to manage access permissions on the target backup vault. Requires a
     * backup vault name and an access policy document in JSON format.
     * </p>
     *
     * @param putBackupVaultAccessPolicyRequest
     * @return Result of the PutBackupVaultAccessPolicy operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.PutBackupVaultAccessPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/PutBackupVaultAccessPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutBackupVaultAccessPolicyResponse putBackupVaultAccessPolicy(
            PutBackupVaultAccessPolicyRequest putBackupVaultAccessPolicyRequest) throws ResourceNotFoundException,
            InvalidParameterValueException, MissingParameterValueException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutBackupVaultAccessPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutBackupVaultAccessPolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putBackupVaultAccessPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutBackupVaultAccessPolicy");

            return clientHandler
                    .execute(new ClientExecutionParams<PutBackupVaultAccessPolicyRequest, PutBackupVaultAccessPolicyResponse>()
                            .withOperationName("PutBackupVaultAccessPolicy").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(putBackupVaultAccessPolicyRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutBackupVaultAccessPolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Turns on notifications on a backup vault for the specified topic and events.
     * </p>
     *
     * @param putBackupVaultNotificationsRequest
     * @return Result of the PutBackupVaultNotifications operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.PutBackupVaultNotifications
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/PutBackupVaultNotifications"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutBackupVaultNotificationsResponse putBackupVaultNotifications(
            PutBackupVaultNotificationsRequest putBackupVaultNotificationsRequest) throws ResourceNotFoundException,
            InvalidParameterValueException, MissingParameterValueException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutBackupVaultNotificationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutBackupVaultNotificationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putBackupVaultNotificationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutBackupVaultNotifications");

            return clientHandler
                    .execute(new ClientExecutionParams<PutBackupVaultNotificationsRequest, PutBackupVaultNotificationsResponse>()
                            .withOperationName("PutBackupVaultNotifications").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(putBackupVaultNotificationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutBackupVaultNotificationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts an on-demand backup job for the specified resource.
     * </p>
     *
     * @param startBackupJobRequest
     * @return Result of the StartBackupJob operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws InvalidRequestException
     *         Indicates that something is wrong with the input to the request. For example, a parameter is of the wrong
     *         type.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws LimitExceededException
     *         A limit in the request has been exceeded; for example, a maximum number of items allowed in a request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.StartBackupJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/StartBackupJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartBackupJobResponse startBackupJob(StartBackupJobRequest startBackupJobRequest) throws ResourceNotFoundException,
            InvalidParameterValueException, MissingParameterValueException, InvalidRequestException, ServiceUnavailableException,
            LimitExceededException, AwsServiceException, SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartBackupJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartBackupJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startBackupJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartBackupJob");

            return clientHandler.execute(new ClientExecutionParams<StartBackupJobRequest, StartBackupJobResponse>()
                    .withOperationName("StartBackupJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startBackupJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartBackupJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts a job to create a one-time copy of the specified resource.
     * </p>
     * <p>
     * Does not support continuous backups.
     * </p>
     *
     * @param startCopyJobRequest
     * @return Result of the StartCopyJob operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws LimitExceededException
     *         A limit in the request has been exceeded; for example, a maximum number of items allowed in a request.
     * @throws InvalidRequestException
     *         Indicates that something is wrong with the input to the request. For example, a parameter is of the wrong
     *         type.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.StartCopyJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/StartCopyJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartCopyJobResponse startCopyJob(StartCopyJobRequest startCopyJobRequest) throws ResourceNotFoundException,
            InvalidParameterValueException, MissingParameterValueException, ServiceUnavailableException, LimitExceededException,
            InvalidRequestException, AwsServiceException, SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartCopyJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartCopyJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startCopyJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartCopyJob");

            return clientHandler.execute(new ClientExecutionParams<StartCopyJobRequest, StartCopyJobResponse>()
                    .withOperationName("StartCopyJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startCopyJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartCopyJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Recovers the saved resource identified by an Amazon Resource Name (ARN).
     * </p>
     *
     * @param startRestoreJobRequest
     * @return Result of the StartRestoreJob operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.StartRestoreJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/StartRestoreJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartRestoreJobResponse startRestoreJob(StartRestoreJobRequest startRestoreJobRequest)
            throws ResourceNotFoundException, InvalidParameterValueException, MissingParameterValueException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartRestoreJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartRestoreJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startRestoreJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartRestoreJob");

            return clientHandler.execute(new ClientExecutionParams<StartRestoreJobRequest, StartRestoreJobResponse>()
                    .withOperationName("StartRestoreJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startRestoreJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartRestoreJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Attempts to cancel a job to create a one-time backup of a resource.
     * </p>
     *
     * @param stopBackupJobRequest
     * @return Result of the StopBackupJob operation returned by the service.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws InvalidRequestException
     *         Indicates that something is wrong with the input to the request. For example, a parameter is of the wrong
     *         type.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.StopBackupJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/StopBackupJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StopBackupJobResponse stopBackupJob(StopBackupJobRequest stopBackupJobRequest) throws MissingParameterValueException,
            ResourceNotFoundException, InvalidParameterValueException, InvalidRequestException, ServiceUnavailableException,
            AwsServiceException, SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopBackupJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopBackupJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopBackupJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopBackupJob");

            return clientHandler.execute(new ClientExecutionParams<StopBackupJobRequest, StopBackupJobResponse>()
                    .withOperationName("StopBackupJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(stopBackupJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StopBackupJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Assigns a set of key-value pairs to a recovery point, backup plan, or backup vault identified by an Amazon
     * Resource Name (ARN).
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws LimitExceededException
     *         A limit in the request has been exceeded; for example, a maximum number of items allowed in a request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            InvalidParameterValueException, MissingParameterValueException, ServiceUnavailableException, LimitExceededException,
            AwsServiceException, SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes a set of key-value pairs from a recovery point, backup plan, or backup vault identified by an Amazon
     * Resource Name (ARN)
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            InvalidParameterValueException, MissingParameterValueException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an existing backup plan identified by its <code>backupPlanId</code> with the input document in JSON
     * format. The new version is uniquely identified by a <code>VersionId</code>.
     * </p>
     *
     * @param updateBackupPlanRequest
     * @return Result of the UpdateBackupPlan operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.UpdateBackupPlan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/UpdateBackupPlan" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateBackupPlanResponse updateBackupPlan(UpdateBackupPlanRequest updateBackupPlanRequest)
            throws ResourceNotFoundException, InvalidParameterValueException, MissingParameterValueException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateBackupPlanResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateBackupPlanResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateBackupPlanRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateBackupPlan");

            return clientHandler.execute(new ClientExecutionParams<UpdateBackupPlanRequest, UpdateBackupPlanResponse>()
                    .withOperationName("UpdateBackupPlan").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateBackupPlanRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateBackupPlanRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the current global settings for the AWS account. Use the <code>DescribeGlobalSettings</code> API to
     * determine the current settings.
     * </p>
     *
     * @param updateGlobalSettingsRequest
     * @return Result of the UpdateGlobalSettings operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws InvalidRequestException
     *         Indicates that something is wrong with the input to the request. For example, a parameter is of the wrong
     *         type.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.UpdateGlobalSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/UpdateGlobalSettings" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateGlobalSettingsResponse updateGlobalSettings(UpdateGlobalSettingsRequest updateGlobalSettingsRequest)
            throws ServiceUnavailableException, MissingParameterValueException, InvalidParameterValueException,
            InvalidRequestException, AwsServiceException, SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateGlobalSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateGlobalSettingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateGlobalSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateGlobalSettings");

            return clientHandler.execute(new ClientExecutionParams<UpdateGlobalSettingsRequest, UpdateGlobalSettingsResponse>()
                    .withOperationName("UpdateGlobalSettings").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateGlobalSettingsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateGlobalSettingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Sets the transition lifecycle of a recovery point.
     * </p>
     * <p>
     * The lifecycle defines when a protected resource is transitioned to cold storage and when it expires. AWS Backup
     * transitions and expires backups automatically according to the lifecycle that you define.
     * </p>
     * <p>
     * Backups transitioned to cold storage must be stored in cold storage for a minimum of 90 days. Therefore, the
     * “expire after days” setting must be 90 days greater than the “transition to cold after days” setting. The
     * “transition to cold after days” setting cannot be changed after a backup has been transitioned to cold.
     * </p>
     * <p>
     * Only Amazon EFS file system backups can be transitioned to cold storage.
     * </p>
     * <p>
     * Does not support continuous backups.
     * </p>
     *
     * @param updateRecoveryPointLifecycleRequest
     * @return Result of the UpdateRecoveryPointLifecycle operation returned by the service.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.UpdateRecoveryPointLifecycle
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/UpdateRecoveryPointLifecycle"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateRecoveryPointLifecycleResponse updateRecoveryPointLifecycle(
            UpdateRecoveryPointLifecycleRequest updateRecoveryPointLifecycleRequest) throws ResourceNotFoundException,
            InvalidParameterValueException, MissingParameterValueException, ServiceUnavailableException, AwsServiceException,
            SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateRecoveryPointLifecycleResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateRecoveryPointLifecycleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRecoveryPointLifecycleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRecoveryPointLifecycle");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateRecoveryPointLifecycleRequest, UpdateRecoveryPointLifecycleResponse>()
                            .withOperationName("UpdateRecoveryPointLifecycle").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateRecoveryPointLifecycleRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateRecoveryPointLifecycleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the current service opt-in settings for the Region. If service-opt-in is enabled for a service, AWS
     * Backup tries to protect that service's resources in this Region, when the resource is included in an on-demand
     * backup or scheduled backup plan. Otherwise, AWS Backup does not try to protect that service's resources in this
     * Region. Use the <code>DescribeRegionSettings</code> API to determine the resource types that are supported.
     * </p>
     *
     * @param updateRegionSettingsRequest
     * @return Result of the UpdateRegionSettings operation returned by the service.
     * @throws ServiceUnavailableException
     *         The request failed due to a temporary failure of the server.
     * @throws MissingParameterValueException
     *         Indicates that a required parameter is missing.
     * @throws InvalidParameterValueException
     *         Indicates that something is wrong with a parameter's value. For example, the value is out of range.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupClient.UpdateRegionSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-2018-11-15/UpdateRegionSettings" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateRegionSettingsResponse updateRegionSettings(UpdateRegionSettingsRequest updateRegionSettingsRequest)
            throws ServiceUnavailableException, MissingParameterValueException, InvalidParameterValueException,
            AwsServiceException, SdkClientException, BackupException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateRegionSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateRegionSettingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRegionSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRegionSettings");

            return clientHandler.execute(new ClientExecutionParams<UpdateRegionSettingsRequest, UpdateRegionSettingsResponse>()
                    .withOperationName("UpdateRegionSettings").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateRegionSettingsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateRegionSettingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(BackupException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MissingParameterValueException")
                                .exceptionBuilderSupplier(MissingParameterValueException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                                .exceptionBuilderSupplier(InvalidParameterValueException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DependencyFailureException")
                                .exceptionBuilderSupplier(DependencyFailureException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidResourceStateException")
                                .exceptionBuilderSupplier(InvalidResourceStateException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AlreadyExistsException")
                                .exceptionBuilderSupplier(AlreadyExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BackupRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
