/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.backup.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Used to specify a set of resources to a backup plan.
 * </p>
 * <p>
 * Specifying your desired <code>Conditions</code>, <code>ListOfTags</code>, <code>NotResources</code>, and/or
 * <code>Resources</code> is recommended. If none of these are specified, Backup will attempt to select all supported
 * and opted-in storage resources, which could have unintended cost implications.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class BackupSelection implements SdkPojo, Serializable, ToCopyableBuilder<BackupSelection.Builder, BackupSelection> {
    private static final SdkField<String> SELECTION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SelectionName").getter(getter(BackupSelection::selectionName)).setter(setter(Builder::selectionName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SelectionName").build()).build();

    private static final SdkField<String> IAM_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IamRoleArn").getter(getter(BackupSelection::iamRoleArn)).setter(setter(Builder::iamRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IamRoleArn").build()).build();

    private static final SdkField<List<String>> RESOURCES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Resources")
            .getter(getter(BackupSelection::resources))
            .setter(setter(Builder::resources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Resources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Condition>> LIST_OF_TAGS_FIELD = SdkField
            .<List<Condition>> builder(MarshallingType.LIST)
            .memberName("ListOfTags")
            .getter(getter(BackupSelection::listOfTags))
            .setter(setter(Builder::listOfTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ListOfTags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Condition> builder(MarshallingType.SDK_POJO)
                                            .constructor(Condition::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> NOT_RESOURCES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("NotResources")
            .getter(getter(BackupSelection::notResources))
            .setter(setter(Builder::notResources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotResources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Conditions> CONDITIONS_FIELD = SdkField.<Conditions> builder(MarshallingType.SDK_POJO)
            .memberName("Conditions").getter(getter(BackupSelection::conditions)).setter(setter(Builder::conditions))
            .constructor(Conditions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Conditions").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SELECTION_NAME_FIELD,
            IAM_ROLE_ARN_FIELD, RESOURCES_FIELD, LIST_OF_TAGS_FIELD, NOT_RESOURCES_FIELD, CONDITIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String selectionName;

    private final String iamRoleArn;

    private final List<String> resources;

    private final List<Condition> listOfTags;

    private final List<String> notResources;

    private final Conditions conditions;

    private BackupSelection(BuilderImpl builder) {
        this.selectionName = builder.selectionName;
        this.iamRoleArn = builder.iamRoleArn;
        this.resources = builder.resources;
        this.listOfTags = builder.listOfTags;
        this.notResources = builder.notResources;
        this.conditions = builder.conditions;
    }

    /**
     * <p>
     * The display name of a resource selection document. Must contain 1 to 50 alphanumeric or '-_.' characters.
     * </p>
     * 
     * @return The display name of a resource selection document. Must contain 1 to 50 alphanumeric or '-_.' characters.
     */
    public final String selectionName() {
        return selectionName;
    }

    /**
     * <p>
     * The ARN of the IAM role that Backup uses to authenticate when backing up the target resource; for example,
     * <code>arn:aws:iam::123456789012:role/S3Access</code>.
     * </p>
     * 
     * @return The ARN of the IAM role that Backup uses to authenticate when backing up the target resource; for
     *         example, <code>arn:aws:iam::123456789012:role/S3Access</code>.
     */
    public final String iamRoleArn() {
        return iamRoleArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the Resources property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasResources() {
        return resources != null && !(resources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of Amazon Resource Names (ARNs) to assign to a backup plan. The maximum number of ARNs is 500 without
     * wildcards, or 30 ARNs with wildcards.
     * </p>
     * <p>
     * If you need to assign many resources to a backup plan, consider a different resource selection strategy, such as
     * assigning all resources of a resource type or refining your resource selection using tags.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResources} method.
     * </p>
     * 
     * @return A list of Amazon Resource Names (ARNs) to assign to a backup plan. The maximum number of ARNs is 500
     *         without wildcards, or 30 ARNs with wildcards.</p>
     *         <p>
     *         If you need to assign many resources to a backup plan, consider a different resource selection strategy,
     *         such as assigning all resources of a resource type or refining your resource selection using tags.
     */
    public final List<String> resources() {
        return resources;
    }

    /**
     * For responses, this returns true if the service returned a value for the ListOfTags property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasListOfTags() {
        return listOfTags != null && !(listOfTags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of conditions that you define to assign resources to your backup plans using tags. For example,
     * <code>"StringEquals": { "ConditionKey": "aws:ResourceTag/CreatedByCryo", "ConditionValue": "true" },</code>.
     * Condition operators are case sensitive.
     * </p>
     * <p>
     * <code>ListOfTags</code> differs from <code>Conditions</code> as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * When you specify more than one condition, you assign all resources that match AT LEAST ONE condition (using OR
     * logic).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ListOfTags</code> only supports <code>StringEquals</code>. <code>Conditions</code> supports
     * <code>StringEquals</code>, <code>StringLike</code>, <code>StringNotEquals</code>, and <code>StringNotLike</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasListOfTags} method.
     * </p>
     * 
     * @return A list of conditions that you define to assign resources to your backup plans using tags. For example,
     *         <code>"StringEquals": { "ConditionKey": "aws:ResourceTag/CreatedByCryo", "ConditionValue": "true" },</code>
     *         . Condition operators are case sensitive.</p>
     *         <p>
     *         <code>ListOfTags</code> differs from <code>Conditions</code> as follows:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         When you specify more than one condition, you assign all resources that match AT LEAST ONE condition
     *         (using OR logic).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ListOfTags</code> only supports <code>StringEquals</code>. <code>Conditions</code> supports
     *         <code>StringEquals</code>, <code>StringLike</code>, <code>StringNotEquals</code>, and
     *         <code>StringNotLike</code>.
     *         </p>
     *         </li>
     */
    public final List<Condition> listOfTags() {
        return listOfTags;
    }

    /**
     * For responses, this returns true if the service returned a value for the NotResources property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasNotResources() {
        return notResources != null && !(notResources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of Amazon Resource Names (ARNs) to exclude from a backup plan. The maximum number of ARNs is 500 without
     * wildcards, or 30 ARNs with wildcards.
     * </p>
     * <p>
     * If you need to exclude many resources from a backup plan, consider a different resource selection strategy, such
     * as assigning only one or a few resource types or refining your resource selection using tags.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNotResources} method.
     * </p>
     * 
     * @return A list of Amazon Resource Names (ARNs) to exclude from a backup plan. The maximum number of ARNs is 500
     *         without wildcards, or 30 ARNs with wildcards.</p>
     *         <p>
     *         If you need to exclude many resources from a backup plan, consider a different resource selection
     *         strategy, such as assigning only one or a few resource types or refining your resource selection using
     *         tags.
     */
    public final List<String> notResources() {
        return notResources;
    }

    /**
     * <p>
     * A list of conditions that you define to assign resources to your backup plans using tags. For example,
     * <code>"StringEquals": { "ConditionKey": "aws:ResourceTag/CreatedByCryo", "ConditionValue": "true" },</code>.
     * Condition operators are case sensitive.
     * </p>
     * <p>
     * <code>Conditions</code> differs from <code>ListOfTags</code> as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * When you specify more than one condition, you only assign the resources that match ALL conditions (using AND
     * logic).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Conditions</code> supports <code>StringEquals</code>, <code>StringLike</code>, <code>StringNotEquals</code>
     * , and <code>StringNotLike</code>. <code>ListOfTags</code> only supports <code>StringEquals</code>.
     * </p>
     * </li>
     * </ul>
     * 
     * @return A list of conditions that you define to assign resources to your backup plans using tags. For example,
     *         <code>"StringEquals": { "ConditionKey": "aws:ResourceTag/CreatedByCryo", "ConditionValue": "true" },</code>
     *         . Condition operators are case sensitive.</p>
     *         <p>
     *         <code>Conditions</code> differs from <code>ListOfTags</code> as follows:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         When you specify more than one condition, you only assign the resources that match ALL conditions (using
     *         AND logic).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Conditions</code> supports <code>StringEquals</code>, <code>StringLike</code>,
     *         <code>StringNotEquals</code>, and <code>StringNotLike</code>. <code>ListOfTags</code> only supports
     *         <code>StringEquals</code>.
     *         </p>
     *         </li>
     */
    public final Conditions conditions() {
        return conditions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(selectionName());
        hashCode = 31 * hashCode + Objects.hashCode(iamRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasResources() ? resources() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasListOfTags() ? listOfTags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasNotResources() ? notResources() : null);
        hashCode = 31 * hashCode + Objects.hashCode(conditions());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof BackupSelection)) {
            return false;
        }
        BackupSelection other = (BackupSelection) obj;
        return Objects.equals(selectionName(), other.selectionName()) && Objects.equals(iamRoleArn(), other.iamRoleArn())
                && hasResources() == other.hasResources() && Objects.equals(resources(), other.resources())
                && hasListOfTags() == other.hasListOfTags() && Objects.equals(listOfTags(), other.listOfTags())
                && hasNotResources() == other.hasNotResources() && Objects.equals(notResources(), other.notResources())
                && Objects.equals(conditions(), other.conditions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("BackupSelection").add("SelectionName", selectionName()).add("IamRoleArn", iamRoleArn())
                .add("Resources", hasResources() ? resources() : null).add("ListOfTags", hasListOfTags() ? listOfTags() : null)
                .add("NotResources", hasNotResources() ? notResources() : null).add("Conditions", conditions()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SelectionName":
            return Optional.ofNullable(clazz.cast(selectionName()));
        case "IamRoleArn":
            return Optional.ofNullable(clazz.cast(iamRoleArn()));
        case "Resources":
            return Optional.ofNullable(clazz.cast(resources()));
        case "ListOfTags":
            return Optional.ofNullable(clazz.cast(listOfTags()));
        case "NotResources":
            return Optional.ofNullable(clazz.cast(notResources()));
        case "Conditions":
            return Optional.ofNullable(clazz.cast(conditions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<BackupSelection, T> g) {
        return obj -> g.apply((BackupSelection) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, BackupSelection> {
        /**
         * <p>
         * The display name of a resource selection document. Must contain 1 to 50 alphanumeric or '-_.' characters.
         * </p>
         * 
         * @param selectionName
         *        The display name of a resource selection document. Must contain 1 to 50 alphanumeric or '-_.'
         *        characters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder selectionName(String selectionName);

        /**
         * <p>
         * The ARN of the IAM role that Backup uses to authenticate when backing up the target resource; for example,
         * <code>arn:aws:iam::123456789012:role/S3Access</code>.
         * </p>
         * 
         * @param iamRoleArn
         *        The ARN of the IAM role that Backup uses to authenticate when backing up the target resource; for
         *        example, <code>arn:aws:iam::123456789012:role/S3Access</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iamRoleArn(String iamRoleArn);

        /**
         * <p>
         * A list of Amazon Resource Names (ARNs) to assign to a backup plan. The maximum number of ARNs is 500 without
         * wildcards, or 30 ARNs with wildcards.
         * </p>
         * <p>
         * If you need to assign many resources to a backup plan, consider a different resource selection strategy, such
         * as assigning all resources of a resource type or refining your resource selection using tags.
         * </p>
         * 
         * @param resources
         *        A list of Amazon Resource Names (ARNs) to assign to a backup plan. The maximum number of ARNs is 500
         *        without wildcards, or 30 ARNs with wildcards.</p>
         *        <p>
         *        If you need to assign many resources to a backup plan, consider a different resource selection
         *        strategy, such as assigning all resources of a resource type or refining your resource selection using
         *        tags.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resources(Collection<String> resources);

        /**
         * <p>
         * A list of Amazon Resource Names (ARNs) to assign to a backup plan. The maximum number of ARNs is 500 without
         * wildcards, or 30 ARNs with wildcards.
         * </p>
         * <p>
         * If you need to assign many resources to a backup plan, consider a different resource selection strategy, such
         * as assigning all resources of a resource type or refining your resource selection using tags.
         * </p>
         * 
         * @param resources
         *        A list of Amazon Resource Names (ARNs) to assign to a backup plan. The maximum number of ARNs is 500
         *        without wildcards, or 30 ARNs with wildcards.</p>
         *        <p>
         *        If you need to assign many resources to a backup plan, consider a different resource selection
         *        strategy, such as assigning all resources of a resource type or refining your resource selection using
         *        tags.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resources(String... resources);

        /**
         * <p>
         * A list of conditions that you define to assign resources to your backup plans using tags. For example,
         * <code>"StringEquals": { "ConditionKey": "aws:ResourceTag/CreatedByCryo", "ConditionValue": "true" },</code>.
         * Condition operators are case sensitive.
         * </p>
         * <p>
         * <code>ListOfTags</code> differs from <code>Conditions</code> as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * When you specify more than one condition, you assign all resources that match AT LEAST ONE condition (using
         * OR logic).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ListOfTags</code> only supports <code>StringEquals</code>. <code>Conditions</code> supports
         * <code>StringEquals</code>, <code>StringLike</code>, <code>StringNotEquals</code>, and
         * <code>StringNotLike</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param listOfTags
         *        A list of conditions that you define to assign resources to your backup plans using tags. For example,
         *        <code>"StringEquals": { "ConditionKey": "aws:ResourceTag/CreatedByCryo", "ConditionValue": "true" },</code>
         *        . Condition operators are case sensitive.</p>
         *        <p>
         *        <code>ListOfTags</code> differs from <code>Conditions</code> as follows:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        When you specify more than one condition, you assign all resources that match AT LEAST ONE condition
         *        (using OR logic).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ListOfTags</code> only supports <code>StringEquals</code>. <code>Conditions</code> supports
         *        <code>StringEquals</code>, <code>StringLike</code>, <code>StringNotEquals</code>, and
         *        <code>StringNotLike</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder listOfTags(Collection<Condition> listOfTags);

        /**
         * <p>
         * A list of conditions that you define to assign resources to your backup plans using tags. For example,
         * <code>"StringEquals": { "ConditionKey": "aws:ResourceTag/CreatedByCryo", "ConditionValue": "true" },</code>.
         * Condition operators are case sensitive.
         * </p>
         * <p>
         * <code>ListOfTags</code> differs from <code>Conditions</code> as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * When you specify more than one condition, you assign all resources that match AT LEAST ONE condition (using
         * OR logic).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ListOfTags</code> only supports <code>StringEquals</code>. <code>Conditions</code> supports
         * <code>StringEquals</code>, <code>StringLike</code>, <code>StringNotEquals</code>, and
         * <code>StringNotLike</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param listOfTags
         *        A list of conditions that you define to assign resources to your backup plans using tags. For example,
         *        <code>"StringEquals": { "ConditionKey": "aws:ResourceTag/CreatedByCryo", "ConditionValue": "true" },</code>
         *        . Condition operators are case sensitive.</p>
         *        <p>
         *        <code>ListOfTags</code> differs from <code>Conditions</code> as follows:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        When you specify more than one condition, you assign all resources that match AT LEAST ONE condition
         *        (using OR logic).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ListOfTags</code> only supports <code>StringEquals</code>. <code>Conditions</code> supports
         *        <code>StringEquals</code>, <code>StringLike</code>, <code>StringNotEquals</code>, and
         *        <code>StringNotLike</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder listOfTags(Condition... listOfTags);

        /**
         * <p>
         * A list of conditions that you define to assign resources to your backup plans using tags. For example,
         * <code>"StringEquals": { "ConditionKey": "aws:ResourceTag/CreatedByCryo", "ConditionValue": "true" },</code>.
         * Condition operators are case sensitive.
         * </p>
         * <p>
         * <code>ListOfTags</code> differs from <code>Conditions</code> as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * When you specify more than one condition, you assign all resources that match AT LEAST ONE condition (using
         * OR logic).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ListOfTags</code> only supports <code>StringEquals</code>. <code>Conditions</code> supports
         * <code>StringEquals</code>, <code>StringLike</code>, <code>StringNotEquals</code>, and
         * <code>StringNotLike</code>.
         * </p>
         * </li>
         * </ul>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.backup.model.Condition.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.backup.model.Condition#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.backup.model.Condition.Builder#build()} is called immediately and its
         * result is passed to {@link #listOfTags(List<Condition>)}.
         * 
         * @param listOfTags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.backup.model.Condition.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #listOfTags(java.util.Collection<Condition>)
         */
        Builder listOfTags(Consumer<Condition.Builder>... listOfTags);

        /**
         * <p>
         * A list of Amazon Resource Names (ARNs) to exclude from a backup plan. The maximum number of ARNs is 500
         * without wildcards, or 30 ARNs with wildcards.
         * </p>
         * <p>
         * If you need to exclude many resources from a backup plan, consider a different resource selection strategy,
         * such as assigning only one or a few resource types or refining your resource selection using tags.
         * </p>
         * 
         * @param notResources
         *        A list of Amazon Resource Names (ARNs) to exclude from a backup plan. The maximum number of ARNs is
         *        500 without wildcards, or 30 ARNs with wildcards.</p>
         *        <p>
         *        If you need to exclude many resources from a backup plan, consider a different resource selection
         *        strategy, such as assigning only one or a few resource types or refining your resource selection using
         *        tags.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notResources(Collection<String> notResources);

        /**
         * <p>
         * A list of Amazon Resource Names (ARNs) to exclude from a backup plan. The maximum number of ARNs is 500
         * without wildcards, or 30 ARNs with wildcards.
         * </p>
         * <p>
         * If you need to exclude many resources from a backup plan, consider a different resource selection strategy,
         * such as assigning only one or a few resource types or refining your resource selection using tags.
         * </p>
         * 
         * @param notResources
         *        A list of Amazon Resource Names (ARNs) to exclude from a backup plan. The maximum number of ARNs is
         *        500 without wildcards, or 30 ARNs with wildcards.</p>
         *        <p>
         *        If you need to exclude many resources from a backup plan, consider a different resource selection
         *        strategy, such as assigning only one or a few resource types or refining your resource selection using
         *        tags.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notResources(String... notResources);

        /**
         * <p>
         * A list of conditions that you define to assign resources to your backup plans using tags. For example,
         * <code>"StringEquals": { "ConditionKey": "aws:ResourceTag/CreatedByCryo", "ConditionValue": "true" },</code>.
         * Condition operators are case sensitive.
         * </p>
         * <p>
         * <code>Conditions</code> differs from <code>ListOfTags</code> as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * When you specify more than one condition, you only assign the resources that match ALL conditions (using AND
         * logic).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Conditions</code> supports <code>StringEquals</code>, <code>StringLike</code>,
         * <code>StringNotEquals</code>, and <code>StringNotLike</code>. <code>ListOfTags</code> only supports
         * <code>StringEquals</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param conditions
         *        A list of conditions that you define to assign resources to your backup plans using tags. For example,
         *        <code>"StringEquals": { "ConditionKey": "aws:ResourceTag/CreatedByCryo", "ConditionValue": "true" },</code>
         *        . Condition operators are case sensitive.</p>
         *        <p>
         *        <code>Conditions</code> differs from <code>ListOfTags</code> as follows:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        When you specify more than one condition, you only assign the resources that match ALL conditions
         *        (using AND logic).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Conditions</code> supports <code>StringEquals</code>, <code>StringLike</code>,
         *        <code>StringNotEquals</code>, and <code>StringNotLike</code>. <code>ListOfTags</code> only supports
         *        <code>StringEquals</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder conditions(Conditions conditions);

        /**
         * <p>
         * A list of conditions that you define to assign resources to your backup plans using tags. For example,
         * <code>"StringEquals": { "ConditionKey": "aws:ResourceTag/CreatedByCryo", "ConditionValue": "true" },</code>.
         * Condition operators are case sensitive.
         * </p>
         * <p>
         * <code>Conditions</code> differs from <code>ListOfTags</code> as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * When you specify more than one condition, you only assign the resources that match ALL conditions (using AND
         * logic).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Conditions</code> supports <code>StringEquals</code>, <code>StringLike</code>,
         * <code>StringNotEquals</code>, and <code>StringNotLike</code>. <code>ListOfTags</code> only supports
         * <code>StringEquals</code>.
         * </p>
         * </li>
         * </ul>
         * This is a convenience method that creates an instance of the {@link Conditions.Builder} avoiding the need to
         * create one manually via {@link Conditions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Conditions.Builder#build()} is called immediately and its result
         * is passed to {@link #conditions(Conditions)}.
         * 
         * @param conditions
         *        a consumer that will call methods on {@link Conditions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #conditions(Conditions)
         */
        default Builder conditions(Consumer<Conditions.Builder> conditions) {
            return conditions(Conditions.builder().applyMutation(conditions).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String selectionName;

        private String iamRoleArn;

        private List<String> resources = DefaultSdkAutoConstructList.getInstance();

        private List<Condition> listOfTags = DefaultSdkAutoConstructList.getInstance();

        private List<String> notResources = DefaultSdkAutoConstructList.getInstance();

        private Conditions conditions;

        private BuilderImpl() {
        }

        private BuilderImpl(BackupSelection model) {
            selectionName(model.selectionName);
            iamRoleArn(model.iamRoleArn);
            resources(model.resources);
            listOfTags(model.listOfTags);
            notResources(model.notResources);
            conditions(model.conditions);
        }

        public final String getSelectionName() {
            return selectionName;
        }

        public final void setSelectionName(String selectionName) {
            this.selectionName = selectionName;
        }

        @Override
        public final Builder selectionName(String selectionName) {
            this.selectionName = selectionName;
            return this;
        }

        public final String getIamRoleArn() {
            return iamRoleArn;
        }

        public final void setIamRoleArn(String iamRoleArn) {
            this.iamRoleArn = iamRoleArn;
        }

        @Override
        public final Builder iamRoleArn(String iamRoleArn) {
            this.iamRoleArn = iamRoleArn;
            return this;
        }

        public final Collection<String> getResources() {
            if (resources instanceof SdkAutoConstructList) {
                return null;
            }
            return resources;
        }

        public final void setResources(Collection<String> resources) {
            this.resources = ResourceArnsCopier.copy(resources);
        }

        @Override
        public final Builder resources(Collection<String> resources) {
            this.resources = ResourceArnsCopier.copy(resources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resources(String... resources) {
            resources(Arrays.asList(resources));
            return this;
        }

        public final List<Condition.Builder> getListOfTags() {
            List<Condition.Builder> result = ListOfTagsCopier.copyToBuilder(this.listOfTags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setListOfTags(Collection<Condition.BuilderImpl> listOfTags) {
            this.listOfTags = ListOfTagsCopier.copyFromBuilder(listOfTags);
        }

        @Override
        public final Builder listOfTags(Collection<Condition> listOfTags) {
            this.listOfTags = ListOfTagsCopier.copy(listOfTags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder listOfTags(Condition... listOfTags) {
            listOfTags(Arrays.asList(listOfTags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder listOfTags(Consumer<Condition.Builder>... listOfTags) {
            listOfTags(Stream.of(listOfTags).map(c -> Condition.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Collection<String> getNotResources() {
            if (notResources instanceof SdkAutoConstructList) {
                return null;
            }
            return notResources;
        }

        public final void setNotResources(Collection<String> notResources) {
            this.notResources = ResourceArnsCopier.copy(notResources);
        }

        @Override
        public final Builder notResources(Collection<String> notResources) {
            this.notResources = ResourceArnsCopier.copy(notResources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder notResources(String... notResources) {
            notResources(Arrays.asList(notResources));
            return this;
        }

        public final Conditions.Builder getConditions() {
            return conditions != null ? conditions.toBuilder() : null;
        }

        public final void setConditions(Conditions.BuilderImpl conditions) {
            this.conditions = conditions != null ? conditions.build() : null;
        }

        @Override
        public final Builder conditions(Conditions conditions) {
            this.conditions = conditions;
            return this;
        }

        @Override
        public BackupSelection build() {
            return new BackupSelection(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
