/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.backup.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class PutBackupVaultNotificationsRequest extends BackupRequest implements
        ToCopyableBuilder<PutBackupVaultNotificationsRequest.Builder, PutBackupVaultNotificationsRequest> {
    private static final SdkField<String> BACKUP_VAULT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BackupVaultName").getter(getter(PutBackupVaultNotificationsRequest::backupVaultName))
            .setter(setter(Builder::backupVaultName))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("backupVaultName").build()).build();

    private static final SdkField<String> SNS_TOPIC_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SNSTopicArn").getter(getter(PutBackupVaultNotificationsRequest::snsTopicArn))
            .setter(setter(Builder::snsTopicArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SNSTopicArn").build()).build();

    private static final SdkField<List<String>> BACKUP_VAULT_EVENTS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("BackupVaultEvents")
            .getter(getter(PutBackupVaultNotificationsRequest::backupVaultEventsAsStrings))
            .setter(setter(Builder::backupVaultEventsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BackupVaultEvents").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BACKUP_VAULT_NAME_FIELD,
            SNS_TOPIC_ARN_FIELD, BACKUP_VAULT_EVENTS_FIELD));

    private final String backupVaultName;

    private final String snsTopicArn;

    private final List<String> backupVaultEvents;

    private PutBackupVaultNotificationsRequest(BuilderImpl builder) {
        super(builder);
        this.backupVaultName = builder.backupVaultName;
        this.snsTopicArn = builder.snsTopicArn;
        this.backupVaultEvents = builder.backupVaultEvents;
    }

    /**
     * <p>
     * The name of a logical container where backups are stored. Backup vaults are identified by names that are unique
     * to the account used to create them and the Amazon Web Services Region where they are created. They consist of
     * lowercase letters, numbers, and hyphens.
     * </p>
     * 
     * @return The name of a logical container where backups are stored. Backup vaults are identified by names that are
     *         unique to the account used to create them and the Amazon Web Services Region where they are created. They
     *         consist of lowercase letters, numbers, and hyphens.
     */
    public final String backupVaultName() {
        return backupVaultName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) that specifies the topic for a backup vault’s events; for example,
     * <code>arn:aws:sns:us-west-2:111122223333:MyVaultTopic</code>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) that specifies the topic for a backup vault’s events; for example,
     *         <code>arn:aws:sns:us-west-2:111122223333:MyVaultTopic</code>.
     */
    public final String snsTopicArn() {
        return snsTopicArn;
    }

    /**
     * <p>
     * An array of events that indicate the status of jobs to back up resources to the backup vault.
     * </p>
     * <p>
     * For common use cases and code samples, see <a
     * href="https://docs.aws.amazon.com/aws-backup/latest/devguide/sns-notifications.html">Using Amazon SNS to track
     * Backup events</a>.
     * </p>
     * <p>
     * The following events are supported:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>BACKUP_JOB_STARTED</code> | <code>BACKUP_JOB_COMPLETED</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>COPY_JOB_STARTED</code> | <code>COPY_JOB_SUCCESSFUL</code> | <code>COPY_JOB_FAILED</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RESTORE_JOB_STARTED</code> | <code>RESTORE_JOB_COMPLETED</code> | <code>RECOVERY_POINT_MODIFIED</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>S3_BACKUP_OBJECT_FAILED</code> | <code>S3_RESTORE_OBJECT_FAILED</code>
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The list below shows items that are deprecated events (for reference) and are no longer in use. They are no
     * longer supported and will not return statuses or notifications. Refer to the list above for current supported
     * events.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasBackupVaultEvents} method.
     * </p>
     * 
     * @return An array of events that indicate the status of jobs to back up resources to the backup vault.</p>
     *         <p>
     *         For common use cases and code samples, see <a
     *         href="https://docs.aws.amazon.com/aws-backup/latest/devguide/sns-notifications.html">Using Amazon SNS to
     *         track Backup events</a>.
     *         </p>
     *         <p>
     *         The following events are supported:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>BACKUP_JOB_STARTED</code> | <code>BACKUP_JOB_COMPLETED</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>COPY_JOB_STARTED</code> | <code>COPY_JOB_SUCCESSFUL</code> | <code>COPY_JOB_FAILED</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RESTORE_JOB_STARTED</code> | <code>RESTORE_JOB_COMPLETED</code> |
     *         <code>RECOVERY_POINT_MODIFIED</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>S3_BACKUP_OBJECT_FAILED</code> | <code>S3_RESTORE_OBJECT_FAILED</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         The list below shows items that are deprecated events (for reference) and are no longer in use. They are
     *         no longer supported and will not return statuses or notifications. Refer to the list above for current
     *         supported events.
     *         </p>
     */
    public final List<BackupVaultEvent> backupVaultEvents() {
        return BackupVaultEventsCopier.copyStringToEnum(backupVaultEvents);
    }

    /**
     * For responses, this returns true if the service returned a value for the BackupVaultEvents property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasBackupVaultEvents() {
        return backupVaultEvents != null && !(backupVaultEvents instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of events that indicate the status of jobs to back up resources to the backup vault.
     * </p>
     * <p>
     * For common use cases and code samples, see <a
     * href="https://docs.aws.amazon.com/aws-backup/latest/devguide/sns-notifications.html">Using Amazon SNS to track
     * Backup events</a>.
     * </p>
     * <p>
     * The following events are supported:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>BACKUP_JOB_STARTED</code> | <code>BACKUP_JOB_COMPLETED</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>COPY_JOB_STARTED</code> | <code>COPY_JOB_SUCCESSFUL</code> | <code>COPY_JOB_FAILED</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RESTORE_JOB_STARTED</code> | <code>RESTORE_JOB_COMPLETED</code> | <code>RECOVERY_POINT_MODIFIED</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>S3_BACKUP_OBJECT_FAILED</code> | <code>S3_RESTORE_OBJECT_FAILED</code>
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The list below shows items that are deprecated events (for reference) and are no longer in use. They are no
     * longer supported and will not return statuses or notifications. Refer to the list above for current supported
     * events.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasBackupVaultEvents} method.
     * </p>
     * 
     * @return An array of events that indicate the status of jobs to back up resources to the backup vault.</p>
     *         <p>
     *         For common use cases and code samples, see <a
     *         href="https://docs.aws.amazon.com/aws-backup/latest/devguide/sns-notifications.html">Using Amazon SNS to
     *         track Backup events</a>.
     *         </p>
     *         <p>
     *         The following events are supported:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>BACKUP_JOB_STARTED</code> | <code>BACKUP_JOB_COMPLETED</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>COPY_JOB_STARTED</code> | <code>COPY_JOB_SUCCESSFUL</code> | <code>COPY_JOB_FAILED</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RESTORE_JOB_STARTED</code> | <code>RESTORE_JOB_COMPLETED</code> |
     *         <code>RECOVERY_POINT_MODIFIED</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>S3_BACKUP_OBJECT_FAILED</code> | <code>S3_RESTORE_OBJECT_FAILED</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         The list below shows items that are deprecated events (for reference) and are no longer in use. They are
     *         no longer supported and will not return statuses or notifications. Refer to the list above for current
     *         supported events.
     *         </p>
     */
    public final List<String> backupVaultEventsAsStrings() {
        return backupVaultEvents;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(backupVaultName());
        hashCode = 31 * hashCode + Objects.hashCode(snsTopicArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasBackupVaultEvents() ? backupVaultEventsAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PutBackupVaultNotificationsRequest)) {
            return false;
        }
        PutBackupVaultNotificationsRequest other = (PutBackupVaultNotificationsRequest) obj;
        return Objects.equals(backupVaultName(), other.backupVaultName()) && Objects.equals(snsTopicArn(), other.snsTopicArn())
                && hasBackupVaultEvents() == other.hasBackupVaultEvents()
                && Objects.equals(backupVaultEventsAsStrings(), other.backupVaultEventsAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PutBackupVaultNotificationsRequest").add("BackupVaultName", backupVaultName())
                .add("SNSTopicArn", snsTopicArn())
                .add("BackupVaultEvents", hasBackupVaultEvents() ? backupVaultEventsAsStrings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "BackupVaultName":
            return Optional.ofNullable(clazz.cast(backupVaultName()));
        case "SNSTopicArn":
            return Optional.ofNullable(clazz.cast(snsTopicArn()));
        case "BackupVaultEvents":
            return Optional.ofNullable(clazz.cast(backupVaultEventsAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PutBackupVaultNotificationsRequest, T> g) {
        return obj -> g.apply((PutBackupVaultNotificationsRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends BackupRequest.Builder, SdkPojo, CopyableBuilder<Builder, PutBackupVaultNotificationsRequest> {
        /**
         * <p>
         * The name of a logical container where backups are stored. Backup vaults are identified by names that are
         * unique to the account used to create them and the Amazon Web Services Region where they are created. They
         * consist of lowercase letters, numbers, and hyphens.
         * </p>
         * 
         * @param backupVaultName
         *        The name of a logical container where backups are stored. Backup vaults are identified by names that
         *        are unique to the account used to create them and the Amazon Web Services Region where they are
         *        created. They consist of lowercase letters, numbers, and hyphens.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backupVaultName(String backupVaultName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) that specifies the topic for a backup vault’s events; for example,
         * <code>arn:aws:sns:us-west-2:111122223333:MyVaultTopic</code>.
         * </p>
         * 
         * @param snsTopicArn
         *        The Amazon Resource Name (ARN) that specifies the topic for a backup vault’s events; for example,
         *        <code>arn:aws:sns:us-west-2:111122223333:MyVaultTopic</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snsTopicArn(String snsTopicArn);

        /**
         * <p>
         * An array of events that indicate the status of jobs to back up resources to the backup vault.
         * </p>
         * <p>
         * For common use cases and code samples, see <a
         * href="https://docs.aws.amazon.com/aws-backup/latest/devguide/sns-notifications.html">Using Amazon SNS to
         * track Backup events</a>.
         * </p>
         * <p>
         * The following events are supported:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>BACKUP_JOB_STARTED</code> | <code>BACKUP_JOB_COMPLETED</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>COPY_JOB_STARTED</code> | <code>COPY_JOB_SUCCESSFUL</code> | <code>COPY_JOB_FAILED</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RESTORE_JOB_STARTED</code> | <code>RESTORE_JOB_COMPLETED</code> | <code>RECOVERY_POINT_MODIFIED</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>S3_BACKUP_OBJECT_FAILED</code> | <code>S3_RESTORE_OBJECT_FAILED</code>
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The list below shows items that are deprecated events (for reference) and are no longer in use. They are no
         * longer supported and will not return statuses or notifications. Refer to the list above for current supported
         * events.
         * </p>
         * </note>
         * 
         * @param backupVaultEvents
         *        An array of events that indicate the status of jobs to back up resources to the backup vault.</p>
         *        <p>
         *        For common use cases and code samples, see <a
         *        href="https://docs.aws.amazon.com/aws-backup/latest/devguide/sns-notifications.html">Using Amazon SNS
         *        to track Backup events</a>.
         *        </p>
         *        <p>
         *        The following events are supported:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>BACKUP_JOB_STARTED</code> | <code>BACKUP_JOB_COMPLETED</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>COPY_JOB_STARTED</code> | <code>COPY_JOB_SUCCESSFUL</code> | <code>COPY_JOB_FAILED</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RESTORE_JOB_STARTED</code> | <code>RESTORE_JOB_COMPLETED</code> |
         *        <code>RECOVERY_POINT_MODIFIED</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>S3_BACKUP_OBJECT_FAILED</code> | <code>S3_RESTORE_OBJECT_FAILED</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        The list below shows items that are deprecated events (for reference) and are no longer in use. They
         *        are no longer supported and will not return statuses or notifications. Refer to the list above for
         *        current supported events.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backupVaultEventsWithStrings(Collection<String> backupVaultEvents);

        /**
         * <p>
         * An array of events that indicate the status of jobs to back up resources to the backup vault.
         * </p>
         * <p>
         * For common use cases and code samples, see <a
         * href="https://docs.aws.amazon.com/aws-backup/latest/devguide/sns-notifications.html">Using Amazon SNS to
         * track Backup events</a>.
         * </p>
         * <p>
         * The following events are supported:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>BACKUP_JOB_STARTED</code> | <code>BACKUP_JOB_COMPLETED</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>COPY_JOB_STARTED</code> | <code>COPY_JOB_SUCCESSFUL</code> | <code>COPY_JOB_FAILED</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RESTORE_JOB_STARTED</code> | <code>RESTORE_JOB_COMPLETED</code> | <code>RECOVERY_POINT_MODIFIED</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>S3_BACKUP_OBJECT_FAILED</code> | <code>S3_RESTORE_OBJECT_FAILED</code>
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The list below shows items that are deprecated events (for reference) and are no longer in use. They are no
         * longer supported and will not return statuses or notifications. Refer to the list above for current supported
         * events.
         * </p>
         * </note>
         * 
         * @param backupVaultEvents
         *        An array of events that indicate the status of jobs to back up resources to the backup vault.</p>
         *        <p>
         *        For common use cases and code samples, see <a
         *        href="https://docs.aws.amazon.com/aws-backup/latest/devguide/sns-notifications.html">Using Amazon SNS
         *        to track Backup events</a>.
         *        </p>
         *        <p>
         *        The following events are supported:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>BACKUP_JOB_STARTED</code> | <code>BACKUP_JOB_COMPLETED</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>COPY_JOB_STARTED</code> | <code>COPY_JOB_SUCCESSFUL</code> | <code>COPY_JOB_FAILED</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RESTORE_JOB_STARTED</code> | <code>RESTORE_JOB_COMPLETED</code> |
         *        <code>RECOVERY_POINT_MODIFIED</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>S3_BACKUP_OBJECT_FAILED</code> | <code>S3_RESTORE_OBJECT_FAILED</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        The list below shows items that are deprecated events (for reference) and are no longer in use. They
         *        are no longer supported and will not return statuses or notifications. Refer to the list above for
         *        current supported events.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backupVaultEventsWithStrings(String... backupVaultEvents);

        /**
         * <p>
         * An array of events that indicate the status of jobs to back up resources to the backup vault.
         * </p>
         * <p>
         * For common use cases and code samples, see <a
         * href="https://docs.aws.amazon.com/aws-backup/latest/devguide/sns-notifications.html">Using Amazon SNS to
         * track Backup events</a>.
         * </p>
         * <p>
         * The following events are supported:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>BACKUP_JOB_STARTED</code> | <code>BACKUP_JOB_COMPLETED</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>COPY_JOB_STARTED</code> | <code>COPY_JOB_SUCCESSFUL</code> | <code>COPY_JOB_FAILED</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RESTORE_JOB_STARTED</code> | <code>RESTORE_JOB_COMPLETED</code> | <code>RECOVERY_POINT_MODIFIED</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>S3_BACKUP_OBJECT_FAILED</code> | <code>S3_RESTORE_OBJECT_FAILED</code>
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The list below shows items that are deprecated events (for reference) and are no longer in use. They are no
         * longer supported and will not return statuses or notifications. Refer to the list above for current supported
         * events.
         * </p>
         * </note>
         * 
         * @param backupVaultEvents
         *        An array of events that indicate the status of jobs to back up resources to the backup vault.</p>
         *        <p>
         *        For common use cases and code samples, see <a
         *        href="https://docs.aws.amazon.com/aws-backup/latest/devguide/sns-notifications.html">Using Amazon SNS
         *        to track Backup events</a>.
         *        </p>
         *        <p>
         *        The following events are supported:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>BACKUP_JOB_STARTED</code> | <code>BACKUP_JOB_COMPLETED</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>COPY_JOB_STARTED</code> | <code>COPY_JOB_SUCCESSFUL</code> | <code>COPY_JOB_FAILED</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RESTORE_JOB_STARTED</code> | <code>RESTORE_JOB_COMPLETED</code> |
         *        <code>RECOVERY_POINT_MODIFIED</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>S3_BACKUP_OBJECT_FAILED</code> | <code>S3_RESTORE_OBJECT_FAILED</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        The list below shows items that are deprecated events (for reference) and are no longer in use. They
         *        are no longer supported and will not return statuses or notifications. Refer to the list above for
         *        current supported events.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backupVaultEvents(Collection<BackupVaultEvent> backupVaultEvents);

        /**
         * <p>
         * An array of events that indicate the status of jobs to back up resources to the backup vault.
         * </p>
         * <p>
         * For common use cases and code samples, see <a
         * href="https://docs.aws.amazon.com/aws-backup/latest/devguide/sns-notifications.html">Using Amazon SNS to
         * track Backup events</a>.
         * </p>
         * <p>
         * The following events are supported:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>BACKUP_JOB_STARTED</code> | <code>BACKUP_JOB_COMPLETED</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>COPY_JOB_STARTED</code> | <code>COPY_JOB_SUCCESSFUL</code> | <code>COPY_JOB_FAILED</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RESTORE_JOB_STARTED</code> | <code>RESTORE_JOB_COMPLETED</code> | <code>RECOVERY_POINT_MODIFIED</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>S3_BACKUP_OBJECT_FAILED</code> | <code>S3_RESTORE_OBJECT_FAILED</code>
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The list below shows items that are deprecated events (for reference) and are no longer in use. They are no
         * longer supported and will not return statuses or notifications. Refer to the list above for current supported
         * events.
         * </p>
         * </note>
         * 
         * @param backupVaultEvents
         *        An array of events that indicate the status of jobs to back up resources to the backup vault.</p>
         *        <p>
         *        For common use cases and code samples, see <a
         *        href="https://docs.aws.amazon.com/aws-backup/latest/devguide/sns-notifications.html">Using Amazon SNS
         *        to track Backup events</a>.
         *        </p>
         *        <p>
         *        The following events are supported:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>BACKUP_JOB_STARTED</code> | <code>BACKUP_JOB_COMPLETED</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>COPY_JOB_STARTED</code> | <code>COPY_JOB_SUCCESSFUL</code> | <code>COPY_JOB_FAILED</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RESTORE_JOB_STARTED</code> | <code>RESTORE_JOB_COMPLETED</code> |
         *        <code>RECOVERY_POINT_MODIFIED</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>S3_BACKUP_OBJECT_FAILED</code> | <code>S3_RESTORE_OBJECT_FAILED</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        The list below shows items that are deprecated events (for reference) and are no longer in use. They
         *        are no longer supported and will not return statuses or notifications. Refer to the list above for
         *        current supported events.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backupVaultEvents(BackupVaultEvent... backupVaultEvents);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends BackupRequest.BuilderImpl implements Builder {
        private String backupVaultName;

        private String snsTopicArn;

        private List<String> backupVaultEvents = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(PutBackupVaultNotificationsRequest model) {
            super(model);
            backupVaultName(model.backupVaultName);
            snsTopicArn(model.snsTopicArn);
            backupVaultEventsWithStrings(model.backupVaultEvents);
        }

        public final String getBackupVaultName() {
            return backupVaultName;
        }

        public final void setBackupVaultName(String backupVaultName) {
            this.backupVaultName = backupVaultName;
        }

        @Override
        public final Builder backupVaultName(String backupVaultName) {
            this.backupVaultName = backupVaultName;
            return this;
        }

        public final String getSnsTopicArn() {
            return snsTopicArn;
        }

        public final void setSnsTopicArn(String snsTopicArn) {
            this.snsTopicArn = snsTopicArn;
        }

        @Override
        public final Builder snsTopicArn(String snsTopicArn) {
            this.snsTopicArn = snsTopicArn;
            return this;
        }

        public final Collection<String> getBackupVaultEvents() {
            if (backupVaultEvents instanceof SdkAutoConstructList) {
                return null;
            }
            return backupVaultEvents;
        }

        public final void setBackupVaultEvents(Collection<String> backupVaultEvents) {
            this.backupVaultEvents = BackupVaultEventsCopier.copy(backupVaultEvents);
        }

        @Override
        public final Builder backupVaultEventsWithStrings(Collection<String> backupVaultEvents) {
            this.backupVaultEvents = BackupVaultEventsCopier.copy(backupVaultEvents);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder backupVaultEventsWithStrings(String... backupVaultEvents) {
            backupVaultEventsWithStrings(Arrays.asList(backupVaultEvents));
            return this;
        }

        @Override
        public final Builder backupVaultEvents(Collection<BackupVaultEvent> backupVaultEvents) {
            this.backupVaultEvents = BackupVaultEventsCopier.copyEnumToString(backupVaultEvents);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder backupVaultEvents(BackupVaultEvent... backupVaultEvents) {
            backupVaultEvents(Arrays.asList(backupVaultEvents));
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public PutBackupVaultNotificationsRequest build() {
            return new PutBackupVaultNotificationsRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
