/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.backup.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class DescribeBackupJobResponse extends BackupResponse implements
        ToCopyableBuilder<DescribeBackupJobResponse.Builder, DescribeBackupJobResponse> {
    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AccountId").getter(getter(DescribeBackupJobResponse::accountId)).setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccountId").build()).build();

    private static final SdkField<String> BACKUP_JOB_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BackupJobId").getter(getter(DescribeBackupJobResponse::backupJobId))
            .setter(setter(Builder::backupJobId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BackupJobId").build()).build();

    private static final SdkField<String> BACKUP_VAULT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BackupVaultName").getter(getter(DescribeBackupJobResponse::backupVaultName))
            .setter(setter(Builder::backupVaultName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BackupVaultName").build()).build();

    private static final SdkField<String> BACKUP_VAULT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BackupVaultArn").getter(getter(DescribeBackupJobResponse::backupVaultArn))
            .setter(setter(Builder::backupVaultArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BackupVaultArn").build()).build();

    private static final SdkField<String> RECOVERY_POINT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RecoveryPointArn").getter(getter(DescribeBackupJobResponse::recoveryPointArn))
            .setter(setter(Builder::recoveryPointArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RecoveryPointArn").build()).build();

    private static final SdkField<String> RESOURCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceArn").getter(getter(DescribeBackupJobResponse::resourceArn))
            .setter(setter(Builder::resourceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceArn").build()).build();

    private static final SdkField<Instant> CREATION_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationDate").getter(getter(DescribeBackupJobResponse::creationDate))
            .setter(setter(Builder::creationDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationDate").build()).build();

    private static final SdkField<Instant> COMPLETION_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CompletionDate").getter(getter(DescribeBackupJobResponse::completionDate))
            .setter(setter(Builder::completionDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CompletionDate").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("State")
            .getter(getter(DescribeBackupJobResponse::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State").build()).build();

    private static final SdkField<String> STATUS_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StatusMessage").getter(getter(DescribeBackupJobResponse::statusMessage))
            .setter(setter(Builder::statusMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusMessage").build()).build();

    private static final SdkField<String> PERCENT_DONE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PercentDone").getter(getter(DescribeBackupJobResponse::percentDone))
            .setter(setter(Builder::percentDone))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PercentDone").build()).build();

    private static final SdkField<Long> BACKUP_SIZE_IN_BYTES_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("BackupSizeInBytes").getter(getter(DescribeBackupJobResponse::backupSizeInBytes))
            .setter(setter(Builder::backupSizeInBytes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BackupSizeInBytes").build()).build();

    private static final SdkField<String> IAM_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IamRoleArn").getter(getter(DescribeBackupJobResponse::iamRoleArn)).setter(setter(Builder::iamRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IamRoleArn").build()).build();

    private static final SdkField<RecoveryPointCreator> CREATED_BY_FIELD = SdkField
            .<RecoveryPointCreator> builder(MarshallingType.SDK_POJO).memberName("CreatedBy")
            .getter(getter(DescribeBackupJobResponse::createdBy)).setter(setter(Builder::createdBy))
            .constructor(RecoveryPointCreator::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedBy").build()).build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceType").getter(getter(DescribeBackupJobResponse::resourceType))
            .setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceType").build()).build();

    private static final SdkField<Long> BYTES_TRANSFERRED_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("BytesTransferred").getter(getter(DescribeBackupJobResponse::bytesTransferred))
            .setter(setter(Builder::bytesTransferred))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BytesTransferred").build()).build();

    private static final SdkField<Instant> EXPECTED_COMPLETION_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("ExpectedCompletionDate").getter(getter(DescribeBackupJobResponse::expectedCompletionDate))
            .setter(setter(Builder::expectedCompletionDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExpectedCompletionDate").build())
            .build();

    private static final SdkField<Instant> START_BY_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartBy").getter(getter(DescribeBackupJobResponse::startBy)).setter(setter(Builder::startBy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartBy").build()).build();

    private static final SdkField<Map<String, String>> BACKUP_OPTIONS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("BackupOptions")
            .getter(getter(DescribeBackupJobResponse::backupOptions))
            .setter(setter(Builder::backupOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BackupOptions").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> BACKUP_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BackupType").getter(getter(DescribeBackupJobResponse::backupType)).setter(setter(Builder::backupType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BackupType").build()).build();

    private static final SdkField<String> PARENT_JOB_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ParentJobId").getter(getter(DescribeBackupJobResponse::parentJobId))
            .setter(setter(Builder::parentJobId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ParentJobId").build()).build();

    private static final SdkField<Boolean> IS_PARENT_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("IsParent").getter(getter(DescribeBackupJobResponse::isParent)).setter(setter(Builder::isParent))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IsParent").build()).build();

    private static final SdkField<Long> NUMBER_OF_CHILD_JOBS_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("NumberOfChildJobs").getter(getter(DescribeBackupJobResponse::numberOfChildJobs))
            .setter(setter(Builder::numberOfChildJobs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NumberOfChildJobs").build()).build();

    private static final SdkField<Map<String, Long>> CHILD_JOBS_IN_STATE_FIELD = SdkField
            .<Map<String, Long>> builder(MarshallingType.MAP)
            .memberName("ChildJobsInState")
            .getter(getter(DescribeBackupJobResponse::childJobsInStateAsStrings))
            .setter(setter(Builder::childJobsInStateWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ChildJobsInState").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<Long> builder(MarshallingType.LONG)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> RESOURCE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceName").getter(getter(DescribeBackupJobResponse::resourceName))
            .setter(setter(Builder::resourceName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceName").build()).build();

    private static final SdkField<Instant> INITIATION_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("InitiationDate").getter(getter(DescribeBackupJobResponse::initiationDate))
            .setter(setter(Builder::initiationDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InitiationDate").build()).build();

    private static final SdkField<String> MESSAGE_CATEGORY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MessageCategory").getter(getter(DescribeBackupJobResponse::messageCategory))
            .setter(setter(Builder::messageCategory))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MessageCategory").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCOUNT_ID_FIELD,
            BACKUP_JOB_ID_FIELD, BACKUP_VAULT_NAME_FIELD, BACKUP_VAULT_ARN_FIELD, RECOVERY_POINT_ARN_FIELD, RESOURCE_ARN_FIELD,
            CREATION_DATE_FIELD, COMPLETION_DATE_FIELD, STATE_FIELD, STATUS_MESSAGE_FIELD, PERCENT_DONE_FIELD,
            BACKUP_SIZE_IN_BYTES_FIELD, IAM_ROLE_ARN_FIELD, CREATED_BY_FIELD, RESOURCE_TYPE_FIELD, BYTES_TRANSFERRED_FIELD,
            EXPECTED_COMPLETION_DATE_FIELD, START_BY_FIELD, BACKUP_OPTIONS_FIELD, BACKUP_TYPE_FIELD, PARENT_JOB_ID_FIELD,
            IS_PARENT_FIELD, NUMBER_OF_CHILD_JOBS_FIELD, CHILD_JOBS_IN_STATE_FIELD, RESOURCE_NAME_FIELD, INITIATION_DATE_FIELD,
            MESSAGE_CATEGORY_FIELD));

    private final String accountId;

    private final String backupJobId;

    private final String backupVaultName;

    private final String backupVaultArn;

    private final String recoveryPointArn;

    private final String resourceArn;

    private final Instant creationDate;

    private final Instant completionDate;

    private final String state;

    private final String statusMessage;

    private final String percentDone;

    private final Long backupSizeInBytes;

    private final String iamRoleArn;

    private final RecoveryPointCreator createdBy;

    private final String resourceType;

    private final Long bytesTransferred;

    private final Instant expectedCompletionDate;

    private final Instant startBy;

    private final Map<String, String> backupOptions;

    private final String backupType;

    private final String parentJobId;

    private final Boolean isParent;

    private final Long numberOfChildJobs;

    private final Map<String, Long> childJobsInState;

    private final String resourceName;

    private final Instant initiationDate;

    private final String messageCategory;

    private DescribeBackupJobResponse(BuilderImpl builder) {
        super(builder);
        this.accountId = builder.accountId;
        this.backupJobId = builder.backupJobId;
        this.backupVaultName = builder.backupVaultName;
        this.backupVaultArn = builder.backupVaultArn;
        this.recoveryPointArn = builder.recoveryPointArn;
        this.resourceArn = builder.resourceArn;
        this.creationDate = builder.creationDate;
        this.completionDate = builder.completionDate;
        this.state = builder.state;
        this.statusMessage = builder.statusMessage;
        this.percentDone = builder.percentDone;
        this.backupSizeInBytes = builder.backupSizeInBytes;
        this.iamRoleArn = builder.iamRoleArn;
        this.createdBy = builder.createdBy;
        this.resourceType = builder.resourceType;
        this.bytesTransferred = builder.bytesTransferred;
        this.expectedCompletionDate = builder.expectedCompletionDate;
        this.startBy = builder.startBy;
        this.backupOptions = builder.backupOptions;
        this.backupType = builder.backupType;
        this.parentJobId = builder.parentJobId;
        this.isParent = builder.isParent;
        this.numberOfChildJobs = builder.numberOfChildJobs;
        this.childJobsInState = builder.childJobsInState;
        this.resourceName = builder.resourceName;
        this.initiationDate = builder.initiationDate;
        this.messageCategory = builder.messageCategory;
    }

    /**
     * <p>
     * Returns the account ID that owns the backup job.
     * </p>
     * 
     * @return Returns the account ID that owns the backup job.
     */
    public final String accountId() {
        return accountId;
    }

    /**
     * <p>
     * Uniquely identifies a request to Backup to back up a resource.
     * </p>
     * 
     * @return Uniquely identifies a request to Backup to back up a resource.
     */
    public final String backupJobId() {
        return backupJobId;
    }

    /**
     * <p>
     * The name of a logical container where backups are stored. Backup vaults are identified by names that are unique
     * to the account used to create them and the Amazon Web Services Region where they are created. They consist of
     * lowercase letters, numbers, and hyphens.
     * </p>
     * 
     * @return The name of a logical container where backups are stored. Backup vaults are identified by names that are
     *         unique to the account used to create them and the Amazon Web Services Region where they are created. They
     *         consist of lowercase letters, numbers, and hyphens.
     */
    public final String backupVaultName() {
        return backupVaultName;
    }

    /**
     * <p>
     * An Amazon Resource Name (ARN) that uniquely identifies a backup vault; for example,
     * <code>arn:aws:backup:us-east-1:123456789012:vault:aBackupVault</code>.
     * </p>
     * 
     * @return An Amazon Resource Name (ARN) that uniquely identifies a backup vault; for example,
     *         <code>arn:aws:backup:us-east-1:123456789012:vault:aBackupVault</code>.
     */
    public final String backupVaultArn() {
        return backupVaultArn;
    }

    /**
     * <p>
     * An ARN that uniquely identifies a recovery point; for example,
     * <code>arn:aws:backup:us-east-1:123456789012:recovery-point:1EB3B5E7-9EB0-435A-A80B-108B488B0D45</code>.
     * </p>
     * 
     * @return An ARN that uniquely identifies a recovery point; for example,
     *         <code>arn:aws:backup:us-east-1:123456789012:recovery-point:1EB3B5E7-9EB0-435A-A80B-108B488B0D45</code>.
     */
    public final String recoveryPointArn() {
        return recoveryPointArn;
    }

    /**
     * <p>
     * An ARN that uniquely identifies a saved resource. The format of the ARN depends on the resource type.
     * </p>
     * 
     * @return An ARN that uniquely identifies a saved resource. The format of the ARN depends on the resource type.
     */
    public final String resourceArn() {
        return resourceArn;
    }

    /**
     * <p>
     * The date and time that a backup job is created, in Unix format and Coordinated Universal Time (UTC). The value of
     * <code>CreationDate</code> is accurate to milliseconds. For example, the value 1516925490.087 represents Friday,
     * January 26, 2018 12:11:30.087 AM.
     * </p>
     * 
     * @return The date and time that a backup job is created, in Unix format and Coordinated Universal Time (UTC). The
     *         value of <code>CreationDate</code> is accurate to milliseconds. For example, the value 1516925490.087
     *         represents Friday, January 26, 2018 12:11:30.087 AM.
     */
    public final Instant creationDate() {
        return creationDate;
    }

    /**
     * <p>
     * The date and time that a job to create a backup job is completed, in Unix format and Coordinated Universal Time
     * (UTC). The value of <code>CompletionDate</code> is accurate to milliseconds. For example, the value
     * 1516925490.087 represents Friday, January 26, 2018 12:11:30.087 AM.
     * </p>
     * 
     * @return The date and time that a job to create a backup job is completed, in Unix format and Coordinated
     *         Universal Time (UTC). The value of <code>CompletionDate</code> is accurate to milliseconds. For example,
     *         the value 1516925490.087 represents Friday, January 26, 2018 12:11:30.087 AM.
     */
    public final Instant completionDate() {
        return completionDate;
    }

    /**
     * <p>
     * The current state of a backup job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link BackupJobState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of a backup job.
     * @see BackupJobState
     */
    public final BackupJobState state() {
        return BackupJobState.fromValue(state);
    }

    /**
     * <p>
     * The current state of a backup job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link BackupJobState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of a backup job.
     * @see BackupJobState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * A detailed message explaining the status of the job to back up a resource.
     * </p>
     * 
     * @return A detailed message explaining the status of the job to back up a resource.
     */
    public final String statusMessage() {
        return statusMessage;
    }

    /**
     * <p>
     * Contains an estimated percentage that is complete of a job at the time the job status was queried.
     * </p>
     * 
     * @return Contains an estimated percentage that is complete of a job at the time the job status was queried.
     */
    public final String percentDone() {
        return percentDone;
    }

    /**
     * <p>
     * The size, in bytes, of a backup.
     * </p>
     * 
     * @return The size, in bytes, of a backup.
     */
    public final Long backupSizeInBytes() {
        return backupSizeInBytes;
    }

    /**
     * <p>
     * Specifies the IAM role ARN used to create the target recovery point; for example,
     * <code>arn:aws:iam::123456789012:role/S3Access</code>.
     * </p>
     * 
     * @return Specifies the IAM role ARN used to create the target recovery point; for example,
     *         <code>arn:aws:iam::123456789012:role/S3Access</code>.
     */
    public final String iamRoleArn() {
        return iamRoleArn;
    }

    /**
     * <p>
     * Contains identifying information about the creation of a backup job, including the <code>BackupPlanArn</code>,
     * <code>BackupPlanId</code>, <code>BackupPlanVersion</code>, and <code>BackupRuleId</code> of the backup plan that
     * is used to create it.
     * </p>
     * 
     * @return Contains identifying information about the creation of a backup job, including the
     *         <code>BackupPlanArn</code>, <code>BackupPlanId</code>, <code>BackupPlanVersion</code>, and
     *         <code>BackupRuleId</code> of the backup plan that is used to create it.
     */
    public final RecoveryPointCreator createdBy() {
        return createdBy;
    }

    /**
     * <p>
     * The type of Amazon Web Services resource to be backed up; for example, an Amazon Elastic Block Store (Amazon EBS)
     * volume or an Amazon Relational Database Service (Amazon RDS) database.
     * </p>
     * 
     * @return The type of Amazon Web Services resource to be backed up; for example, an Amazon Elastic Block Store
     *         (Amazon EBS) volume or an Amazon Relational Database Service (Amazon RDS) database.
     */
    public final String resourceType() {
        return resourceType;
    }

    /**
     * <p>
     * The size in bytes transferred to a backup vault at the time that the job status was queried.
     * </p>
     * 
     * @return The size in bytes transferred to a backup vault at the time that the job status was queried.
     */
    public final Long bytesTransferred() {
        return bytesTransferred;
    }

    /**
     * <p>
     * The date and time that a job to back up resources is expected to be completed, in Unix format and Coordinated
     * Universal Time (UTC). The value of <code>ExpectedCompletionDate</code> is accurate to milliseconds. For example,
     * the value 1516925490.087 represents Friday, January 26, 2018 12:11:30.087 AM.
     * </p>
     * 
     * @return The date and time that a job to back up resources is expected to be completed, in Unix format and
     *         Coordinated Universal Time (UTC). The value of <code>ExpectedCompletionDate</code> is accurate to
     *         milliseconds. For example, the value 1516925490.087 represents Friday, January 26, 2018 12:11:30.087 AM.
     */
    public final Instant expectedCompletionDate() {
        return expectedCompletionDate;
    }

    /**
     * <p>
     * Specifies the time in Unix format and Coordinated Universal Time (UTC) when a backup job must be started before
     * it is canceled. The value is calculated by adding the start window to the scheduled time. So if the scheduled
     * time were 6:00 PM and the start window is 2 hours, the <code>StartBy</code> time would be 8:00 PM on the date
     * specified. The value of <code>StartBy</code> is accurate to milliseconds. For example, the value 1516925490.087
     * represents Friday, January 26, 2018 12:11:30.087 AM.
     * </p>
     * 
     * @return Specifies the time in Unix format and Coordinated Universal Time (UTC) when a backup job must be started
     *         before it is canceled. The value is calculated by adding the start window to the scheduled time. So if
     *         the scheduled time were 6:00 PM and the start window is 2 hours, the <code>StartBy</code> time would be
     *         8:00 PM on the date specified. The value of <code>StartBy</code> is accurate to milliseconds. For
     *         example, the value 1516925490.087 represents Friday, January 26, 2018 12:11:30.087 AM.
     */
    public final Instant startBy() {
        return startBy;
    }

    /**
     * For responses, this returns true if the service returned a value for the BackupOptions property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasBackupOptions() {
        return backupOptions != null && !(backupOptions instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Represents the options specified as part of backup plan or on-demand backup job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasBackupOptions} method.
     * </p>
     * 
     * @return Represents the options specified as part of backup plan or on-demand backup job.
     */
    public final Map<String, String> backupOptions() {
        return backupOptions;
    }

    /**
     * <p>
     * Represents the actual backup type selected for a backup job. For example, if a successful Windows Volume Shadow
     * Copy Service (VSS) backup was taken, <code>BackupType</code> returns <code>"WindowsVSS"</code>. If
     * <code>BackupType</code> is empty, then the backup type was a regular backup.
     * </p>
     * 
     * @return Represents the actual backup type selected for a backup job. For example, if a successful Windows Volume
     *         Shadow Copy Service (VSS) backup was taken, <code>BackupType</code> returns <code>"WindowsVSS"</code>. If
     *         <code>BackupType</code> is empty, then the backup type was a regular backup.
     */
    public final String backupType() {
        return backupType;
    }

    /**
     * <p>
     * This returns the parent (composite) resource backup job ID.
     * </p>
     * 
     * @return This returns the parent (composite) resource backup job ID.
     */
    public final String parentJobId() {
        return parentJobId;
    }

    /**
     * <p>
     * This returns the boolean value that a backup job is a parent (composite) job.
     * </p>
     * 
     * @return This returns the boolean value that a backup job is a parent (composite) job.
     */
    public final Boolean isParent() {
        return isParent;
    }

    /**
     * <p>
     * This returns the number of child (nested) backup jobs.
     * </p>
     * 
     * @return This returns the number of child (nested) backup jobs.
     */
    public final Long numberOfChildJobs() {
        return numberOfChildJobs;
    }

    /**
     * <p>
     * This returns the statistics of the included child (nested) backup jobs.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasChildJobsInState} method.
     * </p>
     * 
     * @return This returns the statistics of the included child (nested) backup jobs.
     */
    public final Map<BackupJobState, Long> childJobsInState() {
        return BackupJobChildJobsInStateCopier.copyStringToEnum(childJobsInState);
    }

    /**
     * For responses, this returns true if the service returned a value for the ChildJobsInState property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasChildJobsInState() {
        return childJobsInState != null && !(childJobsInState instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * This returns the statistics of the included child (nested) backup jobs.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasChildJobsInState} method.
     * </p>
     * 
     * @return This returns the statistics of the included child (nested) backup jobs.
     */
    public final Map<String, Long> childJobsInStateAsStrings() {
        return childJobsInState;
    }

    /**
     * <p>
     * This is the non-unique name of the resource that belongs to the specified backup.
     * </p>
     * 
     * @return This is the non-unique name of the resource that belongs to the specified backup.
     */
    public final String resourceName() {
        return resourceName;
    }

    /**
     * <p>
     * This is the date a backup job was initiated.
     * </p>
     * 
     * @return This is the date a backup job was initiated.
     */
    public final Instant initiationDate() {
        return initiationDate;
    }

    /**
     * <p>
     * This is the job count for the specified message category.
     * </p>
     * <p>
     * Example strings may include <code>AccessDenied</code>, <code>SUCCESS</code>, <code>AGGREGATE_ALL</code>, and
     * <code>INVALIDPARAMETERS</code>. View <a
     * href="https://docs.aws.amazon.com/aws-backup/latest/devguide/monitoring.html">Monitoring</a> for a list of
     * accepted MessageCategory strings.
     * </p>
     * 
     * @return This is the job count for the specified message category.</p>
     *         <p>
     *         Example strings may include <code>AccessDenied</code>, <code>SUCCESS</code>, <code>AGGREGATE_ALL</code>,
     *         and <code>INVALIDPARAMETERS</code>. View <a
     *         href="https://docs.aws.amazon.com/aws-backup/latest/devguide/monitoring.html">Monitoring</a> for a list
     *         of accepted MessageCategory strings.
     */
    public final String messageCategory() {
        return messageCategory;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(backupJobId());
        hashCode = 31 * hashCode + Objects.hashCode(backupVaultName());
        hashCode = 31 * hashCode + Objects.hashCode(backupVaultArn());
        hashCode = 31 * hashCode + Objects.hashCode(recoveryPointArn());
        hashCode = 31 * hashCode + Objects.hashCode(resourceArn());
        hashCode = 31 * hashCode + Objects.hashCode(creationDate());
        hashCode = 31 * hashCode + Objects.hashCode(completionDate());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusMessage());
        hashCode = 31 * hashCode + Objects.hashCode(percentDone());
        hashCode = 31 * hashCode + Objects.hashCode(backupSizeInBytes());
        hashCode = 31 * hashCode + Objects.hashCode(iamRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(createdBy());
        hashCode = 31 * hashCode + Objects.hashCode(resourceType());
        hashCode = 31 * hashCode + Objects.hashCode(bytesTransferred());
        hashCode = 31 * hashCode + Objects.hashCode(expectedCompletionDate());
        hashCode = 31 * hashCode + Objects.hashCode(startBy());
        hashCode = 31 * hashCode + Objects.hashCode(hasBackupOptions() ? backupOptions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(backupType());
        hashCode = 31 * hashCode + Objects.hashCode(parentJobId());
        hashCode = 31 * hashCode + Objects.hashCode(isParent());
        hashCode = 31 * hashCode + Objects.hashCode(numberOfChildJobs());
        hashCode = 31 * hashCode + Objects.hashCode(hasChildJobsInState() ? childJobsInStateAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(resourceName());
        hashCode = 31 * hashCode + Objects.hashCode(initiationDate());
        hashCode = 31 * hashCode + Objects.hashCode(messageCategory());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DescribeBackupJobResponse)) {
            return false;
        }
        DescribeBackupJobResponse other = (DescribeBackupJobResponse) obj;
        return Objects.equals(accountId(), other.accountId()) && Objects.equals(backupJobId(), other.backupJobId())
                && Objects.equals(backupVaultName(), other.backupVaultName())
                && Objects.equals(backupVaultArn(), other.backupVaultArn())
                && Objects.equals(recoveryPointArn(), other.recoveryPointArn())
                && Objects.equals(resourceArn(), other.resourceArn()) && Objects.equals(creationDate(), other.creationDate())
                && Objects.equals(completionDate(), other.completionDate())
                && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(statusMessage(), other.statusMessage()) && Objects.equals(percentDone(), other.percentDone())
                && Objects.equals(backupSizeInBytes(), other.backupSizeInBytes())
                && Objects.equals(iamRoleArn(), other.iamRoleArn()) && Objects.equals(createdBy(), other.createdBy())
                && Objects.equals(resourceType(), other.resourceType())
                && Objects.equals(bytesTransferred(), other.bytesTransferred())
                && Objects.equals(expectedCompletionDate(), other.expectedCompletionDate())
                && Objects.equals(startBy(), other.startBy()) && hasBackupOptions() == other.hasBackupOptions()
                && Objects.equals(backupOptions(), other.backupOptions()) && Objects.equals(backupType(), other.backupType())
                && Objects.equals(parentJobId(), other.parentJobId()) && Objects.equals(isParent(), other.isParent())
                && Objects.equals(numberOfChildJobs(), other.numberOfChildJobs())
                && hasChildJobsInState() == other.hasChildJobsInState()
                && Objects.equals(childJobsInStateAsStrings(), other.childJobsInStateAsStrings())
                && Objects.equals(resourceName(), other.resourceName())
                && Objects.equals(initiationDate(), other.initiationDate())
                && Objects.equals(messageCategory(), other.messageCategory());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DescribeBackupJobResponse").add("AccountId", accountId()).add("BackupJobId", backupJobId())
                .add("BackupVaultName", backupVaultName()).add("BackupVaultArn", backupVaultArn())
                .add("RecoveryPointArn", recoveryPointArn()).add("ResourceArn", resourceArn())
                .add("CreationDate", creationDate()).add("CompletionDate", completionDate()).add("State", stateAsString())
                .add("StatusMessage", statusMessage()).add("PercentDone", percentDone())
                .add("BackupSizeInBytes", backupSizeInBytes()).add("IamRoleArn", iamRoleArn()).add("CreatedBy", createdBy())
                .add("ResourceType", resourceType()).add("BytesTransferred", bytesTransferred())
                .add("ExpectedCompletionDate", expectedCompletionDate()).add("StartBy", startBy())
                .add("BackupOptions", hasBackupOptions() ? backupOptions() : null).add("BackupType", backupType())
                .add("ParentJobId", parentJobId()).add("IsParent", isParent()).add("NumberOfChildJobs", numberOfChildJobs())
                .add("ChildJobsInState", hasChildJobsInState() ? childJobsInStateAsStrings() : null)
                .add("ResourceName", resourceName()).add("InitiationDate", initiationDate())
                .add("MessageCategory", messageCategory()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AccountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "BackupJobId":
            return Optional.ofNullable(clazz.cast(backupJobId()));
        case "BackupVaultName":
            return Optional.ofNullable(clazz.cast(backupVaultName()));
        case "BackupVaultArn":
            return Optional.ofNullable(clazz.cast(backupVaultArn()));
        case "RecoveryPointArn":
            return Optional.ofNullable(clazz.cast(recoveryPointArn()));
        case "ResourceArn":
            return Optional.ofNullable(clazz.cast(resourceArn()));
        case "CreationDate":
            return Optional.ofNullable(clazz.cast(creationDate()));
        case "CompletionDate":
            return Optional.ofNullable(clazz.cast(completionDate()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "StatusMessage":
            return Optional.ofNullable(clazz.cast(statusMessage()));
        case "PercentDone":
            return Optional.ofNullable(clazz.cast(percentDone()));
        case "BackupSizeInBytes":
            return Optional.ofNullable(clazz.cast(backupSizeInBytes()));
        case "IamRoleArn":
            return Optional.ofNullable(clazz.cast(iamRoleArn()));
        case "CreatedBy":
            return Optional.ofNullable(clazz.cast(createdBy()));
        case "ResourceType":
            return Optional.ofNullable(clazz.cast(resourceType()));
        case "BytesTransferred":
            return Optional.ofNullable(clazz.cast(bytesTransferred()));
        case "ExpectedCompletionDate":
            return Optional.ofNullable(clazz.cast(expectedCompletionDate()));
        case "StartBy":
            return Optional.ofNullable(clazz.cast(startBy()));
        case "BackupOptions":
            return Optional.ofNullable(clazz.cast(backupOptions()));
        case "BackupType":
            return Optional.ofNullable(clazz.cast(backupType()));
        case "ParentJobId":
            return Optional.ofNullable(clazz.cast(parentJobId()));
        case "IsParent":
            return Optional.ofNullable(clazz.cast(isParent()));
        case "NumberOfChildJobs":
            return Optional.ofNullable(clazz.cast(numberOfChildJobs()));
        case "ChildJobsInState":
            return Optional.ofNullable(clazz.cast(childJobsInStateAsStrings()));
        case "ResourceName":
            return Optional.ofNullable(clazz.cast(resourceName()));
        case "InitiationDate":
            return Optional.ofNullable(clazz.cast(initiationDate()));
        case "MessageCategory":
            return Optional.ofNullable(clazz.cast(messageCategory()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DescribeBackupJobResponse, T> g) {
        return obj -> g.apply((DescribeBackupJobResponse) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends BackupResponse.Builder, SdkPojo, CopyableBuilder<Builder, DescribeBackupJobResponse> {
        /**
         * <p>
         * Returns the account ID that owns the backup job.
         * </p>
         * 
         * @param accountId
         *        Returns the account ID that owns the backup job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * Uniquely identifies a request to Backup to back up a resource.
         * </p>
         * 
         * @param backupJobId
         *        Uniquely identifies a request to Backup to back up a resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backupJobId(String backupJobId);

        /**
         * <p>
         * The name of a logical container where backups are stored. Backup vaults are identified by names that are
         * unique to the account used to create them and the Amazon Web Services Region where they are created. They
         * consist of lowercase letters, numbers, and hyphens.
         * </p>
         * 
         * @param backupVaultName
         *        The name of a logical container where backups are stored. Backup vaults are identified by names that
         *        are unique to the account used to create them and the Amazon Web Services Region where they are
         *        created. They consist of lowercase letters, numbers, and hyphens.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backupVaultName(String backupVaultName);

        /**
         * <p>
         * An Amazon Resource Name (ARN) that uniquely identifies a backup vault; for example,
         * <code>arn:aws:backup:us-east-1:123456789012:vault:aBackupVault</code>.
         * </p>
         * 
         * @param backupVaultArn
         *        An Amazon Resource Name (ARN) that uniquely identifies a backup vault; for example,
         *        <code>arn:aws:backup:us-east-1:123456789012:vault:aBackupVault</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backupVaultArn(String backupVaultArn);

        /**
         * <p>
         * An ARN that uniquely identifies a recovery point; for example,
         * <code>arn:aws:backup:us-east-1:123456789012:recovery-point:1EB3B5E7-9EB0-435A-A80B-108B488B0D45</code>.
         * </p>
         * 
         * @param recoveryPointArn
         *        An ARN that uniquely identifies a recovery point; for example,
         *        <code>arn:aws:backup:us-east-1:123456789012:recovery-point:1EB3B5E7-9EB0-435A-A80B-108B488B0D45</code>
         *        .
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recoveryPointArn(String recoveryPointArn);

        /**
         * <p>
         * An ARN that uniquely identifies a saved resource. The format of the ARN depends on the resource type.
         * </p>
         * 
         * @param resourceArn
         *        An ARN that uniquely identifies a saved resource. The format of the ARN depends on the resource type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceArn(String resourceArn);

        /**
         * <p>
         * The date and time that a backup job is created, in Unix format and Coordinated Universal Time (UTC). The
         * value of <code>CreationDate</code> is accurate to milliseconds. For example, the value 1516925490.087
         * represents Friday, January 26, 2018 12:11:30.087 AM.
         * </p>
         * 
         * @param creationDate
         *        The date and time that a backup job is created, in Unix format and Coordinated Universal Time (UTC).
         *        The value of <code>CreationDate</code> is accurate to milliseconds. For example, the value
         *        1516925490.087 represents Friday, January 26, 2018 12:11:30.087 AM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationDate(Instant creationDate);

        /**
         * <p>
         * The date and time that a job to create a backup job is completed, in Unix format and Coordinated Universal
         * Time (UTC). The value of <code>CompletionDate</code> is accurate to milliseconds. For example, the value
         * 1516925490.087 represents Friday, January 26, 2018 12:11:30.087 AM.
         * </p>
         * 
         * @param completionDate
         *        The date and time that a job to create a backup job is completed, in Unix format and Coordinated
         *        Universal Time (UTC). The value of <code>CompletionDate</code> is accurate to milliseconds. For
         *        example, the value 1516925490.087 represents Friday, January 26, 2018 12:11:30.087 AM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder completionDate(Instant completionDate);

        /**
         * <p>
         * The current state of a backup job.
         * </p>
         * 
         * @param state
         *        The current state of a backup job.
         * @see BackupJobState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BackupJobState
         */
        Builder state(String state);

        /**
         * <p>
         * The current state of a backup job.
         * </p>
         * 
         * @param state
         *        The current state of a backup job.
         * @see BackupJobState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BackupJobState
         */
        Builder state(BackupJobState state);

        /**
         * <p>
         * A detailed message explaining the status of the job to back up a resource.
         * </p>
         * 
         * @param statusMessage
         *        A detailed message explaining the status of the job to back up a resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusMessage(String statusMessage);

        /**
         * <p>
         * Contains an estimated percentage that is complete of a job at the time the job status was queried.
         * </p>
         * 
         * @param percentDone
         *        Contains an estimated percentage that is complete of a job at the time the job status was queried.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder percentDone(String percentDone);

        /**
         * <p>
         * The size, in bytes, of a backup.
         * </p>
         * 
         * @param backupSizeInBytes
         *        The size, in bytes, of a backup.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backupSizeInBytes(Long backupSizeInBytes);

        /**
         * <p>
         * Specifies the IAM role ARN used to create the target recovery point; for example,
         * <code>arn:aws:iam::123456789012:role/S3Access</code>.
         * </p>
         * 
         * @param iamRoleArn
         *        Specifies the IAM role ARN used to create the target recovery point; for example,
         *        <code>arn:aws:iam::123456789012:role/S3Access</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iamRoleArn(String iamRoleArn);

        /**
         * <p>
         * Contains identifying information about the creation of a backup job, including the <code>BackupPlanArn</code>, <code>BackupPlanId</code>, <code>BackupPlanVersion</code>, and <code>BackupRuleId</code> of the backup plan
         * that is used to create it.
         * </p>
         * 
         * @param createdBy
         *        Contains identifying information about the creation of a backup job, including the
         *        <code>BackupPlanArn</code>, <code>BackupPlanId</code>, <code>BackupPlanVersion</code>, and
         *        <code>BackupRuleId</code> of the backup plan that is used to create it.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdBy(RecoveryPointCreator createdBy);

        /**
         * <p>
         * Contains identifying information about the creation of a backup job, including the <code>BackupPlanArn</code>, <code>BackupPlanId</code>, <code>BackupPlanVersion</code>, and <code>BackupRuleId</code> of the backup plan
         * that is used to create it.
         * </p>
         * This is a convenience method that creates an instance of the {@link RecoveryPointCreator.Builder} avoiding
         * the need to create one manually via {@link RecoveryPointCreator#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RecoveryPointCreator.Builder#build()} is called immediately and
         * its result is passed to {@link #createdBy(RecoveryPointCreator)}.
         * 
         * @param createdBy
         *        a consumer that will call methods on {@link RecoveryPointCreator.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #createdBy(RecoveryPointCreator)
         */
        default Builder createdBy(Consumer<RecoveryPointCreator.Builder> createdBy) {
            return createdBy(RecoveryPointCreator.builder().applyMutation(createdBy).build());
        }

        /**
         * <p>
         * The type of Amazon Web Services resource to be backed up; for example, an Amazon Elastic Block Store (Amazon
         * EBS) volume or an Amazon Relational Database Service (Amazon RDS) database.
         * </p>
         * 
         * @param resourceType
         *        The type of Amazon Web Services resource to be backed up; for example, an Amazon Elastic Block Store
         *        (Amazon EBS) volume or an Amazon Relational Database Service (Amazon RDS) database.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The size in bytes transferred to a backup vault at the time that the job status was queried.
         * </p>
         * 
         * @param bytesTransferred
         *        The size in bytes transferred to a backup vault at the time that the job status was queried.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bytesTransferred(Long bytesTransferred);

        /**
         * <p>
         * The date and time that a job to back up resources is expected to be completed, in Unix format and Coordinated
         * Universal Time (UTC). The value of <code>ExpectedCompletionDate</code> is accurate to milliseconds. For
         * example, the value 1516925490.087 represents Friday, January 26, 2018 12:11:30.087 AM.
         * </p>
         * 
         * @param expectedCompletionDate
         *        The date and time that a job to back up resources is expected to be completed, in Unix format and
         *        Coordinated Universal Time (UTC). The value of <code>ExpectedCompletionDate</code> is accurate to
         *        milliseconds. For example, the value 1516925490.087 represents Friday, January 26, 2018 12:11:30.087
         *        AM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expectedCompletionDate(Instant expectedCompletionDate);

        /**
         * <p>
         * Specifies the time in Unix format and Coordinated Universal Time (UTC) when a backup job must be started
         * before it is canceled. The value is calculated by adding the start window to the scheduled time. So if the
         * scheduled time were 6:00 PM and the start window is 2 hours, the <code>StartBy</code> time would be 8:00 PM
         * on the date specified. The value of <code>StartBy</code> is accurate to milliseconds. For example, the value
         * 1516925490.087 represents Friday, January 26, 2018 12:11:30.087 AM.
         * </p>
         * 
         * @param startBy
         *        Specifies the time in Unix format and Coordinated Universal Time (UTC) when a backup job must be
         *        started before it is canceled. The value is calculated by adding the start window to the scheduled
         *        time. So if the scheduled time were 6:00 PM and the start window is 2 hours, the <code>StartBy</code>
         *        time would be 8:00 PM on the date specified. The value of <code>StartBy</code> is accurate to
         *        milliseconds. For example, the value 1516925490.087 represents Friday, January 26, 2018 12:11:30.087
         *        AM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startBy(Instant startBy);

        /**
         * <p>
         * Represents the options specified as part of backup plan or on-demand backup job.
         * </p>
         * 
         * @param backupOptions
         *        Represents the options specified as part of backup plan or on-demand backup job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backupOptions(Map<String, String> backupOptions);

        /**
         * <p>
         * Represents the actual backup type selected for a backup job. For example, if a successful Windows Volume
         * Shadow Copy Service (VSS) backup was taken, <code>BackupType</code> returns <code>"WindowsVSS"</code>. If
         * <code>BackupType</code> is empty, then the backup type was a regular backup.
         * </p>
         * 
         * @param backupType
         *        Represents the actual backup type selected for a backup job. For example, if a successful Windows
         *        Volume Shadow Copy Service (VSS) backup was taken, <code>BackupType</code> returns
         *        <code>"WindowsVSS"</code>. If <code>BackupType</code> is empty, then the backup type was a regular
         *        backup.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backupType(String backupType);

        /**
         * <p>
         * This returns the parent (composite) resource backup job ID.
         * </p>
         * 
         * @param parentJobId
         *        This returns the parent (composite) resource backup job ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parentJobId(String parentJobId);

        /**
         * <p>
         * This returns the boolean value that a backup job is a parent (composite) job.
         * </p>
         * 
         * @param isParent
         *        This returns the boolean value that a backup job is a parent (composite) job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isParent(Boolean isParent);

        /**
         * <p>
         * This returns the number of child (nested) backup jobs.
         * </p>
         * 
         * @param numberOfChildJobs
         *        This returns the number of child (nested) backup jobs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder numberOfChildJobs(Long numberOfChildJobs);

        /**
         * <p>
         * This returns the statistics of the included child (nested) backup jobs.
         * </p>
         * 
         * @param childJobsInState
         *        This returns the statistics of the included child (nested) backup jobs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder childJobsInStateWithStrings(Map<String, Long> childJobsInState);

        /**
         * <p>
         * This returns the statistics of the included child (nested) backup jobs.
         * </p>
         * 
         * @param childJobsInState
         *        This returns the statistics of the included child (nested) backup jobs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder childJobsInState(Map<BackupJobState, Long> childJobsInState);

        /**
         * <p>
         * This is the non-unique name of the resource that belongs to the specified backup.
         * </p>
         * 
         * @param resourceName
         *        This is the non-unique name of the resource that belongs to the specified backup.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceName(String resourceName);

        /**
         * <p>
         * This is the date a backup job was initiated.
         * </p>
         * 
         * @param initiationDate
         *        This is the date a backup job was initiated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder initiationDate(Instant initiationDate);

        /**
         * <p>
         * This is the job count for the specified message category.
         * </p>
         * <p>
         * Example strings may include <code>AccessDenied</code>, <code>SUCCESS</code>, <code>AGGREGATE_ALL</code>, and
         * <code>INVALIDPARAMETERS</code>. View <a
         * href="https://docs.aws.amazon.com/aws-backup/latest/devguide/monitoring.html">Monitoring</a> for a list of
         * accepted MessageCategory strings.
         * </p>
         * 
         * @param messageCategory
         *        This is the job count for the specified message category.</p>
         *        <p>
         *        Example strings may include <code>AccessDenied</code>, <code>SUCCESS</code>,
         *        <code>AGGREGATE_ALL</code>, and <code>INVALIDPARAMETERS</code>. View <a
         *        href="https://docs.aws.amazon.com/aws-backup/latest/devguide/monitoring.html">Monitoring</a> for a
         *        list of accepted MessageCategory strings.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageCategory(String messageCategory);
    }

    static final class BuilderImpl extends BackupResponse.BuilderImpl implements Builder {
        private String accountId;

        private String backupJobId;

        private String backupVaultName;

        private String backupVaultArn;

        private String recoveryPointArn;

        private String resourceArn;

        private Instant creationDate;

        private Instant completionDate;

        private String state;

        private String statusMessage;

        private String percentDone;

        private Long backupSizeInBytes;

        private String iamRoleArn;

        private RecoveryPointCreator createdBy;

        private String resourceType;

        private Long bytesTransferred;

        private Instant expectedCompletionDate;

        private Instant startBy;

        private Map<String, String> backupOptions = DefaultSdkAutoConstructMap.getInstance();

        private String backupType;

        private String parentJobId;

        private Boolean isParent;

        private Long numberOfChildJobs;

        private Map<String, Long> childJobsInState = DefaultSdkAutoConstructMap.getInstance();

        private String resourceName;

        private Instant initiationDate;

        private String messageCategory;

        private BuilderImpl() {
        }

        private BuilderImpl(DescribeBackupJobResponse model) {
            super(model);
            accountId(model.accountId);
            backupJobId(model.backupJobId);
            backupVaultName(model.backupVaultName);
            backupVaultArn(model.backupVaultArn);
            recoveryPointArn(model.recoveryPointArn);
            resourceArn(model.resourceArn);
            creationDate(model.creationDate);
            completionDate(model.completionDate);
            state(model.state);
            statusMessage(model.statusMessage);
            percentDone(model.percentDone);
            backupSizeInBytes(model.backupSizeInBytes);
            iamRoleArn(model.iamRoleArn);
            createdBy(model.createdBy);
            resourceType(model.resourceType);
            bytesTransferred(model.bytesTransferred);
            expectedCompletionDate(model.expectedCompletionDate);
            startBy(model.startBy);
            backupOptions(model.backupOptions);
            backupType(model.backupType);
            parentJobId(model.parentJobId);
            isParent(model.isParent);
            numberOfChildJobs(model.numberOfChildJobs);
            childJobsInStateWithStrings(model.childJobsInState);
            resourceName(model.resourceName);
            initiationDate(model.initiationDate);
            messageCategory(model.messageCategory);
        }

        public final String getAccountId() {
            return accountId;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final String getBackupJobId() {
            return backupJobId;
        }

        public final void setBackupJobId(String backupJobId) {
            this.backupJobId = backupJobId;
        }

        @Override
        public final Builder backupJobId(String backupJobId) {
            this.backupJobId = backupJobId;
            return this;
        }

        public final String getBackupVaultName() {
            return backupVaultName;
        }

        public final void setBackupVaultName(String backupVaultName) {
            this.backupVaultName = backupVaultName;
        }

        @Override
        public final Builder backupVaultName(String backupVaultName) {
            this.backupVaultName = backupVaultName;
            return this;
        }

        public final String getBackupVaultArn() {
            return backupVaultArn;
        }

        public final void setBackupVaultArn(String backupVaultArn) {
            this.backupVaultArn = backupVaultArn;
        }

        @Override
        public final Builder backupVaultArn(String backupVaultArn) {
            this.backupVaultArn = backupVaultArn;
            return this;
        }

        public final String getRecoveryPointArn() {
            return recoveryPointArn;
        }

        public final void setRecoveryPointArn(String recoveryPointArn) {
            this.recoveryPointArn = recoveryPointArn;
        }

        @Override
        public final Builder recoveryPointArn(String recoveryPointArn) {
            this.recoveryPointArn = recoveryPointArn;
            return this;
        }

        public final String getResourceArn() {
            return resourceArn;
        }

        public final void setResourceArn(String resourceArn) {
            this.resourceArn = resourceArn;
        }

        @Override
        public final Builder resourceArn(String resourceArn) {
            this.resourceArn = resourceArn;
            return this;
        }

        public final Instant getCreationDate() {
            return creationDate;
        }

        public final void setCreationDate(Instant creationDate) {
            this.creationDate = creationDate;
        }

        @Override
        public final Builder creationDate(Instant creationDate) {
            this.creationDate = creationDate;
            return this;
        }

        public final Instant getCompletionDate() {
            return completionDate;
        }

        public final void setCompletionDate(Instant completionDate) {
            this.completionDate = completionDate;
        }

        @Override
        public final Builder completionDate(Instant completionDate) {
            this.completionDate = completionDate;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(BackupJobState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final String getStatusMessage() {
            return statusMessage;
        }

        public final void setStatusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
        }

        @Override
        public final Builder statusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
            return this;
        }

        public final String getPercentDone() {
            return percentDone;
        }

        public final void setPercentDone(String percentDone) {
            this.percentDone = percentDone;
        }

        @Override
        public final Builder percentDone(String percentDone) {
            this.percentDone = percentDone;
            return this;
        }

        public final Long getBackupSizeInBytes() {
            return backupSizeInBytes;
        }

        public final void setBackupSizeInBytes(Long backupSizeInBytes) {
            this.backupSizeInBytes = backupSizeInBytes;
        }

        @Override
        public final Builder backupSizeInBytes(Long backupSizeInBytes) {
            this.backupSizeInBytes = backupSizeInBytes;
            return this;
        }

        public final String getIamRoleArn() {
            return iamRoleArn;
        }

        public final void setIamRoleArn(String iamRoleArn) {
            this.iamRoleArn = iamRoleArn;
        }

        @Override
        public final Builder iamRoleArn(String iamRoleArn) {
            this.iamRoleArn = iamRoleArn;
            return this;
        }

        public final RecoveryPointCreator.Builder getCreatedBy() {
            return createdBy != null ? createdBy.toBuilder() : null;
        }

        public final void setCreatedBy(RecoveryPointCreator.BuilderImpl createdBy) {
            this.createdBy = createdBy != null ? createdBy.build() : null;
        }

        @Override
        public final Builder createdBy(RecoveryPointCreator createdBy) {
            this.createdBy = createdBy;
            return this;
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        public final Long getBytesTransferred() {
            return bytesTransferred;
        }

        public final void setBytesTransferred(Long bytesTransferred) {
            this.bytesTransferred = bytesTransferred;
        }

        @Override
        public final Builder bytesTransferred(Long bytesTransferred) {
            this.bytesTransferred = bytesTransferred;
            return this;
        }

        public final Instant getExpectedCompletionDate() {
            return expectedCompletionDate;
        }

        public final void setExpectedCompletionDate(Instant expectedCompletionDate) {
            this.expectedCompletionDate = expectedCompletionDate;
        }

        @Override
        public final Builder expectedCompletionDate(Instant expectedCompletionDate) {
            this.expectedCompletionDate = expectedCompletionDate;
            return this;
        }

        public final Instant getStartBy() {
            return startBy;
        }

        public final void setStartBy(Instant startBy) {
            this.startBy = startBy;
        }

        @Override
        public final Builder startBy(Instant startBy) {
            this.startBy = startBy;
            return this;
        }

        public final Map<String, String> getBackupOptions() {
            if (backupOptions instanceof SdkAutoConstructMap) {
                return null;
            }
            return backupOptions;
        }

        public final void setBackupOptions(Map<String, String> backupOptions) {
            this.backupOptions = BackupOptionsCopier.copy(backupOptions);
        }

        @Override
        public final Builder backupOptions(Map<String, String> backupOptions) {
            this.backupOptions = BackupOptionsCopier.copy(backupOptions);
            return this;
        }

        public final String getBackupType() {
            return backupType;
        }

        public final void setBackupType(String backupType) {
            this.backupType = backupType;
        }

        @Override
        public final Builder backupType(String backupType) {
            this.backupType = backupType;
            return this;
        }

        public final String getParentJobId() {
            return parentJobId;
        }

        public final void setParentJobId(String parentJobId) {
            this.parentJobId = parentJobId;
        }

        @Override
        public final Builder parentJobId(String parentJobId) {
            this.parentJobId = parentJobId;
            return this;
        }

        public final Boolean getIsParent() {
            return isParent;
        }

        public final void setIsParent(Boolean isParent) {
            this.isParent = isParent;
        }

        @Override
        public final Builder isParent(Boolean isParent) {
            this.isParent = isParent;
            return this;
        }

        public final Long getNumberOfChildJobs() {
            return numberOfChildJobs;
        }

        public final void setNumberOfChildJobs(Long numberOfChildJobs) {
            this.numberOfChildJobs = numberOfChildJobs;
        }

        @Override
        public final Builder numberOfChildJobs(Long numberOfChildJobs) {
            this.numberOfChildJobs = numberOfChildJobs;
            return this;
        }

        public final Map<String, Long> getChildJobsInState() {
            if (childJobsInState instanceof SdkAutoConstructMap) {
                return null;
            }
            return childJobsInState;
        }

        public final void setChildJobsInState(Map<String, Long> childJobsInState) {
            this.childJobsInState = BackupJobChildJobsInStateCopier.copy(childJobsInState);
        }

        @Override
        public final Builder childJobsInStateWithStrings(Map<String, Long> childJobsInState) {
            this.childJobsInState = BackupJobChildJobsInStateCopier.copy(childJobsInState);
            return this;
        }

        @Override
        public final Builder childJobsInState(Map<BackupJobState, Long> childJobsInState) {
            this.childJobsInState = BackupJobChildJobsInStateCopier.copyEnumToString(childJobsInState);
            return this;
        }

        public final String getResourceName() {
            return resourceName;
        }

        public final void setResourceName(String resourceName) {
            this.resourceName = resourceName;
        }

        @Override
        public final Builder resourceName(String resourceName) {
            this.resourceName = resourceName;
            return this;
        }

        public final Instant getInitiationDate() {
            return initiationDate;
        }

        public final void setInitiationDate(Instant initiationDate) {
            this.initiationDate = initiationDate;
        }

        @Override
        public final Builder initiationDate(Instant initiationDate) {
            this.initiationDate = initiationDate;
            return this;
        }

        public final String getMessageCategory() {
            return messageCategory;
        }

        public final void setMessageCategory(String messageCategory) {
            this.messageCategory = messageCategory;
        }

        @Override
        public final Builder messageCategory(String messageCategory) {
            this.messageCategory = messageCategory;
            return this;
        }

        @Override
        public DescribeBackupJobResponse build() {
            return new DescribeBackupJobResponse(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
