/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.backup.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * <code>RecoveryPointSelection</code> has five parameters (three required and two optional). The values you specify
 * determine which recovery point is included in the restore test. You must indicate with <code>Algorithm</code> if you
 * want the latest recovery point within your <code>SelectionWindowDays</code> or if you want a random recovery point,
 * and you must indicate through <code>IncludeVaults</code> from which vaults the recovery points can be chosen.
 * </p>
 * <p>
 * <code>Algorithm</code> (<i>required</i>) Valid values: "<code>LATEST_WITHIN_WINDOW</code>" or "
 * <code>RANDOM_WITHIN_WINDOW</code>".
 * </p>
 * <p>
 * <code>Recovery point types</code> (<i>required</i>) Valid values: "<code>SNAPSHOT</code>" and/or "
 * <code>CONTINUOUS</code>". Include <code>SNAPSHOT</code> to restore only snapshot recovery points; include
 * <code>CONTINUOUS</code> to restore continuous recovery points (point in time restore / PITR); use both to restore
 * either a snapshot or a continuous recovery point. The recovery point will be determined by the value for
 * <code>Algorithm</code>.
 * </p>
 * <p>
 * <code>IncludeVaults</code> (<i>required</i>). You must include one or more backup vaults. Use the wildcard ["*"] or
 * specific ARNs.
 * </p>
 * <p>
 * <code>SelectionWindowDays</code> (<i>optional</i>) Value must be an integer (in days) from 1 to 365. If not included,
 * the value defaults to <code>30</code>.
 * </p>
 * <p>
 * <code>ExcludeVaults</code> (<i>optional</i>). You can choose to input one or more specific backup vault ARNs to
 * exclude those vaults' contents from restore eligibility. Or, you can include a list of selectors. If this parameter
 * and its value are not included, it defaults to empty list.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RestoreTestingRecoveryPointSelection implements SdkPojo, Serializable,
        ToCopyableBuilder<RestoreTestingRecoveryPointSelection.Builder, RestoreTestingRecoveryPointSelection> {
    private static final SdkField<String> ALGORITHM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Algorithm").getter(getter(RestoreTestingRecoveryPointSelection::algorithmAsString))
            .setter(setter(Builder::algorithm))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Algorithm").build()).build();

    private static final SdkField<List<String>> EXCLUDE_VAULTS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ExcludeVaults")
            .getter(getter(RestoreTestingRecoveryPointSelection::excludeVaults))
            .setter(setter(Builder::excludeVaults))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExcludeVaults").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> INCLUDE_VAULTS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("IncludeVaults")
            .getter(getter(RestoreTestingRecoveryPointSelection::includeVaults))
            .setter(setter(Builder::includeVaults))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IncludeVaults").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> RECOVERY_POINT_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("RecoveryPointTypes")
            .getter(getter(RestoreTestingRecoveryPointSelection::recoveryPointTypesAsStrings))
            .setter(setter(Builder::recoveryPointTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RecoveryPointTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> SELECTION_WINDOW_DAYS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("SelectionWindowDays").getter(getter(RestoreTestingRecoveryPointSelection::selectionWindowDays))
            .setter(setter(Builder::selectionWindowDays))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SelectionWindowDays").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ALGORITHM_FIELD,
            EXCLUDE_VAULTS_FIELD, INCLUDE_VAULTS_FIELD, RECOVERY_POINT_TYPES_FIELD, SELECTION_WINDOW_DAYS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String algorithm;

    private final List<String> excludeVaults;

    private final List<String> includeVaults;

    private final List<String> recoveryPointTypes;

    private final Integer selectionWindowDays;

    private RestoreTestingRecoveryPointSelection(BuilderImpl builder) {
        this.algorithm = builder.algorithm;
        this.excludeVaults = builder.excludeVaults;
        this.includeVaults = builder.includeVaults;
        this.recoveryPointTypes = builder.recoveryPointTypes;
        this.selectionWindowDays = builder.selectionWindowDays;
    }

    /**
     * <p>
     * Acceptable values include "LATEST_WITHIN_WINDOW" or "RANDOM_WITHIN_WINDOW"
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #algorithm} will
     * return {@link RestoreTestingRecoveryPointSelectionAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value returned by
     * the service is available from {@link #algorithmAsString}.
     * </p>
     * 
     * @return Acceptable values include "LATEST_WITHIN_WINDOW" or "RANDOM_WITHIN_WINDOW"
     * @see RestoreTestingRecoveryPointSelectionAlgorithm
     */
    public final RestoreTestingRecoveryPointSelectionAlgorithm algorithm() {
        return RestoreTestingRecoveryPointSelectionAlgorithm.fromValue(algorithm);
    }

    /**
     * <p>
     * Acceptable values include "LATEST_WITHIN_WINDOW" or "RANDOM_WITHIN_WINDOW"
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #algorithm} will
     * return {@link RestoreTestingRecoveryPointSelectionAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value returned by
     * the service is available from {@link #algorithmAsString}.
     * </p>
     * 
     * @return Acceptable values include "LATEST_WITHIN_WINDOW" or "RANDOM_WITHIN_WINDOW"
     * @see RestoreTestingRecoveryPointSelectionAlgorithm
     */
    public final String algorithmAsString() {
        return algorithm;
    }

    /**
     * For responses, this returns true if the service returned a value for the ExcludeVaults property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasExcludeVaults() {
        return excludeVaults != null && !(excludeVaults instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Accepted values include specific ARNs or list of selectors. Defaults to empty list if not listed.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasExcludeVaults} method.
     * </p>
     * 
     * @return Accepted values include specific ARNs or list of selectors. Defaults to empty list if not listed.
     */
    public final List<String> excludeVaults() {
        return excludeVaults;
    }

    /**
     * For responses, this returns true if the service returned a value for the IncludeVaults property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasIncludeVaults() {
        return includeVaults != null && !(includeVaults instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Accepted values include wildcard ["*"] or by specific ARNs or ARN wilcard replacement
     * ["arn:aws:backup:us-west-2:123456789012:backup-vault:asdf", ...] ["arn:aws:backup:*:*:backup-vault:asdf-*", ...]
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasIncludeVaults} method.
     * </p>
     * 
     * @return Accepted values include wildcard ["*"] or by specific ARNs or ARN wilcard replacement
     *         ["arn:aws:backup:us-west-2:123456789012:backup-vault:asdf", ...]
     *         ["arn:aws:backup:*:*:backup-vault:asdf-*", ...]
     */
    public final List<String> includeVaults() {
        return includeVaults;
    }

    /**
     * <p>
     * These are the types of recovery points.
     * </p>
     * <p>
     * Include <code>SNAPSHOT</code> to restore only snapshot recovery points; include <code>CONTINUOUS</code> to
     * restore continuous recovery points (point in time restore / PITR); use both to restore either a snapshot or a
     * continuous recovery point. The recovery point will be determined by the value for <code>Algorithm</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRecoveryPointTypes} method.
     * </p>
     * 
     * @return These are the types of recovery points.</p>
     *         <p>
     *         Include <code>SNAPSHOT</code> to restore only snapshot recovery points; include <code>CONTINUOUS</code>
     *         to restore continuous recovery points (point in time restore / PITR); use both to restore either a
     *         snapshot or a continuous recovery point. The recovery point will be determined by the value for
     *         <code>Algorithm</code>.
     */
    public final List<RestoreTestingRecoveryPointType> recoveryPointTypes() {
        return RestoreTestingRecoveryPointTypeListCopier.copyStringToEnum(recoveryPointTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the RecoveryPointTypes property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRecoveryPointTypes() {
        return recoveryPointTypes != null && !(recoveryPointTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * These are the types of recovery points.
     * </p>
     * <p>
     * Include <code>SNAPSHOT</code> to restore only snapshot recovery points; include <code>CONTINUOUS</code> to
     * restore continuous recovery points (point in time restore / PITR); use both to restore either a snapshot or a
     * continuous recovery point. The recovery point will be determined by the value for <code>Algorithm</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRecoveryPointTypes} method.
     * </p>
     * 
     * @return These are the types of recovery points.</p>
     *         <p>
     *         Include <code>SNAPSHOT</code> to restore only snapshot recovery points; include <code>CONTINUOUS</code>
     *         to restore continuous recovery points (point in time restore / PITR); use both to restore either a
     *         snapshot or a continuous recovery point. The recovery point will be determined by the value for
     *         <code>Algorithm</code>.
     */
    public final List<String> recoveryPointTypesAsStrings() {
        return recoveryPointTypes;
    }

    /**
     * <p>
     * Accepted values are integers from 1 to 365.
     * </p>
     * 
     * @return Accepted values are integers from 1 to 365.
     */
    public final Integer selectionWindowDays() {
        return selectionWindowDays;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(algorithmAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasExcludeVaults() ? excludeVaults() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasIncludeVaults() ? includeVaults() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasRecoveryPointTypes() ? recoveryPointTypesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(selectionWindowDays());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RestoreTestingRecoveryPointSelection)) {
            return false;
        }
        RestoreTestingRecoveryPointSelection other = (RestoreTestingRecoveryPointSelection) obj;
        return Objects.equals(algorithmAsString(), other.algorithmAsString()) && hasExcludeVaults() == other.hasExcludeVaults()
                && Objects.equals(excludeVaults(), other.excludeVaults()) && hasIncludeVaults() == other.hasIncludeVaults()
                && Objects.equals(includeVaults(), other.includeVaults())
                && hasRecoveryPointTypes() == other.hasRecoveryPointTypes()
                && Objects.equals(recoveryPointTypesAsStrings(), other.recoveryPointTypesAsStrings())
                && Objects.equals(selectionWindowDays(), other.selectionWindowDays());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RestoreTestingRecoveryPointSelection").add("Algorithm", algorithmAsString())
                .add("ExcludeVaults", hasExcludeVaults() ? excludeVaults() : null)
                .add("IncludeVaults", hasIncludeVaults() ? includeVaults() : null)
                .add("RecoveryPointTypes", hasRecoveryPointTypes() ? recoveryPointTypesAsStrings() : null)
                .add("SelectionWindowDays", selectionWindowDays()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Algorithm":
            return Optional.ofNullable(clazz.cast(algorithmAsString()));
        case "ExcludeVaults":
            return Optional.ofNullable(clazz.cast(excludeVaults()));
        case "IncludeVaults":
            return Optional.ofNullable(clazz.cast(includeVaults()));
        case "RecoveryPointTypes":
            return Optional.ofNullable(clazz.cast(recoveryPointTypesAsStrings()));
        case "SelectionWindowDays":
            return Optional.ofNullable(clazz.cast(selectionWindowDays()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RestoreTestingRecoveryPointSelection, T> g) {
        return obj -> g.apply((RestoreTestingRecoveryPointSelection) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RestoreTestingRecoveryPointSelection> {
        /**
         * <p>
         * Acceptable values include "LATEST_WITHIN_WINDOW" or "RANDOM_WITHIN_WINDOW"
         * </p>
         * 
         * @param algorithm
         *        Acceptable values include "LATEST_WITHIN_WINDOW" or "RANDOM_WITHIN_WINDOW"
         * @see RestoreTestingRecoveryPointSelectionAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RestoreTestingRecoveryPointSelectionAlgorithm
         */
        Builder algorithm(String algorithm);

        /**
         * <p>
         * Acceptable values include "LATEST_WITHIN_WINDOW" or "RANDOM_WITHIN_WINDOW"
         * </p>
         * 
         * @param algorithm
         *        Acceptable values include "LATEST_WITHIN_WINDOW" or "RANDOM_WITHIN_WINDOW"
         * @see RestoreTestingRecoveryPointSelectionAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RestoreTestingRecoveryPointSelectionAlgorithm
         */
        Builder algorithm(RestoreTestingRecoveryPointSelectionAlgorithm algorithm);

        /**
         * <p>
         * Accepted values include specific ARNs or list of selectors. Defaults to empty list if not listed.
         * </p>
         * 
         * @param excludeVaults
         *        Accepted values include specific ARNs or list of selectors. Defaults to empty list if not listed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludeVaults(Collection<String> excludeVaults);

        /**
         * <p>
         * Accepted values include specific ARNs or list of selectors. Defaults to empty list if not listed.
         * </p>
         * 
         * @param excludeVaults
         *        Accepted values include specific ARNs or list of selectors. Defaults to empty list if not listed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludeVaults(String... excludeVaults);

        /**
         * <p>
         * Accepted values include wildcard ["*"] or by specific ARNs or ARN wilcard replacement
         * ["arn:aws:backup:us-west-2:123456789012:backup-vault:asdf", ...] ["arn:aws:backup:*:*:backup-vault:asdf-*",
         * ...]
         * </p>
         * 
         * @param includeVaults
         *        Accepted values include wildcard ["*"] or by specific ARNs or ARN wilcard replacement
         *        ["arn:aws:backup:us-west-2:123456789012:backup-vault:asdf", ...]
         *        ["arn:aws:backup:*:*:backup-vault:asdf-*", ...]
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includeVaults(Collection<String> includeVaults);

        /**
         * <p>
         * Accepted values include wildcard ["*"] or by specific ARNs or ARN wilcard replacement
         * ["arn:aws:backup:us-west-2:123456789012:backup-vault:asdf", ...] ["arn:aws:backup:*:*:backup-vault:asdf-*",
         * ...]
         * </p>
         * 
         * @param includeVaults
         *        Accepted values include wildcard ["*"] or by specific ARNs or ARN wilcard replacement
         *        ["arn:aws:backup:us-west-2:123456789012:backup-vault:asdf", ...]
         *        ["arn:aws:backup:*:*:backup-vault:asdf-*", ...]
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includeVaults(String... includeVaults);

        /**
         * <p>
         * These are the types of recovery points.
         * </p>
         * <p>
         * Include <code>SNAPSHOT</code> to restore only snapshot recovery points; include <code>CONTINUOUS</code> to
         * restore continuous recovery points (point in time restore / PITR); use both to restore either a snapshot or a
         * continuous recovery point. The recovery point will be determined by the value for <code>Algorithm</code>.
         * </p>
         * 
         * @param recoveryPointTypes
         *        These are the types of recovery points.</p>
         *        <p>
         *        Include <code>SNAPSHOT</code> to restore only snapshot recovery points; include
         *        <code>CONTINUOUS</code> to restore continuous recovery points (point in time restore / PITR); use both
         *        to restore either a snapshot or a continuous recovery point. The recovery point will be determined by
         *        the value for <code>Algorithm</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recoveryPointTypesWithStrings(Collection<String> recoveryPointTypes);

        /**
         * <p>
         * These are the types of recovery points.
         * </p>
         * <p>
         * Include <code>SNAPSHOT</code> to restore only snapshot recovery points; include <code>CONTINUOUS</code> to
         * restore continuous recovery points (point in time restore / PITR); use both to restore either a snapshot or a
         * continuous recovery point. The recovery point will be determined by the value for <code>Algorithm</code>.
         * </p>
         * 
         * @param recoveryPointTypes
         *        These are the types of recovery points.</p>
         *        <p>
         *        Include <code>SNAPSHOT</code> to restore only snapshot recovery points; include
         *        <code>CONTINUOUS</code> to restore continuous recovery points (point in time restore / PITR); use both
         *        to restore either a snapshot or a continuous recovery point. The recovery point will be determined by
         *        the value for <code>Algorithm</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recoveryPointTypesWithStrings(String... recoveryPointTypes);

        /**
         * <p>
         * These are the types of recovery points.
         * </p>
         * <p>
         * Include <code>SNAPSHOT</code> to restore only snapshot recovery points; include <code>CONTINUOUS</code> to
         * restore continuous recovery points (point in time restore / PITR); use both to restore either a snapshot or a
         * continuous recovery point. The recovery point will be determined by the value for <code>Algorithm</code>.
         * </p>
         * 
         * @param recoveryPointTypes
         *        These are the types of recovery points.</p>
         *        <p>
         *        Include <code>SNAPSHOT</code> to restore only snapshot recovery points; include
         *        <code>CONTINUOUS</code> to restore continuous recovery points (point in time restore / PITR); use both
         *        to restore either a snapshot or a continuous recovery point. The recovery point will be determined by
         *        the value for <code>Algorithm</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recoveryPointTypes(Collection<RestoreTestingRecoveryPointType> recoveryPointTypes);

        /**
         * <p>
         * These are the types of recovery points.
         * </p>
         * <p>
         * Include <code>SNAPSHOT</code> to restore only snapshot recovery points; include <code>CONTINUOUS</code> to
         * restore continuous recovery points (point in time restore / PITR); use both to restore either a snapshot or a
         * continuous recovery point. The recovery point will be determined by the value for <code>Algorithm</code>.
         * </p>
         * 
         * @param recoveryPointTypes
         *        These are the types of recovery points.</p>
         *        <p>
         *        Include <code>SNAPSHOT</code> to restore only snapshot recovery points; include
         *        <code>CONTINUOUS</code> to restore continuous recovery points (point in time restore / PITR); use both
         *        to restore either a snapshot or a continuous recovery point. The recovery point will be determined by
         *        the value for <code>Algorithm</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recoveryPointTypes(RestoreTestingRecoveryPointType... recoveryPointTypes);

        /**
         * <p>
         * Accepted values are integers from 1 to 365.
         * </p>
         * 
         * @param selectionWindowDays
         *        Accepted values are integers from 1 to 365.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder selectionWindowDays(Integer selectionWindowDays);
    }

    static final class BuilderImpl implements Builder {
        private String algorithm;

        private List<String> excludeVaults = DefaultSdkAutoConstructList.getInstance();

        private List<String> includeVaults = DefaultSdkAutoConstructList.getInstance();

        private List<String> recoveryPointTypes = DefaultSdkAutoConstructList.getInstance();

        private Integer selectionWindowDays;

        private BuilderImpl() {
        }

        private BuilderImpl(RestoreTestingRecoveryPointSelection model) {
            algorithm(model.algorithm);
            excludeVaults(model.excludeVaults);
            includeVaults(model.includeVaults);
            recoveryPointTypesWithStrings(model.recoveryPointTypes);
            selectionWindowDays(model.selectionWindowDays);
        }

        public final String getAlgorithm() {
            return algorithm;
        }

        public final void setAlgorithm(String algorithm) {
            this.algorithm = algorithm;
        }

        @Override
        public final Builder algorithm(String algorithm) {
            this.algorithm = algorithm;
            return this;
        }

        @Override
        public final Builder algorithm(RestoreTestingRecoveryPointSelectionAlgorithm algorithm) {
            this.algorithm(algorithm == null ? null : algorithm.toString());
            return this;
        }

        public final Collection<String> getExcludeVaults() {
            if (excludeVaults instanceof SdkAutoConstructList) {
                return null;
            }
            return excludeVaults;
        }

        public final void setExcludeVaults(Collection<String> excludeVaults) {
            this.excludeVaults = _stringListCopier.copy(excludeVaults);
        }

        @Override
        public final Builder excludeVaults(Collection<String> excludeVaults) {
            this.excludeVaults = _stringListCopier.copy(excludeVaults);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder excludeVaults(String... excludeVaults) {
            excludeVaults(Arrays.asList(excludeVaults));
            return this;
        }

        public final Collection<String> getIncludeVaults() {
            if (includeVaults instanceof SdkAutoConstructList) {
                return null;
            }
            return includeVaults;
        }

        public final void setIncludeVaults(Collection<String> includeVaults) {
            this.includeVaults = _stringListCopier.copy(includeVaults);
        }

        @Override
        public final Builder includeVaults(Collection<String> includeVaults) {
            this.includeVaults = _stringListCopier.copy(includeVaults);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder includeVaults(String... includeVaults) {
            includeVaults(Arrays.asList(includeVaults));
            return this;
        }

        public final Collection<String> getRecoveryPointTypes() {
            if (recoveryPointTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return recoveryPointTypes;
        }

        public final void setRecoveryPointTypes(Collection<String> recoveryPointTypes) {
            this.recoveryPointTypes = RestoreTestingRecoveryPointTypeListCopier.copy(recoveryPointTypes);
        }

        @Override
        public final Builder recoveryPointTypesWithStrings(Collection<String> recoveryPointTypes) {
            this.recoveryPointTypes = RestoreTestingRecoveryPointTypeListCopier.copy(recoveryPointTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recoveryPointTypesWithStrings(String... recoveryPointTypes) {
            recoveryPointTypesWithStrings(Arrays.asList(recoveryPointTypes));
            return this;
        }

        @Override
        public final Builder recoveryPointTypes(Collection<RestoreTestingRecoveryPointType> recoveryPointTypes) {
            this.recoveryPointTypes = RestoreTestingRecoveryPointTypeListCopier.copyEnumToString(recoveryPointTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recoveryPointTypes(RestoreTestingRecoveryPointType... recoveryPointTypes) {
            recoveryPointTypes(Arrays.asList(recoveryPointTypes));
            return this;
        }

        public final Integer getSelectionWindowDays() {
            return selectionWindowDays;
        }

        public final void setSelectionWindowDays(Integer selectionWindowDays) {
            this.selectionWindowDays = selectionWindowDays;
        }

        @Override
        public final Builder selectionWindowDays(Integer selectionWindowDays) {
            this.selectionWindowDays = selectionWindowDays;
            return this;
        }

        @Override
        public RestoreTestingRecoveryPointSelection build() {
            return new RestoreTestingRecoveryPointSelection(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
