/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.backup.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class StartRestoreJobRequest extends BackupRequest implements
        ToCopyableBuilder<StartRestoreJobRequest.Builder, StartRestoreJobRequest> {
    private static final SdkField<String> RECOVERY_POINT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RecoveryPointArn").getter(getter(StartRestoreJobRequest::recoveryPointArn))
            .setter(setter(Builder::recoveryPointArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RecoveryPointArn").build()).build();

    private static final SdkField<Map<String, String>> METADATA_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Metadata")
            .getter(getter(StartRestoreJobRequest::metadata))
            .setter(setter(Builder::metadata))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Metadata").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> IAM_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IamRoleArn").getter(getter(StartRestoreJobRequest::iamRoleArn)).setter(setter(Builder::iamRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IamRoleArn").build()).build();

    private static final SdkField<String> IDEMPOTENCY_TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IdempotencyToken").getter(getter(StartRestoreJobRequest::idempotencyToken))
            .setter(setter(Builder::idempotencyToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IdempotencyToken").build()).build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceType").getter(getter(StartRestoreJobRequest::resourceType))
            .setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceType").build()).build();

    private static final SdkField<Boolean> COPY_SOURCE_TAGS_TO_RESTORED_RESOURCE_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("CopySourceTagsToRestoredResource")
            .getter(getter(StartRestoreJobRequest::copySourceTagsToRestoredResource))
            .setter(setter(Builder::copySourceTagsToRestoredResource))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CopySourceTagsToRestoredResource")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RECOVERY_POINT_ARN_FIELD,
            METADATA_FIELD, IAM_ROLE_ARN_FIELD, IDEMPOTENCY_TOKEN_FIELD, RESOURCE_TYPE_FIELD,
            COPY_SOURCE_TAGS_TO_RESTORED_RESOURCE_FIELD));

    private final String recoveryPointArn;

    private final Map<String, String> metadata;

    private final String iamRoleArn;

    private final String idempotencyToken;

    private final String resourceType;

    private final Boolean copySourceTagsToRestoredResource;

    private StartRestoreJobRequest(BuilderImpl builder) {
        super(builder);
        this.recoveryPointArn = builder.recoveryPointArn;
        this.metadata = builder.metadata;
        this.iamRoleArn = builder.iamRoleArn;
        this.idempotencyToken = builder.idempotencyToken;
        this.resourceType = builder.resourceType;
        this.copySourceTagsToRestoredResource = builder.copySourceTagsToRestoredResource;
    }

    /**
     * <p>
     * An ARN that uniquely identifies a recovery point; for example,
     * <code>arn:aws:backup:us-east-1:123456789012:recovery-point:1EB3B5E7-9EB0-435A-A80B-108B488B0D45</code>.
     * </p>
     * 
     * @return An ARN that uniquely identifies a recovery point; for example,
     *         <code>arn:aws:backup:us-east-1:123456789012:recovery-point:1EB3B5E7-9EB0-435A-A80B-108B488B0D45</code>.
     */
    public final String recoveryPointArn() {
        return recoveryPointArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the Metadata property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasMetadata() {
        return metadata != null && !(metadata instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A set of metadata key-value pairs.
     * </p>
     * <p>
     * You can get configuration metadata about a resource at the time it was backed up by calling
     * <code>GetRecoveryPointRestoreMetadata</code>. However, values in addition to those provided by
     * <code>GetRecoveryPointRestoreMetadata</code> might be required to restore a resource. For example, you might need
     * to provide a new resource name if the original already exists.
     * </p>
     * <p>
     * For more information about the metadata for each resource, see the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-aur.html#aur-restore-cli">Metadata for
     * Amazon Aurora</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-docdb.html#docdb-restore-cli">Metadata
     * for Amazon DocumentDB</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restore-application-stacks.html#restoring-cfn-cli"
     * >Metadata for CloudFormation</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-dynamodb.html#ddb-restore-cli">Metadata
     * for Amazon DynamoDB</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-ebs.html#ebs-restore-cli"> Metadata for
     * Amazon EBS</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-ec2.html#restoring-ec2-cli">Metadata
     * for Amazon EC2</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-efs.html#efs-restore-cli">Metadata for
     * Amazon EFS</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-fsx.html#fsx-restore-cli">Metadata for
     * Amazon FSx</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-nep.html#nep-restore-cli">Metadata for
     * Amazon Neptune</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-rds.html#rds-restore-cli">Metadata for
     * Amazon RDS</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/redshift-restores.html#redshift-restore-api">
     * Metadata for Amazon Redshift</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-storage-gateway.html#restoring-sgw-cli">
     * Metadata for Storage Gateway</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-s3.html#s3-restore-cli">Metadata for
     * Amazon S3</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/timestream-restore.html#timestream-restore-api">
     * Metadata for Amazon Timestream</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-vm.html#vm-restore-cli">Metadata for
     * virtual machines</a>
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMetadata} method.
     * </p>
     * 
     * @return A set of metadata key-value pairs.</p>
     *         <p>
     *         You can get configuration metadata about a resource at the time it was backed up by calling
     *         <code>GetRecoveryPointRestoreMetadata</code>. However, values in addition to those provided by
     *         <code>GetRecoveryPointRestoreMetadata</code> might be required to restore a resource. For example, you
     *         might need to provide a new resource name if the original already exists.
     *         </p>
     *         <p>
     *         For more information about the metadata for each resource, see the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-aur.html#aur-restore-cli">
     *         Metadata for Amazon Aurora</a>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-docdb.html#docdb-restore-cli">
     *         Metadata for Amazon DocumentDB</a>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <a href=
     *         "https://docs.aws.amazon.com/aws-backup/latest/devguide/restore-application-stacks.html#restoring-cfn-cli"
     *         >Metadata for CloudFormation</a>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-dynamodb.html#ddb-restore-cli">
     *         Metadata for Amazon DynamoDB</a>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-ebs.html#ebs-restore-cli">
     *         Metadata for Amazon EBS</a>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-ec2.html#restoring-ec2-cli">
     *         Metadata for Amazon EC2</a>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-efs.html#efs-restore-cli">
     *         Metadata for Amazon EFS</a>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-fsx.html#fsx-restore-cli">
     *         Metadata for Amazon FSx</a>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-nep.html#nep-restore-cli">
     *         Metadata for Amazon Neptune</a>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-rds.html#rds-restore-cli">
     *         Metadata for Amazon RDS</a>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <a href=
     *         "https://docs.aws.amazon.com/aws-backup/latest/devguide/redshift-restores.html#redshift-restore-api"
     *         >Metadata for Amazon Redshift</a>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <a href=
     *         "https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-storage-gateway.html#restoring-sgw-cli"
     *         >Metadata for Storage Gateway</a>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <a
     *         href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-s3.html#s3-restore-cli">Metadata
     *         for Amazon S3</a>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <a href=
     *         "https://docs.aws.amazon.com/aws-backup/latest/devguide/timestream-restore.html#timestream-restore-api"
     *         >Metadata for Amazon Timestream</a>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <a
     *         href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-vm.html#vm-restore-cli">Metadata
     *         for virtual machines</a>
     *         </p>
     *         </li>
     */
    public final Map<String, String> metadata() {
        return metadata;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the IAM role that Backup uses to create the target resource; for example:
     * <code>arn:aws:iam::123456789012:role/S3Access</code>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the IAM role that Backup uses to create the target resource; for
     *         example: <code>arn:aws:iam::123456789012:role/S3Access</code>.
     */
    public final String iamRoleArn() {
        return iamRoleArn;
    }

    /**
     * <p>
     * A customer-chosen string that you can use to distinguish between otherwise identical calls to
     * <code>StartRestoreJob</code>. Retrying a successful request with the same idempotency token results in a success
     * message with no action taken.
     * </p>
     * 
     * @return A customer-chosen string that you can use to distinguish between otherwise identical calls to
     *         <code>StartRestoreJob</code>. Retrying a successful request with the same idempotency token results in a
     *         success message with no action taken.
     */
    public final String idempotencyToken() {
        return idempotencyToken;
    }

    /**
     * <p>
     * Starts a job to restore a recovery point for one of the following resources:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Aurora</code> - Amazon Aurora
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DocumentDB</code> - Amazon DocumentDB
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CloudFormation</code> - CloudFormation
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DynamoDB</code> - Amazon DynamoDB
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>EBS</code> - Amazon Elastic Block Store
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>EC2</code> - Amazon Elastic Compute Cloud
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>EFS</code> - Amazon Elastic File System
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FSx</code> - Amazon FSx
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Neptune</code> - Amazon Neptune
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RDS</code> - Amazon Relational Database Service
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Redshift</code> - Amazon Redshift
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Storage Gateway</code> - Storage Gateway
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>S3</code> - Amazon Simple Storage Service
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Timestream</code> - Amazon Timestream
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>VirtualMachine</code> - Virtual machines
     * </p>
     * </li>
     * </ul>
     * 
     * @return Starts a job to restore a recovery point for one of the following resources:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Aurora</code> - Amazon Aurora
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DocumentDB</code> - Amazon DocumentDB
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CloudFormation</code> - CloudFormation
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DynamoDB</code> - Amazon DynamoDB
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>EBS</code> - Amazon Elastic Block Store
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>EC2</code> - Amazon Elastic Compute Cloud
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>EFS</code> - Amazon Elastic File System
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FSx</code> - Amazon FSx
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Neptune</code> - Amazon Neptune
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RDS</code> - Amazon Relational Database Service
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Redshift</code> - Amazon Redshift
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Storage Gateway</code> - Storage Gateway
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>S3</code> - Amazon Simple Storage Service
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Timestream</code> - Amazon Timestream
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>VirtualMachine</code> - Virtual machines
     *         </p>
     *         </li>
     */
    public final String resourceType() {
        return resourceType;
    }

    /**
     * <p>
     * This is an optional parameter. If this equals <code>True</code>, tags included in the backup will be copied to
     * the restored resource.
     * </p>
     * <p>
     * This can only be applied to backups created through Backup.
     * </p>
     * 
     * @return This is an optional parameter. If this equals <code>True</code>, tags included in the backup will be
     *         copied to the restored resource.</p>
     *         <p>
     *         This can only be applied to backups created through Backup.
     */
    public final Boolean copySourceTagsToRestoredResource() {
        return copySourceTagsToRestoredResource;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(recoveryPointArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasMetadata() ? metadata() : null);
        hashCode = 31 * hashCode + Objects.hashCode(iamRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(idempotencyToken());
        hashCode = 31 * hashCode + Objects.hashCode(resourceType());
        hashCode = 31 * hashCode + Objects.hashCode(copySourceTagsToRestoredResource());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StartRestoreJobRequest)) {
            return false;
        }
        StartRestoreJobRequest other = (StartRestoreJobRequest) obj;
        return Objects.equals(recoveryPointArn(), other.recoveryPointArn()) && hasMetadata() == other.hasMetadata()
                && Objects.equals(metadata(), other.metadata()) && Objects.equals(iamRoleArn(), other.iamRoleArn())
                && Objects.equals(idempotencyToken(), other.idempotencyToken())
                && Objects.equals(resourceType(), other.resourceType())
                && Objects.equals(copySourceTagsToRestoredResource(), other.copySourceTagsToRestoredResource());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("StartRestoreJobRequest").add("RecoveryPointArn", recoveryPointArn())
                .add("Metadata", metadata() == null ? null : "*** Sensitive Data Redacted ***").add("IamRoleArn", iamRoleArn())
                .add("IdempotencyToken", idempotencyToken()).add("ResourceType", resourceType())
                .add("CopySourceTagsToRestoredResource", copySourceTagsToRestoredResource()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RecoveryPointArn":
            return Optional.ofNullable(clazz.cast(recoveryPointArn()));
        case "Metadata":
            return Optional.ofNullable(clazz.cast(metadata()));
        case "IamRoleArn":
            return Optional.ofNullable(clazz.cast(iamRoleArn()));
        case "IdempotencyToken":
            return Optional.ofNullable(clazz.cast(idempotencyToken()));
        case "ResourceType":
            return Optional.ofNullable(clazz.cast(resourceType()));
        case "CopySourceTagsToRestoredResource":
            return Optional.ofNullable(clazz.cast(copySourceTagsToRestoredResource()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<StartRestoreJobRequest, T> g) {
        return obj -> g.apply((StartRestoreJobRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends BackupRequest.Builder, SdkPojo, CopyableBuilder<Builder, StartRestoreJobRequest> {
        /**
         * <p>
         * An ARN that uniquely identifies a recovery point; for example,
         * <code>arn:aws:backup:us-east-1:123456789012:recovery-point:1EB3B5E7-9EB0-435A-A80B-108B488B0D45</code>.
         * </p>
         * 
         * @param recoveryPointArn
         *        An ARN that uniquely identifies a recovery point; for example,
         *        <code>arn:aws:backup:us-east-1:123456789012:recovery-point:1EB3B5E7-9EB0-435A-A80B-108B488B0D45</code>
         *        .
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recoveryPointArn(String recoveryPointArn);

        /**
         * <p>
         * A set of metadata key-value pairs.
         * </p>
         * <p>
         * You can get configuration metadata about a resource at the time it was backed up by calling
         * <code>GetRecoveryPointRestoreMetadata</code>. However, values in addition to those provided by
         * <code>GetRecoveryPointRestoreMetadata</code> might be required to restore a resource. For example, you might
         * need to provide a new resource name if the original already exists.
         * </p>
         * <p>
         * For more information about the metadata for each resource, see the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-aur.html#aur-restore-cli">Metadata
         * for Amazon Aurora</a>
         * </p>
         * </li>
         * <li>
         * <p>
         * <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-docdb.html#docdb-restore-cli">
         * Metadata for Amazon DocumentDB</a>
         * </p>
         * </li>
         * <li>
         * <p>
         * <a href=
         * "https://docs.aws.amazon.com/aws-backup/latest/devguide/restore-application-stacks.html#restoring-cfn-cli"
         * >Metadata for CloudFormation</a>
         * </p>
         * </li>
         * <li>
         * <p>
         * <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-dynamodb.html#ddb-restore-cli">
         * Metadata for Amazon DynamoDB</a>
         * </p>
         * </li>
         * <li>
         * <p>
         * <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-ebs.html#ebs-restore-cli"> Metadata
         * for Amazon EBS</a>
         * </p>
         * </li>
         * <li>
         * <p>
         * <a
         * href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-ec2.html#restoring-ec2-cli">Metadata
         * for Amazon EC2</a>
         * </p>
         * </li>
         * <li>
         * <p>
         * <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-efs.html#efs-restore-cli">Metadata
         * for Amazon EFS</a>
         * </p>
         * </li>
         * <li>
         * <p>
         * <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-fsx.html#fsx-restore-cli">Metadata
         * for Amazon FSx</a>
         * </p>
         * </li>
         * <li>
         * <p>
         * <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-nep.html#nep-restore-cli">Metadata
         * for Amazon Neptune</a>
         * </p>
         * </li>
         * <li>
         * <p>
         * <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-rds.html#rds-restore-cli">Metadata
         * for Amazon RDS</a>
         * </p>
         * </li>
         * <li>
         * <p>
         * <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/redshift-restores.html#redshift-restore-api">
         * Metadata for Amazon Redshift</a>
         * </p>
         * </li>
         * <li>
         * <p>
         * <a href=
         * "https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-storage-gateway.html#restoring-sgw-cli"
         * >Metadata for Storage Gateway</a>
         * </p>
         * </li>
         * <li>
         * <p>
         * <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-s3.html#s3-restore-cli">Metadata
         * for Amazon S3</a>
         * </p>
         * </li>
         * <li>
         * <p>
         * <a href=
         * "https://docs.aws.amazon.com/aws-backup/latest/devguide/timestream-restore.html#timestream-restore-api"
         * >Metadata for Amazon Timestream</a>
         * </p>
         * </li>
         * <li>
         * <p>
         * <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-vm.html#vm-restore-cli">Metadata
         * for virtual machines</a>
         * </p>
         * </li>
         * </ul>
         * 
         * @param metadata
         *        A set of metadata key-value pairs.</p>
         *        <p>
         *        You can get configuration metadata about a resource at the time it was backed up by calling
         *        <code>GetRecoveryPointRestoreMetadata</code>. However, values in addition to those provided by
         *        <code>GetRecoveryPointRestoreMetadata</code> might be required to restore a resource. For example, you
         *        might need to provide a new resource name if the original already exists.
         *        </p>
         *        <p>
         *        For more information about the metadata for each resource, see the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-aur.html#aur-restore-cli">
         *        Metadata for Amazon Aurora</a>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <a
         *        href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-docdb.html#docdb-restore-cli"
         *        >Metadata for Amazon DocumentDB</a>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <a href=
         *        "https://docs.aws.amazon.com/aws-backup/latest/devguide/restore-application-stacks.html#restoring-cfn-cli"
         *        >Metadata for CloudFormation</a>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <a href=
         *        "https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-dynamodb.html#ddb-restore-cli"
         *        >Metadata for Amazon DynamoDB</a>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-ebs.html#ebs-restore-cli">
         *        Metadata for Amazon EBS</a>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-ec2.html#restoring-ec2-cli">
         *        Metadata for Amazon EC2</a>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-efs.html#efs-restore-cli">
         *        Metadata for Amazon EFS</a>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-fsx.html#fsx-restore-cli">
         *        Metadata for Amazon FSx</a>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-nep.html#nep-restore-cli">
         *        Metadata for Amazon Neptune</a>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-rds.html#rds-restore-cli">
         *        Metadata for Amazon RDS</a>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <a href=
         *        "https://docs.aws.amazon.com/aws-backup/latest/devguide/redshift-restores.html#redshift-restore-api"
         *        >Metadata for Amazon Redshift</a>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <a href=
         *        "https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-storage-gateway.html#restoring-sgw-cli"
         *        >Metadata for Storage Gateway</a>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-s3.html#s3-restore-cli">
         *        Metadata for Amazon S3</a>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <a href=
         *        "https://docs.aws.amazon.com/aws-backup/latest/devguide/timestream-restore.html#timestream-restore-api"
         *        >Metadata for Amazon Timestream</a>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/restoring-vm.html#vm-restore-cli">
         *        Metadata for virtual machines</a>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metadata(Map<String, String> metadata);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the IAM role that Backup uses to create the target resource; for example:
         * <code>arn:aws:iam::123456789012:role/S3Access</code>.
         * </p>
         * 
         * @param iamRoleArn
         *        The Amazon Resource Name (ARN) of the IAM role that Backup uses to create the target resource; for
         *        example: <code>arn:aws:iam::123456789012:role/S3Access</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iamRoleArn(String iamRoleArn);

        /**
         * <p>
         * A customer-chosen string that you can use to distinguish between otherwise identical calls to
         * <code>StartRestoreJob</code>. Retrying a successful request with the same idempotency token results in a
         * success message with no action taken.
         * </p>
         * 
         * @param idempotencyToken
         *        A customer-chosen string that you can use to distinguish between otherwise identical calls to
         *        <code>StartRestoreJob</code>. Retrying a successful request with the same idempotency token results in
         *        a success message with no action taken.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder idempotencyToken(String idempotencyToken);

        /**
         * <p>
         * Starts a job to restore a recovery point for one of the following resources:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Aurora</code> - Amazon Aurora
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DocumentDB</code> - Amazon DocumentDB
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CloudFormation</code> - CloudFormation
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DynamoDB</code> - Amazon DynamoDB
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>EBS</code> - Amazon Elastic Block Store
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>EC2</code> - Amazon Elastic Compute Cloud
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>EFS</code> - Amazon Elastic File System
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FSx</code> - Amazon FSx
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Neptune</code> - Amazon Neptune
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RDS</code> - Amazon Relational Database Service
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Redshift</code> - Amazon Redshift
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Storage Gateway</code> - Storage Gateway
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>S3</code> - Amazon Simple Storage Service
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Timestream</code> - Amazon Timestream
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>VirtualMachine</code> - Virtual machines
         * </p>
         * </li>
         * </ul>
         * 
         * @param resourceType
         *        Starts a job to restore a recovery point for one of the following resources:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Aurora</code> - Amazon Aurora
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DocumentDB</code> - Amazon DocumentDB
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CloudFormation</code> - CloudFormation
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DynamoDB</code> - Amazon DynamoDB
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>EBS</code> - Amazon Elastic Block Store
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>EC2</code> - Amazon Elastic Compute Cloud
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>EFS</code> - Amazon Elastic File System
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FSx</code> - Amazon FSx
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Neptune</code> - Amazon Neptune
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RDS</code> - Amazon Relational Database Service
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Redshift</code> - Amazon Redshift
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Storage Gateway</code> - Storage Gateway
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>S3</code> - Amazon Simple Storage Service
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Timestream</code> - Amazon Timestream
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>VirtualMachine</code> - Virtual machines
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * This is an optional parameter. If this equals <code>True</code>, tags included in the backup will be copied
         * to the restored resource.
         * </p>
         * <p>
         * This can only be applied to backups created through Backup.
         * </p>
         * 
         * @param copySourceTagsToRestoredResource
         *        This is an optional parameter. If this equals <code>True</code>, tags included in the backup will be
         *        copied to the restored resource.</p>
         *        <p>
         *        This can only be applied to backups created through Backup.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder copySourceTagsToRestoredResource(Boolean copySourceTagsToRestoredResource);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends BackupRequest.BuilderImpl implements Builder {
        private String recoveryPointArn;

        private Map<String, String> metadata = DefaultSdkAutoConstructMap.getInstance();

        private String iamRoleArn;

        private String idempotencyToken;

        private String resourceType;

        private Boolean copySourceTagsToRestoredResource;

        private BuilderImpl() {
        }

        private BuilderImpl(StartRestoreJobRequest model) {
            super(model);
            recoveryPointArn(model.recoveryPointArn);
            metadata(model.metadata);
            iamRoleArn(model.iamRoleArn);
            idempotencyToken(model.idempotencyToken);
            resourceType(model.resourceType);
            copySourceTagsToRestoredResource(model.copySourceTagsToRestoredResource);
        }

        public final String getRecoveryPointArn() {
            return recoveryPointArn;
        }

        public final void setRecoveryPointArn(String recoveryPointArn) {
            this.recoveryPointArn = recoveryPointArn;
        }

        @Override
        public final Builder recoveryPointArn(String recoveryPointArn) {
            this.recoveryPointArn = recoveryPointArn;
            return this;
        }

        public final Map<String, String> getMetadata() {
            if (metadata instanceof SdkAutoConstructMap) {
                return null;
            }
            return metadata;
        }

        public final void setMetadata(Map<String, String> metadata) {
            this.metadata = MetadataCopier.copy(metadata);
        }

        @Override
        public final Builder metadata(Map<String, String> metadata) {
            this.metadata = MetadataCopier.copy(metadata);
            return this;
        }

        public final String getIamRoleArn() {
            return iamRoleArn;
        }

        public final void setIamRoleArn(String iamRoleArn) {
            this.iamRoleArn = iamRoleArn;
        }

        @Override
        public final Builder iamRoleArn(String iamRoleArn) {
            this.iamRoleArn = iamRoleArn;
            return this;
        }

        public final String getIdempotencyToken() {
            return idempotencyToken;
        }

        public final void setIdempotencyToken(String idempotencyToken) {
            this.idempotencyToken = idempotencyToken;
        }

        @Override
        public final Builder idempotencyToken(String idempotencyToken) {
            this.idempotencyToken = idempotencyToken;
            return this;
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        public final Boolean getCopySourceTagsToRestoredResource() {
            return copySourceTagsToRestoredResource;
        }

        public final void setCopySourceTagsToRestoredResource(Boolean copySourceTagsToRestoredResource) {
            this.copySourceTagsToRestoredResource = copySourceTagsToRestoredResource;
        }

        @Override
        public final Builder copySourceTagsToRestoredResource(Boolean copySourceTagsToRestoredResource) {
            this.copySourceTagsToRestoredResource = copySourceTagsToRestoredResource;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public StartRestoreJobRequest build() {
            return new StartRestoreJobRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
