/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.backup.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The backup options for each resource type.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AdvancedBackupSetting implements SdkPojo, Serializable,
        ToCopyableBuilder<AdvancedBackupSetting.Builder, AdvancedBackupSetting> {
    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceType").getter(getter(AdvancedBackupSetting::resourceType)).setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceType").build()).build();

    private static final SdkField<Map<String, String>> BACKUP_OPTIONS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("BackupOptions")
            .getter(getter(AdvancedBackupSetting::backupOptions))
            .setter(setter(Builder::backupOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BackupOptions").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RESOURCE_TYPE_FIELD,
            BACKUP_OPTIONS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String resourceType;

    private final Map<String, String> backupOptions;

    private AdvancedBackupSetting(BuilderImpl builder) {
        this.resourceType = builder.resourceType;
        this.backupOptions = builder.backupOptions;
    }

    /**
     * <p>
     * Specifies an object containing resource type and backup options. The only supported resource type is Amazon EC2
     * instances with Windows Volume Shadow Copy Service (VSS). For a CloudFormation example, see the <a
     * href="https://docs.aws.amazon.com/aws-backup/latest/devguide/integrate-cloudformation-with-aws-backup.html"
     * >sample CloudFormation template to enable Windows VSS</a> in the <i>Backup User Guide</i>.
     * </p>
     * <p>
     * Valid values: <code>EC2</code>.
     * </p>
     * 
     * @return Specifies an object containing resource type and backup options. The only supported resource type is
     *         Amazon EC2 instances with Windows Volume Shadow Copy Service (VSS). For a CloudFormation example, see the
     *         <a href=
     *         "https://docs.aws.amazon.com/aws-backup/latest/devguide/integrate-cloudformation-with-aws-backup.html"
     *         >sample CloudFormation template to enable Windows VSS</a> in the <i>Backup User Guide</i>.</p>
     *         <p>
     *         Valid values: <code>EC2</code>.
     */
    public final String resourceType() {
        return resourceType;
    }

    /**
     * For responses, this returns true if the service returned a value for the BackupOptions property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasBackupOptions() {
        return backupOptions != null && !(backupOptions instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Specifies the backup option for a selected resource. This option is only available for Windows VSS backup jobs.
     * </p>
     * <p>
     * Valid values:
     * </p>
     * <p>
     * Set to <code>"WindowsVSS":"enabled"</code> to enable the <code>WindowsVSS</code> backup option and create a
     * Windows VSS backup.
     * </p>
     * <p>
     * Set to <code>"WindowsVSS":"disabled"</code> to create a regular backup. The <code>WindowsVSS</code> option is not
     * enabled by default.
     * </p>
     * <p>
     * If you specify an invalid option, you get an <code>InvalidParameterValueException</code> exception.
     * </p>
     * <p>
     * For more information about Windows VSS backups, see <a
     * href="https://docs.aws.amazon.com/aws-backup/latest/devguide/windows-backups.html">Creating a VSS-Enabled Windows
     * Backup</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasBackupOptions} method.
     * </p>
     * 
     * @return Specifies the backup option for a selected resource. This option is only available for Windows VSS backup
     *         jobs.</p>
     *         <p>
     *         Valid values:
     *         </p>
     *         <p>
     *         Set to <code>"WindowsVSS":"enabled"</code> to enable the <code>WindowsVSS</code> backup option and create
     *         a Windows VSS backup.
     *         </p>
     *         <p>
     *         Set to <code>"WindowsVSS":"disabled"</code> to create a regular backup. The <code>WindowsVSS</code>
     *         option is not enabled by default.
     *         </p>
     *         <p>
     *         If you specify an invalid option, you get an <code>InvalidParameterValueException</code> exception.
     *         </p>
     *         <p>
     *         For more information about Windows VSS backups, see <a
     *         href="https://docs.aws.amazon.com/aws-backup/latest/devguide/windows-backups.html">Creating a VSS-Enabled
     *         Windows Backup</a>.
     */
    public final Map<String, String> backupOptions() {
        return backupOptions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(resourceType());
        hashCode = 31 * hashCode + Objects.hashCode(hasBackupOptions() ? backupOptions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AdvancedBackupSetting)) {
            return false;
        }
        AdvancedBackupSetting other = (AdvancedBackupSetting) obj;
        return Objects.equals(resourceType(), other.resourceType()) && hasBackupOptions() == other.hasBackupOptions()
                && Objects.equals(backupOptions(), other.backupOptions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AdvancedBackupSetting").add("ResourceType", resourceType())
                .add("BackupOptions", hasBackupOptions() ? backupOptions() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ResourceType":
            return Optional.ofNullable(clazz.cast(resourceType()));
        case "BackupOptions":
            return Optional.ofNullable(clazz.cast(backupOptions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ResourceType", RESOURCE_TYPE_FIELD);
        map.put("BackupOptions", BACKUP_OPTIONS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AdvancedBackupSetting, T> g) {
        return obj -> g.apply((AdvancedBackupSetting) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AdvancedBackupSetting> {
        /**
         * <p>
         * Specifies an object containing resource type and backup options. The only supported resource type is Amazon
         * EC2 instances with Windows Volume Shadow Copy Service (VSS). For a CloudFormation example, see the <a
         * href="https://docs.aws.amazon.com/aws-backup/latest/devguide/integrate-cloudformation-with-aws-backup.html"
         * >sample CloudFormation template to enable Windows VSS</a> in the <i>Backup User Guide</i>.
         * </p>
         * <p>
         * Valid values: <code>EC2</code>.
         * </p>
         * 
         * @param resourceType
         *        Specifies an object containing resource type and backup options. The only supported resource type is
         *        Amazon EC2 instances with Windows Volume Shadow Copy Service (VSS). For a CloudFormation example, see
         *        the <a href=
         *        "https://docs.aws.amazon.com/aws-backup/latest/devguide/integrate-cloudformation-with-aws-backup.html"
         *        >sample CloudFormation template to enable Windows VSS</a> in the <i>Backup User Guide</i>.</p>
         *        <p>
         *        Valid values: <code>EC2</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * Specifies the backup option for a selected resource. This option is only available for Windows VSS backup
         * jobs.
         * </p>
         * <p>
         * Valid values:
         * </p>
         * <p>
         * Set to <code>"WindowsVSS":"enabled"</code> to enable the <code>WindowsVSS</code> backup option and create a
         * Windows VSS backup.
         * </p>
         * <p>
         * Set to <code>"WindowsVSS":"disabled"</code> to create a regular backup. The <code>WindowsVSS</code> option is
         * not enabled by default.
         * </p>
         * <p>
         * If you specify an invalid option, you get an <code>InvalidParameterValueException</code> exception.
         * </p>
         * <p>
         * For more information about Windows VSS backups, see <a
         * href="https://docs.aws.amazon.com/aws-backup/latest/devguide/windows-backups.html">Creating a VSS-Enabled
         * Windows Backup</a>.
         * </p>
         * 
         * @param backupOptions
         *        Specifies the backup option for a selected resource. This option is only available for Windows VSS
         *        backup jobs.</p>
         *        <p>
         *        Valid values:
         *        </p>
         *        <p>
         *        Set to <code>"WindowsVSS":"enabled"</code> to enable the <code>WindowsVSS</code> backup option and
         *        create a Windows VSS backup.
         *        </p>
         *        <p>
         *        Set to <code>"WindowsVSS":"disabled"</code> to create a regular backup. The <code>WindowsVSS</code>
         *        option is not enabled by default.
         *        </p>
         *        <p>
         *        If you specify an invalid option, you get an <code>InvalidParameterValueException</code> exception.
         *        </p>
         *        <p>
         *        For more information about Windows VSS backups, see <a
         *        href="https://docs.aws.amazon.com/aws-backup/latest/devguide/windows-backups.html">Creating a
         *        VSS-Enabled Windows Backup</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backupOptions(Map<String, String> backupOptions);
    }

    static final class BuilderImpl implements Builder {
        private String resourceType;

        private Map<String, String> backupOptions = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(AdvancedBackupSetting model) {
            resourceType(model.resourceType);
            backupOptions(model.backupOptions);
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        public final Map<String, String> getBackupOptions() {
            if (backupOptions instanceof SdkAutoConstructMap) {
                return null;
            }
            return backupOptions;
        }

        public final void setBackupOptions(Map<String, String> backupOptions) {
            this.backupOptions = BackupOptionsCopier.copy(backupOptions);
        }

        @Override
        public final Builder backupOptions(Map<String, String> backupOptions) {
            this.backupOptions = BackupOptionsCopier.copy(backupOptions);
            return this;
        }

        @Override
        public AdvancedBackupSetting build() {
            return new AdvancedBackupSetting(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
