/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.backup.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies the time period, in days, before a recovery point transitions to cold storage or is deleted.
 * </p>
 * <p>
 * Backups transitioned to cold storage must be stored in cold storage for a minimum of 90 days. Therefore, on the
 * console, the retention setting must be 90 days greater than the transition to cold after days setting. The transition
 * to cold after days setting can't be changed after a backup has been transitioned to cold.
 * </p>
 * <p>
 * Resource types that can transition to cold storage are listed in the <a
 * href="https://docs.aws.amazon.com/aws-backup/latest/devguide/backup-feature-availability.html#features-by-resource"
 * >Feature availability by resource</a> table. Backup ignores this expression for other resource types.
 * </p>
 * <p>
 * To remove the existing lifecycle and retention periods and keep your recovery points indefinitely, specify -1 for
 * <code>MoveToColdStorageAfterDays</code> and <code>DeleteAfterDays</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Lifecycle implements SdkPojo, Serializable, ToCopyableBuilder<Lifecycle.Builder, Lifecycle> {
    private static final SdkField<Long> MOVE_TO_COLD_STORAGE_AFTER_DAYS_FIELD = SdkField
            .<Long> builder(MarshallingType.LONG)
            .memberName("MoveToColdStorageAfterDays")
            .getter(getter(Lifecycle::moveToColdStorageAfterDays))
            .setter(setter(Builder::moveToColdStorageAfterDays))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MoveToColdStorageAfterDays").build())
            .build();

    private static final SdkField<Long> DELETE_AFTER_DAYS_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("DeleteAfterDays").getter(getter(Lifecycle::deleteAfterDays)).setter(setter(Builder::deleteAfterDays))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeleteAfterDays").build()).build();

    private static final SdkField<Boolean> OPT_IN_TO_ARCHIVE_FOR_SUPPORTED_RESOURCES_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("OptInToArchiveForSupportedResources")
            .getter(getter(Lifecycle::optInToArchiveForSupportedResources))
            .setter(setter(Builder::optInToArchiveForSupportedResources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("OptInToArchiveForSupportedResources").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            MOVE_TO_COLD_STORAGE_AFTER_DAYS_FIELD, DELETE_AFTER_DAYS_FIELD, OPT_IN_TO_ARCHIVE_FOR_SUPPORTED_RESOURCES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Long moveToColdStorageAfterDays;

    private final Long deleteAfterDays;

    private final Boolean optInToArchiveForSupportedResources;

    private Lifecycle(BuilderImpl builder) {
        this.moveToColdStorageAfterDays = builder.moveToColdStorageAfterDays;
        this.deleteAfterDays = builder.deleteAfterDays;
        this.optInToArchiveForSupportedResources = builder.optInToArchiveForSupportedResources;
    }

    /**
     * <p>
     * The number of days after creation that a recovery point is moved to cold storage.
     * </p>
     * 
     * @return The number of days after creation that a recovery point is moved to cold storage.
     */
    public final Long moveToColdStorageAfterDays() {
        return moveToColdStorageAfterDays;
    }

    /**
     * <p>
     * The number of days after creation that a recovery point is deleted. This value must be at least 90 days after the
     * number of days specified in <code>MoveToColdStorageAfterDays</code>.
     * </p>
     * 
     * @return The number of days after creation that a recovery point is deleted. This value must be at least 90 days
     *         after the number of days specified in <code>MoveToColdStorageAfterDays</code>.
     */
    public final Long deleteAfterDays() {
        return deleteAfterDays;
    }

    /**
     * <p>
     * If the value is true, your backup plan transitions supported resources to archive (cold) storage tier in
     * accordance with your lifecycle settings.
     * </p>
     * 
     * @return If the value is true, your backup plan transitions supported resources to archive (cold) storage tier in
     *         accordance with your lifecycle settings.
     */
    public final Boolean optInToArchiveForSupportedResources() {
        return optInToArchiveForSupportedResources;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(moveToColdStorageAfterDays());
        hashCode = 31 * hashCode + Objects.hashCode(deleteAfterDays());
        hashCode = 31 * hashCode + Objects.hashCode(optInToArchiveForSupportedResources());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Lifecycle)) {
            return false;
        }
        Lifecycle other = (Lifecycle) obj;
        return Objects.equals(moveToColdStorageAfterDays(), other.moveToColdStorageAfterDays())
                && Objects.equals(deleteAfterDays(), other.deleteAfterDays())
                && Objects.equals(optInToArchiveForSupportedResources(), other.optInToArchiveForSupportedResources());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Lifecycle").add("MoveToColdStorageAfterDays", moveToColdStorageAfterDays())
                .add("DeleteAfterDays", deleteAfterDays())
                .add("OptInToArchiveForSupportedResources", optInToArchiveForSupportedResources()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MoveToColdStorageAfterDays":
            return Optional.ofNullable(clazz.cast(moveToColdStorageAfterDays()));
        case "DeleteAfterDays":
            return Optional.ofNullable(clazz.cast(deleteAfterDays()));
        case "OptInToArchiveForSupportedResources":
            return Optional.ofNullable(clazz.cast(optInToArchiveForSupportedResources()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("MoveToColdStorageAfterDays", MOVE_TO_COLD_STORAGE_AFTER_DAYS_FIELD);
        map.put("DeleteAfterDays", DELETE_AFTER_DAYS_FIELD);
        map.put("OptInToArchiveForSupportedResources", OPT_IN_TO_ARCHIVE_FOR_SUPPORTED_RESOURCES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Lifecycle, T> g) {
        return obj -> g.apply((Lifecycle) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Lifecycle> {
        /**
         * <p>
         * The number of days after creation that a recovery point is moved to cold storage.
         * </p>
         * 
         * @param moveToColdStorageAfterDays
         *        The number of days after creation that a recovery point is moved to cold storage.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder moveToColdStorageAfterDays(Long moveToColdStorageAfterDays);

        /**
         * <p>
         * The number of days after creation that a recovery point is deleted. This value must be at least 90 days after
         * the number of days specified in <code>MoveToColdStorageAfterDays</code>.
         * </p>
         * 
         * @param deleteAfterDays
         *        The number of days after creation that a recovery point is deleted. This value must be at least 90
         *        days after the number of days specified in <code>MoveToColdStorageAfterDays</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deleteAfterDays(Long deleteAfterDays);

        /**
         * <p>
         * If the value is true, your backup plan transitions supported resources to archive (cold) storage tier in
         * accordance with your lifecycle settings.
         * </p>
         * 
         * @param optInToArchiveForSupportedResources
         *        If the value is true, your backup plan transitions supported resources to archive (cold) storage tier
         *        in accordance with your lifecycle settings.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder optInToArchiveForSupportedResources(Boolean optInToArchiveForSupportedResources);
    }

    static final class BuilderImpl implements Builder {
        private Long moveToColdStorageAfterDays;

        private Long deleteAfterDays;

        private Boolean optInToArchiveForSupportedResources;

        private BuilderImpl() {
        }

        private BuilderImpl(Lifecycle model) {
            moveToColdStorageAfterDays(model.moveToColdStorageAfterDays);
            deleteAfterDays(model.deleteAfterDays);
            optInToArchiveForSupportedResources(model.optInToArchiveForSupportedResources);
        }

        public final Long getMoveToColdStorageAfterDays() {
            return moveToColdStorageAfterDays;
        }

        public final void setMoveToColdStorageAfterDays(Long moveToColdStorageAfterDays) {
            this.moveToColdStorageAfterDays = moveToColdStorageAfterDays;
        }

        @Override
        public final Builder moveToColdStorageAfterDays(Long moveToColdStorageAfterDays) {
            this.moveToColdStorageAfterDays = moveToColdStorageAfterDays;
            return this;
        }

        public final Long getDeleteAfterDays() {
            return deleteAfterDays;
        }

        public final void setDeleteAfterDays(Long deleteAfterDays) {
            this.deleteAfterDays = deleteAfterDays;
        }

        @Override
        public final Builder deleteAfterDays(Long deleteAfterDays) {
            this.deleteAfterDays = deleteAfterDays;
            return this;
        }

        public final Boolean getOptInToArchiveForSupportedResources() {
            return optInToArchiveForSupportedResources;
        }

        public final void setOptInToArchiveForSupportedResources(Boolean optInToArchiveForSupportedResources) {
            this.optInToArchiveForSupportedResources = optInToArchiveForSupportedResources;
        }

        @Override
        public final Builder optInToArchiveForSupportedResources(Boolean optInToArchiveForSupportedResources) {
            this.optInToArchiveForSupportedResources = optInToArchiveForSupportedResources;
            return this;
        }

        @Override
        public Lifecycle build() {
            return new Lifecycle(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
