/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.backup.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies a scheduled task used to back up a selection of resources.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class BackupRule implements SdkPojo, Serializable, ToCopyableBuilder<BackupRule.Builder, BackupRule> {
    private static final SdkField<String> RULE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(BackupRule::ruleName)).setter(setter(Builder::ruleName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleName").build()).build();

    private static final SdkField<String> TARGET_BACKUP_VAULT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(BackupRule::targetBackupVaultName)).setter(setter(Builder::targetBackupVaultName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetBackupVaultName").build())
            .build();

    private static final SdkField<String> SCHEDULE_EXPRESSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(BackupRule::scheduleExpression)).setter(setter(Builder::scheduleExpression))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScheduleExpression").build())
            .build();

    private static final SdkField<Long> START_WINDOW_MINUTES_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .getter(getter(BackupRule::startWindowMinutes)).setter(setter(Builder::startWindowMinutes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartWindowMinutes").build())
            .build();

    private static final SdkField<Long> COMPLETION_WINDOW_MINUTES_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .getter(getter(BackupRule::completionWindowMinutes)).setter(setter(Builder::completionWindowMinutes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CompletionWindowMinutes").build())
            .build();

    private static final SdkField<Lifecycle> LIFECYCLE_FIELD = SdkField.<Lifecycle> builder(MarshallingType.SDK_POJO)
            .getter(getter(BackupRule::lifecycle)).setter(setter(Builder::lifecycle)).constructor(Lifecycle::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Lifecycle").build()).build();

    private static final SdkField<Map<String, String>> RECOVERY_POINT_TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(BackupRule::recoveryPointTags))
            .setter(setter(Builder::recoveryPointTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RecoveryPointTags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> RULE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(BackupRule::ruleId)).setter(setter(Builder::ruleId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RULE_NAME_FIELD,
            TARGET_BACKUP_VAULT_NAME_FIELD, SCHEDULE_EXPRESSION_FIELD, START_WINDOW_MINUTES_FIELD,
            COMPLETION_WINDOW_MINUTES_FIELD, LIFECYCLE_FIELD, RECOVERY_POINT_TAGS_FIELD, RULE_ID_FIELD));

    private static final long serialVersionUID = 1L;

    private final String ruleName;

    private final String targetBackupVaultName;

    private final String scheduleExpression;

    private final Long startWindowMinutes;

    private final Long completionWindowMinutes;

    private final Lifecycle lifecycle;

    private final Map<String, String> recoveryPointTags;

    private final String ruleId;

    private BackupRule(BuilderImpl builder) {
        this.ruleName = builder.ruleName;
        this.targetBackupVaultName = builder.targetBackupVaultName;
        this.scheduleExpression = builder.scheduleExpression;
        this.startWindowMinutes = builder.startWindowMinutes;
        this.completionWindowMinutes = builder.completionWindowMinutes;
        this.lifecycle = builder.lifecycle;
        this.recoveryPointTags = builder.recoveryPointTags;
        this.ruleId = builder.ruleId;
    }

    /**
     * <p>
     * An optional display name for a backup rule.
     * </p>
     * 
     * @return An optional display name for a backup rule.
     */
    public String ruleName() {
        return ruleName;
    }

    /**
     * <p>
     * The name of a logical container where backups are stored. Backup vaults are identified by names that are unique
     * to the account used to create them and the AWS Region where they are created. They consist of lowercase letters,
     * numbers, and hyphens.
     * </p>
     * 
     * @return The name of a logical container where backups are stored. Backup vaults are identified by names that are
     *         unique to the account used to create them and the AWS Region where they are created. They consist of
     *         lowercase letters, numbers, and hyphens.
     */
    public String targetBackupVaultName() {
        return targetBackupVaultName;
    }

    /**
     * <p>
     * A CRON expression specifying when AWS Backup initiates a backup job.
     * </p>
     * 
     * @return A CRON expression specifying when AWS Backup initiates a backup job.
     */
    public String scheduleExpression() {
        return scheduleExpression;
    }

    /**
     * <p>
     * An optional value that specifies a period of time in minutes after a backup is scheduled before a job is canceled
     * if it doesn't start successfully.
     * </p>
     * 
     * @return An optional value that specifies a period of time in minutes after a backup is scheduled before a job is
     *         canceled if it doesn't start successfully.
     */
    public Long startWindowMinutes() {
        return startWindowMinutes;
    }

    /**
     * <p>
     * A value in minutes after a backup job is successfully started before it must be completed or it is canceled by
     * AWS Backup. This value is optional.
     * </p>
     * 
     * @return A value in minutes after a backup job is successfully started before it must be completed or it is
     *         canceled by AWS Backup. This value is optional.
     */
    public Long completionWindowMinutes() {
        return completionWindowMinutes;
    }

    /**
     * <p>
     * The lifecycle defines when a protected resource is transitioned to cold storage and when it expires. AWS Backup
     * transitions and expires backups automatically according to the lifecycle that you define.
     * </p>
     * <p>
     * Backups transitioned to cold storage must be stored in cold storage for a minimum of 90 days. Therefore, the
     * “expire after days” setting must be 90 days greater than the “transition to cold after days” setting. The
     * “transition to cold after days” setting cannot be changed after a backup has been transitioned to cold.
     * </p>
     * 
     * @return The lifecycle defines when a protected resource is transitioned to cold storage and when it expires. AWS
     *         Backup transitions and expires backups automatically according to the lifecycle that you define. </p>
     *         <p>
     *         Backups transitioned to cold storage must be stored in cold storage for a minimum of 90 days. Therefore,
     *         the “expire after days” setting must be 90 days greater than the “transition to cold after days” setting.
     *         The “transition to cold after days” setting cannot be changed after a backup has been transitioned to
     *         cold.
     */
    public Lifecycle lifecycle() {
        return lifecycle;
    }

    /**
     * <p>
     * An array of key-value pair strings that are assigned to resources that are associated with this rule when
     * restored from backup.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return An array of key-value pair strings that are assigned to resources that are associated with this rule when
     *         restored from backup.
     */
    public Map<String, String> recoveryPointTags() {
        return recoveryPointTags;
    }

    /**
     * <p>
     * Uniquely identifies a rule that is used to schedule the backup of a selection of resources.
     * </p>
     * 
     * @return Uniquely identifies a rule that is used to schedule the backup of a selection of resources.
     */
    public String ruleId() {
        return ruleId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(ruleName());
        hashCode = 31 * hashCode + Objects.hashCode(targetBackupVaultName());
        hashCode = 31 * hashCode + Objects.hashCode(scheduleExpression());
        hashCode = 31 * hashCode + Objects.hashCode(startWindowMinutes());
        hashCode = 31 * hashCode + Objects.hashCode(completionWindowMinutes());
        hashCode = 31 * hashCode + Objects.hashCode(lifecycle());
        hashCode = 31 * hashCode + Objects.hashCode(recoveryPointTags());
        hashCode = 31 * hashCode + Objects.hashCode(ruleId());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof BackupRule)) {
            return false;
        }
        BackupRule other = (BackupRule) obj;
        return Objects.equals(ruleName(), other.ruleName())
                && Objects.equals(targetBackupVaultName(), other.targetBackupVaultName())
                && Objects.equals(scheduleExpression(), other.scheduleExpression())
                && Objects.equals(startWindowMinutes(), other.startWindowMinutes())
                && Objects.equals(completionWindowMinutes(), other.completionWindowMinutes())
                && Objects.equals(lifecycle(), other.lifecycle())
                && Objects.equals(recoveryPointTags(), other.recoveryPointTags()) && Objects.equals(ruleId(), other.ruleId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("BackupRule").add("RuleName", ruleName()).add("TargetBackupVaultName", targetBackupVaultName())
                .add("ScheduleExpression", scheduleExpression()).add("StartWindowMinutes", startWindowMinutes())
                .add("CompletionWindowMinutes", completionWindowMinutes()).add("Lifecycle", lifecycle())
                .add("RecoveryPointTags", recoveryPointTags() == null ? null : "*** Sensitive Data Redacted ***")
                .add("RuleId", ruleId()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RuleName":
            return Optional.ofNullable(clazz.cast(ruleName()));
        case "TargetBackupVaultName":
            return Optional.ofNullable(clazz.cast(targetBackupVaultName()));
        case "ScheduleExpression":
            return Optional.ofNullable(clazz.cast(scheduleExpression()));
        case "StartWindowMinutes":
            return Optional.ofNullable(clazz.cast(startWindowMinutes()));
        case "CompletionWindowMinutes":
            return Optional.ofNullable(clazz.cast(completionWindowMinutes()));
        case "Lifecycle":
            return Optional.ofNullable(clazz.cast(lifecycle()));
        case "RecoveryPointTags":
            return Optional.ofNullable(clazz.cast(recoveryPointTags()));
        case "RuleId":
            return Optional.ofNullable(clazz.cast(ruleId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<BackupRule, T> g) {
        return obj -> g.apply((BackupRule) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, BackupRule> {
        /**
         * <p>
         * An optional display name for a backup rule.
         * </p>
         * 
         * @param ruleName
         *        An optional display name for a backup rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleName(String ruleName);

        /**
         * <p>
         * The name of a logical container where backups are stored. Backup vaults are identified by names that are
         * unique to the account used to create them and the AWS Region where they are created. They consist of
         * lowercase letters, numbers, and hyphens.
         * </p>
         * 
         * @param targetBackupVaultName
         *        The name of a logical container where backups are stored. Backup vaults are identified by names that
         *        are unique to the account used to create them and the AWS Region where they are created. They consist
         *        of lowercase letters, numbers, and hyphens.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetBackupVaultName(String targetBackupVaultName);

        /**
         * <p>
         * A CRON expression specifying when AWS Backup initiates a backup job.
         * </p>
         * 
         * @param scheduleExpression
         *        A CRON expression specifying when AWS Backup initiates a backup job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scheduleExpression(String scheduleExpression);

        /**
         * <p>
         * An optional value that specifies a period of time in minutes after a backup is scheduled before a job is
         * canceled if it doesn't start successfully.
         * </p>
         * 
         * @param startWindowMinutes
         *        An optional value that specifies a period of time in minutes after a backup is scheduled before a job
         *        is canceled if it doesn't start successfully.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startWindowMinutes(Long startWindowMinutes);

        /**
         * <p>
         * A value in minutes after a backup job is successfully started before it must be completed or it is canceled
         * by AWS Backup. This value is optional.
         * </p>
         * 
         * @param completionWindowMinutes
         *        A value in minutes after a backup job is successfully started before it must be completed or it is
         *        canceled by AWS Backup. This value is optional.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder completionWindowMinutes(Long completionWindowMinutes);

        /**
         * <p>
         * The lifecycle defines when a protected resource is transitioned to cold storage and when it expires. AWS
         * Backup transitions and expires backups automatically according to the lifecycle that you define.
         * </p>
         * <p>
         * Backups transitioned to cold storage must be stored in cold storage for a minimum of 90 days. Therefore, the
         * “expire after days” setting must be 90 days greater than the “transition to cold after days” setting. The
         * “transition to cold after days” setting cannot be changed after a backup has been transitioned to cold.
         * </p>
         * 
         * @param lifecycle
         *        The lifecycle defines when a protected resource is transitioned to cold storage and when it expires.
         *        AWS Backup transitions and expires backups automatically according to the lifecycle that you define.
         *        </p>
         *        <p>
         *        Backups transitioned to cold storage must be stored in cold storage for a minimum of 90 days.
         *        Therefore, the “expire after days” setting must be 90 days greater than the “transition to cold after
         *        days” setting. The “transition to cold after days” setting cannot be changed after a backup has been
         *        transitioned to cold.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lifecycle(Lifecycle lifecycle);

        /**
         * <p>
         * The lifecycle defines when a protected resource is transitioned to cold storage and when it expires. AWS
         * Backup transitions and expires backups automatically according to the lifecycle that you define.
         * </p>
         * <p>
         * Backups transitioned to cold storage must be stored in cold storage for a minimum of 90 days. Therefore, the
         * “expire after days” setting must be 90 days greater than the “transition to cold after days” setting. The
         * “transition to cold after days” setting cannot be changed after a backup has been transitioned to cold.
         * </p>
         * This is a convenience that creates an instance of the {@link Lifecycle.Builder} avoiding the need to create
         * one manually via {@link Lifecycle#builder()}.
         *
         * When the {@link Consumer} completes, {@link Lifecycle.Builder#build()} is called immediately and its result
         * is passed to {@link #lifecycle(Lifecycle)}.
         * 
         * @param lifecycle
         *        a consumer that will call methods on {@link Lifecycle.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #lifecycle(Lifecycle)
         */
        default Builder lifecycle(Consumer<Lifecycle.Builder> lifecycle) {
            return lifecycle(Lifecycle.builder().applyMutation(lifecycle).build());
        }

        /**
         * <p>
         * An array of key-value pair strings that are assigned to resources that are associated with this rule when
         * restored from backup.
         * </p>
         * 
         * @param recoveryPointTags
         *        An array of key-value pair strings that are assigned to resources that are associated with this rule
         *        when restored from backup.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recoveryPointTags(Map<String, String> recoveryPointTags);

        /**
         * <p>
         * Uniquely identifies a rule that is used to schedule the backup of a selection of resources.
         * </p>
         * 
         * @param ruleId
         *        Uniquely identifies a rule that is used to schedule the backup of a selection of resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleId(String ruleId);
    }

    static final class BuilderImpl implements Builder {
        private String ruleName;

        private String targetBackupVaultName;

        private String scheduleExpression;

        private Long startWindowMinutes;

        private Long completionWindowMinutes;

        private Lifecycle lifecycle;

        private Map<String, String> recoveryPointTags = DefaultSdkAutoConstructMap.getInstance();

        private String ruleId;

        private BuilderImpl() {
        }

        private BuilderImpl(BackupRule model) {
            ruleName(model.ruleName);
            targetBackupVaultName(model.targetBackupVaultName);
            scheduleExpression(model.scheduleExpression);
            startWindowMinutes(model.startWindowMinutes);
            completionWindowMinutes(model.completionWindowMinutes);
            lifecycle(model.lifecycle);
            recoveryPointTags(model.recoveryPointTags);
            ruleId(model.ruleId);
        }

        public final String getRuleName() {
            return ruleName;
        }

        @Override
        public final Builder ruleName(String ruleName) {
            this.ruleName = ruleName;
            return this;
        }

        public final void setRuleName(String ruleName) {
            this.ruleName = ruleName;
        }

        public final String getTargetBackupVaultName() {
            return targetBackupVaultName;
        }

        @Override
        public final Builder targetBackupVaultName(String targetBackupVaultName) {
            this.targetBackupVaultName = targetBackupVaultName;
            return this;
        }

        public final void setTargetBackupVaultName(String targetBackupVaultName) {
            this.targetBackupVaultName = targetBackupVaultName;
        }

        public final String getScheduleExpression() {
            return scheduleExpression;
        }

        @Override
        public final Builder scheduleExpression(String scheduleExpression) {
            this.scheduleExpression = scheduleExpression;
            return this;
        }

        public final void setScheduleExpression(String scheduleExpression) {
            this.scheduleExpression = scheduleExpression;
        }

        public final Long getStartWindowMinutes() {
            return startWindowMinutes;
        }

        @Override
        public final Builder startWindowMinutes(Long startWindowMinutes) {
            this.startWindowMinutes = startWindowMinutes;
            return this;
        }

        public final void setStartWindowMinutes(Long startWindowMinutes) {
            this.startWindowMinutes = startWindowMinutes;
        }

        public final Long getCompletionWindowMinutes() {
            return completionWindowMinutes;
        }

        @Override
        public final Builder completionWindowMinutes(Long completionWindowMinutes) {
            this.completionWindowMinutes = completionWindowMinutes;
            return this;
        }

        public final void setCompletionWindowMinutes(Long completionWindowMinutes) {
            this.completionWindowMinutes = completionWindowMinutes;
        }

        public final Lifecycle.Builder getLifecycle() {
            return lifecycle != null ? lifecycle.toBuilder() : null;
        }

        @Override
        public final Builder lifecycle(Lifecycle lifecycle) {
            this.lifecycle = lifecycle;
            return this;
        }

        public final void setLifecycle(Lifecycle.BuilderImpl lifecycle) {
            this.lifecycle = lifecycle != null ? lifecycle.build() : null;
        }

        public final Map<String, String> getRecoveryPointTags() {
            return recoveryPointTags;
        }

        @Override
        public final Builder recoveryPointTags(Map<String, String> recoveryPointTags) {
            this.recoveryPointTags = TagsCopier.copy(recoveryPointTags);
            return this;
        }

        public final void setRecoveryPointTags(Map<String, String> recoveryPointTags) {
            this.recoveryPointTags = TagsCopier.copy(recoveryPointTags);
        }

        public final String getRuleId() {
            return ruleId;
        }

        @Override
        public final Builder ruleId(String ruleId) {
            this.ruleId = ruleId;
            return this;
        }

        public final void setRuleId(String ruleId) {
            this.ruleId = ruleId;
        }

        @Override
        public BackupRule build() {
            return new BackupRule(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
