/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.backupgateway;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.backupgateway.internal.BackupGatewayServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.backupgateway.model.AccessDeniedException;
import software.amazon.awssdk.services.backupgateway.model.AssociateGatewayToServerRequest;
import software.amazon.awssdk.services.backupgateway.model.AssociateGatewayToServerResponse;
import software.amazon.awssdk.services.backupgateway.model.BackupGatewayException;
import software.amazon.awssdk.services.backupgateway.model.ConflictException;
import software.amazon.awssdk.services.backupgateway.model.CreateGatewayRequest;
import software.amazon.awssdk.services.backupgateway.model.CreateGatewayResponse;
import software.amazon.awssdk.services.backupgateway.model.DeleteGatewayRequest;
import software.amazon.awssdk.services.backupgateway.model.DeleteGatewayResponse;
import software.amazon.awssdk.services.backupgateway.model.DeleteHypervisorRequest;
import software.amazon.awssdk.services.backupgateway.model.DeleteHypervisorResponse;
import software.amazon.awssdk.services.backupgateway.model.DisassociateGatewayFromServerRequest;
import software.amazon.awssdk.services.backupgateway.model.DisassociateGatewayFromServerResponse;
import software.amazon.awssdk.services.backupgateway.model.GetBandwidthRateLimitScheduleRequest;
import software.amazon.awssdk.services.backupgateway.model.GetBandwidthRateLimitScheduleResponse;
import software.amazon.awssdk.services.backupgateway.model.GetGatewayRequest;
import software.amazon.awssdk.services.backupgateway.model.GetGatewayResponse;
import software.amazon.awssdk.services.backupgateway.model.GetHypervisorPropertyMappingsRequest;
import software.amazon.awssdk.services.backupgateway.model.GetHypervisorPropertyMappingsResponse;
import software.amazon.awssdk.services.backupgateway.model.GetHypervisorRequest;
import software.amazon.awssdk.services.backupgateway.model.GetHypervisorResponse;
import software.amazon.awssdk.services.backupgateway.model.GetVirtualMachineRequest;
import software.amazon.awssdk.services.backupgateway.model.GetVirtualMachineResponse;
import software.amazon.awssdk.services.backupgateway.model.ImportHypervisorConfigurationRequest;
import software.amazon.awssdk.services.backupgateway.model.ImportHypervisorConfigurationResponse;
import software.amazon.awssdk.services.backupgateway.model.InternalServerException;
import software.amazon.awssdk.services.backupgateway.model.ListGatewaysRequest;
import software.amazon.awssdk.services.backupgateway.model.ListGatewaysResponse;
import software.amazon.awssdk.services.backupgateway.model.ListHypervisorsRequest;
import software.amazon.awssdk.services.backupgateway.model.ListHypervisorsResponse;
import software.amazon.awssdk.services.backupgateway.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.backupgateway.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.backupgateway.model.ListVirtualMachinesRequest;
import software.amazon.awssdk.services.backupgateway.model.ListVirtualMachinesResponse;
import software.amazon.awssdk.services.backupgateway.model.PutBandwidthRateLimitScheduleRequest;
import software.amazon.awssdk.services.backupgateway.model.PutBandwidthRateLimitScheduleResponse;
import software.amazon.awssdk.services.backupgateway.model.PutHypervisorPropertyMappingsRequest;
import software.amazon.awssdk.services.backupgateway.model.PutHypervisorPropertyMappingsResponse;
import software.amazon.awssdk.services.backupgateway.model.PutMaintenanceStartTimeRequest;
import software.amazon.awssdk.services.backupgateway.model.PutMaintenanceStartTimeResponse;
import software.amazon.awssdk.services.backupgateway.model.ResourceNotFoundException;
import software.amazon.awssdk.services.backupgateway.model.StartVirtualMachinesMetadataSyncRequest;
import software.amazon.awssdk.services.backupgateway.model.StartVirtualMachinesMetadataSyncResponse;
import software.amazon.awssdk.services.backupgateway.model.TagResourceRequest;
import software.amazon.awssdk.services.backupgateway.model.TagResourceResponse;
import software.amazon.awssdk.services.backupgateway.model.TestHypervisorConfigurationRequest;
import software.amazon.awssdk.services.backupgateway.model.TestHypervisorConfigurationResponse;
import software.amazon.awssdk.services.backupgateway.model.ThrottlingException;
import software.amazon.awssdk.services.backupgateway.model.UntagResourceRequest;
import software.amazon.awssdk.services.backupgateway.model.UntagResourceResponse;
import software.amazon.awssdk.services.backupgateway.model.UpdateGatewayInformationRequest;
import software.amazon.awssdk.services.backupgateway.model.UpdateGatewayInformationResponse;
import software.amazon.awssdk.services.backupgateway.model.UpdateGatewaySoftwareNowRequest;
import software.amazon.awssdk.services.backupgateway.model.UpdateGatewaySoftwareNowResponse;
import software.amazon.awssdk.services.backupgateway.model.UpdateHypervisorRequest;
import software.amazon.awssdk.services.backupgateway.model.UpdateHypervisorResponse;
import software.amazon.awssdk.services.backupgateway.model.ValidationException;
import software.amazon.awssdk.services.backupgateway.transform.AssociateGatewayToServerRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.CreateGatewayRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.DeleteGatewayRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.DeleteHypervisorRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.DisassociateGatewayFromServerRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.GetBandwidthRateLimitScheduleRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.GetGatewayRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.GetHypervisorPropertyMappingsRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.GetHypervisorRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.GetVirtualMachineRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.ImportHypervisorConfigurationRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.ListGatewaysRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.ListHypervisorsRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.ListVirtualMachinesRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.PutBandwidthRateLimitScheduleRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.PutHypervisorPropertyMappingsRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.PutMaintenanceStartTimeRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.StartVirtualMachinesMetadataSyncRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.TestHypervisorConfigurationRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.UpdateGatewayInformationRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.UpdateGatewaySoftwareNowRequestMarshaller;
import software.amazon.awssdk.services.backupgateway.transform.UpdateHypervisorRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link BackupGatewayClient}.
 *
 * @see BackupGatewayClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultBackupGatewayClient implements BackupGatewayClient {
    private static final Logger log = Logger.loggerFor(DefaultBackupGatewayClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultBackupGatewayClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Associates a backup gateway with your server. After you complete the association process, you can back up and
     * restore your VMs through the gateway.
     * </p>
     *
     * @param associateGatewayToServerRequest
     * @return Result of the AssociateGatewayToServer operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws ConflictException
     *         The operation cannot proceed because it is not supported.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws ThrottlingException
     *         TPS has been limited to protect against intentional or unintentional high request volumes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.AssociateGatewayToServer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/AssociateGatewayToServer"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AssociateGatewayToServerResponse associateGatewayToServer(
            AssociateGatewayToServerRequest associateGatewayToServerRequest) throws ValidationException, ConflictException,
            InternalServerException, ThrottlingException, AwsServiceException, SdkClientException, BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateGatewayToServerResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AssociateGatewayToServerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateGatewayToServerRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateGatewayToServerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateGatewayToServer");

            return clientHandler
                    .execute(new ClientExecutionParams<AssociateGatewayToServerRequest, AssociateGatewayToServerResponse>()
                            .withOperationName("AssociateGatewayToServer").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(associateGatewayToServerRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AssociateGatewayToServerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a backup gateway. After you create a gateway, you can associate it with a server using the
     * <code>AssociateGatewayToServer</code> operation.
     * </p>
     *
     * @param createGatewayRequest
     * @return Result of the CreateGateway operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws ThrottlingException
     *         TPS has been limited to protect against intentional or unintentional high request volumes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.CreateGateway
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/CreateGateway" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateGatewayResponse createGateway(CreateGatewayRequest createGatewayRequest) throws ValidationException,
            InternalServerException, ThrottlingException, AwsServiceException, SdkClientException, BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateGatewayResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateGatewayResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createGatewayRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGatewayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGateway");

            return clientHandler.execute(new ClientExecutionParams<CreateGatewayRequest, CreateGatewayResponse>()
                    .withOperationName("CreateGateway").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createGatewayRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateGatewayRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a backup gateway.
     * </p>
     *
     * @param deleteGatewayRequest
     * @return Result of the DeleteGateway operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action wasn't found.
     * @throws ThrottlingException
     *         TPS has been limited to protect against intentional or unintentional high request volumes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.DeleteGateway
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/DeleteGateway" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteGatewayResponse deleteGateway(DeleteGatewayRequest deleteGatewayRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException,
            BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteGatewayResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteGatewayResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteGatewayRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteGatewayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteGateway");

            return clientHandler.execute(new ClientExecutionParams<DeleteGatewayRequest, DeleteGatewayResponse>()
                    .withOperationName("DeleteGateway").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteGatewayRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteGatewayRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a hypervisor.
     * </p>
     *
     * @param deleteHypervisorRequest
     * @return Result of the DeleteHypervisor operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws ConflictException
     *         The operation cannot proceed because it is not supported.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws AccessDeniedException
     *         The operation cannot proceed because you have insufficient permissions.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action wasn't found.
     * @throws ThrottlingException
     *         TPS has been limited to protect against intentional or unintentional high request volumes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.DeleteHypervisor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/DeleteHypervisor"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteHypervisorResponse deleteHypervisor(DeleteHypervisorRequest deleteHypervisorRequest) throws ValidationException,
            ConflictException, InternalServerException, AccessDeniedException, ResourceNotFoundException, ThrottlingException,
            AwsServiceException, SdkClientException, BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteHypervisorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteHypervisorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteHypervisorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteHypervisorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteHypervisor");

            return clientHandler.execute(new ClientExecutionParams<DeleteHypervisorRequest, DeleteHypervisorResponse>()
                    .withOperationName("DeleteHypervisor").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteHypervisorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteHypervisorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disassociates a backup gateway from the specified server. After the disassociation process finishes, the gateway
     * can no longer access the virtual machines on the server.
     * </p>
     *
     * @param disassociateGatewayFromServerRequest
     * @return Result of the DisassociateGatewayFromServer operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws ConflictException
     *         The operation cannot proceed because it is not supported.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action wasn't found.
     * @throws ThrottlingException
     *         TPS has been limited to protect against intentional or unintentional high request volumes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.DisassociateGatewayFromServer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/DisassociateGatewayFromServer"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateGatewayFromServerResponse disassociateGatewayFromServer(
            DisassociateGatewayFromServerRequest disassociateGatewayFromServerRequest) throws ValidationException,
            ConflictException, InternalServerException, ResourceNotFoundException, ThrottlingException, AwsServiceException,
            SdkClientException, BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateGatewayFromServerResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateGatewayFromServerResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateGatewayFromServerRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateGatewayFromServerRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateGatewayFromServer");

            return clientHandler
                    .execute(new ClientExecutionParams<DisassociateGatewayFromServerRequest, DisassociateGatewayFromServerResponse>()
                            .withOperationName("DisassociateGatewayFromServer").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(disassociateGatewayFromServerRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisassociateGatewayFromServerRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the bandwidth rate limit schedule for a specified gateway. By default, gateways do not have bandwidth
     * rate limit schedules, which means no bandwidth rate limiting is in effect. Use this to get a gateway's bandwidth
     * rate limit schedule.
     * </p>
     *
     * @param getBandwidthRateLimitScheduleRequest
     * @return Result of the GetBandwidthRateLimitSchedule operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action wasn't found.
     * @throws ThrottlingException
     *         TPS has been limited to protect against intentional or unintentional high request volumes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.GetBandwidthRateLimitSchedule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/GetBandwidthRateLimitSchedule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetBandwidthRateLimitScheduleResponse getBandwidthRateLimitSchedule(
            GetBandwidthRateLimitScheduleRequest getBandwidthRateLimitScheduleRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException,
            BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetBandwidthRateLimitScheduleResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetBandwidthRateLimitScheduleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getBandwidthRateLimitScheduleRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getBandwidthRateLimitScheduleRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBandwidthRateLimitSchedule");

            return clientHandler
                    .execute(new ClientExecutionParams<GetBandwidthRateLimitScheduleRequest, GetBandwidthRateLimitScheduleResponse>()
                            .withOperationName("GetBandwidthRateLimitSchedule").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getBandwidthRateLimitScheduleRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetBandwidthRateLimitScheduleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * By providing the ARN (Amazon Resource Name), this API returns the gateway.
     * </p>
     *
     * @param getGatewayRequest
     * @return Result of the GetGateway operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action wasn't found.
     * @throws ThrottlingException
     *         TPS has been limited to protect against intentional or unintentional high request volumes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.GetGateway
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/GetGateway" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetGatewayResponse getGateway(GetGatewayRequest getGatewayRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException,
            BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetGatewayResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetGatewayResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getGatewayRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGatewayRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGateway");

            return clientHandler.execute(new ClientExecutionParams<GetGatewayRequest, GetGatewayResponse>()
                    .withOperationName("GetGateway").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getGatewayRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetGatewayRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This action requests information about the specified hypervisor to which the gateway will connect. A hypervisor
     * is hardware, software, or firmware that creates and manages virtual machines, and allocates resources to them.
     * </p>
     *
     * @param getHypervisorRequest
     * @return Result of the GetHypervisor operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action wasn't found.
     * @throws ThrottlingException
     *         TPS has been limited to protect against intentional or unintentional high request volumes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.GetHypervisor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/GetHypervisor" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetHypervisorResponse getHypervisor(GetHypervisorRequest getHypervisorRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException,
            BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetHypervisorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetHypervisorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getHypervisorRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getHypervisorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetHypervisor");

            return clientHandler.execute(new ClientExecutionParams<GetHypervisorRequest, GetHypervisorResponse>()
                    .withOperationName("GetHypervisor").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getHypervisorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetHypervisorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This action retrieves the property mappings for the specified hypervisor. A hypervisor property mapping displays
     * the relationship of entity properties available from the on-premises hypervisor to the properties available in
     * Amazon Web Services.
     * </p>
     *
     * @param getHypervisorPropertyMappingsRequest
     * @return Result of the GetHypervisorPropertyMappings operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action wasn't found.
     * @throws ThrottlingException
     *         TPS has been limited to protect against intentional or unintentional high request volumes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.GetHypervisorPropertyMappings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/GetHypervisorPropertyMappings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetHypervisorPropertyMappingsResponse getHypervisorPropertyMappings(
            GetHypervisorPropertyMappingsRequest getHypervisorPropertyMappingsRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException,
            BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetHypervisorPropertyMappingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetHypervisorPropertyMappingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getHypervisorPropertyMappingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getHypervisorPropertyMappingsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetHypervisorPropertyMappings");

            return clientHandler
                    .execute(new ClientExecutionParams<GetHypervisorPropertyMappingsRequest, GetHypervisorPropertyMappingsResponse>()
                            .withOperationName("GetHypervisorPropertyMappings").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getHypervisorPropertyMappingsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetHypervisorPropertyMappingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * By providing the ARN (Amazon Resource Name), this API returns the virtual machine.
     * </p>
     *
     * @param getVirtualMachineRequest
     * @return Result of the GetVirtualMachine operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action wasn't found.
     * @throws ThrottlingException
     *         TPS has been limited to protect against intentional or unintentional high request volumes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.GetVirtualMachine
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/GetVirtualMachine"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetVirtualMachineResponse getVirtualMachine(GetVirtualMachineRequest getVirtualMachineRequest)
            throws ValidationException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            AwsServiceException, SdkClientException, BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetVirtualMachineResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetVirtualMachineResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getVirtualMachineRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getVirtualMachineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetVirtualMachine");

            return clientHandler.execute(new ClientExecutionParams<GetVirtualMachineRequest, GetVirtualMachineResponse>()
                    .withOperationName("GetVirtualMachine").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getVirtualMachineRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetVirtualMachineRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Connect to a hypervisor by importing its configuration.
     * </p>
     *
     * @param importHypervisorConfigurationRequest
     * @return Result of the ImportHypervisorConfiguration operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws ConflictException
     *         The operation cannot proceed because it is not supported.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws AccessDeniedException
     *         The operation cannot proceed because you have insufficient permissions.
     * @throws ThrottlingException
     *         TPS has been limited to protect against intentional or unintentional high request volumes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.ImportHypervisorConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/ImportHypervisorConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ImportHypervisorConfigurationResponse importHypervisorConfiguration(
            ImportHypervisorConfigurationRequest importHypervisorConfigurationRequest) throws ValidationException,
            ConflictException, InternalServerException, AccessDeniedException, ThrottlingException, AwsServiceException,
            SdkClientException, BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ImportHypervisorConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ImportHypervisorConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(importHypervisorConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                importHypervisorConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ImportHypervisorConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<ImportHypervisorConfigurationRequest, ImportHypervisorConfigurationResponse>()
                            .withOperationName("ImportHypervisorConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(importHypervisorConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ImportHypervisorConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists backup gateways owned by an Amazon Web Services account in an Amazon Web Services Region. The returned list
     * is ordered by gateway Amazon Resource Name (ARN).
     * </p>
     *
     * @param listGatewaysRequest
     * @return Result of the ListGateways operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws ThrottlingException
     *         TPS has been limited to protect against intentional or unintentional high request volumes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.ListGateways
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/ListGateways" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListGatewaysResponse listGateways(ListGatewaysRequest listGatewaysRequest) throws ValidationException,
            InternalServerException, ThrottlingException, AwsServiceException, SdkClientException, BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListGatewaysResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListGatewaysResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listGatewaysRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGatewaysRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGateways");

            return clientHandler.execute(new ClientExecutionParams<ListGatewaysRequest, ListGatewaysResponse>()
                    .withOperationName("ListGateways").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listGatewaysRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListGatewaysRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists your hypervisors.
     * </p>
     *
     * @param listHypervisorsRequest
     * @return Result of the ListHypervisors operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws ThrottlingException
     *         TPS has been limited to protect against intentional or unintentional high request volumes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.ListHypervisors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/ListHypervisors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListHypervisorsResponse listHypervisors(ListHypervisorsRequest listHypervisorsRequest) throws ValidationException,
            InternalServerException, ThrottlingException, AwsServiceException, SdkClientException, BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListHypervisorsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListHypervisorsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listHypervisorsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listHypervisorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListHypervisors");

            return clientHandler.execute(new ClientExecutionParams<ListHypervisorsRequest, ListHypervisorsResponse>()
                    .withOperationName("ListHypervisors").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listHypervisorsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListHypervisorsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the tags applied to the resource identified by its Amazon Resource Name (ARN).
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action wasn't found.
     * @throws ThrottlingException
     *         TPS has been limited to protect against intentional or unintentional high request volumes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ValidationException, InternalServerException, ResourceNotFoundException, ThrottlingException,
            AwsServiceException, SdkClientException, BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists your virtual machines.
     * </p>
     *
     * @param listVirtualMachinesRequest
     * @return Result of the ListVirtualMachines operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws ThrottlingException
     *         TPS has been limited to protect against intentional or unintentional high request volumes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.ListVirtualMachines
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/ListVirtualMachines"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListVirtualMachinesResponse listVirtualMachines(ListVirtualMachinesRequest listVirtualMachinesRequest)
            throws ValidationException, InternalServerException, ThrottlingException, AwsServiceException, SdkClientException,
            BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListVirtualMachinesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListVirtualMachinesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listVirtualMachinesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVirtualMachinesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVirtualMachines");

            return clientHandler.execute(new ClientExecutionParams<ListVirtualMachinesRequest, ListVirtualMachinesResponse>()
                    .withOperationName("ListVirtualMachines").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listVirtualMachinesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListVirtualMachinesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This action sets the bandwidth rate limit schedule for a specified gateway. By default, gateways do not have a
     * bandwidth rate limit schedule, which means no bandwidth rate limiting is in effect. Use this to initiate a
     * gateway's bandwidth rate limit schedule.
     * </p>
     *
     * @param putBandwidthRateLimitScheduleRequest
     * @return Result of the PutBandwidthRateLimitSchedule operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action wasn't found.
     * @throws ThrottlingException
     *         TPS has been limited to protect against intentional or unintentional high request volumes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.PutBandwidthRateLimitSchedule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/PutBandwidthRateLimitSchedule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutBandwidthRateLimitScheduleResponse putBandwidthRateLimitSchedule(
            PutBandwidthRateLimitScheduleRequest putBandwidthRateLimitScheduleRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException,
            BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutBandwidthRateLimitScheduleResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutBandwidthRateLimitScheduleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putBandwidthRateLimitScheduleRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putBandwidthRateLimitScheduleRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutBandwidthRateLimitSchedule");

            return clientHandler
                    .execute(new ClientExecutionParams<PutBandwidthRateLimitScheduleRequest, PutBandwidthRateLimitScheduleResponse>()
                            .withOperationName("PutBandwidthRateLimitSchedule").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(putBandwidthRateLimitScheduleRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutBandwidthRateLimitScheduleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This action sets the property mappings for the specified hypervisor. A hypervisor property mapping displays the
     * relationship of entity properties available from the on-premises hypervisor to the properties available in Amazon
     * Web Services.
     * </p>
     *
     * @param putHypervisorPropertyMappingsRequest
     * @return Result of the PutHypervisorPropertyMappings operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws ConflictException
     *         The operation cannot proceed because it is not supported.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws AccessDeniedException
     *         The operation cannot proceed because you have insufficient permissions.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action wasn't found.
     * @throws ThrottlingException
     *         TPS has been limited to protect against intentional or unintentional high request volumes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.PutHypervisorPropertyMappings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/PutHypervisorPropertyMappings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutHypervisorPropertyMappingsResponse putHypervisorPropertyMappings(
            PutHypervisorPropertyMappingsRequest putHypervisorPropertyMappingsRequest) throws ValidationException,
            ConflictException, InternalServerException, AccessDeniedException, ResourceNotFoundException, ThrottlingException,
            AwsServiceException, SdkClientException, BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutHypervisorPropertyMappingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutHypervisorPropertyMappingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putHypervisorPropertyMappingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putHypervisorPropertyMappingsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutHypervisorPropertyMappings");

            return clientHandler
                    .execute(new ClientExecutionParams<PutHypervisorPropertyMappingsRequest, PutHypervisorPropertyMappingsResponse>()
                            .withOperationName("PutHypervisorPropertyMappings").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(putHypervisorPropertyMappingsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutHypervisorPropertyMappingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Set the maintenance start time for a gateway.
     * </p>
     *
     * @param putMaintenanceStartTimeRequest
     * @return Result of the PutMaintenanceStartTime operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws ConflictException
     *         The operation cannot proceed because it is not supported.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action wasn't found.
     * @throws ThrottlingException
     *         TPS has been limited to protect against intentional or unintentional high request volumes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.PutMaintenanceStartTime
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/PutMaintenanceStartTime"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutMaintenanceStartTimeResponse putMaintenanceStartTime(PutMaintenanceStartTimeRequest putMaintenanceStartTimeRequest)
            throws ValidationException, ConflictException, InternalServerException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutMaintenanceStartTimeResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutMaintenanceStartTimeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putMaintenanceStartTimeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putMaintenanceStartTimeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutMaintenanceStartTime");

            return clientHandler
                    .execute(new ClientExecutionParams<PutMaintenanceStartTimeRequest, PutMaintenanceStartTimeResponse>()
                            .withOperationName("PutMaintenanceStartTime").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(putMaintenanceStartTimeRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutMaintenanceStartTimeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This action sends a request to sync metadata across the specified virtual machines.
     * </p>
     *
     * @param startVirtualMachinesMetadataSyncRequest
     * @return Result of the StartVirtualMachinesMetadataSync operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws AccessDeniedException
     *         The operation cannot proceed because you have insufficient permissions.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action wasn't found.
     * @throws ThrottlingException
     *         TPS has been limited to protect against intentional or unintentional high request volumes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.StartVirtualMachinesMetadataSync
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/StartVirtualMachinesMetadataSync"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartVirtualMachinesMetadataSyncResponse startVirtualMachinesMetadataSync(
            StartVirtualMachinesMetadataSyncRequest startVirtualMachinesMetadataSyncRequest) throws ValidationException,
            InternalServerException, AccessDeniedException, ResourceNotFoundException, ThrottlingException, AwsServiceException,
            SdkClientException, BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartVirtualMachinesMetadataSyncResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartVirtualMachinesMetadataSyncResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startVirtualMachinesMetadataSyncRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                startVirtualMachinesMetadataSyncRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartVirtualMachinesMetadataSync");

            return clientHandler
                    .execute(new ClientExecutionParams<StartVirtualMachinesMetadataSyncRequest, StartVirtualMachinesMetadataSyncResponse>()
                            .withOperationName("StartVirtualMachinesMetadataSync").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(startVirtualMachinesMetadataSyncRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StartVirtualMachinesMetadataSyncRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Tag the resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action wasn't found.
     * @throws ThrottlingException
     *         TPS has been limited to protect against intentional or unintentional high request volumes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException,
            BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Tests your hypervisor configuration to validate that backup gateway can connect with the hypervisor and its
     * resources.
     * </p>
     *
     * @param testHypervisorConfigurationRequest
     * @return Result of the TestHypervisorConfiguration operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws ConflictException
     *         The operation cannot proceed because it is not supported.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action wasn't found.
     * @throws ThrottlingException
     *         TPS has been limited to protect against intentional or unintentional high request volumes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.TestHypervisorConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/TestHypervisorConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public TestHypervisorConfigurationResponse testHypervisorConfiguration(
            TestHypervisorConfigurationRequest testHypervisorConfigurationRequest) throws ValidationException, ConflictException,
            InternalServerException, ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException,
            BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TestHypervisorConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, TestHypervisorConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(testHypervisorConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, testHypervisorConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TestHypervisorConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<TestHypervisorConfigurationRequest, TestHypervisorConfigurationResponse>()
                            .withOperationName("TestHypervisorConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(testHypervisorConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new TestHypervisorConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes tags from the resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action wasn't found.
     * @throws ThrottlingException
     *         TPS has been limited to protect against intentional or unintentional high request volumes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException,
            BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a gateway's name. Specify which gateway to update using the Amazon Resource Name (ARN) of the gateway in
     * your request.
     * </p>
     *
     * @param updateGatewayInformationRequest
     * @return Result of the UpdateGatewayInformation operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws ConflictException
     *         The operation cannot proceed because it is not supported.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action wasn't found.
     * @throws ThrottlingException
     *         TPS has been limited to protect against intentional or unintentional high request volumes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.UpdateGatewayInformation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/UpdateGatewayInformation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateGatewayInformationResponse updateGatewayInformation(
            UpdateGatewayInformationRequest updateGatewayInformationRequest) throws ValidationException, ConflictException,
            InternalServerException, ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException,
            BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateGatewayInformationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateGatewayInformationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateGatewayInformationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateGatewayInformationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateGatewayInformation");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateGatewayInformationRequest, UpdateGatewayInformationResponse>()
                            .withOperationName("UpdateGatewayInformation").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateGatewayInformationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateGatewayInformationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the gateway virtual machine (VM) software. The request immediately triggers the software update.
     * </p>
     * <note>
     * <p>
     * When you make this request, you get a <code>200 OK</code> success response immediately. However, it might take
     * some time for the update to complete.
     * </p>
     * </note>
     *
     * @param updateGatewaySoftwareNowRequest
     * @return Result of the UpdateGatewaySoftwareNow operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action wasn't found.
     * @throws ThrottlingException
     *         TPS has been limited to protect against intentional or unintentional high request volumes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.UpdateGatewaySoftwareNow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/UpdateGatewaySoftwareNow"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateGatewaySoftwareNowResponse updateGatewaySoftwareNow(
            UpdateGatewaySoftwareNowRequest updateGatewaySoftwareNowRequest) throws ValidationException, InternalServerException,
            ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException, BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateGatewaySoftwareNowResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateGatewaySoftwareNowResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateGatewaySoftwareNowRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateGatewaySoftwareNowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateGatewaySoftwareNow");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateGatewaySoftwareNowRequest, UpdateGatewaySoftwareNowResponse>()
                            .withOperationName("UpdateGatewaySoftwareNow").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateGatewaySoftwareNowRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateGatewaySoftwareNowRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a hypervisor metadata, including its host, username, and password. Specify which hypervisor to update
     * using the Amazon Resource Name (ARN) of the hypervisor in your request.
     * </p>
     *
     * @param updateHypervisorRequest
     * @return Result of the UpdateHypervisor operation returned by the service.
     * @throws ValidationException
     *         The operation did not succeed because a validation error occurred.
     * @throws ConflictException
     *         The operation cannot proceed because it is not supported.
     * @throws InternalServerException
     *         The operation did not succeed because an internal error occurred. Try again later.
     * @throws AccessDeniedException
     *         The operation cannot proceed because you have insufficient permissions.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action wasn't found.
     * @throws ThrottlingException
     *         TPS has been limited to protect against intentional or unintentional high request volumes.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupGatewayException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupGatewayClient.UpdateHypervisor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backup-gateway-2021-01-01/UpdateHypervisor"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateHypervisorResponse updateHypervisor(UpdateHypervisorRequest updateHypervisorRequest) throws ValidationException,
            ConflictException, InternalServerException, AccessDeniedException, ResourceNotFoundException, ThrottlingException,
            AwsServiceException, SdkClientException, BackupGatewayException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateHypervisorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateHypervisorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateHypervisorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateHypervisorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Backup Gateway");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateHypervisor");

            return clientHandler.execute(new ClientExecutionParams<UpdateHypervisorRequest, UpdateHypervisorResponse>()
                    .withOperationName("UpdateHypervisor").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateHypervisorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateHypervisorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        BackupGatewayServiceClientConfigurationBuilder serviceConfigBuilder = new BackupGatewayServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(BackupGatewayException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    @Override
    public final BackupGatewayServiceClientConfiguration serviceClientConfiguration() {
        return new BackupGatewayServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
