/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.backupsearch;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.backupsearch.model.AccessDeniedException;
import software.amazon.awssdk.services.backupsearch.model.BackupSearchException;
import software.amazon.awssdk.services.backupsearch.model.ConflictException;
import software.amazon.awssdk.services.backupsearch.model.GetSearchJobRequest;
import software.amazon.awssdk.services.backupsearch.model.GetSearchJobResponse;
import software.amazon.awssdk.services.backupsearch.model.GetSearchResultExportJobRequest;
import software.amazon.awssdk.services.backupsearch.model.GetSearchResultExportJobResponse;
import software.amazon.awssdk.services.backupsearch.model.InternalServerException;
import software.amazon.awssdk.services.backupsearch.model.ListSearchJobBackupsRequest;
import software.amazon.awssdk.services.backupsearch.model.ListSearchJobBackupsResponse;
import software.amazon.awssdk.services.backupsearch.model.ListSearchJobResultsRequest;
import software.amazon.awssdk.services.backupsearch.model.ListSearchJobResultsResponse;
import software.amazon.awssdk.services.backupsearch.model.ListSearchJobsRequest;
import software.amazon.awssdk.services.backupsearch.model.ListSearchJobsResponse;
import software.amazon.awssdk.services.backupsearch.model.ListSearchResultExportJobsRequest;
import software.amazon.awssdk.services.backupsearch.model.ListSearchResultExportJobsResponse;
import software.amazon.awssdk.services.backupsearch.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.backupsearch.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.backupsearch.model.ResourceNotFoundException;
import software.amazon.awssdk.services.backupsearch.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.backupsearch.model.StartSearchJobRequest;
import software.amazon.awssdk.services.backupsearch.model.StartSearchJobResponse;
import software.amazon.awssdk.services.backupsearch.model.StartSearchResultExportJobRequest;
import software.amazon.awssdk.services.backupsearch.model.StartSearchResultExportJobResponse;
import software.amazon.awssdk.services.backupsearch.model.StopSearchJobRequest;
import software.amazon.awssdk.services.backupsearch.model.StopSearchJobResponse;
import software.amazon.awssdk.services.backupsearch.model.TagResourceRequest;
import software.amazon.awssdk.services.backupsearch.model.TagResourceResponse;
import software.amazon.awssdk.services.backupsearch.model.ThrottlingException;
import software.amazon.awssdk.services.backupsearch.model.UntagResourceRequest;
import software.amazon.awssdk.services.backupsearch.model.UntagResourceResponse;
import software.amazon.awssdk.services.backupsearch.model.ValidationException;
import software.amazon.awssdk.services.backupsearch.paginators.ListSearchJobBackupsIterable;
import software.amazon.awssdk.services.backupsearch.paginators.ListSearchJobResultsIterable;
import software.amazon.awssdk.services.backupsearch.paginators.ListSearchJobsIterable;
import software.amazon.awssdk.services.backupsearch.paginators.ListSearchResultExportJobsIterable;

/**
 * Service client for accessing AWS Backup Search. This can be created using the static {@link #builder()} method.
 *
 * <p>
 * <fullname>Backup Search</fullname>
 * <p>
 * Backup Search is the recovery point and item level search for Backup.
 * </p>
 * <p>
 * For additional information, see:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/api-reference.html">Backup API Reference</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/aws-backup/latest/devguide/whatisbackup.html">Backup Developer Guide</a>
 * </p>
 * </li>
 * </ul>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface BackupSearchClient extends AwsClient {
    String SERVICE_NAME = "backup-search";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "backup-search";

    /**
     * <p>
     * This operation retrieves metadata of a search job, including its progress.
     * </p>
     *
     * @param getSearchJobRequest
     * @return Result of the GetSearchJob operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found for this request.</p>
     *         <p>
     *         Confirm the resource information, such as the ARN or type is correct and exists, then retry the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupSearchClient.GetSearchJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/GetSearchJob" target="_top">AWS API
     *      Documentation</a>
     */
    default GetSearchJobResponse getSearchJob(GetSearchJobRequest getSearchJobRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, BackupSearchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * This operation retrieves metadata of a search job, including its progress.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetSearchJobRequest.Builder} avoiding the need to
     * create one manually via {@link GetSearchJobRequest#builder()}
     * </p>
     *
     * @param getSearchJobRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.backupsearch.model.GetSearchJobRequest.Builder} to create a
     *        request.
     * @return Result of the GetSearchJob operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found for this request.</p>
     *         <p>
     *         Confirm the resource information, such as the ARN or type is correct and exists, then retry the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupSearchClient.GetSearchJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/GetSearchJob" target="_top">AWS API
     *      Documentation</a>
     */
    default GetSearchJobResponse getSearchJob(Consumer<GetSearchJobRequest.Builder> getSearchJobRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, BackupSearchException {
        return getSearchJob(GetSearchJobRequest.builder().applyMutation(getSearchJobRequest).build());
    }

    /**
     * <p>
     * This operation retrieves the metadata of an export job.
     * </p>
     * <p>
     * An export job is an operation that transmits the results of a search job to a specified S3 bucket in a .csv file.
     * </p>
     * <p>
     * An export job allows you to retain results of a search beyond the search job's scheduled retention of 7 days.
     * </p>
     *
     * @param getSearchResultExportJobRequest
     * @return Result of the GetSearchResultExportJob operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found for this request.</p>
     *         <p>
     *         Confirm the resource information, such as the ARN or type is correct and exists, then retry the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupSearchClient.GetSearchResultExportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/GetSearchResultExportJob"
     *      target="_top">AWS API Documentation</a>
     */
    default GetSearchResultExportJobResponse getSearchResultExportJob(
            GetSearchResultExportJobRequest getSearchResultExportJobRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, BackupSearchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * This operation retrieves the metadata of an export job.
     * </p>
     * <p>
     * An export job is an operation that transmits the results of a search job to a specified S3 bucket in a .csv file.
     * </p>
     * <p>
     * An export job allows you to retain results of a search beyond the search job's scheduled retention of 7 days.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetSearchResultExportJobRequest.Builder} avoiding
     * the need to create one manually via {@link GetSearchResultExportJobRequest#builder()}
     * </p>
     *
     * @param getSearchResultExportJobRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.backupsearch.model.GetSearchResultExportJobRequest.Builder} to
     *        create a request.
     * @return Result of the GetSearchResultExportJob operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found for this request.</p>
     *         <p>
     *         Confirm the resource information, such as the ARN or type is correct and exists, then retry the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupSearchClient.GetSearchResultExportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/GetSearchResultExportJob"
     *      target="_top">AWS API Documentation</a>
     */
    default GetSearchResultExportJobResponse getSearchResultExportJob(
            Consumer<GetSearchResultExportJobRequest.Builder> getSearchResultExportJobRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, BackupSearchException {
        return getSearchResultExportJob(GetSearchResultExportJobRequest.builder().applyMutation(getSearchResultExportJobRequest)
                .build());
    }

    /**
     * <p>
     * This operation returns a list of all backups (recovery points) in a paginated format that were included in the
     * search job.
     * </p>
     * <p>
     * If a search does not display an expected backup in the results, you can call this operation to display each
     * backup included in the search. Any backups that were not included because they have a <code>FAILED</code> status
     * from a permissions issue will be displayed, along with a status message.
     * </p>
     * <p>
     * Only recovery points with a backup index that has a status of <code>ACTIVE</code> will be included in search
     * results. If the index has any other status, its status will be displayed along with a status message.
     * </p>
     *
     * @param listSearchJobBackupsRequest
     * @return Result of the ListSearchJobBackups operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found for this request.</p>
     *         <p>
     *         Confirm the resource information, such as the ARN or type is correct and exists, then retry the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupSearchClient.ListSearchJobBackups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/ListSearchJobBackups"
     *      target="_top">AWS API Documentation</a>
     */
    default ListSearchJobBackupsResponse listSearchJobBackups(ListSearchJobBackupsRequest listSearchJobBackupsRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, BackupSearchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * This operation returns a list of all backups (recovery points) in a paginated format that were included in the
     * search job.
     * </p>
     * <p>
     * If a search does not display an expected backup in the results, you can call this operation to display each
     * backup included in the search. Any backups that were not included because they have a <code>FAILED</code> status
     * from a permissions issue will be displayed, along with a status message.
     * </p>
     * <p>
     * Only recovery points with a backup index that has a status of <code>ACTIVE</code> will be included in search
     * results. If the index has any other status, its status will be displayed along with a status message.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListSearchJobBackupsRequest.Builder} avoiding the
     * need to create one manually via {@link ListSearchJobBackupsRequest#builder()}
     * </p>
     *
     * @param listSearchJobBackupsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.backupsearch.model.ListSearchJobBackupsRequest.Builder} to create a
     *        request.
     * @return Result of the ListSearchJobBackups operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found for this request.</p>
     *         <p>
     *         Confirm the resource information, such as the ARN or type is correct and exists, then retry the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupSearchClient.ListSearchJobBackups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/ListSearchJobBackups"
     *      target="_top">AWS API Documentation</a>
     */
    default ListSearchJobBackupsResponse listSearchJobBackups(
            Consumer<ListSearchJobBackupsRequest.Builder> listSearchJobBackupsRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, BackupSearchException {
        return listSearchJobBackups(ListSearchJobBackupsRequest.builder().applyMutation(listSearchJobBackupsRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listSearchJobBackups(software.amazon.awssdk.services.backupsearch.model.ListSearchJobBackupsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupsearch.paginators.ListSearchJobBackupsIterable responses = client.listSearchJobBackupsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.backupsearch.paginators.ListSearchJobBackupsIterable responses = client
     *             .listSearchJobBackupsPaginator(request);
     *     for (software.amazon.awssdk.services.backupsearch.model.ListSearchJobBackupsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupsearch.paginators.ListSearchJobBackupsIterable responses = client.listSearchJobBackupsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSearchJobBackups(software.amazon.awssdk.services.backupsearch.model.ListSearchJobBackupsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSearchJobBackupsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The resource was not found for this request.</p>
     *         <p>
     *         Confirm the resource information, such as the ARN or type is correct and exists, then retry the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupSearchClient.ListSearchJobBackups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/ListSearchJobBackups"
     *      target="_top">AWS API Documentation</a>
     */
    default ListSearchJobBackupsIterable listSearchJobBackupsPaginator(ListSearchJobBackupsRequest listSearchJobBackupsRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, BackupSearchException {
        return new ListSearchJobBackupsIterable(this, listSearchJobBackupsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listSearchJobBackups(software.amazon.awssdk.services.backupsearch.model.ListSearchJobBackupsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupsearch.paginators.ListSearchJobBackupsIterable responses = client.listSearchJobBackupsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.backupsearch.paginators.ListSearchJobBackupsIterable responses = client
     *             .listSearchJobBackupsPaginator(request);
     *     for (software.amazon.awssdk.services.backupsearch.model.ListSearchJobBackupsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupsearch.paginators.ListSearchJobBackupsIterable responses = client.listSearchJobBackupsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSearchJobBackups(software.amazon.awssdk.services.backupsearch.model.ListSearchJobBackupsRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListSearchJobBackupsRequest.Builder} avoiding the
     * need to create one manually via {@link ListSearchJobBackupsRequest#builder()}
     * </p>
     *
     * @param listSearchJobBackupsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.backupsearch.model.ListSearchJobBackupsRequest.Builder} to create a
     *        request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The resource was not found for this request.</p>
     *         <p>
     *         Confirm the resource information, such as the ARN or type is correct and exists, then retry the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupSearchClient.ListSearchJobBackups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/ListSearchJobBackups"
     *      target="_top">AWS API Documentation</a>
     */
    default ListSearchJobBackupsIterable listSearchJobBackupsPaginator(
            Consumer<ListSearchJobBackupsRequest.Builder> listSearchJobBackupsRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, BackupSearchException {
        return listSearchJobBackupsPaginator(ListSearchJobBackupsRequest.builder().applyMutation(listSearchJobBackupsRequest)
                .build());
    }

    /**
     * <p>
     * This operation returns a list of a specified search job.
     * </p>
     *
     * @param listSearchJobResultsRequest
     * @return Result of the ListSearchJobResults operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found for this request.</p>
     *         <p>
     *         Confirm the resource information, such as the ARN or type is correct and exists, then retry the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupSearchClient.ListSearchJobResults
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/ListSearchJobResults"
     *      target="_top">AWS API Documentation</a>
     */
    default ListSearchJobResultsResponse listSearchJobResults(ListSearchJobResultsRequest listSearchJobResultsRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, BackupSearchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * This operation returns a list of a specified search job.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListSearchJobResultsRequest.Builder} avoiding the
     * need to create one manually via {@link ListSearchJobResultsRequest#builder()}
     * </p>
     *
     * @param listSearchJobResultsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.backupsearch.model.ListSearchJobResultsRequest.Builder} to create a
     *        request.
     * @return Result of the ListSearchJobResults operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found for this request.</p>
     *         <p>
     *         Confirm the resource information, such as the ARN or type is correct and exists, then retry the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupSearchClient.ListSearchJobResults
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/ListSearchJobResults"
     *      target="_top">AWS API Documentation</a>
     */
    default ListSearchJobResultsResponse listSearchJobResults(
            Consumer<ListSearchJobResultsRequest.Builder> listSearchJobResultsRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, BackupSearchException {
        return listSearchJobResults(ListSearchJobResultsRequest.builder().applyMutation(listSearchJobResultsRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listSearchJobResults(software.amazon.awssdk.services.backupsearch.model.ListSearchJobResultsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupsearch.paginators.ListSearchJobResultsIterable responses = client.listSearchJobResultsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.backupsearch.paginators.ListSearchJobResultsIterable responses = client
     *             .listSearchJobResultsPaginator(request);
     *     for (software.amazon.awssdk.services.backupsearch.model.ListSearchJobResultsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupsearch.paginators.ListSearchJobResultsIterable responses = client.listSearchJobResultsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSearchJobResults(software.amazon.awssdk.services.backupsearch.model.ListSearchJobResultsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSearchJobResultsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The resource was not found for this request.</p>
     *         <p>
     *         Confirm the resource information, such as the ARN or type is correct and exists, then retry the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupSearchClient.ListSearchJobResults
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/ListSearchJobResults"
     *      target="_top">AWS API Documentation</a>
     */
    default ListSearchJobResultsIterable listSearchJobResultsPaginator(ListSearchJobResultsRequest listSearchJobResultsRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, BackupSearchException {
        return new ListSearchJobResultsIterable(this, listSearchJobResultsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listSearchJobResults(software.amazon.awssdk.services.backupsearch.model.ListSearchJobResultsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupsearch.paginators.ListSearchJobResultsIterable responses = client.listSearchJobResultsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.backupsearch.paginators.ListSearchJobResultsIterable responses = client
     *             .listSearchJobResultsPaginator(request);
     *     for (software.amazon.awssdk.services.backupsearch.model.ListSearchJobResultsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupsearch.paginators.ListSearchJobResultsIterable responses = client.listSearchJobResultsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSearchJobResults(software.amazon.awssdk.services.backupsearch.model.ListSearchJobResultsRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListSearchJobResultsRequest.Builder} avoiding the
     * need to create one manually via {@link ListSearchJobResultsRequest#builder()}
     * </p>
     *
     * @param listSearchJobResultsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.backupsearch.model.ListSearchJobResultsRequest.Builder} to create a
     *        request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The resource was not found for this request.</p>
     *         <p>
     *         Confirm the resource information, such as the ARN or type is correct and exists, then retry the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupSearchClient.ListSearchJobResults
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/ListSearchJobResults"
     *      target="_top">AWS API Documentation</a>
     */
    default ListSearchJobResultsIterable listSearchJobResultsPaginator(
            Consumer<ListSearchJobResultsRequest.Builder> listSearchJobResultsRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, BackupSearchException {
        return listSearchJobResultsPaginator(ListSearchJobResultsRequest.builder().applyMutation(listSearchJobResultsRequest)
                .build());
    }

    /**
     * <p>
     * This operation returns a list of search jobs belonging to an account.
     * </p>
     *
     * @param listSearchJobsRequest
     * @return Result of the ListSearchJobs operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupSearchClient.ListSearchJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/ListSearchJobs" target="_top">AWS
     *      API Documentation</a>
     */
    default ListSearchJobsResponse listSearchJobs(ListSearchJobsRequest listSearchJobsRequest) throws ThrottlingException,
            AccessDeniedException, ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            BackupSearchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * This operation returns a list of search jobs belonging to an account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListSearchJobsRequest.Builder} avoiding the need to
     * create one manually via {@link ListSearchJobsRequest#builder()}
     * </p>
     *
     * @param listSearchJobsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.backupsearch.model.ListSearchJobsRequest.Builder} to create a
     *        request.
     * @return Result of the ListSearchJobs operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupSearchClient.ListSearchJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/ListSearchJobs" target="_top">AWS
     *      API Documentation</a>
     */
    default ListSearchJobsResponse listSearchJobs(Consumer<ListSearchJobsRequest.Builder> listSearchJobsRequest)
            throws ThrottlingException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, BackupSearchException {
        return listSearchJobs(ListSearchJobsRequest.builder().applyMutation(listSearchJobsRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listSearchJobs(software.amazon.awssdk.services.backupsearch.model.ListSearchJobsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupsearch.paginators.ListSearchJobsIterable responses = client.listSearchJobsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.backupsearch.paginators.ListSearchJobsIterable responses = client
     *             .listSearchJobsPaginator(request);
     *     for (software.amazon.awssdk.services.backupsearch.model.ListSearchJobsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupsearch.paginators.ListSearchJobsIterable responses = client.listSearchJobsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSearchJobs(software.amazon.awssdk.services.backupsearch.model.ListSearchJobsRequest)} operation.</b>
     * </p>
     *
     * @param listSearchJobsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupSearchClient.ListSearchJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/ListSearchJobs" target="_top">AWS
     *      API Documentation</a>
     */
    default ListSearchJobsIterable listSearchJobsPaginator(ListSearchJobsRequest listSearchJobsRequest)
            throws ThrottlingException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, BackupSearchException {
        return new ListSearchJobsIterable(this, listSearchJobsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listSearchJobs(software.amazon.awssdk.services.backupsearch.model.ListSearchJobsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupsearch.paginators.ListSearchJobsIterable responses = client.listSearchJobsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.backupsearch.paginators.ListSearchJobsIterable responses = client
     *             .listSearchJobsPaginator(request);
     *     for (software.amazon.awssdk.services.backupsearch.model.ListSearchJobsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupsearch.paginators.ListSearchJobsIterable responses = client.listSearchJobsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSearchJobs(software.amazon.awssdk.services.backupsearch.model.ListSearchJobsRequest)} operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListSearchJobsRequest.Builder} avoiding the need to
     * create one manually via {@link ListSearchJobsRequest#builder()}
     * </p>
     *
     * @param listSearchJobsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.backupsearch.model.ListSearchJobsRequest.Builder} to create a
     *        request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupSearchClient.ListSearchJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/ListSearchJobs" target="_top">AWS
     *      API Documentation</a>
     */
    default ListSearchJobsIterable listSearchJobsPaginator(Consumer<ListSearchJobsRequest.Builder> listSearchJobsRequest)
            throws ThrottlingException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, BackupSearchException {
        return listSearchJobsPaginator(ListSearchJobsRequest.builder().applyMutation(listSearchJobsRequest).build());
    }

    /**
     * <p>
     * This operation exports search results of a search job to a specified destination S3 bucket.
     * </p>
     *
     * @param listSearchResultExportJobsRequest
     * @return Result of the ListSearchResultExportJobs operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found for this request.</p>
     *         <p>
     *         Confirm the resource information, such as the ARN or type is correct and exists, then retry the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ServiceQuotaExceededException
     *         The request denied due to exceeding the quota limits permitted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupSearchClient.ListSearchResultExportJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/ListSearchResultExportJobs"
     *      target="_top">AWS API Documentation</a>
     */
    default ListSearchResultExportJobsResponse listSearchResultExportJobs(
            ListSearchResultExportJobsRequest listSearchResultExportJobsRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, ValidationException, InternalServerException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, BackupSearchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * This operation exports search results of a search job to a specified destination S3 bucket.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListSearchResultExportJobsRequest.Builder} avoiding
     * the need to create one manually via {@link ListSearchResultExportJobsRequest#builder()}
     * </p>
     *
     * @param listSearchResultExportJobsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.backupsearch.model.ListSearchResultExportJobsRequest.Builder} to
     *        create a request.
     * @return Result of the ListSearchResultExportJobs operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found for this request.</p>
     *         <p>
     *         Confirm the resource information, such as the ARN or type is correct and exists, then retry the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ServiceQuotaExceededException
     *         The request denied due to exceeding the quota limits permitted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupSearchClient.ListSearchResultExportJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/ListSearchResultExportJobs"
     *      target="_top">AWS API Documentation</a>
     */
    default ListSearchResultExportJobsResponse listSearchResultExportJobs(
            Consumer<ListSearchResultExportJobsRequest.Builder> listSearchResultExportJobsRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, ValidationException,
            InternalServerException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            BackupSearchException {
        return listSearchResultExportJobs(ListSearchResultExportJobsRequest.builder()
                .applyMutation(listSearchResultExportJobsRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listSearchResultExportJobs(software.amazon.awssdk.services.backupsearch.model.ListSearchResultExportJobsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupsearch.paginators.ListSearchResultExportJobsIterable responses = client.listSearchResultExportJobsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.backupsearch.paginators.ListSearchResultExportJobsIterable responses = client
     *             .listSearchResultExportJobsPaginator(request);
     *     for (software.amazon.awssdk.services.backupsearch.model.ListSearchResultExportJobsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupsearch.paginators.ListSearchResultExportJobsIterable responses = client.listSearchResultExportJobsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSearchResultExportJobs(software.amazon.awssdk.services.backupsearch.model.ListSearchResultExportJobsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSearchResultExportJobsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The resource was not found for this request.</p>
     *         <p>
     *         Confirm the resource information, such as the ARN or type is correct and exists, then retry the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ServiceQuotaExceededException
     *         The request denied due to exceeding the quota limits permitted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupSearchClient.ListSearchResultExportJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/ListSearchResultExportJobs"
     *      target="_top">AWS API Documentation</a>
     */
    default ListSearchResultExportJobsIterable listSearchResultExportJobsPaginator(
            ListSearchResultExportJobsRequest listSearchResultExportJobsRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, ValidationException, InternalServerException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, BackupSearchException {
        return new ListSearchResultExportJobsIterable(this, listSearchResultExportJobsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listSearchResultExportJobs(software.amazon.awssdk.services.backupsearch.model.ListSearchResultExportJobsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupsearch.paginators.ListSearchResultExportJobsIterable responses = client.listSearchResultExportJobsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.backupsearch.paginators.ListSearchResultExportJobsIterable responses = client
     *             .listSearchResultExportJobsPaginator(request);
     *     for (software.amazon.awssdk.services.backupsearch.model.ListSearchResultExportJobsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.backupsearch.paginators.ListSearchResultExportJobsIterable responses = client.listSearchResultExportJobsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSearchResultExportJobs(software.amazon.awssdk.services.backupsearch.model.ListSearchResultExportJobsRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListSearchResultExportJobsRequest.Builder} avoiding
     * the need to create one manually via {@link ListSearchResultExportJobsRequest#builder()}
     * </p>
     *
     * @param listSearchResultExportJobsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.backupsearch.model.ListSearchResultExportJobsRequest.Builder} to
     *        create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The resource was not found for this request.</p>
     *         <p>
     *         Confirm the resource information, such as the ARN or type is correct and exists, then retry the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ServiceQuotaExceededException
     *         The request denied due to exceeding the quota limits permitted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupSearchClient.ListSearchResultExportJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/ListSearchResultExportJobs"
     *      target="_top">AWS API Documentation</a>
     */
    default ListSearchResultExportJobsIterable listSearchResultExportJobsPaginator(
            Consumer<ListSearchResultExportJobsRequest.Builder> listSearchResultExportJobsRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, ValidationException,
            InternalServerException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            BackupSearchException {
        return listSearchResultExportJobsPaginator(ListSearchResultExportJobsRequest.builder()
                .applyMutation(listSearchResultExportJobsRequest).build());
    }

    /**
     * <p>
     * This operation returns the tags for a resource type.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found for this request.</p>
     *         <p>
     *         Confirm the resource information, such as the ARN or type is correct and exists, then retry the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupSearchClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, BackupSearchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * This operation returns the tags for a resource type.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.backupsearch.model.ListTagsForResourceRequest.Builder} to create a
     *        request.
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found for this request.</p>
     *         <p>
     *         Confirm the resource information, such as the ARN or type is correct and exists, then retry the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupSearchClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, BackupSearchException {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * This operation creates a search job which returns recovery points filtered by SearchScope and items filtered by
     * ItemFilters.
     * </p>
     * <p>
     * You can optionally include ClientToken, EncryptionKeyArn, Name, and/or Tags.
     * </p>
     *
     * @param startSearchJobRequest
     * @return Result of the StartSearchJob operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ServiceQuotaExceededException
     *         The request denied due to exceeding the quota limits permitted.
     * @throws ConflictException
     *         This exception occurs when a conflict with a previous successful operation is detected. This generally
     *         occurs when the previous operation did not have time to propagate to the host serving the current
     *         request.</p>
     *         <p>
     *         A retry (with appropriate backoff logic) is the recommended response to this exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupSearchClient.StartSearchJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/StartSearchJob" target="_top">AWS
     *      API Documentation</a>
     */
    default StartSearchJobResponse startSearchJob(StartSearchJobRequest startSearchJobRequest) throws ThrottlingException,
            AccessDeniedException, ValidationException, InternalServerException, ServiceQuotaExceededException,
            ConflictException, AwsServiceException, SdkClientException, BackupSearchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * This operation creates a search job which returns recovery points filtered by SearchScope and items filtered by
     * ItemFilters.
     * </p>
     * <p>
     * You can optionally include ClientToken, EncryptionKeyArn, Name, and/or Tags.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StartSearchJobRequest.Builder} avoiding the need to
     * create one manually via {@link StartSearchJobRequest#builder()}
     * </p>
     *
     * @param startSearchJobRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.backupsearch.model.StartSearchJobRequest.Builder} to create a
     *        request.
     * @return Result of the StartSearchJob operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ServiceQuotaExceededException
     *         The request denied due to exceeding the quota limits permitted.
     * @throws ConflictException
     *         This exception occurs when a conflict with a previous successful operation is detected. This generally
     *         occurs when the previous operation did not have time to propagate to the host serving the current
     *         request.</p>
     *         <p>
     *         A retry (with appropriate backoff logic) is the recommended response to this exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupSearchClient.StartSearchJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/StartSearchJob" target="_top">AWS
     *      API Documentation</a>
     */
    default StartSearchJobResponse startSearchJob(Consumer<StartSearchJobRequest.Builder> startSearchJobRequest)
            throws ThrottlingException, AccessDeniedException, ValidationException, InternalServerException,
            ServiceQuotaExceededException, ConflictException, AwsServiceException, SdkClientException, BackupSearchException {
        return startSearchJob(StartSearchJobRequest.builder().applyMutation(startSearchJobRequest).build());
    }

    /**
     * <p>
     * This operations starts a job to export the results of search job to a designated S3 bucket.
     * </p>
     *
     * @param startSearchResultExportJobRequest
     * @return Result of the StartSearchResultExportJob operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found for this request.</p>
     *         <p>
     *         Confirm the resource information, such as the ARN or type is correct and exists, then retry the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ServiceQuotaExceededException
     *         The request denied due to exceeding the quota limits permitted.
     * @throws ConflictException
     *         This exception occurs when a conflict with a previous successful operation is detected. This generally
     *         occurs when the previous operation did not have time to propagate to the host serving the current
     *         request.
     *         </p>
     *         <p>
     *         A retry (with appropriate backoff logic) is the recommended response to this exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupSearchClient.StartSearchResultExportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/StartSearchResultExportJob"
     *      target="_top">AWS API Documentation</a>
     */
    default StartSearchResultExportJobResponse startSearchResultExportJob(
            StartSearchResultExportJobRequest startSearchResultExportJobRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, ValidationException, InternalServerException,
            ServiceQuotaExceededException, ConflictException, AwsServiceException, SdkClientException, BackupSearchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * This operations starts a job to export the results of search job to a designated S3 bucket.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StartSearchResultExportJobRequest.Builder} avoiding
     * the need to create one manually via {@link StartSearchResultExportJobRequest#builder()}
     * </p>
     *
     * @param startSearchResultExportJobRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.backupsearch.model.StartSearchResultExportJobRequest.Builder} to
     *        create a request.
     * @return Result of the StartSearchResultExportJob operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found for this request.</p>
     *         <p>
     *         Confirm the resource information, such as the ARN or type is correct and exists, then retry the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ServiceQuotaExceededException
     *         The request denied due to exceeding the quota limits permitted.
     * @throws ConflictException
     *         This exception occurs when a conflict with a previous successful operation is detected. This generally
     *         occurs when the previous operation did not have time to propagate to the host serving the current
     *         request.
     *         </p>
     *         <p>
     *         A retry (with appropriate backoff logic) is the recommended response to this exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupSearchClient.StartSearchResultExportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/StartSearchResultExportJob"
     *      target="_top">AWS API Documentation</a>
     */
    default StartSearchResultExportJobResponse startSearchResultExportJob(
            Consumer<StartSearchResultExportJobRequest.Builder> startSearchResultExportJobRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, ValidationException,
            InternalServerException, ServiceQuotaExceededException, ConflictException, AwsServiceException, SdkClientException,
            BackupSearchException {
        return startSearchResultExportJob(StartSearchResultExportJobRequest.builder()
                .applyMutation(startSearchResultExportJobRequest).build());
    }

    /**
     * <p>
     * This operations ends a search job.
     * </p>
     * <p>
     * Only a search job with a status of <code>RUNNING</code> can be stopped.
     * </p>
     *
     * @param stopSearchJobRequest
     * @return Result of the StopSearchJob operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found for this request.</p>
     *         <p>
     *         Confirm the resource information, such as the ARN or type is correct and exists, then retry the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ConflictException
     *         This exception occurs when a conflict with a previous successful operation is detected. This generally
     *         occurs when the previous operation did not have time to propagate to the host serving the current
     *         request.
     *         </p>
     *         <p>
     *         A retry (with appropriate backoff logic) is the recommended response to this exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupSearchClient.StopSearchJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/StopSearchJob" target="_top">AWS
     *      API Documentation</a>
     */
    default StopSearchJobResponse stopSearchJob(StopSearchJobRequest stopSearchJobRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, ValidationException, InternalServerException, ConflictException,
            AwsServiceException, SdkClientException, BackupSearchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * This operations ends a search job.
     * </p>
     * <p>
     * Only a search job with a status of <code>RUNNING</code> can be stopped.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StopSearchJobRequest.Builder} avoiding the need to
     * create one manually via {@link StopSearchJobRequest#builder()}
     * </p>
     *
     * @param stopSearchJobRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.backupsearch.model.StopSearchJobRequest.Builder} to create a
     *        request.
     * @return Result of the StopSearchJob operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found for this request.</p>
     *         <p>
     *         Confirm the resource information, such as the ARN or type is correct and exists, then retry the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ConflictException
     *         This exception occurs when a conflict with a previous successful operation is detected. This generally
     *         occurs when the previous operation did not have time to propagate to the host serving the current
     *         request.
     *         </p>
     *         <p>
     *         A retry (with appropriate backoff logic) is the recommended response to this exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupSearchClient.StopSearchJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/StopSearchJob" target="_top">AWS
     *      API Documentation</a>
     */
    default StopSearchJobResponse stopSearchJob(Consumer<StopSearchJobRequest.Builder> stopSearchJobRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, ValidationException,
            InternalServerException, ConflictException, AwsServiceException, SdkClientException, BackupSearchException {
        return stopSearchJob(StopSearchJobRequest.builder().applyMutation(stopSearchJobRequest).build());
    }

    /**
     * <p>
     * This operation puts tags on the resource you indicate.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found for this request.</p>
     *         <p>
     *         Confirm the resource information, such as the ARN or type is correct and exists, then retry the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupSearchClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, BackupSearchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * This operation puts tags on the resource you indicate.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.backupsearch.model.TagResourceRequest.Builder} to create a request.
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found for this request.</p>
     *         <p>
     *         Confirm the resource information, such as the ARN or type is correct and exists, then retry the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupSearchClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default TagResourceResponse tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, BackupSearchException {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * This operation removes tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found for this request.</p>
     *         <p>
     *         Confirm the resource information, such as the ARN or type is correct and exists, then retry the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupSearchClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    default UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, BackupSearchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * This operation removes tags from the specified resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.backupsearch.model.UntagResourceRequest.Builder} to create a
     *        request.
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The resource was not found for this request.</p>
     *         <p>
     *         Confirm the resource information, such as the ARN or type is correct and exists, then retry the request.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by a service.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BackupSearchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BackupSearchClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    default UntagResourceResponse untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest)
            throws ResourceNotFoundException, ThrottlingException, AccessDeniedException, ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, BackupSearchException {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    /**
     * Create a {@link BackupSearchClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static BackupSearchClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link BackupSearchClient}.
     */
    static BackupSearchClientBuilder builder() {
        return new DefaultBackupSearchClientBuilder();
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }

    @Override
    default BackupSearchServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }
}
