/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.backupsearch;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.backupsearch.internal.BackupSearchServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.backupsearch.model.AccessDeniedException;
import software.amazon.awssdk.services.backupsearch.model.BackupSearchException;
import software.amazon.awssdk.services.backupsearch.model.ConflictException;
import software.amazon.awssdk.services.backupsearch.model.GetSearchJobRequest;
import software.amazon.awssdk.services.backupsearch.model.GetSearchJobResponse;
import software.amazon.awssdk.services.backupsearch.model.GetSearchResultExportJobRequest;
import software.amazon.awssdk.services.backupsearch.model.GetSearchResultExportJobResponse;
import software.amazon.awssdk.services.backupsearch.model.InternalServerException;
import software.amazon.awssdk.services.backupsearch.model.ListSearchJobBackupsRequest;
import software.amazon.awssdk.services.backupsearch.model.ListSearchJobBackupsResponse;
import software.amazon.awssdk.services.backupsearch.model.ListSearchJobResultsRequest;
import software.amazon.awssdk.services.backupsearch.model.ListSearchJobResultsResponse;
import software.amazon.awssdk.services.backupsearch.model.ListSearchJobsRequest;
import software.amazon.awssdk.services.backupsearch.model.ListSearchJobsResponse;
import software.amazon.awssdk.services.backupsearch.model.ListSearchResultExportJobsRequest;
import software.amazon.awssdk.services.backupsearch.model.ListSearchResultExportJobsResponse;
import software.amazon.awssdk.services.backupsearch.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.backupsearch.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.backupsearch.model.ResourceNotFoundException;
import software.amazon.awssdk.services.backupsearch.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.backupsearch.model.StartSearchJobRequest;
import software.amazon.awssdk.services.backupsearch.model.StartSearchJobResponse;
import software.amazon.awssdk.services.backupsearch.model.StartSearchResultExportJobRequest;
import software.amazon.awssdk.services.backupsearch.model.StartSearchResultExportJobResponse;
import software.amazon.awssdk.services.backupsearch.model.StopSearchJobRequest;
import software.amazon.awssdk.services.backupsearch.model.StopSearchJobResponse;
import software.amazon.awssdk.services.backupsearch.model.TagResourceRequest;
import software.amazon.awssdk.services.backupsearch.model.TagResourceResponse;
import software.amazon.awssdk.services.backupsearch.model.ThrottlingException;
import software.amazon.awssdk.services.backupsearch.model.UntagResourceRequest;
import software.amazon.awssdk.services.backupsearch.model.UntagResourceResponse;
import software.amazon.awssdk.services.backupsearch.model.ValidationException;
import software.amazon.awssdk.services.backupsearch.transform.GetSearchJobRequestMarshaller;
import software.amazon.awssdk.services.backupsearch.transform.GetSearchResultExportJobRequestMarshaller;
import software.amazon.awssdk.services.backupsearch.transform.ListSearchJobBackupsRequestMarshaller;
import software.amazon.awssdk.services.backupsearch.transform.ListSearchJobResultsRequestMarshaller;
import software.amazon.awssdk.services.backupsearch.transform.ListSearchJobsRequestMarshaller;
import software.amazon.awssdk.services.backupsearch.transform.ListSearchResultExportJobsRequestMarshaller;
import software.amazon.awssdk.services.backupsearch.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.backupsearch.transform.StartSearchJobRequestMarshaller;
import software.amazon.awssdk.services.backupsearch.transform.StartSearchResultExportJobRequestMarshaller;
import software.amazon.awssdk.services.backupsearch.transform.StopSearchJobRequestMarshaller;
import software.amazon.awssdk.services.backupsearch.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.backupsearch.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link BackupSearchAsyncClient}.
 *
 * @see BackupSearchAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultBackupSearchAsyncClient implements BackupSearchAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultBackupSearchAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultBackupSearchAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * This operation retrieves metadata of a search job, including its progress.
     * </p>
     *
     * @param getSearchJobRequest
     * @return A Java Future containing the result of the GetSearchJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found for this request.</p>
     *         <p>
     *         Confirm the resource information, such as the ARN or type is correct and exists, then retry the request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by a service.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupSearchAsyncClient.GetSearchJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/GetSearchJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetSearchJobResponse> getSearchJob(GetSearchJobRequest getSearchJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSearchJobRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSearchJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BackupSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSearchJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSearchJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetSearchJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSearchJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSearchJobRequest, GetSearchJobResponse>()
                            .withOperationName("GetSearchJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetSearchJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getSearchJobRequest));
            CompletableFuture<GetSearchJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation retrieves the metadata of an export job.
     * </p>
     * <p>
     * An export job is an operation that transmits the results of a search job to a specified S3 bucket in a .csv file.
     * </p>
     * <p>
     * An export job allows you to retain results of a search beyond the search job's scheduled retention of 7 days.
     * </p>
     *
     * @param getSearchResultExportJobRequest
     * @return A Java Future containing the result of the GetSearchResultExportJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found for this request.</p>
     *         <p>
     *         Confirm the resource information, such as the ARN or type is correct and exists, then retry the request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by a service.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupSearchAsyncClient.GetSearchResultExportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/GetSearchResultExportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSearchResultExportJobResponse> getSearchResultExportJob(
            GetSearchResultExportJobRequest getSearchResultExportJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSearchResultExportJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSearchResultExportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BackupSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSearchResultExportJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSearchResultExportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSearchResultExportJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSearchResultExportJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSearchResultExportJobRequest, GetSearchResultExportJobResponse>()
                            .withOperationName("GetSearchResultExportJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetSearchResultExportJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getSearchResultExportJobRequest));
            CompletableFuture<GetSearchResultExportJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation returns a list of all backups (recovery points) in a paginated format that were included in the
     * search job.
     * </p>
     * <p>
     * If a search does not display an expected backup in the results, you can call this operation to display each
     * backup included in the search. Any backups that were not included because they have a <code>FAILED</code> status
     * from a permissions issue will be displayed, along with a status message.
     * </p>
     * <p>
     * Only recovery points with a backup index that has a status of <code>ACTIVE</code> will be included in search
     * results. If the index has any other status, its status will be displayed along with a status message.
     * </p>
     *
     * @param listSearchJobBackupsRequest
     * @return A Java Future containing the result of the ListSearchJobBackups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found for this request.</p>
     *         <p>
     *         Confirm the resource information, such as the ARN or type is correct and exists, then retry the request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by a service.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupSearchAsyncClient.ListSearchJobBackups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/ListSearchJobBackups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSearchJobBackupsResponse> listSearchJobBackups(
            ListSearchJobBackupsRequest listSearchJobBackupsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSearchJobBackupsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSearchJobBackupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BackupSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSearchJobBackups");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSearchJobBackupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSearchJobBackupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSearchJobBackupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSearchJobBackupsRequest, ListSearchJobBackupsResponse>()
                            .withOperationName("ListSearchJobBackups").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListSearchJobBackupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSearchJobBackupsRequest));
            CompletableFuture<ListSearchJobBackupsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation returns a list of a specified search job.
     * </p>
     *
     * @param listSearchJobResultsRequest
     * @return A Java Future containing the result of the ListSearchJobResults operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found for this request.</p>
     *         <p>
     *         Confirm the resource information, such as the ARN or type is correct and exists, then retry the request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by a service.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupSearchAsyncClient.ListSearchJobResults
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/ListSearchJobResults"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSearchJobResultsResponse> listSearchJobResults(
            ListSearchJobResultsRequest listSearchJobResultsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSearchJobResultsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSearchJobResultsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BackupSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSearchJobResults");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSearchJobResultsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSearchJobResultsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSearchJobResultsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSearchJobResultsRequest, ListSearchJobResultsResponse>()
                            .withOperationName("ListSearchJobResults").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListSearchJobResultsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSearchJobResultsRequest));
            CompletableFuture<ListSearchJobResultsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation returns a list of search jobs belonging to an account.
     * </p>
     *
     * @param listSearchJobsRequest
     * @return A Java Future containing the result of the ListSearchJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by a service.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupSearchAsyncClient.ListSearchJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/ListSearchJobs" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSearchJobsResponse> listSearchJobs(ListSearchJobsRequest listSearchJobsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSearchJobsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSearchJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BackupSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSearchJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSearchJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSearchJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSearchJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSearchJobsRequest, ListSearchJobsResponse>()
                            .withOperationName("ListSearchJobs").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListSearchJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSearchJobsRequest));
            CompletableFuture<ListSearchJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation exports search results of a search job to a specified destination S3 bucket.
     * </p>
     *
     * @param listSearchResultExportJobsRequest
     * @return A Java Future containing the result of the ListSearchResultExportJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found for this request.</p>
     *         <p>
     *         Confirm the resource information, such as the ARN or type is correct and exists, then retry the request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by a service.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ServiceQuotaExceededException The request denied due to exceeding the quota limits permitted.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupSearchAsyncClient.ListSearchResultExportJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/ListSearchResultExportJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSearchResultExportJobsResponse> listSearchResultExportJobs(
            ListSearchResultExportJobsRequest listSearchResultExportJobsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSearchResultExportJobsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSearchResultExportJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BackupSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSearchResultExportJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSearchResultExportJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSearchResultExportJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSearchResultExportJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSearchResultExportJobsRequest, ListSearchResultExportJobsResponse>()
                            .withOperationName("ListSearchResultExportJobs").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListSearchResultExportJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSearchResultExportJobsRequest));
            CompletableFuture<ListSearchResultExportJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation returns the tags for a resource type.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found for this request.</p>
     *         <p>
     *         Confirm the resource information, such as the ARN or type is correct and exists, then retry the request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by a service.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupSearchAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BackupSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation creates a search job which returns recovery points filtered by SearchScope and items filtered by
     * ItemFilters.
     * </p>
     * <p>
     * You can optionally include ClientToken, EncryptionKeyArn, Name, and/or Tags.
     * </p>
     *
     * @param startSearchJobRequest
     * @return A Java Future containing the result of the StartSearchJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by a service.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ServiceQuotaExceededException The request denied due to exceeding the quota limits permitted.</li>
     *         <li>ConflictException This exception occurs when a conflict with a previous successful operation is
     *         detected. This generally occurs when the previous operation did not have time to propagate to the host
     *         serving the current request.</p>
     *         <p>
     *         A retry (with appropriate backoff logic) is the recommended response to this exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupSearchAsyncClient.StartSearchJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/StartSearchJob" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<StartSearchJobResponse> startSearchJob(StartSearchJobRequest startSearchJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startSearchJobRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startSearchJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BackupSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartSearchJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartSearchJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartSearchJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartSearchJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartSearchJobRequest, StartSearchJobResponse>()
                            .withOperationName("StartSearchJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartSearchJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startSearchJobRequest));
            CompletableFuture<StartSearchJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operations starts a job to export the results of search job to a designated S3 bucket.
     * </p>
     *
     * @param startSearchResultExportJobRequest
     * @return A Java Future containing the result of the StartSearchResultExportJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found for this request.</p>
     *         <p>
     *         Confirm the resource information, such as the ARN or type is correct and exists, then retry the request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by a service.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ServiceQuotaExceededException The request denied due to exceeding the quota limits permitted.</li>
     *         <li>ConflictException This exception occurs when a conflict with a previous successful operation is
     *         detected. This generally occurs when the previous operation did not have time to propagate to the host
     *         serving the current request.</p>
     *         <p>
     *         A retry (with appropriate backoff logic) is the recommended response to this exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupSearchAsyncClient.StartSearchResultExportJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/StartSearchResultExportJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartSearchResultExportJobResponse> startSearchResultExportJob(
            StartSearchResultExportJobRequest startSearchResultExportJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startSearchResultExportJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startSearchResultExportJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BackupSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartSearchResultExportJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartSearchResultExportJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartSearchResultExportJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartSearchResultExportJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartSearchResultExportJobRequest, StartSearchResultExportJobResponse>()
                            .withOperationName("StartSearchResultExportJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartSearchResultExportJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startSearchResultExportJobRequest));
            CompletableFuture<StartSearchResultExportJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operations ends a search job.
     * </p>
     * <p>
     * Only a search job with a status of <code>RUNNING</code> can be stopped.
     * </p>
     *
     * @param stopSearchJobRequest
     * @return A Java Future containing the result of the StopSearchJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found for this request.</p>
     *         <p>
     *         Confirm the resource information, such as the ARN or type is correct and exists, then retry the request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by a service.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ConflictException This exception occurs when a conflict with a previous successful operation is
     *         detected. This generally occurs when the previous operation did not have time to propagate to the host
     *         serving the current request.</p>
     *         <p>
     *         A retry (with appropriate backoff logic) is the recommended response to this exception.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupSearchAsyncClient.StopSearchJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/StopSearchJob" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<StopSearchJobResponse> stopSearchJob(StopSearchJobRequest stopSearchJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopSearchJobRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopSearchJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BackupSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopSearchJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopSearchJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StopSearchJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopSearchJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopSearchJobRequest, StopSearchJobResponse>()
                            .withOperationName("StopSearchJob").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StopSearchJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(stopSearchJobRequest));
            CompletableFuture<StopSearchJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation puts tags on the resource you indicate.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found for this request.</p>
     *         <p>
     *         Confirm the resource information, such as the ARN or type is correct and exists, then retry the request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by a service.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupSearchAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BackupSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation removes tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The resource was not found for this request.</p>
     *         <p>
     *         Confirm the resource information, such as the ARN or type is correct and exists, then retry the request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by a service.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupSearchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupSearchAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupsearch-2018-05-10/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BackupSearch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final BackupSearchServiceClientConfiguration serviceClientConfiguration() {
        return new BackupSearchServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(BackupSearchException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        BackupSearchServiceClientConfigurationBuilder serviceConfigBuilder = new BackupSearchServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
