/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.backupstorage;

import static software.amazon.awssdk.utils.FunctionalUtils.runAndLogError;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.auth.signer.Aws4UnsignedPayloadSigner;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.async.AsyncRequestBody;
import software.amazon.awssdk.core.async.AsyncResponseTransformer;
import software.amazon.awssdk.core.async.AsyncResponseTransformerUtils;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.runtime.transform.AsyncStreamingRequestMarshaller;
import software.amazon.awssdk.core.signer.Signer;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.backupstorage.internal.BackupStorageServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.backupstorage.model.AccessDeniedException;
import software.amazon.awssdk.services.backupstorage.model.BackupStorageException;
import software.amazon.awssdk.services.backupstorage.model.BackupStorageRequest;
import software.amazon.awssdk.services.backupstorage.model.DataAlreadyExistsException;
import software.amazon.awssdk.services.backupstorage.model.DeleteObjectRequest;
import software.amazon.awssdk.services.backupstorage.model.DeleteObjectResponse;
import software.amazon.awssdk.services.backupstorage.model.GetChunkRequest;
import software.amazon.awssdk.services.backupstorage.model.GetChunkResponse;
import software.amazon.awssdk.services.backupstorage.model.GetObjectMetadataRequest;
import software.amazon.awssdk.services.backupstorage.model.GetObjectMetadataResponse;
import software.amazon.awssdk.services.backupstorage.model.IllegalArgumentException;
import software.amazon.awssdk.services.backupstorage.model.KmsInvalidKeyUsageException;
import software.amazon.awssdk.services.backupstorage.model.ListChunksRequest;
import software.amazon.awssdk.services.backupstorage.model.ListChunksResponse;
import software.amazon.awssdk.services.backupstorage.model.ListObjectsRequest;
import software.amazon.awssdk.services.backupstorage.model.ListObjectsResponse;
import software.amazon.awssdk.services.backupstorage.model.NotReadableInputStreamException;
import software.amazon.awssdk.services.backupstorage.model.NotifyObjectCompleteRequest;
import software.amazon.awssdk.services.backupstorage.model.NotifyObjectCompleteResponse;
import software.amazon.awssdk.services.backupstorage.model.PutChunkRequest;
import software.amazon.awssdk.services.backupstorage.model.PutChunkResponse;
import software.amazon.awssdk.services.backupstorage.model.PutObjectRequest;
import software.amazon.awssdk.services.backupstorage.model.PutObjectResponse;
import software.amazon.awssdk.services.backupstorage.model.ResourceNotFoundException;
import software.amazon.awssdk.services.backupstorage.model.RetryableException;
import software.amazon.awssdk.services.backupstorage.model.ServiceInternalException;
import software.amazon.awssdk.services.backupstorage.model.ServiceUnavailableException;
import software.amazon.awssdk.services.backupstorage.model.StartObjectRequest;
import software.amazon.awssdk.services.backupstorage.model.StartObjectResponse;
import software.amazon.awssdk.services.backupstorage.model.ThrottlingException;
import software.amazon.awssdk.services.backupstorage.transform.DeleteObjectRequestMarshaller;
import software.amazon.awssdk.services.backupstorage.transform.GetChunkRequestMarshaller;
import software.amazon.awssdk.services.backupstorage.transform.GetObjectMetadataRequestMarshaller;
import software.amazon.awssdk.services.backupstorage.transform.ListChunksRequestMarshaller;
import software.amazon.awssdk.services.backupstorage.transform.ListObjectsRequestMarshaller;
import software.amazon.awssdk.services.backupstorage.transform.NotifyObjectCompleteRequestMarshaller;
import software.amazon.awssdk.services.backupstorage.transform.PutChunkRequestMarshaller;
import software.amazon.awssdk.services.backupstorage.transform.PutObjectRequestMarshaller;
import software.amazon.awssdk.services.backupstorage.transform.StartObjectRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;
import software.amazon.awssdk.utils.Pair;

/**
 * Internal implementation of {@link BackupStorageAsyncClient}.
 *
 * @see BackupStorageAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultBackupStorageAsyncClient implements BackupStorageAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultBackupStorageAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final BackupStorageServiceClientConfiguration serviceClientConfiguration;

    protected DefaultBackupStorageAsyncClient(BackupStorageServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * Delete Object from the incremental base Backup.
     *
     * @param deleteObjectRequest
     * @return A Java Future containing the result of the DeleteObject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException Retryable exception, indicates internal server error.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>ResourceNotFoundException Non-retryable exception. Attempted to make an operation on non-existing or
     *         expired resource.</li>
     *         <li>ThrottlingException Increased rate over throttling limits. Can be retried with exponential backoff.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.DeleteObject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/DeleteObject" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteObjectResponse> deleteObject(DeleteObjectRequest deleteObjectRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteObjectRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteObjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BackupStorage");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteObject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteObjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteObjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteObjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteObjectRequest, DeleteObjectResponse>()
                            .withOperationName("DeleteObject").withMarshaller(new DeleteObjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteObjectRequest));
            CompletableFuture<DeleteObjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Gets the specified object's chunk.
     *
     * @param getChunkRequest
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows 'Chunk data'.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>ResourceNotFoundException Non-retryable exception. Attempted to make an operation on non-existing or
     *         expired resource.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>ThrottlingException Increased rate over throttling limits. Can be retried with exponential backoff.</li>
     *         <li>KmsInvalidKeyUsageException Non-retryable exception. Indicates the KMS key usage is incorrect. See
     *         exception message for details.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.GetChunk
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/GetChunk" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public <ReturnT> CompletableFuture<ReturnT> getChunk(GetChunkRequest getChunkRequest,
            AsyncResponseTransformer<GetChunkResponse, ReturnT> asyncResponseTransformer) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getChunkRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getChunkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BackupStorage");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetChunk");
            Pair<AsyncResponseTransformer<GetChunkResponse, ReturnT>, CompletableFuture<Void>> pair = AsyncResponseTransformerUtils
                    .wrapWithEndOfStreamFuture(asyncResponseTransformer);
            asyncResponseTransformer = pair.left();
            CompletableFuture<Void> endOfStreamFuture = pair.right();
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(true)
                    .isPayloadJson(false).build();

            HttpResponseHandler<GetChunkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetChunkResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ReturnT> executeFuture = clientHandler.execute(
                    new ClientExecutionParams<GetChunkRequest, GetChunkResponse>().withOperationName("GetChunk")
                            .withMarshaller(new GetChunkRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getChunkRequest), asyncResponseTransformer);
            AsyncResponseTransformer<GetChunkResponse, ReturnT> finalAsyncResponseTransformer = asyncResponseTransformer;
            CompletableFuture<ReturnT> whenCompleted = executeFuture.whenComplete((r, e) -> {
                if (e != null) {
                    runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                            () -> finalAsyncResponseTransformer.exceptionOccurred(e));
                }
                endOfStreamFuture.whenComplete((r2, e2) -> {
                    metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                });
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            AsyncResponseTransformer<GetChunkResponse, ReturnT> finalAsyncResponseTransformer = asyncResponseTransformer;
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> finalAsyncResponseTransformer.exceptionOccurred(t));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Get metadata associated with an Object.
     *
     * @param getObjectMetadataRequest
     * @param asyncResponseTransformer
     *        The response transformer for processing the streaming response in a non-blocking manner. See
     *        {@link AsyncResponseTransformer} for details on how this callback should be implemented and for links to
     *        precanned implementations for common scenarios like downloading to a file. The service documentation for
     *        the response content is as follows 'Metadata blob.'.
     * @return A future to the transformed result of the AsyncResponseTransformer.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException Retryable exception, indicates internal server error.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>ResourceNotFoundException Non-retryable exception. Attempted to make an operation on non-existing or
     *         expired resource.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>ThrottlingException Increased rate over throttling limits. Can be retried with exponential backoff.</li>
     *         <li>KmsInvalidKeyUsageException Non-retryable exception. Indicates the KMS key usage is incorrect. See
     *         exception message for details.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.GetObjectMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/GetObjectMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public <ReturnT> CompletableFuture<ReturnT> getObjectMetadata(GetObjectMetadataRequest getObjectMetadataRequest,
            AsyncResponseTransformer<GetObjectMetadataResponse, ReturnT> asyncResponseTransformer) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getObjectMetadataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getObjectMetadataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BackupStorage");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetObjectMetadata");
            Pair<AsyncResponseTransformer<GetObjectMetadataResponse, ReturnT>, CompletableFuture<Void>> pair = AsyncResponseTransformerUtils
                    .wrapWithEndOfStreamFuture(asyncResponseTransformer);
            asyncResponseTransformer = pair.left();
            CompletableFuture<Void> endOfStreamFuture = pair.right();
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(true)
                    .isPayloadJson(false).build();

            HttpResponseHandler<GetObjectMetadataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetObjectMetadataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ReturnT> executeFuture = clientHandler.execute(
                    new ClientExecutionParams<GetObjectMetadataRequest, GetObjectMetadataResponse>()
                            .withOperationName("GetObjectMetadata")
                            .withMarshaller(new GetObjectMetadataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getObjectMetadataRequest), asyncResponseTransformer);
            AsyncResponseTransformer<GetObjectMetadataResponse, ReturnT> finalAsyncResponseTransformer = asyncResponseTransformer;
            CompletableFuture<ReturnT> whenCompleted = executeFuture.whenComplete((r, e) -> {
                if (e != null) {
                    runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                            () -> finalAsyncResponseTransformer.exceptionOccurred(e));
                }
                endOfStreamFuture.whenComplete((r2, e2) -> {
                    metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                });
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            AsyncResponseTransformer<GetObjectMetadataResponse, ReturnT> finalAsyncResponseTransformer = asyncResponseTransformer;
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> finalAsyncResponseTransformer.exceptionOccurred(t));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * List chunks in a given Object
     *
     * @param listChunksRequest
     * @return A Java Future containing the result of the ListChunks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException Retryable exception, indicates internal server error.</li>
     *         <li>ResourceNotFoundException Non-retryable exception. Attempted to make an operation on non-existing or
     *         expired resource.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.ListChunks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/ListChunks" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListChunksResponse> listChunks(ListChunksRequest listChunksRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listChunksRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listChunksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BackupStorage");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListChunks");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListChunksResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListChunksResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListChunksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListChunksRequest, ListChunksResponse>().withOperationName("ListChunks")
                            .withMarshaller(new ListChunksRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listChunksRequest));
            CompletableFuture<ListChunksResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * List all Objects in a given Backup.
     *
     * @param listObjectsRequest
     * @return A Java Future containing the result of the ListObjects operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException Retryable exception, indicates internal server error.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>ThrottlingException Increased rate over throttling limits. Can be retried with exponential backoff.</li>
     *         <li>ResourceNotFoundException Non-retryable exception. Attempted to make an operation on non-existing or
     *         expired resource.</li>
     *         <li>KmsInvalidKeyUsageException Non-retryable exception. Indicates the KMS key usage is incorrect. See
     *         exception message for details.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.ListObjects
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/ListObjects" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListObjectsResponse> listObjects(ListObjectsRequest listObjectsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listObjectsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listObjectsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BackupStorage");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListObjects");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListObjectsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListObjectsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListObjectsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListObjectsRequest, ListObjectsResponse>()
                            .withOperationName("ListObjects").withMarshaller(new ListObjectsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listObjectsRequest));
            CompletableFuture<ListObjectsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Complete upload
     *
     * @param notifyObjectCompleteRequest
     * @param requestBody
     *        Functional interface that can be implemented to produce the request content in a non-blocking manner. The
     *        size of the content is expected to be known up front. See {@link AsyncRequestBody} for specific details on
     *        implementing this interface as well as links to precanned implementations for common scenarios like
     *        uploading from a file. The service documentation for the request content is as follows 'Optional metadata
     *        associated with an Object. Maximum length is 4MB.'
     * @return A Java Future containing the result of the NotifyObjectComplete operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException Retryable exception, indicates internal server error.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>NotReadableInputStreamException Retryalble exception. Indicated issues while reading an input stream
     *         due to the networking issues or connection drop on the client side.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>ThrottlingException Increased rate over throttling limits. Can be retried with exponential backoff.</li>
     *         <li>KmsInvalidKeyUsageException Non-retryable exception. Indicates the KMS key usage is incorrect. See
     *         exception message for details.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.NotifyObjectComplete
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/NotifyObjectComplete"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<NotifyObjectCompleteResponse> notifyObjectComplete(
            NotifyObjectCompleteRequest notifyObjectCompleteRequest, AsyncRequestBody requestBody) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(notifyObjectCompleteRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, notifyObjectCompleteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BackupStorage");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "NotifyObjectComplete");
            notifyObjectCompleteRequest = applySignerOverride(notifyObjectCompleteRequest, Aws4UnsignedPayloadSigner.create());
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<NotifyObjectCompleteResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, NotifyObjectCompleteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<NotifyObjectCompleteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<NotifyObjectCompleteRequest, NotifyObjectCompleteResponse>()
                            .withOperationName("NotifyObjectComplete")
                            .withMarshaller(
                                    AsyncStreamingRequestMarshaller.builder()
                                            .delegateMarshaller(new NotifyObjectCompleteRequestMarshaller(protocolFactory))
                                            .asyncRequestBody(requestBody).transferEncoding(true).build())
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withAsyncRequestBody(requestBody).withInput(notifyObjectCompleteRequest));
            CompletableFuture<NotifyObjectCompleteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Upload chunk.
     *
     * @param putChunkRequest
     * @param requestBody
     *        Functional interface that can be implemented to produce the request content in a non-blocking manner. The
     *        size of the content is expected to be known up front. See {@link AsyncRequestBody} for specific details on
     *        implementing this interface as well as links to precanned implementations for common scenarios like
     *        uploading from a file. The service documentation for the request content is as follows 'Data to be
     *        uploaded'
     * @return A Java Future containing the result of the PutChunk operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException Retryable exception, indicates internal server error.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>NotReadableInputStreamException Retryalble exception. Indicated issues while reading an input stream
     *         due to the networking issues or connection drop on the client side.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>ThrottlingException Increased rate over throttling limits. Can be retried with exponential backoff.</li>
     *         <li>KmsInvalidKeyUsageException Non-retryable exception. Indicates the KMS key usage is incorrect. See
     *         exception message for details.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.PutChunk
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/PutChunk" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutChunkResponse> putChunk(PutChunkRequest putChunkRequest, AsyncRequestBody requestBody) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putChunkRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putChunkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BackupStorage");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutChunk");
            putChunkRequest = applySignerOverride(putChunkRequest, Aws4UnsignedPayloadSigner.create());
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutChunkResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutChunkResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutChunkResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutChunkRequest, PutChunkResponse>()
                            .withOperationName("PutChunk")
                            .withMarshaller(
                                    AsyncStreamingRequestMarshaller.builder()
                                            .delegateMarshaller(new PutChunkRequestMarshaller(protocolFactory))
                                            .asyncRequestBody(requestBody).transferEncoding(true).build())
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withAsyncRequestBody(requestBody).withInput(putChunkRequest));
            CompletableFuture<PutChunkResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Upload object that can store object metadata String and data blob in single API call using inline chunk field.
     *
     * @param putObjectRequest
     * @param requestBody
     *        Functional interface that can be implemented to produce the request content in a non-blocking manner. The
     *        size of the content is expected to be known up front. See {@link AsyncRequestBody} for specific details on
     *        implementing this interface as well as links to precanned implementations for common scenarios like
     *        uploading from a file. The service documentation for the request content is as follows 'Inline chunk data
     *        to be uploaded.'
     * @return A Java Future containing the result of the PutObject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException Retryable exception, indicates internal server error.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>NotReadableInputStreamException Retryalble exception. Indicated issues while reading an input stream
     *         due to the networking issues or connection drop on the client side.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>ThrottlingException Increased rate over throttling limits. Can be retried with exponential backoff.</li>
     *         <li>KmsInvalidKeyUsageException Non-retryable exception. Indicates the KMS key usage is incorrect. See
     *         exception message for details.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.PutObject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/PutObject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutObjectResponse> putObject(PutObjectRequest putObjectRequest, AsyncRequestBody requestBody) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putObjectRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putObjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BackupStorage");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutObject");
            putObjectRequest = applySignerOverride(putObjectRequest, Aws4UnsignedPayloadSigner.create());
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutObjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutObjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutObjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutObjectRequest, PutObjectResponse>()
                            .withOperationName("PutObject")
                            .withMarshaller(
                                    AsyncStreamingRequestMarshaller.builder()
                                            .delegateMarshaller(new PutObjectRequestMarshaller(protocolFactory))
                                            .asyncRequestBody(requestBody).transferEncoding(true).build())
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withAsyncRequestBody(requestBody).withInput(putObjectRequest));
            CompletableFuture<PutObjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Start upload containing one or many chunks.
     *
     * @param startObjectRequest
     * @return A Java Future containing the result of the StartObject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceUnavailableException Retryable exception, indicates internal server error.</li>
     *         <li>ServiceInternalException Deprecated. To be removed from the model.</li>
     *         <li>RetryableException Retryable exception. In general indicates internal failure that can be fixed by
     *         retry.</li>
     *         <li>IllegalArgumentException Non-retryable exception, indicates client error (wrong argument passed to
     *         API). See exception message for details.</li>
     *         <li>ResourceNotFoundException Non-retryable exception. Attempted to make an operation on non-existing or
     *         expired resource.</li>
     *         <li>DataAlreadyExistsException Non-retryable exception. Attempted to create already existing object or
     *         chunk. This message contains a checksum of already presented data.</li>
     *         <li>ThrottlingException Increased rate over throttling limits. Can be retried with exponential backoff.</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BackupStorageException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BackupStorageAsyncClient.StartObject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/backupstorage-2018-04-10/StartObject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartObjectResponse> startObject(StartObjectRequest startObjectRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startObjectRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startObjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BackupStorage");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartObject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartObjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StartObjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartObjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartObjectRequest, StartObjectResponse>()
                            .withOperationName("StartObject").withMarshaller(new StartObjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startObjectRequest));
            CompletableFuture<StartObjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final BackupStorageServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(BackupStorageException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KMSInvalidKeyUsageException")
                                .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotReadableInputStreamException")
                                .exceptionBuilderSupplier(NotReadableInputStreamException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IllegalArgumentException")
                                .exceptionBuilderSupplier(IllegalArgumentException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DataAlreadyExistsException")
                                .exceptionBuilderSupplier(DataAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceInternalException")
                                .exceptionBuilderSupplier(ServiceInternalException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RetryableException")
                                .exceptionBuilderSupplier(RetryableException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends BackupStorageRequest> T applySignerOverride(T request, Signer signer) {
        if (request.overrideConfiguration().flatMap(c -> c.signer()).isPresent()) {
            return request;
        }
        Consumer<AwsRequestOverrideConfiguration.Builder> signerOverride = b -> b.signer(signer).build();
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(signerOverride).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(signerOverride).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private static boolean isSignerOverridden(SdkClientConfiguration clientConfiguration) {
        return Boolean.TRUE.equals(clientConfiguration.option(SdkClientOption.SIGNER_OVERRIDDEN));
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        BackupStorageServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = BackupStorageServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
