/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.batch.model;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import javax.annotation.Generated;
import software.amazon.awssdk.annotation.SdkInternalApi;
import software.amazon.awssdk.protocol.ProtocolMarshaller;
import software.amazon.awssdk.protocol.StructuredPojo;
import software.amazon.awssdk.services.batch.transform.ContainerPropertiesMarshaller;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Container properties are used in job definitions to describe the container that is launched as part of a job.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class ContainerProperties implements StructuredPojo, ToCopyableBuilder<ContainerProperties.Builder, ContainerProperties> {
    private final String image;

    private final Integer vcpus;

    private final Integer memory;

    private final List<String> command;

    private final String jobRoleArn;

    private final List<Volume> volumes;

    private final List<KeyValuePair> environment;

    private final List<MountPoint> mountPoints;

    private final Boolean readonlyRootFilesystem;

    private final Boolean privileged;

    private final List<Ulimit> ulimits;

    private final String user;

    private ContainerProperties(BuilderImpl builder) {
        this.image = builder.image;
        this.vcpus = builder.vcpus;
        this.memory = builder.memory;
        this.command = builder.command;
        this.jobRoleArn = builder.jobRoleArn;
        this.volumes = builder.volumes;
        this.environment = builder.environment;
        this.mountPoints = builder.mountPoints;
        this.readonlyRootFilesystem = builder.readonlyRootFilesystem;
        this.privileged = builder.privileged;
        this.ulimits = builder.ulimits;
        this.user = builder.user;
    }

    /**
     * <p>
     * The image used to start a container. This string is passed directly to the Docker daemon. Images in the Docker
     * Hub registry are available by default. Other repositories are specified with
     * <code> <i>repository-url</i>/<i>image</i>:<i>tag</i> </code>. Up to 255 letters (uppercase and lowercase),
     * numbers, hyphens, underscores, colons, periods, forward slashes, and number signs are allowed. This parameter
     * maps to <code>Image</code> in the <a
     * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create a
     * container</a> section of the <a
     * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and the
     * <code>IMAGE</code> parameter of <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Images in Amazon ECR repositories use the full registry and repository URI (for example,
     * <code>012345678910.dkr.ecr.&lt;region-name&gt;.amazonaws.com/&lt;repository-name&gt;</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * Images in official repositories on Docker Hub use a single name (for example, <code>ubuntu</code> or
     * <code>mongo</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * Images in other repositories on Docker Hub are qualified with an organization name (for example,
     * <code>amazon/amazon-ecs-agent</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * Images in other online repositories are qualified further by a domain name (for example,
     * <code>quay.io/assemblyline/ubuntu</code>).
     * </p>
     * </li>
     * </ul>
     * 
     * @return The image used to start a container. This string is passed directly to the Docker daemon. Images in the
     *         Docker Hub registry are available by default. Other repositories are specified with
     *         <code> <i>repository-url</i>/<i>image</i>:<i>tag</i> </code>. Up to 255 letters (uppercase and
     *         lowercase), numbers, hyphens, underscores, colons, periods, forward slashes, and number signs are
     *         allowed. This parameter maps to <code>Image</code> in the <a
     *         href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create a
     *         container</a> section of the <a
     *         href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and
     *         the <code>IMAGE</code> parameter of <a href="https://docs.docker.com/engine/reference/run/">docker
     *         run</a>.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Images in Amazon ECR repositories use the full registry and repository URI (for example,
     *         <code>012345678910.dkr.ecr.&lt;region-name&gt;.amazonaws.com/&lt;repository-name&gt;</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Images in official repositories on Docker Hub use a single name (for example, <code>ubuntu</code> or
     *         <code>mongo</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Images in other repositories on Docker Hub are qualified with an organization name (for example,
     *         <code>amazon/amazon-ecs-agent</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Images in other online repositories are qualified further by a domain name (for example,
     *         <code>quay.io/assemblyline/ubuntu</code>).
     *         </p>
     *         </li>
     */
    public String image() {
        return image;
    }

    /**
     * <p>
     * The number of vCPUs reserved for the container. This parameter maps to <code>CpuShares</code> in the <a
     * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create a
     * container</a> section of the <a
     * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and the
     * <code>--cpu-shares</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>. Each
     * vCPU is equivalent to 1,024 CPU shares.
     * </p>
     * 
     * @return The number of vCPUs reserved for the container. This parameter maps to <code>CpuShares</code> in the <a
     *         href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create a
     *         container</a> section of the <a
     *         href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and
     *         the <code>--cpu-shares</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
     *         run</a>. Each vCPU is equivalent to 1,024 CPU shares.
     */
    public Integer vcpus() {
        return vcpus;
    }

    /**
     * <p>
     * The hard limit (in MiB) of memory to present to the container. If your container attempts to exceed the memory
     * specified here, the container is killed. This parameter maps to <code>Memory</code> in the <a
     * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create a
     * container</a> section of the <a
     * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and the
     * <code>--memory</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
     * </p>
     * 
     * @return The hard limit (in MiB) of memory to present to the container. If your container attempts to exceed the
     *         memory specified here, the container is killed. This parameter maps to <code>Memory</code> in the <a
     *         href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create a
     *         container</a> section of the <a
     *         href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and
     *         the <code>--memory</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
     *         run</a>.
     */
    public Integer memory() {
        return memory;
    }

    /**
     * <p>
     * The command that is passed to the container. This parameter maps to <code>Cmd</code> in the <a
     * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create a
     * container</a> section of the <a
     * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and the
     * <code>COMMAND</code> parameter to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>. For
     * more information, see <a
     * href="https://docs.docker.com/engine/reference/builder/#cmd">https://docs.docker.com/engine
     * /reference/builder/#cmd</a>.
     * </p>
     * 
     * @return The command that is passed to the container. This parameter maps to <code>Cmd</code> in the <a
     *         href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create a
     *         container</a> section of the <a
     *         href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and
     *         the <code>COMMAND</code> parameter to <a href="https://docs.docker.com/engine/reference/run/">docker
     *         run</a>. For more information, see <a
     *         href="https://docs.docker.com/engine/reference/builder/#cmd">https:/
     *         /docs.docker.com/engine/reference/builder/#cmd</a>.
     */
    public List<String> command() {
        return command;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the IAM role that the container can assume for AWS permissions.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the IAM role that the container can assume for AWS permissions.
     */
    public String jobRoleArn() {
        return jobRoleArn;
    }

    /**
     * <p>
     * A list of data volumes used in a job.
     * </p>
     * 
     * @return A list of data volumes used in a job.
     */
    public List<Volume> volumes() {
        return volumes;
    }

    /**
     * <p>
     * The environment variables to pass to a container. This parameter maps to <code>Env</code> in the <a
     * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create a
     * container</a> section of the <a
     * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and the
     * <code>--env</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
     * </p>
     * <important>
     * <p>
     * We do not recommend using plain text environment variables for sensitive information, such as credential data.
     * </p>
     * </important>
     * 
     * @return The environment variables to pass to a container. This parameter maps to <code>Env</code> in the <a
     *         href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create a
     *         container</a> section of the <a
     *         href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and
     *         the <code>--env</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
     *         run</a>.</p> <important>
     *         <p>
     *         We do not recommend using plain text environment variables for sensitive information, such as credential
     *         data.
     *         </p>
     */
    public List<KeyValuePair> environment() {
        return environment;
    }

    /**
     * <p>
     * The mount points for data volumes in your container. This parameter maps to <code>Volumes</code> in the <a
     * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create a
     * container</a> section of the <a
     * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and the
     * <code>--volume</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
     * </p>
     * 
     * @return The mount points for data volumes in your container. This parameter maps to <code>Volumes</code> in the
     *         <a href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create
     *         a container</a> section of the <a
     *         href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and
     *         the <code>--volume</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
     *         run</a>.
     */
    public List<MountPoint> mountPoints() {
        return mountPoints;
    }

    /**
     * <p>
     * When this parameter is true, the container is given read-only access to its root file system. This parameter maps
     * to <code>ReadonlyRootfs</code> in the <a
     * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create a
     * container</a> section of the <a
     * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and the
     * <code>--read-only</code> option to <code>docker run</code>.
     * </p>
     * 
     * @return When this parameter is true, the container is given read-only access to its root file system. This
     *         parameter maps to <code>ReadonlyRootfs</code> in the <a
     *         href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create a
     *         container</a> section of the <a
     *         href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and
     *         the <code>--read-only</code> option to <code>docker run</code>.
     */
    public Boolean readonlyRootFilesystem() {
        return readonlyRootFilesystem;
    }

    /**
     * <p>
     * When this parameter is true, the container is given elevated privileges on the host container instance (similar
     * to the <code>root</code> user). This parameter maps to <code>Privileged</code> in the <a
     * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create a
     * container</a> section of the <a
     * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and the
     * <code>--privileged</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
     * </p>
     * 
     * @return When this parameter is true, the container is given elevated privileges on the host container instance
     *         (similar to the <code>root</code> user). This parameter maps to <code>Privileged</code> in the <a
     *         href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create a
     *         container</a> section of the <a
     *         href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and
     *         the <code>--privileged</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
     *         run</a>.
     */
    public Boolean privileged() {
        return privileged;
    }

    /**
     * <p>
     * A list of <code>ulimits</code> to set in the container. This parameter maps to <code>Ulimits</code> in the <a
     * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create a
     * container</a> section of the <a
     * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and the
     * <code>--ulimit</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
     * </p>
     * 
     * @return A list of <code>ulimits</code> to set in the container. This parameter maps to <code>Ulimits</code> in
     *         the <a
     *         href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create a
     *         container</a> section of the <a
     *         href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and
     *         the <code>--ulimit</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
     *         run</a>.
     */
    public List<Ulimit> ulimits() {
        return ulimits;
    }

    /**
     * <p>
     * The user name to use inside the container. This parameter maps to <code>User</code> in the <a
     * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create a
     * container</a> section of the <a
     * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and the
     * <code>--user</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
     * </p>
     * 
     * @return The user name to use inside the container. This parameter maps to <code>User</code> in the <a
     *         href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create a
     *         container</a> section of the <a
     *         href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and
     *         the <code>--user</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
     */
    public String user() {
        return user;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + ((image() == null) ? 0 : image().hashCode());
        hashCode = 31 * hashCode + ((vcpus() == null) ? 0 : vcpus().hashCode());
        hashCode = 31 * hashCode + ((memory() == null) ? 0 : memory().hashCode());
        hashCode = 31 * hashCode + ((command() == null) ? 0 : command().hashCode());
        hashCode = 31 * hashCode + ((jobRoleArn() == null) ? 0 : jobRoleArn().hashCode());
        hashCode = 31 * hashCode + ((volumes() == null) ? 0 : volumes().hashCode());
        hashCode = 31 * hashCode + ((environment() == null) ? 0 : environment().hashCode());
        hashCode = 31 * hashCode + ((mountPoints() == null) ? 0 : mountPoints().hashCode());
        hashCode = 31 * hashCode + ((readonlyRootFilesystem() == null) ? 0 : readonlyRootFilesystem().hashCode());
        hashCode = 31 * hashCode + ((privileged() == null) ? 0 : privileged().hashCode());
        hashCode = 31 * hashCode + ((ulimits() == null) ? 0 : ulimits().hashCode());
        hashCode = 31 * hashCode + ((user() == null) ? 0 : user().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ContainerProperties)) {
            return false;
        }
        ContainerProperties other = (ContainerProperties) obj;
        if (other.image() == null ^ this.image() == null) {
            return false;
        }
        if (other.image() != null && !other.image().equals(this.image())) {
            return false;
        }
        if (other.vcpus() == null ^ this.vcpus() == null) {
            return false;
        }
        if (other.vcpus() != null && !other.vcpus().equals(this.vcpus())) {
            return false;
        }
        if (other.memory() == null ^ this.memory() == null) {
            return false;
        }
        if (other.memory() != null && !other.memory().equals(this.memory())) {
            return false;
        }
        if (other.command() == null ^ this.command() == null) {
            return false;
        }
        if (other.command() != null && !other.command().equals(this.command())) {
            return false;
        }
        if (other.jobRoleArn() == null ^ this.jobRoleArn() == null) {
            return false;
        }
        if (other.jobRoleArn() != null && !other.jobRoleArn().equals(this.jobRoleArn())) {
            return false;
        }
        if (other.volumes() == null ^ this.volumes() == null) {
            return false;
        }
        if (other.volumes() != null && !other.volumes().equals(this.volumes())) {
            return false;
        }
        if (other.environment() == null ^ this.environment() == null) {
            return false;
        }
        if (other.environment() != null && !other.environment().equals(this.environment())) {
            return false;
        }
        if (other.mountPoints() == null ^ this.mountPoints() == null) {
            return false;
        }
        if (other.mountPoints() != null && !other.mountPoints().equals(this.mountPoints())) {
            return false;
        }
        if (other.readonlyRootFilesystem() == null ^ this.readonlyRootFilesystem() == null) {
            return false;
        }
        if (other.readonlyRootFilesystem() != null && !other.readonlyRootFilesystem().equals(this.readonlyRootFilesystem())) {
            return false;
        }
        if (other.privileged() == null ^ this.privileged() == null) {
            return false;
        }
        if (other.privileged() != null && !other.privileged().equals(this.privileged())) {
            return false;
        }
        if (other.ulimits() == null ^ this.ulimits() == null) {
            return false;
        }
        if (other.ulimits() != null && !other.ulimits().equals(this.ulimits())) {
            return false;
        }
        if (other.user() == null ^ this.user() == null) {
            return false;
        }
        if (other.user() != null && !other.user().equals(this.user())) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        if (image() != null) {
            sb.append("Image: ").append(image()).append(",");
        }
        if (vcpus() != null) {
            sb.append("Vcpus: ").append(vcpus()).append(",");
        }
        if (memory() != null) {
            sb.append("Memory: ").append(memory()).append(",");
        }
        if (command() != null) {
            sb.append("Command: ").append(command()).append(",");
        }
        if (jobRoleArn() != null) {
            sb.append("JobRoleArn: ").append(jobRoleArn()).append(",");
        }
        if (volumes() != null) {
            sb.append("Volumes: ").append(volumes()).append(",");
        }
        if (environment() != null) {
            sb.append("Environment: ").append(environment()).append(",");
        }
        if (mountPoints() != null) {
            sb.append("MountPoints: ").append(mountPoints()).append(",");
        }
        if (readonlyRootFilesystem() != null) {
            sb.append("ReadonlyRootFilesystem: ").append(readonlyRootFilesystem()).append(",");
        }
        if (privileged() != null) {
            sb.append("Privileged: ").append(privileged()).append(",");
        }
        if (ulimits() != null) {
            sb.append("Ulimits: ").append(ulimits()).append(",");
        }
        if (user() != null) {
            sb.append("User: ").append(user()).append(",");
        }
        sb.append("}");
        return sb.toString();
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        ContainerPropertiesMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, ContainerProperties> {
        /**
         * <p>
         * The image used to start a container. This string is passed directly to the Docker daemon. Images in the
         * Docker Hub registry are available by default. Other repositories are specified with
         * <code> <i>repository-url</i>/<i>image</i>:<i>tag</i> </code>. Up to 255 letters (uppercase and lowercase),
         * numbers, hyphens, underscores, colons, periods, forward slashes, and number signs are allowed. This parameter
         * maps to <code>Image</code> in the <a
         * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create a
         * container</a> section of the <a
         * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and the
         * <code>IMAGE</code> parameter of <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Images in Amazon ECR repositories use the full registry and repository URI (for example,
         * <code>012345678910.dkr.ecr.&lt;region-name&gt;.amazonaws.com/&lt;repository-name&gt;</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * Images in official repositories on Docker Hub use a single name (for example, <code>ubuntu</code> or
         * <code>mongo</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * Images in other repositories on Docker Hub are qualified with an organization name (for example,
         * <code>amazon/amazon-ecs-agent</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * Images in other online repositories are qualified further by a domain name (for example,
         * <code>quay.io/assemblyline/ubuntu</code>).
         * </p>
         * </li>
         * </ul>
         * 
         * @param image
         *        The image used to start a container. This string is passed directly to the Docker daemon. Images in
         *        the Docker Hub registry are available by default. Other repositories are specified with
         *        <code> <i>repository-url</i>/<i>image</i>:<i>tag</i> </code>. Up to 255 letters (uppercase and
         *        lowercase), numbers, hyphens, underscores, colons, periods, forward slashes, and number signs are
         *        allowed. This parameter maps to <code>Image</code> in the <a
         *        href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create
         *        a container</a> section of the <a
         *        href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and
         *        the <code>IMAGE</code> parameter of <a href="https://docs.docker.com/engine/reference/run/">docker
         *        run</a>.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Images in Amazon ECR repositories use the full registry and repository URI (for example,
         *        <code>012345678910.dkr.ecr.&lt;region-name&gt;.amazonaws.com/&lt;repository-name&gt;</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Images in official repositories on Docker Hub use a single name (for example, <code>ubuntu</code> or
         *        <code>mongo</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Images in other repositories on Docker Hub are qualified with an organization name (for example,
         *        <code>amazon/amazon-ecs-agent</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Images in other online repositories are qualified further by a domain name (for example,
         *        <code>quay.io/assemblyline/ubuntu</code>).
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder image(String image);

        /**
         * <p>
         * The number of vCPUs reserved for the container. This parameter maps to <code>CpuShares</code> in the <a
         * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create a
         * container</a> section of the <a
         * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and the
         * <code>--cpu-shares</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
         * Each vCPU is equivalent to 1,024 CPU shares.
         * </p>
         * 
         * @param vcpus
         *        The number of vCPUs reserved for the container. This parameter maps to <code>CpuShares</code> in the
         *        <a href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">
         *        Create a container</a> section of the <a
         *        href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and
         *        the <code>--cpu-shares</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
         *        run</a>. Each vCPU is equivalent to 1,024 CPU shares.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vcpus(Integer vcpus);

        /**
         * <p>
         * The hard limit (in MiB) of memory to present to the container. If your container attempts to exceed the
         * memory specified here, the container is killed. This parameter maps to <code>Memory</code> in the <a
         * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create a
         * container</a> section of the <a
         * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and the
         * <code>--memory</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
         * </p>
         * 
         * @param memory
         *        The hard limit (in MiB) of memory to present to the container. If your container attempts to exceed
         *        the memory specified here, the container is killed. This parameter maps to <code>Memory</code> in the
         *        <a href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">
         *        Create a container</a> section of the <a
         *        href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and
         *        the <code>--memory</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
         *        run</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder memory(Integer memory);

        /**
         * <p>
         * The command that is passed to the container. This parameter maps to <code>Cmd</code> in the <a
         * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create a
         * container</a> section of the <a
         * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and the
         * <code>COMMAND</code> parameter to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>. For
         * more information, see <a
         * href="https://docs.docker.com/engine/reference/builder/#cmd">https://docs.docker.com/
         * engine/reference/builder/#cmd</a>.
         * </p>
         * 
         * @param command
         *        The command that is passed to the container. This parameter maps to <code>Cmd</code> in the <a
         *        href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create
         *        a container</a> section of the <a
         *        href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and
         *        the <code>COMMAND</code> parameter to <a href="https://docs.docker.com/engine/reference/run/">docker
         *        run</a>. For more information, see <a
         *        href="https://docs.docker.com/engine/reference/builder/#cmd">https
         *        ://docs.docker.com/engine/reference/builder/#cmd</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(Collection<String> command);

        /**
         * <p>
         * The command that is passed to the container. This parameter maps to <code>Cmd</code> in the <a
         * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create a
         * container</a> section of the <a
         * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and the
         * <code>COMMAND</code> parameter to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>. For
         * more information, see <a
         * href="https://docs.docker.com/engine/reference/builder/#cmd">https://docs.docker.com/
         * engine/reference/builder/#cmd</a>.
         * </p>
         * <p>
         * <b>NOTE:</b> This method appends the values to the existing list (if any). Use
         * {@link #setCommand(java.util.Collection)} or {@link #withCommand(java.util.Collection)} if you want to
         * override the existing values.
         * </p>
         * 
         * @param command
         *        The command that is passed to the container. This parameter maps to <code>Cmd</code> in the <a
         *        href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create
         *        a container</a> section of the <a
         *        href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and
         *        the <code>COMMAND</code> parameter to <a href="https://docs.docker.com/engine/reference/run/">docker
         *        run</a>. For more information, see <a
         *        href="https://docs.docker.com/engine/reference/builder/#cmd">https
         *        ://docs.docker.com/engine/reference/builder/#cmd</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(String... command);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the IAM role that the container can assume for AWS permissions.
         * </p>
         * 
         * @param jobRoleArn
         *        The Amazon Resource Name (ARN) of the IAM role that the container can assume for AWS permissions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobRoleArn(String jobRoleArn);

        /**
         * <p>
         * A list of data volumes used in a job.
         * </p>
         * 
         * @param volumes
         *        A list of data volumes used in a job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumes(Collection<Volume> volumes);

        /**
         * <p>
         * A list of data volumes used in a job.
         * </p>
         * <p>
         * <b>NOTE:</b> This method appends the values to the existing list (if any). Use
         * {@link #setVolumes(java.util.Collection)} or {@link #withVolumes(java.util.Collection)} if you want to
         * override the existing values.
         * </p>
         * 
         * @param volumes
         *        A list of data volumes used in a job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumes(Volume... volumes);

        /**
         * <p>
         * The environment variables to pass to a container. This parameter maps to <code>Env</code> in the <a
         * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create a
         * container</a> section of the <a
         * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and the
         * <code>--env</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
         * </p>
         * <important>
         * <p>
         * We do not recommend using plain text environment variables for sensitive information, such as credential
         * data.
         * </p>
         * </important>
         * 
         * @param environment
         *        The environment variables to pass to a container. This parameter maps to <code>Env</code> in the <a
         *        href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create
         *        a container</a> section of the <a
         *        href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and
         *        the <code>--env</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
         *        run</a>.</p> <important>
         *        <p>
         *        We do not recommend using plain text environment variables for sensitive information, such as
         *        credential data.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environment(Collection<KeyValuePair> environment);

        /**
         * <p>
         * The environment variables to pass to a container. This parameter maps to <code>Env</code> in the <a
         * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create a
         * container</a> section of the <a
         * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and the
         * <code>--env</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
         * </p>
         * <important>
         * <p>
         * We do not recommend using plain text environment variables for sensitive information, such as credential
         * data.
         * </p>
         * </important>
         * <p>
         * <b>NOTE:</b> This method appends the values to the existing list (if any). Use
         * {@link #setEnvironment(java.util.Collection)} or {@link #withEnvironment(java.util.Collection)} if you want
         * to override the existing values.
         * </p>
         * 
         * @param environment
         *        The environment variables to pass to a container. This parameter maps to <code>Env</code> in the <a
         *        href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create
         *        a container</a> section of the <a
         *        href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and
         *        the <code>--env</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
         *        run</a>.</p> <important>
         *        <p>
         *        We do not recommend using plain text environment variables for sensitive information, such as
         *        credential data.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environment(KeyValuePair... environment);

        /**
         * <p>
         * The mount points for data volumes in your container. This parameter maps to <code>Volumes</code> in the <a
         * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create a
         * container</a> section of the <a
         * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and the
         * <code>--volume</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
         * </p>
         * 
         * @param mountPoints
         *        The mount points for data volumes in your container. This parameter maps to <code>Volumes</code> in
         *        the <a
         *        href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create
         *        a container</a> section of the <a
         *        href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and
         *        the <code>--volume</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
         *        run</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mountPoints(Collection<MountPoint> mountPoints);

        /**
         * <p>
         * The mount points for data volumes in your container. This parameter maps to <code>Volumes</code> in the <a
         * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create a
         * container</a> section of the <a
         * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and the
         * <code>--volume</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
         * </p>
         * <p>
         * <b>NOTE:</b> This method appends the values to the existing list (if any). Use
         * {@link #setMountPoints(java.util.Collection)} or {@link #withMountPoints(java.util.Collection)} if you want
         * to override the existing values.
         * </p>
         * 
         * @param mountPoints
         *        The mount points for data volumes in your container. This parameter maps to <code>Volumes</code> in
         *        the <a
         *        href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create
         *        a container</a> section of the <a
         *        href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and
         *        the <code>--volume</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
         *        run</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mountPoints(MountPoint... mountPoints);

        /**
         * <p>
         * When this parameter is true, the container is given read-only access to its root file system. This parameter
         * maps to <code>ReadonlyRootfs</code> in the <a
         * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create a
         * container</a> section of the <a
         * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and the
         * <code>--read-only</code> option to <code>docker run</code>.
         * </p>
         * 
         * @param readonlyRootFilesystem
         *        When this parameter is true, the container is given read-only access to its root file system. This
         *        parameter maps to <code>ReadonlyRootfs</code> in the <a
         *        href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create
         *        a container</a> section of the <a
         *        href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and
         *        the <code>--read-only</code> option to <code>docker run</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder readonlyRootFilesystem(Boolean readonlyRootFilesystem);

        /**
         * <p>
         * When this parameter is true, the container is given elevated privileges on the host container instance
         * (similar to the <code>root</code> user). This parameter maps to <code>Privileged</code> in the <a
         * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create a
         * container</a> section of the <a
         * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and the
         * <code>--privileged</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
         * </p>
         * 
         * @param privileged
         *        When this parameter is true, the container is given elevated privileges on the host container instance
         *        (similar to the <code>root</code> user). This parameter maps to <code>Privileged</code> in the <a
         *        href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create
         *        a container</a> section of the <a
         *        href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and
         *        the <code>--privileged</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
         *        run</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder privileged(Boolean privileged);

        /**
         * <p>
         * A list of <code>ulimits</code> to set in the container. This parameter maps to <code>Ulimits</code> in the <a
         * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create a
         * container</a> section of the <a
         * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and the
         * <code>--ulimit</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
         * </p>
         * 
         * @param ulimits
         *        A list of <code>ulimits</code> to set in the container. This parameter maps to <code>Ulimits</code> in
         *        the <a
         *        href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create
         *        a container</a> section of the <a
         *        href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and
         *        the <code>--ulimit</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
         *        run</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ulimits(Collection<Ulimit> ulimits);

        /**
         * <p>
         * A list of <code>ulimits</code> to set in the container. This parameter maps to <code>Ulimits</code> in the <a
         * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create a
         * container</a> section of the <a
         * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and the
         * <code>--ulimit</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
         * </p>
         * <p>
         * <b>NOTE:</b> This method appends the values to the existing list (if any). Use
         * {@link #setUlimits(java.util.Collection)} or {@link #withUlimits(java.util.Collection)} if you want to
         * override the existing values.
         * </p>
         * 
         * @param ulimits
         *        A list of <code>ulimits</code> to set in the container. This parameter maps to <code>Ulimits</code> in
         *        the <a
         *        href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create
         *        a container</a> section of the <a
         *        href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and
         *        the <code>--ulimit</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
         *        run</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ulimits(Ulimit... ulimits);

        /**
         * <p>
         * The user name to use inside the container. This parameter maps to <code>User</code> in the <a
         * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create a
         * container</a> section of the <a
         * href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and the
         * <code>--user</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
         * </p>
         * 
         * @param user
         *        The user name to use inside the container. This parameter maps to <code>User</code> in the <a
         *        href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/#create-a-container">Create
         *        a container</a> section of the <a
         *        href="https://docs.docker.com/engine/reference/api/docker_remote_api_v1.23/">Docker Remote API</a> and
         *        the <code>--user</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
         *        run</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder user(String user);
    }

    private static final class BuilderImpl implements Builder {
        private String image;

        private Integer vcpus;

        private Integer memory;

        private List<String> command;

        private String jobRoleArn;

        private List<Volume> volumes;

        private List<KeyValuePair> environment;

        private List<MountPoint> mountPoints;

        private Boolean readonlyRootFilesystem;

        private Boolean privileged;

        private List<Ulimit> ulimits;

        private String user;

        private BuilderImpl() {
        }

        private BuilderImpl(ContainerProperties model) {
            setImage(model.image);
            setVcpus(model.vcpus);
            setMemory(model.memory);
            setCommand(model.command);
            setJobRoleArn(model.jobRoleArn);
            setVolumes(model.volumes);
            setEnvironment(model.environment);
            setMountPoints(model.mountPoints);
            setReadonlyRootFilesystem(model.readonlyRootFilesystem);
            setPrivileged(model.privileged);
            setUlimits(model.ulimits);
            setUser(model.user);
        }

        public final String getImage() {
            return image;
        }

        @Override
        public final Builder image(String image) {
            this.image = image;
            return this;
        }

        public final void setImage(String image) {
            this.image = image;
        }

        public final Integer getVcpus() {
            return vcpus;
        }

        @Override
        public final Builder vcpus(Integer vcpus) {
            this.vcpus = vcpus;
            return this;
        }

        public final void setVcpus(Integer vcpus) {
            this.vcpus = vcpus;
        }

        public final Integer getMemory() {
            return memory;
        }

        @Override
        public final Builder memory(Integer memory) {
            this.memory = memory;
            return this;
        }

        public final void setMemory(Integer memory) {
            this.memory = memory;
        }

        public final Collection<String> getCommand() {
            return command;
        }

        @Override
        public final Builder command(Collection<String> command) {
            this.command = StringListCopier.copy(command);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder command(String... command) {
            if (this.command == null) {
                this.command = new ArrayList<>(command.length);
            }
            for (String e : command) {
                this.command.add(e);
            }
            return this;
        }

        public final void setCommand(Collection<String> command) {
            this.command = StringListCopier.copy(command);
        }

        @SafeVarargs
        public final void setCommand(String... command) {
            if (this.command == null) {
                this.command = new ArrayList<>(command.length);
            }
            for (String e : command) {
                this.command.add(e);
            }
        }

        public final String getJobRoleArn() {
            return jobRoleArn;
        }

        @Override
        public final Builder jobRoleArn(String jobRoleArn) {
            this.jobRoleArn = jobRoleArn;
            return this;
        }

        public final void setJobRoleArn(String jobRoleArn) {
            this.jobRoleArn = jobRoleArn;
        }

        public final Collection<Volume> getVolumes() {
            return volumes;
        }

        @Override
        public final Builder volumes(Collection<Volume> volumes) {
            this.volumes = VolumesCopier.copy(volumes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder volumes(Volume... volumes) {
            if (this.volumes == null) {
                this.volumes = new ArrayList<>(volumes.length);
            }
            for (Volume e : volumes) {
                this.volumes.add(e);
            }
            return this;
        }

        public final void setVolumes(Collection<Volume> volumes) {
            this.volumes = VolumesCopier.copy(volumes);
        }

        @SafeVarargs
        public final void setVolumes(Volume... volumes) {
            if (this.volumes == null) {
                this.volumes = new ArrayList<>(volumes.length);
            }
            for (Volume e : volumes) {
                this.volumes.add(e);
            }
        }

        public final Collection<KeyValuePair> getEnvironment() {
            return environment;
        }

        @Override
        public final Builder environment(Collection<KeyValuePair> environment) {
            this.environment = EnvironmentVariablesCopier.copy(environment);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder environment(KeyValuePair... environment) {
            if (this.environment == null) {
                this.environment = new ArrayList<>(environment.length);
            }
            for (KeyValuePair e : environment) {
                this.environment.add(e);
            }
            return this;
        }

        public final void setEnvironment(Collection<KeyValuePair> environment) {
            this.environment = EnvironmentVariablesCopier.copy(environment);
        }

        @SafeVarargs
        public final void setEnvironment(KeyValuePair... environment) {
            if (this.environment == null) {
                this.environment = new ArrayList<>(environment.length);
            }
            for (KeyValuePair e : environment) {
                this.environment.add(e);
            }
        }

        public final Collection<MountPoint> getMountPoints() {
            return mountPoints;
        }

        @Override
        public final Builder mountPoints(Collection<MountPoint> mountPoints) {
            this.mountPoints = MountPointsCopier.copy(mountPoints);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder mountPoints(MountPoint... mountPoints) {
            if (this.mountPoints == null) {
                this.mountPoints = new ArrayList<>(mountPoints.length);
            }
            for (MountPoint e : mountPoints) {
                this.mountPoints.add(e);
            }
            return this;
        }

        public final void setMountPoints(Collection<MountPoint> mountPoints) {
            this.mountPoints = MountPointsCopier.copy(mountPoints);
        }

        @SafeVarargs
        public final void setMountPoints(MountPoint... mountPoints) {
            if (this.mountPoints == null) {
                this.mountPoints = new ArrayList<>(mountPoints.length);
            }
            for (MountPoint e : mountPoints) {
                this.mountPoints.add(e);
            }
        }

        public final Boolean getReadonlyRootFilesystem() {
            return readonlyRootFilesystem;
        }

        @Override
        public final Builder readonlyRootFilesystem(Boolean readonlyRootFilesystem) {
            this.readonlyRootFilesystem = readonlyRootFilesystem;
            return this;
        }

        public final void setReadonlyRootFilesystem(Boolean readonlyRootFilesystem) {
            this.readonlyRootFilesystem = readonlyRootFilesystem;
        }

        public final Boolean getPrivileged() {
            return privileged;
        }

        @Override
        public final Builder privileged(Boolean privileged) {
            this.privileged = privileged;
            return this;
        }

        public final void setPrivileged(Boolean privileged) {
            this.privileged = privileged;
        }

        public final Collection<Ulimit> getUlimits() {
            return ulimits;
        }

        @Override
        public final Builder ulimits(Collection<Ulimit> ulimits) {
            this.ulimits = UlimitsCopier.copy(ulimits);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ulimits(Ulimit... ulimits) {
            if (this.ulimits == null) {
                this.ulimits = new ArrayList<>(ulimits.length);
            }
            for (Ulimit e : ulimits) {
                this.ulimits.add(e);
            }
            return this;
        }

        public final void setUlimits(Collection<Ulimit> ulimits) {
            this.ulimits = UlimitsCopier.copy(ulimits);
        }

        @SafeVarargs
        public final void setUlimits(Ulimit... ulimits) {
            if (this.ulimits == null) {
                this.ulimits = new ArrayList<>(ulimits.length);
            }
            for (Ulimit e : ulimits) {
                this.ulimits.add(e);
            }
        }

        public final String getUser() {
            return user;
        }

        @Override
        public final Builder user(String user) {
            this.user = user;
            return this;
        }

        public final void setUser(String user) {
            this.user = user;
        }

        @Override
        public ContainerProperties build() {
            return new ContainerProperties(this);
        }
    }
}
