/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.batch.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.batch.transform.ComputeResourceMarshaller;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object representing an AWS Batch compute resource.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class ComputeResource implements StructuredPojo, ToCopyableBuilder<ComputeResource.Builder, ComputeResource> {
    private final String type;

    private final Integer minvCpus;

    private final Integer maxvCpus;

    private final Integer desiredvCpus;

    private final List<String> instanceTypes;

    private final String imageId;

    private final List<String> subnets;

    private final List<String> securityGroupIds;

    private final String ec2KeyPair;

    private final String instanceRole;

    private final Map<String, String> tags;

    private final Integer bidPercentage;

    private final String spotIamFleetRole;

    private ComputeResource(BuilderImpl builder) {
        this.type = builder.type;
        this.minvCpus = builder.minvCpus;
        this.maxvCpus = builder.maxvCpus;
        this.desiredvCpus = builder.desiredvCpus;
        this.instanceTypes = builder.instanceTypes;
        this.imageId = builder.imageId;
        this.subnets = builder.subnets;
        this.securityGroupIds = builder.securityGroupIds;
        this.ec2KeyPair = builder.ec2KeyPair;
        this.instanceRole = builder.instanceRole;
        this.tags = builder.tags;
        this.bidPercentage = builder.bidPercentage;
        this.spotIamFleetRole = builder.spotIamFleetRole;
    }

    /**
     * <p>
     * The type of compute environment.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link CRType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeString}.
     * </p>
     * 
     * @return The type of compute environment.
     * @see CRType
     */
    public CRType type() {
        return CRType.fromValue(type);
    }

    /**
     * <p>
     * The type of compute environment.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link CRType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeString}.
     * </p>
     * 
     * @return The type of compute environment.
     * @see CRType
     */
    public String typeString() {
        return type;
    }

    /**
     * <p>
     * The minimum number of EC2 vCPUs that an environment should maintain.
     * </p>
     * 
     * @return The minimum number of EC2 vCPUs that an environment should maintain.
     */
    public Integer minvCpus() {
        return minvCpus;
    }

    /**
     * <p>
     * The maximum number of EC2 vCPUs that an environment can reach.
     * </p>
     * 
     * @return The maximum number of EC2 vCPUs that an environment can reach.
     */
    public Integer maxvCpus() {
        return maxvCpus;
    }

    /**
     * <p>
     * The desired number of EC2 vCPUS in the compute environment.
     * </p>
     * 
     * @return The desired number of EC2 vCPUS in the compute environment.
     */
    public Integer desiredvCpus() {
        return desiredvCpus;
    }

    /**
     * <p>
     * The instances types that may be launched. You can specify instance families to launch any instance type within
     * those families (for example, <code>c4</code> or <code>p3</code>), or you can specify specific sizes within a
     * family (such as <code>c4.8xlarge</code>). You can also choose <code>optimal</code> to pick instance types (from
     * the latest C, M, and R instance families) on the fly that match the demand of your job queues.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The instances types that may be launched. You can specify instance families to launch any instance type
     *         within those families (for example, <code>c4</code> or <code>p3</code>), or you can specify specific
     *         sizes within a family (such as <code>c4.8xlarge</code>). You can also choose <code>optimal</code> to pick
     *         instance types (from the latest C, M, and R instance families) on the fly that match the demand of your
     *         job queues.
     */
    public List<String> instanceTypes() {
        return instanceTypes;
    }

    /**
     * <p>
     * The Amazon Machine Image (AMI) ID used for instances launched in the compute environment.
     * </p>
     * 
     * @return The Amazon Machine Image (AMI) ID used for instances launched in the compute environment.
     */
    public String imageId() {
        return imageId;
    }

    /**
     * <p>
     * The VPC subnets into which the compute resources are launched.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The VPC subnets into which the compute resources are launched.
     */
    public List<String> subnets() {
        return subnets;
    }

    /**
     * <p>
     * The EC2 security group that is associated with instances launched in the compute environment.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The EC2 security group that is associated with instances launched in the compute environment.
     */
    public List<String> securityGroupIds() {
        return securityGroupIds;
    }

    /**
     * <p>
     * The EC2 key pair that is used for instances launched in the compute environment.
     * </p>
     * 
     * @return The EC2 key pair that is used for instances launched in the compute environment.
     */
    public String ec2KeyPair() {
        return ec2KeyPair;
    }

    /**
     * <p>
     * The Amazon ECS instance profile applied to Amazon EC2 instances in a compute environment. You can specify the
     * short name or full Amazon Resource Name (ARN) of an instance profile. For example, <code>ecsInstanceRole</code>
     * or <code>arn:aws:iam::&lt;aws_account_id&gt;:instance-profile/ecsInstanceRole</code>. For more information, see
     * <a href="http://docs.aws.amazon.com/batch/latest/userguide/instance_IAM_role.html">Amazon ECS Instance Role</a>
     * in the <i>AWS Batch User Guide</i>.
     * </p>
     * 
     * @return The Amazon ECS instance profile applied to Amazon EC2 instances in a compute environment. You can specify
     *         the short name or full Amazon Resource Name (ARN) of an instance profile. For example,
     *         <code>ecsInstanceRole</code> or
     *         <code>arn:aws:iam::&lt;aws_account_id&gt;:instance-profile/ecsInstanceRole</code>. For more information,
     *         see <a href="http://docs.aws.amazon.com/batch/latest/userguide/instance_IAM_role.html">Amazon ECS
     *         Instance Role</a> in the <i>AWS Batch User Guide</i>.
     */
    public String instanceRole() {
        return instanceRole;
    }

    /**
     * <p>
     * Key-value pair tags to be applied to resources that are launched in the compute environment.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Key-value pair tags to be applied to resources that are launched in the compute environment.
     */
    public Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * The minimum percentage that a Spot Instance price must be when compared with the On-Demand price for that
     * instance type before instances are launched. For example, if your bid percentage is 20%, then the Spot price must
     * be below 20% of the current On-Demand price for that EC2 instance.
     * </p>
     * 
     * @return The minimum percentage that a Spot Instance price must be when compared with the On-Demand price for that
     *         instance type before instances are launched. For example, if your bid percentage is 20%, then the Spot
     *         price must be below 20% of the current On-Demand price for that EC2 instance.
     */
    public Integer bidPercentage() {
        return bidPercentage;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the Amazon EC2 Spot Fleet IAM role applied to a <code>SPOT</code> compute
     * environment.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the Amazon EC2 Spot Fleet IAM role applied to a <code>SPOT</code>
     *         compute environment.
     */
    public String spotIamFleetRole() {
        return spotIamFleetRole;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typeString());
        hashCode = 31 * hashCode + Objects.hashCode(minvCpus());
        hashCode = 31 * hashCode + Objects.hashCode(maxvCpus());
        hashCode = 31 * hashCode + Objects.hashCode(desiredvCpus());
        hashCode = 31 * hashCode + Objects.hashCode(instanceTypes());
        hashCode = 31 * hashCode + Objects.hashCode(imageId());
        hashCode = 31 * hashCode + Objects.hashCode(subnets());
        hashCode = 31 * hashCode + Objects.hashCode(securityGroupIds());
        hashCode = 31 * hashCode + Objects.hashCode(ec2KeyPair());
        hashCode = 31 * hashCode + Objects.hashCode(instanceRole());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        hashCode = 31 * hashCode + Objects.hashCode(bidPercentage());
        hashCode = 31 * hashCode + Objects.hashCode(spotIamFleetRole());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ComputeResource)) {
            return false;
        }
        ComputeResource other = (ComputeResource) obj;
        return Objects.equals(typeString(), other.typeString()) && Objects.equals(minvCpus(), other.minvCpus())
                && Objects.equals(maxvCpus(), other.maxvCpus()) && Objects.equals(desiredvCpus(), other.desiredvCpus())
                && Objects.equals(instanceTypes(), other.instanceTypes()) && Objects.equals(imageId(), other.imageId())
                && Objects.equals(subnets(), other.subnets()) && Objects.equals(securityGroupIds(), other.securityGroupIds())
                && Objects.equals(ec2KeyPair(), other.ec2KeyPair()) && Objects.equals(instanceRole(), other.instanceRole())
                && Objects.equals(tags(), other.tags()) && Objects.equals(bidPercentage(), other.bidPercentage())
                && Objects.equals(spotIamFleetRole(), other.spotIamFleetRole());
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (typeString() != null) {
            sb.append("Type: ").append(typeString()).append(",");
        }
        if (minvCpus() != null) {
            sb.append("MinvCpus: ").append(minvCpus()).append(",");
        }
        if (maxvCpus() != null) {
            sb.append("MaxvCpus: ").append(maxvCpus()).append(",");
        }
        if (desiredvCpus() != null) {
            sb.append("DesiredvCpus: ").append(desiredvCpus()).append(",");
        }
        if (instanceTypes() != null) {
            sb.append("InstanceTypes: ").append(instanceTypes()).append(",");
        }
        if (imageId() != null) {
            sb.append("ImageId: ").append(imageId()).append(",");
        }
        if (subnets() != null) {
            sb.append("Subnets: ").append(subnets()).append(",");
        }
        if (securityGroupIds() != null) {
            sb.append("SecurityGroupIds: ").append(securityGroupIds()).append(",");
        }
        if (ec2KeyPair() != null) {
            sb.append("Ec2KeyPair: ").append(ec2KeyPair()).append(",");
        }
        if (instanceRole() != null) {
            sb.append("InstanceRole: ").append(instanceRole()).append(",");
        }
        if (tags() != null) {
            sb.append("Tags: ").append(tags()).append(",");
        }
        if (bidPercentage() != null) {
            sb.append("BidPercentage: ").append(bidPercentage()).append(",");
        }
        if (spotIamFleetRole() != null) {
            sb.append("SpotIamFleetRole: ").append(spotIamFleetRole()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "type":
            return Optional.of(clazz.cast(typeString()));
        case "minvCpus":
            return Optional.of(clazz.cast(minvCpus()));
        case "maxvCpus":
            return Optional.of(clazz.cast(maxvCpus()));
        case "desiredvCpus":
            return Optional.of(clazz.cast(desiredvCpus()));
        case "instanceTypes":
            return Optional.of(clazz.cast(instanceTypes()));
        case "imageId":
            return Optional.of(clazz.cast(imageId()));
        case "subnets":
            return Optional.of(clazz.cast(subnets()));
        case "securityGroupIds":
            return Optional.of(clazz.cast(securityGroupIds()));
        case "ec2KeyPair":
            return Optional.of(clazz.cast(ec2KeyPair()));
        case "instanceRole":
            return Optional.of(clazz.cast(instanceRole()));
        case "tags":
            return Optional.of(clazz.cast(tags()));
        case "bidPercentage":
            return Optional.of(clazz.cast(bidPercentage()));
        case "spotIamFleetRole":
            return Optional.of(clazz.cast(spotIamFleetRole()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        ComputeResourceMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, ComputeResource> {
        /**
         * <p>
         * The type of compute environment.
         * </p>
         * 
         * @param type
         *        The type of compute environment.
         * @see CRType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CRType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of compute environment.
         * </p>
         * 
         * @param type
         *        The type of compute environment.
         * @see CRType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CRType
         */
        Builder type(CRType type);

        /**
         * <p>
         * The minimum number of EC2 vCPUs that an environment should maintain.
         * </p>
         * 
         * @param minvCpus
         *        The minimum number of EC2 vCPUs that an environment should maintain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minvCpus(Integer minvCpus);

        /**
         * <p>
         * The maximum number of EC2 vCPUs that an environment can reach.
         * </p>
         * 
         * @param maxvCpus
         *        The maximum number of EC2 vCPUs that an environment can reach.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxvCpus(Integer maxvCpus);

        /**
         * <p>
         * The desired number of EC2 vCPUS in the compute environment.
         * </p>
         * 
         * @param desiredvCpus
         *        The desired number of EC2 vCPUS in the compute environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder desiredvCpus(Integer desiredvCpus);

        /**
         * <p>
         * The instances types that may be launched. You can specify instance families to launch any instance type
         * within those families (for example, <code>c4</code> or <code>p3</code>), or you can specify specific sizes
         * within a family (such as <code>c4.8xlarge</code>). You can also choose <code>optimal</code> to pick instance
         * types (from the latest C, M, and R instance families) on the fly that match the demand of your job queues.
         * </p>
         * 
         * @param instanceTypes
         *        The instances types that may be launched. You can specify instance families to launch any instance
         *        type within those families (for example, <code>c4</code> or <code>p3</code>), or you can specify
         *        specific sizes within a family (such as <code>c4.8xlarge</code>). You can also choose
         *        <code>optimal</code> to pick instance types (from the latest C, M, and R instance families) on the fly
         *        that match the demand of your job queues.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceTypes(Collection<String> instanceTypes);

        /**
         * <p>
         * The instances types that may be launched. You can specify instance families to launch any instance type
         * within those families (for example, <code>c4</code> or <code>p3</code>), or you can specify specific sizes
         * within a family (such as <code>c4.8xlarge</code>). You can also choose <code>optimal</code> to pick instance
         * types (from the latest C, M, and R instance families) on the fly that match the demand of your job queues.
         * </p>
         * 
         * @param instanceTypes
         *        The instances types that may be launched. You can specify instance families to launch any instance
         *        type within those families (for example, <code>c4</code> or <code>p3</code>), or you can specify
         *        specific sizes within a family (such as <code>c4.8xlarge</code>). You can also choose
         *        <code>optimal</code> to pick instance types (from the latest C, M, and R instance families) on the fly
         *        that match the demand of your job queues.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceTypes(String... instanceTypes);

        /**
         * <p>
         * The Amazon Machine Image (AMI) ID used for instances launched in the compute environment.
         * </p>
         * 
         * @param imageId
         *        The Amazon Machine Image (AMI) ID used for instances launched in the compute environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageId(String imageId);

        /**
         * <p>
         * The VPC subnets into which the compute resources are launched.
         * </p>
         * 
         * @param subnets
         *        The VPC subnets into which the compute resources are launched.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnets(Collection<String> subnets);

        /**
         * <p>
         * The VPC subnets into which the compute resources are launched.
         * </p>
         * 
         * @param subnets
         *        The VPC subnets into which the compute resources are launched.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnets(String... subnets);

        /**
         * <p>
         * The EC2 security group that is associated with instances launched in the compute environment.
         * </p>
         * 
         * @param securityGroupIds
         *        The EC2 security group that is associated with instances launched in the compute environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupIds(Collection<String> securityGroupIds);

        /**
         * <p>
         * The EC2 security group that is associated with instances launched in the compute environment.
         * </p>
         * 
         * @param securityGroupIds
         *        The EC2 security group that is associated with instances launched in the compute environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupIds(String... securityGroupIds);

        /**
         * <p>
         * The EC2 key pair that is used for instances launched in the compute environment.
         * </p>
         * 
         * @param ec2KeyPair
         *        The EC2 key pair that is used for instances launched in the compute environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ec2KeyPair(String ec2KeyPair);

        /**
         * <p>
         * The Amazon ECS instance profile applied to Amazon EC2 instances in a compute environment. You can specify the
         * short name or full Amazon Resource Name (ARN) of an instance profile. For example,
         * <code>ecsInstanceRole</code> or
         * <code>arn:aws:iam::&lt;aws_account_id&gt;:instance-profile/ecsInstanceRole</code>. For more information, see
         * <a href="http://docs.aws.amazon.com/batch/latest/userguide/instance_IAM_role.html">Amazon ECS Instance
         * Role</a> in the <i>AWS Batch User Guide</i>.
         * </p>
         * 
         * @param instanceRole
         *        The Amazon ECS instance profile applied to Amazon EC2 instances in a compute environment. You can
         *        specify the short name or full Amazon Resource Name (ARN) of an instance profile. For example,
         *        <code>ecsInstanceRole</code> or
         *        <code>arn:aws:iam::&lt;aws_account_id&gt;:instance-profile/ecsInstanceRole</code>. For more
         *        information, see <a
         *        href="http://docs.aws.amazon.com/batch/latest/userguide/instance_IAM_role.html">Amazon ECS Instance
         *        Role</a> in the <i>AWS Batch User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceRole(String instanceRole);

        /**
         * <p>
         * Key-value pair tags to be applied to resources that are launched in the compute environment.
         * </p>
         * 
         * @param tags
         *        Key-value pair tags to be applied to resources that are launched in the compute environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * The minimum percentage that a Spot Instance price must be when compared with the On-Demand price for that
         * instance type before instances are launched. For example, if your bid percentage is 20%, then the Spot price
         * must be below 20% of the current On-Demand price for that EC2 instance.
         * </p>
         * 
         * @param bidPercentage
         *        The minimum percentage that a Spot Instance price must be when compared with the On-Demand price for
         *        that instance type before instances are launched. For example, if your bid percentage is 20%, then the
         *        Spot price must be below 20% of the current On-Demand price for that EC2 instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bidPercentage(Integer bidPercentage);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the Amazon EC2 Spot Fleet IAM role applied to a <code>SPOT</code> compute
         * environment.
         * </p>
         * 
         * @param spotIamFleetRole
         *        The Amazon Resource Name (ARN) of the Amazon EC2 Spot Fleet IAM role applied to a <code>SPOT</code>
         *        compute environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder spotIamFleetRole(String spotIamFleetRole);
    }

    static final class BuilderImpl implements Builder {
        private String type;

        private Integer minvCpus;

        private Integer maxvCpus;

        private Integer desiredvCpus;

        private List<String> instanceTypes;

        private String imageId;

        private List<String> subnets;

        private List<String> securityGroupIds;

        private String ec2KeyPair;

        private String instanceRole;

        private Map<String, String> tags;

        private Integer bidPercentage;

        private String spotIamFleetRole;

        private BuilderImpl() {
        }

        private BuilderImpl(ComputeResource model) {
            type(model.type);
            minvCpus(model.minvCpus);
            maxvCpus(model.maxvCpus);
            desiredvCpus(model.desiredvCpus);
            instanceTypes(model.instanceTypes);
            imageId(model.imageId);
            subnets(model.subnets);
            securityGroupIds(model.securityGroupIds);
            ec2KeyPair(model.ec2KeyPair);
            instanceRole(model.instanceRole);
            tags(model.tags);
            bidPercentage(model.bidPercentage);
            spotIamFleetRole(model.spotIamFleetRole);
        }

        public final String getType() {
            return type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(CRType type) {
            this.type(type.toString());
            return this;
        }

        public final void setType(String type) {
            this.type = type;
        }

        public final Integer getMinvCpus() {
            return minvCpus;
        }

        @Override
        public final Builder minvCpus(Integer minvCpus) {
            this.minvCpus = minvCpus;
            return this;
        }

        public final void setMinvCpus(Integer minvCpus) {
            this.minvCpus = minvCpus;
        }

        public final Integer getMaxvCpus() {
            return maxvCpus;
        }

        @Override
        public final Builder maxvCpus(Integer maxvCpus) {
            this.maxvCpus = maxvCpus;
            return this;
        }

        public final void setMaxvCpus(Integer maxvCpus) {
            this.maxvCpus = maxvCpus;
        }

        public final Integer getDesiredvCpus() {
            return desiredvCpus;
        }

        @Override
        public final Builder desiredvCpus(Integer desiredvCpus) {
            this.desiredvCpus = desiredvCpus;
            return this;
        }

        public final void setDesiredvCpus(Integer desiredvCpus) {
            this.desiredvCpus = desiredvCpus;
        }

        public final Collection<String> getInstanceTypes() {
            return instanceTypes;
        }

        @Override
        public final Builder instanceTypes(Collection<String> instanceTypes) {
            this.instanceTypes = StringListCopier.copy(instanceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceTypes(String... instanceTypes) {
            instanceTypes(Arrays.asList(instanceTypes));
            return this;
        }

        public final void setInstanceTypes(Collection<String> instanceTypes) {
            this.instanceTypes = StringListCopier.copy(instanceTypes);
        }

        public final String getImageId() {
            return imageId;
        }

        @Override
        public final Builder imageId(String imageId) {
            this.imageId = imageId;
            return this;
        }

        public final void setImageId(String imageId) {
            this.imageId = imageId;
        }

        public final Collection<String> getSubnets() {
            return subnets;
        }

        @Override
        public final Builder subnets(Collection<String> subnets) {
            this.subnets = StringListCopier.copy(subnets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subnets(String... subnets) {
            subnets(Arrays.asList(subnets));
            return this;
        }

        public final void setSubnets(Collection<String> subnets) {
            this.subnets = StringListCopier.copy(subnets);
        }

        public final Collection<String> getSecurityGroupIds() {
            return securityGroupIds;
        }

        @Override
        public final Builder securityGroupIds(Collection<String> securityGroupIds) {
            this.securityGroupIds = StringListCopier.copy(securityGroupIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder securityGroupIds(String... securityGroupIds) {
            securityGroupIds(Arrays.asList(securityGroupIds));
            return this;
        }

        public final void setSecurityGroupIds(Collection<String> securityGroupIds) {
            this.securityGroupIds = StringListCopier.copy(securityGroupIds);
        }

        public final String getEc2KeyPair() {
            return ec2KeyPair;
        }

        @Override
        public final Builder ec2KeyPair(String ec2KeyPair) {
            this.ec2KeyPair = ec2KeyPair;
            return this;
        }

        public final void setEc2KeyPair(String ec2KeyPair) {
            this.ec2KeyPair = ec2KeyPair;
        }

        public final String getInstanceRole() {
            return instanceRole;
        }

        @Override
        public final Builder instanceRole(String instanceRole) {
            this.instanceRole = instanceRole;
            return this;
        }

        public final void setInstanceRole(String instanceRole) {
            this.instanceRole = instanceRole;
        }

        public final Map<String, String> getTags() {
            return tags;
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagsMapCopier.copy(tags);
            return this;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagsMapCopier.copy(tags);
        }

        public final Integer getBidPercentage() {
            return bidPercentage;
        }

        @Override
        public final Builder bidPercentage(Integer bidPercentage) {
            this.bidPercentage = bidPercentage;
            return this;
        }

        public final void setBidPercentage(Integer bidPercentage) {
            this.bidPercentage = bidPercentage;
        }

        public final String getSpotIamFleetRole() {
            return spotIamFleetRole;
        }

        @Override
        public final Builder spotIamFleetRole(String spotIamFleetRole) {
            this.spotIamFleetRole = spotIamFleetRole;
            return this;
        }

        public final void setSpotIamFleetRole(String spotIamFleetRole) {
            this.spotIamFleetRole = spotIamFleetRole;
        }

        @Override
        public ComputeResource build() {
            return new ComputeResource(this);
        }
    }
}
