/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.batch.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.batch.transform.ContainerOverridesMarshaller;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The overrides that should be sent to a container.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class ContainerOverrides implements StructuredPojo, ToCopyableBuilder<ContainerOverrides.Builder, ContainerOverrides> {
    private final Integer vcpus;

    private final Integer memory;

    private final List<String> command;

    private final List<KeyValuePair> environment;

    private ContainerOverrides(BuilderImpl builder) {
        this.vcpus = builder.vcpus;
        this.memory = builder.memory;
        this.command = builder.command;
        this.environment = builder.environment;
    }

    /**
     * <p>
     * The number of vCPUs to reserve for the container. This value overrides the value set in the job definition.
     * </p>
     * 
     * @return The number of vCPUs to reserve for the container. This value overrides the value set in the job
     *         definition.
     */
    public Integer vcpus() {
        return vcpus;
    }

    /**
     * <p>
     * The number of MiB of memory reserved for the job. This value overrides the value set in the job definition.
     * </p>
     * 
     * @return The number of MiB of memory reserved for the job. This value overrides the value set in the job
     *         definition.
     */
    public Integer memory() {
        return memory;
    }

    /**
     * <p>
     * The command to send to the container that overrides the default command from the Docker image or the job
     * definition.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The command to send to the container that overrides the default command from the Docker image or the job
     *         definition.
     */
    public List<String> command() {
        return command;
    }

    /**
     * <p>
     * The environment variables to send to the container. You can add new environment variables, which are added to the
     * container at launch, or you can override the existing environment variables from the Docker image or the job
     * definition.
     * </p>
     * <note>
     * <p>
     * Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved for
     * variables that are set by the AWS Batch service.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The environment variables to send to the container. You can add new environment variables, which are
     *         added to the container at launch, or you can override the existing environment variables from the Docker
     *         image or the job definition.</p> <note>
     *         <p>
     *         Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved for
     *         variables that are set by the AWS Batch service.
     *         </p>
     */
    public List<KeyValuePair> environment() {
        return environment;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(vcpus());
        hashCode = 31 * hashCode + Objects.hashCode(memory());
        hashCode = 31 * hashCode + Objects.hashCode(command());
        hashCode = 31 * hashCode + Objects.hashCode(environment());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ContainerOverrides)) {
            return false;
        }
        ContainerOverrides other = (ContainerOverrides) obj;
        return Objects.equals(vcpus(), other.vcpus()) && Objects.equals(memory(), other.memory())
                && Objects.equals(command(), other.command()) && Objects.equals(environment(), other.environment());
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (vcpus() != null) {
            sb.append("Vcpus: ").append(vcpus()).append(",");
        }
        if (memory() != null) {
            sb.append("Memory: ").append(memory()).append(",");
        }
        if (command() != null) {
            sb.append("Command: ").append(command()).append(",");
        }
        if (environment() != null) {
            sb.append("Environment: ").append(environment()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "vcpus":
            return Optional.of(clazz.cast(vcpus()));
        case "memory":
            return Optional.of(clazz.cast(memory()));
        case "command":
            return Optional.of(clazz.cast(command()));
        case "environment":
            return Optional.of(clazz.cast(environment()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        ContainerOverridesMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, ContainerOverrides> {
        /**
         * <p>
         * The number of vCPUs to reserve for the container. This value overrides the value set in the job definition.
         * </p>
         * 
         * @param vcpus
         *        The number of vCPUs to reserve for the container. This value overrides the value set in the job
         *        definition.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vcpus(Integer vcpus);

        /**
         * <p>
         * The number of MiB of memory reserved for the job. This value overrides the value set in the job definition.
         * </p>
         * 
         * @param memory
         *        The number of MiB of memory reserved for the job. This value overrides the value set in the job
         *        definition.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder memory(Integer memory);

        /**
         * <p>
         * The command to send to the container that overrides the default command from the Docker image or the job
         * definition.
         * </p>
         * 
         * @param command
         *        The command to send to the container that overrides the default command from the Docker image or the
         *        job definition.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(Collection<String> command);

        /**
         * <p>
         * The command to send to the container that overrides the default command from the Docker image or the job
         * definition.
         * </p>
         * 
         * @param command
         *        The command to send to the container that overrides the default command from the Docker image or the
         *        job definition.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(String... command);

        /**
         * <p>
         * The environment variables to send to the container. You can add new environment variables, which are added to
         * the container at launch, or you can override the existing environment variables from the Docker image or the
         * job definition.
         * </p>
         * <note>
         * <p>
         * Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved for
         * variables that are set by the AWS Batch service.
         * </p>
         * </note>
         * 
         * @param environment
         *        The environment variables to send to the container. You can add new environment variables, which are
         *        added to the container at launch, or you can override the existing environment variables from the
         *        Docker image or the job definition.</p> <note>
         *        <p>
         *        Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved
         *        for variables that are set by the AWS Batch service.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environment(Collection<KeyValuePair> environment);

        /**
         * <p>
         * The environment variables to send to the container. You can add new environment variables, which are added to
         * the container at launch, or you can override the existing environment variables from the Docker image or the
         * job definition.
         * </p>
         * <note>
         * <p>
         * Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved for
         * variables that are set by the AWS Batch service.
         * </p>
         * </note>
         * 
         * @param environment
         *        The environment variables to send to the container. You can add new environment variables, which are
         *        added to the container at launch, or you can override the existing environment variables from the
         *        Docker image or the job definition.</p> <note>
         *        <p>
         *        Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved
         *        for variables that are set by the AWS Batch service.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environment(KeyValuePair... environment);
    }

    static final class BuilderImpl implements Builder {
        private Integer vcpus;

        private Integer memory;

        private List<String> command;

        private List<KeyValuePair> environment;

        private BuilderImpl() {
        }

        private BuilderImpl(ContainerOverrides model) {
            vcpus(model.vcpus);
            memory(model.memory);
            command(model.command);
            environment(model.environment);
        }

        public final Integer getVcpus() {
            return vcpus;
        }

        @Override
        public final Builder vcpus(Integer vcpus) {
            this.vcpus = vcpus;
            return this;
        }

        public final void setVcpus(Integer vcpus) {
            this.vcpus = vcpus;
        }

        public final Integer getMemory() {
            return memory;
        }

        @Override
        public final Builder memory(Integer memory) {
            this.memory = memory;
            return this;
        }

        public final void setMemory(Integer memory) {
            this.memory = memory;
        }

        public final Collection<String> getCommand() {
            return command;
        }

        @Override
        public final Builder command(Collection<String> command) {
            this.command = StringListCopier.copy(command);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder command(String... command) {
            command(Arrays.asList(command));
            return this;
        }

        public final void setCommand(Collection<String> command) {
            this.command = StringListCopier.copy(command);
        }

        public final Collection<KeyValuePair.Builder> getEnvironment() {
            return environment != null ? environment.stream().map(KeyValuePair::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder environment(Collection<KeyValuePair> environment) {
            this.environment = EnvironmentVariablesCopier.copy(environment);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder environment(KeyValuePair... environment) {
            environment(Arrays.asList(environment));
            return this;
        }

        public final void setEnvironment(Collection<KeyValuePair.BuilderImpl> environment) {
            this.environment = EnvironmentVariablesCopier.copyFromBuilder(environment);
        }

        @Override
        public ContainerOverrides build() {
            return new ContainerOverrides(this);
        }
    }
}
