/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.batch;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.batch.model.BatchException;
import software.amazon.awssdk.services.batch.model.BatchRequest;
import software.amazon.awssdk.services.batch.model.CancelJobRequest;
import software.amazon.awssdk.services.batch.model.CancelJobResponse;
import software.amazon.awssdk.services.batch.model.ClientException;
import software.amazon.awssdk.services.batch.model.CreateComputeEnvironmentRequest;
import software.amazon.awssdk.services.batch.model.CreateComputeEnvironmentResponse;
import software.amazon.awssdk.services.batch.model.CreateJobQueueRequest;
import software.amazon.awssdk.services.batch.model.CreateJobQueueResponse;
import software.amazon.awssdk.services.batch.model.DeleteComputeEnvironmentRequest;
import software.amazon.awssdk.services.batch.model.DeleteComputeEnvironmentResponse;
import software.amazon.awssdk.services.batch.model.DeleteJobQueueRequest;
import software.amazon.awssdk.services.batch.model.DeleteJobQueueResponse;
import software.amazon.awssdk.services.batch.model.DeregisterJobDefinitionRequest;
import software.amazon.awssdk.services.batch.model.DeregisterJobDefinitionResponse;
import software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsRequest;
import software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsResponse;
import software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsRequest;
import software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsResponse;
import software.amazon.awssdk.services.batch.model.DescribeJobQueuesRequest;
import software.amazon.awssdk.services.batch.model.DescribeJobQueuesResponse;
import software.amazon.awssdk.services.batch.model.DescribeJobsRequest;
import software.amazon.awssdk.services.batch.model.DescribeJobsResponse;
import software.amazon.awssdk.services.batch.model.ListJobsRequest;
import software.amazon.awssdk.services.batch.model.ListJobsResponse;
import software.amazon.awssdk.services.batch.model.RegisterJobDefinitionRequest;
import software.amazon.awssdk.services.batch.model.RegisterJobDefinitionResponse;
import software.amazon.awssdk.services.batch.model.ServerException;
import software.amazon.awssdk.services.batch.model.SubmitJobRequest;
import software.amazon.awssdk.services.batch.model.SubmitJobResponse;
import software.amazon.awssdk.services.batch.model.TerminateJobRequest;
import software.amazon.awssdk.services.batch.model.TerminateJobResponse;
import software.amazon.awssdk.services.batch.model.UpdateComputeEnvironmentRequest;
import software.amazon.awssdk.services.batch.model.UpdateComputeEnvironmentResponse;
import software.amazon.awssdk.services.batch.model.UpdateJobQueueRequest;
import software.amazon.awssdk.services.batch.model.UpdateJobQueueResponse;
import software.amazon.awssdk.services.batch.paginators.DescribeComputeEnvironmentsPublisher;
import software.amazon.awssdk.services.batch.paginators.DescribeJobDefinitionsPublisher;
import software.amazon.awssdk.services.batch.paginators.DescribeJobQueuesPublisher;
import software.amazon.awssdk.services.batch.paginators.ListJobsPublisher;
import software.amazon.awssdk.services.batch.transform.CancelJobRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.CreateComputeEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.CreateJobQueueRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.DeleteComputeEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.DeleteJobQueueRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.DeregisterJobDefinitionRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.DescribeComputeEnvironmentsRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.DescribeJobDefinitionsRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.DescribeJobQueuesRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.DescribeJobsRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.ListJobsRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.RegisterJobDefinitionRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.SubmitJobRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.TerminateJobRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.UpdateComputeEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.UpdateJobQueueRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link BatchAsyncClient}.
 *
 * @see BatchAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultBatchAsyncClient implements BatchAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultBatchAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultBatchAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Cancels a job in an AWS Batch job queue. Jobs that are in the <code>SUBMITTED</code>, <code>PENDING</code>, or
     * <code>RUNNABLE</code> state are cancelled. Jobs that have progressed to <code>STARTING</code> or
     * <code>RUNNING</code> are not cancelled (but the API operation still succeeds, even if no job is cancelled); these
     * jobs must be terminated with the <a>TerminateJob</a> operation.
     * </p>
     *
     * @param cancelJobRequest
     * @return A Java Future containing the result of the CancelJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.CancelJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/CancelJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CancelJobResponse> cancelJob(CancelJobRequest cancelJobRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CancelJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CancelJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelJobRequest, CancelJobResponse>().withOperationName("CancelJob")
                            .withMarshaller(new CancelJobRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(cancelJobRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an AWS Batch compute environment. You can create <code>MANAGED</code> or <code>UNMANAGED</code> compute
     * environments.
     * </p>
     * <p>
     * In a managed compute environment, AWS Batch manages the capacity and instance types of the compute resources
     * within the environment. This is based on the compute resource specification that you define or the <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html">launch template</a> that you
     * specify when you create the compute environment. You can choose to use Amazon EC2 On-Demand Instances or Spot
     * Instances in your managed compute environment. You can optionally set a maximum price so that Spot Instances only
     * launch when the Spot Instance price is below a specified percentage of the On-Demand price.
     * </p>
     * <note>
     * <p>
     * Multi-node parallel jobs are not supported on Spot Instances.
     * </p>
     * </note>
     * <p>
     * In an unmanaged compute environment, you can manage your own compute resources. This provides more compute
     * resource configuration options, such as using a custom AMI, but you must ensure that your AMI meets the Amazon
     * ECS container instance AMI specification. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container_instance_AMIs.html">Container
     * Instance AMIs</a> in the <i>Amazon Elastic Container Service Developer Guide</i>. After you have created your
     * unmanaged compute environment, you can use the <a>DescribeComputeEnvironments</a> operation to find the Amazon
     * ECS cluster that is associated with it. Then, manually launch your container instances into that Amazon ECS
     * cluster. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_container_instance.html">Launching an
     * Amazon ECS Container Instance</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <note>
     * <p>
     * AWS Batch does not upgrade the AMIs in a compute environment after it is created (for example, when a newer
     * version of the Amazon ECS-optimized AMI is available). You are responsible for the management of the guest
     * operating system (including updates and security patches) and any additional application software or utilities
     * that you install on the compute resources. To use a new AMI for your AWS Batch jobs:
     * </p>
     * <ol>
     * <li>
     * <p>
     * Create a new compute environment with the new AMI.
     * </p>
     * </li>
     * <li>
     * <p>
     * Add the compute environment to an existing job queue.
     * </p>
     * </li>
     * <li>
     * <p>
     * Remove the old compute environment from your job queue.
     * </p>
     * </li>
     * <li>
     * <p>
     * Delete the old compute environment.
     * </p>
     * </li>
     * </ol>
     * </note>
     *
     * @param createComputeEnvironmentRequest
     * @return A Java Future containing the result of the CreateComputeEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.CreateComputeEnvironment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/CreateComputeEnvironment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateComputeEnvironmentResponse> createComputeEnvironment(
            CreateComputeEnvironmentRequest createComputeEnvironmentRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateComputeEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateComputeEnvironmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateComputeEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateComputeEnvironmentRequest, CreateComputeEnvironmentResponse>()
                            .withOperationName("CreateComputeEnvironment")
                            .withMarshaller(new CreateComputeEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createComputeEnvironmentRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an AWS Batch job queue. When you create a job queue, you associate one or more compute environments to
     * the queue and assign an order of preference for the compute environments.
     * </p>
     * <p>
     * You also set a priority to the job queue that determines the order in which the AWS Batch scheduler places jobs
     * onto its associated compute environments. For example, if a compute environment is associated with more than one
     * job queue, the job queue with a higher priority is given preference for scheduling jobs to that compute
     * environment.
     * </p>
     *
     * @param createJobQueueRequest
     * @return A Java Future containing the result of the CreateJobQueue operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.CreateJobQueue
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/CreateJobQueue" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateJobQueueResponse> createJobQueue(CreateJobQueueRequest createJobQueueRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateJobQueueResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateJobQueueResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateJobQueueResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateJobQueueRequest, CreateJobQueueResponse>()
                            .withOperationName("CreateJobQueue")
                            .withMarshaller(new CreateJobQueueRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createJobQueueRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an AWS Batch compute environment.
     * </p>
     * <p>
     * Before you can delete a compute environment, you must set its state to <code>DISABLED</code> with the
     * <a>UpdateComputeEnvironment</a> API operation and disassociate it from any job queues with the
     * <a>UpdateJobQueue</a> API operation.
     * </p>
     *
     * @param deleteComputeEnvironmentRequest
     * @return A Java Future containing the result of the DeleteComputeEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DeleteComputeEnvironment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DeleteComputeEnvironment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteComputeEnvironmentResponse> deleteComputeEnvironment(
            DeleteComputeEnvironmentRequest deleteComputeEnvironmentRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteComputeEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteComputeEnvironmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteComputeEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteComputeEnvironmentRequest, DeleteComputeEnvironmentResponse>()
                            .withOperationName("DeleteComputeEnvironment")
                            .withMarshaller(new DeleteComputeEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteComputeEnvironmentRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified job queue. You must first disable submissions for a queue with the <a>UpdateJobQueue</a>
     * operation. All jobs in the queue are terminated when you delete a job queue.
     * </p>
     * <p>
     * It is not necessary to disassociate compute environments from a queue before submitting a
     * <code>DeleteJobQueue</code> request.
     * </p>
     *
     * @param deleteJobQueueRequest
     * @return A Java Future containing the result of the DeleteJobQueue operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DeleteJobQueue
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DeleteJobQueue" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteJobQueueResponse> deleteJobQueue(DeleteJobQueueRequest deleteJobQueueRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteJobQueueResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteJobQueueResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteJobQueueResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteJobQueueRequest, DeleteJobQueueResponse>()
                            .withOperationName("DeleteJobQueue")
                            .withMarshaller(new DeleteJobQueueRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteJobQueueRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deregisters an AWS Batch job definition.
     * </p>
     *
     * @param deregisterJobDefinitionRequest
     * @return A Java Future containing the result of the DeregisterJobDefinition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DeregisterJobDefinition
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DeregisterJobDefinition" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeregisterJobDefinitionResponse> deregisterJobDefinition(
            DeregisterJobDefinitionRequest deregisterJobDefinitionRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeregisterJobDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeregisterJobDefinitionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeregisterJobDefinitionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeregisterJobDefinitionRequest, DeregisterJobDefinitionResponse>()
                            .withOperationName("DeregisterJobDefinition")
                            .withMarshaller(new DeregisterJobDefinitionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deregisterJobDefinitionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes one or more of your compute environments.
     * </p>
     * <p>
     * If you are using an unmanaged compute environment, you can use the <code>DescribeComputeEnvironment</code>
     * operation to determine the <code>ecsClusterArn</code> that you should launch your Amazon ECS container instances
     * into.
     * </p>
     *
     * @param describeComputeEnvironmentsRequest
     * @return A Java Future containing the result of the DescribeComputeEnvironments operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeComputeEnvironments
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeComputeEnvironments"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeComputeEnvironmentsResponse> describeComputeEnvironments(
            DescribeComputeEnvironmentsRequest describeComputeEnvironmentsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeComputeEnvironmentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeComputeEnvironmentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeComputeEnvironmentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeComputeEnvironmentsRequest, DescribeComputeEnvironmentsResponse>()
                            .withOperationName("DescribeComputeEnvironments")
                            .withMarshaller(new DescribeComputeEnvironmentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeComputeEnvironmentsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes one or more of your compute environments.
     * </p>
     * <p>
     * If you are using an unmanaged compute environment, you can use the <code>DescribeComputeEnvironment</code>
     * operation to determine the <code>ecsClusterArn</code> that you should launch your Amazon ECS container instances
     * into.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeComputeEnvironments(software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeComputeEnvironmentsPublisher publisher = client.describeComputeEnvironmentsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeComputeEnvironmentsPublisher publisher = client.describeComputeEnvironmentsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeComputeEnvironments(software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeComputeEnvironmentsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeComputeEnvironments
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeComputeEnvironments"
     *      target="_top">AWS API Documentation</a>
     */
    public DescribeComputeEnvironmentsPublisher describeComputeEnvironmentsPaginator(
            DescribeComputeEnvironmentsRequest describeComputeEnvironmentsRequest) {
        return new DescribeComputeEnvironmentsPublisher(this, applyPaginatorUserAgent(describeComputeEnvironmentsRequest));
    }

    /**
     * <p>
     * Describes a list of job definitions. You can specify a <code>status</code> (such as <code>ACTIVE</code>) to only
     * return job definitions that match that status.
     * </p>
     *
     * @param describeJobDefinitionsRequest
     * @return A Java Future containing the result of the DescribeJobDefinitions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeJobDefinitions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobDefinitions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeJobDefinitionsResponse> describeJobDefinitions(
            DescribeJobDefinitionsRequest describeJobDefinitionsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeJobDefinitionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeJobDefinitionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeJobDefinitionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeJobDefinitionsRequest, DescribeJobDefinitionsResponse>()
                            .withOperationName("DescribeJobDefinitions")
                            .withMarshaller(new DescribeJobDefinitionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeJobDefinitionsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes a list of job definitions. You can specify a <code>status</code> (such as <code>ACTIVE</code>) to only
     * return job definitions that match that status.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeJobDefinitions(software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeJobDefinitionsPublisher publisher = client.describeJobDefinitionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeJobDefinitionsPublisher publisher = client.describeJobDefinitionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeJobDefinitions(software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeJobDefinitionsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeJobDefinitions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobDefinitions" target="_top">AWS
     *      API Documentation</a>
     */
    public DescribeJobDefinitionsPublisher describeJobDefinitionsPaginator(
            DescribeJobDefinitionsRequest describeJobDefinitionsRequest) {
        return new DescribeJobDefinitionsPublisher(this, applyPaginatorUserAgent(describeJobDefinitionsRequest));
    }

    /**
     * <p>
     * Describes one or more of your job queues.
     * </p>
     *
     * @param describeJobQueuesRequest
     * @return A Java Future containing the result of the DescribeJobQueues operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeJobQueues
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobQueues" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeJobQueuesResponse> describeJobQueues(DescribeJobQueuesRequest describeJobQueuesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeJobQueuesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeJobQueuesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeJobQueuesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeJobQueuesRequest, DescribeJobQueuesResponse>()
                            .withOperationName("DescribeJobQueues")
                            .withMarshaller(new DescribeJobQueuesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeJobQueuesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes one or more of your job queues.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeJobQueues(software.amazon.awssdk.services.batch.model.DescribeJobQueuesRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeJobQueuesPublisher publisher = client.describeJobQueuesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeJobQueuesPublisher publisher = client.describeJobQueuesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.batch.model.DescribeJobQueuesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.batch.model.DescribeJobQueuesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeJobQueues(software.amazon.awssdk.services.batch.model.DescribeJobQueuesRequest)} operation.</b>
     * </p>
     *
     * @param describeJobQueuesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeJobQueues
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobQueues" target="_top">AWS API
     *      Documentation</a>
     */
    public DescribeJobQueuesPublisher describeJobQueuesPaginator(DescribeJobQueuesRequest describeJobQueuesRequest) {
        return new DescribeJobQueuesPublisher(this, applyPaginatorUserAgent(describeJobQueuesRequest));
    }

    /**
     * <p>
     * Describes a list of AWS Batch jobs.
     * </p>
     *
     * @param describeJobsRequest
     * @return A Java Future containing the result of the DescribeJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeJobsResponse> describeJobs(DescribeJobsRequest describeJobsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeJobsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeJobsRequest, DescribeJobsResponse>()
                            .withOperationName("DescribeJobs").withMarshaller(new DescribeJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeJobsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of AWS Batch jobs.
     * </p>
     * <p>
     * You must specify only one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * a job queue ID to return a list of jobs in that job queue
     * </p>
     * </li>
     * <li>
     * <p>
     * a multi-node parallel job ID to return a list of that job's nodes
     * </p>
     * </li>
     * <li>
     * <p>
     * an array job ID to return a list of that job's children
     * </p>
     * </li>
     * </ul>
     * <p>
     * You can filter the results by job status with the <code>jobStatus</code> parameter. If you do not specify a
     * status, only <code>RUNNING</code> jobs are returned.
     * </p>
     *
     * @param listJobsRequest
     * @return A Java Future containing the result of the ListJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.ListJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListJobsResponse> listJobs(ListJobsRequest listJobsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListJobsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListJobsRequest, ListJobsResponse>().withOperationName("ListJobs")
                            .withMarshaller(new ListJobsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listJobsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of AWS Batch jobs.
     * </p>
     * <p>
     * You must specify only one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * a job queue ID to return a list of jobs in that job queue
     * </p>
     * </li>
     * <li>
     * <p>
     * a multi-node parallel job ID to return a list of that job's nodes
     * </p>
     * </li>
     * <li>
     * <p>
     * an array job ID to return a list of that job's children
     * </p>
     * </li>
     * </ul>
     * <p>
     * You can filter the results by job status with the <code>jobStatus</code> parameter. If you do not specify a
     * status, only <code>RUNNING</code> jobs are returned.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listJobs(software.amazon.awssdk.services.batch.model.ListJobsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.ListJobsPublisher publisher = client.listJobsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.ListJobsPublisher publisher = client.listJobsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.batch.model.ListJobsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.batch.model.ListJobsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listJobs(software.amazon.awssdk.services.batch.model.ListJobsRequest)} operation.</b>
     * </p>
     *
     * @param listJobsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.ListJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    public ListJobsPublisher listJobsPaginator(ListJobsRequest listJobsRequest) {
        return new ListJobsPublisher(this, applyPaginatorUserAgent(listJobsRequest));
    }

    /**
     * <p>
     * Registers an AWS Batch job definition.
     * </p>
     *
     * @param registerJobDefinitionRequest
     * @return A Java Future containing the result of the RegisterJobDefinition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.RegisterJobDefinition
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/RegisterJobDefinition" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<RegisterJobDefinitionResponse> registerJobDefinition(
            RegisterJobDefinitionRequest registerJobDefinitionRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RegisterJobDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RegisterJobDefinitionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RegisterJobDefinitionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RegisterJobDefinitionRequest, RegisterJobDefinitionResponse>()
                            .withOperationName("RegisterJobDefinition")
                            .withMarshaller(new RegisterJobDefinitionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(registerJobDefinitionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Submits an AWS Batch job from a job definition. Parameters specified during <a>SubmitJob</a> override parameters
     * defined in the job definition.
     * </p>
     *
     * @param submitJobRequest
     * @return A Java Future containing the result of the SubmitJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.SubmitJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/SubmitJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<SubmitJobResponse> submitJob(SubmitJobRequest submitJobRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SubmitJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    SubmitJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SubmitJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SubmitJobRequest, SubmitJobResponse>().withOperationName("SubmitJob")
                            .withMarshaller(new SubmitJobRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(submitJobRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Terminates a job in a job queue. Jobs that are in the <code>STARTING</code> or <code>RUNNING</code> state are
     * terminated, which causes them to transition to <code>FAILED</code>. Jobs that have not progressed to the
     * <code>STARTING</code> state are cancelled.
     * </p>
     *
     * @param terminateJobRequest
     * @return A Java Future containing the result of the TerminateJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.TerminateJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/TerminateJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TerminateJobResponse> terminateJob(TerminateJobRequest terminateJobRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TerminateJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TerminateJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TerminateJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TerminateJobRequest, TerminateJobResponse>()
                            .withOperationName("TerminateJob").withMarshaller(new TerminateJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(terminateJobRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an AWS Batch compute environment.
     * </p>
     *
     * @param updateComputeEnvironmentRequest
     * @return A Java Future containing the result of the UpdateComputeEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.UpdateComputeEnvironment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/UpdateComputeEnvironment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateComputeEnvironmentResponse> updateComputeEnvironment(
            UpdateComputeEnvironmentRequest updateComputeEnvironmentRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateComputeEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateComputeEnvironmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateComputeEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateComputeEnvironmentRequest, UpdateComputeEnvironmentResponse>()
                            .withOperationName("UpdateComputeEnvironment")
                            .withMarshaller(new UpdateComputeEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateComputeEnvironmentRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a job queue.
     * </p>
     *
     * @param updateJobQueueRequest
     * @return A Java Future containing the result of the UpdateJobQueue operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that is not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.UpdateJobQueue
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/UpdateJobQueue" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateJobQueueResponse> updateJobQueue(UpdateJobQueueRequest updateJobQueueRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateJobQueueResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateJobQueueResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateJobQueueResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateJobQueueRequest, UpdateJobQueueResponse>()
                            .withOperationName("UpdateJobQueue")
                            .withMarshaller(new UpdateJobQueueRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateJobQueueRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(BatchException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServerException")
                                .exceptionBuilderSupplier(ServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ClientException")
                                .exceptionBuilderSupplier(ClientException::builder).httpStatusCode(400).build());
    }

    private <T extends BatchRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
