/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.batch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Linux-specific modifications that are applied to the container, such as details for device mappings.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LinuxParameters implements SdkPojo, Serializable, ToCopyableBuilder<LinuxParameters.Builder, LinuxParameters> {
    private static final SdkField<List<Device>> DEVICES_FIELD = SdkField
            .<List<Device>> builder(MarshallingType.LIST)
            .getter(getter(LinuxParameters::devices))
            .setter(setter(Builder::devices))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("devices").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Device> builder(MarshallingType.SDK_POJO)
                                            .constructor(Device::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DEVICES_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<Device> devices;

    private LinuxParameters(BuilderImpl builder) {
        this.devices = builder.devices;
    }

    /**
     * Returns true if the Devices property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasDevices() {
        return devices != null && !(devices instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Any host devices to expose to the container. This parameter maps to <code>Devices</code> in the <a
     * href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the <a
     * href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--device</code> option to <a
     * href="https://docs.docker.com/engine/reference/run/">docker run</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasDevices()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Any host devices to expose to the container. This parameter maps to <code>Devices</code> in the <a
     *         href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the
     *         <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--device</code>
     *         option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
     */
    public List<Device> devices() {
        return devices;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(devices());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LinuxParameters)) {
            return false;
        }
        LinuxParameters other = (LinuxParameters) obj;
        return Objects.equals(devices(), other.devices());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("LinuxParameters").add("Devices", devices()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "devices":
            return Optional.ofNullable(clazz.cast(devices()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LinuxParameters, T> g) {
        return obj -> g.apply((LinuxParameters) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LinuxParameters> {
        /**
         * <p>
         * Any host devices to expose to the container. This parameter maps to <code>Devices</code> in the <a
         * href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the <a
         * href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--device</code> option
         * to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
         * </p>
         * 
         * @param devices
         *        Any host devices to expose to the container. This parameter maps to <code>Devices</code> in the <a
         *        href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of
         *        the <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the
         *        <code>--device</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
         *        run</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder devices(Collection<Device> devices);

        /**
         * <p>
         * Any host devices to expose to the container. This parameter maps to <code>Devices</code> in the <a
         * href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the <a
         * href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--device</code> option
         * to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
         * </p>
         * 
         * @param devices
         *        Any host devices to expose to the container. This parameter maps to <code>Devices</code> in the <a
         *        href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of
         *        the <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the
         *        <code>--device</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
         *        run</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder devices(Device... devices);

        /**
         * <p>
         * Any host devices to expose to the container. This parameter maps to <code>Devices</code> in the <a
         * href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the <a
         * href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--device</code> option
         * to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Device>.Builder} avoiding the need to
         * create one manually via {@link List<Device>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Device>.Builder#build()} is called immediately and its
         * result is passed to {@link #devices(List<Device>)}.
         * 
         * @param devices
         *        a consumer that will call methods on {@link List<Device>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #devices(List<Device>)
         */
        Builder devices(Consumer<Device.Builder>... devices);
    }

    static final class BuilderImpl implements Builder {
        private List<Device> devices = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(LinuxParameters model) {
            devices(model.devices);
        }

        public final Collection<Device.Builder> getDevices() {
            return devices != null ? devices.stream().map(Device::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder devices(Collection<Device> devices) {
            this.devices = DevicesListCopier.copy(devices);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder devices(Device... devices) {
            devices(Arrays.asList(devices));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder devices(Consumer<Device.Builder>... devices) {
            devices(Stream.of(devices).map(c -> Device.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setDevices(Collection<Device.BuilderImpl> devices) {
            this.devices = DevicesListCopier.copyFromBuilder(devices);
        }

        @Override
        public LinuxParameters build() {
            return new LinuxParameters(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
