/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.batch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Object representing any node overrides to a job definition that is used in a <a>SubmitJob</a> API operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class NodeOverrides implements SdkPojo, Serializable, ToCopyableBuilder<NodeOverrides.Builder, NodeOverrides> {
    private static final SdkField<Integer> NUM_NODES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(NodeOverrides::numNodes)).setter(setter(Builder::numNodes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("numNodes").build()).build();

    private static final SdkField<List<NodePropertyOverride>> NODE_PROPERTY_OVERRIDES_FIELD = SdkField
            .<List<NodePropertyOverride>> builder(MarshallingType.LIST)
            .getter(getter(NodeOverrides::nodePropertyOverrides))
            .setter(setter(Builder::nodePropertyOverrides))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("nodePropertyOverrides").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<NodePropertyOverride> builder(MarshallingType.SDK_POJO)
                                            .constructor(NodePropertyOverride::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NUM_NODES_FIELD,
            NODE_PROPERTY_OVERRIDES_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer numNodes;

    private final List<NodePropertyOverride> nodePropertyOverrides;

    private NodeOverrides(BuilderImpl builder) {
        this.numNodes = builder.numNodes;
        this.nodePropertyOverrides = builder.nodePropertyOverrides;
    }

    /**
     * <p>
     * The number of nodes to use with a multi-node parallel job. This value overrides the number of nodes that are
     * specified in the job definition. To use this override:
     * </p>
     * <ul>
     * <li>
     * <p>
     * There must be at least one node range in your job definition that has an open upper boundary (such as
     * <code>:</code> or <code>n:</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * The lower boundary of the node range specified in the job definition must be fewer than the number of nodes
     * specified in the override.
     * </p>
     * </li>
     * <li>
     * <p>
     * The main node index specified in the job definition must be fewer than the number of nodes specified in the
     * override.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The number of nodes to use with a multi-node parallel job. This value overrides the number of nodes that
     *         are specified in the job definition. To use this override:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         There must be at least one node range in your job definition that has an open upper boundary (such as
     *         <code>:</code> or <code>n:</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The lower boundary of the node range specified in the job definition must be fewer than the number of
     *         nodes specified in the override.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The main node index specified in the job definition must be fewer than the number of nodes specified in
     *         the override.
     *         </p>
     *         </li>
     */
    public Integer numNodes() {
        return numNodes;
    }

    /**
     * Returns true if the NodePropertyOverrides property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasNodePropertyOverrides() {
        return nodePropertyOverrides != null && !(nodePropertyOverrides instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The node property overrides for the job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasNodePropertyOverrides()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The node property overrides for the job.
     */
    public List<NodePropertyOverride> nodePropertyOverrides() {
        return nodePropertyOverrides;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(numNodes());
        hashCode = 31 * hashCode + Objects.hashCode(nodePropertyOverrides());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof NodeOverrides)) {
            return false;
        }
        NodeOverrides other = (NodeOverrides) obj;
        return Objects.equals(numNodes(), other.numNodes())
                && Objects.equals(nodePropertyOverrides(), other.nodePropertyOverrides());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("NodeOverrides").add("NumNodes", numNodes())
                .add("NodePropertyOverrides", nodePropertyOverrides()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "numNodes":
            return Optional.ofNullable(clazz.cast(numNodes()));
        case "nodePropertyOverrides":
            return Optional.ofNullable(clazz.cast(nodePropertyOverrides()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<NodeOverrides, T> g) {
        return obj -> g.apply((NodeOverrides) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, NodeOverrides> {
        /**
         * <p>
         * The number of nodes to use with a multi-node parallel job. This value overrides the number of nodes that are
         * specified in the job definition. To use this override:
         * </p>
         * <ul>
         * <li>
         * <p>
         * There must be at least one node range in your job definition that has an open upper boundary (such as
         * <code>:</code> or <code>n:</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * The lower boundary of the node range specified in the job definition must be fewer than the number of nodes
         * specified in the override.
         * </p>
         * </li>
         * <li>
         * <p>
         * The main node index specified in the job definition must be fewer than the number of nodes specified in the
         * override.
         * </p>
         * </li>
         * </ul>
         * 
         * @param numNodes
         *        The number of nodes to use with a multi-node parallel job. This value overrides the number of nodes
         *        that are specified in the job definition. To use this override:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        There must be at least one node range in your job definition that has an open upper boundary (such as
         *        <code>:</code> or <code>n:</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The lower boundary of the node range specified in the job definition must be fewer than the number of
         *        nodes specified in the override.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The main node index specified in the job definition must be fewer than the number of nodes specified
         *        in the override.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder numNodes(Integer numNodes);

        /**
         * <p>
         * The node property overrides for the job.
         * </p>
         * 
         * @param nodePropertyOverrides
         *        The node property overrides for the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodePropertyOverrides(Collection<NodePropertyOverride> nodePropertyOverrides);

        /**
         * <p>
         * The node property overrides for the job.
         * </p>
         * 
         * @param nodePropertyOverrides
         *        The node property overrides for the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodePropertyOverrides(NodePropertyOverride... nodePropertyOverrides);

        /**
         * <p>
         * The node property overrides for the job.
         * </p>
         * This is a convenience that creates an instance of the {@link List<NodePropertyOverride>.Builder} avoiding the
         * need to create one manually via {@link List<NodePropertyOverride>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<NodePropertyOverride>.Builder#build()} is called immediately
         * and its result is passed to {@link #nodePropertyOverrides(List<NodePropertyOverride>)}.
         * 
         * @param nodePropertyOverrides
         *        a consumer that will call methods on {@link List<NodePropertyOverride>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #nodePropertyOverrides(List<NodePropertyOverride>)
         */
        Builder nodePropertyOverrides(Consumer<NodePropertyOverride.Builder>... nodePropertyOverrides);
    }

    static final class BuilderImpl implements Builder {
        private Integer numNodes;

        private List<NodePropertyOverride> nodePropertyOverrides = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(NodeOverrides model) {
            numNodes(model.numNodes);
            nodePropertyOverrides(model.nodePropertyOverrides);
        }

        public final Integer getNumNodes() {
            return numNodes;
        }

        @Override
        public final Builder numNodes(Integer numNodes) {
            this.numNodes = numNodes;
            return this;
        }

        public final void setNumNodes(Integer numNodes) {
            this.numNodes = numNodes;
        }

        public final Collection<NodePropertyOverride.Builder> getNodePropertyOverrides() {
            return nodePropertyOverrides != null ? nodePropertyOverrides.stream().map(NodePropertyOverride::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder nodePropertyOverrides(Collection<NodePropertyOverride> nodePropertyOverrides) {
            this.nodePropertyOverrides = NodePropertyOverridesCopier.copy(nodePropertyOverrides);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nodePropertyOverrides(NodePropertyOverride... nodePropertyOverrides) {
            nodePropertyOverrides(Arrays.asList(nodePropertyOverrides));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nodePropertyOverrides(Consumer<NodePropertyOverride.Builder>... nodePropertyOverrides) {
            nodePropertyOverrides(Stream.of(nodePropertyOverrides)
                    .map(c -> NodePropertyOverride.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setNodePropertyOverrides(Collection<NodePropertyOverride.BuilderImpl> nodePropertyOverrides) {
            this.nodePropertyOverrides = NodePropertyOverridesCopier.copyFromBuilder(nodePropertyOverrides);
        }

        @Override
        public NodeOverrides build() {
            return new NodeOverrides(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
