/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.batch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object representing a container instance host device.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Device implements SdkPojo, Serializable, ToCopyableBuilder<Device.Builder, Device> {
    private static final SdkField<String> HOST_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Device::hostPath)).setter(setter(Builder::hostPath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("hostPath").build()).build();

    private static final SdkField<String> CONTAINER_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Device::containerPath)).setter(setter(Builder::containerPath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("containerPath").build()).build();

    private static final SdkField<List<String>> PERMISSIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(Device::permissionsAsStrings))
            .setter(setter(Builder::permissionsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("permissions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(HOST_PATH_FIELD,
            CONTAINER_PATH_FIELD, PERMISSIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String hostPath;

    private final String containerPath;

    private final List<String> permissions;

    private Device(BuilderImpl builder) {
        this.hostPath = builder.hostPath;
        this.containerPath = builder.containerPath;
        this.permissions = builder.permissions;
    }

    /**
     * <p>
     * The path for the device on the host container instance.
     * </p>
     * 
     * @return The path for the device on the host container instance.
     */
    public String hostPath() {
        return hostPath;
    }

    /**
     * <p>
     * The path inside the container at which to expose the host device. By default the <code>hostPath</code> value is
     * used.
     * </p>
     * 
     * @return The path inside the container at which to expose the host device. By default the <code>hostPath</code>
     *         value is used.
     */
    public String containerPath() {
        return containerPath;
    }

    /**
     * <p>
     * The explicit permissions to provide to the container for the device. By default, the container has permissions
     * for <code>read</code>, <code>write</code>, and <code>mknod</code> for the device.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasPermissions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The explicit permissions to provide to the container for the device. By default, the container has
     *         permissions for <code>read</code>, <code>write</code>, and <code>mknod</code> for the device.
     */
    public List<DeviceCgroupPermission> permissions() {
        return DeviceCgroupPermissionsCopier.copyStringToEnum(permissions);
    }

    /**
     * Returns true if the Permissions property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasPermissions() {
        return permissions != null && !(permissions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The explicit permissions to provide to the container for the device. By default, the container has permissions
     * for <code>read</code>, <code>write</code>, and <code>mknod</code> for the device.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasPermissions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The explicit permissions to provide to the container for the device. By default, the container has
     *         permissions for <code>read</code>, <code>write</code>, and <code>mknod</code> for the device.
     */
    public List<String> permissionsAsStrings() {
        return permissions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hostPath());
        hashCode = 31 * hashCode + Objects.hashCode(containerPath());
        hashCode = 31 * hashCode + Objects.hashCode(permissionsAsStrings());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Device)) {
            return false;
        }
        Device other = (Device) obj;
        return Objects.equals(hostPath(), other.hostPath()) && Objects.equals(containerPath(), other.containerPath())
                && Objects.equals(permissionsAsStrings(), other.permissionsAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Device").add("HostPath", hostPath()).add("ContainerPath", containerPath())
                .add("Permissions", permissionsAsStrings()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "hostPath":
            return Optional.ofNullable(clazz.cast(hostPath()));
        case "containerPath":
            return Optional.ofNullable(clazz.cast(containerPath()));
        case "permissions":
            return Optional.ofNullable(clazz.cast(permissionsAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Device, T> g) {
        return obj -> g.apply((Device) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Device> {
        /**
         * <p>
         * The path for the device on the host container instance.
         * </p>
         * 
         * @param hostPath
         *        The path for the device on the host container instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hostPath(String hostPath);

        /**
         * <p>
         * The path inside the container at which to expose the host device. By default the <code>hostPath</code> value
         * is used.
         * </p>
         * 
         * @param containerPath
         *        The path inside the container at which to expose the host device. By default the <code>hostPath</code>
         *        value is used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerPath(String containerPath);

        /**
         * <p>
         * The explicit permissions to provide to the container for the device. By default, the container has
         * permissions for <code>read</code>, <code>write</code>, and <code>mknod</code> for the device.
         * </p>
         * 
         * @param permissions
         *        The explicit permissions to provide to the container for the device. By default, the container has
         *        permissions for <code>read</code>, <code>write</code>, and <code>mknod</code> for the device.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder permissionsWithStrings(Collection<String> permissions);

        /**
         * <p>
         * The explicit permissions to provide to the container for the device. By default, the container has
         * permissions for <code>read</code>, <code>write</code>, and <code>mknod</code> for the device.
         * </p>
         * 
         * @param permissions
         *        The explicit permissions to provide to the container for the device. By default, the container has
         *        permissions for <code>read</code>, <code>write</code>, and <code>mknod</code> for the device.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder permissionsWithStrings(String... permissions);

        /**
         * <p>
         * The explicit permissions to provide to the container for the device. By default, the container has
         * permissions for <code>read</code>, <code>write</code>, and <code>mknod</code> for the device.
         * </p>
         * 
         * @param permissions
         *        The explicit permissions to provide to the container for the device. By default, the container has
         *        permissions for <code>read</code>, <code>write</code>, and <code>mknod</code> for the device.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder permissions(Collection<DeviceCgroupPermission> permissions);

        /**
         * <p>
         * The explicit permissions to provide to the container for the device. By default, the container has
         * permissions for <code>read</code>, <code>write</code>, and <code>mknod</code> for the device.
         * </p>
         * 
         * @param permissions
         *        The explicit permissions to provide to the container for the device. By default, the container has
         *        permissions for <code>read</code>, <code>write</code>, and <code>mknod</code> for the device.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder permissions(DeviceCgroupPermission... permissions);
    }

    static final class BuilderImpl implements Builder {
        private String hostPath;

        private String containerPath;

        private List<String> permissions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Device model) {
            hostPath(model.hostPath);
            containerPath(model.containerPath);
            permissionsWithStrings(model.permissions);
        }

        public final String getHostPath() {
            return hostPath;
        }

        @Override
        public final Builder hostPath(String hostPath) {
            this.hostPath = hostPath;
            return this;
        }

        public final void setHostPath(String hostPath) {
            this.hostPath = hostPath;
        }

        public final String getContainerPath() {
            return containerPath;
        }

        @Override
        public final Builder containerPath(String containerPath) {
            this.containerPath = containerPath;
            return this;
        }

        public final void setContainerPath(String containerPath) {
            this.containerPath = containerPath;
        }

        public final Collection<String> getPermissions() {
            return permissions;
        }

        @Override
        public final Builder permissionsWithStrings(Collection<String> permissions) {
            this.permissions = DeviceCgroupPermissionsCopier.copy(permissions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder permissionsWithStrings(String... permissions) {
            permissionsWithStrings(Arrays.asList(permissions));
            return this;
        }

        @Override
        public final Builder permissions(Collection<DeviceCgroupPermission> permissions) {
            this.permissions = DeviceCgroupPermissionsCopier.copyEnumToString(permissions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder permissions(DeviceCgroupPermission... permissions) {
            permissions(Arrays.asList(permissions));
            return this;
        }

        public final void setPermissions(Collection<String> permissions) {
            this.permissions = DeviceCgroupPermissionsCopier.copy(permissions);
        }

        @Override
        public Device build() {
            return new Device(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
