/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.batch;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.batch.model.BatchException;
import software.amazon.awssdk.services.batch.model.BatchRequest;
import software.amazon.awssdk.services.batch.model.CancelJobRequest;
import software.amazon.awssdk.services.batch.model.CancelJobResponse;
import software.amazon.awssdk.services.batch.model.ClientException;
import software.amazon.awssdk.services.batch.model.CreateComputeEnvironmentRequest;
import software.amazon.awssdk.services.batch.model.CreateComputeEnvironmentResponse;
import software.amazon.awssdk.services.batch.model.CreateJobQueueRequest;
import software.amazon.awssdk.services.batch.model.CreateJobQueueResponse;
import software.amazon.awssdk.services.batch.model.DeleteComputeEnvironmentRequest;
import software.amazon.awssdk.services.batch.model.DeleteComputeEnvironmentResponse;
import software.amazon.awssdk.services.batch.model.DeleteJobQueueRequest;
import software.amazon.awssdk.services.batch.model.DeleteJobQueueResponse;
import software.amazon.awssdk.services.batch.model.DeregisterJobDefinitionRequest;
import software.amazon.awssdk.services.batch.model.DeregisterJobDefinitionResponse;
import software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsRequest;
import software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsResponse;
import software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsRequest;
import software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsResponse;
import software.amazon.awssdk.services.batch.model.DescribeJobQueuesRequest;
import software.amazon.awssdk.services.batch.model.DescribeJobQueuesResponse;
import software.amazon.awssdk.services.batch.model.DescribeJobsRequest;
import software.amazon.awssdk.services.batch.model.DescribeJobsResponse;
import software.amazon.awssdk.services.batch.model.ListJobsRequest;
import software.amazon.awssdk.services.batch.model.ListJobsResponse;
import software.amazon.awssdk.services.batch.model.RegisterJobDefinitionRequest;
import software.amazon.awssdk.services.batch.model.RegisterJobDefinitionResponse;
import software.amazon.awssdk.services.batch.model.ServerException;
import software.amazon.awssdk.services.batch.model.SubmitJobRequest;
import software.amazon.awssdk.services.batch.model.SubmitJobResponse;
import software.amazon.awssdk.services.batch.model.TerminateJobRequest;
import software.amazon.awssdk.services.batch.model.TerminateJobResponse;
import software.amazon.awssdk.services.batch.model.UpdateComputeEnvironmentRequest;
import software.amazon.awssdk.services.batch.model.UpdateComputeEnvironmentResponse;
import software.amazon.awssdk.services.batch.model.UpdateJobQueueRequest;
import software.amazon.awssdk.services.batch.model.UpdateJobQueueResponse;
import software.amazon.awssdk.services.batch.paginators.DescribeComputeEnvironmentsIterable;
import software.amazon.awssdk.services.batch.paginators.DescribeJobDefinitionsIterable;
import software.amazon.awssdk.services.batch.paginators.DescribeJobQueuesIterable;
import software.amazon.awssdk.services.batch.paginators.ListJobsIterable;
import software.amazon.awssdk.services.batch.transform.CancelJobRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.CreateComputeEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.CreateJobQueueRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.DeleteComputeEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.DeleteJobQueueRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.DeregisterJobDefinitionRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.DescribeComputeEnvironmentsRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.DescribeJobDefinitionsRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.DescribeJobQueuesRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.DescribeJobsRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.ListJobsRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.RegisterJobDefinitionRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.SubmitJobRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.TerminateJobRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.UpdateComputeEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.UpdateJobQueueRequestMarshaller;

/**
 * Internal implementation of {@link BatchClient}.
 *
 * @see BatchClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultBatchClient implements BatchClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultBatchClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Cancels a job in an AWS Batch job queue. Jobs that are in the <code>SUBMITTED</code>, <code>PENDING</code>, or
     * <code>RUNNABLE</code> state are cancelled. Jobs that have progressed to <code>STARTING</code> or
     * <code>RUNNING</code> are not cancelled (but the API operation still succeeds, even if no job is cancelled); these
     * jobs must be terminated with the <a>TerminateJob</a> operation.
     * </p>
     *
     * @param cancelJobRequest
     * @return Result of the CancelJob operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action, such as using an action or resource on behalf of a
     *         user that doesn't have permissions to use the action or resource, or specifying an identifier that is not
     *         valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.CancelJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/CancelJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CancelJobResponse cancelJob(CancelJobRequest cancelJobRequest) throws ClientException, ServerException,
            AwsServiceException, SdkClientException, BatchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CancelJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelJob");

            return clientHandler.execute(new ClientExecutionParams<CancelJobRequest, CancelJobResponse>()
                    .withOperationName("CancelJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(cancelJobRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new CancelJobRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelJobRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an AWS Batch compute environment. You can create <code>MANAGED</code> or <code>UNMANAGED</code> compute
     * environments.
     * </p>
     * <p>
     * In a managed compute environment, AWS Batch manages the capacity and instance types of the compute resources
     * within the environment. This is based on the compute resource specification that you define or the <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html">launch template</a> that you
     * specify when you create the compute environment. You can choose to use Amazon EC2 On-Demand Instances or Spot
     * Instances in your managed compute environment. You can optionally set a maximum price so that Spot Instances only
     * launch when the Spot Instance price is below a specified percentage of the On-Demand price.
     * </p>
     * <note>
     * <p>
     * Multi-node parallel jobs are not supported on Spot Instances.
     * </p>
     * </note>
     * <p>
     * In an unmanaged compute environment, you can manage your own compute resources. This provides more compute
     * resource configuration options, such as using a custom AMI, but you must ensure that your AMI meets the Amazon
     * ECS container instance AMI specification. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container_instance_AMIs.html">Container
     * Instance AMIs</a> in the <i>Amazon Elastic Container Service Developer Guide</i>. After you have created your
     * unmanaged compute environment, you can use the <a>DescribeComputeEnvironments</a> operation to find the Amazon
     * ECS cluster that is associated with it. Then, manually launch your container instances into that Amazon ECS
     * cluster. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_container_instance.html">Launching an
     * Amazon ECS Container Instance</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <note>
     * <p>
     * AWS Batch does not upgrade the AMIs in a compute environment after it is created (for example, when a newer
     * version of the Amazon ECS-optimized AMI is available). You are responsible for the management of the guest
     * operating system (including updates and security patches) and any additional application software or utilities
     * that you install on the compute resources. To use a new AMI for your AWS Batch jobs:
     * </p>
     * <ol>
     * <li>
     * <p>
     * Create a new compute environment with the new AMI.
     * </p>
     * </li>
     * <li>
     * <p>
     * Add the compute environment to an existing job queue.
     * </p>
     * </li>
     * <li>
     * <p>
     * Remove the old compute environment from your job queue.
     * </p>
     * </li>
     * <li>
     * <p>
     * Delete the old compute environment.
     * </p>
     * </li>
     * </ol>
     * </note>
     *
     * @param createComputeEnvironmentRequest
     * @return Result of the CreateComputeEnvironment operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action, such as using an action or resource on behalf of a
     *         user that doesn't have permissions to use the action or resource, or specifying an identifier that is not
     *         valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.CreateComputeEnvironment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/CreateComputeEnvironment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateComputeEnvironmentResponse createComputeEnvironment(
            CreateComputeEnvironmentRequest createComputeEnvironmentRequest) throws ClientException, ServerException,
            AwsServiceException, SdkClientException, BatchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateComputeEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateComputeEnvironmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateComputeEnvironment");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateComputeEnvironmentRequest, CreateComputeEnvironmentResponse>()
                            .withOperationName("CreateComputeEnvironment").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createComputeEnvironmentRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateComputeEnvironmentRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createComputeEnvironmentRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an AWS Batch job queue. When you create a job queue, you associate one or more compute environments to
     * the queue and assign an order of preference for the compute environments.
     * </p>
     * <p>
     * You also set a priority to the job queue that determines the order in which the AWS Batch scheduler places jobs
     * onto its associated compute environments. For example, if a compute environment is associated with more than one
     * job queue, the job queue with a higher priority is given preference for scheduling jobs to that compute
     * environment.
     * </p>
     *
     * @param createJobQueueRequest
     * @return Result of the CreateJobQueue operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action, such as using an action or resource on behalf of a
     *         user that doesn't have permissions to use the action or resource, or specifying an identifier that is not
     *         valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.CreateJobQueue
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/CreateJobQueue" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateJobQueueResponse createJobQueue(CreateJobQueueRequest createJobQueueRequest) throws ClientException,
            ServerException, AwsServiceException, SdkClientException, BatchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateJobQueueResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateJobQueueResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateJobQueue");

            return clientHandler.execute(new ClientExecutionParams<CreateJobQueueRequest, CreateJobQueueResponse>()
                    .withOperationName("CreateJobQueue").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createJobQueueRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateJobQueueRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createJobQueueRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an AWS Batch compute environment.
     * </p>
     * <p>
     * Before you can delete a compute environment, you must set its state to <code>DISABLED</code> with the
     * <a>UpdateComputeEnvironment</a> API operation and disassociate it from any job queues with the
     * <a>UpdateJobQueue</a> API operation.
     * </p>
     *
     * @param deleteComputeEnvironmentRequest
     * @return Result of the DeleteComputeEnvironment operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action, such as using an action or resource on behalf of a
     *         user that doesn't have permissions to use the action or resource, or specifying an identifier that is not
     *         valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DeleteComputeEnvironment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DeleteComputeEnvironment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteComputeEnvironmentResponse deleteComputeEnvironment(
            DeleteComputeEnvironmentRequest deleteComputeEnvironmentRequest) throws ClientException, ServerException,
            AwsServiceException, SdkClientException, BatchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteComputeEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteComputeEnvironmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteComputeEnvironment");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteComputeEnvironmentRequest, DeleteComputeEnvironmentResponse>()
                            .withOperationName("DeleteComputeEnvironment").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteComputeEnvironmentRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteComputeEnvironmentRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteComputeEnvironmentRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified job queue. You must first disable submissions for a queue with the <a>UpdateJobQueue</a>
     * operation. All jobs in the queue are terminated when you delete a job queue.
     * </p>
     * <p>
     * It is not necessary to disassociate compute environments from a queue before submitting a
     * <code>DeleteJobQueue</code> request.
     * </p>
     *
     * @param deleteJobQueueRequest
     * @return Result of the DeleteJobQueue operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action, such as using an action or resource on behalf of a
     *         user that doesn't have permissions to use the action or resource, or specifying an identifier that is not
     *         valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DeleteJobQueue
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DeleteJobQueue" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteJobQueueResponse deleteJobQueue(DeleteJobQueueRequest deleteJobQueueRequest) throws ClientException,
            ServerException, AwsServiceException, SdkClientException, BatchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteJobQueueResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteJobQueueResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteJobQueue");

            return clientHandler.execute(new ClientExecutionParams<DeleteJobQueueRequest, DeleteJobQueueResponse>()
                    .withOperationName("DeleteJobQueue").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteJobQueueRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteJobQueueRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteJobQueueRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deregisters an AWS Batch job definition. Job definitions will be permanently deleted after 180 days.
     * </p>
     *
     * @param deregisterJobDefinitionRequest
     * @return Result of the DeregisterJobDefinition operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action, such as using an action or resource on behalf of a
     *         user that doesn't have permissions to use the action or resource, or specifying an identifier that is not
     *         valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DeregisterJobDefinition
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DeregisterJobDefinition" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeregisterJobDefinitionResponse deregisterJobDefinition(DeregisterJobDefinitionRequest deregisterJobDefinitionRequest)
            throws ClientException, ServerException, AwsServiceException, SdkClientException, BatchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeregisterJobDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeregisterJobDefinitionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeregisterJobDefinition");

            return clientHandler
                    .execute(new ClientExecutionParams<DeregisterJobDefinitionRequest, DeregisterJobDefinitionResponse>()
                            .withOperationName("DeregisterJobDefinition").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deregisterJobDefinitionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeregisterJobDefinitionRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deregisterJobDefinitionRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes one or more of your compute environments.
     * </p>
     * <p>
     * If you are using an unmanaged compute environment, you can use the <code>DescribeComputeEnvironment</code>
     * operation to determine the <code>ecsClusterArn</code> that you should launch your Amazon ECS container instances
     * into.
     * </p>
     *
     * @param describeComputeEnvironmentsRequest
     * @return Result of the DescribeComputeEnvironments operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action, such as using an action or resource on behalf of a
     *         user that doesn't have permissions to use the action or resource, or specifying an identifier that is not
     *         valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DescribeComputeEnvironments
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeComputeEnvironments"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeComputeEnvironmentsResponse describeComputeEnvironments(
            DescribeComputeEnvironmentsRequest describeComputeEnvironmentsRequest) throws ClientException, ServerException,
            AwsServiceException, SdkClientException, BatchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeComputeEnvironmentsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeComputeEnvironmentsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeComputeEnvironments");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeComputeEnvironmentsRequest, DescribeComputeEnvironmentsResponse>()
                            .withOperationName("DescribeComputeEnvironments").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeComputeEnvironmentsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeComputeEnvironmentsRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                    describeComputeEnvironmentsRequest.overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes one or more of your compute environments.
     * </p>
     * <p>
     * If you are using an unmanaged compute environment, you can use the <code>DescribeComputeEnvironment</code>
     * operation to determine the <code>ecsClusterArn</code> that you should launch your Amazon ECS container instances
     * into.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeComputeEnvironments(software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeComputeEnvironmentsIterable responses = client.describeComputeEnvironmentsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.batch.paginators.DescribeComputeEnvironmentsIterable responses = client
     *             .describeComputeEnvironmentsPaginator(request);
     *     for (software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeComputeEnvironmentsIterable responses = client.describeComputeEnvironmentsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeComputeEnvironments(software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeComputeEnvironmentsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ClientException
     *         These errors are usually caused by a client action, such as using an action or resource on behalf of a
     *         user that doesn't have permissions to use the action or resource, or specifying an identifier that is not
     *         valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DescribeComputeEnvironments
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeComputeEnvironments"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeComputeEnvironmentsIterable describeComputeEnvironmentsPaginator(
            DescribeComputeEnvironmentsRequest describeComputeEnvironmentsRequest) throws ClientException, ServerException,
            AwsServiceException, SdkClientException, BatchException {
        return new DescribeComputeEnvironmentsIterable(this, applyPaginatorUserAgent(describeComputeEnvironmentsRequest));
    }

    /**
     * <p>
     * Describes a list of job definitions. You can specify a <code>status</code> (such as <code>ACTIVE</code>) to only
     * return job definitions that match that status.
     * </p>
     *
     * @param describeJobDefinitionsRequest
     * @return Result of the DescribeJobDefinitions operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action, such as using an action or resource on behalf of a
     *         user that doesn't have permissions to use the action or resource, or specifying an identifier that is not
     *         valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DescribeJobDefinitions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobDefinitions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeJobDefinitionsResponse describeJobDefinitions(DescribeJobDefinitionsRequest describeJobDefinitionsRequest)
            throws ClientException, ServerException, AwsServiceException, SdkClientException, BatchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeJobDefinitionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeJobDefinitionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeJobDefinitions");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeJobDefinitionsRequest, DescribeJobDefinitionsResponse>()
                            .withOperationName("DescribeJobDefinitions").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeJobDefinitionsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeJobDefinitionsRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeJobDefinitionsRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes a list of job definitions. You can specify a <code>status</code> (such as <code>ACTIVE</code>) to only
     * return job definitions that match that status.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeJobDefinitions(software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeJobDefinitionsIterable responses = client.describeJobDefinitionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.batch.paginators.DescribeJobDefinitionsIterable responses = client
     *             .describeJobDefinitionsPaginator(request);
     *     for (software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeJobDefinitionsIterable responses = client.describeJobDefinitionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeJobDefinitions(software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeJobDefinitionsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ClientException
     *         These errors are usually caused by a client action, such as using an action or resource on behalf of a
     *         user that doesn't have permissions to use the action or resource, or specifying an identifier that is not
     *         valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DescribeJobDefinitions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobDefinitions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeJobDefinitionsIterable describeJobDefinitionsPaginator(
            DescribeJobDefinitionsRequest describeJobDefinitionsRequest) throws ClientException, ServerException,
            AwsServiceException, SdkClientException, BatchException {
        return new DescribeJobDefinitionsIterable(this, applyPaginatorUserAgent(describeJobDefinitionsRequest));
    }

    /**
     * <p>
     * Describes one or more of your job queues.
     * </p>
     *
     * @param describeJobQueuesRequest
     * @return Result of the DescribeJobQueues operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action, such as using an action or resource on behalf of a
     *         user that doesn't have permissions to use the action or resource, or specifying an identifier that is not
     *         valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DescribeJobQueues
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobQueues" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeJobQueuesResponse describeJobQueues(DescribeJobQueuesRequest describeJobQueuesRequest) throws ClientException,
            ServerException, AwsServiceException, SdkClientException, BatchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeJobQueuesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeJobQueuesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeJobQueues");

            return clientHandler.execute(new ClientExecutionParams<DescribeJobQueuesRequest, DescribeJobQueuesResponse>()
                    .withOperationName("DescribeJobQueues").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeJobQueuesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeJobQueuesRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeJobQueuesRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes one or more of your job queues.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeJobQueues(software.amazon.awssdk.services.batch.model.DescribeJobQueuesRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeJobQueuesIterable responses = client.describeJobQueuesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.batch.paginators.DescribeJobQueuesIterable responses = client
     *             .describeJobQueuesPaginator(request);
     *     for (software.amazon.awssdk.services.batch.model.DescribeJobQueuesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeJobQueuesIterable responses = client.describeJobQueuesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeJobQueues(software.amazon.awssdk.services.batch.model.DescribeJobQueuesRequest)} operation.</b>
     * </p>
     *
     * @param describeJobQueuesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ClientException
     *         These errors are usually caused by a client action, such as using an action or resource on behalf of a
     *         user that doesn't have permissions to use the action or resource, or specifying an identifier that is not
     *         valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DescribeJobQueues
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobQueues" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeJobQueuesIterable describeJobQueuesPaginator(DescribeJobQueuesRequest describeJobQueuesRequest)
            throws ClientException, ServerException, AwsServiceException, SdkClientException, BatchException {
        return new DescribeJobQueuesIterable(this, applyPaginatorUserAgent(describeJobQueuesRequest));
    }

    /**
     * <p>
     * Describes a list of AWS Batch jobs.
     * </p>
     *
     * @param describeJobsRequest
     * @return Result of the DescribeJobs operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action, such as using an action or resource on behalf of a
     *         user that doesn't have permissions to use the action or resource, or specifying an identifier that is not
     *         valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DescribeJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeJobsResponse describeJobs(DescribeJobsRequest describeJobsRequest) throws ClientException, ServerException,
            AwsServiceException, SdkClientException, BatchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeJobsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeJobs");

            return clientHandler.execute(new ClientExecutionParams<DescribeJobsRequest, DescribeJobsResponse>()
                    .withOperationName("DescribeJobs").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeJobsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeJobsRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeJobsRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of AWS Batch jobs.
     * </p>
     * <p>
     * You must specify only one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * a job queue ID to return a list of jobs in that job queue
     * </p>
     * </li>
     * <li>
     * <p>
     * a multi-node parallel job ID to return a list of that job's nodes
     * </p>
     * </li>
     * <li>
     * <p>
     * an array job ID to return a list of that job's children
     * </p>
     * </li>
     * </ul>
     * <p>
     * You can filter the results by job status with the <code>jobStatus</code> parameter. If you do not specify a
     * status, only <code>RUNNING</code> jobs are returned.
     * </p>
     *
     * @param listJobsRequest
     * @return Result of the ListJobs operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action, such as using an action or resource on behalf of a
     *         user that doesn't have permissions to use the action or resource, or specifying an identifier that is not
     *         valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.ListJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListJobsResponse listJobs(ListJobsRequest listJobsRequest) throws ClientException, ServerException,
            AwsServiceException, SdkClientException, BatchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListJobsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListJobs");

            return clientHandler.execute(new ClientExecutionParams<ListJobsRequest, ListJobsResponse>()
                    .withOperationName("ListJobs").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listJobsRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new ListJobsRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listJobsRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of AWS Batch jobs.
     * </p>
     * <p>
     * You must specify only one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * a job queue ID to return a list of jobs in that job queue
     * </p>
     * </li>
     * <li>
     * <p>
     * a multi-node parallel job ID to return a list of that job's nodes
     * </p>
     * </li>
     * <li>
     * <p>
     * an array job ID to return a list of that job's children
     * </p>
     * </li>
     * </ul>
     * <p>
     * You can filter the results by job status with the <code>jobStatus</code> parameter. If you do not specify a
     * status, only <code>RUNNING</code> jobs are returned.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listJobs(software.amazon.awssdk.services.batch.model.ListJobsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.ListJobsIterable responses = client.listJobsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.batch.paginators.ListJobsIterable responses = client.listJobsPaginator(request);
     *     for (software.amazon.awssdk.services.batch.model.ListJobsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.ListJobsIterable responses = client.listJobsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listJobs(software.amazon.awssdk.services.batch.model.ListJobsRequest)} operation.</b>
     * </p>
     *
     * @param listJobsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ClientException
     *         These errors are usually caused by a client action, such as using an action or resource on behalf of a
     *         user that doesn't have permissions to use the action or resource, or specifying an identifier that is not
     *         valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.ListJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListJobsIterable listJobsPaginator(ListJobsRequest listJobsRequest) throws ClientException, ServerException,
            AwsServiceException, SdkClientException, BatchException {
        return new ListJobsIterable(this, applyPaginatorUserAgent(listJobsRequest));
    }

    /**
     * <p>
     * Registers an AWS Batch job definition.
     * </p>
     *
     * @param registerJobDefinitionRequest
     * @return Result of the RegisterJobDefinition operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action, such as using an action or resource on behalf of a
     *         user that doesn't have permissions to use the action or resource, or specifying an identifier that is not
     *         valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.RegisterJobDefinition
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/RegisterJobDefinition" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public RegisterJobDefinitionResponse registerJobDefinition(RegisterJobDefinitionRequest registerJobDefinitionRequest)
            throws ClientException, ServerException, AwsServiceException, SdkClientException, BatchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RegisterJobDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RegisterJobDefinitionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterJobDefinition");

            return clientHandler.execute(new ClientExecutionParams<RegisterJobDefinitionRequest, RegisterJobDefinitionResponse>()
                    .withOperationName("RegisterJobDefinition").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(registerJobDefinitionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RegisterJobDefinitionRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, registerJobDefinitionRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Submits an AWS Batch job from a job definition. Parameters specified during <a>SubmitJob</a> override parameters
     * defined in the job definition.
     * </p>
     *
     * @param submitJobRequest
     * @return Result of the SubmitJob operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action, such as using an action or resource on behalf of a
     *         user that doesn't have permissions to use the action or resource, or specifying an identifier that is not
     *         valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.SubmitJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/SubmitJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SubmitJobResponse submitJob(SubmitJobRequest submitJobRequest) throws ClientException, ServerException,
            AwsServiceException, SdkClientException, BatchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SubmitJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                SubmitJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SubmitJob");

            return clientHandler.execute(new ClientExecutionParams<SubmitJobRequest, SubmitJobResponse>()
                    .withOperationName("SubmitJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(submitJobRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new SubmitJobRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, submitJobRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Terminates a job in a job queue. Jobs that are in the <code>STARTING</code> or <code>RUNNING</code> state are
     * terminated, which causes them to transition to <code>FAILED</code>. Jobs that have not progressed to the
     * <code>STARTING</code> state are cancelled.
     * </p>
     *
     * @param terminateJobRequest
     * @return Result of the TerminateJob operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action, such as using an action or resource on behalf of a
     *         user that doesn't have permissions to use the action or resource, or specifying an identifier that is not
     *         valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.TerminateJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/TerminateJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TerminateJobResponse terminateJob(TerminateJobRequest terminateJobRequest) throws ClientException, ServerException,
            AwsServiceException, SdkClientException, BatchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TerminateJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TerminateJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TerminateJob");

            return clientHandler.execute(new ClientExecutionParams<TerminateJobRequest, TerminateJobResponse>()
                    .withOperationName("TerminateJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(terminateJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TerminateJobRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, terminateJobRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an AWS Batch compute environment.
     * </p>
     *
     * @param updateComputeEnvironmentRequest
     * @return Result of the UpdateComputeEnvironment operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action, such as using an action or resource on behalf of a
     *         user that doesn't have permissions to use the action or resource, or specifying an identifier that is not
     *         valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.UpdateComputeEnvironment
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/UpdateComputeEnvironment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateComputeEnvironmentResponse updateComputeEnvironment(
            UpdateComputeEnvironmentRequest updateComputeEnvironmentRequest) throws ClientException, ServerException,
            AwsServiceException, SdkClientException, BatchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateComputeEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateComputeEnvironmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateComputeEnvironment");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateComputeEnvironmentRequest, UpdateComputeEnvironmentResponse>()
                            .withOperationName("UpdateComputeEnvironment").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateComputeEnvironmentRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateComputeEnvironmentRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateComputeEnvironmentRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a job queue.
     * </p>
     *
     * @param updateJobQueueRequest
     * @return Result of the UpdateJobQueue operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action, such as using an action or resource on behalf of a
     *         user that doesn't have permissions to use the action or resource, or specifying an identifier that is not
     *         valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.UpdateJobQueue
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/UpdateJobQueue" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateJobQueueResponse updateJobQueue(UpdateJobQueueRequest updateJobQueueRequest) throws ClientException,
            ServerException, AwsServiceException, SdkClientException, BatchException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateJobQueueResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateJobQueueResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateJobQueue");

            return clientHandler.execute(new ClientExecutionParams<UpdateJobQueueRequest, UpdateJobQueueResponse>()
                    .withOperationName("UpdateJobQueue").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateJobQueueRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateJobQueueRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateJobQueueRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(BatchException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServerException")
                                .exceptionBuilderSupplier(ServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ClientException")
                                .exceptionBuilderSupplier(ClientException::builder).httpStatusCode(400).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BatchRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
