/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.batch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object representing the details of a container that is part of a job.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ContainerDetail implements SdkPojo, Serializable, ToCopyableBuilder<ContainerDetail.Builder, ContainerDetail> {
    private static final SdkField<String> IMAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("image")
            .getter(getter(ContainerDetail::image)).setter(setter(Builder::image))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("image").build()).build();

    private static final SdkField<Integer> VCPUS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("vcpus")
            .getter(getter(ContainerDetail::vcpus)).setter(setter(Builder::vcpus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("vcpus").build()).build();

    private static final SdkField<Integer> MEMORY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("memory").getter(getter(ContainerDetail::memory)).setter(setter(Builder::memory))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("memory").build()).build();

    private static final SdkField<List<String>> COMMAND_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("command")
            .getter(getter(ContainerDetail::command))
            .setter(setter(Builder::command))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("command").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> JOB_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("jobRoleArn").getter(getter(ContainerDetail::jobRoleArn)).setter(setter(Builder::jobRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("jobRoleArn").build()).build();

    private static final SdkField<String> EXECUTION_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("executionRoleArn").getter(getter(ContainerDetail::executionRoleArn))
            .setter(setter(Builder::executionRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("executionRoleArn").build()).build();

    private static final SdkField<List<Volume>> VOLUMES_FIELD = SdkField
            .<List<Volume>> builder(MarshallingType.LIST)
            .memberName("volumes")
            .getter(getter(ContainerDetail::volumes))
            .setter(setter(Builder::volumes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("volumes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Volume> builder(MarshallingType.SDK_POJO)
                                            .constructor(Volume::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<KeyValuePair>> ENVIRONMENT_FIELD = SdkField
            .<List<KeyValuePair>> builder(MarshallingType.LIST)
            .memberName("environment")
            .getter(getter(ContainerDetail::environment))
            .setter(setter(Builder::environment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("environment").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<KeyValuePair> builder(MarshallingType.SDK_POJO)
                                            .constructor(KeyValuePair::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<MountPoint>> MOUNT_POINTS_FIELD = SdkField
            .<List<MountPoint>> builder(MarshallingType.LIST)
            .memberName("mountPoints")
            .getter(getter(ContainerDetail::mountPoints))
            .setter(setter(Builder::mountPoints))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("mountPoints").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<MountPoint> builder(MarshallingType.SDK_POJO)
                                            .constructor(MountPoint::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> READONLY_ROOT_FILESYSTEM_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("readonlyRootFilesystem").getter(getter(ContainerDetail::readonlyRootFilesystem))
            .setter(setter(Builder::readonlyRootFilesystem))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("readonlyRootFilesystem").build())
            .build();

    private static final SdkField<List<Ulimit>> ULIMITS_FIELD = SdkField
            .<List<Ulimit>> builder(MarshallingType.LIST)
            .memberName("ulimits")
            .getter(getter(ContainerDetail::ulimits))
            .setter(setter(Builder::ulimits))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ulimits").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Ulimit> builder(MarshallingType.SDK_POJO)
                                            .constructor(Ulimit::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> PRIVILEGED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("privileged").getter(getter(ContainerDetail::privileged)).setter(setter(Builder::privileged))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("privileged").build()).build();

    private static final SdkField<String> USER_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("user")
            .getter(getter(ContainerDetail::user)).setter(setter(Builder::user))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("user").build()).build();

    private static final SdkField<Integer> EXIT_CODE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("exitCode").getter(getter(ContainerDetail::exitCode)).setter(setter(Builder::exitCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("exitCode").build()).build();

    private static final SdkField<String> REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("reason")
            .getter(getter(ContainerDetail::reason)).setter(setter(Builder::reason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("reason").build()).build();

    private static final SdkField<String> CONTAINER_INSTANCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("containerInstanceArn").getter(getter(ContainerDetail::containerInstanceArn))
            .setter(setter(Builder::containerInstanceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("containerInstanceArn").build())
            .build();

    private static final SdkField<String> TASK_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("taskArn").getter(getter(ContainerDetail::taskArn)).setter(setter(Builder::taskArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("taskArn").build()).build();

    private static final SdkField<String> LOG_STREAM_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("logStreamName").getter(getter(ContainerDetail::logStreamName)).setter(setter(Builder::logStreamName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("logStreamName").build()).build();

    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("instanceType").getter(getter(ContainerDetail::instanceType)).setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instanceType").build()).build();

    private static final SdkField<List<NetworkInterface>> NETWORK_INTERFACES_FIELD = SdkField
            .<List<NetworkInterface>> builder(MarshallingType.LIST)
            .memberName("networkInterfaces")
            .getter(getter(ContainerDetail::networkInterfaces))
            .setter(setter(Builder::networkInterfaces))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("networkInterfaces").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<NetworkInterface> builder(MarshallingType.SDK_POJO)
                                            .constructor(NetworkInterface::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<ResourceRequirement>> RESOURCE_REQUIREMENTS_FIELD = SdkField
            .<List<ResourceRequirement>> builder(MarshallingType.LIST)
            .memberName("resourceRequirements")
            .getter(getter(ContainerDetail::resourceRequirements))
            .setter(setter(Builder::resourceRequirements))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceRequirements").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ResourceRequirement> builder(MarshallingType.SDK_POJO)
                                            .constructor(ResourceRequirement::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<LinuxParameters> LINUX_PARAMETERS_FIELD = SdkField
            .<LinuxParameters> builder(MarshallingType.SDK_POJO).memberName("linuxParameters")
            .getter(getter(ContainerDetail::linuxParameters)).setter(setter(Builder::linuxParameters))
            .constructor(LinuxParameters::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("linuxParameters").build()).build();

    private static final SdkField<LogConfiguration> LOG_CONFIGURATION_FIELD = SdkField
            .<LogConfiguration> builder(MarshallingType.SDK_POJO).memberName("logConfiguration")
            .getter(getter(ContainerDetail::logConfiguration)).setter(setter(Builder::logConfiguration))
            .constructor(LogConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("logConfiguration").build()).build();

    private static final SdkField<List<Secret>> SECRETS_FIELD = SdkField
            .<List<Secret>> builder(MarshallingType.LIST)
            .memberName("secrets")
            .getter(getter(ContainerDetail::secrets))
            .setter(setter(Builder::secrets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("secrets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Secret> builder(MarshallingType.SDK_POJO)
                                            .constructor(Secret::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays
            .asList(IMAGE_FIELD, VCPUS_FIELD, MEMORY_FIELD, COMMAND_FIELD, JOB_ROLE_ARN_FIELD, EXECUTION_ROLE_ARN_FIELD,
                    VOLUMES_FIELD, ENVIRONMENT_FIELD, MOUNT_POINTS_FIELD, READONLY_ROOT_FILESYSTEM_FIELD, ULIMITS_FIELD,
                    PRIVILEGED_FIELD, USER_FIELD, EXIT_CODE_FIELD, REASON_FIELD, CONTAINER_INSTANCE_ARN_FIELD, TASK_ARN_FIELD,
                    LOG_STREAM_NAME_FIELD, INSTANCE_TYPE_FIELD, NETWORK_INTERFACES_FIELD, RESOURCE_REQUIREMENTS_FIELD,
                    LINUX_PARAMETERS_FIELD, LOG_CONFIGURATION_FIELD, SECRETS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String image;

    private final Integer vcpus;

    private final Integer memory;

    private final List<String> command;

    private final String jobRoleArn;

    private final String executionRoleArn;

    private final List<Volume> volumes;

    private final List<KeyValuePair> environment;

    private final List<MountPoint> mountPoints;

    private final Boolean readonlyRootFilesystem;

    private final List<Ulimit> ulimits;

    private final Boolean privileged;

    private final String user;

    private final Integer exitCode;

    private final String reason;

    private final String containerInstanceArn;

    private final String taskArn;

    private final String logStreamName;

    private final String instanceType;

    private final List<NetworkInterface> networkInterfaces;

    private final List<ResourceRequirement> resourceRequirements;

    private final LinuxParameters linuxParameters;

    private final LogConfiguration logConfiguration;

    private final List<Secret> secrets;

    private ContainerDetail(BuilderImpl builder) {
        this.image = builder.image;
        this.vcpus = builder.vcpus;
        this.memory = builder.memory;
        this.command = builder.command;
        this.jobRoleArn = builder.jobRoleArn;
        this.executionRoleArn = builder.executionRoleArn;
        this.volumes = builder.volumes;
        this.environment = builder.environment;
        this.mountPoints = builder.mountPoints;
        this.readonlyRootFilesystem = builder.readonlyRootFilesystem;
        this.ulimits = builder.ulimits;
        this.privileged = builder.privileged;
        this.user = builder.user;
        this.exitCode = builder.exitCode;
        this.reason = builder.reason;
        this.containerInstanceArn = builder.containerInstanceArn;
        this.taskArn = builder.taskArn;
        this.logStreamName = builder.logStreamName;
        this.instanceType = builder.instanceType;
        this.networkInterfaces = builder.networkInterfaces;
        this.resourceRequirements = builder.resourceRequirements;
        this.linuxParameters = builder.linuxParameters;
        this.logConfiguration = builder.logConfiguration;
        this.secrets = builder.secrets;
    }

    /**
     * <p>
     * The image used to start the container.
     * </p>
     * 
     * @return The image used to start the container.
     */
    public String image() {
        return image;
    }

    /**
     * <p>
     * The number of VCPUs allocated for the job. This is a required parameter.
     * </p>
     * 
     * @return The number of VCPUs allocated for the job. This is a required parameter.
     */
    public Integer vcpus() {
        return vcpus;
    }

    /**
     * <p>
     * The number of MiB of memory reserved for the job. This is a required parameter.
     * </p>
     * 
     * @return The number of MiB of memory reserved for the job. This is a required parameter.
     */
    public Integer memory() {
        return memory;
    }

    /**
     * Returns true if the Command property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasCommand() {
        return command != null && !(command instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The command that is passed to the container.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasCommand()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The command that is passed to the container.
     */
    public List<String> command() {
        return command;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) associated with the job upon execution.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) associated with the job upon execution.
     */
    public String jobRoleArn() {
        return jobRoleArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the execution role that AWS Batch can assume. For more information, see <a
     * href="https://docs.aws.amazon.com/batch/latest/userguide/execution-IAM-role.html">AWS Batch execution IAM
     * role</a>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the execution role that AWS Batch can assume. For more information, see
     *         <a href="https://docs.aws.amazon.com/batch/latest/userguide/execution-IAM-role.html">AWS Batch execution
     *         IAM role</a>.
     */
    public String executionRoleArn() {
        return executionRoleArn;
    }

    /**
     * Returns true if the Volumes property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasVolumes() {
        return volumes != null && !(volumes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of volumes associated with the job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasVolumes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of volumes associated with the job.
     */
    public List<Volume> volumes() {
        return volumes;
    }

    /**
     * Returns true if the Environment property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasEnvironment() {
        return environment != null && !(environment instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The environment variables to pass to a container.
     * </p>
     * <note>
     * <p>
     * Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved for
     * variables that are set by the AWS Batch service.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasEnvironment()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The environment variables to pass to a container.</p> <note>
     *         <p>
     *         Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved for
     *         variables that are set by the AWS Batch service.
     *         </p>
     */
    public List<KeyValuePair> environment() {
        return environment;
    }

    /**
     * Returns true if the MountPoints property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasMountPoints() {
        return mountPoints != null && !(mountPoints instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The mount points for data volumes in your container.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasMountPoints()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The mount points for data volumes in your container.
     */
    public List<MountPoint> mountPoints() {
        return mountPoints;
    }

    /**
     * <p>
     * When this parameter is true, the container is given read-only access to its root file system.
     * </p>
     * 
     * @return When this parameter is true, the container is given read-only access to its root file system.
     */
    public Boolean readonlyRootFilesystem() {
        return readonlyRootFilesystem;
    }

    /**
     * Returns true if the Ulimits property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasUlimits() {
        return ulimits != null && !(ulimits instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of <code>ulimit</code> values to set in the container.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasUlimits()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of <code>ulimit</code> values to set in the container.
     */
    public List<Ulimit> ulimits() {
        return ulimits;
    }

    /**
     * <p>
     * When this parameter is true, the container is given elevated privileges on the host container instance (similar
     * to the <code>root</code> user).
     * </p>
     * 
     * @return When this parameter is true, the container is given elevated privileges on the host container instance
     *         (similar to the <code>root</code> user).
     */
    public Boolean privileged() {
        return privileged;
    }

    /**
     * <p>
     * The user name to use inside the container.
     * </p>
     * 
     * @return The user name to use inside the container.
     */
    public String user() {
        return user;
    }

    /**
     * <p>
     * The exit code to return upon completion.
     * </p>
     * 
     * @return The exit code to return upon completion.
     */
    public Integer exitCode() {
        return exitCode;
    }

    /**
     * <p>
     * A short (255 max characters) human-readable string to provide additional details about a running or stopped
     * container.
     * </p>
     * 
     * @return A short (255 max characters) human-readable string to provide additional details about a running or
     *         stopped container.
     */
    public String reason() {
        return reason;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the container instance on which the container is running.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the container instance on which the container is running.
     */
    public String containerInstanceArn() {
        return containerInstanceArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the Amazon ECS task that is associated with the container job. Each container
     * attempt receives a task ARN when they reach the <code>STARTING</code> status.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the Amazon ECS task that is associated with the container job. Each
     *         container attempt receives a task ARN when they reach the <code>STARTING</code> status.
     */
    public String taskArn() {
        return taskArn;
    }

    /**
     * <p>
     * The name of the CloudWatch Logs log stream associated with the container. The log group for AWS Batch jobs is
     * <code>/aws/batch/job</code>. Each container attempt receives a log stream name when they reach the
     * <code>RUNNING</code> status.
     * </p>
     * 
     * @return The name of the CloudWatch Logs log stream associated with the container. The log group for AWS Batch
     *         jobs is <code>/aws/batch/job</code>. Each container attempt receives a log stream name when they reach
     *         the <code>RUNNING</code> status.
     */
    public String logStreamName() {
        return logStreamName;
    }

    /**
     * <p>
     * The instance type of the underlying host infrastructure of a multi-node parallel job.
     * </p>
     * 
     * @return The instance type of the underlying host infrastructure of a multi-node parallel job.
     */
    public String instanceType() {
        return instanceType;
    }

    /**
     * Returns true if the NetworkInterfaces property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasNetworkInterfaces() {
        return networkInterfaces != null && !(networkInterfaces instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The network interfaces associated with the job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasNetworkInterfaces()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The network interfaces associated with the job.
     */
    public List<NetworkInterface> networkInterfaces() {
        return networkInterfaces;
    }

    /**
     * Returns true if the ResourceRequirements property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasResourceRequirements() {
        return resourceRequirements != null && !(resourceRequirements instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The type and amount of a resource to assign to a container. Currently, the only supported resource is
     * <code>GPU</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasResourceRequirements()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The type and amount of a resource to assign to a container. Currently, the only supported resource is
     *         <code>GPU</code>.
     */
    public List<ResourceRequirement> resourceRequirements() {
        return resourceRequirements;
    }

    /**
     * <p>
     * Linux-specific modifications that are applied to the container, such as details for device mappings.
     * </p>
     * 
     * @return Linux-specific modifications that are applied to the container, such as details for device mappings.
     */
    public LinuxParameters linuxParameters() {
        return linuxParameters;
    }

    /**
     * <p>
     * The log configuration specification for the container.
     * </p>
     * <p>
     * This parameter maps to <code>LogConfig</code> in the <a
     * href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the <a
     * href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--log-driver</code> option
     * to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>. By default, containers use the same
     * logging driver that the Docker daemon uses. However the container may use a different logging driver than the
     * Docker daemon by specifying a log driver with this parameter in the container definition. To use a different
     * logging driver for a container, the log system must be configured properly on the container instance (or on a
     * different log server for remote logging options). For more information on the options for different supported log
     * drivers, see <a href="https://docs.docker.com/engine/admin/logging/overview/">Configure logging drivers</a> in
     * the Docker documentation.
     * </p>
     * <note>
     * <p>
     * AWS Batch currently supports a subset of the logging drivers available to the Docker daemon (shown in the
     * <a>LogConfiguration</a> data type). Additional log drivers may be available in future releases of the Amazon ECS
     * container agent.
     * </p>
     * </note>
     * <p>
     * This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check the
     * Docker Remote API version on your container instance, log into your container instance and run the following
     * command: <code>sudo docker version | grep "Server API version"</code>
     * </p>
     * <note>
     * <p>
     * The Amazon ECS container agent running on a container instance must register the logging drivers available on
     * that instance with the <code>ECS_AVAILABLE_LOGGING_DRIVERS</code> environment variable before containers placed
     * on that instance can use these log configuration options. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html">Amazon ECS Container
     * Agent Configuration</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * </note>
     * 
     * @return The log configuration specification for the container.</p>
     *         <p>
     *         This parameter maps to <code>LogConfig</code> in the <a
     *         href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the
     *         <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the
     *         <code>--log-driver</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
     *         run</a>. By default, containers use the same logging driver that the Docker daemon uses. However the
     *         container may use a different logging driver than the Docker daemon by specifying a log driver with this
     *         parameter in the container definition. To use a different logging driver for a container, the log system
     *         must be configured properly on the container instance (or on a different log server for remote logging
     *         options). For more information on the options for different supported log drivers, see <a
     *         href="https://docs.docker.com/engine/admin/logging/overview/">Configure logging drivers</a> in the Docker
     *         documentation.
     *         </p>
     *         <note>
     *         <p>
     *         AWS Batch currently supports a subset of the logging drivers available to the Docker daemon (shown in the
     *         <a>LogConfiguration</a> data type). Additional log drivers may be available in future releases of the
     *         Amazon ECS container agent.
     *         </p>
     *         </note>
     *         <p>
     *         This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To
     *         check the Docker Remote API version on your container instance, log into your container instance and run
     *         the following command: <code>sudo docker version | grep "Server API version"</code>
     *         </p>
     *         <note>
     *         <p>
     *         The Amazon ECS container agent running on a container instance must register the logging drivers
     *         available on that instance with the <code>ECS_AVAILABLE_LOGGING_DRIVERS</code> environment variable
     *         before containers placed on that instance can use these log configuration options. For more information,
     *         see <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html">Amazon
     *         ECS Container Agent Configuration</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     *         </p>
     */
    public LogConfiguration logConfiguration() {
        return logConfiguration;
    }

    /**
     * Returns true if the Secrets property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasSecrets() {
        return secrets != null && !(secrets instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The secrets to pass to the container. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html">Specifying
     * Sensitive Data</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSecrets()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The secrets to pass to the container. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html"
     *         >Specifying Sensitive Data</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     */
    public List<Secret> secrets() {
        return secrets;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(image());
        hashCode = 31 * hashCode + Objects.hashCode(vcpus());
        hashCode = 31 * hashCode + Objects.hashCode(memory());
        hashCode = 31 * hashCode + Objects.hashCode(hasCommand() ? command() : null);
        hashCode = 31 * hashCode + Objects.hashCode(jobRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(executionRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasVolumes() ? volumes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasEnvironment() ? environment() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasMountPoints() ? mountPoints() : null);
        hashCode = 31 * hashCode + Objects.hashCode(readonlyRootFilesystem());
        hashCode = 31 * hashCode + Objects.hashCode(hasUlimits() ? ulimits() : null);
        hashCode = 31 * hashCode + Objects.hashCode(privileged());
        hashCode = 31 * hashCode + Objects.hashCode(user());
        hashCode = 31 * hashCode + Objects.hashCode(exitCode());
        hashCode = 31 * hashCode + Objects.hashCode(reason());
        hashCode = 31 * hashCode + Objects.hashCode(containerInstanceArn());
        hashCode = 31 * hashCode + Objects.hashCode(taskArn());
        hashCode = 31 * hashCode + Objects.hashCode(logStreamName());
        hashCode = 31 * hashCode + Objects.hashCode(instanceType());
        hashCode = 31 * hashCode + Objects.hashCode(hasNetworkInterfaces() ? networkInterfaces() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceRequirements() ? resourceRequirements() : null);
        hashCode = 31 * hashCode + Objects.hashCode(linuxParameters());
        hashCode = 31 * hashCode + Objects.hashCode(logConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(hasSecrets() ? secrets() : null);
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ContainerDetail)) {
            return false;
        }
        ContainerDetail other = (ContainerDetail) obj;
        return Objects.equals(image(), other.image()) && Objects.equals(vcpus(), other.vcpus())
                && Objects.equals(memory(), other.memory()) && hasCommand() == other.hasCommand()
                && Objects.equals(command(), other.command()) && Objects.equals(jobRoleArn(), other.jobRoleArn())
                && Objects.equals(executionRoleArn(), other.executionRoleArn()) && hasVolumes() == other.hasVolumes()
                && Objects.equals(volumes(), other.volumes()) && hasEnvironment() == other.hasEnvironment()
                && Objects.equals(environment(), other.environment()) && hasMountPoints() == other.hasMountPoints()
                && Objects.equals(mountPoints(), other.mountPoints())
                && Objects.equals(readonlyRootFilesystem(), other.readonlyRootFilesystem()) && hasUlimits() == other.hasUlimits()
                && Objects.equals(ulimits(), other.ulimits()) && Objects.equals(privileged(), other.privileged())
                && Objects.equals(user(), other.user()) && Objects.equals(exitCode(), other.exitCode())
                && Objects.equals(reason(), other.reason())
                && Objects.equals(containerInstanceArn(), other.containerInstanceArn())
                && Objects.equals(taskArn(), other.taskArn()) && Objects.equals(logStreamName(), other.logStreamName())
                && Objects.equals(instanceType(), other.instanceType()) && hasNetworkInterfaces() == other.hasNetworkInterfaces()
                && Objects.equals(networkInterfaces(), other.networkInterfaces())
                && hasResourceRequirements() == other.hasResourceRequirements()
                && Objects.equals(resourceRequirements(), other.resourceRequirements())
                && Objects.equals(linuxParameters(), other.linuxParameters())
                && Objects.equals(logConfiguration(), other.logConfiguration()) && hasSecrets() == other.hasSecrets()
                && Objects.equals(secrets(), other.secrets());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ContainerDetail").add("Image", image()).add("Vcpus", vcpus()).add("Memory", memory())
                .add("Command", hasCommand() ? command() : null).add("JobRoleArn", jobRoleArn())
                .add("ExecutionRoleArn", executionRoleArn()).add("Volumes", hasVolumes() ? volumes() : null)
                .add("Environment", hasEnvironment() ? environment() : null)
                .add("MountPoints", hasMountPoints() ? mountPoints() : null)
                .add("ReadonlyRootFilesystem", readonlyRootFilesystem()).add("Ulimits", hasUlimits() ? ulimits() : null)
                .add("Privileged", privileged()).add("User", user()).add("ExitCode", exitCode()).add("Reason", reason())
                .add("ContainerInstanceArn", containerInstanceArn()).add("TaskArn", taskArn())
                .add("LogStreamName", logStreamName()).add("InstanceType", instanceType())
                .add("NetworkInterfaces", hasNetworkInterfaces() ? networkInterfaces() : null)
                .add("ResourceRequirements", hasResourceRequirements() ? resourceRequirements() : null)
                .add("LinuxParameters", linuxParameters()).add("LogConfiguration", logConfiguration())
                .add("Secrets", hasSecrets() ? secrets() : null).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "image":
            return Optional.ofNullable(clazz.cast(image()));
        case "vcpus":
            return Optional.ofNullable(clazz.cast(vcpus()));
        case "memory":
            return Optional.ofNullable(clazz.cast(memory()));
        case "command":
            return Optional.ofNullable(clazz.cast(command()));
        case "jobRoleArn":
            return Optional.ofNullable(clazz.cast(jobRoleArn()));
        case "executionRoleArn":
            return Optional.ofNullable(clazz.cast(executionRoleArn()));
        case "volumes":
            return Optional.ofNullable(clazz.cast(volumes()));
        case "environment":
            return Optional.ofNullable(clazz.cast(environment()));
        case "mountPoints":
            return Optional.ofNullable(clazz.cast(mountPoints()));
        case "readonlyRootFilesystem":
            return Optional.ofNullable(clazz.cast(readonlyRootFilesystem()));
        case "ulimits":
            return Optional.ofNullable(clazz.cast(ulimits()));
        case "privileged":
            return Optional.ofNullable(clazz.cast(privileged()));
        case "user":
            return Optional.ofNullable(clazz.cast(user()));
        case "exitCode":
            return Optional.ofNullable(clazz.cast(exitCode()));
        case "reason":
            return Optional.ofNullable(clazz.cast(reason()));
        case "containerInstanceArn":
            return Optional.ofNullable(clazz.cast(containerInstanceArn()));
        case "taskArn":
            return Optional.ofNullable(clazz.cast(taskArn()));
        case "logStreamName":
            return Optional.ofNullable(clazz.cast(logStreamName()));
        case "instanceType":
            return Optional.ofNullable(clazz.cast(instanceType()));
        case "networkInterfaces":
            return Optional.ofNullable(clazz.cast(networkInterfaces()));
        case "resourceRequirements":
            return Optional.ofNullable(clazz.cast(resourceRequirements()));
        case "linuxParameters":
            return Optional.ofNullable(clazz.cast(linuxParameters()));
        case "logConfiguration":
            return Optional.ofNullable(clazz.cast(logConfiguration()));
        case "secrets":
            return Optional.ofNullable(clazz.cast(secrets()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ContainerDetail, T> g) {
        return obj -> g.apply((ContainerDetail) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ContainerDetail> {
        /**
         * <p>
         * The image used to start the container.
         * </p>
         * 
         * @param image
         *        The image used to start the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder image(String image);

        /**
         * <p>
         * The number of VCPUs allocated for the job. This is a required parameter.
         * </p>
         * 
         * @param vcpus
         *        The number of VCPUs allocated for the job. This is a required parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vcpus(Integer vcpus);

        /**
         * <p>
         * The number of MiB of memory reserved for the job. This is a required parameter.
         * </p>
         * 
         * @param memory
         *        The number of MiB of memory reserved for the job. This is a required parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder memory(Integer memory);

        /**
         * <p>
         * The command that is passed to the container.
         * </p>
         * 
         * @param command
         *        The command that is passed to the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(Collection<String> command);

        /**
         * <p>
         * The command that is passed to the container.
         * </p>
         * 
         * @param command
         *        The command that is passed to the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(String... command);

        /**
         * <p>
         * The Amazon Resource Name (ARN) associated with the job upon execution.
         * </p>
         * 
         * @param jobRoleArn
         *        The Amazon Resource Name (ARN) associated with the job upon execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobRoleArn(String jobRoleArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the execution role that AWS Batch can assume. For more information, see <a
         * href="https://docs.aws.amazon.com/batch/latest/userguide/execution-IAM-role.html">AWS Batch execution IAM
         * role</a>.
         * </p>
         * 
         * @param executionRoleArn
         *        The Amazon Resource Name (ARN) of the execution role that AWS Batch can assume. For more information,
         *        see <a href="https://docs.aws.amazon.com/batch/latest/userguide/execution-IAM-role.html">AWS Batch
         *        execution IAM role</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionRoleArn(String executionRoleArn);

        /**
         * <p>
         * A list of volumes associated with the job.
         * </p>
         * 
         * @param volumes
         *        A list of volumes associated with the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumes(Collection<Volume> volumes);

        /**
         * <p>
         * A list of volumes associated with the job.
         * </p>
         * 
         * @param volumes
         *        A list of volumes associated with the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumes(Volume... volumes);

        /**
         * <p>
         * A list of volumes associated with the job.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Volume>.Builder} avoiding the need to
         * create one manually via {@link List<Volume>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Volume>.Builder#build()} is called immediately and its
         * result is passed to {@link #volumes(List<Volume>)}.
         * 
         * @param volumes
         *        a consumer that will call methods on {@link List<Volume>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #volumes(List<Volume>)
         */
        Builder volumes(Consumer<Volume.Builder>... volumes);

        /**
         * <p>
         * The environment variables to pass to a container.
         * </p>
         * <note>
         * <p>
         * Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved for
         * variables that are set by the AWS Batch service.
         * </p>
         * </note>
         * 
         * @param environment
         *        The environment variables to pass to a container.</p> <note>
         *        <p>
         *        Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved
         *        for variables that are set by the AWS Batch service.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environment(Collection<KeyValuePair> environment);

        /**
         * <p>
         * The environment variables to pass to a container.
         * </p>
         * <note>
         * <p>
         * Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved for
         * variables that are set by the AWS Batch service.
         * </p>
         * </note>
         * 
         * @param environment
         *        The environment variables to pass to a container.</p> <note>
         *        <p>
         *        Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved
         *        for variables that are set by the AWS Batch service.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environment(KeyValuePair... environment);

        /**
         * <p>
         * The environment variables to pass to a container.
         * </p>
         * <note>
         * <p>
         * Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved for
         * variables that are set by the AWS Batch service.
         * </p>
         * </note> This is a convenience that creates an instance of the {@link List<KeyValuePair>.Builder} avoiding the
         * need to create one manually via {@link List<KeyValuePair>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<KeyValuePair>.Builder#build()} is called immediately and its
         * result is passed to {@link #environment(List<KeyValuePair>)}.
         * 
         * @param environment
         *        a consumer that will call methods on {@link List<KeyValuePair>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #environment(List<KeyValuePair>)
         */
        Builder environment(Consumer<KeyValuePair.Builder>... environment);

        /**
         * <p>
         * The mount points for data volumes in your container.
         * </p>
         * 
         * @param mountPoints
         *        The mount points for data volumes in your container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mountPoints(Collection<MountPoint> mountPoints);

        /**
         * <p>
         * The mount points for data volumes in your container.
         * </p>
         * 
         * @param mountPoints
         *        The mount points for data volumes in your container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mountPoints(MountPoint... mountPoints);

        /**
         * <p>
         * The mount points for data volumes in your container.
         * </p>
         * This is a convenience that creates an instance of the {@link List<MountPoint>.Builder} avoiding the need to
         * create one manually via {@link List<MountPoint>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<MountPoint>.Builder#build()} is called immediately and its
         * result is passed to {@link #mountPoints(List<MountPoint>)}.
         * 
         * @param mountPoints
         *        a consumer that will call methods on {@link List<MountPoint>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #mountPoints(List<MountPoint>)
         */
        Builder mountPoints(Consumer<MountPoint.Builder>... mountPoints);

        /**
         * <p>
         * When this parameter is true, the container is given read-only access to its root file system.
         * </p>
         * 
         * @param readonlyRootFilesystem
         *        When this parameter is true, the container is given read-only access to its root file system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder readonlyRootFilesystem(Boolean readonlyRootFilesystem);

        /**
         * <p>
         * A list of <code>ulimit</code> values to set in the container.
         * </p>
         * 
         * @param ulimits
         *        A list of <code>ulimit</code> values to set in the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ulimits(Collection<Ulimit> ulimits);

        /**
         * <p>
         * A list of <code>ulimit</code> values to set in the container.
         * </p>
         * 
         * @param ulimits
         *        A list of <code>ulimit</code> values to set in the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ulimits(Ulimit... ulimits);

        /**
         * <p>
         * A list of <code>ulimit</code> values to set in the container.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Ulimit>.Builder} avoiding the need to
         * create one manually via {@link List<Ulimit>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Ulimit>.Builder#build()} is called immediately and its
         * result is passed to {@link #ulimits(List<Ulimit>)}.
         * 
         * @param ulimits
         *        a consumer that will call methods on {@link List<Ulimit>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ulimits(List<Ulimit>)
         */
        Builder ulimits(Consumer<Ulimit.Builder>... ulimits);

        /**
         * <p>
         * When this parameter is true, the container is given elevated privileges on the host container instance
         * (similar to the <code>root</code> user).
         * </p>
         * 
         * @param privileged
         *        When this parameter is true, the container is given elevated privileges on the host container instance
         *        (similar to the <code>root</code> user).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder privileged(Boolean privileged);

        /**
         * <p>
         * The user name to use inside the container.
         * </p>
         * 
         * @param user
         *        The user name to use inside the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder user(String user);

        /**
         * <p>
         * The exit code to return upon completion.
         * </p>
         * 
         * @param exitCode
         *        The exit code to return upon completion.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exitCode(Integer exitCode);

        /**
         * <p>
         * A short (255 max characters) human-readable string to provide additional details about a running or stopped
         * container.
         * </p>
         * 
         * @param reason
         *        A short (255 max characters) human-readable string to provide additional details about a running or
         *        stopped container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reason(String reason);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the container instance on which the container is running.
         * </p>
         * 
         * @param containerInstanceArn
         *        The Amazon Resource Name (ARN) of the container instance on which the container is running.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerInstanceArn(String containerInstanceArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the Amazon ECS task that is associated with the container job. Each
         * container attempt receives a task ARN when they reach the <code>STARTING</code> status.
         * </p>
         * 
         * @param taskArn
         *        The Amazon Resource Name (ARN) of the Amazon ECS task that is associated with the container job. Each
         *        container attempt receives a task ARN when they reach the <code>STARTING</code> status.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskArn(String taskArn);

        /**
         * <p>
         * The name of the CloudWatch Logs log stream associated with the container. The log group for AWS Batch jobs is
         * <code>/aws/batch/job</code>. Each container attempt receives a log stream name when they reach the
         * <code>RUNNING</code> status.
         * </p>
         * 
         * @param logStreamName
         *        The name of the CloudWatch Logs log stream associated with the container. The log group for AWS Batch
         *        jobs is <code>/aws/batch/job</code>. Each container attempt receives a log stream name when they reach
         *        the <code>RUNNING</code> status.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logStreamName(String logStreamName);

        /**
         * <p>
         * The instance type of the underlying host infrastructure of a multi-node parallel job.
         * </p>
         * 
         * @param instanceType
         *        The instance type of the underlying host infrastructure of a multi-node parallel job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * The network interfaces associated with the job.
         * </p>
         * 
         * @param networkInterfaces
         *        The network interfaces associated with the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaces(Collection<NetworkInterface> networkInterfaces);

        /**
         * <p>
         * The network interfaces associated with the job.
         * </p>
         * 
         * @param networkInterfaces
         *        The network interfaces associated with the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaces(NetworkInterface... networkInterfaces);

        /**
         * <p>
         * The network interfaces associated with the job.
         * </p>
         * This is a convenience that creates an instance of the {@link List<NetworkInterface>.Builder} avoiding the
         * need to create one manually via {@link List<NetworkInterface>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<NetworkInterface>.Builder#build()} is called immediately and
         * its result is passed to {@link #networkInterfaces(List<NetworkInterface>)}.
         * 
         * @param networkInterfaces
         *        a consumer that will call methods on {@link List<NetworkInterface>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #networkInterfaces(List<NetworkInterface>)
         */
        Builder networkInterfaces(Consumer<NetworkInterface.Builder>... networkInterfaces);

        /**
         * <p>
         * The type and amount of a resource to assign to a container. Currently, the only supported resource is
         * <code>GPU</code>.
         * </p>
         * 
         * @param resourceRequirements
         *        The type and amount of a resource to assign to a container. Currently, the only supported resource is
         *        <code>GPU</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceRequirements(Collection<ResourceRequirement> resourceRequirements);

        /**
         * <p>
         * The type and amount of a resource to assign to a container. Currently, the only supported resource is
         * <code>GPU</code>.
         * </p>
         * 
         * @param resourceRequirements
         *        The type and amount of a resource to assign to a container. Currently, the only supported resource is
         *        <code>GPU</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceRequirements(ResourceRequirement... resourceRequirements);

        /**
         * <p>
         * The type and amount of a resource to assign to a container. Currently, the only supported resource is
         * <code>GPU</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ResourceRequirement>.Builder} avoiding the
         * need to create one manually via {@link List<ResourceRequirement>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ResourceRequirement>.Builder#build()} is called immediately
         * and its result is passed to {@link #resourceRequirements(List<ResourceRequirement>)}.
         * 
         * @param resourceRequirements
         *        a consumer that will call methods on {@link List<ResourceRequirement>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resourceRequirements(List<ResourceRequirement>)
         */
        Builder resourceRequirements(Consumer<ResourceRequirement.Builder>... resourceRequirements);

        /**
         * <p>
         * Linux-specific modifications that are applied to the container, such as details for device mappings.
         * </p>
         * 
         * @param linuxParameters
         *        Linux-specific modifications that are applied to the container, such as details for device mappings.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder linuxParameters(LinuxParameters linuxParameters);

        /**
         * <p>
         * Linux-specific modifications that are applied to the container, such as details for device mappings.
         * </p>
         * This is a convenience that creates an instance of the {@link LinuxParameters.Builder} avoiding the need to
         * create one manually via {@link LinuxParameters#builder()}.
         *
         * When the {@link Consumer} completes, {@link LinuxParameters.Builder#build()} is called immediately and its
         * result is passed to {@link #linuxParameters(LinuxParameters)}.
         * 
         * @param linuxParameters
         *        a consumer that will call methods on {@link LinuxParameters.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #linuxParameters(LinuxParameters)
         */
        default Builder linuxParameters(Consumer<LinuxParameters.Builder> linuxParameters) {
            return linuxParameters(LinuxParameters.builder().applyMutation(linuxParameters).build());
        }

        /**
         * <p>
         * The log configuration specification for the container.
         * </p>
         * <p>
         * This parameter maps to <code>LogConfig</code> in the <a
         * href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the <a
         * href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--log-driver</code>
         * option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>. By default, containers use
         * the same logging driver that the Docker daemon uses. However the container may use a different logging driver
         * than the Docker daemon by specifying a log driver with this parameter in the container definition. To use a
         * different logging driver for a container, the log system must be configured properly on the container
         * instance (or on a different log server for remote logging options). For more information on the options for
         * different supported log drivers, see <a
         * href="https://docs.docker.com/engine/admin/logging/overview/">Configure logging drivers</a> in the Docker
         * documentation.
         * </p>
         * <note>
         * <p>
         * AWS Batch currently supports a subset of the logging drivers available to the Docker daemon (shown in the
         * <a>LogConfiguration</a> data type). Additional log drivers may be available in future releases of the Amazon
         * ECS container agent.
         * </p>
         * </note>
         * <p>
         * This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check
         * the Docker Remote API version on your container instance, log into your container instance and run the
         * following command: <code>sudo docker version | grep "Server API version"</code>
         * </p>
         * <note>
         * <p>
         * The Amazon ECS container agent running on a container instance must register the logging drivers available on
         * that instance with the <code>ECS_AVAILABLE_LOGGING_DRIVERS</code> environment variable before containers
         * placed on that instance can use these log configuration options. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html">Amazon ECS Container
         * Agent Configuration</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * </note>
         * 
         * @param logConfiguration
         *        The log configuration specification for the container.</p>
         *        <p>
         *        This parameter maps to <code>LogConfig</code> in the <a
         *        href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of
         *        the <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the
         *        <code>--log-driver</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
         *        run</a>. By default, containers use the same logging driver that the Docker daemon uses. However the
         *        container may use a different logging driver than the Docker daemon by specifying a log driver with
         *        this parameter in the container definition. To use a different logging driver for a container, the log
         *        system must be configured properly on the container instance (or on a different log server for remote
         *        logging options). For more information on the options for different supported log drivers, see <a
         *        href="https://docs.docker.com/engine/admin/logging/overview/">Configure logging drivers</a> in the
         *        Docker documentation.
         *        </p>
         *        <note>
         *        <p>
         *        AWS Batch currently supports a subset of the logging drivers available to the Docker daemon (shown in
         *        the <a>LogConfiguration</a> data type). Additional log drivers may be available in future releases of
         *        the Amazon ECS container agent.
         *        </p>
         *        </note>
         *        <p>
         *        This parameter requires version 1.18 of the Docker Remote API or greater on your container instance.
         *        To check the Docker Remote API version on your container instance, log into your container instance
         *        and run the following command: <code>sudo docker version | grep "Server API version"</code>
         *        </p>
         *        <note>
         *        <p>
         *        The Amazon ECS container agent running on a container instance must register the logging drivers
         *        available on that instance with the <code>ECS_AVAILABLE_LOGGING_DRIVERS</code> environment variable
         *        before containers placed on that instance can use these log configuration options. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html">Amazon ECS
         *        Container Agent Configuration</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logConfiguration(LogConfiguration logConfiguration);

        /**
         * <p>
         * The log configuration specification for the container.
         * </p>
         * <p>
         * This parameter maps to <code>LogConfig</code> in the <a
         * href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the <a
         * href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--log-driver</code>
         * option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>. By default, containers use
         * the same logging driver that the Docker daemon uses. However the container may use a different logging driver
         * than the Docker daemon by specifying a log driver with this parameter in the container definition. To use a
         * different logging driver for a container, the log system must be configured properly on the container
         * instance (or on a different log server for remote logging options). For more information on the options for
         * different supported log drivers, see <a
         * href="https://docs.docker.com/engine/admin/logging/overview/">Configure logging drivers</a> in the Docker
         * documentation.
         * </p>
         * <note>
         * <p>
         * AWS Batch currently supports a subset of the logging drivers available to the Docker daemon (shown in the
         * <a>LogConfiguration</a> data type). Additional log drivers may be available in future releases of the Amazon
         * ECS container agent.
         * </p>
         * </note>
         * <p>
         * This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check
         * the Docker Remote API version on your container instance, log into your container instance and run the
         * following command: <code>sudo docker version | grep "Server API version"</code>
         * </p>
         * <note>
         * <p>
         * The Amazon ECS container agent running on a container instance must register the logging drivers available on
         * that instance with the <code>ECS_AVAILABLE_LOGGING_DRIVERS</code> environment variable before containers
         * placed on that instance can use these log configuration options. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html">Amazon ECS Container
         * Agent Configuration</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * </note> This is a convenience that creates an instance of the {@link LogConfiguration.Builder} avoiding the
         * need to create one manually via {@link LogConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link LogConfiguration.Builder#build()} is called immediately and its
         * result is passed to {@link #logConfiguration(LogConfiguration)}.
         * 
         * @param logConfiguration
         *        a consumer that will call methods on {@link LogConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #logConfiguration(LogConfiguration)
         */
        default Builder logConfiguration(Consumer<LogConfiguration.Builder> logConfiguration) {
            return logConfiguration(LogConfiguration.builder().applyMutation(logConfiguration).build());
        }

        /**
         * <p>
         * The secrets to pass to the container. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html">Specifying
         * Sensitive Data</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param secrets
         *        The secrets to pass to the container. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html"
         *        >Specifying Sensitive Data</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secrets(Collection<Secret> secrets);

        /**
         * <p>
         * The secrets to pass to the container. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html">Specifying
         * Sensitive Data</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param secrets
         *        The secrets to pass to the container. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html"
         *        >Specifying Sensitive Data</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secrets(Secret... secrets);

        /**
         * <p>
         * The secrets to pass to the container. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html">Specifying
         * Sensitive Data</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Secret>.Builder} avoiding the need to
         * create one manually via {@link List<Secret>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Secret>.Builder#build()} is called immediately and its
         * result is passed to {@link #secrets(List<Secret>)}.
         * 
         * @param secrets
         *        a consumer that will call methods on {@link List<Secret>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #secrets(List<Secret>)
         */
        Builder secrets(Consumer<Secret.Builder>... secrets);
    }

    static final class BuilderImpl implements Builder {
        private String image;

        private Integer vcpus;

        private Integer memory;

        private List<String> command = DefaultSdkAutoConstructList.getInstance();

        private String jobRoleArn;

        private String executionRoleArn;

        private List<Volume> volumes = DefaultSdkAutoConstructList.getInstance();

        private List<KeyValuePair> environment = DefaultSdkAutoConstructList.getInstance();

        private List<MountPoint> mountPoints = DefaultSdkAutoConstructList.getInstance();

        private Boolean readonlyRootFilesystem;

        private List<Ulimit> ulimits = DefaultSdkAutoConstructList.getInstance();

        private Boolean privileged;

        private String user;

        private Integer exitCode;

        private String reason;

        private String containerInstanceArn;

        private String taskArn;

        private String logStreamName;

        private String instanceType;

        private List<NetworkInterface> networkInterfaces = DefaultSdkAutoConstructList.getInstance();

        private List<ResourceRequirement> resourceRequirements = DefaultSdkAutoConstructList.getInstance();

        private LinuxParameters linuxParameters;

        private LogConfiguration logConfiguration;

        private List<Secret> secrets = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ContainerDetail model) {
            image(model.image);
            vcpus(model.vcpus);
            memory(model.memory);
            command(model.command);
            jobRoleArn(model.jobRoleArn);
            executionRoleArn(model.executionRoleArn);
            volumes(model.volumes);
            environment(model.environment);
            mountPoints(model.mountPoints);
            readonlyRootFilesystem(model.readonlyRootFilesystem);
            ulimits(model.ulimits);
            privileged(model.privileged);
            user(model.user);
            exitCode(model.exitCode);
            reason(model.reason);
            containerInstanceArn(model.containerInstanceArn);
            taskArn(model.taskArn);
            logStreamName(model.logStreamName);
            instanceType(model.instanceType);
            networkInterfaces(model.networkInterfaces);
            resourceRequirements(model.resourceRequirements);
            linuxParameters(model.linuxParameters);
            logConfiguration(model.logConfiguration);
            secrets(model.secrets);
        }

        public final String getImage() {
            return image;
        }

        @Override
        public final Builder image(String image) {
            this.image = image;
            return this;
        }

        public final void setImage(String image) {
            this.image = image;
        }

        public final Integer getVcpus() {
            return vcpus;
        }

        @Override
        public final Builder vcpus(Integer vcpus) {
            this.vcpus = vcpus;
            return this;
        }

        public final void setVcpus(Integer vcpus) {
            this.vcpus = vcpus;
        }

        public final Integer getMemory() {
            return memory;
        }

        @Override
        public final Builder memory(Integer memory) {
            this.memory = memory;
            return this;
        }

        public final void setMemory(Integer memory) {
            this.memory = memory;
        }

        public final Collection<String> getCommand() {
            if (command instanceof SdkAutoConstructList) {
                return null;
            }
            return command;
        }

        @Override
        public final Builder command(Collection<String> command) {
            this.command = StringListCopier.copy(command);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder command(String... command) {
            command(Arrays.asList(command));
            return this;
        }

        public final void setCommand(Collection<String> command) {
            this.command = StringListCopier.copy(command);
        }

        public final String getJobRoleArn() {
            return jobRoleArn;
        }

        @Override
        public final Builder jobRoleArn(String jobRoleArn) {
            this.jobRoleArn = jobRoleArn;
            return this;
        }

        public final void setJobRoleArn(String jobRoleArn) {
            this.jobRoleArn = jobRoleArn;
        }

        public final String getExecutionRoleArn() {
            return executionRoleArn;
        }

        @Override
        public final Builder executionRoleArn(String executionRoleArn) {
            this.executionRoleArn = executionRoleArn;
            return this;
        }

        public final void setExecutionRoleArn(String executionRoleArn) {
            this.executionRoleArn = executionRoleArn;
        }

        public final Collection<Volume.Builder> getVolumes() {
            if (volumes instanceof SdkAutoConstructList) {
                return null;
            }
            return volumes != null ? volumes.stream().map(Volume::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder volumes(Collection<Volume> volumes) {
            this.volumes = VolumesCopier.copy(volumes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder volumes(Volume... volumes) {
            volumes(Arrays.asList(volumes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder volumes(Consumer<Volume.Builder>... volumes) {
            volumes(Stream.of(volumes).map(c -> Volume.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setVolumes(Collection<Volume.BuilderImpl> volumes) {
            this.volumes = VolumesCopier.copyFromBuilder(volumes);
        }

        public final Collection<KeyValuePair.Builder> getEnvironment() {
            if (environment instanceof SdkAutoConstructList) {
                return null;
            }
            return environment != null ? environment.stream().map(KeyValuePair::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder environment(Collection<KeyValuePair> environment) {
            this.environment = EnvironmentVariablesCopier.copy(environment);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder environment(KeyValuePair... environment) {
            environment(Arrays.asList(environment));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder environment(Consumer<KeyValuePair.Builder>... environment) {
            environment(Stream.of(environment).map(c -> KeyValuePair.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setEnvironment(Collection<KeyValuePair.BuilderImpl> environment) {
            this.environment = EnvironmentVariablesCopier.copyFromBuilder(environment);
        }

        public final Collection<MountPoint.Builder> getMountPoints() {
            if (mountPoints instanceof SdkAutoConstructList) {
                return null;
            }
            return mountPoints != null ? mountPoints.stream().map(MountPoint::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder mountPoints(Collection<MountPoint> mountPoints) {
            this.mountPoints = MountPointsCopier.copy(mountPoints);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder mountPoints(MountPoint... mountPoints) {
            mountPoints(Arrays.asList(mountPoints));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder mountPoints(Consumer<MountPoint.Builder>... mountPoints) {
            mountPoints(Stream.of(mountPoints).map(c -> MountPoint.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setMountPoints(Collection<MountPoint.BuilderImpl> mountPoints) {
            this.mountPoints = MountPointsCopier.copyFromBuilder(mountPoints);
        }

        public final Boolean getReadonlyRootFilesystem() {
            return readonlyRootFilesystem;
        }

        @Override
        public final Builder readonlyRootFilesystem(Boolean readonlyRootFilesystem) {
            this.readonlyRootFilesystem = readonlyRootFilesystem;
            return this;
        }

        public final void setReadonlyRootFilesystem(Boolean readonlyRootFilesystem) {
            this.readonlyRootFilesystem = readonlyRootFilesystem;
        }

        public final Collection<Ulimit.Builder> getUlimits() {
            if (ulimits instanceof SdkAutoConstructList) {
                return null;
            }
            return ulimits != null ? ulimits.stream().map(Ulimit::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder ulimits(Collection<Ulimit> ulimits) {
            this.ulimits = UlimitsCopier.copy(ulimits);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ulimits(Ulimit... ulimits) {
            ulimits(Arrays.asList(ulimits));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ulimits(Consumer<Ulimit.Builder>... ulimits) {
            ulimits(Stream.of(ulimits).map(c -> Ulimit.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setUlimits(Collection<Ulimit.BuilderImpl> ulimits) {
            this.ulimits = UlimitsCopier.copyFromBuilder(ulimits);
        }

        public final Boolean getPrivileged() {
            return privileged;
        }

        @Override
        public final Builder privileged(Boolean privileged) {
            this.privileged = privileged;
            return this;
        }

        public final void setPrivileged(Boolean privileged) {
            this.privileged = privileged;
        }

        public final String getUser() {
            return user;
        }

        @Override
        public final Builder user(String user) {
            this.user = user;
            return this;
        }

        public final void setUser(String user) {
            this.user = user;
        }

        public final Integer getExitCode() {
            return exitCode;
        }

        @Override
        public final Builder exitCode(Integer exitCode) {
            this.exitCode = exitCode;
            return this;
        }

        public final void setExitCode(Integer exitCode) {
            this.exitCode = exitCode;
        }

        public final String getReason() {
            return reason;
        }

        @Override
        public final Builder reason(String reason) {
            this.reason = reason;
            return this;
        }

        public final void setReason(String reason) {
            this.reason = reason;
        }

        public final String getContainerInstanceArn() {
            return containerInstanceArn;
        }

        @Override
        public final Builder containerInstanceArn(String containerInstanceArn) {
            this.containerInstanceArn = containerInstanceArn;
            return this;
        }

        public final void setContainerInstanceArn(String containerInstanceArn) {
            this.containerInstanceArn = containerInstanceArn;
        }

        public final String getTaskArn() {
            return taskArn;
        }

        @Override
        public final Builder taskArn(String taskArn) {
            this.taskArn = taskArn;
            return this;
        }

        public final void setTaskArn(String taskArn) {
            this.taskArn = taskArn;
        }

        public final String getLogStreamName() {
            return logStreamName;
        }

        @Override
        public final Builder logStreamName(String logStreamName) {
            this.logStreamName = logStreamName;
            return this;
        }

        public final void setLogStreamName(String logStreamName) {
            this.logStreamName = logStreamName;
        }

        public final String getInstanceType() {
            return instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        public final Collection<NetworkInterface.Builder> getNetworkInterfaces() {
            if (networkInterfaces instanceof SdkAutoConstructList) {
                return null;
            }
            return networkInterfaces != null ? networkInterfaces.stream().map(NetworkInterface::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder networkInterfaces(Collection<NetworkInterface> networkInterfaces) {
            this.networkInterfaces = NetworkInterfaceListCopier.copy(networkInterfaces);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder networkInterfaces(NetworkInterface... networkInterfaces) {
            networkInterfaces(Arrays.asList(networkInterfaces));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder networkInterfaces(Consumer<NetworkInterface.Builder>... networkInterfaces) {
            networkInterfaces(Stream.of(networkInterfaces).map(c -> NetworkInterface.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setNetworkInterfaces(Collection<NetworkInterface.BuilderImpl> networkInterfaces) {
            this.networkInterfaces = NetworkInterfaceListCopier.copyFromBuilder(networkInterfaces);
        }

        public final Collection<ResourceRequirement.Builder> getResourceRequirements() {
            if (resourceRequirements instanceof SdkAutoConstructList) {
                return null;
            }
            return resourceRequirements != null ? resourceRequirements.stream().map(ResourceRequirement::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder resourceRequirements(Collection<ResourceRequirement> resourceRequirements) {
            this.resourceRequirements = ResourceRequirementsCopier.copy(resourceRequirements);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceRequirements(ResourceRequirement... resourceRequirements) {
            resourceRequirements(Arrays.asList(resourceRequirements));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceRequirements(Consumer<ResourceRequirement.Builder>... resourceRequirements) {
            resourceRequirements(Stream.of(resourceRequirements).map(c -> ResourceRequirement.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setResourceRequirements(Collection<ResourceRequirement.BuilderImpl> resourceRequirements) {
            this.resourceRequirements = ResourceRequirementsCopier.copyFromBuilder(resourceRequirements);
        }

        public final LinuxParameters.Builder getLinuxParameters() {
            return linuxParameters != null ? linuxParameters.toBuilder() : null;
        }

        @Override
        public final Builder linuxParameters(LinuxParameters linuxParameters) {
            this.linuxParameters = linuxParameters;
            return this;
        }

        public final void setLinuxParameters(LinuxParameters.BuilderImpl linuxParameters) {
            this.linuxParameters = linuxParameters != null ? linuxParameters.build() : null;
        }

        public final LogConfiguration.Builder getLogConfiguration() {
            return logConfiguration != null ? logConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder logConfiguration(LogConfiguration logConfiguration) {
            this.logConfiguration = logConfiguration;
            return this;
        }

        public final void setLogConfiguration(LogConfiguration.BuilderImpl logConfiguration) {
            this.logConfiguration = logConfiguration != null ? logConfiguration.build() : null;
        }

        public final Collection<Secret.Builder> getSecrets() {
            if (secrets instanceof SdkAutoConstructList) {
                return null;
            }
            return secrets != null ? secrets.stream().map(Secret::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder secrets(Collection<Secret> secrets) {
            this.secrets = SecretListCopier.copy(secrets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder secrets(Secret... secrets) {
            secrets(Arrays.asList(secrets));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder secrets(Consumer<Secret.Builder>... secrets) {
            secrets(Stream.of(secrets).map(c -> Secret.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setSecrets(Collection<Secret.BuilderImpl> secrets) {
            this.secrets = SecretListCopier.copyFromBuilder(secrets);
        }

        @Override
        public ContainerDetail build() {
            return new ContainerDetail(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
