/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.batch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object representing an AWS Batch compute resource.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ComputeResource implements SdkPojo, Serializable, ToCopyableBuilder<ComputeResource.Builder, ComputeResource> {
    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(ComputeResource::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<String> ALLOCATION_STRATEGY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("allocationStrategy").getter(getter(ComputeResource::allocationStrategyAsString))
            .setter(setter(Builder::allocationStrategy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("allocationStrategy").build())
            .build();

    private static final SdkField<Integer> MINV_CPUS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("minvCpus").getter(getter(ComputeResource::minvCpus)).setter(setter(Builder::minvCpus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("minvCpus").build()).build();

    private static final SdkField<Integer> MAXV_CPUS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("maxvCpus").getter(getter(ComputeResource::maxvCpus)).setter(setter(Builder::maxvCpus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maxvCpus").build()).build();

    private static final SdkField<Integer> DESIREDV_CPUS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("desiredvCpus").getter(getter(ComputeResource::desiredvCpus)).setter(setter(Builder::desiredvCpus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("desiredvCpus").build()).build();

    private static final SdkField<List<String>> INSTANCE_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("instanceTypes")
            .getter(getter(ComputeResource::instanceTypes))
            .setter(setter(Builder::instanceTypes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instanceTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> IMAGE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("imageId").getter(getter(ComputeResource::imageId)).setter(setter(Builder::imageId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("imageId").build()).build();

    private static final SdkField<List<String>> SUBNETS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("subnets")
            .getter(getter(ComputeResource::subnets))
            .setter(setter(Builder::subnets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("subnets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> SECURITY_GROUP_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("securityGroupIds")
            .getter(getter(ComputeResource::securityGroupIds))
            .setter(setter(Builder::securityGroupIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("securityGroupIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> EC2_KEY_PAIR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ec2KeyPair").getter(getter(ComputeResource::ec2KeyPair)).setter(setter(Builder::ec2KeyPair))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ec2KeyPair").build()).build();

    private static final SdkField<String> INSTANCE_ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("instanceRole").getter(getter(ComputeResource::instanceRole)).setter(setter(Builder::instanceRole))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instanceRole").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(ComputeResource::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> PLACEMENT_GROUP_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("placementGroup").getter(getter(ComputeResource::placementGroup)).setter(setter(Builder::placementGroup))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("placementGroup").build()).build();

    private static final SdkField<Integer> BID_PERCENTAGE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("bidPercentage").getter(getter(ComputeResource::bidPercentage)).setter(setter(Builder::bidPercentage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("bidPercentage").build()).build();

    private static final SdkField<String> SPOT_IAM_FLEET_ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("spotIamFleetRole").getter(getter(ComputeResource::spotIamFleetRole))
            .setter(setter(Builder::spotIamFleetRole))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("spotIamFleetRole").build()).build();

    private static final SdkField<LaunchTemplateSpecification> LAUNCH_TEMPLATE_FIELD = SdkField
            .<LaunchTemplateSpecification> builder(MarshallingType.SDK_POJO).memberName("launchTemplate")
            .getter(getter(ComputeResource::launchTemplate)).setter(setter(Builder::launchTemplate))
            .constructor(LaunchTemplateSpecification::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("launchTemplate").build()).build();

    private static final SdkField<List<Ec2Configuration>> EC2_CONFIGURATION_FIELD = SdkField
            .<List<Ec2Configuration>> builder(MarshallingType.LIST)
            .memberName("ec2Configuration")
            .getter(getter(ComputeResource::ec2Configuration))
            .setter(setter(Builder::ec2Configuration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ec2Configuration").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Ec2Configuration> builder(MarshallingType.SDK_POJO)
                                            .constructor(Ec2Configuration::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_FIELD,
            ALLOCATION_STRATEGY_FIELD, MINV_CPUS_FIELD, MAXV_CPUS_FIELD, DESIREDV_CPUS_FIELD, INSTANCE_TYPES_FIELD,
            IMAGE_ID_FIELD, SUBNETS_FIELD, SECURITY_GROUP_IDS_FIELD, EC2_KEY_PAIR_FIELD, INSTANCE_ROLE_FIELD, TAGS_FIELD,
            PLACEMENT_GROUP_FIELD, BID_PERCENTAGE_FIELD, SPOT_IAM_FLEET_ROLE_FIELD, LAUNCH_TEMPLATE_FIELD,
            EC2_CONFIGURATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String type;

    private final String allocationStrategy;

    private final Integer minvCpus;

    private final Integer maxvCpus;

    private final Integer desiredvCpus;

    private final List<String> instanceTypes;

    private final String imageId;

    private final List<String> subnets;

    private final List<String> securityGroupIds;

    private final String ec2KeyPair;

    private final String instanceRole;

    private final Map<String, String> tags;

    private final String placementGroup;

    private final Integer bidPercentage;

    private final String spotIamFleetRole;

    private final LaunchTemplateSpecification launchTemplate;

    private final List<Ec2Configuration> ec2Configuration;

    private ComputeResource(BuilderImpl builder) {
        this.type = builder.type;
        this.allocationStrategy = builder.allocationStrategy;
        this.minvCpus = builder.minvCpus;
        this.maxvCpus = builder.maxvCpus;
        this.desiredvCpus = builder.desiredvCpus;
        this.instanceTypes = builder.instanceTypes;
        this.imageId = builder.imageId;
        this.subnets = builder.subnets;
        this.securityGroupIds = builder.securityGroupIds;
        this.ec2KeyPair = builder.ec2KeyPair;
        this.instanceRole = builder.instanceRole;
        this.tags = builder.tags;
        this.placementGroup = builder.placementGroup;
        this.bidPercentage = builder.bidPercentage;
        this.spotIamFleetRole = builder.spotIamFleetRole;
        this.launchTemplate = builder.launchTemplate;
        this.ec2Configuration = builder.ec2Configuration;
    }

    /**
     * <p>
     * The type of compute environment: <code>EC2</code>, <code>SPOT</code>, <code>FARGATE</code>, or
     * <code>FARGATE_SPOT</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html">Compute Environments</a> in
     * the <i>AWS Batch User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link CRType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of compute environment: <code>EC2</code>, <code>SPOT</code>, <code>FARGATE</code>, or
     *         <code>FARGATE_SPOT</code>. For more information, see <a
     *         href="https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html">Compute
     *         Environments</a> in the <i>AWS Batch User Guide</i>.
     * @see CRType
     */
    public CRType type() {
        return CRType.fromValue(type);
    }

    /**
     * <p>
     * The type of compute environment: <code>EC2</code>, <code>SPOT</code>, <code>FARGATE</code>, or
     * <code>FARGATE_SPOT</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html">Compute Environments</a> in
     * the <i>AWS Batch User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link CRType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of compute environment: <code>EC2</code>, <code>SPOT</code>, <code>FARGATE</code>, or
     *         <code>FARGATE_SPOT</code>. For more information, see <a
     *         href="https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html">Compute
     *         Environments</a> in the <i>AWS Batch User Guide</i>.
     * @see CRType
     */
    public String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The allocation strategy to use for the compute resource if not enough instances of the best fitting instance type
     * can be allocated. This might be because of availability of the instance type in the Region or <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-resource-limits.html">Amazon EC2 service
     * limits</a>. For more information, see <a
     * href="https://docs.aws.amazon.com/batch/latest/userguide/allocation-strategies.html">Allocation Strategies</a> in
     * the <i>AWS Batch User Guide</i>.
     * </p>
     * <note>
     * <p>
     * This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
     * </p>
     * </note>
     * <dl>
     * <dt>BEST_FIT (default)</dt>
     * <dd>
     * <p>
     * AWS Batch selects an instance type that best fits the needs of the jobs with a preference for the lowest-cost
     * instance type. If additional instances of the selected instance type aren't available, AWS Batch will wait for
     * the additional instances to be available. If there are not enough instances available, or if the user is hitting
     * <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-resource-limits.html">Amazon EC2 service
     * limits</a> then additional jobs aren't run until currently running jobs have completed. This allocation strategy
     * keeps costs lower but can limit scaling. If you are using Spot Fleets with <code>BEST_FIT</code> then the Spot
     * Fleet IAM Role must be specified.
     * </p>
     * </dd>
     * <dt>BEST_FIT_PROGRESSIVE</dt>
     * <dd>
     * <p>
     * AWS Batch will select additional instance types that are large enough to meet the requirements of the jobs in the
     * queue, with a preference for instance types with a lower cost per unit vCPU. If additional instances of the
     * previously selected instance types aren't available, AWS Batch will select new instance types.
     * </p>
     * </dd>
     * <dt>SPOT_CAPACITY_OPTIMIZED</dt>
     * <dd>
     * <p>
     * AWS Batch will select one or more instance types that are large enough to meet the requirements of the jobs in
     * the queue, with a preference for instance types that are less likely to be interrupted. This allocation strategy
     * is only available for Spot Instance compute resources.
     * </p>
     * </dd>
     * </dl>
     * <p>
     * With both <code>BEST_FIT_PROGRESSIVE</code> and <code>SPOT_CAPACITY_OPTIMIZED</code> strategies, AWS Batch might
     * need to go above <code>maxvCpus</code> to meet your capacity requirements. In this event, AWS Batch never exceeds
     * <code>maxvCpus</code> by more than a single instance.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #allocationStrategy} will return {@link CRAllocationStrategy#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #allocationStrategyAsString}.
     * </p>
     * 
     * @return The allocation strategy to use for the compute resource if not enough instances of the best fitting
     *         instance type can be allocated. This might be because of availability of the instance type in the Region
     *         or <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-resource-limits.html">Amazon EC2
     *         service limits</a>. For more information, see <a
     *         href="https://docs.aws.amazon.com/batch/latest/userguide/allocation-strategies.html">Allocation
     *         Strategies</a> in the <i>AWS Batch User Guide</i>.</p> <note>
     *         <p>
     *         This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
     *         </p>
     *         </note>
     *         <dl>
     *         <dt>BEST_FIT (default)</dt>
     *         <dd>
     *         <p>
     *         AWS Batch selects an instance type that best fits the needs of the jobs with a preference for the
     *         lowest-cost instance type. If additional instances of the selected instance type aren't available, AWS
     *         Batch will wait for the additional instances to be available. If there are not enough instances
     *         available, or if the user is hitting <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-resource-limits.html">Amazon EC2 service
     *         limits</a> then additional jobs aren't run until currently running jobs have completed. This allocation
     *         strategy keeps costs lower but can limit scaling. If you are using Spot Fleets with <code>BEST_FIT</code>
     *         then the Spot Fleet IAM Role must be specified.
     *         </p>
     *         </dd>
     *         <dt>BEST_FIT_PROGRESSIVE</dt>
     *         <dd>
     *         <p>
     *         AWS Batch will select additional instance types that are large enough to meet the requirements of the
     *         jobs in the queue, with a preference for instance types with a lower cost per unit vCPU. If additional
     *         instances of the previously selected instance types aren't available, AWS Batch will select new instance
     *         types.
     *         </p>
     *         </dd>
     *         <dt>SPOT_CAPACITY_OPTIMIZED</dt>
     *         <dd>
     *         <p>
     *         AWS Batch will select one or more instance types that are large enough to meet the requirements of the
     *         jobs in the queue, with a preference for instance types that are less likely to be interrupted. This
     *         allocation strategy is only available for Spot Instance compute resources.
     *         </p>
     *         </dd>
     *         </dl>
     *         <p>
     *         With both <code>BEST_FIT_PROGRESSIVE</code> and <code>SPOT_CAPACITY_OPTIMIZED</code> strategies, AWS
     *         Batch might need to go above <code>maxvCpus</code> to meet your capacity requirements. In this event, AWS
     *         Batch never exceeds <code>maxvCpus</code> by more than a single instance.
     * @see CRAllocationStrategy
     */
    public CRAllocationStrategy allocationStrategy() {
        return CRAllocationStrategy.fromValue(allocationStrategy);
    }

    /**
     * <p>
     * The allocation strategy to use for the compute resource if not enough instances of the best fitting instance type
     * can be allocated. This might be because of availability of the instance type in the Region or <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-resource-limits.html">Amazon EC2 service
     * limits</a>. For more information, see <a
     * href="https://docs.aws.amazon.com/batch/latest/userguide/allocation-strategies.html">Allocation Strategies</a> in
     * the <i>AWS Batch User Guide</i>.
     * </p>
     * <note>
     * <p>
     * This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
     * </p>
     * </note>
     * <dl>
     * <dt>BEST_FIT (default)</dt>
     * <dd>
     * <p>
     * AWS Batch selects an instance type that best fits the needs of the jobs with a preference for the lowest-cost
     * instance type. If additional instances of the selected instance type aren't available, AWS Batch will wait for
     * the additional instances to be available. If there are not enough instances available, or if the user is hitting
     * <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-resource-limits.html">Amazon EC2 service
     * limits</a> then additional jobs aren't run until currently running jobs have completed. This allocation strategy
     * keeps costs lower but can limit scaling. If you are using Spot Fleets with <code>BEST_FIT</code> then the Spot
     * Fleet IAM Role must be specified.
     * </p>
     * </dd>
     * <dt>BEST_FIT_PROGRESSIVE</dt>
     * <dd>
     * <p>
     * AWS Batch will select additional instance types that are large enough to meet the requirements of the jobs in the
     * queue, with a preference for instance types with a lower cost per unit vCPU. If additional instances of the
     * previously selected instance types aren't available, AWS Batch will select new instance types.
     * </p>
     * </dd>
     * <dt>SPOT_CAPACITY_OPTIMIZED</dt>
     * <dd>
     * <p>
     * AWS Batch will select one or more instance types that are large enough to meet the requirements of the jobs in
     * the queue, with a preference for instance types that are less likely to be interrupted. This allocation strategy
     * is only available for Spot Instance compute resources.
     * </p>
     * </dd>
     * </dl>
     * <p>
     * With both <code>BEST_FIT_PROGRESSIVE</code> and <code>SPOT_CAPACITY_OPTIMIZED</code> strategies, AWS Batch might
     * need to go above <code>maxvCpus</code> to meet your capacity requirements. In this event, AWS Batch never exceeds
     * <code>maxvCpus</code> by more than a single instance.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #allocationStrategy} will return {@link CRAllocationStrategy#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #allocationStrategyAsString}.
     * </p>
     * 
     * @return The allocation strategy to use for the compute resource if not enough instances of the best fitting
     *         instance type can be allocated. This might be because of availability of the instance type in the Region
     *         or <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-resource-limits.html">Amazon EC2
     *         service limits</a>. For more information, see <a
     *         href="https://docs.aws.amazon.com/batch/latest/userguide/allocation-strategies.html">Allocation
     *         Strategies</a> in the <i>AWS Batch User Guide</i>.</p> <note>
     *         <p>
     *         This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
     *         </p>
     *         </note>
     *         <dl>
     *         <dt>BEST_FIT (default)</dt>
     *         <dd>
     *         <p>
     *         AWS Batch selects an instance type that best fits the needs of the jobs with a preference for the
     *         lowest-cost instance type. If additional instances of the selected instance type aren't available, AWS
     *         Batch will wait for the additional instances to be available. If there are not enough instances
     *         available, or if the user is hitting <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-resource-limits.html">Amazon EC2 service
     *         limits</a> then additional jobs aren't run until currently running jobs have completed. This allocation
     *         strategy keeps costs lower but can limit scaling. If you are using Spot Fleets with <code>BEST_FIT</code>
     *         then the Spot Fleet IAM Role must be specified.
     *         </p>
     *         </dd>
     *         <dt>BEST_FIT_PROGRESSIVE</dt>
     *         <dd>
     *         <p>
     *         AWS Batch will select additional instance types that are large enough to meet the requirements of the
     *         jobs in the queue, with a preference for instance types with a lower cost per unit vCPU. If additional
     *         instances of the previously selected instance types aren't available, AWS Batch will select new instance
     *         types.
     *         </p>
     *         </dd>
     *         <dt>SPOT_CAPACITY_OPTIMIZED</dt>
     *         <dd>
     *         <p>
     *         AWS Batch will select one or more instance types that are large enough to meet the requirements of the
     *         jobs in the queue, with a preference for instance types that are less likely to be interrupted. This
     *         allocation strategy is only available for Spot Instance compute resources.
     *         </p>
     *         </dd>
     *         </dl>
     *         <p>
     *         With both <code>BEST_FIT_PROGRESSIVE</code> and <code>SPOT_CAPACITY_OPTIMIZED</code> strategies, AWS
     *         Batch might need to go above <code>maxvCpus</code> to meet your capacity requirements. In this event, AWS
     *         Batch never exceeds <code>maxvCpus</code> by more than a single instance.
     * @see CRAllocationStrategy
     */
    public String allocationStrategyAsString() {
        return allocationStrategy;
    }

    /**
     * <p>
     * The minimum number of Amazon EC2 vCPUs that an environment should maintain (even if the compute environment is
     * <code>DISABLED</code>).
     * </p>
     * <note>
     * <p>
     * This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
     * </p>
     * </note>
     * 
     * @return The minimum number of Amazon EC2 vCPUs that an environment should maintain (even if the compute
     *         environment is <code>DISABLED</code>).</p> <note>
     *         <p>
     *         This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
     *         </p>
     */
    public Integer minvCpus() {
        return minvCpus;
    }

    /**
     * <p>
     * The maximum number of Amazon EC2 vCPUs that an environment can reach.
     * </p>
     * <note>
     * <p>
     * With both <code>BEST_FIT_PROGRESSIVE</code> and <code>SPOT_CAPACITY_OPTIMIZED</code> allocation strategies, AWS
     * Batch may need to go above <code>maxvCpus</code> to meet your capacity requirements. In this event, AWS Batch
     * will never go above <code>maxvCpus</code> by more than a single instance (e.g., no more than a single instance
     * from among those specified in your compute environment).
     * </p>
     * </note>
     * 
     * @return The maximum number of Amazon EC2 vCPUs that an environment can reach.</p> <note>
     *         <p>
     *         With both <code>BEST_FIT_PROGRESSIVE</code> and <code>SPOT_CAPACITY_OPTIMIZED</code> allocation
     *         strategies, AWS Batch may need to go above <code>maxvCpus</code> to meet your capacity requirements. In
     *         this event, AWS Batch will never go above <code>maxvCpus</code> by more than a single instance (e.g., no
     *         more than a single instance from among those specified in your compute environment).
     *         </p>
     */
    public Integer maxvCpus() {
        return maxvCpus;
    }

    /**
     * <p>
     * The desired number of Amazon EC2 vCPUS in the compute environment. AWS Batch modifies this value between the
     * minimum and maximum values, based on job queue demand.
     * </p>
     * <note>
     * <p>
     * This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
     * </p>
     * </note>
     * 
     * @return The desired number of Amazon EC2 vCPUS in the compute environment. AWS Batch modifies this value between
     *         the minimum and maximum values, based on job queue demand.</p> <note>
     *         <p>
     *         This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
     *         </p>
     */
    public Integer desiredvCpus() {
        return desiredvCpus;
    }

    /**
     * Returns true if the InstanceTypes property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasInstanceTypes() {
        return instanceTypes != null && !(instanceTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The instances types that may be launched. You can specify instance families to launch any instance type within
     * those families (for example, <code>c5</code> or <code>p3</code>), or you can specify specific sizes within a
     * family (such as <code>c5.8xlarge</code>). You can also choose <code>optimal</code> to select instance types (from
     * the C, M, and R instance families) on the fly that match the demand of your job queues.
     * </p>
     * <note>
     * <p>
     * This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
     * </p>
     * </note> <note>
     * <p>
     * When you create a compute environment, the instance types that you select for the compute environment must share
     * the same architecture. For example, you can't mix x86 and ARM instances in the same compute environment.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasInstanceTypes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The instances types that may be launched. You can specify instance families to launch any instance type
     *         within those families (for example, <code>c5</code> or <code>p3</code>), or you can specify specific
     *         sizes within a family (such as <code>c5.8xlarge</code>). You can also choose <code>optimal</code> to
     *         select instance types (from the C, M, and R instance families) on the fly that match the demand of your
     *         job queues.</p> <note>
     *         <p>
     *         This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
     *         </p>
     *         </note> <note>
     *         <p>
     *         When you create a compute environment, the instance types that you select for the compute environment
     *         must share the same architecture. For example, you can't mix x86 and ARM instances in the same compute
     *         environment.
     *         </p>
     */
    public List<String> instanceTypes() {
        return instanceTypes;
    }

    /**
     * <p>
     * The Amazon Machine Image (AMI) ID used for instances launched in the compute environment. This parameter is
     * overridden by the <code>imageIdOverride</code> member of the <code>Ec2Configuration</code> structure.
     * </p>
     * <note>
     * <p>
     * This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
     * </p>
     * </note> <note>
     * <p>
     * The AMI that you choose for a compute environment must match the architecture of the instance types that you
     * intend to use for that compute environment. For example, if your compute environment uses A1 instance types, the
     * compute resource AMI that you choose must support ARM instances. Amazon ECS vends both x86 and ARM versions of
     * the Amazon ECS-optimized Amazon Linux 2 AMI. For more information, see <a href=
     * "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#ecs-optimized-ami-linux-variants.html"
     * >Amazon ECS-optimized Amazon Linux 2 AMI</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * </note>
     * 
     * @return The Amazon Machine Image (AMI) ID used for instances launched in the compute environment. This parameter
     *         is overridden by the <code>imageIdOverride</code> member of the <code>Ec2Configuration</code>
     *         structure.</p> <note>
     *         <p>
     *         This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
     *         </p>
     *         </note> <note>
     *         <p>
     *         The AMI that you choose for a compute environment must match the architecture of the instance types that
     *         you intend to use for that compute environment. For example, if your compute environment uses A1 instance
     *         types, the compute resource AMI that you choose must support ARM instances. Amazon ECS vends both x86 and
     *         ARM versions of the Amazon ECS-optimized Amazon Linux 2 AMI. For more information, see <a href=
     *         "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#ecs-optimized-ami-linux-variants.html"
     *         >Amazon ECS-optimized Amazon Linux 2 AMI</a> in the <i>Amazon Elastic Container Service Developer
     *         Guide</i>.
     *         </p>
     */
    public String imageId() {
        return imageId;
    }

    /**
     * Returns true if the Subnets property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasSubnets() {
        return subnets != null && !(subnets instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The VPC subnets into which the compute resources are launched. These subnets must be within the same VPC. This
     * parameter is required for jobs running on Fargate resources, where it can contain up to 16 subnets. For more
     * information, see <a href="https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html">VPCs and Subnets</a>
     * in the <i>Amazon VPC User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSubnets()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The VPC subnets into which the compute resources are launched. These subnets must be within the same VPC.
     *         This parameter is required for jobs running on Fargate resources, where it can contain up to 16 subnets.
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html">VPCs and Subnets</a> in the
     *         <i>Amazon VPC User Guide</i>.
     */
    public List<String> subnets() {
        return subnets;
    }

    /**
     * Returns true if the SecurityGroupIds property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasSecurityGroupIds() {
        return securityGroupIds != null && !(securityGroupIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon EC2 security groups associated with instances launched in the compute environment. One or more
     * security groups must be specified, either in <code>securityGroupIds</code> or using a launch template referenced
     * in <code>launchTemplate</code>. This parameter is required for jobs running on Fargate resources and must contain
     * at least one security group. (Fargate does not support launch templates.) If security groups are specified using
     * both <code>securityGroupIds</code> and <code>launchTemplate</code>, the values in <code>securityGroupIds</code>
     * will be used.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSecurityGroupIds()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The Amazon EC2 security groups associated with instances launched in the compute environment. One or more
     *         security groups must be specified, either in <code>securityGroupIds</code> or using a launch template
     *         referenced in <code>launchTemplate</code>. This parameter is required for jobs running on Fargate
     *         resources and must contain at least one security group. (Fargate does not support launch templates.) If
     *         security groups are specified using both <code>securityGroupIds</code> and <code>launchTemplate</code>,
     *         the values in <code>securityGroupIds</code> will be used.
     */
    public List<String> securityGroupIds() {
        return securityGroupIds;
    }

    /**
     * <p>
     * The Amazon EC2 key pair that is used for instances launched in the compute environment. You can use this key pair
     * to log in to your instances with SSH.
     * </p>
     * <note>
     * <p>
     * This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
     * </p>
     * </note>
     * 
     * @return The Amazon EC2 key pair that is used for instances launched in the compute environment. You can use this
     *         key pair to log in to your instances with SSH.</p> <note>
     *         <p>
     *         This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
     *         </p>
     */
    public String ec2KeyPair() {
        return ec2KeyPair;
    }

    /**
     * <p>
     * The Amazon ECS instance profile applied to Amazon EC2 instances in a compute environment. You can specify the
     * short name or full Amazon Resource Name (ARN) of an instance profile. For example,
     * <code> <i>ecsInstanceRole</i> </code> or
     * <code>arn:aws:iam::<i>&lt;aws_account_id&gt;</i>:instance-profile/<i>ecsInstanceRole</i> </code>. For more
     * information, see <a href="https://docs.aws.amazon.com/batch/latest/userguide/instance_IAM_role.html">Amazon ECS
     * Instance Role</a> in the <i>AWS Batch User Guide</i>.
     * </p>
     * <note>
     * <p>
     * This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
     * </p>
     * </note>
     * 
     * @return The Amazon ECS instance profile applied to Amazon EC2 instances in a compute environment. You can specify
     *         the short name or full Amazon Resource Name (ARN) of an instance profile. For example,
     *         <code> <i>ecsInstanceRole</i> </code> or
     *         <code>arn:aws:iam::<i>&lt;aws_account_id&gt;</i>:instance-profile/<i>ecsInstanceRole</i> </code>. For
     *         more information, see <a
     *         href="https://docs.aws.amazon.com/batch/latest/userguide/instance_IAM_role.html">Amazon ECS Instance
     *         Role</a> in the <i>AWS Batch User Guide</i>.</p> <note>
     *         <p>
     *         This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
     *         </p>
     */
    public String instanceRole() {
        return instanceRole;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Key-value pair tags to be applied to EC2 resources that are launched in the compute environment. For AWS Batch,
     * these take the form of "String1": "String2", where String1 is the tag key and String2 is the tag value—for
     * example, { "Name": "AWS Batch Instance - C4OnDemand" }. This is helpful for recognizing your AWS Batch instances
     * in the Amazon EC2 console. These tags can not be updated or removed after the compute environment has been
     * created; any changes require creating a new compute environment and removing the old compute environment. These
     * tags are not seen when using the AWS Batch <code>ListTagsForResource</code> API operation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Key-value pair tags to be applied to EC2 resources that are launched in the compute environment. For AWS
     *         Batch, these take the form of "String1": "String2", where String1 is the tag key and String2 is the tag
     *         value—for example, { "Name": "AWS Batch Instance - C4OnDemand" }. This is helpful for recognizing your
     *         AWS Batch instances in the Amazon EC2 console. These tags can not be updated or removed after the compute
     *         environment has been created; any changes require creating a new compute environment and removing the old
     *         compute environment. These tags are not seen when using the AWS Batch <code>ListTagsForResource</code>
     *         API operation.
     */
    public Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * The Amazon EC2 placement group to associate with your compute resources. If you intend to submit multi-node
     * parallel jobs to your compute environment, you should consider creating a cluster placement group and associate
     * it with your compute resources. This keeps your multi-node parallel job on a logical grouping of instances within
     * a single Availability Zone with high network flow potential. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/placement-groups.html">Placement Groups</a> in the
     * <i>Amazon EC2 User Guide for Linux Instances</i>.
     * </p>
     * <note>
     * <p>
     * This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
     * </p>
     * </note>
     * 
     * @return The Amazon EC2 placement group to associate with your compute resources. If you intend to submit
     *         multi-node parallel jobs to your compute environment, you should consider creating a cluster placement
     *         group and associate it with your compute resources. This keeps your multi-node parallel job on a logical
     *         grouping of instances within a single Availability Zone with high network flow potential. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/placement-groups.html">Placement Groups</a> in
     *         the <i>Amazon EC2 User Guide for Linux Instances</i>.</p> <note>
     *         <p>
     *         This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
     *         </p>
     */
    public String placementGroup() {
        return placementGroup;
    }

    /**
     * <p>
     * The maximum percentage that a Spot Instance price can be when compared with the On-Demand price for that instance
     * type before instances are launched. For example, if your maximum percentage is 20%, then the Spot price must be
     * below 20% of the current On-Demand price for that Amazon EC2 instance. You always pay the lowest (market) price
     * and never more than your maximum percentage. If you leave this field empty, the default value is 100% of the
     * On-Demand price.
     * </p>
     * <note>
     * <p>
     * This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
     * </p>
     * </note>
     * 
     * @return The maximum percentage that a Spot Instance price can be when compared with the On-Demand price for that
     *         instance type before instances are launched. For example, if your maximum percentage is 20%, then the
     *         Spot price must be below 20% of the current On-Demand price for that Amazon EC2 instance. You always pay
     *         the lowest (market) price and never more than your maximum percentage. If you leave this field empty, the
     *         default value is 100% of the On-Demand price.</p> <note>
     *         <p>
     *         This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
     *         </p>
     */
    public Integer bidPercentage() {
        return bidPercentage;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the Amazon EC2 Spot Fleet IAM role applied to a <code>SPOT</code> compute
     * environment. This role is required if the allocation strategy set to <code>BEST_FIT</code> or if the allocation
     * strategy isn't specified. For more information, see <a
     * href="https://docs.aws.amazon.com/batch/latest/userguide/spot_fleet_IAM_role.html">Amazon EC2 Spot Fleet Role</a>
     * in the <i>AWS Batch User Guide</i>.
     * </p>
     * <note>
     * <p>
     * This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
     * </p>
     * </note> <important>
     * <p>
     * To tag your Spot Instances on creation, the Spot Fleet IAM role specified here must use the newer
     * <b>AmazonEC2SpotFleetTaggingRole</b> managed policy. The previously recommended <b>AmazonEC2SpotFleetRole</b>
     * managed policy doesn't have the required permissions to tag Spot Instances. For more information, see <a
     * href="https://docs.aws.amazon.com/batch/latest/userguide/troubleshooting.html#spot-instance-no-tag">Spot
     * Instances not tagged on creation</a> in the <i>AWS Batch User Guide</i>.
     * </p>
     * </important>
     * 
     * @return The Amazon Resource Name (ARN) of the Amazon EC2 Spot Fleet IAM role applied to a <code>SPOT</code>
     *         compute environment. This role is required if the allocation strategy set to <code>BEST_FIT</code> or if
     *         the allocation strategy isn't specified. For more information, see <a
     *         href="https://docs.aws.amazon.com/batch/latest/userguide/spot_fleet_IAM_role.html">Amazon EC2 Spot Fleet
     *         Role</a> in the <i>AWS Batch User Guide</i>.</p> <note>
     *         <p>
     *         This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
     *         </p>
     *         </note> <important>
     *         <p>
     *         To tag your Spot Instances on creation, the Spot Fleet IAM role specified here must use the newer
     *         <b>AmazonEC2SpotFleetTaggingRole</b> managed policy. The previously recommended
     *         <b>AmazonEC2SpotFleetRole</b> managed policy doesn't have the required permissions to tag Spot Instances.
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/batch/latest/userguide/troubleshooting.html#spot-instance-no-tag">Spot
     *         Instances not tagged on creation</a> in the <i>AWS Batch User Guide</i>.
     *         </p>
     */
    public String spotIamFleetRole() {
        return spotIamFleetRole;
    }

    /**
     * <p>
     * The launch template to use for your compute resources. Any other compute resource parameters that you specify in
     * a <a>CreateComputeEnvironment</a> API operation override the same parameters in the launch template. You must
     * specify either the launch template ID or launch template name in the request, but not both. For more information,
     * see <a href="https://docs.aws.amazon.com/batch/latest/userguide/launch-templates.html">Launch Template
     * Support</a> in the <i>AWS Batch User Guide</i>.
     * </p>
     * <note>
     * <p>
     * This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
     * </p>
     * </note>
     * 
     * @return The launch template to use for your compute resources. Any other compute resource parameters that you
     *         specify in a <a>CreateComputeEnvironment</a> API operation override the same parameters in the launch
     *         template. You must specify either the launch template ID or launch template name in the request, but not
     *         both. For more information, see <a
     *         href="https://docs.aws.amazon.com/batch/latest/userguide/launch-templates.html">Launch Template
     *         Support</a> in the <i>AWS Batch User Guide</i>.</p> <note>
     *         <p>
     *         This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
     *         </p>
     */
    public LaunchTemplateSpecification launchTemplate() {
        return launchTemplate;
    }

    /**
     * Returns true if the Ec2Configuration property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasEc2Configuration() {
        return ec2Configuration != null && !(ec2Configuration instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Provides information used to select Amazon Machine Images (AMIs) for instances in the compute environment. If
     * <code>Ec2Configuration</code> isn't specified, the default is <code>ECS_AL1</code>.
     * </p>
     * <note>
     * <p>
     * This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasEc2Configuration()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Provides information used to select Amazon Machine Images (AMIs) for instances in the compute
     *         environment. If <code>Ec2Configuration</code> isn't specified, the default is <code>ECS_AL1</code>.</p>
     *         <note>
     *         <p>
     *         This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
     *         </p>
     */
    public List<Ec2Configuration> ec2Configuration() {
        return ec2Configuration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(allocationStrategyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(minvCpus());
        hashCode = 31 * hashCode + Objects.hashCode(maxvCpus());
        hashCode = 31 * hashCode + Objects.hashCode(desiredvCpus());
        hashCode = 31 * hashCode + Objects.hashCode(hasInstanceTypes() ? instanceTypes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(imageId());
        hashCode = 31 * hashCode + Objects.hashCode(hasSubnets() ? subnets() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSecurityGroupIds() ? securityGroupIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(ec2KeyPair());
        hashCode = 31 * hashCode + Objects.hashCode(instanceRole());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(placementGroup());
        hashCode = 31 * hashCode + Objects.hashCode(bidPercentage());
        hashCode = 31 * hashCode + Objects.hashCode(spotIamFleetRole());
        hashCode = 31 * hashCode + Objects.hashCode(launchTemplate());
        hashCode = 31 * hashCode + Objects.hashCode(hasEc2Configuration() ? ec2Configuration() : null);
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ComputeResource)) {
            return false;
        }
        ComputeResource other = (ComputeResource) obj;
        return Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(allocationStrategyAsString(), other.allocationStrategyAsString())
                && Objects.equals(minvCpus(), other.minvCpus()) && Objects.equals(maxvCpus(), other.maxvCpus())
                && Objects.equals(desiredvCpus(), other.desiredvCpus()) && hasInstanceTypes() == other.hasInstanceTypes()
                && Objects.equals(instanceTypes(), other.instanceTypes()) && Objects.equals(imageId(), other.imageId())
                && hasSubnets() == other.hasSubnets() && Objects.equals(subnets(), other.subnets())
                && hasSecurityGroupIds() == other.hasSecurityGroupIds()
                && Objects.equals(securityGroupIds(), other.securityGroupIds())
                && Objects.equals(ec2KeyPair(), other.ec2KeyPair()) && Objects.equals(instanceRole(), other.instanceRole())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && Objects.equals(placementGroup(), other.placementGroup())
                && Objects.equals(bidPercentage(), other.bidPercentage())
                && Objects.equals(spotIamFleetRole(), other.spotIamFleetRole())
                && Objects.equals(launchTemplate(), other.launchTemplate())
                && hasEc2Configuration() == other.hasEc2Configuration()
                && Objects.equals(ec2Configuration(), other.ec2Configuration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ComputeResource").add("Type", typeAsString())
                .add("AllocationStrategy", allocationStrategyAsString()).add("MinvCpus", minvCpus()).add("MaxvCpus", maxvCpus())
                .add("DesiredvCpus", desiredvCpus()).add("InstanceTypes", hasInstanceTypes() ? instanceTypes() : null)
                .add("ImageId", imageId()).add("Subnets", hasSubnets() ? subnets() : null)
                .add("SecurityGroupIds", hasSecurityGroupIds() ? securityGroupIds() : null).add("Ec2KeyPair", ec2KeyPair())
                .add("InstanceRole", instanceRole()).add("Tags", hasTags() ? tags() : null)
                .add("PlacementGroup", placementGroup()).add("BidPercentage", bidPercentage())
                .add("SpotIamFleetRole", spotIamFleetRole()).add("LaunchTemplate", launchTemplate())
                .add("Ec2Configuration", hasEc2Configuration() ? ec2Configuration() : null).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "allocationStrategy":
            return Optional.ofNullable(clazz.cast(allocationStrategyAsString()));
        case "minvCpus":
            return Optional.ofNullable(clazz.cast(minvCpus()));
        case "maxvCpus":
            return Optional.ofNullable(clazz.cast(maxvCpus()));
        case "desiredvCpus":
            return Optional.ofNullable(clazz.cast(desiredvCpus()));
        case "instanceTypes":
            return Optional.ofNullable(clazz.cast(instanceTypes()));
        case "imageId":
            return Optional.ofNullable(clazz.cast(imageId()));
        case "subnets":
            return Optional.ofNullable(clazz.cast(subnets()));
        case "securityGroupIds":
            return Optional.ofNullable(clazz.cast(securityGroupIds()));
        case "ec2KeyPair":
            return Optional.ofNullable(clazz.cast(ec2KeyPair()));
        case "instanceRole":
            return Optional.ofNullable(clazz.cast(instanceRole()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "placementGroup":
            return Optional.ofNullable(clazz.cast(placementGroup()));
        case "bidPercentage":
            return Optional.ofNullable(clazz.cast(bidPercentage()));
        case "spotIamFleetRole":
            return Optional.ofNullable(clazz.cast(spotIamFleetRole()));
        case "launchTemplate":
            return Optional.ofNullable(clazz.cast(launchTemplate()));
        case "ec2Configuration":
            return Optional.ofNullable(clazz.cast(ec2Configuration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ComputeResource, T> g) {
        return obj -> g.apply((ComputeResource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ComputeResource> {
        /**
         * <p>
         * The type of compute environment: <code>EC2</code>, <code>SPOT</code>, <code>FARGATE</code>, or
         * <code>FARGATE_SPOT</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html">Compute Environments</a>
         * in the <i>AWS Batch User Guide</i>.
         * </p>
         * 
         * @param type
         *        The type of compute environment: <code>EC2</code>, <code>SPOT</code>, <code>FARGATE</code>, or
         *        <code>FARGATE_SPOT</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html">Compute
         *        Environments</a> in the <i>AWS Batch User Guide</i>.
         * @see CRType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CRType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of compute environment: <code>EC2</code>, <code>SPOT</code>, <code>FARGATE</code>, or
         * <code>FARGATE_SPOT</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html">Compute Environments</a>
         * in the <i>AWS Batch User Guide</i>.
         * </p>
         * 
         * @param type
         *        The type of compute environment: <code>EC2</code>, <code>SPOT</code>, <code>FARGATE</code>, or
         *        <code>FARGATE_SPOT</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html">Compute
         *        Environments</a> in the <i>AWS Batch User Guide</i>.
         * @see CRType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CRType
         */
        Builder type(CRType type);

        /**
         * <p>
         * The allocation strategy to use for the compute resource if not enough instances of the best fitting instance
         * type can be allocated. This might be because of availability of the instance type in the Region or <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-resource-limits.html">Amazon EC2 service
         * limits</a>. For more information, see <a
         * href="https://docs.aws.amazon.com/batch/latest/userguide/allocation-strategies.html">Allocation
         * Strategies</a> in the <i>AWS Batch User Guide</i>.
         * </p>
         * <note>
         * <p>
         * This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
         * </p>
         * </note>
         * <dl>
         * <dt>BEST_FIT (default)</dt>
         * <dd>
         * <p>
         * AWS Batch selects an instance type that best fits the needs of the jobs with a preference for the lowest-cost
         * instance type. If additional instances of the selected instance type aren't available, AWS Batch will wait
         * for the additional instances to be available. If there are not enough instances available, or if the user is
         * hitting <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-resource-limits.html">Amazon EC2
         * service limits</a> then additional jobs aren't run until currently running jobs have completed. This
         * allocation strategy keeps costs lower but can limit scaling. If you are using Spot Fleets with
         * <code>BEST_FIT</code> then the Spot Fleet IAM Role must be specified.
         * </p>
         * </dd>
         * <dt>BEST_FIT_PROGRESSIVE</dt>
         * <dd>
         * <p>
         * AWS Batch will select additional instance types that are large enough to meet the requirements of the jobs in
         * the queue, with a preference for instance types with a lower cost per unit vCPU. If additional instances of
         * the previously selected instance types aren't available, AWS Batch will select new instance types.
         * </p>
         * </dd>
         * <dt>SPOT_CAPACITY_OPTIMIZED</dt>
         * <dd>
         * <p>
         * AWS Batch will select one or more instance types that are large enough to meet the requirements of the jobs
         * in the queue, with a preference for instance types that are less likely to be interrupted. This allocation
         * strategy is only available for Spot Instance compute resources.
         * </p>
         * </dd>
         * </dl>
         * <p>
         * With both <code>BEST_FIT_PROGRESSIVE</code> and <code>SPOT_CAPACITY_OPTIMIZED</code> strategies, AWS Batch
         * might need to go above <code>maxvCpus</code> to meet your capacity requirements. In this event, AWS Batch
         * never exceeds <code>maxvCpus</code> by more than a single instance.
         * </p>
         * 
         * @param allocationStrategy
         *        The allocation strategy to use for the compute resource if not enough instances of the best fitting
         *        instance type can be allocated. This might be because of availability of the instance type in the
         *        Region or <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-resource-limits.html">Amazon EC2 service
         *        limits</a>. For more information, see <a
         *        href="https://docs.aws.amazon.com/batch/latest/userguide/allocation-strategies.html">Allocation
         *        Strategies</a> in the <i>AWS Batch User Guide</i>.</p> <note>
         *        <p>
         *        This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
         *        </p>
         *        </note>
         *        <dl>
         *        <dt>BEST_FIT (default)</dt>
         *        <dd>
         *        <p>
         *        AWS Batch selects an instance type that best fits the needs of the jobs with a preference for the
         *        lowest-cost instance type. If additional instances of the selected instance type aren't available, AWS
         *        Batch will wait for the additional instances to be available. If there are not enough instances
         *        available, or if the user is hitting <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-resource-limits.html">Amazon EC2 service
         *        limits</a> then additional jobs aren't run until currently running jobs have completed. This
         *        allocation strategy keeps costs lower but can limit scaling. If you are using Spot Fleets with
         *        <code>BEST_FIT</code> then the Spot Fleet IAM Role must be specified.
         *        </p>
         *        </dd>
         *        <dt>BEST_FIT_PROGRESSIVE</dt>
         *        <dd>
         *        <p>
         *        AWS Batch will select additional instance types that are large enough to meet the requirements of the
         *        jobs in the queue, with a preference for instance types with a lower cost per unit vCPU. If additional
         *        instances of the previously selected instance types aren't available, AWS Batch will select new
         *        instance types.
         *        </p>
         *        </dd>
         *        <dt>SPOT_CAPACITY_OPTIMIZED</dt>
         *        <dd>
         *        <p>
         *        AWS Batch will select one or more instance types that are large enough to meet the requirements of the
         *        jobs in the queue, with a preference for instance types that are less likely to be interrupted. This
         *        allocation strategy is only available for Spot Instance compute resources.
         *        </p>
         *        </dd>
         *        </dl>
         *        <p>
         *        With both <code>BEST_FIT_PROGRESSIVE</code> and <code>SPOT_CAPACITY_OPTIMIZED</code> strategies, AWS
         *        Batch might need to go above <code>maxvCpus</code> to meet your capacity requirements. In this event,
         *        AWS Batch never exceeds <code>maxvCpus</code> by more than a single instance.
         * @see CRAllocationStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CRAllocationStrategy
         */
        Builder allocationStrategy(String allocationStrategy);

        /**
         * <p>
         * The allocation strategy to use for the compute resource if not enough instances of the best fitting instance
         * type can be allocated. This might be because of availability of the instance type in the Region or <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-resource-limits.html">Amazon EC2 service
         * limits</a>. For more information, see <a
         * href="https://docs.aws.amazon.com/batch/latest/userguide/allocation-strategies.html">Allocation
         * Strategies</a> in the <i>AWS Batch User Guide</i>.
         * </p>
         * <note>
         * <p>
         * This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
         * </p>
         * </note>
         * <dl>
         * <dt>BEST_FIT (default)</dt>
         * <dd>
         * <p>
         * AWS Batch selects an instance type that best fits the needs of the jobs with a preference for the lowest-cost
         * instance type. If additional instances of the selected instance type aren't available, AWS Batch will wait
         * for the additional instances to be available. If there are not enough instances available, or if the user is
         * hitting <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-resource-limits.html">Amazon EC2
         * service limits</a> then additional jobs aren't run until currently running jobs have completed. This
         * allocation strategy keeps costs lower but can limit scaling. If you are using Spot Fleets with
         * <code>BEST_FIT</code> then the Spot Fleet IAM Role must be specified.
         * </p>
         * </dd>
         * <dt>BEST_FIT_PROGRESSIVE</dt>
         * <dd>
         * <p>
         * AWS Batch will select additional instance types that are large enough to meet the requirements of the jobs in
         * the queue, with a preference for instance types with a lower cost per unit vCPU. If additional instances of
         * the previously selected instance types aren't available, AWS Batch will select new instance types.
         * </p>
         * </dd>
         * <dt>SPOT_CAPACITY_OPTIMIZED</dt>
         * <dd>
         * <p>
         * AWS Batch will select one or more instance types that are large enough to meet the requirements of the jobs
         * in the queue, with a preference for instance types that are less likely to be interrupted. This allocation
         * strategy is only available for Spot Instance compute resources.
         * </p>
         * </dd>
         * </dl>
         * <p>
         * With both <code>BEST_FIT_PROGRESSIVE</code> and <code>SPOT_CAPACITY_OPTIMIZED</code> strategies, AWS Batch
         * might need to go above <code>maxvCpus</code> to meet your capacity requirements. In this event, AWS Batch
         * never exceeds <code>maxvCpus</code> by more than a single instance.
         * </p>
         * 
         * @param allocationStrategy
         *        The allocation strategy to use for the compute resource if not enough instances of the best fitting
         *        instance type can be allocated. This might be because of availability of the instance type in the
         *        Region or <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-resource-limits.html">Amazon EC2 service
         *        limits</a>. For more information, see <a
         *        href="https://docs.aws.amazon.com/batch/latest/userguide/allocation-strategies.html">Allocation
         *        Strategies</a> in the <i>AWS Batch User Guide</i>.</p> <note>
         *        <p>
         *        This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
         *        </p>
         *        </note>
         *        <dl>
         *        <dt>BEST_FIT (default)</dt>
         *        <dd>
         *        <p>
         *        AWS Batch selects an instance type that best fits the needs of the jobs with a preference for the
         *        lowest-cost instance type. If additional instances of the selected instance type aren't available, AWS
         *        Batch will wait for the additional instances to be available. If there are not enough instances
         *        available, or if the user is hitting <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-resource-limits.html">Amazon EC2 service
         *        limits</a> then additional jobs aren't run until currently running jobs have completed. This
         *        allocation strategy keeps costs lower but can limit scaling. If you are using Spot Fleets with
         *        <code>BEST_FIT</code> then the Spot Fleet IAM Role must be specified.
         *        </p>
         *        </dd>
         *        <dt>BEST_FIT_PROGRESSIVE</dt>
         *        <dd>
         *        <p>
         *        AWS Batch will select additional instance types that are large enough to meet the requirements of the
         *        jobs in the queue, with a preference for instance types with a lower cost per unit vCPU. If additional
         *        instances of the previously selected instance types aren't available, AWS Batch will select new
         *        instance types.
         *        </p>
         *        </dd>
         *        <dt>SPOT_CAPACITY_OPTIMIZED</dt>
         *        <dd>
         *        <p>
         *        AWS Batch will select one or more instance types that are large enough to meet the requirements of the
         *        jobs in the queue, with a preference for instance types that are less likely to be interrupted. This
         *        allocation strategy is only available for Spot Instance compute resources.
         *        </p>
         *        </dd>
         *        </dl>
         *        <p>
         *        With both <code>BEST_FIT_PROGRESSIVE</code> and <code>SPOT_CAPACITY_OPTIMIZED</code> strategies, AWS
         *        Batch might need to go above <code>maxvCpus</code> to meet your capacity requirements. In this event,
         *        AWS Batch never exceeds <code>maxvCpus</code> by more than a single instance.
         * @see CRAllocationStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CRAllocationStrategy
         */
        Builder allocationStrategy(CRAllocationStrategy allocationStrategy);

        /**
         * <p>
         * The minimum number of Amazon EC2 vCPUs that an environment should maintain (even if the compute environment
         * is <code>DISABLED</code>).
         * </p>
         * <note>
         * <p>
         * This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
         * </p>
         * </note>
         * 
         * @param minvCpus
         *        The minimum number of Amazon EC2 vCPUs that an environment should maintain (even if the compute
         *        environment is <code>DISABLED</code>).</p> <note>
         *        <p>
         *        This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minvCpus(Integer minvCpus);

        /**
         * <p>
         * The maximum number of Amazon EC2 vCPUs that an environment can reach.
         * </p>
         * <note>
         * <p>
         * With both <code>BEST_FIT_PROGRESSIVE</code> and <code>SPOT_CAPACITY_OPTIMIZED</code> allocation strategies,
         * AWS Batch may need to go above <code>maxvCpus</code> to meet your capacity requirements. In this event, AWS
         * Batch will never go above <code>maxvCpus</code> by more than a single instance (e.g., no more than a single
         * instance from among those specified in your compute environment).
         * </p>
         * </note>
         * 
         * @param maxvCpus
         *        The maximum number of Amazon EC2 vCPUs that an environment can reach.</p> <note>
         *        <p>
         *        With both <code>BEST_FIT_PROGRESSIVE</code> and <code>SPOT_CAPACITY_OPTIMIZED</code> allocation
         *        strategies, AWS Batch may need to go above <code>maxvCpus</code> to meet your capacity requirements.
         *        In this event, AWS Batch will never go above <code>maxvCpus</code> by more than a single instance
         *        (e.g., no more than a single instance from among those specified in your compute environment).
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxvCpus(Integer maxvCpus);

        /**
         * <p>
         * The desired number of Amazon EC2 vCPUS in the compute environment. AWS Batch modifies this value between the
         * minimum and maximum values, based on job queue demand.
         * </p>
         * <note>
         * <p>
         * This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
         * </p>
         * </note>
         * 
         * @param desiredvCpus
         *        The desired number of Amazon EC2 vCPUS in the compute environment. AWS Batch modifies this value
         *        between the minimum and maximum values, based on job queue demand.</p> <note>
         *        <p>
         *        This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder desiredvCpus(Integer desiredvCpus);

        /**
         * <p>
         * The instances types that may be launched. You can specify instance families to launch any instance type
         * within those families (for example, <code>c5</code> or <code>p3</code>), or you can specify specific sizes
         * within a family (such as <code>c5.8xlarge</code>). You can also choose <code>optimal</code> to select
         * instance types (from the C, M, and R instance families) on the fly that match the demand of your job queues.
         * </p>
         * <note>
         * <p>
         * This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
         * </p>
         * </note> <note>
         * <p>
         * When you create a compute environment, the instance types that you select for the compute environment must
         * share the same architecture. For example, you can't mix x86 and ARM instances in the same compute
         * environment.
         * </p>
         * </note>
         * 
         * @param instanceTypes
         *        The instances types that may be launched. You can specify instance families to launch any instance
         *        type within those families (for example, <code>c5</code> or <code>p3</code>), or you can specify
         *        specific sizes within a family (such as <code>c5.8xlarge</code>). You can also choose
         *        <code>optimal</code> to select instance types (from the C, M, and R instance families) on the fly that
         *        match the demand of your job queues.</p> <note>
         *        <p>
         *        This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
         *        </p>
         *        </note> <note>
         *        <p>
         *        When you create a compute environment, the instance types that you select for the compute environment
         *        must share the same architecture. For example, you can't mix x86 and ARM instances in the same compute
         *        environment.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceTypes(Collection<String> instanceTypes);

        /**
         * <p>
         * The instances types that may be launched. You can specify instance families to launch any instance type
         * within those families (for example, <code>c5</code> or <code>p3</code>), or you can specify specific sizes
         * within a family (such as <code>c5.8xlarge</code>). You can also choose <code>optimal</code> to select
         * instance types (from the C, M, and R instance families) on the fly that match the demand of your job queues.
         * </p>
         * <note>
         * <p>
         * This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
         * </p>
         * </note> <note>
         * <p>
         * When you create a compute environment, the instance types that you select for the compute environment must
         * share the same architecture. For example, you can't mix x86 and ARM instances in the same compute
         * environment.
         * </p>
         * </note>
         * 
         * @param instanceTypes
         *        The instances types that may be launched. You can specify instance families to launch any instance
         *        type within those families (for example, <code>c5</code> or <code>p3</code>), or you can specify
         *        specific sizes within a family (such as <code>c5.8xlarge</code>). You can also choose
         *        <code>optimal</code> to select instance types (from the C, M, and R instance families) on the fly that
         *        match the demand of your job queues.</p> <note>
         *        <p>
         *        This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
         *        </p>
         *        </note> <note>
         *        <p>
         *        When you create a compute environment, the instance types that you select for the compute environment
         *        must share the same architecture. For example, you can't mix x86 and ARM instances in the same compute
         *        environment.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceTypes(String... instanceTypes);

        /**
         * <p>
         * The Amazon Machine Image (AMI) ID used for instances launched in the compute environment. This parameter is
         * overridden by the <code>imageIdOverride</code> member of the <code>Ec2Configuration</code> structure.
         * </p>
         * <note>
         * <p>
         * This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
         * </p>
         * </note> <note>
         * <p>
         * The AMI that you choose for a compute environment must match the architecture of the instance types that you
         * intend to use for that compute environment. For example, if your compute environment uses A1 instance types,
         * the compute resource AMI that you choose must support ARM instances. Amazon ECS vends both x86 and ARM
         * versions of the Amazon ECS-optimized Amazon Linux 2 AMI. For more information, see <a href=
         * "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#ecs-optimized-ami-linux-variants.html"
         * >Amazon ECS-optimized Amazon Linux 2 AMI</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * </note>
         * 
         * @param imageId
         *        The Amazon Machine Image (AMI) ID used for instances launched in the compute environment. This
         *        parameter is overridden by the <code>imageIdOverride</code> member of the
         *        <code>Ec2Configuration</code> structure.</p> <note>
         *        <p>
         *        This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
         *        </p>
         *        </note> <note>
         *        <p>
         *        The AMI that you choose for a compute environment must match the architecture of the instance types
         *        that you intend to use for that compute environment. For example, if your compute environment uses A1
         *        instance types, the compute resource AMI that you choose must support ARM instances. Amazon ECS vends
         *        both x86 and ARM versions of the Amazon ECS-optimized Amazon Linux 2 AMI. For more information, see <a
         *        href=
         *        "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html#ecs-optimized-ami-linux-variants.html"
         *        >Amazon ECS-optimized Amazon Linux 2 AMI</a> in the <i>Amazon Elastic Container Service Developer
         *        Guide</i>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageId(String imageId);

        /**
         * <p>
         * The VPC subnets into which the compute resources are launched. These subnets must be within the same VPC.
         * This parameter is required for jobs running on Fargate resources, where it can contain up to 16 subnets. For
         * more information, see <a href="https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html">VPCs and
         * Subnets</a> in the <i>Amazon VPC User Guide</i>.
         * </p>
         * 
         * @param subnets
         *        The VPC subnets into which the compute resources are launched. These subnets must be within the same
         *        VPC. This parameter is required for jobs running on Fargate resources, where it can contain up to 16
         *        subnets. For more information, see <a
         *        href="https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html">VPCs and Subnets</a> in the
         *        <i>Amazon VPC User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnets(Collection<String> subnets);

        /**
         * <p>
         * The VPC subnets into which the compute resources are launched. These subnets must be within the same VPC.
         * This parameter is required for jobs running on Fargate resources, where it can contain up to 16 subnets. For
         * more information, see <a href="https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html">VPCs and
         * Subnets</a> in the <i>Amazon VPC User Guide</i>.
         * </p>
         * 
         * @param subnets
         *        The VPC subnets into which the compute resources are launched. These subnets must be within the same
         *        VPC. This parameter is required for jobs running on Fargate resources, where it can contain up to 16
         *        subnets. For more information, see <a
         *        href="https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html">VPCs and Subnets</a> in the
         *        <i>Amazon VPC User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnets(String... subnets);

        /**
         * <p>
         * The Amazon EC2 security groups associated with instances launched in the compute environment. One or more
         * security groups must be specified, either in <code>securityGroupIds</code> or using a launch template
         * referenced in <code>launchTemplate</code>. This parameter is required for jobs running on Fargate resources
         * and must contain at least one security group. (Fargate does not support launch templates.) If security groups
         * are specified using both <code>securityGroupIds</code> and <code>launchTemplate</code>, the values in
         * <code>securityGroupIds</code> will be used.
         * </p>
         * 
         * @param securityGroupIds
         *        The Amazon EC2 security groups associated with instances launched in the compute environment. One or
         *        more security groups must be specified, either in <code>securityGroupIds</code> or using a launch
         *        template referenced in <code>launchTemplate</code>. This parameter is required for jobs running on
         *        Fargate resources and must contain at least one security group. (Fargate does not support launch
         *        templates.) If security groups are specified using both <code>securityGroupIds</code> and
         *        <code>launchTemplate</code>, the values in <code>securityGroupIds</code> will be used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupIds(Collection<String> securityGroupIds);

        /**
         * <p>
         * The Amazon EC2 security groups associated with instances launched in the compute environment. One or more
         * security groups must be specified, either in <code>securityGroupIds</code> or using a launch template
         * referenced in <code>launchTemplate</code>. This parameter is required for jobs running on Fargate resources
         * and must contain at least one security group. (Fargate does not support launch templates.) If security groups
         * are specified using both <code>securityGroupIds</code> and <code>launchTemplate</code>, the values in
         * <code>securityGroupIds</code> will be used.
         * </p>
         * 
         * @param securityGroupIds
         *        The Amazon EC2 security groups associated with instances launched in the compute environment. One or
         *        more security groups must be specified, either in <code>securityGroupIds</code> or using a launch
         *        template referenced in <code>launchTemplate</code>. This parameter is required for jobs running on
         *        Fargate resources and must contain at least one security group. (Fargate does not support launch
         *        templates.) If security groups are specified using both <code>securityGroupIds</code> and
         *        <code>launchTemplate</code>, the values in <code>securityGroupIds</code> will be used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupIds(String... securityGroupIds);

        /**
         * <p>
         * The Amazon EC2 key pair that is used for instances launched in the compute environment. You can use this key
         * pair to log in to your instances with SSH.
         * </p>
         * <note>
         * <p>
         * This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
         * </p>
         * </note>
         * 
         * @param ec2KeyPair
         *        The Amazon EC2 key pair that is used for instances launched in the compute environment. You can use
         *        this key pair to log in to your instances with SSH.</p> <note>
         *        <p>
         *        This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ec2KeyPair(String ec2KeyPair);

        /**
         * <p>
         * The Amazon ECS instance profile applied to Amazon EC2 instances in a compute environment. You can specify the
         * short name or full Amazon Resource Name (ARN) of an instance profile. For example,
         * <code> <i>ecsInstanceRole</i> </code> or
         * <code>arn:aws:iam::<i>&lt;aws_account_id&gt;</i>:instance-profile/<i>ecsInstanceRole</i> </code>. For more
         * information, see <a href="https://docs.aws.amazon.com/batch/latest/userguide/instance_IAM_role.html">Amazon
         * ECS Instance Role</a> in the <i>AWS Batch User Guide</i>.
         * </p>
         * <note>
         * <p>
         * This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
         * </p>
         * </note>
         * 
         * @param instanceRole
         *        The Amazon ECS instance profile applied to Amazon EC2 instances in a compute environment. You can
         *        specify the short name or full Amazon Resource Name (ARN) of an instance profile. For example,
         *        <code> <i>ecsInstanceRole</i> </code> or
         *        <code>arn:aws:iam::<i>&lt;aws_account_id&gt;</i>:instance-profile/<i>ecsInstanceRole</i> </code>. For
         *        more information, see <a
         *        href="https://docs.aws.amazon.com/batch/latest/userguide/instance_IAM_role.html">Amazon ECS Instance
         *        Role</a> in the <i>AWS Batch User Guide</i>.</p> <note>
         *        <p>
         *        This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceRole(String instanceRole);

        /**
         * <p>
         * Key-value pair tags to be applied to EC2 resources that are launched in the compute environment. For AWS
         * Batch, these take the form of "String1": "String2", where String1 is the tag key and String2 is the tag
         * value—for example, { "Name": "AWS Batch Instance - C4OnDemand" }. This is helpful for recognizing your AWS
         * Batch instances in the Amazon EC2 console. These tags can not be updated or removed after the compute
         * environment has been created; any changes require creating a new compute environment and removing the old
         * compute environment. These tags are not seen when using the AWS Batch <code>ListTagsForResource</code> API
         * operation.
         * </p>
         * 
         * @param tags
         *        Key-value pair tags to be applied to EC2 resources that are launched in the compute environment. For
         *        AWS Batch, these take the form of "String1": "String2", where String1 is the tag key and String2 is
         *        the tag value—for example, { "Name": "AWS Batch Instance - C4OnDemand" }. This is helpful for
         *        recognizing your AWS Batch instances in the Amazon EC2 console. These tags can not be updated or
         *        removed after the compute environment has been created; any changes require creating a new compute
         *        environment and removing the old compute environment. These tags are not seen when using the AWS Batch
         *        <code>ListTagsForResource</code> API operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * The Amazon EC2 placement group to associate with your compute resources. If you intend to submit multi-node
         * parallel jobs to your compute environment, you should consider creating a cluster placement group and
         * associate it with your compute resources. This keeps your multi-node parallel job on a logical grouping of
         * instances within a single Availability Zone with high network flow potential. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/placement-groups.html">Placement Groups</a> in the
         * <i>Amazon EC2 User Guide for Linux Instances</i>.
         * </p>
         * <note>
         * <p>
         * This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
         * </p>
         * </note>
         * 
         * @param placementGroup
         *        The Amazon EC2 placement group to associate with your compute resources. If you intend to submit
         *        multi-node parallel jobs to your compute environment, you should consider creating a cluster placement
         *        group and associate it with your compute resources. This keeps your multi-node parallel job on a
         *        logical grouping of instances within a single Availability Zone with high network flow potential. For
         *        more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/placement-groups.html">Placement Groups</a>
         *        in the <i>Amazon EC2 User Guide for Linux Instances</i>.</p> <note>
         *        <p>
         *        This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder placementGroup(String placementGroup);

        /**
         * <p>
         * The maximum percentage that a Spot Instance price can be when compared with the On-Demand price for that
         * instance type before instances are launched. For example, if your maximum percentage is 20%, then the Spot
         * price must be below 20% of the current On-Demand price for that Amazon EC2 instance. You always pay the
         * lowest (market) price and never more than your maximum percentage. If you leave this field empty, the default
         * value is 100% of the On-Demand price.
         * </p>
         * <note>
         * <p>
         * This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
         * </p>
         * </note>
         * 
         * @param bidPercentage
         *        The maximum percentage that a Spot Instance price can be when compared with the On-Demand price for
         *        that instance type before instances are launched. For example, if your maximum percentage is 20%, then
         *        the Spot price must be below 20% of the current On-Demand price for that Amazon EC2 instance. You
         *        always pay the lowest (market) price and never more than your maximum percentage. If you leave this
         *        field empty, the default value is 100% of the On-Demand price.</p> <note>
         *        <p>
         *        This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bidPercentage(Integer bidPercentage);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the Amazon EC2 Spot Fleet IAM role applied to a <code>SPOT</code> compute
         * environment. This role is required if the allocation strategy set to <code>BEST_FIT</code> or if the
         * allocation strategy isn't specified. For more information, see <a
         * href="https://docs.aws.amazon.com/batch/latest/userguide/spot_fleet_IAM_role.html">Amazon EC2 Spot Fleet
         * Role</a> in the <i>AWS Batch User Guide</i>.
         * </p>
         * <note>
         * <p>
         * This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
         * </p>
         * </note> <important>
         * <p>
         * To tag your Spot Instances on creation, the Spot Fleet IAM role specified here must use the newer
         * <b>AmazonEC2SpotFleetTaggingRole</b> managed policy. The previously recommended <b>AmazonEC2SpotFleetRole</b>
         * managed policy doesn't have the required permissions to tag Spot Instances. For more information, see <a
         * href="https://docs.aws.amazon.com/batch/latest/userguide/troubleshooting.html#spot-instance-no-tag">Spot
         * Instances not tagged on creation</a> in the <i>AWS Batch User Guide</i>.
         * </p>
         * </important>
         * 
         * @param spotIamFleetRole
         *        The Amazon Resource Name (ARN) of the Amazon EC2 Spot Fleet IAM role applied to a <code>SPOT</code>
         *        compute environment. This role is required if the allocation strategy set to <code>BEST_FIT</code> or
         *        if the allocation strategy isn't specified. For more information, see <a
         *        href="https://docs.aws.amazon.com/batch/latest/userguide/spot_fleet_IAM_role.html">Amazon EC2 Spot
         *        Fleet Role</a> in the <i>AWS Batch User Guide</i>.</p> <note>
         *        <p>
         *        This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
         *        </p>
         *        </note> <important>
         *        <p>
         *        To tag your Spot Instances on creation, the Spot Fleet IAM role specified here must use the newer
         *        <b>AmazonEC2SpotFleetTaggingRole</b> managed policy. The previously recommended
         *        <b>AmazonEC2SpotFleetRole</b> managed policy doesn't have the required permissions to tag Spot
         *        Instances. For more information, see <a
         *        href="https://docs.aws.amazon.com/batch/latest/userguide/troubleshooting.html#spot-instance-no-tag"
         *        >Spot Instances not tagged on creation</a> in the <i>AWS Batch User Guide</i>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder spotIamFleetRole(String spotIamFleetRole);

        /**
         * <p>
         * The launch template to use for your compute resources. Any other compute resource parameters that you specify
         * in a <a>CreateComputeEnvironment</a> API operation override the same parameters in the launch template. You
         * must specify either the launch template ID or launch template name in the request, but not both. For more
         * information, see <a href="https://docs.aws.amazon.com/batch/latest/userguide/launch-templates.html">Launch
         * Template Support</a> in the <i>AWS Batch User Guide</i>.
         * </p>
         * <note>
         * <p>
         * This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
         * </p>
         * </note>
         * 
         * @param launchTemplate
         *        The launch template to use for your compute resources. Any other compute resource parameters that you
         *        specify in a <a>CreateComputeEnvironment</a> API operation override the same parameters in the launch
         *        template. You must specify either the launch template ID or launch template name in the request, but
         *        not both. For more information, see <a
         *        href="https://docs.aws.amazon.com/batch/latest/userguide/launch-templates.html">Launch Template
         *        Support</a> in the <i>AWS Batch User Guide</i>.</p> <note>
         *        <p>
         *        This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder launchTemplate(LaunchTemplateSpecification launchTemplate);

        /**
         * <p>
         * The launch template to use for your compute resources. Any other compute resource parameters that you specify
         * in a <a>CreateComputeEnvironment</a> API operation override the same parameters in the launch template. You
         * must specify either the launch template ID or launch template name in the request, but not both. For more
         * information, see <a href="https://docs.aws.amazon.com/batch/latest/userguide/launch-templates.html">Launch
         * Template Support</a> in the <i>AWS Batch User Guide</i>.
         * </p>
         * <note>
         * <p>
         * This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
         * </p>
         * </note> This is a convenience that creates an instance of the {@link LaunchTemplateSpecification.Builder}
         * avoiding the need to create one manually via {@link LaunchTemplateSpecification#builder()}.
         *
         * When the {@link Consumer} completes, {@link LaunchTemplateSpecification.Builder#build()} is called
         * immediately and its result is passed to {@link #launchTemplate(LaunchTemplateSpecification)}.
         * 
         * @param launchTemplate
         *        a consumer that will call methods on {@link LaunchTemplateSpecification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #launchTemplate(LaunchTemplateSpecification)
         */
        default Builder launchTemplate(Consumer<LaunchTemplateSpecification.Builder> launchTemplate) {
            return launchTemplate(LaunchTemplateSpecification.builder().applyMutation(launchTemplate).build());
        }

        /**
         * <p>
         * Provides information used to select Amazon Machine Images (AMIs) for instances in the compute environment. If
         * <code>Ec2Configuration</code> isn't specified, the default is <code>ECS_AL1</code>.
         * </p>
         * <note>
         * <p>
         * This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
         * </p>
         * </note>
         * 
         * @param ec2Configuration
         *        Provides information used to select Amazon Machine Images (AMIs) for instances in the compute
         *        environment. If <code>Ec2Configuration</code> isn't specified, the default is <code>ECS_AL1</code>
         *        .</p> <note>
         *        <p>
         *        This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ec2Configuration(Collection<Ec2Configuration> ec2Configuration);

        /**
         * <p>
         * Provides information used to select Amazon Machine Images (AMIs) for instances in the compute environment. If
         * <code>Ec2Configuration</code> isn't specified, the default is <code>ECS_AL1</code>.
         * </p>
         * <note>
         * <p>
         * This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
         * </p>
         * </note>
         * 
         * @param ec2Configuration
         *        Provides information used to select Amazon Machine Images (AMIs) for instances in the compute
         *        environment. If <code>Ec2Configuration</code> isn't specified, the default is <code>ECS_AL1</code>
         *        .</p> <note>
         *        <p>
         *        This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ec2Configuration(Ec2Configuration... ec2Configuration);

        /**
         * <p>
         * Provides information used to select Amazon Machine Images (AMIs) for instances in the compute environment. If
         * <code>Ec2Configuration</code> isn't specified, the default is <code>ECS_AL1</code>.
         * </p>
         * <note>
         * <p>
         * This parameter isn't applicable to jobs running on Fargate resources, and shouldn't be specified.
         * </p>
         * </note> This is a convenience that creates an instance of the {@link List<Ec2Configuration>.Builder} avoiding
         * the need to create one manually via {@link List<Ec2Configuration>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Ec2Configuration>.Builder#build()} is called immediately and
         * its result is passed to {@link #ec2Configuration(List<Ec2Configuration>)}.
         * 
         * @param ec2Configuration
         *        a consumer that will call methods on {@link List<Ec2Configuration>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ec2Configuration(List<Ec2Configuration>)
         */
        Builder ec2Configuration(Consumer<Ec2Configuration.Builder>... ec2Configuration);
    }

    static final class BuilderImpl implements Builder {
        private String type;

        private String allocationStrategy;

        private Integer minvCpus;

        private Integer maxvCpus;

        private Integer desiredvCpus;

        private List<String> instanceTypes = DefaultSdkAutoConstructList.getInstance();

        private String imageId;

        private List<String> subnets = DefaultSdkAutoConstructList.getInstance();

        private List<String> securityGroupIds = DefaultSdkAutoConstructList.getInstance();

        private String ec2KeyPair;

        private String instanceRole;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private String placementGroup;

        private Integer bidPercentage;

        private String spotIamFleetRole;

        private LaunchTemplateSpecification launchTemplate;

        private List<Ec2Configuration> ec2Configuration = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ComputeResource model) {
            type(model.type);
            allocationStrategy(model.allocationStrategy);
            minvCpus(model.minvCpus);
            maxvCpus(model.maxvCpus);
            desiredvCpus(model.desiredvCpus);
            instanceTypes(model.instanceTypes);
            imageId(model.imageId);
            subnets(model.subnets);
            securityGroupIds(model.securityGroupIds);
            ec2KeyPair(model.ec2KeyPair);
            instanceRole(model.instanceRole);
            tags(model.tags);
            placementGroup(model.placementGroup);
            bidPercentage(model.bidPercentage);
            spotIamFleetRole(model.spotIamFleetRole);
            launchTemplate(model.launchTemplate);
            ec2Configuration(model.ec2Configuration);
        }

        public final String getType() {
            return type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(CRType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final void setType(String type) {
            this.type = type;
        }

        public final String getAllocationStrategy() {
            return allocationStrategy;
        }

        @Override
        public final Builder allocationStrategy(String allocationStrategy) {
            this.allocationStrategy = allocationStrategy;
            return this;
        }

        @Override
        public final Builder allocationStrategy(CRAllocationStrategy allocationStrategy) {
            this.allocationStrategy(allocationStrategy == null ? null : allocationStrategy.toString());
            return this;
        }

        public final void setAllocationStrategy(String allocationStrategy) {
            this.allocationStrategy = allocationStrategy;
        }

        public final Integer getMinvCpus() {
            return minvCpus;
        }

        @Override
        public final Builder minvCpus(Integer minvCpus) {
            this.minvCpus = minvCpus;
            return this;
        }

        public final void setMinvCpus(Integer minvCpus) {
            this.minvCpus = minvCpus;
        }

        public final Integer getMaxvCpus() {
            return maxvCpus;
        }

        @Override
        public final Builder maxvCpus(Integer maxvCpus) {
            this.maxvCpus = maxvCpus;
            return this;
        }

        public final void setMaxvCpus(Integer maxvCpus) {
            this.maxvCpus = maxvCpus;
        }

        public final Integer getDesiredvCpus() {
            return desiredvCpus;
        }

        @Override
        public final Builder desiredvCpus(Integer desiredvCpus) {
            this.desiredvCpus = desiredvCpus;
            return this;
        }

        public final void setDesiredvCpus(Integer desiredvCpus) {
            this.desiredvCpus = desiredvCpus;
        }

        public final Collection<String> getInstanceTypes() {
            if (instanceTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return instanceTypes;
        }

        @Override
        public final Builder instanceTypes(Collection<String> instanceTypes) {
            this.instanceTypes = StringListCopier.copy(instanceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceTypes(String... instanceTypes) {
            instanceTypes(Arrays.asList(instanceTypes));
            return this;
        }

        public final void setInstanceTypes(Collection<String> instanceTypes) {
            this.instanceTypes = StringListCopier.copy(instanceTypes);
        }

        public final String getImageId() {
            return imageId;
        }

        @Override
        public final Builder imageId(String imageId) {
            this.imageId = imageId;
            return this;
        }

        public final void setImageId(String imageId) {
            this.imageId = imageId;
        }

        public final Collection<String> getSubnets() {
            if (subnets instanceof SdkAutoConstructList) {
                return null;
            }
            return subnets;
        }

        @Override
        public final Builder subnets(Collection<String> subnets) {
            this.subnets = StringListCopier.copy(subnets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subnets(String... subnets) {
            subnets(Arrays.asList(subnets));
            return this;
        }

        public final void setSubnets(Collection<String> subnets) {
            this.subnets = StringListCopier.copy(subnets);
        }

        public final Collection<String> getSecurityGroupIds() {
            if (securityGroupIds instanceof SdkAutoConstructList) {
                return null;
            }
            return securityGroupIds;
        }

        @Override
        public final Builder securityGroupIds(Collection<String> securityGroupIds) {
            this.securityGroupIds = StringListCopier.copy(securityGroupIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder securityGroupIds(String... securityGroupIds) {
            securityGroupIds(Arrays.asList(securityGroupIds));
            return this;
        }

        public final void setSecurityGroupIds(Collection<String> securityGroupIds) {
            this.securityGroupIds = StringListCopier.copy(securityGroupIds);
        }

        public final String getEc2KeyPair() {
            return ec2KeyPair;
        }

        @Override
        public final Builder ec2KeyPair(String ec2KeyPair) {
            this.ec2KeyPair = ec2KeyPair;
            return this;
        }

        public final void setEc2KeyPair(String ec2KeyPair) {
            this.ec2KeyPair = ec2KeyPair;
        }

        public final String getInstanceRole() {
            return instanceRole;
        }

        @Override
        public final Builder instanceRole(String instanceRole) {
            this.instanceRole = instanceRole;
            return this;
        }

        public final void setInstanceRole(String instanceRole) {
            this.instanceRole = instanceRole;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagsMapCopier.copy(tags);
            return this;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagsMapCopier.copy(tags);
        }

        public final String getPlacementGroup() {
            return placementGroup;
        }

        @Override
        public final Builder placementGroup(String placementGroup) {
            this.placementGroup = placementGroup;
            return this;
        }

        public final void setPlacementGroup(String placementGroup) {
            this.placementGroup = placementGroup;
        }

        public final Integer getBidPercentage() {
            return bidPercentage;
        }

        @Override
        public final Builder bidPercentage(Integer bidPercentage) {
            this.bidPercentage = bidPercentage;
            return this;
        }

        public final void setBidPercentage(Integer bidPercentage) {
            this.bidPercentage = bidPercentage;
        }

        public final String getSpotIamFleetRole() {
            return spotIamFleetRole;
        }

        @Override
        public final Builder spotIamFleetRole(String spotIamFleetRole) {
            this.spotIamFleetRole = spotIamFleetRole;
            return this;
        }

        public final void setSpotIamFleetRole(String spotIamFleetRole) {
            this.spotIamFleetRole = spotIamFleetRole;
        }

        public final LaunchTemplateSpecification.Builder getLaunchTemplate() {
            return launchTemplate != null ? launchTemplate.toBuilder() : null;
        }

        @Override
        public final Builder launchTemplate(LaunchTemplateSpecification launchTemplate) {
            this.launchTemplate = launchTemplate;
            return this;
        }

        public final void setLaunchTemplate(LaunchTemplateSpecification.BuilderImpl launchTemplate) {
            this.launchTemplate = launchTemplate != null ? launchTemplate.build() : null;
        }

        public final Collection<Ec2Configuration.Builder> getEc2Configuration() {
            if (ec2Configuration instanceof SdkAutoConstructList) {
                return null;
            }
            return ec2Configuration != null ? ec2Configuration.stream().map(Ec2Configuration::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder ec2Configuration(Collection<Ec2Configuration> ec2Configuration) {
            this.ec2Configuration = Ec2ConfigurationListCopier.copy(ec2Configuration);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ec2Configuration(Ec2Configuration... ec2Configuration) {
            ec2Configuration(Arrays.asList(ec2Configuration));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ec2Configuration(Consumer<Ec2Configuration.Builder>... ec2Configuration) {
            ec2Configuration(Stream.of(ec2Configuration).map(c -> Ec2Configuration.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setEc2Configuration(Collection<Ec2Configuration.BuilderImpl> ec2Configuration) {
            this.ec2Configuration = Ec2ConfigurationListCopier.copyFromBuilder(ec2Configuration);
        }

        @Override
        public ComputeResource build() {
            return new ComputeResource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
