/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.batch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Linux-specific modifications that are applied to the container, such as details for device mappings.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LinuxParameters implements SdkPojo, Serializable, ToCopyableBuilder<LinuxParameters.Builder, LinuxParameters> {
    private static final SdkField<List<Device>> DEVICES_FIELD = SdkField
            .<List<Device>> builder(MarshallingType.LIST)
            .memberName("devices")
            .getter(getter(LinuxParameters::devices))
            .setter(setter(Builder::devices))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("devices").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Device> builder(MarshallingType.SDK_POJO)
                                            .constructor(Device::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> INIT_PROCESS_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("initProcessEnabled").getter(getter(LinuxParameters::initProcessEnabled))
            .setter(setter(Builder::initProcessEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("initProcessEnabled").build())
            .build();

    private static final SdkField<Integer> SHARED_MEMORY_SIZE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("sharedMemorySize").getter(getter(LinuxParameters::sharedMemorySize))
            .setter(setter(Builder::sharedMemorySize))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sharedMemorySize").build()).build();

    private static final SdkField<List<Tmpfs>> TMPFS_FIELD = SdkField
            .<List<Tmpfs>> builder(MarshallingType.LIST)
            .memberName("tmpfs")
            .getter(getter(LinuxParameters::tmpfs))
            .setter(setter(Builder::tmpfs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tmpfs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tmpfs> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tmpfs::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> MAX_SWAP_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("maxSwap").getter(getter(LinuxParameters::maxSwap)).setter(setter(Builder::maxSwap))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maxSwap").build()).build();

    private static final SdkField<Integer> SWAPPINESS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("swappiness").getter(getter(LinuxParameters::swappiness)).setter(setter(Builder::swappiness))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("swappiness").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DEVICES_FIELD,
            INIT_PROCESS_ENABLED_FIELD, SHARED_MEMORY_SIZE_FIELD, TMPFS_FIELD, MAX_SWAP_FIELD, SWAPPINESS_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<Device> devices;

    private final Boolean initProcessEnabled;

    private final Integer sharedMemorySize;

    private final List<Tmpfs> tmpfs;

    private final Integer maxSwap;

    private final Integer swappiness;

    private LinuxParameters(BuilderImpl builder) {
        this.devices = builder.devices;
        this.initProcessEnabled = builder.initProcessEnabled;
        this.sharedMemorySize = builder.sharedMemorySize;
        this.tmpfs = builder.tmpfs;
        this.maxSwap = builder.maxSwap;
        this.swappiness = builder.swappiness;
    }

    /**
     * Returns true if the Devices property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasDevices() {
        return devices != null && !(devices instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Any host devices to expose to the container. This parameter maps to <code>Devices</code> in the <a
     * href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the <a
     * href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--device</code> option to <a
     * href="https://docs.docker.com/engine/reference/run/">docker run</a>.
     * </p>
     * <note>
     * <p>
     * This parameter isn't applicable to jobs running on Fargate resources and shouldn't be provided.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasDevices()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Any host devices to expose to the container. This parameter maps to <code>Devices</code> in the <a
     *         href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the
     *         <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--device</code>
     *         option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.</p> <note>
     *         <p>
     *         This parameter isn't applicable to jobs running on Fargate resources and shouldn't be provided.
     *         </p>
     */
    public List<Device> devices() {
        return devices;
    }

    /**
     * <p>
     * If true, run an <code>init</code> process inside the container that forwards signals and reaps processes. This
     * parameter maps to the <code>--init</code> option to <a
     * href="https://docs.docker.com/engine/reference/run/">docker run</a>. This parameter requires version 1.25 of the
     * Docker Remote API or greater on your container instance. To check the Docker Remote API version on your container
     * instance, log into your container instance and run the following command:
     * <code>sudo docker version | grep "Server API version"</code>
     * </p>
     * 
     * @return If true, run an <code>init</code> process inside the container that forwards signals and reaps processes.
     *         This parameter maps to the <code>--init</code> option to <a
     *         href="https://docs.docker.com/engine/reference/run/">docker run</a>. This parameter requires version 1.25
     *         of the Docker Remote API or greater on your container instance. To check the Docker Remote API version on
     *         your container instance, log into your container instance and run the following command:
     *         <code>sudo docker version | grep "Server API version"</code>
     */
    public Boolean initProcessEnabled() {
        return initProcessEnabled;
    }

    /**
     * <p>
     * The value for the size (in MiB) of the <code>/dev/shm</code> volume. This parameter maps to the
     * <code>--shm-size</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
     * </p>
     * <note>
     * <p>
     * This parameter isn't applicable to jobs running on Fargate resources and shouldn't be provided.
     * </p>
     * </note>
     * 
     * @return The value for the size (in MiB) of the <code>/dev/shm</code> volume. This parameter maps to the
     *         <code>--shm-size</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
     *         run</a>.</p> <note>
     *         <p>
     *         This parameter isn't applicable to jobs running on Fargate resources and shouldn't be provided.
     *         </p>
     */
    public Integer sharedMemorySize() {
        return sharedMemorySize;
    }

    /**
     * Returns true if the Tmpfs property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTmpfs() {
        return tmpfs != null && !(tmpfs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The container path, mount options, and size (in MiB) of the tmpfs mount. This parameter maps to the
     * <code>--tmpfs</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
     * </p>
     * <note>
     * <p>
     * This parameter isn't applicable to jobs running on Fargate resources and shouldn't be provided.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTmpfs()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The container path, mount options, and size (in MiB) of the tmpfs mount. This parameter maps to the
     *         <code>--tmpfs</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
     *         run</a>.</p> <note>
     *         <p>
     *         This parameter isn't applicable to jobs running on Fargate resources and shouldn't be provided.
     *         </p>
     */
    public List<Tmpfs> tmpfs() {
        return tmpfs;
    }

    /**
     * <p>
     * The total amount of swap memory (in MiB) a container can use. This parameter is translated to the
     * <code>--memory-swap</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a> where
     * the value is the sum of the container memory plus the <code>maxSwap</code> value. For more information, see <a
     * href="https://docs.docker.com/config/containers/resource_constraints/#--memory-swap-details">
     * <code>--memory-swap</code> details</a> in the Docker documentation.
     * </p>
     * <p>
     * If a <code>maxSwap</code> value of <code>0</code> is specified, the container doesn't use swap. Accepted values
     * are <code>0</code> or any positive integer. If the <code>maxSwap</code> parameter is omitted, the container
     * doesn't use the swap configuration for the container instance it is running on. A <code>maxSwap</code> value must
     * be set for the <code>swappiness</code> parameter to be used.
     * </p>
     * <note>
     * <p>
     * This parameter isn't applicable to jobs running on Fargate resources and shouldn't be provided.
     * </p>
     * </note>
     * 
     * @return The total amount of swap memory (in MiB) a container can use. This parameter is translated to the
     *         <code>--memory-swap</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
     *         run</a> where the value is the sum of the container memory plus the <code>maxSwap</code> value. For more
     *         information, see <a
     *         href="https://docs.docker.com/config/containers/resource_constraints/#--memory-swap-details">
     *         <code>--memory-swap</code> details</a> in the Docker documentation.</p>
     *         <p>
     *         If a <code>maxSwap</code> value of <code>0</code> is specified, the container doesn't use swap. Accepted
     *         values are <code>0</code> or any positive integer. If the <code>maxSwap</code> parameter is omitted, the
     *         container doesn't use the swap configuration for the container instance it is running on. A
     *         <code>maxSwap</code> value must be set for the <code>swappiness</code> parameter to be used.
     *         </p>
     *         <note>
     *         <p>
     *         This parameter isn't applicable to jobs running on Fargate resources and shouldn't be provided.
     *         </p>
     */
    public Integer maxSwap() {
        return maxSwap;
    }

    /**
     * <p>
     * This allows you to tune a container's memory swappiness behavior. A <code>swappiness</code> value of
     * <code>0</code> causes swapping not to happen unless absolutely necessary. A <code>swappiness</code> value of
     * <code>100</code> causes pages to be swapped very aggressively. Accepted values are whole numbers between
     * <code>0</code> and <code>100</code>. If the <code>swappiness</code> parameter isn't specified, a default value of
     * <code>60</code> is used. If a value isn't specified for <code>maxSwap</code> then this parameter is ignored. This
     * parameter maps to the <code>--memory-swappiness</code> option to <a
     * href="https://docs.docker.com/engine/reference/run/">docker run</a>.
     * </p>
     * <note>
     * <p>
     * This parameter isn't applicable to jobs running on Fargate resources and shouldn't be provided.
     * </p>
     * </note>
     * 
     * @return This allows you to tune a container's memory swappiness behavior. A <code>swappiness</code> value of
     *         <code>0</code> causes swapping not to happen unless absolutely necessary. A <code>swappiness</code> value
     *         of <code>100</code> causes pages to be swapped very aggressively. Accepted values are whole numbers
     *         between <code>0</code> and <code>100</code>. If the <code>swappiness</code> parameter isn't specified, a
     *         default value of <code>60</code> is used. If a value isn't specified for <code>maxSwap</code> then this
     *         parameter is ignored. This parameter maps to the <code>--memory-swappiness</code> option to <a
     *         href="https://docs.docker.com/engine/reference/run/">docker run</a>.</p> <note>
     *         <p>
     *         This parameter isn't applicable to jobs running on Fargate resources and shouldn't be provided.
     *         </p>
     */
    public Integer swappiness() {
        return swappiness;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasDevices() ? devices() : null);
        hashCode = 31 * hashCode + Objects.hashCode(initProcessEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(sharedMemorySize());
        hashCode = 31 * hashCode + Objects.hashCode(hasTmpfs() ? tmpfs() : null);
        hashCode = 31 * hashCode + Objects.hashCode(maxSwap());
        hashCode = 31 * hashCode + Objects.hashCode(swappiness());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LinuxParameters)) {
            return false;
        }
        LinuxParameters other = (LinuxParameters) obj;
        return hasDevices() == other.hasDevices() && Objects.equals(devices(), other.devices())
                && Objects.equals(initProcessEnabled(), other.initProcessEnabled())
                && Objects.equals(sharedMemorySize(), other.sharedMemorySize()) && hasTmpfs() == other.hasTmpfs()
                && Objects.equals(tmpfs(), other.tmpfs()) && Objects.equals(maxSwap(), other.maxSwap())
                && Objects.equals(swappiness(), other.swappiness());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("LinuxParameters").add("Devices", hasDevices() ? devices() : null)
                .add("InitProcessEnabled", initProcessEnabled()).add("SharedMemorySize", sharedMemorySize())
                .add("Tmpfs", hasTmpfs() ? tmpfs() : null).add("MaxSwap", maxSwap()).add("Swappiness", swappiness()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "devices":
            return Optional.ofNullable(clazz.cast(devices()));
        case "initProcessEnabled":
            return Optional.ofNullable(clazz.cast(initProcessEnabled()));
        case "sharedMemorySize":
            return Optional.ofNullable(clazz.cast(sharedMemorySize()));
        case "tmpfs":
            return Optional.ofNullable(clazz.cast(tmpfs()));
        case "maxSwap":
            return Optional.ofNullable(clazz.cast(maxSwap()));
        case "swappiness":
            return Optional.ofNullable(clazz.cast(swappiness()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LinuxParameters, T> g) {
        return obj -> g.apply((LinuxParameters) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LinuxParameters> {
        /**
         * <p>
         * Any host devices to expose to the container. This parameter maps to <code>Devices</code> in the <a
         * href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the <a
         * href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--device</code> option
         * to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
         * </p>
         * <note>
         * <p>
         * This parameter isn't applicable to jobs running on Fargate resources and shouldn't be provided.
         * </p>
         * </note>
         * 
         * @param devices
         *        Any host devices to expose to the container. This parameter maps to <code>Devices</code> in the <a
         *        href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of
         *        the <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the
         *        <code>--device</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
         *        run</a>.</p> <note>
         *        <p>
         *        This parameter isn't applicable to jobs running on Fargate resources and shouldn't be provided.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder devices(Collection<Device> devices);

        /**
         * <p>
         * Any host devices to expose to the container. This parameter maps to <code>Devices</code> in the <a
         * href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the <a
         * href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--device</code> option
         * to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
         * </p>
         * <note>
         * <p>
         * This parameter isn't applicable to jobs running on Fargate resources and shouldn't be provided.
         * </p>
         * </note>
         * 
         * @param devices
         *        Any host devices to expose to the container. This parameter maps to <code>Devices</code> in the <a
         *        href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of
         *        the <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the
         *        <code>--device</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
         *        run</a>.</p> <note>
         *        <p>
         *        This parameter isn't applicable to jobs running on Fargate resources and shouldn't be provided.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder devices(Device... devices);

        /**
         * <p>
         * Any host devices to expose to the container. This parameter maps to <code>Devices</code> in the <a
         * href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the <a
         * href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--device</code> option
         * to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
         * </p>
         * <note>
         * <p>
         * This parameter isn't applicable to jobs running on Fargate resources and shouldn't be provided.
         * </p>
         * </note> This is a convenience that creates an instance of the {@link List<Device>.Builder} avoiding the need
         * to create one manually via {@link List<Device>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Device>.Builder#build()} is called immediately and its
         * result is passed to {@link #devices(List<Device>)}.
         * 
         * @param devices
         *        a consumer that will call methods on {@link List<Device>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #devices(List<Device>)
         */
        Builder devices(Consumer<Device.Builder>... devices);

        /**
         * <p>
         * If true, run an <code>init</code> process inside the container that forwards signals and reaps processes.
         * This parameter maps to the <code>--init</code> option to <a
         * href="https://docs.docker.com/engine/reference/run/">docker run</a>. This parameter requires version 1.25 of
         * the Docker Remote API or greater on your container instance. To check the Docker Remote API version on your
         * container instance, log into your container instance and run the following command:
         * <code>sudo docker version | grep "Server API version"</code>
         * </p>
         * 
         * @param initProcessEnabled
         *        If true, run an <code>init</code> process inside the container that forwards signals and reaps
         *        processes. This parameter maps to the <code>--init</code> option to <a
         *        href="https://docs.docker.com/engine/reference/run/">docker run</a>. This parameter requires version
         *        1.25 of the Docker Remote API or greater on your container instance. To check the Docker Remote API
         *        version on your container instance, log into your container instance and run the following command:
         *        <code>sudo docker version | grep "Server API version"</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder initProcessEnabled(Boolean initProcessEnabled);

        /**
         * <p>
         * The value for the size (in MiB) of the <code>/dev/shm</code> volume. This parameter maps to the
         * <code>--shm-size</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
         * </p>
         * <note>
         * <p>
         * This parameter isn't applicable to jobs running on Fargate resources and shouldn't be provided.
         * </p>
         * </note>
         * 
         * @param sharedMemorySize
         *        The value for the size (in MiB) of the <code>/dev/shm</code> volume. This parameter maps to the
         *        <code>--shm-size</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
         *        run</a>.</p> <note>
         *        <p>
         *        This parameter isn't applicable to jobs running on Fargate resources and shouldn't be provided.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sharedMemorySize(Integer sharedMemorySize);

        /**
         * <p>
         * The container path, mount options, and size (in MiB) of the tmpfs mount. This parameter maps to the
         * <code>--tmpfs</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
         * </p>
         * <note>
         * <p>
         * This parameter isn't applicable to jobs running on Fargate resources and shouldn't be provided.
         * </p>
         * </note>
         * 
         * @param tmpfs
         *        The container path, mount options, and size (in MiB) of the tmpfs mount. This parameter maps to the
         *        <code>--tmpfs</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
         *        run</a>.</p> <note>
         *        <p>
         *        This parameter isn't applicable to jobs running on Fargate resources and shouldn't be provided.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tmpfs(Collection<Tmpfs> tmpfs);

        /**
         * <p>
         * The container path, mount options, and size (in MiB) of the tmpfs mount. This parameter maps to the
         * <code>--tmpfs</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
         * </p>
         * <note>
         * <p>
         * This parameter isn't applicable to jobs running on Fargate resources and shouldn't be provided.
         * </p>
         * </note>
         * 
         * @param tmpfs
         *        The container path, mount options, and size (in MiB) of the tmpfs mount. This parameter maps to the
         *        <code>--tmpfs</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
         *        run</a>.</p> <note>
         *        <p>
         *        This parameter isn't applicable to jobs running on Fargate resources and shouldn't be provided.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tmpfs(Tmpfs... tmpfs);

        /**
         * <p>
         * The container path, mount options, and size (in MiB) of the tmpfs mount. This parameter maps to the
         * <code>--tmpfs</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
         * </p>
         * <note>
         * <p>
         * This parameter isn't applicable to jobs running on Fargate resources and shouldn't be provided.
         * </p>
         * </note> This is a convenience that creates an instance of the {@link List<Tmpfs>.Builder} avoiding the need
         * to create one manually via {@link List<Tmpfs>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tmpfs>.Builder#build()} is called immediately and its result
         * is passed to {@link #tmpfs(List<Tmpfs>)}.
         * 
         * @param tmpfs
         *        a consumer that will call methods on {@link List<Tmpfs>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tmpfs(List<Tmpfs>)
         */
        Builder tmpfs(Consumer<Tmpfs.Builder>... tmpfs);

        /**
         * <p>
         * The total amount of swap memory (in MiB) a container can use. This parameter is translated to the
         * <code>--memory-swap</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>
         * where the value is the sum of the container memory plus the <code>maxSwap</code> value. For more information,
         * see <a href="https://docs.docker.com/config/containers/resource_constraints/#--memory-swap-details">
         * <code>--memory-swap</code> details</a> in the Docker documentation.
         * </p>
         * <p>
         * If a <code>maxSwap</code> value of <code>0</code> is specified, the container doesn't use swap. Accepted
         * values are <code>0</code> or any positive integer. If the <code>maxSwap</code> parameter is omitted, the
         * container doesn't use the swap configuration for the container instance it is running on. A
         * <code>maxSwap</code> value must be set for the <code>swappiness</code> parameter to be used.
         * </p>
         * <note>
         * <p>
         * This parameter isn't applicable to jobs running on Fargate resources and shouldn't be provided.
         * </p>
         * </note>
         * 
         * @param maxSwap
         *        The total amount of swap memory (in MiB) a container can use. This parameter is translated to the
         *        <code>--memory-swap</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
         *        run</a> where the value is the sum of the container memory plus the <code>maxSwap</code> value. For
         *        more information, see <a
         *        href="https://docs.docker.com/config/containers/resource_constraints/#--memory-swap-details">
         *        <code>--memory-swap</code> details</a> in the Docker documentation.</p>
         *        <p>
         *        If a <code>maxSwap</code> value of <code>0</code> is specified, the container doesn't use swap.
         *        Accepted values are <code>0</code> or any positive integer. If the <code>maxSwap</code> parameter is
         *        omitted, the container doesn't use the swap configuration for the container instance it is running on.
         *        A <code>maxSwap</code> value must be set for the <code>swappiness</code> parameter to be used.
         *        </p>
         *        <note>
         *        <p>
         *        This parameter isn't applicable to jobs running on Fargate resources and shouldn't be provided.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxSwap(Integer maxSwap);

        /**
         * <p>
         * This allows you to tune a container's memory swappiness behavior. A <code>swappiness</code> value of
         * <code>0</code> causes swapping not to happen unless absolutely necessary. A <code>swappiness</code> value of
         * <code>100</code> causes pages to be swapped very aggressively. Accepted values are whole numbers between
         * <code>0</code> and <code>100</code>. If the <code>swappiness</code> parameter isn't specified, a default
         * value of <code>60</code> is used. If a value isn't specified for <code>maxSwap</code> then this parameter is
         * ignored. This parameter maps to the <code>--memory-swappiness</code> option to <a
         * href="https://docs.docker.com/engine/reference/run/">docker run</a>.
         * </p>
         * <note>
         * <p>
         * This parameter isn't applicable to jobs running on Fargate resources and shouldn't be provided.
         * </p>
         * </note>
         * 
         * @param swappiness
         *        This allows you to tune a container's memory swappiness behavior. A <code>swappiness</code> value of
         *        <code>0</code> causes swapping not to happen unless absolutely necessary. A <code>swappiness</code>
         *        value of <code>100</code> causes pages to be swapped very aggressively. Accepted values are whole
         *        numbers between <code>0</code> and <code>100</code>. If the <code>swappiness</code> parameter isn't
         *        specified, a default value of <code>60</code> is used. If a value isn't specified for
         *        <code>maxSwap</code> then this parameter is ignored. This parameter maps to the
         *        <code>--memory-swappiness</code> option to <a
         *        href="https://docs.docker.com/engine/reference/run/">docker run</a>.</p> <note>
         *        <p>
         *        This parameter isn't applicable to jobs running on Fargate resources and shouldn't be provided.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder swappiness(Integer swappiness);
    }

    static final class BuilderImpl implements Builder {
        private List<Device> devices = DefaultSdkAutoConstructList.getInstance();

        private Boolean initProcessEnabled;

        private Integer sharedMemorySize;

        private List<Tmpfs> tmpfs = DefaultSdkAutoConstructList.getInstance();

        private Integer maxSwap;

        private Integer swappiness;

        private BuilderImpl() {
        }

        private BuilderImpl(LinuxParameters model) {
            devices(model.devices);
            initProcessEnabled(model.initProcessEnabled);
            sharedMemorySize(model.sharedMemorySize);
            tmpfs(model.tmpfs);
            maxSwap(model.maxSwap);
            swappiness(model.swappiness);
        }

        public final Collection<Device.Builder> getDevices() {
            if (devices instanceof SdkAutoConstructList) {
                return null;
            }
            return devices != null ? devices.stream().map(Device::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder devices(Collection<Device> devices) {
            this.devices = DevicesListCopier.copy(devices);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder devices(Device... devices) {
            devices(Arrays.asList(devices));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder devices(Consumer<Device.Builder>... devices) {
            devices(Stream.of(devices).map(c -> Device.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setDevices(Collection<Device.BuilderImpl> devices) {
            this.devices = DevicesListCopier.copyFromBuilder(devices);
        }

        public final Boolean getInitProcessEnabled() {
            return initProcessEnabled;
        }

        @Override
        public final Builder initProcessEnabled(Boolean initProcessEnabled) {
            this.initProcessEnabled = initProcessEnabled;
            return this;
        }

        public final void setInitProcessEnabled(Boolean initProcessEnabled) {
            this.initProcessEnabled = initProcessEnabled;
        }

        public final Integer getSharedMemorySize() {
            return sharedMemorySize;
        }

        @Override
        public final Builder sharedMemorySize(Integer sharedMemorySize) {
            this.sharedMemorySize = sharedMemorySize;
            return this;
        }

        public final void setSharedMemorySize(Integer sharedMemorySize) {
            this.sharedMemorySize = sharedMemorySize;
        }

        public final Collection<Tmpfs.Builder> getTmpfs() {
            if (tmpfs instanceof SdkAutoConstructList) {
                return null;
            }
            return tmpfs != null ? tmpfs.stream().map(Tmpfs::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tmpfs(Collection<Tmpfs> tmpfs) {
            this.tmpfs = TmpfsListCopier.copy(tmpfs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tmpfs(Tmpfs... tmpfs) {
            tmpfs(Arrays.asList(tmpfs));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tmpfs(Consumer<Tmpfs.Builder>... tmpfs) {
            tmpfs(Stream.of(tmpfs).map(c -> Tmpfs.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTmpfs(Collection<Tmpfs.BuilderImpl> tmpfs) {
            this.tmpfs = TmpfsListCopier.copyFromBuilder(tmpfs);
        }

        public final Integer getMaxSwap() {
            return maxSwap;
        }

        @Override
        public final Builder maxSwap(Integer maxSwap) {
            this.maxSwap = maxSwap;
            return this;
        }

        public final void setMaxSwap(Integer maxSwap) {
            this.maxSwap = maxSwap;
        }

        public final Integer getSwappiness() {
            return swappiness;
        }

        @Override
        public final Builder swappiness(Integer swappiness) {
            this.swappiness = swappiness;
            return this;
        }

        public final void setSwappiness(Integer swappiness) {
            this.swappiness = swappiness;
        }

        @Override
        public LinuxParameters build() {
            return new LinuxParameters(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
