/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.batch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Log configuration options to send to a custom log driver for the container.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LogConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<LogConfiguration.Builder, LogConfiguration> {
    private static final SdkField<String> LOG_DRIVER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("logDriver").getter(getter(LogConfiguration::logDriverAsString)).setter(setter(Builder::logDriver))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("logDriver").build()).build();

    private static final SdkField<Map<String, String>> OPTIONS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("options")
            .getter(getter(LogConfiguration::options))
            .setter(setter(Builder::options))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("options").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<Secret>> SECRET_OPTIONS_FIELD = SdkField
            .<List<Secret>> builder(MarshallingType.LIST)
            .memberName("secretOptions")
            .getter(getter(LogConfiguration::secretOptions))
            .setter(setter(Builder::secretOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("secretOptions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Secret> builder(MarshallingType.SDK_POJO)
                                            .constructor(Secret::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(LOG_DRIVER_FIELD,
            OPTIONS_FIELD, SECRET_OPTIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String logDriver;

    private final Map<String, String> options;

    private final List<Secret> secretOptions;

    private LogConfiguration(BuilderImpl builder) {
        this.logDriver = builder.logDriver;
        this.options = builder.options;
        this.secretOptions = builder.secretOptions;
    }

    /**
     * <p>
     * The log driver to use for the container. The valid values listed for this parameter are log drivers that the
     * Amazon ECS container agent can communicate with by default.
     * </p>
     * <p>
     * The supported log drivers are <code>awslogs</code>, <code>fluentd</code>, <code>gelf</code>,
     * <code>json-file</code>, <code>journald</code>, <code>logentries</code>, <code>syslog</code>, and
     * <code>splunk</code>.
     * </p>
     * <note>
     * <p>
     * Jobs running on Fargate resources are restricted to the <code>awslogs</code> and <code>splunk</code> log drivers.
     * </p>
     * </note>
     * <dl>
     * <dt>awslogs</dt>
     * <dd>
     * <p>
     * Specifies the Amazon CloudWatch Logs logging driver. For more information, see <a
     * href="https://docs.aws.amazon.com/batch/latest/userguide/using_awslogs.html">Using the awslogs Log Driver</a> in
     * the <i>AWS Batch User Guide</i> and <a href="https://docs.docker.com/config/containers/logging/awslogs/">Amazon
     * CloudWatch Logs logging driver</a> in the Docker documentation.
     * </p>
     * </dd>
     * <dt>fluentd</dt>
     * <dd>
     * <p>
     * Specifies the Fluentd logging driver. For more information, including usage and options, see <a
     * href="https://docs.docker.com/config/containers/logging/fluentd/">Fluentd logging driver</a> in the Docker
     * documentation.
     * </p>
     * </dd>
     * <dt>gelf</dt>
     * <dd>
     * <p>
     * Specifies the Graylog Extended Format (GELF) logging driver. For more information, including usage and options,
     * see <a href="https://docs.docker.com/config/containers/logging/gelf/">Graylog Extended Format logging driver</a>
     * in the Docker documentation.
     * </p>
     * </dd>
     * <dt>journald</dt>
     * <dd>
     * <p>
     * Specifies the journald logging driver. For more information, including usage and options, see <a
     * href="https://docs.docker.com/config/containers/logging/journald/">Journald logging driver</a> in the Docker
     * documentation.
     * </p>
     * </dd>
     * <dt>json-file</dt>
     * <dd>
     * <p>
     * Specifies the JSON file logging driver. For more information, including usage and options, see <a
     * href="https://docs.docker.com/config/containers/logging/json-file/">JSON File logging driver</a> in the Docker
     * documentation.
     * </p>
     * </dd>
     * <dt>splunk</dt>
     * <dd>
     * <p>
     * Specifies the Splunk logging driver. For more information, including usage and options, see <a
     * href="https://docs.docker.com/config/containers/logging/splunk/">Splunk logging driver</a> in the Docker
     * documentation.
     * </p>
     * </dd>
     * <dt>syslog</dt>
     * <dd>
     * <p>
     * Specifies the syslog logging driver. For more information, including usage and options, see <a
     * href="https://docs.docker.com/config/containers/logging/syslog/">Syslog logging driver</a> in the Docker
     * documentation.
     * </p>
     * </dd>
     * </dl>
     * <note>
     * <p>
     * If you have a custom driver that isn't listed earlier that you want to work with the Amazon ECS container agent,
     * you can fork the Amazon ECS container agent project that is <a
     * href="https://github.com/aws/amazon-ecs-agent">available on GitHub</a> and customize it to work with that driver.
     * We encourage you to submit pull requests for changes that you want to have included. However, Amazon Web Services
     * doesn't currently support running modified copies of this software.
     * </p>
     * </note>
     * <p>
     * This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check the
     * Docker Remote API version on your container instance, log into your container instance and run the following
     * command: <code>sudo docker version | grep "Server API version"</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #logDriver} will
     * return {@link LogDriver#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #logDriverAsString}.
     * </p>
     * 
     * @return The log driver to use for the container. The valid values listed for this parameter are log drivers that
     *         the Amazon ECS container agent can communicate with by default.</p>
     *         <p>
     *         The supported log drivers are <code>awslogs</code>, <code>fluentd</code>, <code>gelf</code>,
     *         <code>json-file</code>, <code>journald</code>, <code>logentries</code>, <code>syslog</code>, and
     *         <code>splunk</code>.
     *         </p>
     *         <note>
     *         <p>
     *         Jobs running on Fargate resources are restricted to the <code>awslogs</code> and <code>splunk</code> log
     *         drivers.
     *         </p>
     *         </note>
     *         <dl>
     *         <dt>awslogs</dt>
     *         <dd>
     *         <p>
     *         Specifies the Amazon CloudWatch Logs logging driver. For more information, see <a
     *         href="https://docs.aws.amazon.com/batch/latest/userguide/using_awslogs.html">Using the awslogs Log
     *         Driver</a> in the <i>AWS Batch User Guide</i> and <a
     *         href="https://docs.docker.com/config/containers/logging/awslogs/">Amazon CloudWatch Logs logging
     *         driver</a> in the Docker documentation.
     *         </p>
     *         </dd>
     *         <dt>fluentd</dt>
     *         <dd>
     *         <p>
     *         Specifies the Fluentd logging driver. For more information, including usage and options, see <a
     *         href="https://docs.docker.com/config/containers/logging/fluentd/">Fluentd logging driver</a> in the
     *         Docker documentation.
     *         </p>
     *         </dd>
     *         <dt>gelf</dt>
     *         <dd>
     *         <p>
     *         Specifies the Graylog Extended Format (GELF) logging driver. For more information, including usage and
     *         options, see <a href="https://docs.docker.com/config/containers/logging/gelf/">Graylog Extended Format
     *         logging driver</a> in the Docker documentation.
     *         </p>
     *         </dd>
     *         <dt>journald</dt>
     *         <dd>
     *         <p>
     *         Specifies the journald logging driver. For more information, including usage and options, see <a
     *         href="https://docs.docker.com/config/containers/logging/journald/">Journald logging driver</a> in the
     *         Docker documentation.
     *         </p>
     *         </dd>
     *         <dt>json-file</dt>
     *         <dd>
     *         <p>
     *         Specifies the JSON file logging driver. For more information, including usage and options, see <a
     *         href="https://docs.docker.com/config/containers/logging/json-file/">JSON File logging driver</a> in the
     *         Docker documentation.
     *         </p>
     *         </dd>
     *         <dt>splunk</dt>
     *         <dd>
     *         <p>
     *         Specifies the Splunk logging driver. For more information, including usage and options, see <a
     *         href="https://docs.docker.com/config/containers/logging/splunk/">Splunk logging driver</a> in the Docker
     *         documentation.
     *         </p>
     *         </dd>
     *         <dt>syslog</dt>
     *         <dd>
     *         <p>
     *         Specifies the syslog logging driver. For more information, including usage and options, see <a
     *         href="https://docs.docker.com/config/containers/logging/syslog/">Syslog logging driver</a> in the Docker
     *         documentation.
     *         </p>
     *         </dd>
     *         </dl>
     *         <note>
     *         <p>
     *         If you have a custom driver that isn't listed earlier that you want to work with the Amazon ECS container
     *         agent, you can fork the Amazon ECS container agent project that is <a
     *         href="https://github.com/aws/amazon-ecs-agent">available on GitHub</a> and customize it to work with that
     *         driver. We encourage you to submit pull requests for changes that you want to have included. However,
     *         Amazon Web Services doesn't currently support running modified copies of this software.
     *         </p>
     *         </note>
     *         <p>
     *         This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To
     *         check the Docker Remote API version on your container instance, log into your container instance and run
     *         the following command: <code>sudo docker version | grep "Server API version"</code>
     * @see LogDriver
     */
    public final LogDriver logDriver() {
        return LogDriver.fromValue(logDriver);
    }

    /**
     * <p>
     * The log driver to use for the container. The valid values listed for this parameter are log drivers that the
     * Amazon ECS container agent can communicate with by default.
     * </p>
     * <p>
     * The supported log drivers are <code>awslogs</code>, <code>fluentd</code>, <code>gelf</code>,
     * <code>json-file</code>, <code>journald</code>, <code>logentries</code>, <code>syslog</code>, and
     * <code>splunk</code>.
     * </p>
     * <note>
     * <p>
     * Jobs running on Fargate resources are restricted to the <code>awslogs</code> and <code>splunk</code> log drivers.
     * </p>
     * </note>
     * <dl>
     * <dt>awslogs</dt>
     * <dd>
     * <p>
     * Specifies the Amazon CloudWatch Logs logging driver. For more information, see <a
     * href="https://docs.aws.amazon.com/batch/latest/userguide/using_awslogs.html">Using the awslogs Log Driver</a> in
     * the <i>AWS Batch User Guide</i> and <a href="https://docs.docker.com/config/containers/logging/awslogs/">Amazon
     * CloudWatch Logs logging driver</a> in the Docker documentation.
     * </p>
     * </dd>
     * <dt>fluentd</dt>
     * <dd>
     * <p>
     * Specifies the Fluentd logging driver. For more information, including usage and options, see <a
     * href="https://docs.docker.com/config/containers/logging/fluentd/">Fluentd logging driver</a> in the Docker
     * documentation.
     * </p>
     * </dd>
     * <dt>gelf</dt>
     * <dd>
     * <p>
     * Specifies the Graylog Extended Format (GELF) logging driver. For more information, including usage and options,
     * see <a href="https://docs.docker.com/config/containers/logging/gelf/">Graylog Extended Format logging driver</a>
     * in the Docker documentation.
     * </p>
     * </dd>
     * <dt>journald</dt>
     * <dd>
     * <p>
     * Specifies the journald logging driver. For more information, including usage and options, see <a
     * href="https://docs.docker.com/config/containers/logging/journald/">Journald logging driver</a> in the Docker
     * documentation.
     * </p>
     * </dd>
     * <dt>json-file</dt>
     * <dd>
     * <p>
     * Specifies the JSON file logging driver. For more information, including usage and options, see <a
     * href="https://docs.docker.com/config/containers/logging/json-file/">JSON File logging driver</a> in the Docker
     * documentation.
     * </p>
     * </dd>
     * <dt>splunk</dt>
     * <dd>
     * <p>
     * Specifies the Splunk logging driver. For more information, including usage and options, see <a
     * href="https://docs.docker.com/config/containers/logging/splunk/">Splunk logging driver</a> in the Docker
     * documentation.
     * </p>
     * </dd>
     * <dt>syslog</dt>
     * <dd>
     * <p>
     * Specifies the syslog logging driver. For more information, including usage and options, see <a
     * href="https://docs.docker.com/config/containers/logging/syslog/">Syslog logging driver</a> in the Docker
     * documentation.
     * </p>
     * </dd>
     * </dl>
     * <note>
     * <p>
     * If you have a custom driver that isn't listed earlier that you want to work with the Amazon ECS container agent,
     * you can fork the Amazon ECS container agent project that is <a
     * href="https://github.com/aws/amazon-ecs-agent">available on GitHub</a> and customize it to work with that driver.
     * We encourage you to submit pull requests for changes that you want to have included. However, Amazon Web Services
     * doesn't currently support running modified copies of this software.
     * </p>
     * </note>
     * <p>
     * This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check the
     * Docker Remote API version on your container instance, log into your container instance and run the following
     * command: <code>sudo docker version | grep "Server API version"</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #logDriver} will
     * return {@link LogDriver#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #logDriverAsString}.
     * </p>
     * 
     * @return The log driver to use for the container. The valid values listed for this parameter are log drivers that
     *         the Amazon ECS container agent can communicate with by default.</p>
     *         <p>
     *         The supported log drivers are <code>awslogs</code>, <code>fluentd</code>, <code>gelf</code>,
     *         <code>json-file</code>, <code>journald</code>, <code>logentries</code>, <code>syslog</code>, and
     *         <code>splunk</code>.
     *         </p>
     *         <note>
     *         <p>
     *         Jobs running on Fargate resources are restricted to the <code>awslogs</code> and <code>splunk</code> log
     *         drivers.
     *         </p>
     *         </note>
     *         <dl>
     *         <dt>awslogs</dt>
     *         <dd>
     *         <p>
     *         Specifies the Amazon CloudWatch Logs logging driver. For more information, see <a
     *         href="https://docs.aws.amazon.com/batch/latest/userguide/using_awslogs.html">Using the awslogs Log
     *         Driver</a> in the <i>AWS Batch User Guide</i> and <a
     *         href="https://docs.docker.com/config/containers/logging/awslogs/">Amazon CloudWatch Logs logging
     *         driver</a> in the Docker documentation.
     *         </p>
     *         </dd>
     *         <dt>fluentd</dt>
     *         <dd>
     *         <p>
     *         Specifies the Fluentd logging driver. For more information, including usage and options, see <a
     *         href="https://docs.docker.com/config/containers/logging/fluentd/">Fluentd logging driver</a> in the
     *         Docker documentation.
     *         </p>
     *         </dd>
     *         <dt>gelf</dt>
     *         <dd>
     *         <p>
     *         Specifies the Graylog Extended Format (GELF) logging driver. For more information, including usage and
     *         options, see <a href="https://docs.docker.com/config/containers/logging/gelf/">Graylog Extended Format
     *         logging driver</a> in the Docker documentation.
     *         </p>
     *         </dd>
     *         <dt>journald</dt>
     *         <dd>
     *         <p>
     *         Specifies the journald logging driver. For more information, including usage and options, see <a
     *         href="https://docs.docker.com/config/containers/logging/journald/">Journald logging driver</a> in the
     *         Docker documentation.
     *         </p>
     *         </dd>
     *         <dt>json-file</dt>
     *         <dd>
     *         <p>
     *         Specifies the JSON file logging driver. For more information, including usage and options, see <a
     *         href="https://docs.docker.com/config/containers/logging/json-file/">JSON File logging driver</a> in the
     *         Docker documentation.
     *         </p>
     *         </dd>
     *         <dt>splunk</dt>
     *         <dd>
     *         <p>
     *         Specifies the Splunk logging driver. For more information, including usage and options, see <a
     *         href="https://docs.docker.com/config/containers/logging/splunk/">Splunk logging driver</a> in the Docker
     *         documentation.
     *         </p>
     *         </dd>
     *         <dt>syslog</dt>
     *         <dd>
     *         <p>
     *         Specifies the syslog logging driver. For more information, including usage and options, see <a
     *         href="https://docs.docker.com/config/containers/logging/syslog/">Syslog logging driver</a> in the Docker
     *         documentation.
     *         </p>
     *         </dd>
     *         </dl>
     *         <note>
     *         <p>
     *         If you have a custom driver that isn't listed earlier that you want to work with the Amazon ECS container
     *         agent, you can fork the Amazon ECS container agent project that is <a
     *         href="https://github.com/aws/amazon-ecs-agent">available on GitHub</a> and customize it to work with that
     *         driver. We encourage you to submit pull requests for changes that you want to have included. However,
     *         Amazon Web Services doesn't currently support running modified copies of this software.
     *         </p>
     *         </note>
     *         <p>
     *         This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To
     *         check the Docker Remote API version on your container instance, log into your container instance and run
     *         the following command: <code>sudo docker version | grep "Server API version"</code>
     * @see LogDriver
     */
    public final String logDriverAsString() {
        return logDriver;
    }

    /**
     * Returns true if the Options property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasOptions() {
        return options != null && !(options instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The configuration options to send to the log driver. This parameter requires version 1.19 of the Docker Remote
     * API or greater on your container instance. To check the Docker Remote API version on your container instance, log
     * into your container instance and run the following command:
     * <code>sudo docker version | grep "Server API version"</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasOptions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The configuration options to send to the log driver. This parameter requires version 1.19 of the Docker
     *         Remote API or greater on your container instance. To check the Docker Remote API version on your
     *         container instance, log into your container instance and run the following command:
     *         <code>sudo docker version | grep "Server API version"</code>
     */
    public final Map<String, String> options() {
        return options;
    }

    /**
     * Returns true if the SecretOptions property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasSecretOptions() {
        return secretOptions != null && !(secretOptions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The secrets to pass to the log configuration. For more information, see <a
     * href="https://docs.aws.amazon.com/batch/latest/userguide/specifying-sensitive-data.html">Specifying Sensitive
     * Data</a> in the <i>AWS Batch User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSecretOptions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The secrets to pass to the log configuration. For more information, see <a
     *         href="https://docs.aws.amazon.com/batch/latest/userguide/specifying-sensitive-data.html">Specifying
     *         Sensitive Data</a> in the <i>AWS Batch User Guide</i>.
     */
    public final List<Secret> secretOptions() {
        return secretOptions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(logDriverAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasOptions() ? options() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSecretOptions() ? secretOptions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LogConfiguration)) {
            return false;
        }
        LogConfiguration other = (LogConfiguration) obj;
        return Objects.equals(logDriverAsString(), other.logDriverAsString()) && hasOptions() == other.hasOptions()
                && Objects.equals(options(), other.options()) && hasSecretOptions() == other.hasSecretOptions()
                && Objects.equals(secretOptions(), other.secretOptions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("LogConfiguration").add("LogDriver", logDriverAsString())
                .add("Options", hasOptions() ? options() : null)
                .add("SecretOptions", hasSecretOptions() ? secretOptions() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "logDriver":
            return Optional.ofNullable(clazz.cast(logDriverAsString()));
        case "options":
            return Optional.ofNullable(clazz.cast(options()));
        case "secretOptions":
            return Optional.ofNullable(clazz.cast(secretOptions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LogConfiguration, T> g) {
        return obj -> g.apply((LogConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LogConfiguration> {
        /**
         * <p>
         * The log driver to use for the container. The valid values listed for this parameter are log drivers that the
         * Amazon ECS container agent can communicate with by default.
         * </p>
         * <p>
         * The supported log drivers are <code>awslogs</code>, <code>fluentd</code>, <code>gelf</code>,
         * <code>json-file</code>, <code>journald</code>, <code>logentries</code>, <code>syslog</code>, and
         * <code>splunk</code>.
         * </p>
         * <note>
         * <p>
         * Jobs running on Fargate resources are restricted to the <code>awslogs</code> and <code>splunk</code> log
         * drivers.
         * </p>
         * </note>
         * <dl>
         * <dt>awslogs</dt>
         * <dd>
         * <p>
         * Specifies the Amazon CloudWatch Logs logging driver. For more information, see <a
         * href="https://docs.aws.amazon.com/batch/latest/userguide/using_awslogs.html">Using the awslogs Log Driver</a>
         * in the <i>AWS Batch User Guide</i> and <a
         * href="https://docs.docker.com/config/containers/logging/awslogs/">Amazon CloudWatch Logs logging driver</a>
         * in the Docker documentation.
         * </p>
         * </dd>
         * <dt>fluentd</dt>
         * <dd>
         * <p>
         * Specifies the Fluentd logging driver. For more information, including usage and options, see <a
         * href="https://docs.docker.com/config/containers/logging/fluentd/">Fluentd logging driver</a> in the Docker
         * documentation.
         * </p>
         * </dd>
         * <dt>gelf</dt>
         * <dd>
         * <p>
         * Specifies the Graylog Extended Format (GELF) logging driver. For more information, including usage and
         * options, see <a href="https://docs.docker.com/config/containers/logging/gelf/">Graylog Extended Format
         * logging driver</a> in the Docker documentation.
         * </p>
         * </dd>
         * <dt>journald</dt>
         * <dd>
         * <p>
         * Specifies the journald logging driver. For more information, including usage and options, see <a
         * href="https://docs.docker.com/config/containers/logging/journald/">Journald logging driver</a> in the Docker
         * documentation.
         * </p>
         * </dd>
         * <dt>json-file</dt>
         * <dd>
         * <p>
         * Specifies the JSON file logging driver. For more information, including usage and options, see <a
         * href="https://docs.docker.com/config/containers/logging/json-file/">JSON File logging driver</a> in the
         * Docker documentation.
         * </p>
         * </dd>
         * <dt>splunk</dt>
         * <dd>
         * <p>
         * Specifies the Splunk logging driver. For more information, including usage and options, see <a
         * href="https://docs.docker.com/config/containers/logging/splunk/">Splunk logging driver</a> in the Docker
         * documentation.
         * </p>
         * </dd>
         * <dt>syslog</dt>
         * <dd>
         * <p>
         * Specifies the syslog logging driver. For more information, including usage and options, see <a
         * href="https://docs.docker.com/config/containers/logging/syslog/">Syslog logging driver</a> in the Docker
         * documentation.
         * </p>
         * </dd>
         * </dl>
         * <note>
         * <p>
         * If you have a custom driver that isn't listed earlier that you want to work with the Amazon ECS container
         * agent, you can fork the Amazon ECS container agent project that is <a
         * href="https://github.com/aws/amazon-ecs-agent">available on GitHub</a> and customize it to work with that
         * driver. We encourage you to submit pull requests for changes that you want to have included. However, Amazon
         * Web Services doesn't currently support running modified copies of this software.
         * </p>
         * </note>
         * <p>
         * This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check
         * the Docker Remote API version on your container instance, log into your container instance and run the
         * following command: <code>sudo docker version | grep "Server API version"</code>
         * </p>
         * 
         * @param logDriver
         *        The log driver to use for the container. The valid values listed for this parameter are log drivers
         *        that the Amazon ECS container agent can communicate with by default.</p>
         *        <p>
         *        The supported log drivers are <code>awslogs</code>, <code>fluentd</code>, <code>gelf</code>,
         *        <code>json-file</code>, <code>journald</code>, <code>logentries</code>, <code>syslog</code>, and
         *        <code>splunk</code>.
         *        </p>
         *        <note>
         *        <p>
         *        Jobs running on Fargate resources are restricted to the <code>awslogs</code> and <code>splunk</code>
         *        log drivers.
         *        </p>
         *        </note>
         *        <dl>
         *        <dt>awslogs</dt>
         *        <dd>
         *        <p>
         *        Specifies the Amazon CloudWatch Logs logging driver. For more information, see <a
         *        href="https://docs.aws.amazon.com/batch/latest/userguide/using_awslogs.html">Using the awslogs Log
         *        Driver</a> in the <i>AWS Batch User Guide</i> and <a
         *        href="https://docs.docker.com/config/containers/logging/awslogs/">Amazon CloudWatch Logs logging
         *        driver</a> in the Docker documentation.
         *        </p>
         *        </dd>
         *        <dt>fluentd</dt>
         *        <dd>
         *        <p>
         *        Specifies the Fluentd logging driver. For more information, including usage and options, see <a
         *        href="https://docs.docker.com/config/containers/logging/fluentd/">Fluentd logging driver</a> in the
         *        Docker documentation.
         *        </p>
         *        </dd>
         *        <dt>gelf</dt>
         *        <dd>
         *        <p>
         *        Specifies the Graylog Extended Format (GELF) logging driver. For more information, including usage and
         *        options, see <a href="https://docs.docker.com/config/containers/logging/gelf/">Graylog Extended Format
         *        logging driver</a> in the Docker documentation.
         *        </p>
         *        </dd>
         *        <dt>journald</dt>
         *        <dd>
         *        <p>
         *        Specifies the journald logging driver. For more information, including usage and options, see <a
         *        href="https://docs.docker.com/config/containers/logging/journald/">Journald logging driver</a> in the
         *        Docker documentation.
         *        </p>
         *        </dd>
         *        <dt>json-file</dt>
         *        <dd>
         *        <p>
         *        Specifies the JSON file logging driver. For more information, including usage and options, see <a
         *        href="https://docs.docker.com/config/containers/logging/json-file/">JSON File logging driver</a> in
         *        the Docker documentation.
         *        </p>
         *        </dd>
         *        <dt>splunk</dt>
         *        <dd>
         *        <p>
         *        Specifies the Splunk logging driver. For more information, including usage and options, see <a
         *        href="https://docs.docker.com/config/containers/logging/splunk/">Splunk logging driver</a> in the
         *        Docker documentation.
         *        </p>
         *        </dd>
         *        <dt>syslog</dt>
         *        <dd>
         *        <p>
         *        Specifies the syslog logging driver. For more information, including usage and options, see <a
         *        href="https://docs.docker.com/config/containers/logging/syslog/">Syslog logging driver</a> in the
         *        Docker documentation.
         *        </p>
         *        </dd>
         *        </dl>
         *        <note>
         *        <p>
         *        If you have a custom driver that isn't listed earlier that you want to work with the Amazon ECS
         *        container agent, you can fork the Amazon ECS container agent project that is <a
         *        href="https://github.com/aws/amazon-ecs-agent">available on GitHub</a> and customize it to work with
         *        that driver. We encourage you to submit pull requests for changes that you want to have included.
         *        However, Amazon Web Services doesn't currently support running modified copies of this software.
         *        </p>
         *        </note>
         *        <p>
         *        This parameter requires version 1.18 of the Docker Remote API or greater on your container instance.
         *        To check the Docker Remote API version on your container instance, log into your container instance
         *        and run the following command: <code>sudo docker version | grep "Server API version"</code>
         * @see LogDriver
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LogDriver
         */
        Builder logDriver(String logDriver);

        /**
         * <p>
         * The log driver to use for the container. The valid values listed for this parameter are log drivers that the
         * Amazon ECS container agent can communicate with by default.
         * </p>
         * <p>
         * The supported log drivers are <code>awslogs</code>, <code>fluentd</code>, <code>gelf</code>,
         * <code>json-file</code>, <code>journald</code>, <code>logentries</code>, <code>syslog</code>, and
         * <code>splunk</code>.
         * </p>
         * <note>
         * <p>
         * Jobs running on Fargate resources are restricted to the <code>awslogs</code> and <code>splunk</code> log
         * drivers.
         * </p>
         * </note>
         * <dl>
         * <dt>awslogs</dt>
         * <dd>
         * <p>
         * Specifies the Amazon CloudWatch Logs logging driver. For more information, see <a
         * href="https://docs.aws.amazon.com/batch/latest/userguide/using_awslogs.html">Using the awslogs Log Driver</a>
         * in the <i>AWS Batch User Guide</i> and <a
         * href="https://docs.docker.com/config/containers/logging/awslogs/">Amazon CloudWatch Logs logging driver</a>
         * in the Docker documentation.
         * </p>
         * </dd>
         * <dt>fluentd</dt>
         * <dd>
         * <p>
         * Specifies the Fluentd logging driver. For more information, including usage and options, see <a
         * href="https://docs.docker.com/config/containers/logging/fluentd/">Fluentd logging driver</a> in the Docker
         * documentation.
         * </p>
         * </dd>
         * <dt>gelf</dt>
         * <dd>
         * <p>
         * Specifies the Graylog Extended Format (GELF) logging driver. For more information, including usage and
         * options, see <a href="https://docs.docker.com/config/containers/logging/gelf/">Graylog Extended Format
         * logging driver</a> in the Docker documentation.
         * </p>
         * </dd>
         * <dt>journald</dt>
         * <dd>
         * <p>
         * Specifies the journald logging driver. For more information, including usage and options, see <a
         * href="https://docs.docker.com/config/containers/logging/journald/">Journald logging driver</a> in the Docker
         * documentation.
         * </p>
         * </dd>
         * <dt>json-file</dt>
         * <dd>
         * <p>
         * Specifies the JSON file logging driver. For more information, including usage and options, see <a
         * href="https://docs.docker.com/config/containers/logging/json-file/">JSON File logging driver</a> in the
         * Docker documentation.
         * </p>
         * </dd>
         * <dt>splunk</dt>
         * <dd>
         * <p>
         * Specifies the Splunk logging driver. For more information, including usage and options, see <a
         * href="https://docs.docker.com/config/containers/logging/splunk/">Splunk logging driver</a> in the Docker
         * documentation.
         * </p>
         * </dd>
         * <dt>syslog</dt>
         * <dd>
         * <p>
         * Specifies the syslog logging driver. For more information, including usage and options, see <a
         * href="https://docs.docker.com/config/containers/logging/syslog/">Syslog logging driver</a> in the Docker
         * documentation.
         * </p>
         * </dd>
         * </dl>
         * <note>
         * <p>
         * If you have a custom driver that isn't listed earlier that you want to work with the Amazon ECS container
         * agent, you can fork the Amazon ECS container agent project that is <a
         * href="https://github.com/aws/amazon-ecs-agent">available on GitHub</a> and customize it to work with that
         * driver. We encourage you to submit pull requests for changes that you want to have included. However, Amazon
         * Web Services doesn't currently support running modified copies of this software.
         * </p>
         * </note>
         * <p>
         * This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check
         * the Docker Remote API version on your container instance, log into your container instance and run the
         * following command: <code>sudo docker version | grep "Server API version"</code>
         * </p>
         * 
         * @param logDriver
         *        The log driver to use for the container. The valid values listed for this parameter are log drivers
         *        that the Amazon ECS container agent can communicate with by default.</p>
         *        <p>
         *        The supported log drivers are <code>awslogs</code>, <code>fluentd</code>, <code>gelf</code>,
         *        <code>json-file</code>, <code>journald</code>, <code>logentries</code>, <code>syslog</code>, and
         *        <code>splunk</code>.
         *        </p>
         *        <note>
         *        <p>
         *        Jobs running on Fargate resources are restricted to the <code>awslogs</code> and <code>splunk</code>
         *        log drivers.
         *        </p>
         *        </note>
         *        <dl>
         *        <dt>awslogs</dt>
         *        <dd>
         *        <p>
         *        Specifies the Amazon CloudWatch Logs logging driver. For more information, see <a
         *        href="https://docs.aws.amazon.com/batch/latest/userguide/using_awslogs.html">Using the awslogs Log
         *        Driver</a> in the <i>AWS Batch User Guide</i> and <a
         *        href="https://docs.docker.com/config/containers/logging/awslogs/">Amazon CloudWatch Logs logging
         *        driver</a> in the Docker documentation.
         *        </p>
         *        </dd>
         *        <dt>fluentd</dt>
         *        <dd>
         *        <p>
         *        Specifies the Fluentd logging driver. For more information, including usage and options, see <a
         *        href="https://docs.docker.com/config/containers/logging/fluentd/">Fluentd logging driver</a> in the
         *        Docker documentation.
         *        </p>
         *        </dd>
         *        <dt>gelf</dt>
         *        <dd>
         *        <p>
         *        Specifies the Graylog Extended Format (GELF) logging driver. For more information, including usage and
         *        options, see <a href="https://docs.docker.com/config/containers/logging/gelf/">Graylog Extended Format
         *        logging driver</a> in the Docker documentation.
         *        </p>
         *        </dd>
         *        <dt>journald</dt>
         *        <dd>
         *        <p>
         *        Specifies the journald logging driver. For more information, including usage and options, see <a
         *        href="https://docs.docker.com/config/containers/logging/journald/">Journald logging driver</a> in the
         *        Docker documentation.
         *        </p>
         *        </dd>
         *        <dt>json-file</dt>
         *        <dd>
         *        <p>
         *        Specifies the JSON file logging driver. For more information, including usage and options, see <a
         *        href="https://docs.docker.com/config/containers/logging/json-file/">JSON File logging driver</a> in
         *        the Docker documentation.
         *        </p>
         *        </dd>
         *        <dt>splunk</dt>
         *        <dd>
         *        <p>
         *        Specifies the Splunk logging driver. For more information, including usage and options, see <a
         *        href="https://docs.docker.com/config/containers/logging/splunk/">Splunk logging driver</a> in the
         *        Docker documentation.
         *        </p>
         *        </dd>
         *        <dt>syslog</dt>
         *        <dd>
         *        <p>
         *        Specifies the syslog logging driver. For more information, including usage and options, see <a
         *        href="https://docs.docker.com/config/containers/logging/syslog/">Syslog logging driver</a> in the
         *        Docker documentation.
         *        </p>
         *        </dd>
         *        </dl>
         *        <note>
         *        <p>
         *        If you have a custom driver that isn't listed earlier that you want to work with the Amazon ECS
         *        container agent, you can fork the Amazon ECS container agent project that is <a
         *        href="https://github.com/aws/amazon-ecs-agent">available on GitHub</a> and customize it to work with
         *        that driver. We encourage you to submit pull requests for changes that you want to have included.
         *        However, Amazon Web Services doesn't currently support running modified copies of this software.
         *        </p>
         *        </note>
         *        <p>
         *        This parameter requires version 1.18 of the Docker Remote API or greater on your container instance.
         *        To check the Docker Remote API version on your container instance, log into your container instance
         *        and run the following command: <code>sudo docker version | grep "Server API version"</code>
         * @see LogDriver
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LogDriver
         */
        Builder logDriver(LogDriver logDriver);

        /**
         * <p>
         * The configuration options to send to the log driver. This parameter requires version 1.19 of the Docker
         * Remote API or greater on your container instance. To check the Docker Remote API version on your container
         * instance, log into your container instance and run the following command:
         * <code>sudo docker version | grep "Server API version"</code>
         * </p>
         * 
         * @param options
         *        The configuration options to send to the log driver. This parameter requires version 1.19 of the
         *        Docker Remote API or greater on your container instance. To check the Docker Remote API version on
         *        your container instance, log into your container instance and run the following command:
         *        <code>sudo docker version | grep "Server API version"</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder options(Map<String, String> options);

        /**
         * <p>
         * The secrets to pass to the log configuration. For more information, see <a
         * href="https://docs.aws.amazon.com/batch/latest/userguide/specifying-sensitive-data.html">Specifying Sensitive
         * Data</a> in the <i>AWS Batch User Guide</i>.
         * </p>
         * 
         * @param secretOptions
         *        The secrets to pass to the log configuration. For more information, see <a
         *        href="https://docs.aws.amazon.com/batch/latest/userguide/specifying-sensitive-data.html">Specifying
         *        Sensitive Data</a> in the <i>AWS Batch User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretOptions(Collection<Secret> secretOptions);

        /**
         * <p>
         * The secrets to pass to the log configuration. For more information, see <a
         * href="https://docs.aws.amazon.com/batch/latest/userguide/specifying-sensitive-data.html">Specifying Sensitive
         * Data</a> in the <i>AWS Batch User Guide</i>.
         * </p>
         * 
         * @param secretOptions
         *        The secrets to pass to the log configuration. For more information, see <a
         *        href="https://docs.aws.amazon.com/batch/latest/userguide/specifying-sensitive-data.html">Specifying
         *        Sensitive Data</a> in the <i>AWS Batch User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretOptions(Secret... secretOptions);

        /**
         * <p>
         * The secrets to pass to the log configuration. For more information, see <a
         * href="https://docs.aws.amazon.com/batch/latest/userguide/specifying-sensitive-data.html">Specifying Sensitive
         * Data</a> in the <i>AWS Batch User Guide</i>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Secret>.Builder} avoiding the need to
         * create one manually via {@link List<Secret>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Secret>.Builder#build()} is called immediately and its
         * result is passed to {@link #secretOptions(List<Secret>)}.
         * 
         * @param secretOptions
         *        a consumer that will call methods on {@link List<Secret>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #secretOptions(List<Secret>)
         */
        Builder secretOptions(Consumer<Secret.Builder>... secretOptions);
    }

    static final class BuilderImpl implements Builder {
        private String logDriver;

        private Map<String, String> options = DefaultSdkAutoConstructMap.getInstance();

        private List<Secret> secretOptions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(LogConfiguration model) {
            logDriver(model.logDriver);
            options(model.options);
            secretOptions(model.secretOptions);
        }

        public final String getLogDriver() {
            return logDriver;
        }

        @Override
        public final Builder logDriver(String logDriver) {
            this.logDriver = logDriver;
            return this;
        }

        @Override
        public final Builder logDriver(LogDriver logDriver) {
            this.logDriver(logDriver == null ? null : logDriver.toString());
            return this;
        }

        public final void setLogDriver(String logDriver) {
            this.logDriver = logDriver;
        }

        public final Map<String, String> getOptions() {
            if (options instanceof SdkAutoConstructMap) {
                return null;
            }
            return options;
        }

        @Override
        public final Builder options(Map<String, String> options) {
            this.options = LogConfigurationOptionsMapCopier.copy(options);
            return this;
        }

        public final void setOptions(Map<String, String> options) {
            this.options = LogConfigurationOptionsMapCopier.copy(options);
        }

        public final Collection<Secret.Builder> getSecretOptions() {
            if (secretOptions instanceof SdkAutoConstructList) {
                return null;
            }
            return secretOptions != null ? secretOptions.stream().map(Secret::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder secretOptions(Collection<Secret> secretOptions) {
            this.secretOptions = SecretListCopier.copy(secretOptions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder secretOptions(Secret... secretOptions) {
            secretOptions(Arrays.asList(secretOptions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder secretOptions(Consumer<Secret.Builder>... secretOptions) {
            secretOptions(Stream.of(secretOptions).map(c -> Secret.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setSecretOptions(Collection<Secret.BuilderImpl> secretOptions) {
            this.secretOptions = SecretListCopier.copyFromBuilder(secretOptions);
        }

        @Override
        public LogConfiguration build() {
            return new LogConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
