/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.batch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The overrides that should be sent to a container.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ContainerOverrides implements SdkPojo, Serializable,
        ToCopyableBuilder<ContainerOverrides.Builder, ContainerOverrides> {
    private static final SdkField<Integer> VCPUS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("vcpus")
            .getter(getter(ContainerOverrides::vcpus)).setter(setter(Builder::vcpus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("vcpus").build()).build();

    private static final SdkField<Integer> MEMORY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("memory").getter(getter(ContainerOverrides::memory)).setter(setter(Builder::memory))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("memory").build()).build();

    private static final SdkField<List<String>> COMMAND_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("command")
            .getter(getter(ContainerOverrides::command))
            .setter(setter(Builder::command))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("command").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("instanceType").getter(getter(ContainerOverrides::instanceType)).setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instanceType").build()).build();

    private static final SdkField<List<KeyValuePair>> ENVIRONMENT_FIELD = SdkField
            .<List<KeyValuePair>> builder(MarshallingType.LIST)
            .memberName("environment")
            .getter(getter(ContainerOverrides::environment))
            .setter(setter(Builder::environment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("environment").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<KeyValuePair> builder(MarshallingType.SDK_POJO)
                                            .constructor(KeyValuePair::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<ResourceRequirement>> RESOURCE_REQUIREMENTS_FIELD = SdkField
            .<List<ResourceRequirement>> builder(MarshallingType.LIST)
            .memberName("resourceRequirements")
            .getter(getter(ContainerOverrides::resourceRequirements))
            .setter(setter(Builder::resourceRequirements))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceRequirements").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ResourceRequirement> builder(MarshallingType.SDK_POJO)
                                            .constructor(ResourceRequirement::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VCPUS_FIELD, MEMORY_FIELD,
            COMMAND_FIELD, INSTANCE_TYPE_FIELD, ENVIRONMENT_FIELD, RESOURCE_REQUIREMENTS_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer vcpus;

    private final Integer memory;

    private final List<String> command;

    private final String instanceType;

    private final List<KeyValuePair> environment;

    private final List<ResourceRequirement> resourceRequirements;

    private ContainerOverrides(BuilderImpl builder) {
        this.vcpus = builder.vcpus;
        this.memory = builder.memory;
        this.command = builder.command;
        this.instanceType = builder.instanceType;
        this.environment = builder.environment;
        this.resourceRequirements = builder.resourceRequirements;
    }

    /**
     * <p>
     * This parameter indicates the number of vCPUs reserved for the container.It overrides the <code>vcpus</code>
     * parameter that's set in the job definition, but doesn't override any vCPU requirement specified in the
     * <code>resourceRequirement</code> structure in the job definition.
     * </p>
     * <p>
     * This parameter is supported for jobs that run on EC2 resources, but isn't supported for jobs that run on Fargate
     * resources. For Fargate resources, you can only use <code>resourceRequirement</code>. For EC2 resources, you can
     * use either this parameter or <code>resourceRequirement</code> but not both.
     * </p>
     * <p>
     * This parameter maps to <code>CpuShares</code> in the <a
     * href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the <a
     * href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--cpu-shares</code> option
     * to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>. Each vCPU is equivalent to 1,024 CPU
     * shares. You must specify at least one vCPU.
     * </p>
     * <note>
     * <p>
     * This parameter isn't applicable to jobs that run on Fargate resources and shouldn't be provided. For jobs that
     * run on Fargate resources, you must specify the vCPU requirement for the job using
     * <code>resourceRequirements</code>.
     * </p>
     * </note>
     * 
     * @return This parameter indicates the number of vCPUs reserved for the container.It overrides the
     *         <code>vcpus</code> parameter that's set in the job definition, but doesn't override any vCPU requirement
     *         specified in the <code>resourceRequirement</code> structure in the job definition.</p>
     *         <p>
     *         This parameter is supported for jobs that run on EC2 resources, but isn't supported for jobs that run on
     *         Fargate resources. For Fargate resources, you can only use <code>resourceRequirement</code>. For EC2
     *         resources, you can use either this parameter or <code>resourceRequirement</code> but not both.
     *         </p>
     *         <p>
     *         This parameter maps to <code>CpuShares</code> in the <a
     *         href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the
     *         <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the
     *         <code>--cpu-shares</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
     *         run</a>. Each vCPU is equivalent to 1,024 CPU shares. You must specify at least one vCPU.
     *         </p>
     *         <note>
     *         <p>
     *         This parameter isn't applicable to jobs that run on Fargate resources and shouldn't be provided. For jobs
     *         that run on Fargate resources, you must specify the vCPU requirement for the job using
     *         <code>resourceRequirements</code>.
     *         </p>
     */
    public final Integer vcpus() {
        return vcpus;
    }

    /**
     * <p>
     * This parameter indicates the amount of memory (in MiB) that's reserved for the job. It overrides the
     * <code>memory</code> parameter set in the job definition, but doesn't override any memory requirement specified in
     * the <code>ResourceRequirement</code> structure in the job definition.
     * </p>
     * <p>
     * This parameter is supported for jobs that run on EC2 resources, but isn't supported for jobs that run on Fargate
     * resources. For these resources, use <code>resourceRequirement</code> instead.
     * </p>
     * 
     * @return This parameter indicates the amount of memory (in MiB) that's reserved for the job. It overrides the
     *         <code>memory</code> parameter set in the job definition, but doesn't override any memory requirement
     *         specified in the <code>ResourceRequirement</code> structure in the job definition.</p>
     *         <p>
     *         This parameter is supported for jobs that run on EC2 resources, but isn't supported for jobs that run on
     *         Fargate resources. For these resources, use <code>resourceRequirement</code> instead.
     */
    public final Integer memory() {
        return memory;
    }

    /**
     * Returns true if the Command property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasCommand() {
        return command != null && !(command instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The command to send to the container that overrides the default command from the Docker image or the job
     * definition.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasCommand()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The command to send to the container that overrides the default command from the Docker image or the job
     *         definition.
     */
    public final List<String> command() {
        return command;
    }

    /**
     * <p>
     * The instance type to use for a multi-node parallel job.
     * </p>
     * <note>
     * <p>
     * This parameter isn't applicable to single-node container jobs or for jobs running on Fargate resources and
     * shouldn't be provided.
     * </p>
     * </note>
     * 
     * @return The instance type to use for a multi-node parallel job.</p> <note>
     *         <p>
     *         This parameter isn't applicable to single-node container jobs or for jobs running on Fargate resources
     *         and shouldn't be provided.
     *         </p>
     */
    public final String instanceType() {
        return instanceType;
    }

    /**
     * Returns true if the Environment property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasEnvironment() {
        return environment != null && !(environment instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The environment variables to send to the container. You can add new environment variables, which are added to the
     * container at launch, or you can override the existing environment variables from the Docker image or the job
     * definition.
     * </p>
     * <note>
     * <p>
     * Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved for
     * variables that are set by the AWS Batch service.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasEnvironment()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The environment variables to send to the container. You can add new environment variables, which are
     *         added to the container at launch, or you can override the existing environment variables from the Docker
     *         image or the job definition.</p> <note>
     *         <p>
     *         Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved for
     *         variables that are set by the AWS Batch service.
     *         </p>
     */
    public final List<KeyValuePair> environment() {
        return environment;
    }

    /**
     * Returns true if the ResourceRequirements property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasResourceRequirements() {
        return resourceRequirements != null && !(resourceRequirements instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The type and amount of resources to assign to a container. This overrides the settings in the job definition. The
     * supported resources include <code>GPU</code>, <code>MEMORY</code>, and <code>VCPU</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasResourceRequirements()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The type and amount of resources to assign to a container. This overrides the settings in the job
     *         definition. The supported resources include <code>GPU</code>, <code>MEMORY</code>, and <code>VCPU</code>.
     */
    public final List<ResourceRequirement> resourceRequirements() {
        return resourceRequirements;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(vcpus());
        hashCode = 31 * hashCode + Objects.hashCode(memory());
        hashCode = 31 * hashCode + Objects.hashCode(hasCommand() ? command() : null);
        hashCode = 31 * hashCode + Objects.hashCode(instanceType());
        hashCode = 31 * hashCode + Objects.hashCode(hasEnvironment() ? environment() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceRequirements() ? resourceRequirements() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ContainerOverrides)) {
            return false;
        }
        ContainerOverrides other = (ContainerOverrides) obj;
        return Objects.equals(vcpus(), other.vcpus()) && Objects.equals(memory(), other.memory())
                && hasCommand() == other.hasCommand() && Objects.equals(command(), other.command())
                && Objects.equals(instanceType(), other.instanceType()) && hasEnvironment() == other.hasEnvironment()
                && Objects.equals(environment(), other.environment())
                && hasResourceRequirements() == other.hasResourceRequirements()
                && Objects.equals(resourceRequirements(), other.resourceRequirements());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ContainerOverrides").add("Vcpus", vcpus()).add("Memory", memory())
                .add("Command", hasCommand() ? command() : null).add("InstanceType", instanceType())
                .add("Environment", hasEnvironment() ? environment() : null)
                .add("ResourceRequirements", hasResourceRequirements() ? resourceRequirements() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "vcpus":
            return Optional.ofNullable(clazz.cast(vcpus()));
        case "memory":
            return Optional.ofNullable(clazz.cast(memory()));
        case "command":
            return Optional.ofNullable(clazz.cast(command()));
        case "instanceType":
            return Optional.ofNullable(clazz.cast(instanceType()));
        case "environment":
            return Optional.ofNullable(clazz.cast(environment()));
        case "resourceRequirements":
            return Optional.ofNullable(clazz.cast(resourceRequirements()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ContainerOverrides, T> g) {
        return obj -> g.apply((ContainerOverrides) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ContainerOverrides> {
        /**
         * <p>
         * This parameter indicates the number of vCPUs reserved for the container.It overrides the <code>vcpus</code>
         * parameter that's set in the job definition, but doesn't override any vCPU requirement specified in the
         * <code>resourceRequirement</code> structure in the job definition.
         * </p>
         * <p>
         * This parameter is supported for jobs that run on EC2 resources, but isn't supported for jobs that run on
         * Fargate resources. For Fargate resources, you can only use <code>resourceRequirement</code>. For EC2
         * resources, you can use either this parameter or <code>resourceRequirement</code> but not both.
         * </p>
         * <p>
         * This parameter maps to <code>CpuShares</code> in the <a
         * href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the <a
         * href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--cpu-shares</code>
         * option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>. Each vCPU is equivalent to
         * 1,024 CPU shares. You must specify at least one vCPU.
         * </p>
         * <note>
         * <p>
         * This parameter isn't applicable to jobs that run on Fargate resources and shouldn't be provided. For jobs
         * that run on Fargate resources, you must specify the vCPU requirement for the job using
         * <code>resourceRequirements</code>.
         * </p>
         * </note>
         * 
         * @param vcpus
         *        This parameter indicates the number of vCPUs reserved for the container.It overrides the
         *        <code>vcpus</code> parameter that's set in the job definition, but doesn't override any vCPU
         *        requirement specified in the <code>resourceRequirement</code> structure in the job definition.</p>
         *        <p>
         *        This parameter is supported for jobs that run on EC2 resources, but isn't supported for jobs that run
         *        on Fargate resources. For Fargate resources, you can only use <code>resourceRequirement</code>. For
         *        EC2 resources, you can use either this parameter or <code>resourceRequirement</code> but not both.
         *        </p>
         *        <p>
         *        This parameter maps to <code>CpuShares</code> in the <a
         *        href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of
         *        the <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the
         *        <code>--cpu-shares</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
         *        run</a>. Each vCPU is equivalent to 1,024 CPU shares. You must specify at least one vCPU.
         *        </p>
         *        <note>
         *        <p>
         *        This parameter isn't applicable to jobs that run on Fargate resources and shouldn't be provided. For
         *        jobs that run on Fargate resources, you must specify the vCPU requirement for the job using
         *        <code>resourceRequirements</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vcpus(Integer vcpus);

        /**
         * <p>
         * This parameter indicates the amount of memory (in MiB) that's reserved for the job. It overrides the
         * <code>memory</code> parameter set in the job definition, but doesn't override any memory requirement
         * specified in the <code>ResourceRequirement</code> structure in the job definition.
         * </p>
         * <p>
         * This parameter is supported for jobs that run on EC2 resources, but isn't supported for jobs that run on
         * Fargate resources. For these resources, use <code>resourceRequirement</code> instead.
         * </p>
         * 
         * @param memory
         *        This parameter indicates the amount of memory (in MiB) that's reserved for the job. It overrides the
         *        <code>memory</code> parameter set in the job definition, but doesn't override any memory requirement
         *        specified in the <code>ResourceRequirement</code> structure in the job definition.</p>
         *        <p>
         *        This parameter is supported for jobs that run on EC2 resources, but isn't supported for jobs that run
         *        on Fargate resources. For these resources, use <code>resourceRequirement</code> instead.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder memory(Integer memory);

        /**
         * <p>
         * The command to send to the container that overrides the default command from the Docker image or the job
         * definition.
         * </p>
         * 
         * @param command
         *        The command to send to the container that overrides the default command from the Docker image or the
         *        job definition.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(Collection<String> command);

        /**
         * <p>
         * The command to send to the container that overrides the default command from the Docker image or the job
         * definition.
         * </p>
         * 
         * @param command
         *        The command to send to the container that overrides the default command from the Docker image or the
         *        job definition.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(String... command);

        /**
         * <p>
         * The instance type to use for a multi-node parallel job.
         * </p>
         * <note>
         * <p>
         * This parameter isn't applicable to single-node container jobs or for jobs running on Fargate resources and
         * shouldn't be provided.
         * </p>
         * </note>
         * 
         * @param instanceType
         *        The instance type to use for a multi-node parallel job.</p> <note>
         *        <p>
         *        This parameter isn't applicable to single-node container jobs or for jobs running on Fargate resources
         *        and shouldn't be provided.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * The environment variables to send to the container. You can add new environment variables, which are added to
         * the container at launch, or you can override the existing environment variables from the Docker image or the
         * job definition.
         * </p>
         * <note>
         * <p>
         * Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved for
         * variables that are set by the AWS Batch service.
         * </p>
         * </note>
         * 
         * @param environment
         *        The environment variables to send to the container. You can add new environment variables, which are
         *        added to the container at launch, or you can override the existing environment variables from the
         *        Docker image or the job definition.</p> <note>
         *        <p>
         *        Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved
         *        for variables that are set by the AWS Batch service.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environment(Collection<KeyValuePair> environment);

        /**
         * <p>
         * The environment variables to send to the container. You can add new environment variables, which are added to
         * the container at launch, or you can override the existing environment variables from the Docker image or the
         * job definition.
         * </p>
         * <note>
         * <p>
         * Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved for
         * variables that are set by the AWS Batch service.
         * </p>
         * </note>
         * 
         * @param environment
         *        The environment variables to send to the container. You can add new environment variables, which are
         *        added to the container at launch, or you can override the existing environment variables from the
         *        Docker image or the job definition.</p> <note>
         *        <p>
         *        Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved
         *        for variables that are set by the AWS Batch service.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environment(KeyValuePair... environment);

        /**
         * <p>
         * The environment variables to send to the container. You can add new environment variables, which are added to
         * the container at launch, or you can override the existing environment variables from the Docker image or the
         * job definition.
         * </p>
         * <note>
         * <p>
         * Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved for
         * variables that are set by the AWS Batch service.
         * </p>
         * </note> This is a convenience that creates an instance of the {@link List<KeyValuePair>.Builder} avoiding the
         * need to create one manually via {@link List<KeyValuePair>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<KeyValuePair>.Builder#build()} is called immediately and its
         * result is passed to {@link #environment(List<KeyValuePair>)}.
         * 
         * @param environment
         *        a consumer that will call methods on {@link List<KeyValuePair>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #environment(List<KeyValuePair>)
         */
        Builder environment(Consumer<KeyValuePair.Builder>... environment);

        /**
         * <p>
         * The type and amount of resources to assign to a container. This overrides the settings in the job definition.
         * The supported resources include <code>GPU</code>, <code>MEMORY</code>, and <code>VCPU</code>.
         * </p>
         * 
         * @param resourceRequirements
         *        The type and amount of resources to assign to a container. This overrides the settings in the job
         *        definition. The supported resources include <code>GPU</code>, <code>MEMORY</code>, and
         *        <code>VCPU</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceRequirements(Collection<ResourceRequirement> resourceRequirements);

        /**
         * <p>
         * The type and amount of resources to assign to a container. This overrides the settings in the job definition.
         * The supported resources include <code>GPU</code>, <code>MEMORY</code>, and <code>VCPU</code>.
         * </p>
         * 
         * @param resourceRequirements
         *        The type and amount of resources to assign to a container. This overrides the settings in the job
         *        definition. The supported resources include <code>GPU</code>, <code>MEMORY</code>, and
         *        <code>VCPU</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceRequirements(ResourceRequirement... resourceRequirements);

        /**
         * <p>
         * The type and amount of resources to assign to a container. This overrides the settings in the job definition.
         * The supported resources include <code>GPU</code>, <code>MEMORY</code>, and <code>VCPU</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ResourceRequirement>.Builder} avoiding the
         * need to create one manually via {@link List<ResourceRequirement>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ResourceRequirement>.Builder#build()} is called immediately
         * and its result is passed to {@link #resourceRequirements(List<ResourceRequirement>)}.
         * 
         * @param resourceRequirements
         *        a consumer that will call methods on {@link List<ResourceRequirement>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resourceRequirements(List<ResourceRequirement>)
         */
        Builder resourceRequirements(Consumer<ResourceRequirement.Builder>... resourceRequirements);
    }

    static final class BuilderImpl implements Builder {
        private Integer vcpus;

        private Integer memory;

        private List<String> command = DefaultSdkAutoConstructList.getInstance();

        private String instanceType;

        private List<KeyValuePair> environment = DefaultSdkAutoConstructList.getInstance();

        private List<ResourceRequirement> resourceRequirements = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ContainerOverrides model) {
            vcpus(model.vcpus);
            memory(model.memory);
            command(model.command);
            instanceType(model.instanceType);
            environment(model.environment);
            resourceRequirements(model.resourceRequirements);
        }

        public final Integer getVcpus() {
            return vcpus;
        }

        @Override
        public final Builder vcpus(Integer vcpus) {
            this.vcpus = vcpus;
            return this;
        }

        public final void setVcpus(Integer vcpus) {
            this.vcpus = vcpus;
        }

        public final Integer getMemory() {
            return memory;
        }

        @Override
        public final Builder memory(Integer memory) {
            this.memory = memory;
            return this;
        }

        public final void setMemory(Integer memory) {
            this.memory = memory;
        }

        public final Collection<String> getCommand() {
            if (command instanceof SdkAutoConstructList) {
                return null;
            }
            return command;
        }

        @Override
        public final Builder command(Collection<String> command) {
            this.command = StringListCopier.copy(command);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder command(String... command) {
            command(Arrays.asList(command));
            return this;
        }

        public final void setCommand(Collection<String> command) {
            this.command = StringListCopier.copy(command);
        }

        public final String getInstanceType() {
            return instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        public final Collection<KeyValuePair.Builder> getEnvironment() {
            if (environment instanceof SdkAutoConstructList) {
                return null;
            }
            return environment != null ? environment.stream().map(KeyValuePair::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder environment(Collection<KeyValuePair> environment) {
            this.environment = EnvironmentVariablesCopier.copy(environment);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder environment(KeyValuePair... environment) {
            environment(Arrays.asList(environment));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder environment(Consumer<KeyValuePair.Builder>... environment) {
            environment(Stream.of(environment).map(c -> KeyValuePair.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setEnvironment(Collection<KeyValuePair.BuilderImpl> environment) {
            this.environment = EnvironmentVariablesCopier.copyFromBuilder(environment);
        }

        public final Collection<ResourceRequirement.Builder> getResourceRequirements() {
            if (resourceRequirements instanceof SdkAutoConstructList) {
                return null;
            }
            return resourceRequirements != null ? resourceRequirements.stream().map(ResourceRequirement::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder resourceRequirements(Collection<ResourceRequirement> resourceRequirements) {
            this.resourceRequirements = ResourceRequirementsCopier.copy(resourceRequirements);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceRequirements(ResourceRequirement... resourceRequirements) {
            resourceRequirements(Arrays.asList(resourceRequirements));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceRequirements(Consumer<ResourceRequirement.Builder>... resourceRequirements) {
            resourceRequirements(Stream.of(resourceRequirements).map(c -> ResourceRequirement.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setResourceRequirements(Collection<ResourceRequirement.BuilderImpl> resourceRequirements) {
            this.resourceRequirements = ResourceRequirementsCopier.copyFromBuilder(resourceRequirements);
        }

        @Override
        public ContainerOverrides build() {
            return new ContainerOverrides(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
