/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.batch.model;

import java.beans.Transient;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the parameters for <code>CreateJobQueue</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateJobQueueRequest extends BatchRequest implements
        ToCopyableBuilder<CreateJobQueueRequest.Builder, CreateJobQueueRequest> {
    private static final SdkField<String> JOB_QUEUE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("jobQueueName").getter(getter(CreateJobQueueRequest::jobQueueName)).setter(setter(Builder::jobQueueName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("jobQueueName").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("state")
            .getter(getter(CreateJobQueueRequest::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("state").build()).build();

    private static final SdkField<String> SCHEDULING_POLICY_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("schedulingPolicyArn").getter(getter(CreateJobQueueRequest::schedulingPolicyArn))
            .setter(setter(Builder::schedulingPolicyArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("schedulingPolicyArn").build())
            .build();

    private static final SdkField<Integer> PRIORITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("priority").getter(getter(CreateJobQueueRequest::priority)).setter(setter(Builder::priority))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("priority").build()).build();

    private static final SdkField<List<ComputeEnvironmentOrder>> COMPUTE_ENVIRONMENT_ORDER_FIELD = SdkField
            .<List<ComputeEnvironmentOrder>> builder(MarshallingType.LIST)
            .memberName("computeEnvironmentOrder")
            .getter(getter(CreateJobQueueRequest::computeEnvironmentOrder))
            .setter(setter(Builder::computeEnvironmentOrder))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("computeEnvironmentOrder").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ComputeEnvironmentOrder> builder(MarshallingType.SDK_POJO)
                                            .constructor(ComputeEnvironmentOrder::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(CreateJobQueueRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(JOB_QUEUE_NAME_FIELD,
            STATE_FIELD, SCHEDULING_POLICY_ARN_FIELD, PRIORITY_FIELD, COMPUTE_ENVIRONMENT_ORDER_FIELD, TAGS_FIELD));

    private final String jobQueueName;

    private final String state;

    private final String schedulingPolicyArn;

    private final Integer priority;

    private final List<ComputeEnvironmentOrder> computeEnvironmentOrder;

    private final Map<String, String> tags;

    private CreateJobQueueRequest(BuilderImpl builder) {
        super(builder);
        this.jobQueueName = builder.jobQueueName;
        this.state = builder.state;
        this.schedulingPolicyArn = builder.schedulingPolicyArn;
        this.priority = builder.priority;
        this.computeEnvironmentOrder = builder.computeEnvironmentOrder;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The name of the job queue. It can be up to 128 letters long. It can contain uppercase and lowercase letters,
     * numbers, hyphens (-), and underscores (_).
     * </p>
     * 
     * @return The name of the job queue. It can be up to 128 letters long. It can contain uppercase and lowercase
     *         letters, numbers, hyphens (-), and underscores (_).
     */
    public final String jobQueueName() {
        return jobQueueName;
    }

    /**
     * <p>
     * The state of the job queue. If the job queue state is <code>ENABLED</code>, it is able to accept jobs. If the job
     * queue state is <code>DISABLED</code>, new jobs can't be added to the queue, but jobs already in the queue can
     * finish.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link JQState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the job queue. If the job queue state is <code>ENABLED</code>, it is able to accept jobs. If
     *         the job queue state is <code>DISABLED</code>, new jobs can't be added to the queue, but jobs already in
     *         the queue can finish.
     * @see JQState
     */
    public final JQState state() {
        return JQState.fromValue(state);
    }

    /**
     * <p>
     * The state of the job queue. If the job queue state is <code>ENABLED</code>, it is able to accept jobs. If the job
     * queue state is <code>DISABLED</code>, new jobs can't be added to the queue, but jobs already in the queue can
     * finish.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link JQState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the job queue. If the job queue state is <code>ENABLED</code>, it is able to accept jobs. If
     *         the job queue state is <code>DISABLED</code>, new jobs can't be added to the queue, but jobs already in
     *         the queue can finish.
     * @see JQState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the fair share scheduling policy. If this parameter is specified, the job queue
     * uses a fair share scheduling policy. If this parameter isn't specified, the job queue uses a first in, first out
     * (FIFO) scheduling policy. After a job queue is created, you can replace but can't remove the fair share
     * scheduling policy. The format is
     * <code>aws:<i>Partition</i>:batch:<i>Region</i>:<i>Account</i>:scheduling-policy/<i>Name</i> </code>. An example
     * is <code>aws:aws:batch:us-west-2:012345678910:scheduling-policy/MySchedulingPolicy</code>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the fair share scheduling policy. If this parameter is specified, the
     *         job queue uses a fair share scheduling policy. If this parameter isn't specified, the job queue uses a
     *         first in, first out (FIFO) scheduling policy. After a job queue is created, you can replace but can't
     *         remove the fair share scheduling policy. The format is
     *         <code>aws:<i>Partition</i>:batch:<i>Region</i>:<i>Account</i>:scheduling-policy/<i>Name</i> </code>. An
     *         example is <code>aws:aws:batch:us-west-2:012345678910:scheduling-policy/MySchedulingPolicy</code>.
     */
    public final String schedulingPolicyArn() {
        return schedulingPolicyArn;
    }

    /**
     * <p>
     * The priority of the job queue. Job queues with a higher priority (or a higher integer value for the
     * <code>priority</code> parameter) are evaluated first when associated with the same compute environment. Priority
     * is determined in descending order. For example, a job queue with a priority value of <code>10</code> is given
     * scheduling preference over a job queue with a priority value of <code>1</code>. All of the compute environments
     * must be either EC2 (<code>EC2</code> or <code>SPOT</code>) or Fargate (<code>FARGATE</code> or
     * <code>FARGATE_SPOT</code>); EC2 and Fargate compute environments can't be mixed.
     * </p>
     * 
     * @return The priority of the job queue. Job queues with a higher priority (or a higher integer value for the
     *         <code>priority</code> parameter) are evaluated first when associated with the same compute environment.
     *         Priority is determined in descending order. For example, a job queue with a priority value of
     *         <code>10</code> is given scheduling preference over a job queue with a priority value of <code>1</code>.
     *         All of the compute environments must be either EC2 (<code>EC2</code> or <code>SPOT</code>) or Fargate (
     *         <code>FARGATE</code> or <code>FARGATE_SPOT</code>); EC2 and Fargate compute environments can't be mixed.
     */
    public final Integer priority() {
        return priority;
    }

    /**
     * For responses, this returns true if the service returned a value for the ComputeEnvironmentOrder property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasComputeEnvironmentOrder() {
        return computeEnvironmentOrder != null && !(computeEnvironmentOrder instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The set of compute environments mapped to a job queue and their order relative to each other. The job scheduler
     * uses this parameter to determine which compute environment should run a specific job. Compute environments must
     * be in the <code>VALID</code> state before you can associate them with a job queue. You can associate up to three
     * compute environments with a job queue. All of the compute environments must be either EC2 (<code>EC2</code> or
     * <code>SPOT</code>) or Fargate (<code>FARGATE</code> or <code>FARGATE_SPOT</code>); EC2 and Fargate compute
     * environments can't be mixed.
     * </p>
     * <note>
     * <p>
     * All compute environments that are associated with a job queue must share the same architecture. Batch doesn't
     * support mixing compute environment architecture types in a single job queue.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasComputeEnvironmentOrder} method.
     * </p>
     * 
     * @return The set of compute environments mapped to a job queue and their order relative to each other. The job
     *         scheduler uses this parameter to determine which compute environment should run a specific job. Compute
     *         environments must be in the <code>VALID</code> state before you can associate them with a job queue. You
     *         can associate up to three compute environments with a job queue. All of the compute environments must be
     *         either EC2 (<code>EC2</code> or <code>SPOT</code>) or Fargate (<code>FARGATE</code> or
     *         <code>FARGATE_SPOT</code>); EC2 and Fargate compute environments can't be mixed.</p> <note>
     *         <p>
     *         All compute environments that are associated with a job queue must share the same architecture. Batch
     *         doesn't support mixing compute environment architecture types in a single job queue.
     *         </p>
     */
    public final List<ComputeEnvironmentOrder> computeEnvironmentOrder() {
        return computeEnvironmentOrder;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The tags that you apply to the job queue to help you categorize and organize your resources. Each tag consists of
     * a key and an optional value. For more information, see <a
     * href="https://docs.aws.amazon.com/batch/latest/userguide/using-tags.html">Tagging your Batch resources</a> in
     * <i>Batch User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags that you apply to the job queue to help you categorize and organize your resources. Each tag
     *         consists of a key and an optional value. For more information, see <a
     *         href="https://docs.aws.amazon.com/batch/latest/userguide/using-tags.html">Tagging your Batch
     *         resources</a> in <i>Batch User Guide</i>.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(jobQueueName());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(schedulingPolicyArn());
        hashCode = 31 * hashCode + Objects.hashCode(priority());
        hashCode = 31 * hashCode + Objects.hashCode(hasComputeEnvironmentOrder() ? computeEnvironmentOrder() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateJobQueueRequest)) {
            return false;
        }
        CreateJobQueueRequest other = (CreateJobQueueRequest) obj;
        return Objects.equals(jobQueueName(), other.jobQueueName()) && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(schedulingPolicyArn(), other.schedulingPolicyArn())
                && Objects.equals(priority(), other.priority())
                && hasComputeEnvironmentOrder() == other.hasComputeEnvironmentOrder()
                && Objects.equals(computeEnvironmentOrder(), other.computeEnvironmentOrder()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateJobQueueRequest").add("JobQueueName", jobQueueName()).add("State", stateAsString())
                .add("SchedulingPolicyArn", schedulingPolicyArn()).add("Priority", priority())
                .add("ComputeEnvironmentOrder", hasComputeEnvironmentOrder() ? computeEnvironmentOrder() : null)
                .add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "jobQueueName":
            return Optional.ofNullable(clazz.cast(jobQueueName()));
        case "state":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "schedulingPolicyArn":
            return Optional.ofNullable(clazz.cast(schedulingPolicyArn()));
        case "priority":
            return Optional.ofNullable(clazz.cast(priority()));
        case "computeEnvironmentOrder":
            return Optional.ofNullable(clazz.cast(computeEnvironmentOrder()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateJobQueueRequest, T> g) {
        return obj -> g.apply((CreateJobQueueRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends BatchRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateJobQueueRequest> {
        /**
         * <p>
         * The name of the job queue. It can be up to 128 letters long. It can contain uppercase and lowercase letters,
         * numbers, hyphens (-), and underscores (_).
         * </p>
         * 
         * @param jobQueueName
         *        The name of the job queue. It can be up to 128 letters long. It can contain uppercase and lowercase
         *        letters, numbers, hyphens (-), and underscores (_).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobQueueName(String jobQueueName);

        /**
         * <p>
         * The state of the job queue. If the job queue state is <code>ENABLED</code>, it is able to accept jobs. If the
         * job queue state is <code>DISABLED</code>, new jobs can't be added to the queue, but jobs already in the queue
         * can finish.
         * </p>
         * 
         * @param state
         *        The state of the job queue. If the job queue state is <code>ENABLED</code>, it is able to accept jobs.
         *        If the job queue state is <code>DISABLED</code>, new jobs can't be added to the queue, but jobs
         *        already in the queue can finish.
         * @see JQState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JQState
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the job queue. If the job queue state is <code>ENABLED</code>, it is able to accept jobs. If the
         * job queue state is <code>DISABLED</code>, new jobs can't be added to the queue, but jobs already in the queue
         * can finish.
         * </p>
         * 
         * @param state
         *        The state of the job queue. If the job queue state is <code>ENABLED</code>, it is able to accept jobs.
         *        If the job queue state is <code>DISABLED</code>, new jobs can't be added to the queue, but jobs
         *        already in the queue can finish.
         * @see JQState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JQState
         */
        Builder state(JQState state);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the fair share scheduling policy. If this parameter is specified, the job
         * queue uses a fair share scheduling policy. If this parameter isn't specified, the job queue uses a first in,
         * first out (FIFO) scheduling policy. After a job queue is created, you can replace but can't remove the fair
         * share scheduling policy. The format is
         * <code>aws:<i>Partition</i>:batch:<i>Region</i>:<i>Account</i>:scheduling-policy/<i>Name</i> </code>. An
         * example is <code>aws:aws:batch:us-west-2:012345678910:scheduling-policy/MySchedulingPolicy</code>.
         * </p>
         * 
         * @param schedulingPolicyArn
         *        The Amazon Resource Name (ARN) of the fair share scheduling policy. If this parameter is specified,
         *        the job queue uses a fair share scheduling policy. If this parameter isn't specified, the job queue
         *        uses a first in, first out (FIFO) scheduling policy. After a job queue is created, you can replace but
         *        can't remove the fair share scheduling policy. The format is
         *        <code>aws:<i>Partition</i>:batch:<i>Region</i>:<i>Account</i>:scheduling-policy/<i>Name</i> </code>.
         *        An example is <code>aws:aws:batch:us-west-2:012345678910:scheduling-policy/MySchedulingPolicy</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schedulingPolicyArn(String schedulingPolicyArn);

        /**
         * <p>
         * The priority of the job queue. Job queues with a higher priority (or a higher integer value for the
         * <code>priority</code> parameter) are evaluated first when associated with the same compute environment.
         * Priority is determined in descending order. For example, a job queue with a priority value of <code>10</code>
         * is given scheduling preference over a job queue with a priority value of <code>1</code>. All of the compute
         * environments must be either EC2 (<code>EC2</code> or <code>SPOT</code>) or Fargate (<code>FARGATE</code> or
         * <code>FARGATE_SPOT</code>); EC2 and Fargate compute environments can't be mixed.
         * </p>
         * 
         * @param priority
         *        The priority of the job queue. Job queues with a higher priority (or a higher integer value for the
         *        <code>priority</code> parameter) are evaluated first when associated with the same compute
         *        environment. Priority is determined in descending order. For example, a job queue with a priority
         *        value of <code>10</code> is given scheduling preference over a job queue with a priority value of
         *        <code>1</code>. All of the compute environments must be either EC2 (<code>EC2</code> or
         *        <code>SPOT</code>) or Fargate (<code>FARGATE</code> or <code>FARGATE_SPOT</code>); EC2 and Fargate
         *        compute environments can't be mixed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder priority(Integer priority);

        /**
         * <p>
         * The set of compute environments mapped to a job queue and their order relative to each other. The job
         * scheduler uses this parameter to determine which compute environment should run a specific job. Compute
         * environments must be in the <code>VALID</code> state before you can associate them with a job queue. You can
         * associate up to three compute environments with a job queue. All of the compute environments must be either
         * EC2 (<code>EC2</code> or <code>SPOT</code>) or Fargate (<code>FARGATE</code> or <code>FARGATE_SPOT</code>);
         * EC2 and Fargate compute environments can't be mixed.
         * </p>
         * <note>
         * <p>
         * All compute environments that are associated with a job queue must share the same architecture. Batch doesn't
         * support mixing compute environment architecture types in a single job queue.
         * </p>
         * </note>
         * 
         * @param computeEnvironmentOrder
         *        The set of compute environments mapped to a job queue and their order relative to each other. The job
         *        scheduler uses this parameter to determine which compute environment should run a specific job.
         *        Compute environments must be in the <code>VALID</code> state before you can associate them with a job
         *        queue. You can associate up to three compute environments with a job queue. All of the compute
         *        environments must be either EC2 (<code>EC2</code> or <code>SPOT</code>) or Fargate (
         *        <code>FARGATE</code> or <code>FARGATE_SPOT</code>); EC2 and Fargate compute environments can't be
         *        mixed.</p> <note>
         *        <p>
         *        All compute environments that are associated with a job queue must share the same architecture. Batch
         *        doesn't support mixing compute environment architecture types in a single job queue.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder computeEnvironmentOrder(Collection<ComputeEnvironmentOrder> computeEnvironmentOrder);

        /**
         * <p>
         * The set of compute environments mapped to a job queue and their order relative to each other. The job
         * scheduler uses this parameter to determine which compute environment should run a specific job. Compute
         * environments must be in the <code>VALID</code> state before you can associate them with a job queue. You can
         * associate up to three compute environments with a job queue. All of the compute environments must be either
         * EC2 (<code>EC2</code> or <code>SPOT</code>) or Fargate (<code>FARGATE</code> or <code>FARGATE_SPOT</code>);
         * EC2 and Fargate compute environments can't be mixed.
         * </p>
         * <note>
         * <p>
         * All compute environments that are associated with a job queue must share the same architecture. Batch doesn't
         * support mixing compute environment architecture types in a single job queue.
         * </p>
         * </note>
         * 
         * @param computeEnvironmentOrder
         *        The set of compute environments mapped to a job queue and their order relative to each other. The job
         *        scheduler uses this parameter to determine which compute environment should run a specific job.
         *        Compute environments must be in the <code>VALID</code> state before you can associate them with a job
         *        queue. You can associate up to three compute environments with a job queue. All of the compute
         *        environments must be either EC2 (<code>EC2</code> or <code>SPOT</code>) or Fargate (
         *        <code>FARGATE</code> or <code>FARGATE_SPOT</code>); EC2 and Fargate compute environments can't be
         *        mixed.</p> <note>
         *        <p>
         *        All compute environments that are associated with a job queue must share the same architecture. Batch
         *        doesn't support mixing compute environment architecture types in a single job queue.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder computeEnvironmentOrder(ComputeEnvironmentOrder... computeEnvironmentOrder);

        /**
         * <p>
         * The set of compute environments mapped to a job queue and their order relative to each other. The job
         * scheduler uses this parameter to determine which compute environment should run a specific job. Compute
         * environments must be in the <code>VALID</code> state before you can associate them with a job queue. You can
         * associate up to three compute environments with a job queue. All of the compute environments must be either
         * EC2 (<code>EC2</code> or <code>SPOT</code>) or Fargate (<code>FARGATE</code> or <code>FARGATE_SPOT</code>);
         * EC2 and Fargate compute environments can't be mixed.
         * </p>
         * <note>
         * <p>
         * All compute environments that are associated with a job queue must share the same architecture. Batch doesn't
         * support mixing compute environment architecture types in a single job queue.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link List
         * <ComputeEnvironmentOrder>.Builder} avoiding the need to create one manually via {@link List
         * <ComputeEnvironmentOrder>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ComputeEnvironmentOrder>.Builder#build()} is called
         * immediately and its result is passed to {@link #computeEnvironmentOrder(List<ComputeEnvironmentOrder>)}.
         * 
         * @param computeEnvironmentOrder
         *        a consumer that will call methods on {@link List<ComputeEnvironmentOrder>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #computeEnvironmentOrder(List<ComputeEnvironmentOrder>)
         */
        Builder computeEnvironmentOrder(Consumer<ComputeEnvironmentOrder.Builder>... computeEnvironmentOrder);

        /**
         * <p>
         * The tags that you apply to the job queue to help you categorize and organize your resources. Each tag
         * consists of a key and an optional value. For more information, see <a
         * href="https://docs.aws.amazon.com/batch/latest/userguide/using-tags.html">Tagging your Batch resources</a> in
         * <i>Batch User Guide</i>.
         * </p>
         * 
         * @param tags
         *        The tags that you apply to the job queue to help you categorize and organize your resources. Each tag
         *        consists of a key and an optional value. For more information, see <a
         *        href="https://docs.aws.amazon.com/batch/latest/userguide/using-tags.html">Tagging your Batch
         *        resources</a> in <i>Batch User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends BatchRequest.BuilderImpl implements Builder {
        private String jobQueueName;

        private String state;

        private String schedulingPolicyArn;

        private Integer priority;

        private List<ComputeEnvironmentOrder> computeEnvironmentOrder = DefaultSdkAutoConstructList.getInstance();

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateJobQueueRequest model) {
            super(model);
            jobQueueName(model.jobQueueName);
            state(model.state);
            schedulingPolicyArn(model.schedulingPolicyArn);
            priority(model.priority);
            computeEnvironmentOrder(model.computeEnvironmentOrder);
            tags(model.tags);
        }

        public final String getJobQueueName() {
            return jobQueueName;
        }

        public final void setJobQueueName(String jobQueueName) {
            this.jobQueueName = jobQueueName;
        }

        @Override
        @Transient
        public final Builder jobQueueName(String jobQueueName) {
            this.jobQueueName = jobQueueName;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        @Transient
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        @Transient
        public final Builder state(JQState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final String getSchedulingPolicyArn() {
            return schedulingPolicyArn;
        }

        public final void setSchedulingPolicyArn(String schedulingPolicyArn) {
            this.schedulingPolicyArn = schedulingPolicyArn;
        }

        @Override
        @Transient
        public final Builder schedulingPolicyArn(String schedulingPolicyArn) {
            this.schedulingPolicyArn = schedulingPolicyArn;
            return this;
        }

        public final Integer getPriority() {
            return priority;
        }

        public final void setPriority(Integer priority) {
            this.priority = priority;
        }

        @Override
        @Transient
        public final Builder priority(Integer priority) {
            this.priority = priority;
            return this;
        }

        public final List<ComputeEnvironmentOrder.Builder> getComputeEnvironmentOrder() {
            List<ComputeEnvironmentOrder.Builder> result = ComputeEnvironmentOrdersCopier
                    .copyToBuilder(this.computeEnvironmentOrder);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setComputeEnvironmentOrder(Collection<ComputeEnvironmentOrder.BuilderImpl> computeEnvironmentOrder) {
            this.computeEnvironmentOrder = ComputeEnvironmentOrdersCopier.copyFromBuilder(computeEnvironmentOrder);
        }

        @Override
        @Transient
        public final Builder computeEnvironmentOrder(Collection<ComputeEnvironmentOrder> computeEnvironmentOrder) {
            this.computeEnvironmentOrder = ComputeEnvironmentOrdersCopier.copy(computeEnvironmentOrder);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder computeEnvironmentOrder(ComputeEnvironmentOrder... computeEnvironmentOrder) {
            computeEnvironmentOrder(Arrays.asList(computeEnvironmentOrder));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder computeEnvironmentOrder(Consumer<ComputeEnvironmentOrder.Builder>... computeEnvironmentOrder) {
            computeEnvironmentOrder(Stream.of(computeEnvironmentOrder)
                    .map(c -> ComputeEnvironmentOrder.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagrisTagsMapCopier.copy(tags);
        }

        @Override
        @Transient
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagrisTagsMapCopier.copy(tags);
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateJobQueueRequest build() {
            return new CreateJobQueueRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
