/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.batch.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Object representing any node overrides to a job definition that's used in a <a>SubmitJob</a> API operation.
 * </p>
 * <note>
 * <p>
 * This isn't applicable to jobs that are running on Fargate resources and shouldn't be provided; use
 * <code>containerOverrides</code> instead.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class NodeOverrides implements SdkPojo, Serializable, ToCopyableBuilder<NodeOverrides.Builder, NodeOverrides> {
    private static final SdkField<Integer> NUM_NODES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("numNodes").getter(getter(NodeOverrides::numNodes)).setter(setter(Builder::numNodes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("numNodes").build()).build();

    private static final SdkField<List<NodePropertyOverride>> NODE_PROPERTY_OVERRIDES_FIELD = SdkField
            .<List<NodePropertyOverride>> builder(MarshallingType.LIST)
            .memberName("nodePropertyOverrides")
            .getter(getter(NodeOverrides::nodePropertyOverrides))
            .setter(setter(Builder::nodePropertyOverrides))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("nodePropertyOverrides").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<NodePropertyOverride> builder(MarshallingType.SDK_POJO)
                                            .constructor(NodePropertyOverride::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NUM_NODES_FIELD,
            NODE_PROPERTY_OVERRIDES_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer numNodes;

    private final List<NodePropertyOverride> nodePropertyOverrides;

    private NodeOverrides(BuilderImpl builder) {
        this.numNodes = builder.numNodes;
        this.nodePropertyOverrides = builder.nodePropertyOverrides;
    }

    /**
     * <p>
     * The number of nodes to use with a multi-node parallel job. This value overrides the number of nodes that are
     * specified in the job definition. To use this override:
     * </p>
     * <ul>
     * <li>
     * <p>
     * There must be at least one node range in your job definition that has an open upper boundary (such as
     * <code>:</code> or <code>n:</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * The lower boundary of the node range specified in the job definition must be fewer than the number of nodes
     * specified in the override.
     * </p>
     * </li>
     * <li>
     * <p>
     * The main node index specified in the job definition must be fewer than the number of nodes specified in the
     * override.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The number of nodes to use with a multi-node parallel job. This value overrides the number of nodes that
     *         are specified in the job definition. To use this override:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         There must be at least one node range in your job definition that has an open upper boundary (such as
     *         <code>:</code> or <code>n:</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The lower boundary of the node range specified in the job definition must be fewer than the number of
     *         nodes specified in the override.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The main node index specified in the job definition must be fewer than the number of nodes specified in
     *         the override.
     *         </p>
     *         </li>
     */
    public final Integer numNodes() {
        return numNodes;
    }

    /**
     * For responses, this returns true if the service returned a value for the NodePropertyOverrides property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasNodePropertyOverrides() {
        return nodePropertyOverrides != null && !(nodePropertyOverrides instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The node property overrides for the job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNodePropertyOverrides} method.
     * </p>
     * 
     * @return The node property overrides for the job.
     */
    public final List<NodePropertyOverride> nodePropertyOverrides() {
        return nodePropertyOverrides;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(numNodes());
        hashCode = 31 * hashCode + Objects.hashCode(hasNodePropertyOverrides() ? nodePropertyOverrides() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof NodeOverrides)) {
            return false;
        }
        NodeOverrides other = (NodeOverrides) obj;
        return Objects.equals(numNodes(), other.numNodes()) && hasNodePropertyOverrides() == other.hasNodePropertyOverrides()
                && Objects.equals(nodePropertyOverrides(), other.nodePropertyOverrides());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("NodeOverrides").add("NumNodes", numNodes())
                .add("NodePropertyOverrides", hasNodePropertyOverrides() ? nodePropertyOverrides() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "numNodes":
            return Optional.ofNullable(clazz.cast(numNodes()));
        case "nodePropertyOverrides":
            return Optional.ofNullable(clazz.cast(nodePropertyOverrides()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<NodeOverrides, T> g) {
        return obj -> g.apply((NodeOverrides) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, NodeOverrides> {
        /**
         * <p>
         * The number of nodes to use with a multi-node parallel job. This value overrides the number of nodes that are
         * specified in the job definition. To use this override:
         * </p>
         * <ul>
         * <li>
         * <p>
         * There must be at least one node range in your job definition that has an open upper boundary (such as
         * <code>:</code> or <code>n:</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * The lower boundary of the node range specified in the job definition must be fewer than the number of nodes
         * specified in the override.
         * </p>
         * </li>
         * <li>
         * <p>
         * The main node index specified in the job definition must be fewer than the number of nodes specified in the
         * override.
         * </p>
         * </li>
         * </ul>
         * 
         * @param numNodes
         *        The number of nodes to use with a multi-node parallel job. This value overrides the number of nodes
         *        that are specified in the job definition. To use this override:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        There must be at least one node range in your job definition that has an open upper boundary (such as
         *        <code>:</code> or <code>n:</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The lower boundary of the node range specified in the job definition must be fewer than the number of
         *        nodes specified in the override.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The main node index specified in the job definition must be fewer than the number of nodes specified
         *        in the override.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder numNodes(Integer numNodes);

        /**
         * <p>
         * The node property overrides for the job.
         * </p>
         * 
         * @param nodePropertyOverrides
         *        The node property overrides for the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodePropertyOverrides(Collection<NodePropertyOverride> nodePropertyOverrides);

        /**
         * <p>
         * The node property overrides for the job.
         * </p>
         * 
         * @param nodePropertyOverrides
         *        The node property overrides for the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodePropertyOverrides(NodePropertyOverride... nodePropertyOverrides);

        /**
         * <p>
         * The node property overrides for the job.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<NodePropertyOverride>.Builder}
         * avoiding the need to create one manually via {@link List<NodePropertyOverride>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<NodePropertyOverride>.Builder#build()} is called immediately
         * and its result is passed to {@link #nodePropertyOverrides(List<NodePropertyOverride>)}.
         * 
         * @param nodePropertyOverrides
         *        a consumer that will call methods on {@link List<NodePropertyOverride>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #nodePropertyOverrides(List<NodePropertyOverride>)
         */
        Builder nodePropertyOverrides(Consumer<NodePropertyOverride.Builder>... nodePropertyOverrides);
    }

    static final class BuilderImpl implements Builder {
        private Integer numNodes;

        private List<NodePropertyOverride> nodePropertyOverrides = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(NodeOverrides model) {
            numNodes(model.numNodes);
            nodePropertyOverrides(model.nodePropertyOverrides);
        }

        public final Integer getNumNodes() {
            return numNodes;
        }

        public final void setNumNodes(Integer numNodes) {
            this.numNodes = numNodes;
        }

        @Override
        @Transient
        public final Builder numNodes(Integer numNodes) {
            this.numNodes = numNodes;
            return this;
        }

        public final List<NodePropertyOverride.Builder> getNodePropertyOverrides() {
            List<NodePropertyOverride.Builder> result = NodePropertyOverridesCopier.copyToBuilder(this.nodePropertyOverrides);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setNodePropertyOverrides(Collection<NodePropertyOverride.BuilderImpl> nodePropertyOverrides) {
            this.nodePropertyOverrides = NodePropertyOverridesCopier.copyFromBuilder(nodePropertyOverrides);
        }

        @Override
        @Transient
        public final Builder nodePropertyOverrides(Collection<NodePropertyOverride> nodePropertyOverrides) {
            this.nodePropertyOverrides = NodePropertyOverridesCopier.copy(nodePropertyOverrides);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder nodePropertyOverrides(NodePropertyOverride... nodePropertyOverrides) {
            nodePropertyOverrides(Arrays.asList(nodePropertyOverrides));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder nodePropertyOverrides(Consumer<NodePropertyOverride.Builder>... nodePropertyOverrides) {
            nodePropertyOverrides(Stream.of(nodePropertyOverrides)
                    .map(c -> NodePropertyOverride.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public NodeOverrides build() {
            return new NodeOverrides(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
