/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.batch;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.core.SdkClient;
import software.amazon.awssdk.services.batch.model.CancelJobRequest;
import software.amazon.awssdk.services.batch.model.CancelJobResponse;
import software.amazon.awssdk.services.batch.model.CreateComputeEnvironmentRequest;
import software.amazon.awssdk.services.batch.model.CreateComputeEnvironmentResponse;
import software.amazon.awssdk.services.batch.model.CreateJobQueueRequest;
import software.amazon.awssdk.services.batch.model.CreateJobQueueResponse;
import software.amazon.awssdk.services.batch.model.CreateSchedulingPolicyRequest;
import software.amazon.awssdk.services.batch.model.CreateSchedulingPolicyResponse;
import software.amazon.awssdk.services.batch.model.DeleteComputeEnvironmentRequest;
import software.amazon.awssdk.services.batch.model.DeleteComputeEnvironmentResponse;
import software.amazon.awssdk.services.batch.model.DeleteJobQueueRequest;
import software.amazon.awssdk.services.batch.model.DeleteJobQueueResponse;
import software.amazon.awssdk.services.batch.model.DeleteSchedulingPolicyRequest;
import software.amazon.awssdk.services.batch.model.DeleteSchedulingPolicyResponse;
import software.amazon.awssdk.services.batch.model.DeregisterJobDefinitionRequest;
import software.amazon.awssdk.services.batch.model.DeregisterJobDefinitionResponse;
import software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsRequest;
import software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsResponse;
import software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsRequest;
import software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsResponse;
import software.amazon.awssdk.services.batch.model.DescribeJobQueuesRequest;
import software.amazon.awssdk.services.batch.model.DescribeJobQueuesResponse;
import software.amazon.awssdk.services.batch.model.DescribeJobsRequest;
import software.amazon.awssdk.services.batch.model.DescribeJobsResponse;
import software.amazon.awssdk.services.batch.model.DescribeSchedulingPoliciesRequest;
import software.amazon.awssdk.services.batch.model.DescribeSchedulingPoliciesResponse;
import software.amazon.awssdk.services.batch.model.ListJobsRequest;
import software.amazon.awssdk.services.batch.model.ListJobsResponse;
import software.amazon.awssdk.services.batch.model.ListSchedulingPoliciesRequest;
import software.amazon.awssdk.services.batch.model.ListSchedulingPoliciesResponse;
import software.amazon.awssdk.services.batch.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.batch.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.batch.model.RegisterJobDefinitionRequest;
import software.amazon.awssdk.services.batch.model.RegisterJobDefinitionResponse;
import software.amazon.awssdk.services.batch.model.SubmitJobRequest;
import software.amazon.awssdk.services.batch.model.SubmitJobResponse;
import software.amazon.awssdk.services.batch.model.TagResourceRequest;
import software.amazon.awssdk.services.batch.model.TagResourceResponse;
import software.amazon.awssdk.services.batch.model.TerminateJobRequest;
import software.amazon.awssdk.services.batch.model.TerminateJobResponse;
import software.amazon.awssdk.services.batch.model.UntagResourceRequest;
import software.amazon.awssdk.services.batch.model.UntagResourceResponse;
import software.amazon.awssdk.services.batch.model.UpdateComputeEnvironmentRequest;
import software.amazon.awssdk.services.batch.model.UpdateComputeEnvironmentResponse;
import software.amazon.awssdk.services.batch.model.UpdateJobQueueRequest;
import software.amazon.awssdk.services.batch.model.UpdateJobQueueResponse;
import software.amazon.awssdk.services.batch.model.UpdateSchedulingPolicyRequest;
import software.amazon.awssdk.services.batch.model.UpdateSchedulingPolicyResponse;
import software.amazon.awssdk.services.batch.paginators.DescribeComputeEnvironmentsPublisher;
import software.amazon.awssdk.services.batch.paginators.DescribeJobDefinitionsPublisher;
import software.amazon.awssdk.services.batch.paginators.DescribeJobQueuesPublisher;
import software.amazon.awssdk.services.batch.paginators.ListJobsPublisher;
import software.amazon.awssdk.services.batch.paginators.ListSchedulingPoliciesPublisher;

/**
 * Service client for accessing AWS Batch asynchronously. This can be created using the static {@link #builder()}
 * method.
 *
 * <fullname>Batch</fullname>
 * <p>
 * Using Batch, you can run batch computing workloads on the Amazon Web Services Cloud. Batch computing is a common
 * means for developers, scientists, and engineers to access large amounts of compute resources. Batch uses the
 * advantages of this computing workload to remove the undifferentiated heavy lifting of configuring and managing
 * required infrastructure. At the same time, it also adopts a familiar batch computing software approach. Given these
 * advantages, Batch can help you to efficiently provision resources in response to jobs submitted, thus effectively
 * helping you to eliminate capacity constraints, reduce compute costs, and deliver your results more quickly.
 * </p>
 * <p>
 * As a fully managed service, Batch can run batch computing workloads of any scale. Batch automatically provisions
 * compute resources and optimizes workload distribution based on the quantity and scale of your specific workloads.
 * With Batch, there's no need to install or manage batch computing software. This means that you can focus your time
 * and energy on analyzing results and solving your specific problems.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface BatchAsyncClient extends SdkClient {
    String SERVICE_NAME = "batch";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "batch";

    /**
     * Create a {@link BatchAsyncClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static BatchAsyncClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link BatchAsyncClient}.
     */
    static BatchAsyncClientBuilder builder() {
        return new DefaultBatchAsyncClientBuilder();
    }

    /**
     * <p>
     * Cancels a job in an Batch job queue. Jobs that are in the <code>SUBMITTED</code>, <code>PENDING</code>, or
     * <code>RUNNABLE</code> state are canceled. Jobs that have progressed to <code>STARTING</code> or
     * <code>RUNNING</code> aren't canceled, but the API operation still succeeds, even if no job is canceled. These
     * jobs must be terminated with the <a>TerminateJob</a> operation.
     * </p>
     *
     * @param cancelJobRequest
     *        Contains the parameters for <code>CancelJob</code>.
     * @return A Java Future containing the result of the CancelJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.CancelJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/CancelJob" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CancelJobResponse> cancelJob(CancelJobRequest cancelJobRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Cancels a job in an Batch job queue. Jobs that are in the <code>SUBMITTED</code>, <code>PENDING</code>, or
     * <code>RUNNABLE</code> state are canceled. Jobs that have progressed to <code>STARTING</code> or
     * <code>RUNNING</code> aren't canceled, but the API operation still succeeds, even if no job is canceled. These
     * jobs must be terminated with the <a>TerminateJob</a> operation.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CancelJobRequest.Builder} avoiding the need to
     * create one manually via {@link CancelJobRequest#builder()}
     * </p>
     *
     * @param cancelJobRequest
     *        A {@link Consumer} that will call methods on {@link CancelJobRequest.Builder} to create a request.
     *        Contains the parameters for <code>CancelJob</code>.
     * @return A Java Future containing the result of the CancelJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.CancelJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/CancelJob" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CancelJobResponse> cancelJob(Consumer<CancelJobRequest.Builder> cancelJobRequest) {
        return cancelJob(CancelJobRequest.builder().applyMutation(cancelJobRequest).build());
    }

    /**
     * <p>
     * Creates an Batch compute environment. You can create <code>MANAGED</code> or <code>UNMANAGED</code> compute
     * environments. <code>MANAGED</code> compute environments can use Amazon EC2 or Fargate resources.
     * <code>UNMANAGED</code> compute environments can only use EC2 resources.
     * </p>
     * <p>
     * In a managed compute environment, Batch manages the capacity and instance types of the compute resources within
     * the environment. This is based on the compute resource specification that you define or the <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html">launch template</a> that you
     * specify when you create the compute environment. Either, you can choose to use EC2 On-Demand Instances and EC2
     * Spot Instances. Or, you can use Fargate and Fargate Spot capacity in your managed compute environment. You can
     * optionally set a maximum price so that Spot Instances only launch when the Spot Instance price is less than a
     * specified percentage of the On-Demand price.
     * </p>
     * <note>
     * <p>
     * Multi-node parallel jobs aren't supported on Spot Instances.
     * </p>
     * </note>
     * <p>
     * In an unmanaged compute environment, you can manage your own EC2 compute resources and have a lot of flexibility
     * with how you configure your compute resources. For example, you can use custom AMIs. However, you must verify
     * that each of your AMIs meet the Amazon ECS container instance AMI specification. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container_instance_AMIs.html">container
     * instance AMIs</a> in the <i>Amazon Elastic Container Service Developer Guide</i>. After you created your
     * unmanaged compute environment, you can use the <a>DescribeComputeEnvironments</a> operation to find the Amazon
     * ECS cluster that's associated with it. Then, launch your container instances into that Amazon ECS cluster. For
     * more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_container_instance.html">Launching an
     * Amazon ECS container instance</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <note>
     * <p>
     * Batch doesn't upgrade the AMIs in a compute environment after the environment is created. For example, it doesn't
     * update the AMIs when a newer version of the Amazon ECS optimized AMI is available. Therefore, you're responsible
     * for managing the guest operating system (including its updates and security patches) and any additional
     * application software or utilities that you install on the compute resources. To use a new AMI for your Batch
     * jobs, complete these steps:
     * </p>
     * <ol>
     * <li>
     * <p>
     * Create a new compute environment with the new AMI.
     * </p>
     * </li>
     * <li>
     * <p>
     * Add the compute environment to an existing job queue.
     * </p>
     * </li>
     * <li>
     * <p>
     * Remove the earlier compute environment from your job queue.
     * </p>
     * </li>
     * <li>
     * <p>
     * Delete the earlier compute environment.
     * </p>
     * </li>
     * </ol>
     * </note>
     *
     * @param createComputeEnvironmentRequest
     *        Contains the parameters for <code>CreateComputeEnvironment</code>.
     * @return A Java Future containing the result of the CreateComputeEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.CreateComputeEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/CreateComputeEnvironment"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<CreateComputeEnvironmentResponse> createComputeEnvironment(
            CreateComputeEnvironmentRequest createComputeEnvironmentRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates an Batch compute environment. You can create <code>MANAGED</code> or <code>UNMANAGED</code> compute
     * environments. <code>MANAGED</code> compute environments can use Amazon EC2 or Fargate resources.
     * <code>UNMANAGED</code> compute environments can only use EC2 resources.
     * </p>
     * <p>
     * In a managed compute environment, Batch manages the capacity and instance types of the compute resources within
     * the environment. This is based on the compute resource specification that you define or the <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html">launch template</a> that you
     * specify when you create the compute environment. Either, you can choose to use EC2 On-Demand Instances and EC2
     * Spot Instances. Or, you can use Fargate and Fargate Spot capacity in your managed compute environment. You can
     * optionally set a maximum price so that Spot Instances only launch when the Spot Instance price is less than a
     * specified percentage of the On-Demand price.
     * </p>
     * <note>
     * <p>
     * Multi-node parallel jobs aren't supported on Spot Instances.
     * </p>
     * </note>
     * <p>
     * In an unmanaged compute environment, you can manage your own EC2 compute resources and have a lot of flexibility
     * with how you configure your compute resources. For example, you can use custom AMIs. However, you must verify
     * that each of your AMIs meet the Amazon ECS container instance AMI specification. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container_instance_AMIs.html">container
     * instance AMIs</a> in the <i>Amazon Elastic Container Service Developer Guide</i>. After you created your
     * unmanaged compute environment, you can use the <a>DescribeComputeEnvironments</a> operation to find the Amazon
     * ECS cluster that's associated with it. Then, launch your container instances into that Amazon ECS cluster. For
     * more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_container_instance.html">Launching an
     * Amazon ECS container instance</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <note>
     * <p>
     * Batch doesn't upgrade the AMIs in a compute environment after the environment is created. For example, it doesn't
     * update the AMIs when a newer version of the Amazon ECS optimized AMI is available. Therefore, you're responsible
     * for managing the guest operating system (including its updates and security patches) and any additional
     * application software or utilities that you install on the compute resources. To use a new AMI for your Batch
     * jobs, complete these steps:
     * </p>
     * <ol>
     * <li>
     * <p>
     * Create a new compute environment with the new AMI.
     * </p>
     * </li>
     * <li>
     * <p>
     * Add the compute environment to an existing job queue.
     * </p>
     * </li>
     * <li>
     * <p>
     * Remove the earlier compute environment from your job queue.
     * </p>
     * </li>
     * <li>
     * <p>
     * Delete the earlier compute environment.
     * </p>
     * </li>
     * </ol>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateComputeEnvironmentRequest.Builder} avoiding
     * the need to create one manually via {@link CreateComputeEnvironmentRequest#builder()}
     * </p>
     *
     * @param createComputeEnvironmentRequest
     *        A {@link Consumer} that will call methods on {@link CreateComputeEnvironmentRequest.Builder} to create a
     *        request. Contains the parameters for <code>CreateComputeEnvironment</code>.
     * @return A Java Future containing the result of the CreateComputeEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.CreateComputeEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/CreateComputeEnvironment"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<CreateComputeEnvironmentResponse> createComputeEnvironment(
            Consumer<CreateComputeEnvironmentRequest.Builder> createComputeEnvironmentRequest) {
        return createComputeEnvironment(CreateComputeEnvironmentRequest.builder().applyMutation(createComputeEnvironmentRequest)
                .build());
    }

    /**
     * <p>
     * Creates an Batch job queue. When you create a job queue, you associate one or more compute environments to the
     * queue and assign an order of preference for the compute environments.
     * </p>
     * <p>
     * You also set a priority to the job queue that determines the order that the Batch scheduler places jobs onto its
     * associated compute environments. For example, if a compute environment is associated with more than one job
     * queue, the job queue with a higher priority is given preference for scheduling jobs to that compute environment.
     * </p>
     *
     * @param createJobQueueRequest
     *        Contains the parameters for <code>CreateJobQueue</code>.
     * @return A Java Future containing the result of the CreateJobQueue operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.CreateJobQueue
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/CreateJobQueue" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreateJobQueueResponse> createJobQueue(CreateJobQueueRequest createJobQueueRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates an Batch job queue. When you create a job queue, you associate one or more compute environments to the
     * queue and assign an order of preference for the compute environments.
     * </p>
     * <p>
     * You also set a priority to the job queue that determines the order that the Batch scheduler places jobs onto its
     * associated compute environments. For example, if a compute environment is associated with more than one job
     * queue, the job queue with a higher priority is given preference for scheduling jobs to that compute environment.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateJobQueueRequest.Builder} avoiding the need to
     * create one manually via {@link CreateJobQueueRequest#builder()}
     * </p>
     *
     * @param createJobQueueRequest
     *        A {@link Consumer} that will call methods on {@link CreateJobQueueRequest.Builder} to create a request.
     *        Contains the parameters for <code>CreateJobQueue</code>.
     * @return A Java Future containing the result of the CreateJobQueue operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.CreateJobQueue
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/CreateJobQueue" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreateJobQueueResponse> createJobQueue(Consumer<CreateJobQueueRequest.Builder> createJobQueueRequest) {
        return createJobQueue(CreateJobQueueRequest.builder().applyMutation(createJobQueueRequest).build());
    }

    /**
     * <p>
     * Creates an Batch scheduling policy.
     * </p>
     *
     * @param createSchedulingPolicyRequest
     *        Contains the parameters for <code>CreateSchedulingPolicy</code>.
     * @return A Java Future containing the result of the CreateSchedulingPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.CreateSchedulingPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/CreateSchedulingPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<CreateSchedulingPolicyResponse> createSchedulingPolicy(
            CreateSchedulingPolicyRequest createSchedulingPolicyRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates an Batch scheduling policy.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateSchedulingPolicyRequest.Builder} avoiding the
     * need to create one manually via {@link CreateSchedulingPolicyRequest#builder()}
     * </p>
     *
     * @param createSchedulingPolicyRequest
     *        A {@link Consumer} that will call methods on {@link CreateSchedulingPolicyRequest.Builder} to create a
     *        request. Contains the parameters for <code>CreateSchedulingPolicy</code>.
     * @return A Java Future containing the result of the CreateSchedulingPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.CreateSchedulingPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/CreateSchedulingPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<CreateSchedulingPolicyResponse> createSchedulingPolicy(
            Consumer<CreateSchedulingPolicyRequest.Builder> createSchedulingPolicyRequest) {
        return createSchedulingPolicy(CreateSchedulingPolicyRequest.builder().applyMutation(createSchedulingPolicyRequest)
                .build());
    }

    /**
     * <p>
     * Deletes an Batch compute environment.
     * </p>
     * <p>
     * Before you can delete a compute environment, you must set its state to <code>DISABLED</code> with the
     * <a>UpdateComputeEnvironment</a> API operation and disassociate it from any job queues with the
     * <a>UpdateJobQueue</a> API operation. Compute environments that use Fargate resources must terminate all active
     * jobs on that compute environment before deleting the compute environment. If this isn't done, the compute
     * environment enters an invalid state.
     * </p>
     *
     * @param deleteComputeEnvironmentRequest
     *        Contains the parameters for <code>DeleteComputeEnvironment</code>.
     * @return A Java Future containing the result of the DeleteComputeEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DeleteComputeEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DeleteComputeEnvironment"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DeleteComputeEnvironmentResponse> deleteComputeEnvironment(
            DeleteComputeEnvironmentRequest deleteComputeEnvironmentRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes an Batch compute environment.
     * </p>
     * <p>
     * Before you can delete a compute environment, you must set its state to <code>DISABLED</code> with the
     * <a>UpdateComputeEnvironment</a> API operation and disassociate it from any job queues with the
     * <a>UpdateJobQueue</a> API operation. Compute environments that use Fargate resources must terminate all active
     * jobs on that compute environment before deleting the compute environment. If this isn't done, the compute
     * environment enters an invalid state.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteComputeEnvironmentRequest.Builder} avoiding
     * the need to create one manually via {@link DeleteComputeEnvironmentRequest#builder()}
     * </p>
     *
     * @param deleteComputeEnvironmentRequest
     *        A {@link Consumer} that will call methods on {@link DeleteComputeEnvironmentRequest.Builder} to create a
     *        request. Contains the parameters for <code>DeleteComputeEnvironment</code>.
     * @return A Java Future containing the result of the DeleteComputeEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DeleteComputeEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DeleteComputeEnvironment"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DeleteComputeEnvironmentResponse> deleteComputeEnvironment(
            Consumer<DeleteComputeEnvironmentRequest.Builder> deleteComputeEnvironmentRequest) {
        return deleteComputeEnvironment(DeleteComputeEnvironmentRequest.builder().applyMutation(deleteComputeEnvironmentRequest)
                .build());
    }

    /**
     * <p>
     * Deletes the specified job queue. You must first disable submissions for a queue with the <a>UpdateJobQueue</a>
     * operation. All jobs in the queue are eventually terminated when you delete a job queue. The jobs are terminated
     * at a rate of about 16 jobs each second.
     * </p>
     * <p>
     * It's not necessary to disassociate compute environments from a queue before submitting a
     * <code>DeleteJobQueue</code> request.
     * </p>
     *
     * @param deleteJobQueueRequest
     *        Contains the parameters for <code>DeleteJobQueue</code>.
     * @return A Java Future containing the result of the DeleteJobQueue operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DeleteJobQueue
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DeleteJobQueue" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteJobQueueResponse> deleteJobQueue(DeleteJobQueueRequest deleteJobQueueRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the specified job queue. You must first disable submissions for a queue with the <a>UpdateJobQueue</a>
     * operation. All jobs in the queue are eventually terminated when you delete a job queue. The jobs are terminated
     * at a rate of about 16 jobs each second.
     * </p>
     * <p>
     * It's not necessary to disassociate compute environments from a queue before submitting a
     * <code>DeleteJobQueue</code> request.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteJobQueueRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteJobQueueRequest#builder()}
     * </p>
     *
     * @param deleteJobQueueRequest
     *        A {@link Consumer} that will call methods on {@link DeleteJobQueueRequest.Builder} to create a request.
     *        Contains the parameters for <code>DeleteJobQueue</code>.
     * @return A Java Future containing the result of the DeleteJobQueue operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DeleteJobQueue
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DeleteJobQueue" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteJobQueueResponse> deleteJobQueue(Consumer<DeleteJobQueueRequest.Builder> deleteJobQueueRequest) {
        return deleteJobQueue(DeleteJobQueueRequest.builder().applyMutation(deleteJobQueueRequest).build());
    }

    /**
     * <p>
     * Deletes the specified scheduling policy.
     * </p>
     * <p>
     * You can't delete a scheduling policy that's used in any job queues.
     * </p>
     *
     * @param deleteSchedulingPolicyRequest
     *        Contains the parameters for <code>DeleteSchedulingPolicy</code>.
     * @return A Java Future containing the result of the DeleteSchedulingPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DeleteSchedulingPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DeleteSchedulingPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<DeleteSchedulingPolicyResponse> deleteSchedulingPolicy(
            DeleteSchedulingPolicyRequest deleteSchedulingPolicyRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the specified scheduling policy.
     * </p>
     * <p>
     * You can't delete a scheduling policy that's used in any job queues.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteSchedulingPolicyRequest.Builder} avoiding the
     * need to create one manually via {@link DeleteSchedulingPolicyRequest#builder()}
     * </p>
     *
     * @param deleteSchedulingPolicyRequest
     *        A {@link Consumer} that will call methods on {@link DeleteSchedulingPolicyRequest.Builder} to create a
     *        request. Contains the parameters for <code>DeleteSchedulingPolicy</code>.
     * @return A Java Future containing the result of the DeleteSchedulingPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DeleteSchedulingPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DeleteSchedulingPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<DeleteSchedulingPolicyResponse> deleteSchedulingPolicy(
            Consumer<DeleteSchedulingPolicyRequest.Builder> deleteSchedulingPolicyRequest) {
        return deleteSchedulingPolicy(DeleteSchedulingPolicyRequest.builder().applyMutation(deleteSchedulingPolicyRequest)
                .build());
    }

    /**
     * <p>
     * Deregisters an Batch job definition. Job definitions are permanently deleted after 180 days.
     * </p>
     *
     * @param deregisterJobDefinitionRequest
     * @return A Java Future containing the result of the DeregisterJobDefinition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DeregisterJobDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DeregisterJobDefinition" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<DeregisterJobDefinitionResponse> deregisterJobDefinition(
            DeregisterJobDefinitionRequest deregisterJobDefinitionRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deregisters an Batch job definition. Job definitions are permanently deleted after 180 days.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeregisterJobDefinitionRequest.Builder} avoiding
     * the need to create one manually via {@link DeregisterJobDefinitionRequest#builder()}
     * </p>
     *
     * @param deregisterJobDefinitionRequest
     *        A {@link Consumer} that will call methods on {@link DeregisterJobDefinitionRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the DeregisterJobDefinition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DeregisterJobDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DeregisterJobDefinition" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<DeregisterJobDefinitionResponse> deregisterJobDefinition(
            Consumer<DeregisterJobDefinitionRequest.Builder> deregisterJobDefinitionRequest) {
        return deregisterJobDefinition(DeregisterJobDefinitionRequest.builder().applyMutation(deregisterJobDefinitionRequest)
                .build());
    }

    /**
     * <p>
     * Describes one or more of your compute environments.
     * </p>
     * <p>
     * If you're using an unmanaged compute environment, you can use the <code>DescribeComputeEnvironment</code>
     * operation to determine the <code>ecsClusterArn</code> that you launch your Amazon ECS container instances into.
     * </p>
     *
     * @param describeComputeEnvironmentsRequest
     *        Contains the parameters for <code>DescribeComputeEnvironments</code>.
     * @return A Java Future containing the result of the DescribeComputeEnvironments operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeComputeEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeComputeEnvironments"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DescribeComputeEnvironmentsResponse> describeComputeEnvironments(
            DescribeComputeEnvironmentsRequest describeComputeEnvironmentsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes one or more of your compute environments.
     * </p>
     * <p>
     * If you're using an unmanaged compute environment, you can use the <code>DescribeComputeEnvironment</code>
     * operation to determine the <code>ecsClusterArn</code> that you launch your Amazon ECS container instances into.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeComputeEnvironmentsRequest.Builder}
     * avoiding the need to create one manually via {@link DescribeComputeEnvironmentsRequest#builder()}
     * </p>
     *
     * @param describeComputeEnvironmentsRequest
     *        A {@link Consumer} that will call methods on {@link DescribeComputeEnvironmentsRequest.Builder} to create
     *        a request. Contains the parameters for <code>DescribeComputeEnvironments</code>.
     * @return A Java Future containing the result of the DescribeComputeEnvironments operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeComputeEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeComputeEnvironments"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DescribeComputeEnvironmentsResponse> describeComputeEnvironments(
            Consumer<DescribeComputeEnvironmentsRequest.Builder> describeComputeEnvironmentsRequest) {
        return describeComputeEnvironments(DescribeComputeEnvironmentsRequest.builder()
                .applyMutation(describeComputeEnvironmentsRequest).build());
    }

    /**
     * <p>
     * Describes one or more of your compute environments.
     * </p>
     * <p>
     * If you're using an unmanaged compute environment, you can use the <code>DescribeComputeEnvironment</code>
     * operation to determine the <code>ecsClusterArn</code> that you launch your Amazon ECS container instances into.
     * </p>
     *
     * @return A Java Future containing the result of the DescribeComputeEnvironments operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeComputeEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeComputeEnvironments"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DescribeComputeEnvironmentsResponse> describeComputeEnvironments() {
        return describeComputeEnvironments(DescribeComputeEnvironmentsRequest.builder().build());
    }

    /**
     * <p>
     * Describes one or more of your compute environments.
     * </p>
     * <p>
     * If you're using an unmanaged compute environment, you can use the <code>DescribeComputeEnvironment</code>
     * operation to determine the <code>ecsClusterArn</code> that you launch your Amazon ECS container instances into.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeComputeEnvironments(software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeComputeEnvironmentsPublisher publisher = client.describeComputeEnvironmentsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeComputeEnvironmentsPublisher publisher = client.describeComputeEnvironmentsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeComputeEnvironments(software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsRequest)}
     * operation.</b>
     * </p>
     *
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeComputeEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeComputeEnvironments"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeComputeEnvironmentsPublisher describeComputeEnvironmentsPaginator() {
        return describeComputeEnvironmentsPaginator(DescribeComputeEnvironmentsRequest.builder().build());
    }

    /**
     * <p>
     * Describes one or more of your compute environments.
     * </p>
     * <p>
     * If you're using an unmanaged compute environment, you can use the <code>DescribeComputeEnvironment</code>
     * operation to determine the <code>ecsClusterArn</code> that you launch your Amazon ECS container instances into.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeComputeEnvironments(software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeComputeEnvironmentsPublisher publisher = client.describeComputeEnvironmentsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeComputeEnvironmentsPublisher publisher = client.describeComputeEnvironmentsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeComputeEnvironments(software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeComputeEnvironmentsRequest
     *        Contains the parameters for <code>DescribeComputeEnvironments</code>.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeComputeEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeComputeEnvironments"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeComputeEnvironmentsPublisher describeComputeEnvironmentsPaginator(
            DescribeComputeEnvironmentsRequest describeComputeEnvironmentsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes one or more of your compute environments.
     * </p>
     * <p>
     * If you're using an unmanaged compute environment, you can use the <code>DescribeComputeEnvironment</code>
     * operation to determine the <code>ecsClusterArn</code> that you launch your Amazon ECS container instances into.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeComputeEnvironments(software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeComputeEnvironmentsPublisher publisher = client.describeComputeEnvironmentsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeComputeEnvironmentsPublisher publisher = client.describeComputeEnvironmentsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeComputeEnvironments(software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsRequest)}
     * operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeComputeEnvironmentsRequest.Builder}
     * avoiding the need to create one manually via {@link DescribeComputeEnvironmentsRequest#builder()}
     * </p>
     *
     * @param describeComputeEnvironmentsRequest
     *        A {@link Consumer} that will call methods on {@link DescribeComputeEnvironmentsRequest.Builder} to create
     *        a request. Contains the parameters for <code>DescribeComputeEnvironments</code>.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeComputeEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeComputeEnvironments"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeComputeEnvironmentsPublisher describeComputeEnvironmentsPaginator(
            Consumer<DescribeComputeEnvironmentsRequest.Builder> describeComputeEnvironmentsRequest) {
        return describeComputeEnvironmentsPaginator(DescribeComputeEnvironmentsRequest.builder()
                .applyMutation(describeComputeEnvironmentsRequest).build());
    }

    /**
     * <p>
     * Describes a list of job definitions. You can specify a <code>status</code> (such as <code>ACTIVE</code>) to only
     * return job definitions that match that status.
     * </p>
     *
     * @param describeJobDefinitionsRequest
     *        Contains the parameters for <code>DescribeJobDefinitions</code>.
     * @return A Java Future containing the result of the DescribeJobDefinitions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeJobDefinitions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobDefinitions" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<DescribeJobDefinitionsResponse> describeJobDefinitions(
            DescribeJobDefinitionsRequest describeJobDefinitionsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes a list of job definitions. You can specify a <code>status</code> (such as <code>ACTIVE</code>) to only
     * return job definitions that match that status.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeJobDefinitionsRequest.Builder} avoiding the
     * need to create one manually via {@link DescribeJobDefinitionsRequest#builder()}
     * </p>
     *
     * @param describeJobDefinitionsRequest
     *        A {@link Consumer} that will call methods on {@link DescribeJobDefinitionsRequest.Builder} to create a
     *        request. Contains the parameters for <code>DescribeJobDefinitions</code>.
     * @return A Java Future containing the result of the DescribeJobDefinitions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeJobDefinitions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobDefinitions" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<DescribeJobDefinitionsResponse> describeJobDefinitions(
            Consumer<DescribeJobDefinitionsRequest.Builder> describeJobDefinitionsRequest) {
        return describeJobDefinitions(DescribeJobDefinitionsRequest.builder().applyMutation(describeJobDefinitionsRequest)
                .build());
    }

    /**
     * <p>
     * Describes a list of job definitions. You can specify a <code>status</code> (such as <code>ACTIVE</code>) to only
     * return job definitions that match that status.
     * </p>
     *
     * @return A Java Future containing the result of the DescribeJobDefinitions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeJobDefinitions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobDefinitions" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<DescribeJobDefinitionsResponse> describeJobDefinitions() {
        return describeJobDefinitions(DescribeJobDefinitionsRequest.builder().build());
    }

    /**
     * <p>
     * Describes a list of job definitions. You can specify a <code>status</code> (such as <code>ACTIVE</code>) to only
     * return job definitions that match that status.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeJobDefinitions(software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeJobDefinitionsPublisher publisher = client.describeJobDefinitionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeJobDefinitionsPublisher publisher = client.describeJobDefinitionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeJobDefinitions(software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsRequest)}
     * operation.</b>
     * </p>
     *
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeJobDefinitions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobDefinitions" target="_top">AWS
     *      API Documentation</a>
     */
    default DescribeJobDefinitionsPublisher describeJobDefinitionsPaginator() {
        return describeJobDefinitionsPaginator(DescribeJobDefinitionsRequest.builder().build());
    }

    /**
     * <p>
     * Describes a list of job definitions. You can specify a <code>status</code> (such as <code>ACTIVE</code>) to only
     * return job definitions that match that status.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeJobDefinitions(software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeJobDefinitionsPublisher publisher = client.describeJobDefinitionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeJobDefinitionsPublisher publisher = client.describeJobDefinitionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeJobDefinitions(software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeJobDefinitionsRequest
     *        Contains the parameters for <code>DescribeJobDefinitions</code>.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeJobDefinitions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobDefinitions" target="_top">AWS
     *      API Documentation</a>
     */
    default DescribeJobDefinitionsPublisher describeJobDefinitionsPaginator(
            DescribeJobDefinitionsRequest describeJobDefinitionsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes a list of job definitions. You can specify a <code>status</code> (such as <code>ACTIVE</code>) to only
     * return job definitions that match that status.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeJobDefinitions(software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeJobDefinitionsPublisher publisher = client.describeJobDefinitionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeJobDefinitionsPublisher publisher = client.describeJobDefinitionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeJobDefinitions(software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsRequest)}
     * operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeJobDefinitionsRequest.Builder} avoiding the
     * need to create one manually via {@link DescribeJobDefinitionsRequest#builder()}
     * </p>
     *
     * @param describeJobDefinitionsRequest
     *        A {@link Consumer} that will call methods on {@link DescribeJobDefinitionsRequest.Builder} to create a
     *        request. Contains the parameters for <code>DescribeJobDefinitions</code>.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeJobDefinitions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobDefinitions" target="_top">AWS
     *      API Documentation</a>
     */
    default DescribeJobDefinitionsPublisher describeJobDefinitionsPaginator(
            Consumer<DescribeJobDefinitionsRequest.Builder> describeJobDefinitionsRequest) {
        return describeJobDefinitionsPaginator(DescribeJobDefinitionsRequest.builder()
                .applyMutation(describeJobDefinitionsRequest).build());
    }

    /**
     * <p>
     * Describes one or more of your job queues.
     * </p>
     *
     * @param describeJobQueuesRequest
     *        Contains the parameters for <code>DescribeJobQueues</code>.
     * @return A Java Future containing the result of the DescribeJobQueues operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeJobQueues
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobQueues" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DescribeJobQueuesResponse> describeJobQueues(DescribeJobQueuesRequest describeJobQueuesRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes one or more of your job queues.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeJobQueuesRequest.Builder} avoiding the need
     * to create one manually via {@link DescribeJobQueuesRequest#builder()}
     * </p>
     *
     * @param describeJobQueuesRequest
     *        A {@link Consumer} that will call methods on {@link DescribeJobQueuesRequest.Builder} to create a request.
     *        Contains the parameters for <code>DescribeJobQueues</code>.
     * @return A Java Future containing the result of the DescribeJobQueues operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeJobQueues
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobQueues" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DescribeJobQueuesResponse> describeJobQueues(
            Consumer<DescribeJobQueuesRequest.Builder> describeJobQueuesRequest) {
        return describeJobQueues(DescribeJobQueuesRequest.builder().applyMutation(describeJobQueuesRequest).build());
    }

    /**
     * <p>
     * Describes one or more of your job queues.
     * </p>
     *
     * @return A Java Future containing the result of the DescribeJobQueues operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeJobQueues
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobQueues" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DescribeJobQueuesResponse> describeJobQueues() {
        return describeJobQueues(DescribeJobQueuesRequest.builder().build());
    }

    /**
     * <p>
     * Describes one or more of your job queues.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeJobQueues(software.amazon.awssdk.services.batch.model.DescribeJobQueuesRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeJobQueuesPublisher publisher = client.describeJobQueuesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeJobQueuesPublisher publisher = client.describeJobQueuesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.batch.model.DescribeJobQueuesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.batch.model.DescribeJobQueuesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeJobQueues(software.amazon.awssdk.services.batch.model.DescribeJobQueuesRequest)} operation.</b>
     * </p>
     *
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeJobQueues
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobQueues" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeJobQueuesPublisher describeJobQueuesPaginator() {
        return describeJobQueuesPaginator(DescribeJobQueuesRequest.builder().build());
    }

    /**
     * <p>
     * Describes one or more of your job queues.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeJobQueues(software.amazon.awssdk.services.batch.model.DescribeJobQueuesRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeJobQueuesPublisher publisher = client.describeJobQueuesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeJobQueuesPublisher publisher = client.describeJobQueuesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.batch.model.DescribeJobQueuesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.batch.model.DescribeJobQueuesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeJobQueues(software.amazon.awssdk.services.batch.model.DescribeJobQueuesRequest)} operation.</b>
     * </p>
     *
     * @param describeJobQueuesRequest
     *        Contains the parameters for <code>DescribeJobQueues</code>.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeJobQueues
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobQueues" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeJobQueuesPublisher describeJobQueuesPaginator(DescribeJobQueuesRequest describeJobQueuesRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes one or more of your job queues.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeJobQueues(software.amazon.awssdk.services.batch.model.DescribeJobQueuesRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeJobQueuesPublisher publisher = client.describeJobQueuesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeJobQueuesPublisher publisher = client.describeJobQueuesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.batch.model.DescribeJobQueuesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.batch.model.DescribeJobQueuesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeJobQueues(software.amazon.awssdk.services.batch.model.DescribeJobQueuesRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeJobQueuesRequest.Builder} avoiding the need
     * to create one manually via {@link DescribeJobQueuesRequest#builder()}
     * </p>
     *
     * @param describeJobQueuesRequest
     *        A {@link Consumer} that will call methods on {@link DescribeJobQueuesRequest.Builder} to create a request.
     *        Contains the parameters for <code>DescribeJobQueues</code>.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeJobQueues
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobQueues" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeJobQueuesPublisher describeJobQueuesPaginator(
            Consumer<DescribeJobQueuesRequest.Builder> describeJobQueuesRequest) {
        return describeJobQueuesPaginator(DescribeJobQueuesRequest.builder().applyMutation(describeJobQueuesRequest).build());
    }

    /**
     * <p>
     * Describes a list of Batch jobs.
     * </p>
     *
     * @param describeJobsRequest
     *        Contains the parameters for <code>DescribeJobs</code>.
     * @return A Java Future containing the result of the DescribeJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobs" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DescribeJobsResponse> describeJobs(DescribeJobsRequest describeJobsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes a list of Batch jobs.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeJobsRequest.Builder} avoiding the need to
     * create one manually via {@link DescribeJobsRequest#builder()}
     * </p>
     *
     * @param describeJobsRequest
     *        A {@link Consumer} that will call methods on {@link DescribeJobsRequest.Builder} to create a request.
     *        Contains the parameters for <code>DescribeJobs</code>.
     * @return A Java Future containing the result of the DescribeJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobs" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DescribeJobsResponse> describeJobs(Consumer<DescribeJobsRequest.Builder> describeJobsRequest) {
        return describeJobs(DescribeJobsRequest.builder().applyMutation(describeJobsRequest).build());
    }

    /**
     * <p>
     * Describes one or more of your scheduling policies.
     * </p>
     *
     * @param describeSchedulingPoliciesRequest
     *        Contains the parameters for <code>DescribeSchedulingPolicies</code>.
     * @return A Java Future containing the result of the DescribeSchedulingPolicies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeSchedulingPolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeSchedulingPolicies"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DescribeSchedulingPoliciesResponse> describeSchedulingPolicies(
            DescribeSchedulingPoliciesRequest describeSchedulingPoliciesRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes one or more of your scheduling policies.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeSchedulingPoliciesRequest.Builder} avoiding
     * the need to create one manually via {@link DescribeSchedulingPoliciesRequest#builder()}
     * </p>
     *
     * @param describeSchedulingPoliciesRequest
     *        A {@link Consumer} that will call methods on {@link DescribeSchedulingPoliciesRequest.Builder} to create a
     *        request. Contains the parameters for <code>DescribeSchedulingPolicies</code>.
     * @return A Java Future containing the result of the DescribeSchedulingPolicies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeSchedulingPolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeSchedulingPolicies"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DescribeSchedulingPoliciesResponse> describeSchedulingPolicies(
            Consumer<DescribeSchedulingPoliciesRequest.Builder> describeSchedulingPoliciesRequest) {
        return describeSchedulingPolicies(DescribeSchedulingPoliciesRequest.builder()
                .applyMutation(describeSchedulingPoliciesRequest).build());
    }

    /**
     * <p>
     * Returns a list of Batch jobs.
     * </p>
     * <p>
     * You must specify only one of the following items:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A job queue ID to return a list of jobs in that job queue
     * </p>
     * </li>
     * <li>
     * <p>
     * A multi-node parallel job ID to return a list of nodes for that job
     * </p>
     * </li>
     * <li>
     * <p>
     * An array job ID to return a list of the children for that job
     * </p>
     * </li>
     * </ul>
     * <p>
     * You can filter the results by job status with the <code>jobStatus</code> parameter. If you don't specify a
     * status, only <code>RUNNING</code> jobs are returned.
     * </p>
     *
     * @param listJobsRequest
     *        Contains the parameters for <code>ListJobs</code>.
     * @return A Java Future containing the result of the ListJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.ListJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListJobsResponse> listJobs(ListJobsRequest listJobsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of Batch jobs.
     * </p>
     * <p>
     * You must specify only one of the following items:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A job queue ID to return a list of jobs in that job queue
     * </p>
     * </li>
     * <li>
     * <p>
     * A multi-node parallel job ID to return a list of nodes for that job
     * </p>
     * </li>
     * <li>
     * <p>
     * An array job ID to return a list of the children for that job
     * </p>
     * </li>
     * </ul>
     * <p>
     * You can filter the results by job status with the <code>jobStatus</code> parameter. If you don't specify a
     * status, only <code>RUNNING</code> jobs are returned.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListJobsRequest.Builder} avoiding the need to
     * create one manually via {@link ListJobsRequest#builder()}
     * </p>
     *
     * @param listJobsRequest
     *        A {@link Consumer} that will call methods on {@link ListJobsRequest.Builder} to create a request. Contains
     *        the parameters for <code>ListJobs</code>.
     * @return A Java Future containing the result of the ListJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.ListJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListJobsResponse> listJobs(Consumer<ListJobsRequest.Builder> listJobsRequest) {
        return listJobs(ListJobsRequest.builder().applyMutation(listJobsRequest).build());
    }

    /**
     * <p>
     * Returns a list of Batch jobs.
     * </p>
     * <p>
     * You must specify only one of the following items:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A job queue ID to return a list of jobs in that job queue
     * </p>
     * </li>
     * <li>
     * <p>
     * A multi-node parallel job ID to return a list of nodes for that job
     * </p>
     * </li>
     * <li>
     * <p>
     * An array job ID to return a list of the children for that job
     * </p>
     * </li>
     * </ul>
     * <p>
     * You can filter the results by job status with the <code>jobStatus</code> parameter. If you don't specify a
     * status, only <code>RUNNING</code> jobs are returned.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listJobs(software.amazon.awssdk.services.batch.model.ListJobsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.ListJobsPublisher publisher = client.listJobsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.ListJobsPublisher publisher = client.listJobsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.batch.model.ListJobsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.batch.model.ListJobsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listJobs(software.amazon.awssdk.services.batch.model.ListJobsRequest)} operation.</b>
     * </p>
     *
     * @param listJobsRequest
     *        Contains the parameters for <code>ListJobs</code>.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.ListJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    default ListJobsPublisher listJobsPaginator(ListJobsRequest listJobsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of Batch jobs.
     * </p>
     * <p>
     * You must specify only one of the following items:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A job queue ID to return a list of jobs in that job queue
     * </p>
     * </li>
     * <li>
     * <p>
     * A multi-node parallel job ID to return a list of nodes for that job
     * </p>
     * </li>
     * <li>
     * <p>
     * An array job ID to return a list of the children for that job
     * </p>
     * </li>
     * </ul>
     * <p>
     * You can filter the results by job status with the <code>jobStatus</code> parameter. If you don't specify a
     * status, only <code>RUNNING</code> jobs are returned.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listJobs(software.amazon.awssdk.services.batch.model.ListJobsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.ListJobsPublisher publisher = client.listJobsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.ListJobsPublisher publisher = client.listJobsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.batch.model.ListJobsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.batch.model.ListJobsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listJobs(software.amazon.awssdk.services.batch.model.ListJobsRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListJobsRequest.Builder} avoiding the need to
     * create one manually via {@link ListJobsRequest#builder()}
     * </p>
     *
     * @param listJobsRequest
     *        A {@link Consumer} that will call methods on {@link ListJobsRequest.Builder} to create a request. Contains
     *        the parameters for <code>ListJobs</code>.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.ListJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    default ListJobsPublisher listJobsPaginator(Consumer<ListJobsRequest.Builder> listJobsRequest) {
        return listJobsPaginator(ListJobsRequest.builder().applyMutation(listJobsRequest).build());
    }

    /**
     * <p>
     * Returns a list of Batch scheduling policies.
     * </p>
     *
     * @param listSchedulingPoliciesRequest
     *        Contains the parameters for <code>ListSchedulingPolicies</code>.
     * @return A Java Future containing the result of the ListSchedulingPolicies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.ListSchedulingPolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/ListSchedulingPolicies" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<ListSchedulingPoliciesResponse> listSchedulingPolicies(
            ListSchedulingPoliciesRequest listSchedulingPoliciesRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of Batch scheduling policies.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListSchedulingPoliciesRequest.Builder} avoiding the
     * need to create one manually via {@link ListSchedulingPoliciesRequest#builder()}
     * </p>
     *
     * @param listSchedulingPoliciesRequest
     *        A {@link Consumer} that will call methods on {@link ListSchedulingPoliciesRequest.Builder} to create a
     *        request. Contains the parameters for <code>ListSchedulingPolicies</code>.
     * @return A Java Future containing the result of the ListSchedulingPolicies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.ListSchedulingPolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/ListSchedulingPolicies" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<ListSchedulingPoliciesResponse> listSchedulingPolicies(
            Consumer<ListSchedulingPoliciesRequest.Builder> listSchedulingPoliciesRequest) {
        return listSchedulingPolicies(ListSchedulingPoliciesRequest.builder().applyMutation(listSchedulingPoliciesRequest)
                .build());
    }

    /**
     * <p>
     * Returns a list of Batch scheduling policies.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSchedulingPolicies(software.amazon.awssdk.services.batch.model.ListSchedulingPoliciesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.ListSchedulingPoliciesPublisher publisher = client.listSchedulingPoliciesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.ListSchedulingPoliciesPublisher publisher = client.listSchedulingPoliciesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.batch.model.ListSchedulingPoliciesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.batch.model.ListSchedulingPoliciesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSchedulingPolicies(software.amazon.awssdk.services.batch.model.ListSchedulingPoliciesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSchedulingPoliciesRequest
     *        Contains the parameters for <code>ListSchedulingPolicies</code>.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.ListSchedulingPolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/ListSchedulingPolicies" target="_top">AWS
     *      API Documentation</a>
     */
    default ListSchedulingPoliciesPublisher listSchedulingPoliciesPaginator(
            ListSchedulingPoliciesRequest listSchedulingPoliciesRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of Batch scheduling policies.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSchedulingPolicies(software.amazon.awssdk.services.batch.model.ListSchedulingPoliciesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.ListSchedulingPoliciesPublisher publisher = client.listSchedulingPoliciesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.ListSchedulingPoliciesPublisher publisher = client.listSchedulingPoliciesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.batch.model.ListSchedulingPoliciesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.batch.model.ListSchedulingPoliciesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSchedulingPolicies(software.amazon.awssdk.services.batch.model.ListSchedulingPoliciesRequest)}
     * operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListSchedulingPoliciesRequest.Builder} avoiding the
     * need to create one manually via {@link ListSchedulingPoliciesRequest#builder()}
     * </p>
     *
     * @param listSchedulingPoliciesRequest
     *        A {@link Consumer} that will call methods on {@link ListSchedulingPoliciesRequest.Builder} to create a
     *        request. Contains the parameters for <code>ListSchedulingPolicies</code>.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.ListSchedulingPolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/ListSchedulingPolicies" target="_top">AWS
     *      API Documentation</a>
     */
    default ListSchedulingPoliciesPublisher listSchedulingPoliciesPaginator(
            Consumer<ListSchedulingPoliciesRequest.Builder> listSchedulingPoliciesRequest) {
        return listSchedulingPoliciesPaginator(ListSchedulingPoliciesRequest.builder()
                .applyMutation(listSchedulingPoliciesRequest).build());
    }

    /**
     * <p>
     * Lists the tags for an Batch resource. Batch resources that support tags are compute environments, jobs, job
     * definitions, job queues, and scheduling policies. ARNs for child jobs of array and multi-node parallel (MNP) jobs
     * are not supported.
     * </p>
     *
     * @param listTagsForResourceRequest
     *        Contains the parameters for <code>ListTagsForResource</code>.
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the tags for an Batch resource. Batch resources that support tags are compute environments, jobs, job
     * definitions, job queues, and scheduling policies. ARNs for child jobs of array and multi-node parallel (MNP) jobs
     * are not supported.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on {@link ListTagsForResourceRequest.Builder} to create a
     *        request. Contains the parameters for <code>ListTagsForResource</code>.
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * Registers an Batch job definition.
     * </p>
     *
     * @param registerJobDefinitionRequest
     *        Contains the parameters for <code>RegisterJobDefinition</code>.
     * @return A Java Future containing the result of the RegisterJobDefinition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.RegisterJobDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/RegisterJobDefinition" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<RegisterJobDefinitionResponse> registerJobDefinition(
            RegisterJobDefinitionRequest registerJobDefinitionRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Registers an Batch job definition.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link RegisterJobDefinitionRequest.Builder} avoiding the
     * need to create one manually via {@link RegisterJobDefinitionRequest#builder()}
     * </p>
     *
     * @param registerJobDefinitionRequest
     *        A {@link Consumer} that will call methods on {@link RegisterJobDefinitionRequest.Builder} to create a
     *        request. Contains the parameters for <code>RegisterJobDefinition</code>.
     * @return A Java Future containing the result of the RegisterJobDefinition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.RegisterJobDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/RegisterJobDefinition" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<RegisterJobDefinitionResponse> registerJobDefinition(
            Consumer<RegisterJobDefinitionRequest.Builder> registerJobDefinitionRequest) {
        return registerJobDefinition(RegisterJobDefinitionRequest.builder().applyMutation(registerJobDefinitionRequest).build());
    }

    /**
     * <p>
     * Submits an Batch job from a job definition. Parameters that are specified during <a>SubmitJob</a> override
     * parameters defined in the job definition. vCPU and memory requirements that are specified in the
     * <code>resourceRequirements</code> objects in the job definition are the exception. They can't be overridden this
     * way using the <code>memory</code> and <code>vcpus</code> parameters. Rather, you must specify updates to job
     * definition parameters in a <code>resourceRequirements</code> object that's included in the
     * <code>containerOverrides</code> parameter.
     * </p>
     * <note>
     * <p>
     * Job queues with a scheduling policy are limited to 500 active fair share identifiers at a time.
     * </p>
     * </note> <important>
     * <p>
     * Jobs that run on Fargate resources can't be guaranteed to run for more than 14 days. This is because, after 14
     * days, Fargate resources might become unavailable and job might be terminated.
     * </p>
     * </important>
     *
     * @param submitJobRequest
     *        Contains the parameters for <code>SubmitJob</code>.
     * @return A Java Future containing the result of the SubmitJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.SubmitJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/SubmitJob" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<SubmitJobResponse> submitJob(SubmitJobRequest submitJobRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Submits an Batch job from a job definition. Parameters that are specified during <a>SubmitJob</a> override
     * parameters defined in the job definition. vCPU and memory requirements that are specified in the
     * <code>resourceRequirements</code> objects in the job definition are the exception. They can't be overridden this
     * way using the <code>memory</code> and <code>vcpus</code> parameters. Rather, you must specify updates to job
     * definition parameters in a <code>resourceRequirements</code> object that's included in the
     * <code>containerOverrides</code> parameter.
     * </p>
     * <note>
     * <p>
     * Job queues with a scheduling policy are limited to 500 active fair share identifiers at a time.
     * </p>
     * </note> <important>
     * <p>
     * Jobs that run on Fargate resources can't be guaranteed to run for more than 14 days. This is because, after 14
     * days, Fargate resources might become unavailable and job might be terminated.
     * </p>
     * </important><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link SubmitJobRequest.Builder} avoiding the need to
     * create one manually via {@link SubmitJobRequest#builder()}
     * </p>
     *
     * @param submitJobRequest
     *        A {@link Consumer} that will call methods on {@link SubmitJobRequest.Builder} to create a request.
     *        Contains the parameters for <code>SubmitJob</code>.
     * @return A Java Future containing the result of the SubmitJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.SubmitJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/SubmitJob" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<SubmitJobResponse> submitJob(Consumer<SubmitJobRequest.Builder> submitJobRequest) {
        return submitJob(SubmitJobRequest.builder().applyMutation(submitJobRequest).build());
    }

    /**
     * <p>
     * Associates the specified tags to a resource with the specified <code>resourceArn</code>. If existing tags on a
     * resource aren't specified in the request parameters, they aren't changed. When a resource is deleted, the tags
     * that are associated with that resource are deleted as well. Batch resources that support tags are compute
     * environments, jobs, job definitions, job queues, and scheduling policies. ARNs for child jobs of array and
     * multi-node parallel (MNP) jobs are not supported.
     * </p>
     *
     * @param tagResourceRequest
     *        Contains the parameters for <code>TagResource</code>.
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Associates the specified tags to a resource with the specified <code>resourceArn</code>. If existing tags on a
     * resource aren't specified in the request parameters, they aren't changed. When a resource is deleted, the tags
     * that are associated with that resource are deleted as well. Batch resources that support tags are compute
     * environments, jobs, job definitions, job queues, and scheduling policies. ARNs for child jobs of array and
     * multi-node parallel (MNP) jobs are not supported.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on {@link TagResourceRequest.Builder} to create a request.
     *        Contains the parameters for <code>TagResource</code>.
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<TagResourceResponse> tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest) {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * Terminates a job in a job queue. Jobs that are in the <code>STARTING</code> or <code>RUNNING</code> state are
     * terminated, which causes them to transition to <code>FAILED</code>. Jobs that have not progressed to the
     * <code>STARTING</code> state are cancelled.
     * </p>
     *
     * @param terminateJobRequest
     *        Contains the parameters for <code>TerminateJob</code>.
     * @return A Java Future containing the result of the TerminateJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.TerminateJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/TerminateJob" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<TerminateJobResponse> terminateJob(TerminateJobRequest terminateJobRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Terminates a job in a job queue. Jobs that are in the <code>STARTING</code> or <code>RUNNING</code> state are
     * terminated, which causes them to transition to <code>FAILED</code>. Jobs that have not progressed to the
     * <code>STARTING</code> state are cancelled.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TerminateJobRequest.Builder} avoiding the need to
     * create one manually via {@link TerminateJobRequest#builder()}
     * </p>
     *
     * @param terminateJobRequest
     *        A {@link Consumer} that will call methods on {@link TerminateJobRequest.Builder} to create a request.
     *        Contains the parameters for <code>TerminateJob</code>.
     * @return A Java Future containing the result of the TerminateJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.TerminateJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/TerminateJob" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<TerminateJobResponse> terminateJob(Consumer<TerminateJobRequest.Builder> terminateJobRequest) {
        return terminateJob(TerminateJobRequest.builder().applyMutation(terminateJobRequest).build());
    }

    /**
     * <p>
     * Deletes specified tags from an Batch resource.
     * </p>
     *
     * @param untagResourceRequest
     *        Contains the parameters for <code>UntagResource</code>.
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes specified tags from an Batch resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on {@link UntagResourceRequest.Builder} to create a request.
     *        Contains the parameters for <code>UntagResource</code>.
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UntagResourceResponse> untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest) {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    /**
     * <p>
     * Updates an Batch compute environment.
     * </p>
     *
     * @param updateComputeEnvironmentRequest
     *        Contains the parameters for <code>UpdateComputeEnvironment</code>.
     * @return A Java Future containing the result of the UpdateComputeEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.UpdateComputeEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/UpdateComputeEnvironment"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<UpdateComputeEnvironmentResponse> updateComputeEnvironment(
            UpdateComputeEnvironmentRequest updateComputeEnvironmentRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates an Batch compute environment.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateComputeEnvironmentRequest.Builder} avoiding
     * the need to create one manually via {@link UpdateComputeEnvironmentRequest#builder()}
     * </p>
     *
     * @param updateComputeEnvironmentRequest
     *        A {@link Consumer} that will call methods on {@link UpdateComputeEnvironmentRequest.Builder} to create a
     *        request. Contains the parameters for <code>UpdateComputeEnvironment</code>.
     * @return A Java Future containing the result of the UpdateComputeEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.UpdateComputeEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/UpdateComputeEnvironment"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<UpdateComputeEnvironmentResponse> updateComputeEnvironment(
            Consumer<UpdateComputeEnvironmentRequest.Builder> updateComputeEnvironmentRequest) {
        return updateComputeEnvironment(UpdateComputeEnvironmentRequest.builder().applyMutation(updateComputeEnvironmentRequest)
                .build());
    }

    /**
     * <p>
     * Updates a job queue.
     * </p>
     *
     * @param updateJobQueueRequest
     *        Contains the parameters for <code>UpdateJobQueue</code>.
     * @return A Java Future containing the result of the UpdateJobQueue operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.UpdateJobQueue
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/UpdateJobQueue" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UpdateJobQueueResponse> updateJobQueue(UpdateJobQueueRequest updateJobQueueRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates a job queue.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateJobQueueRequest.Builder} avoiding the need to
     * create one manually via {@link UpdateJobQueueRequest#builder()}
     * </p>
     *
     * @param updateJobQueueRequest
     *        A {@link Consumer} that will call methods on {@link UpdateJobQueueRequest.Builder} to create a request.
     *        Contains the parameters for <code>UpdateJobQueue</code>.
     * @return A Java Future containing the result of the UpdateJobQueue operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.UpdateJobQueue
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/UpdateJobQueue" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UpdateJobQueueResponse> updateJobQueue(Consumer<UpdateJobQueueRequest.Builder> updateJobQueueRequest) {
        return updateJobQueue(UpdateJobQueueRequest.builder().applyMutation(updateJobQueueRequest).build());
    }

    /**
     * <p>
     * Updates a scheduling policy.
     * </p>
     *
     * @param updateSchedulingPolicyRequest
     *        Contains the parameters for <code>UpdateSchedulingPolicy</code>.
     * @return A Java Future containing the result of the UpdateSchedulingPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.UpdateSchedulingPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/UpdateSchedulingPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<UpdateSchedulingPolicyResponse> updateSchedulingPolicy(
            UpdateSchedulingPolicyRequest updateSchedulingPolicyRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates a scheduling policy.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateSchedulingPolicyRequest.Builder} avoiding the
     * need to create one manually via {@link UpdateSchedulingPolicyRequest#builder()}
     * </p>
     *
     * @param updateSchedulingPolicyRequest
     *        A {@link Consumer} that will call methods on {@link UpdateSchedulingPolicyRequest.Builder} to create a
     *        request. Contains the parameters for <code>UpdateSchedulingPolicy</code>.
     * @return A Java Future containing the result of the UpdateSchedulingPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action, such as using an action or
     *         resource on behalf of a user that doesn't have permissions to use the action or resource, or specifying
     *         an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.UpdateSchedulingPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/UpdateSchedulingPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<UpdateSchedulingPolicyResponse> updateSchedulingPolicy(
            Consumer<UpdateSchedulingPolicyRequest.Builder> updateSchedulingPolicyRequest) {
        return updateSchedulingPolicy(UpdateSchedulingPolicyRequest.builder().applyMutation(updateSchedulingPolicyRequest)
                .build());
    }
}
