/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.batch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object representing the details of a container that's part of a job.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ContainerDetail implements SdkPojo, Serializable, ToCopyableBuilder<ContainerDetail.Builder, ContainerDetail> {
    private static final SdkField<String> IMAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("image")
            .getter(getter(ContainerDetail::image)).setter(setter(Builder::image))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("image").build()).build();

    private static final SdkField<Integer> VCPUS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("vcpus")
            .getter(getter(ContainerDetail::vcpus)).setter(setter(Builder::vcpus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("vcpus").build()).build();

    private static final SdkField<Integer> MEMORY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("memory").getter(getter(ContainerDetail::memory)).setter(setter(Builder::memory))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("memory").build()).build();

    private static final SdkField<List<String>> COMMAND_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("command")
            .getter(getter(ContainerDetail::command))
            .setter(setter(Builder::command))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("command").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> JOB_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("jobRoleArn").getter(getter(ContainerDetail::jobRoleArn)).setter(setter(Builder::jobRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("jobRoleArn").build()).build();

    private static final SdkField<String> EXECUTION_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("executionRoleArn").getter(getter(ContainerDetail::executionRoleArn))
            .setter(setter(Builder::executionRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("executionRoleArn").build()).build();

    private static final SdkField<List<Volume>> VOLUMES_FIELD = SdkField
            .<List<Volume>> builder(MarshallingType.LIST)
            .memberName("volumes")
            .getter(getter(ContainerDetail::volumes))
            .setter(setter(Builder::volumes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("volumes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Volume> builder(MarshallingType.SDK_POJO)
                                            .constructor(Volume::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<KeyValuePair>> ENVIRONMENT_FIELD = SdkField
            .<List<KeyValuePair>> builder(MarshallingType.LIST)
            .memberName("environment")
            .getter(getter(ContainerDetail::environment))
            .setter(setter(Builder::environment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("environment").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<KeyValuePair> builder(MarshallingType.SDK_POJO)
                                            .constructor(KeyValuePair::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<MountPoint>> MOUNT_POINTS_FIELD = SdkField
            .<List<MountPoint>> builder(MarshallingType.LIST)
            .memberName("mountPoints")
            .getter(getter(ContainerDetail::mountPoints))
            .setter(setter(Builder::mountPoints))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("mountPoints").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<MountPoint> builder(MarshallingType.SDK_POJO)
                                            .constructor(MountPoint::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> READONLY_ROOT_FILESYSTEM_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("readonlyRootFilesystem").getter(getter(ContainerDetail::readonlyRootFilesystem))
            .setter(setter(Builder::readonlyRootFilesystem))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("readonlyRootFilesystem").build())
            .build();

    private static final SdkField<List<Ulimit>> ULIMITS_FIELD = SdkField
            .<List<Ulimit>> builder(MarshallingType.LIST)
            .memberName("ulimits")
            .getter(getter(ContainerDetail::ulimits))
            .setter(setter(Builder::ulimits))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ulimits").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Ulimit> builder(MarshallingType.SDK_POJO)
                                            .constructor(Ulimit::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> PRIVILEGED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("privileged").getter(getter(ContainerDetail::privileged)).setter(setter(Builder::privileged))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("privileged").build()).build();

    private static final SdkField<String> USER_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("user")
            .getter(getter(ContainerDetail::user)).setter(setter(Builder::user))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("user").build()).build();

    private static final SdkField<Integer> EXIT_CODE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("exitCode").getter(getter(ContainerDetail::exitCode)).setter(setter(Builder::exitCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("exitCode").build()).build();

    private static final SdkField<String> REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("reason")
            .getter(getter(ContainerDetail::reason)).setter(setter(Builder::reason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("reason").build()).build();

    private static final SdkField<String> CONTAINER_INSTANCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("containerInstanceArn").getter(getter(ContainerDetail::containerInstanceArn))
            .setter(setter(Builder::containerInstanceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("containerInstanceArn").build())
            .build();

    private static final SdkField<String> TASK_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("taskArn").getter(getter(ContainerDetail::taskArn)).setter(setter(Builder::taskArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("taskArn").build()).build();

    private static final SdkField<String> LOG_STREAM_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("logStreamName").getter(getter(ContainerDetail::logStreamName)).setter(setter(Builder::logStreamName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("logStreamName").build()).build();

    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("instanceType").getter(getter(ContainerDetail::instanceType)).setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instanceType").build()).build();

    private static final SdkField<List<NetworkInterface>> NETWORK_INTERFACES_FIELD = SdkField
            .<List<NetworkInterface>> builder(MarshallingType.LIST)
            .memberName("networkInterfaces")
            .getter(getter(ContainerDetail::networkInterfaces))
            .setter(setter(Builder::networkInterfaces))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("networkInterfaces").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<NetworkInterface> builder(MarshallingType.SDK_POJO)
                                            .constructor(NetworkInterface::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<ResourceRequirement>> RESOURCE_REQUIREMENTS_FIELD = SdkField
            .<List<ResourceRequirement>> builder(MarshallingType.LIST)
            .memberName("resourceRequirements")
            .getter(getter(ContainerDetail::resourceRequirements))
            .setter(setter(Builder::resourceRequirements))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceRequirements").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ResourceRequirement> builder(MarshallingType.SDK_POJO)
                                            .constructor(ResourceRequirement::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<LinuxParameters> LINUX_PARAMETERS_FIELD = SdkField
            .<LinuxParameters> builder(MarshallingType.SDK_POJO).memberName("linuxParameters")
            .getter(getter(ContainerDetail::linuxParameters)).setter(setter(Builder::linuxParameters))
            .constructor(LinuxParameters::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("linuxParameters").build()).build();

    private static final SdkField<LogConfiguration> LOG_CONFIGURATION_FIELD = SdkField
            .<LogConfiguration> builder(MarshallingType.SDK_POJO).memberName("logConfiguration")
            .getter(getter(ContainerDetail::logConfiguration)).setter(setter(Builder::logConfiguration))
            .constructor(LogConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("logConfiguration").build()).build();

    private static final SdkField<List<Secret>> SECRETS_FIELD = SdkField
            .<List<Secret>> builder(MarshallingType.LIST)
            .memberName("secrets")
            .getter(getter(ContainerDetail::secrets))
            .setter(setter(Builder::secrets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("secrets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Secret> builder(MarshallingType.SDK_POJO)
                                            .constructor(Secret::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<NetworkConfiguration> NETWORK_CONFIGURATION_FIELD = SdkField
            .<NetworkConfiguration> builder(MarshallingType.SDK_POJO).memberName("networkConfiguration")
            .getter(getter(ContainerDetail::networkConfiguration)).setter(setter(Builder::networkConfiguration))
            .constructor(NetworkConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("networkConfiguration").build())
            .build();

    private static final SdkField<FargatePlatformConfiguration> FARGATE_PLATFORM_CONFIGURATION_FIELD = SdkField
            .<FargatePlatformConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("fargatePlatformConfiguration")
            .getter(getter(ContainerDetail::fargatePlatformConfiguration))
            .setter(setter(Builder::fargatePlatformConfiguration))
            .constructor(FargatePlatformConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("fargatePlatformConfiguration")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(IMAGE_FIELD, VCPUS_FIELD,
            MEMORY_FIELD, COMMAND_FIELD, JOB_ROLE_ARN_FIELD, EXECUTION_ROLE_ARN_FIELD, VOLUMES_FIELD, ENVIRONMENT_FIELD,
            MOUNT_POINTS_FIELD, READONLY_ROOT_FILESYSTEM_FIELD, ULIMITS_FIELD, PRIVILEGED_FIELD, USER_FIELD, EXIT_CODE_FIELD,
            REASON_FIELD, CONTAINER_INSTANCE_ARN_FIELD, TASK_ARN_FIELD, LOG_STREAM_NAME_FIELD, INSTANCE_TYPE_FIELD,
            NETWORK_INTERFACES_FIELD, RESOURCE_REQUIREMENTS_FIELD, LINUX_PARAMETERS_FIELD, LOG_CONFIGURATION_FIELD,
            SECRETS_FIELD, NETWORK_CONFIGURATION_FIELD, FARGATE_PLATFORM_CONFIGURATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String image;

    private final Integer vcpus;

    private final Integer memory;

    private final List<String> command;

    private final String jobRoleArn;

    private final String executionRoleArn;

    private final List<Volume> volumes;

    private final List<KeyValuePair> environment;

    private final List<MountPoint> mountPoints;

    private final Boolean readonlyRootFilesystem;

    private final List<Ulimit> ulimits;

    private final Boolean privileged;

    private final String user;

    private final Integer exitCode;

    private final String reason;

    private final String containerInstanceArn;

    private final String taskArn;

    private final String logStreamName;

    private final String instanceType;

    private final List<NetworkInterface> networkInterfaces;

    private final List<ResourceRequirement> resourceRequirements;

    private final LinuxParameters linuxParameters;

    private final LogConfiguration logConfiguration;

    private final List<Secret> secrets;

    private final NetworkConfiguration networkConfiguration;

    private final FargatePlatformConfiguration fargatePlatformConfiguration;

    private ContainerDetail(BuilderImpl builder) {
        this.image = builder.image;
        this.vcpus = builder.vcpus;
        this.memory = builder.memory;
        this.command = builder.command;
        this.jobRoleArn = builder.jobRoleArn;
        this.executionRoleArn = builder.executionRoleArn;
        this.volumes = builder.volumes;
        this.environment = builder.environment;
        this.mountPoints = builder.mountPoints;
        this.readonlyRootFilesystem = builder.readonlyRootFilesystem;
        this.ulimits = builder.ulimits;
        this.privileged = builder.privileged;
        this.user = builder.user;
        this.exitCode = builder.exitCode;
        this.reason = builder.reason;
        this.containerInstanceArn = builder.containerInstanceArn;
        this.taskArn = builder.taskArn;
        this.logStreamName = builder.logStreamName;
        this.instanceType = builder.instanceType;
        this.networkInterfaces = builder.networkInterfaces;
        this.resourceRequirements = builder.resourceRequirements;
        this.linuxParameters = builder.linuxParameters;
        this.logConfiguration = builder.logConfiguration;
        this.secrets = builder.secrets;
        this.networkConfiguration = builder.networkConfiguration;
        this.fargatePlatformConfiguration = builder.fargatePlatformConfiguration;
    }

    /**
     * <p>
     * The image used to start the container.
     * </p>
     * 
     * @return The image used to start the container.
     */
    public final String image() {
        return image;
    }

    /**
     * <p>
     * The number of vCPUs reserved for the container. For jobs that run on EC2 resources, you can specify the vCPU
     * requirement for the job using <code>resourceRequirements</code>, but you can't specify the vCPU requirements in
     * both the <code>vcpus</code> and <code>resourceRequirements</code> object. This parameter maps to
     * <code>CpuShares</code> in the <a href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a
     * container</a> section of the <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the
     * <code>--cpu-shares</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>. Each
     * vCPU is equivalent to 1,024 CPU shares. You must specify at least one vCPU. This is required but can be specified
     * in several places. It must be specified for each node at least once.
     * </p>
     * <note>
     * <p>
     * This parameter isn't applicable to jobs that run on Fargate resources. For jobs that run on Fargate resources,
     * you must specify the vCPU requirement for the job using <code>resourceRequirements</code>.
     * </p>
     * </note>
     * 
     * @return The number of vCPUs reserved for the container. For jobs that run on EC2 resources, you can specify the
     *         vCPU requirement for the job using <code>resourceRequirements</code>, but you can't specify the vCPU
     *         requirements in both the <code>vcpus</code> and <code>resourceRequirements</code> object. This parameter
     *         maps to <code>CpuShares</code> in the <a
     *         href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the
     *         <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the
     *         <code>--cpu-shares</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
     *         run</a>. Each vCPU is equivalent to 1,024 CPU shares. You must specify at least one vCPU. This is
     *         required but can be specified in several places. It must be specified for each node at least once.</p>
     *         <note>
     *         <p>
     *         This parameter isn't applicable to jobs that run on Fargate resources. For jobs that run on Fargate
     *         resources, you must specify the vCPU requirement for the job using <code>resourceRequirements</code>.
     *         </p>
     */
    public final Integer vcpus() {
        return vcpus;
    }

    /**
     * <p>
     * For jobs running on EC2 resources that didn't specify memory requirements using <code>resourceRequirements</code>
     * , the number of MiB of memory reserved for the job. For other jobs, including all run on Fargate resources, see
     * <code>resourceRequirements</code>.
     * </p>
     * 
     * @return For jobs running on EC2 resources that didn't specify memory requirements using
     *         <code>resourceRequirements</code>, the number of MiB of memory reserved for the job. For other jobs,
     *         including all run on Fargate resources, see <code>resourceRequirements</code>.
     */
    public final Integer memory() {
        return memory;
    }

    /**
     * For responses, this returns true if the service returned a value for the Command property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasCommand() {
        return command != null && !(command instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The command that's passed to the container.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCommand} method.
     * </p>
     * 
     * @return The command that's passed to the container.
     */
    public final List<String> command() {
        return command;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) associated with the job upon execution.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) associated with the job upon execution.
     */
    public final String jobRoleArn() {
        return jobRoleArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the execution role that Batch can assume. For more information, see <a
     * href="https://docs.aws.amazon.com/batch/latest/userguide/execution-IAM-role.html">Batch execution IAM role</a> in
     * the <i>Batch User Guide</i>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the execution role that Batch can assume. For more information, see <a
     *         href="https://docs.aws.amazon.com/batch/latest/userguide/execution-IAM-role.html">Batch execution IAM
     *         role</a> in the <i>Batch User Guide</i>.
     */
    public final String executionRoleArn() {
        return executionRoleArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the Volumes property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasVolumes() {
        return volumes != null && !(volumes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of volumes associated with the job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasVolumes} method.
     * </p>
     * 
     * @return A list of volumes associated with the job.
     */
    public final List<Volume> volumes() {
        return volumes;
    }

    /**
     * For responses, this returns true if the service returned a value for the Environment property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEnvironment() {
        return environment != null && !(environment instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The environment variables to pass to a container.
     * </p>
     * <note>
     * <p>
     * Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved for
     * variables that are set by the Batch service.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEnvironment} method.
     * </p>
     * 
     * @return The environment variables to pass to a container.</p> <note>
     *         <p>
     *         Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved for
     *         variables that are set by the Batch service.
     *         </p>
     */
    public final List<KeyValuePair> environment() {
        return environment;
    }

    /**
     * For responses, this returns true if the service returned a value for the MountPoints property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasMountPoints() {
        return mountPoints != null && !(mountPoints instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The mount points for data volumes in your container.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMountPoints} method.
     * </p>
     * 
     * @return The mount points for data volumes in your container.
     */
    public final List<MountPoint> mountPoints() {
        return mountPoints;
    }

    /**
     * <p>
     * When this parameter is true, the container is given read-only access to its root file system. This parameter maps
     * to <code>ReadonlyRootfs</code> in the <a
     * href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the <a
     * href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--read-only</code> option to
     * <a href="https://docs.docker.com/engine/reference/commandline/run/"> <code>docker run</code> </a>.
     * </p>
     * 
     * @return When this parameter is true, the container is given read-only access to its root file system. This
     *         parameter maps to <code>ReadonlyRootfs</code> in the <a
     *         href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the
     *         <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the
     *         <code>--read-only</code> option to <a href="https://docs.docker.com/engine/reference/commandline/run/">
     *         <code>docker run</code> </a>.
     */
    public final Boolean readonlyRootFilesystem() {
        return readonlyRootFilesystem;
    }

    /**
     * For responses, this returns true if the service returned a value for the Ulimits property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasUlimits() {
        return ulimits != null && !(ulimits instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of <code>ulimit</code> values to set in the container. This parameter maps to <code>Ulimits</code> in the
     * <a href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the <a
     * href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--ulimit</code> option to <a
     * href="https://docs.docker.com/engine/reference/run/">docker run</a>.
     * </p>
     * <note>
     * <p>
     * This parameter isn't applicable to jobs that are running on Fargate resources.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasUlimits} method.
     * </p>
     * 
     * @return A list of <code>ulimit</code> values to set in the container. This parameter maps to <code>Ulimits</code>
     *         in the <a href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a>
     *         section of the <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the
     *         <code>--ulimit</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
     *         run</a>.</p> <note>
     *         <p>
     *         This parameter isn't applicable to jobs that are running on Fargate resources.
     *         </p>
     */
    public final List<Ulimit> ulimits() {
        return ulimits;
    }

    /**
     * <p>
     * When this parameter is true, the container is given elevated permissions on the host container instance (similar
     * to the <code>root</code> user). The default value is false.
     * </p>
     * <note>
     * <p>
     * This parameter isn't applicable to jobs that are running on Fargate resources and shouldn't be provided, or
     * specified as false.
     * </p>
     * </note>
     * 
     * @return When this parameter is true, the container is given elevated permissions on the host container instance
     *         (similar to the <code>root</code> user). The default value is false.</p> <note>
     *         <p>
     *         This parameter isn't applicable to jobs that are running on Fargate resources and shouldn't be provided,
     *         or specified as false.
     *         </p>
     */
    public final Boolean privileged() {
        return privileged;
    }

    /**
     * <p>
     * The user name to use inside the container. This parameter maps to <code>User</code> in the <a
     * href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the <a
     * href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--user</code> option to <a
     * href="https://docs.docker.com/engine/reference/run/">docker run</a>.
     * </p>
     * 
     * @return The user name to use inside the container. This parameter maps to <code>User</code> in the <a
     *         href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the
     *         <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--user</code>
     *         option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
     */
    public final String user() {
        return user;
    }

    /**
     * <p>
     * The exit code to return upon completion.
     * </p>
     * 
     * @return The exit code to return upon completion.
     */
    public final Integer exitCode() {
        return exitCode;
    }

    /**
     * <p>
     * A short (255 max characters) human-readable string to provide additional details about a running or stopped
     * container.
     * </p>
     * 
     * @return A short (255 max characters) human-readable string to provide additional details about a running or
     *         stopped container.
     */
    public final String reason() {
        return reason;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the container instance that the container is running on.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the container instance that the container is running on.
     */
    public final String containerInstanceArn() {
        return containerInstanceArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the Amazon ECS task that's associated with the container job. Each container
     * attempt receives a task ARN when they reach the <code>STARTING</code> status.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the Amazon ECS task that's associated with the container job. Each
     *         container attempt receives a task ARN when they reach the <code>STARTING</code> status.
     */
    public final String taskArn() {
        return taskArn;
    }

    /**
     * <p>
     * The name of the CloudWatch Logs log stream associated with the container. The log group for Batch jobs is
     * <code>/aws/batch/job</code>. Each container attempt receives a log stream name when they reach the
     * <code>RUNNING</code> status.
     * </p>
     * 
     * @return The name of the CloudWatch Logs log stream associated with the container. The log group for Batch jobs is
     *         <code>/aws/batch/job</code>. Each container attempt receives a log stream name when they reach the
     *         <code>RUNNING</code> status.
     */
    public final String logStreamName() {
        return logStreamName;
    }

    /**
     * <p>
     * The instance type of the underlying host infrastructure of a multi-node parallel job.
     * </p>
     * <note>
     * <p>
     * This parameter isn't applicable to jobs that are running on Fargate resources.
     * </p>
     * </note>
     * 
     * @return The instance type of the underlying host infrastructure of a multi-node parallel job.</p> <note>
     *         <p>
     *         This parameter isn't applicable to jobs that are running on Fargate resources.
     *         </p>
     */
    public final String instanceType() {
        return instanceType;
    }

    /**
     * For responses, this returns true if the service returned a value for the NetworkInterfaces property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasNetworkInterfaces() {
        return networkInterfaces != null && !(networkInterfaces instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The network interfaces associated with the job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNetworkInterfaces} method.
     * </p>
     * 
     * @return The network interfaces associated with the job.
     */
    public final List<NetworkInterface> networkInterfaces() {
        return networkInterfaces;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourceRequirements property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourceRequirements() {
        return resourceRequirements != null && !(resourceRequirements instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The type and amount of resources to assign to a container. The supported resources include <code>GPU</code>,
     * <code>MEMORY</code>, and <code>VCPU</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceRequirements} method.
     * </p>
     * 
     * @return The type and amount of resources to assign to a container. The supported resources include
     *         <code>GPU</code>, <code>MEMORY</code>, and <code>VCPU</code>.
     */
    public final List<ResourceRequirement> resourceRequirements() {
        return resourceRequirements;
    }

    /**
     * <p>
     * Linux-specific modifications that are applied to the container, such as details for device mappings.
     * </p>
     * 
     * @return Linux-specific modifications that are applied to the container, such as details for device mappings.
     */
    public final LinuxParameters linuxParameters() {
        return linuxParameters;
    }

    /**
     * <p>
     * The log configuration specification for the container.
     * </p>
     * <p>
     * This parameter maps to <code>LogConfig</code> in the <a
     * href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the <a
     * href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--log-driver</code> option
     * to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>. By default, containers use the same
     * logging driver that the Docker daemon uses. However, the container might use a different logging driver than the
     * Docker daemon by specifying a log driver with this parameter in the container definition. To use a different
     * logging driver for a container, the log system must be configured properly on the container instance. Or,
     * alternatively, it must be configured on a different log server for remote logging options. For more information
     * on the options for different supported log drivers, see <a
     * href="https://docs.docker.com/engine/admin/logging/overview/">Configure logging drivers</a> in the Docker
     * documentation.
     * </p>
     * <note>
     * <p>
     * Batch currently supports a subset of the logging drivers available to the Docker daemon (shown in the
     * <a>LogConfiguration</a> data type). Additional log drivers might be available in future releases of the Amazon
     * ECS container agent.
     * </p>
     * </note>
     * <p>
     * This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check the
     * Docker Remote API version on your container instance, log into your container instance and run the following
     * command: <code>sudo docker version | grep "Server API version"</code>
     * </p>
     * <note>
     * <p>
     * The Amazon ECS container agent running on a container instance must register the logging drivers available on
     * that instance with the <code>ECS_AVAILABLE_LOGGING_DRIVERS</code> environment variable before containers placed
     * on that instance can use these log configuration options. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html">Amazon ECS container
     * agent configuration</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * </note>
     * 
     * @return The log configuration specification for the container.</p>
     *         <p>
     *         This parameter maps to <code>LogConfig</code> in the <a
     *         href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the
     *         <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the
     *         <code>--log-driver</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
     *         run</a>. By default, containers use the same logging driver that the Docker daemon uses. However, the
     *         container might use a different logging driver than the Docker daemon by specifying a log driver with
     *         this parameter in the container definition. To use a different logging driver for a container, the log
     *         system must be configured properly on the container instance. Or, alternatively, it must be configured on
     *         a different log server for remote logging options. For more information on the options for different
     *         supported log drivers, see <a href="https://docs.docker.com/engine/admin/logging/overview/">Configure
     *         logging drivers</a> in the Docker documentation.
     *         </p>
     *         <note>
     *         <p>
     *         Batch currently supports a subset of the logging drivers available to the Docker daemon (shown in the
     *         <a>LogConfiguration</a> data type). Additional log drivers might be available in future releases of the
     *         Amazon ECS container agent.
     *         </p>
     *         </note>
     *         <p>
     *         This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To
     *         check the Docker Remote API version on your container instance, log into your container instance and run
     *         the following command: <code>sudo docker version | grep "Server API version"</code>
     *         </p>
     *         <note>
     *         <p>
     *         The Amazon ECS container agent running on a container instance must register the logging drivers
     *         available on that instance with the <code>ECS_AVAILABLE_LOGGING_DRIVERS</code> environment variable
     *         before containers placed on that instance can use these log configuration options. For more information,
     *         see <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html">Amazon
     *         ECS container agent configuration</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     *         </p>
     */
    public final LogConfiguration logConfiguration() {
        return logConfiguration;
    }

    /**
     * For responses, this returns true if the service returned a value for the Secrets property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasSecrets() {
        return secrets != null && !(secrets instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The secrets to pass to the container. For more information, see <a
     * href="https://docs.aws.amazon.com/batch/latest/userguide/specifying-sensitive-data.html">Specifying sensitive
     * data</a> in the <i>Batch User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSecrets} method.
     * </p>
     * 
     * @return The secrets to pass to the container. For more information, see <a
     *         href="https://docs.aws.amazon.com/batch/latest/userguide/specifying-sensitive-data.html">Specifying
     *         sensitive data</a> in the <i>Batch User Guide</i>.
     */
    public final List<Secret> secrets() {
        return secrets;
    }

    /**
     * <p>
     * The network configuration for jobs that are running on Fargate resources. Jobs that are running on EC2 resources
     * must not specify this parameter.
     * </p>
     * 
     * @return The network configuration for jobs that are running on Fargate resources. Jobs that are running on EC2
     *         resources must not specify this parameter.
     */
    public final NetworkConfiguration networkConfiguration() {
        return networkConfiguration;
    }

    /**
     * <p>
     * The platform configuration for jobs that are running on Fargate resources. Jobs that are running on EC2 resources
     * must not specify this parameter.
     * </p>
     * 
     * @return The platform configuration for jobs that are running on Fargate resources. Jobs that are running on EC2
     *         resources must not specify this parameter.
     */
    public final FargatePlatformConfiguration fargatePlatformConfiguration() {
        return fargatePlatformConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(image());
        hashCode = 31 * hashCode + Objects.hashCode(vcpus());
        hashCode = 31 * hashCode + Objects.hashCode(memory());
        hashCode = 31 * hashCode + Objects.hashCode(hasCommand() ? command() : null);
        hashCode = 31 * hashCode + Objects.hashCode(jobRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(executionRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasVolumes() ? volumes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasEnvironment() ? environment() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasMountPoints() ? mountPoints() : null);
        hashCode = 31 * hashCode + Objects.hashCode(readonlyRootFilesystem());
        hashCode = 31 * hashCode + Objects.hashCode(hasUlimits() ? ulimits() : null);
        hashCode = 31 * hashCode + Objects.hashCode(privileged());
        hashCode = 31 * hashCode + Objects.hashCode(user());
        hashCode = 31 * hashCode + Objects.hashCode(exitCode());
        hashCode = 31 * hashCode + Objects.hashCode(reason());
        hashCode = 31 * hashCode + Objects.hashCode(containerInstanceArn());
        hashCode = 31 * hashCode + Objects.hashCode(taskArn());
        hashCode = 31 * hashCode + Objects.hashCode(logStreamName());
        hashCode = 31 * hashCode + Objects.hashCode(instanceType());
        hashCode = 31 * hashCode + Objects.hashCode(hasNetworkInterfaces() ? networkInterfaces() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceRequirements() ? resourceRequirements() : null);
        hashCode = 31 * hashCode + Objects.hashCode(linuxParameters());
        hashCode = 31 * hashCode + Objects.hashCode(logConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(hasSecrets() ? secrets() : null);
        hashCode = 31 * hashCode + Objects.hashCode(networkConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(fargatePlatformConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ContainerDetail)) {
            return false;
        }
        ContainerDetail other = (ContainerDetail) obj;
        return Objects.equals(image(), other.image()) && Objects.equals(vcpus(), other.vcpus())
                && Objects.equals(memory(), other.memory()) && hasCommand() == other.hasCommand()
                && Objects.equals(command(), other.command()) && Objects.equals(jobRoleArn(), other.jobRoleArn())
                && Objects.equals(executionRoleArn(), other.executionRoleArn()) && hasVolumes() == other.hasVolumes()
                && Objects.equals(volumes(), other.volumes()) && hasEnvironment() == other.hasEnvironment()
                && Objects.equals(environment(), other.environment()) && hasMountPoints() == other.hasMountPoints()
                && Objects.equals(mountPoints(), other.mountPoints())
                && Objects.equals(readonlyRootFilesystem(), other.readonlyRootFilesystem()) && hasUlimits() == other.hasUlimits()
                && Objects.equals(ulimits(), other.ulimits()) && Objects.equals(privileged(), other.privileged())
                && Objects.equals(user(), other.user()) && Objects.equals(exitCode(), other.exitCode())
                && Objects.equals(reason(), other.reason())
                && Objects.equals(containerInstanceArn(), other.containerInstanceArn())
                && Objects.equals(taskArn(), other.taskArn()) && Objects.equals(logStreamName(), other.logStreamName())
                && Objects.equals(instanceType(), other.instanceType()) && hasNetworkInterfaces() == other.hasNetworkInterfaces()
                && Objects.equals(networkInterfaces(), other.networkInterfaces())
                && hasResourceRequirements() == other.hasResourceRequirements()
                && Objects.equals(resourceRequirements(), other.resourceRequirements())
                && Objects.equals(linuxParameters(), other.linuxParameters())
                && Objects.equals(logConfiguration(), other.logConfiguration()) && hasSecrets() == other.hasSecrets()
                && Objects.equals(secrets(), other.secrets())
                && Objects.equals(networkConfiguration(), other.networkConfiguration())
                && Objects.equals(fargatePlatformConfiguration(), other.fargatePlatformConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ContainerDetail").add("Image", image()).add("Vcpus", vcpus()).add("Memory", memory())
                .add("Command", hasCommand() ? command() : null).add("JobRoleArn", jobRoleArn())
                .add("ExecutionRoleArn", executionRoleArn()).add("Volumes", hasVolumes() ? volumes() : null)
                .add("Environment", hasEnvironment() ? environment() : null)
                .add("MountPoints", hasMountPoints() ? mountPoints() : null)
                .add("ReadonlyRootFilesystem", readonlyRootFilesystem()).add("Ulimits", hasUlimits() ? ulimits() : null)
                .add("Privileged", privileged()).add("User", user()).add("ExitCode", exitCode()).add("Reason", reason())
                .add("ContainerInstanceArn", containerInstanceArn()).add("TaskArn", taskArn())
                .add("LogStreamName", logStreamName()).add("InstanceType", instanceType())
                .add("NetworkInterfaces", hasNetworkInterfaces() ? networkInterfaces() : null)
                .add("ResourceRequirements", hasResourceRequirements() ? resourceRequirements() : null)
                .add("LinuxParameters", linuxParameters()).add("LogConfiguration", logConfiguration())
                .add("Secrets", hasSecrets() ? secrets() : null).add("NetworkConfiguration", networkConfiguration())
                .add("FargatePlatformConfiguration", fargatePlatformConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "image":
            return Optional.ofNullable(clazz.cast(image()));
        case "vcpus":
            return Optional.ofNullable(clazz.cast(vcpus()));
        case "memory":
            return Optional.ofNullable(clazz.cast(memory()));
        case "command":
            return Optional.ofNullable(clazz.cast(command()));
        case "jobRoleArn":
            return Optional.ofNullable(clazz.cast(jobRoleArn()));
        case "executionRoleArn":
            return Optional.ofNullable(clazz.cast(executionRoleArn()));
        case "volumes":
            return Optional.ofNullable(clazz.cast(volumes()));
        case "environment":
            return Optional.ofNullable(clazz.cast(environment()));
        case "mountPoints":
            return Optional.ofNullable(clazz.cast(mountPoints()));
        case "readonlyRootFilesystem":
            return Optional.ofNullable(clazz.cast(readonlyRootFilesystem()));
        case "ulimits":
            return Optional.ofNullable(clazz.cast(ulimits()));
        case "privileged":
            return Optional.ofNullable(clazz.cast(privileged()));
        case "user":
            return Optional.ofNullable(clazz.cast(user()));
        case "exitCode":
            return Optional.ofNullable(clazz.cast(exitCode()));
        case "reason":
            return Optional.ofNullable(clazz.cast(reason()));
        case "containerInstanceArn":
            return Optional.ofNullable(clazz.cast(containerInstanceArn()));
        case "taskArn":
            return Optional.ofNullable(clazz.cast(taskArn()));
        case "logStreamName":
            return Optional.ofNullable(clazz.cast(logStreamName()));
        case "instanceType":
            return Optional.ofNullable(clazz.cast(instanceType()));
        case "networkInterfaces":
            return Optional.ofNullable(clazz.cast(networkInterfaces()));
        case "resourceRequirements":
            return Optional.ofNullable(clazz.cast(resourceRequirements()));
        case "linuxParameters":
            return Optional.ofNullable(clazz.cast(linuxParameters()));
        case "logConfiguration":
            return Optional.ofNullable(clazz.cast(logConfiguration()));
        case "secrets":
            return Optional.ofNullable(clazz.cast(secrets()));
        case "networkConfiguration":
            return Optional.ofNullable(clazz.cast(networkConfiguration()));
        case "fargatePlatformConfiguration":
            return Optional.ofNullable(clazz.cast(fargatePlatformConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ContainerDetail, T> g) {
        return obj -> g.apply((ContainerDetail) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ContainerDetail> {
        /**
         * <p>
         * The image used to start the container.
         * </p>
         * 
         * @param image
         *        The image used to start the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder image(String image);

        /**
         * <p>
         * The number of vCPUs reserved for the container. For jobs that run on EC2 resources, you can specify the vCPU
         * requirement for the job using <code>resourceRequirements</code>, but you can't specify the vCPU requirements
         * in both the <code>vcpus</code> and <code>resourceRequirements</code> object. This parameter maps to
         * <code>CpuShares</code> in the <a href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a
         * container</a> section of the <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and
         * the <code>--cpu-shares</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
         * run</a>. Each vCPU is equivalent to 1,024 CPU shares. You must specify at least one vCPU. This is required
         * but can be specified in several places. It must be specified for each node at least once.
         * </p>
         * <note>
         * <p>
         * This parameter isn't applicable to jobs that run on Fargate resources. For jobs that run on Fargate
         * resources, you must specify the vCPU requirement for the job using <code>resourceRequirements</code>.
         * </p>
         * </note>
         * 
         * @param vcpus
         *        The number of vCPUs reserved for the container. For jobs that run on EC2 resources, you can specify
         *        the vCPU requirement for the job using <code>resourceRequirements</code>, but you can't specify the
         *        vCPU requirements in both the <code>vcpus</code> and <code>resourceRequirements</code> object. This
         *        parameter maps to <code>CpuShares</code> in the <a
         *        href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of
         *        the <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the
         *        <code>--cpu-shares</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
         *        run</a>. Each vCPU is equivalent to 1,024 CPU shares. You must specify at least one vCPU. This is
         *        required but can be specified in several places. It must be specified for each node at least once.</p>
         *        <note>
         *        <p>
         *        This parameter isn't applicable to jobs that run on Fargate resources. For jobs that run on Fargate
         *        resources, you must specify the vCPU requirement for the job using <code>resourceRequirements</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vcpus(Integer vcpus);

        /**
         * <p>
         * For jobs running on EC2 resources that didn't specify memory requirements using
         * <code>resourceRequirements</code>, the number of MiB of memory reserved for the job. For other jobs,
         * including all run on Fargate resources, see <code>resourceRequirements</code>.
         * </p>
         * 
         * @param memory
         *        For jobs running on EC2 resources that didn't specify memory requirements using
         *        <code>resourceRequirements</code>, the number of MiB of memory reserved for the job. For other jobs,
         *        including all run on Fargate resources, see <code>resourceRequirements</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder memory(Integer memory);

        /**
         * <p>
         * The command that's passed to the container.
         * </p>
         * 
         * @param command
         *        The command that's passed to the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(Collection<String> command);

        /**
         * <p>
         * The command that's passed to the container.
         * </p>
         * 
         * @param command
         *        The command that's passed to the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(String... command);

        /**
         * <p>
         * The Amazon Resource Name (ARN) associated with the job upon execution.
         * </p>
         * 
         * @param jobRoleArn
         *        The Amazon Resource Name (ARN) associated with the job upon execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobRoleArn(String jobRoleArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the execution role that Batch can assume. For more information, see <a
         * href="https://docs.aws.amazon.com/batch/latest/userguide/execution-IAM-role.html">Batch execution IAM
         * role</a> in the <i>Batch User Guide</i>.
         * </p>
         * 
         * @param executionRoleArn
         *        The Amazon Resource Name (ARN) of the execution role that Batch can assume. For more information, see
         *        <a href="https://docs.aws.amazon.com/batch/latest/userguide/execution-IAM-role.html">Batch execution
         *        IAM role</a> in the <i>Batch User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionRoleArn(String executionRoleArn);

        /**
         * <p>
         * A list of volumes associated with the job.
         * </p>
         * 
         * @param volumes
         *        A list of volumes associated with the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumes(Collection<Volume> volumes);

        /**
         * <p>
         * A list of volumes associated with the job.
         * </p>
         * 
         * @param volumes
         *        A list of volumes associated with the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumes(Volume... volumes);

        /**
         * <p>
         * A list of volumes associated with the job.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<Volume>.Builder} avoiding the need
         * to create one manually via {@link List<Volume>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Volume>.Builder#build()} is called immediately and its
         * result is passed to {@link #volumes(List<Volume>)}.
         * 
         * @param volumes
         *        a consumer that will call methods on {@link List<Volume>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #volumes(List<Volume>)
         */
        Builder volumes(Consumer<Volume.Builder>... volumes);

        /**
         * <p>
         * The environment variables to pass to a container.
         * </p>
         * <note>
         * <p>
         * Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved for
         * variables that are set by the Batch service.
         * </p>
         * </note>
         * 
         * @param environment
         *        The environment variables to pass to a container.</p> <note>
         *        <p>
         *        Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved
         *        for variables that are set by the Batch service.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environment(Collection<KeyValuePair> environment);

        /**
         * <p>
         * The environment variables to pass to a container.
         * </p>
         * <note>
         * <p>
         * Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved for
         * variables that are set by the Batch service.
         * </p>
         * </note>
         * 
         * @param environment
         *        The environment variables to pass to a container.</p> <note>
         *        <p>
         *        Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved
         *        for variables that are set by the Batch service.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environment(KeyValuePair... environment);

        /**
         * <p>
         * The environment variables to pass to a container.
         * </p>
         * <note>
         * <p>
         * Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved for
         * variables that are set by the Batch service.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link List<KeyValuePair>.Builder}
         * avoiding the need to create one manually via {@link List<KeyValuePair>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<KeyValuePair>.Builder#build()} is called immediately and its
         * result is passed to {@link #environment(List<KeyValuePair>)}.
         * 
         * @param environment
         *        a consumer that will call methods on {@link List<KeyValuePair>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #environment(List<KeyValuePair>)
         */
        Builder environment(Consumer<KeyValuePair.Builder>... environment);

        /**
         * <p>
         * The mount points for data volumes in your container.
         * </p>
         * 
         * @param mountPoints
         *        The mount points for data volumes in your container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mountPoints(Collection<MountPoint> mountPoints);

        /**
         * <p>
         * The mount points for data volumes in your container.
         * </p>
         * 
         * @param mountPoints
         *        The mount points for data volumes in your container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mountPoints(MountPoint... mountPoints);

        /**
         * <p>
         * The mount points for data volumes in your container.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<MountPoint>.Builder} avoiding the
         * need to create one manually via {@link List<MountPoint>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<MountPoint>.Builder#build()} is called immediately and its
         * result is passed to {@link #mountPoints(List<MountPoint>)}.
         * 
         * @param mountPoints
         *        a consumer that will call methods on {@link List<MountPoint>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #mountPoints(List<MountPoint>)
         */
        Builder mountPoints(Consumer<MountPoint.Builder>... mountPoints);

        /**
         * <p>
         * When this parameter is true, the container is given read-only access to its root file system. This parameter
         * maps to <code>ReadonlyRootfs</code> in the <a
         * href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the <a
         * href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--read-only</code>
         * option to <a href="https://docs.docker.com/engine/reference/commandline/run/"> <code>docker run</code> </a>.
         * </p>
         * 
         * @param readonlyRootFilesystem
         *        When this parameter is true, the container is given read-only access to its root file system. This
         *        parameter maps to <code>ReadonlyRootfs</code> in the <a
         *        href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of
         *        the <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the
         *        <code>--read-only</code> option to <a
         *        href="https://docs.docker.com/engine/reference/commandline/run/"> <code>docker run</code> </a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder readonlyRootFilesystem(Boolean readonlyRootFilesystem);

        /**
         * <p>
         * A list of <code>ulimit</code> values to set in the container. This parameter maps to <code>Ulimits</code> in
         * the <a href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of
         * the <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--ulimit</code>
         * option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
         * </p>
         * <note>
         * <p>
         * This parameter isn't applicable to jobs that are running on Fargate resources.
         * </p>
         * </note>
         * 
         * @param ulimits
         *        A list of <code>ulimit</code> values to set in the container. This parameter maps to
         *        <code>Ulimits</code> in the <a
         *        href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of
         *        the <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the
         *        <code>--ulimit</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
         *        run</a>.</p> <note>
         *        <p>
         *        This parameter isn't applicable to jobs that are running on Fargate resources.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ulimits(Collection<Ulimit> ulimits);

        /**
         * <p>
         * A list of <code>ulimit</code> values to set in the container. This parameter maps to <code>Ulimits</code> in
         * the <a href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of
         * the <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--ulimit</code>
         * option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
         * </p>
         * <note>
         * <p>
         * This parameter isn't applicable to jobs that are running on Fargate resources.
         * </p>
         * </note>
         * 
         * @param ulimits
         *        A list of <code>ulimit</code> values to set in the container. This parameter maps to
         *        <code>Ulimits</code> in the <a
         *        href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of
         *        the <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the
         *        <code>--ulimit</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
         *        run</a>.</p> <note>
         *        <p>
         *        This parameter isn't applicable to jobs that are running on Fargate resources.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ulimits(Ulimit... ulimits);

        /**
         * <p>
         * A list of <code>ulimit</code> values to set in the container. This parameter maps to <code>Ulimits</code> in
         * the <a href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of
         * the <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--ulimit</code>
         * option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
         * </p>
         * <note>
         * <p>
         * This parameter isn't applicable to jobs that are running on Fargate resources.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link List<Ulimit>.Builder} avoiding
         * the need to create one manually via {@link List<Ulimit>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Ulimit>.Builder#build()} is called immediately and its
         * result is passed to {@link #ulimits(List<Ulimit>)}.
         * 
         * @param ulimits
         *        a consumer that will call methods on {@link List<Ulimit>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ulimits(List<Ulimit>)
         */
        Builder ulimits(Consumer<Ulimit.Builder>... ulimits);

        /**
         * <p>
         * When this parameter is true, the container is given elevated permissions on the host container instance
         * (similar to the <code>root</code> user). The default value is false.
         * </p>
         * <note>
         * <p>
         * This parameter isn't applicable to jobs that are running on Fargate resources and shouldn't be provided, or
         * specified as false.
         * </p>
         * </note>
         * 
         * @param privileged
         *        When this parameter is true, the container is given elevated permissions on the host container
         *        instance (similar to the <code>root</code> user). The default value is false.</p> <note>
         *        <p>
         *        This parameter isn't applicable to jobs that are running on Fargate resources and shouldn't be
         *        provided, or specified as false.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder privileged(Boolean privileged);

        /**
         * <p>
         * The user name to use inside the container. This parameter maps to <code>User</code> in the <a
         * href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the <a
         * href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--user</code> option to
         * <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
         * </p>
         * 
         * @param user
         *        The user name to use inside the container. This parameter maps to <code>User</code> in the <a
         *        href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of
         *        the <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the
         *        <code>--user</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder user(String user);

        /**
         * <p>
         * The exit code to return upon completion.
         * </p>
         * 
         * @param exitCode
         *        The exit code to return upon completion.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exitCode(Integer exitCode);

        /**
         * <p>
         * A short (255 max characters) human-readable string to provide additional details about a running or stopped
         * container.
         * </p>
         * 
         * @param reason
         *        A short (255 max characters) human-readable string to provide additional details about a running or
         *        stopped container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reason(String reason);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the container instance that the container is running on.
         * </p>
         * 
         * @param containerInstanceArn
         *        The Amazon Resource Name (ARN) of the container instance that the container is running on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerInstanceArn(String containerInstanceArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the Amazon ECS task that's associated with the container job. Each
         * container attempt receives a task ARN when they reach the <code>STARTING</code> status.
         * </p>
         * 
         * @param taskArn
         *        The Amazon Resource Name (ARN) of the Amazon ECS task that's associated with the container job. Each
         *        container attempt receives a task ARN when they reach the <code>STARTING</code> status.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskArn(String taskArn);

        /**
         * <p>
         * The name of the CloudWatch Logs log stream associated with the container. The log group for Batch jobs is
         * <code>/aws/batch/job</code>. Each container attempt receives a log stream name when they reach the
         * <code>RUNNING</code> status.
         * </p>
         * 
         * @param logStreamName
         *        The name of the CloudWatch Logs log stream associated with the container. The log group for Batch jobs
         *        is <code>/aws/batch/job</code>. Each container attempt receives a log stream name when they reach the
         *        <code>RUNNING</code> status.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logStreamName(String logStreamName);

        /**
         * <p>
         * The instance type of the underlying host infrastructure of a multi-node parallel job.
         * </p>
         * <note>
         * <p>
         * This parameter isn't applicable to jobs that are running on Fargate resources.
         * </p>
         * </note>
         * 
         * @param instanceType
         *        The instance type of the underlying host infrastructure of a multi-node parallel job.</p> <note>
         *        <p>
         *        This parameter isn't applicable to jobs that are running on Fargate resources.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * The network interfaces associated with the job.
         * </p>
         * 
         * @param networkInterfaces
         *        The network interfaces associated with the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaces(Collection<NetworkInterface> networkInterfaces);

        /**
         * <p>
         * The network interfaces associated with the job.
         * </p>
         * 
         * @param networkInterfaces
         *        The network interfaces associated with the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaces(NetworkInterface... networkInterfaces);

        /**
         * <p>
         * The network interfaces associated with the job.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<NetworkInterface>.Builder} avoiding
         * the need to create one manually via {@link List<NetworkInterface>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<NetworkInterface>.Builder#build()} is called immediately and
         * its result is passed to {@link #networkInterfaces(List<NetworkInterface>)}.
         * 
         * @param networkInterfaces
         *        a consumer that will call methods on {@link List<NetworkInterface>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #networkInterfaces(List<NetworkInterface>)
         */
        Builder networkInterfaces(Consumer<NetworkInterface.Builder>... networkInterfaces);

        /**
         * <p>
         * The type and amount of resources to assign to a container. The supported resources include <code>GPU</code>,
         * <code>MEMORY</code>, and <code>VCPU</code>.
         * </p>
         * 
         * @param resourceRequirements
         *        The type and amount of resources to assign to a container. The supported resources include
         *        <code>GPU</code>, <code>MEMORY</code>, and <code>VCPU</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceRequirements(Collection<ResourceRequirement> resourceRequirements);

        /**
         * <p>
         * The type and amount of resources to assign to a container. The supported resources include <code>GPU</code>,
         * <code>MEMORY</code>, and <code>VCPU</code>.
         * </p>
         * 
         * @param resourceRequirements
         *        The type and amount of resources to assign to a container. The supported resources include
         *        <code>GPU</code>, <code>MEMORY</code>, and <code>VCPU</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceRequirements(ResourceRequirement... resourceRequirements);

        /**
         * <p>
         * The type and amount of resources to assign to a container. The supported resources include <code>GPU</code>,
         * <code>MEMORY</code>, and <code>VCPU</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<ResourceRequirement>.Builder}
         * avoiding the need to create one manually via {@link List<ResourceRequirement>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ResourceRequirement>.Builder#build()} is called immediately
         * and its result is passed to {@link #resourceRequirements(List<ResourceRequirement>)}.
         * 
         * @param resourceRequirements
         *        a consumer that will call methods on {@link List<ResourceRequirement>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resourceRequirements(List<ResourceRequirement>)
         */
        Builder resourceRequirements(Consumer<ResourceRequirement.Builder>... resourceRequirements);

        /**
         * <p>
         * Linux-specific modifications that are applied to the container, such as details for device mappings.
         * </p>
         * 
         * @param linuxParameters
         *        Linux-specific modifications that are applied to the container, such as details for device mappings.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder linuxParameters(LinuxParameters linuxParameters);

        /**
         * <p>
         * Linux-specific modifications that are applied to the container, such as details for device mappings.
         * </p>
         * This is a convenience method that creates an instance of the {@link LinuxParameters.Builder} avoiding the
         * need to create one manually via {@link LinuxParameters#builder()}.
         *
         * When the {@link Consumer} completes, {@link LinuxParameters.Builder#build()} is called immediately and its
         * result is passed to {@link #linuxParameters(LinuxParameters)}.
         * 
         * @param linuxParameters
         *        a consumer that will call methods on {@link LinuxParameters.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #linuxParameters(LinuxParameters)
         */
        default Builder linuxParameters(Consumer<LinuxParameters.Builder> linuxParameters) {
            return linuxParameters(LinuxParameters.builder().applyMutation(linuxParameters).build());
        }

        /**
         * <p>
         * The log configuration specification for the container.
         * </p>
         * <p>
         * This parameter maps to <code>LogConfig</code> in the <a
         * href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the <a
         * href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--log-driver</code>
         * option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>. By default, containers use
         * the same logging driver that the Docker daemon uses. However, the container might use a different logging
         * driver than the Docker daemon by specifying a log driver with this parameter in the container definition. To
         * use a different logging driver for a container, the log system must be configured properly on the container
         * instance. Or, alternatively, it must be configured on a different log server for remote logging options. For
         * more information on the options for different supported log drivers, see <a
         * href="https://docs.docker.com/engine/admin/logging/overview/">Configure logging drivers</a> in the Docker
         * documentation.
         * </p>
         * <note>
         * <p>
         * Batch currently supports a subset of the logging drivers available to the Docker daemon (shown in the
         * <a>LogConfiguration</a> data type). Additional log drivers might be available in future releases of the
         * Amazon ECS container agent.
         * </p>
         * </note>
         * <p>
         * This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check
         * the Docker Remote API version on your container instance, log into your container instance and run the
         * following command: <code>sudo docker version | grep "Server API version"</code>
         * </p>
         * <note>
         * <p>
         * The Amazon ECS container agent running on a container instance must register the logging drivers available on
         * that instance with the <code>ECS_AVAILABLE_LOGGING_DRIVERS</code> environment variable before containers
         * placed on that instance can use these log configuration options. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html">Amazon ECS container
         * agent configuration</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * </note>
         * 
         * @param logConfiguration
         *        The log configuration specification for the container.</p>
         *        <p>
         *        This parameter maps to <code>LogConfig</code> in the <a
         *        href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of
         *        the <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the
         *        <code>--log-driver</code> option to <a href="https://docs.docker.com/engine/reference/run/">docker
         *        run</a>. By default, containers use the same logging driver that the Docker daemon uses. However, the
         *        container might use a different logging driver than the Docker daemon by specifying a log driver with
         *        this parameter in the container definition. To use a different logging driver for a container, the log
         *        system must be configured properly on the container instance. Or, alternatively, it must be configured
         *        on a different log server for remote logging options. For more information on the options for
         *        different supported log drivers, see <a
         *        href="https://docs.docker.com/engine/admin/logging/overview/">Configure logging drivers</a> in the
         *        Docker documentation.
         *        </p>
         *        <note>
         *        <p>
         *        Batch currently supports a subset of the logging drivers available to the Docker daemon (shown in the
         *        <a>LogConfiguration</a> data type). Additional log drivers might be available in future releases of
         *        the Amazon ECS container agent.
         *        </p>
         *        </note>
         *        <p>
         *        This parameter requires version 1.18 of the Docker Remote API or greater on your container instance.
         *        To check the Docker Remote API version on your container instance, log into your container instance
         *        and run the following command: <code>sudo docker version | grep "Server API version"</code>
         *        </p>
         *        <note>
         *        <p>
         *        The Amazon ECS container agent running on a container instance must register the logging drivers
         *        available on that instance with the <code>ECS_AVAILABLE_LOGGING_DRIVERS</code> environment variable
         *        before containers placed on that instance can use these log configuration options. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html">Amazon ECS
         *        container agent configuration</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logConfiguration(LogConfiguration logConfiguration);

        /**
         * <p>
         * The log configuration specification for the container.
         * </p>
         * <p>
         * This parameter maps to <code>LogConfig</code> in the <a
         * href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the <a
         * href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--log-driver</code>
         * option to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>. By default, containers use
         * the same logging driver that the Docker daemon uses. However, the container might use a different logging
         * driver than the Docker daemon by specifying a log driver with this parameter in the container definition. To
         * use a different logging driver for a container, the log system must be configured properly on the container
         * instance. Or, alternatively, it must be configured on a different log server for remote logging options. For
         * more information on the options for different supported log drivers, see <a
         * href="https://docs.docker.com/engine/admin/logging/overview/">Configure logging drivers</a> in the Docker
         * documentation.
         * </p>
         * <note>
         * <p>
         * Batch currently supports a subset of the logging drivers available to the Docker daemon (shown in the
         * <a>LogConfiguration</a> data type). Additional log drivers might be available in future releases of the
         * Amazon ECS container agent.
         * </p>
         * </note>
         * <p>
         * This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check
         * the Docker Remote API version on your container instance, log into your container instance and run the
         * following command: <code>sudo docker version | grep "Server API version"</code>
         * </p>
         * <note>
         * <p>
         * The Amazon ECS container agent running on a container instance must register the logging drivers available on
         * that instance with the <code>ECS_AVAILABLE_LOGGING_DRIVERS</code> environment variable before containers
         * placed on that instance can use these log configuration options. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html">Amazon ECS container
         * agent configuration</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link LogConfiguration.Builder}
         * avoiding the need to create one manually via {@link LogConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link LogConfiguration.Builder#build()} is called immediately and its
         * result is passed to {@link #logConfiguration(LogConfiguration)}.
         * 
         * @param logConfiguration
         *        a consumer that will call methods on {@link LogConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #logConfiguration(LogConfiguration)
         */
        default Builder logConfiguration(Consumer<LogConfiguration.Builder> logConfiguration) {
            return logConfiguration(LogConfiguration.builder().applyMutation(logConfiguration).build());
        }

        /**
         * <p>
         * The secrets to pass to the container. For more information, see <a
         * href="https://docs.aws.amazon.com/batch/latest/userguide/specifying-sensitive-data.html">Specifying sensitive
         * data</a> in the <i>Batch User Guide</i>.
         * </p>
         * 
         * @param secrets
         *        The secrets to pass to the container. For more information, see <a
         *        href="https://docs.aws.amazon.com/batch/latest/userguide/specifying-sensitive-data.html">Specifying
         *        sensitive data</a> in the <i>Batch User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secrets(Collection<Secret> secrets);

        /**
         * <p>
         * The secrets to pass to the container. For more information, see <a
         * href="https://docs.aws.amazon.com/batch/latest/userguide/specifying-sensitive-data.html">Specifying sensitive
         * data</a> in the <i>Batch User Guide</i>.
         * </p>
         * 
         * @param secrets
         *        The secrets to pass to the container. For more information, see <a
         *        href="https://docs.aws.amazon.com/batch/latest/userguide/specifying-sensitive-data.html">Specifying
         *        sensitive data</a> in the <i>Batch User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secrets(Secret... secrets);

        /**
         * <p>
         * The secrets to pass to the container. For more information, see <a
         * href="https://docs.aws.amazon.com/batch/latest/userguide/specifying-sensitive-data.html">Specifying sensitive
         * data</a> in the <i>Batch User Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<Secret>.Builder} avoiding the need
         * to create one manually via {@link List<Secret>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Secret>.Builder#build()} is called immediately and its
         * result is passed to {@link #secrets(List<Secret>)}.
         * 
         * @param secrets
         *        a consumer that will call methods on {@link List<Secret>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #secrets(List<Secret>)
         */
        Builder secrets(Consumer<Secret.Builder>... secrets);

        /**
         * <p>
         * The network configuration for jobs that are running on Fargate resources. Jobs that are running on EC2
         * resources must not specify this parameter.
         * </p>
         * 
         * @param networkConfiguration
         *        The network configuration for jobs that are running on Fargate resources. Jobs that are running on EC2
         *        resources must not specify this parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkConfiguration(NetworkConfiguration networkConfiguration);

        /**
         * <p>
         * The network configuration for jobs that are running on Fargate resources. Jobs that are running on EC2
         * resources must not specify this parameter.
         * </p>
         * This is a convenience method that creates an instance of the {@link NetworkConfiguration.Builder} avoiding
         * the need to create one manually via {@link NetworkConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link NetworkConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #networkConfiguration(NetworkConfiguration)}.
         * 
         * @param networkConfiguration
         *        a consumer that will call methods on {@link NetworkConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #networkConfiguration(NetworkConfiguration)
         */
        default Builder networkConfiguration(Consumer<NetworkConfiguration.Builder> networkConfiguration) {
            return networkConfiguration(NetworkConfiguration.builder().applyMutation(networkConfiguration).build());
        }

        /**
         * <p>
         * The platform configuration for jobs that are running on Fargate resources. Jobs that are running on EC2
         * resources must not specify this parameter.
         * </p>
         * 
         * @param fargatePlatformConfiguration
         *        The platform configuration for jobs that are running on Fargate resources. Jobs that are running on
         *        EC2 resources must not specify this parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fargatePlatformConfiguration(FargatePlatformConfiguration fargatePlatformConfiguration);

        /**
         * <p>
         * The platform configuration for jobs that are running on Fargate resources. Jobs that are running on EC2
         * resources must not specify this parameter.
         * </p>
         * This is a convenience method that creates an instance of the {@link FargatePlatformConfiguration.Builder}
         * avoiding the need to create one manually via {@link FargatePlatformConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link FargatePlatformConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #fargatePlatformConfiguration(FargatePlatformConfiguration)}.
         * 
         * @param fargatePlatformConfiguration
         *        a consumer that will call methods on {@link FargatePlatformConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #fargatePlatformConfiguration(FargatePlatformConfiguration)
         */
        default Builder fargatePlatformConfiguration(Consumer<FargatePlatformConfiguration.Builder> fargatePlatformConfiguration) {
            return fargatePlatformConfiguration(FargatePlatformConfiguration.builder()
                    .applyMutation(fargatePlatformConfiguration).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String image;

        private Integer vcpus;

        private Integer memory;

        private List<String> command = DefaultSdkAutoConstructList.getInstance();

        private String jobRoleArn;

        private String executionRoleArn;

        private List<Volume> volumes = DefaultSdkAutoConstructList.getInstance();

        private List<KeyValuePair> environment = DefaultSdkAutoConstructList.getInstance();

        private List<MountPoint> mountPoints = DefaultSdkAutoConstructList.getInstance();

        private Boolean readonlyRootFilesystem;

        private List<Ulimit> ulimits = DefaultSdkAutoConstructList.getInstance();

        private Boolean privileged;

        private String user;

        private Integer exitCode;

        private String reason;

        private String containerInstanceArn;

        private String taskArn;

        private String logStreamName;

        private String instanceType;

        private List<NetworkInterface> networkInterfaces = DefaultSdkAutoConstructList.getInstance();

        private List<ResourceRequirement> resourceRequirements = DefaultSdkAutoConstructList.getInstance();

        private LinuxParameters linuxParameters;

        private LogConfiguration logConfiguration;

        private List<Secret> secrets = DefaultSdkAutoConstructList.getInstance();

        private NetworkConfiguration networkConfiguration;

        private FargatePlatformConfiguration fargatePlatformConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(ContainerDetail model) {
            image(model.image);
            vcpus(model.vcpus);
            memory(model.memory);
            command(model.command);
            jobRoleArn(model.jobRoleArn);
            executionRoleArn(model.executionRoleArn);
            volumes(model.volumes);
            environment(model.environment);
            mountPoints(model.mountPoints);
            readonlyRootFilesystem(model.readonlyRootFilesystem);
            ulimits(model.ulimits);
            privileged(model.privileged);
            user(model.user);
            exitCode(model.exitCode);
            reason(model.reason);
            containerInstanceArn(model.containerInstanceArn);
            taskArn(model.taskArn);
            logStreamName(model.logStreamName);
            instanceType(model.instanceType);
            networkInterfaces(model.networkInterfaces);
            resourceRequirements(model.resourceRequirements);
            linuxParameters(model.linuxParameters);
            logConfiguration(model.logConfiguration);
            secrets(model.secrets);
            networkConfiguration(model.networkConfiguration);
            fargatePlatformConfiguration(model.fargatePlatformConfiguration);
        }

        public final String getImage() {
            return image;
        }

        public final void setImage(String image) {
            this.image = image;
        }

        @Override
        public final Builder image(String image) {
            this.image = image;
            return this;
        }

        public final Integer getVcpus() {
            return vcpus;
        }

        public final void setVcpus(Integer vcpus) {
            this.vcpus = vcpus;
        }

        @Override
        public final Builder vcpus(Integer vcpus) {
            this.vcpus = vcpus;
            return this;
        }

        public final Integer getMemory() {
            return memory;
        }

        public final void setMemory(Integer memory) {
            this.memory = memory;
        }

        @Override
        public final Builder memory(Integer memory) {
            this.memory = memory;
            return this;
        }

        public final Collection<String> getCommand() {
            if (command instanceof SdkAutoConstructList) {
                return null;
            }
            return command;
        }

        public final void setCommand(Collection<String> command) {
            this.command = StringListCopier.copy(command);
        }

        @Override
        public final Builder command(Collection<String> command) {
            this.command = StringListCopier.copy(command);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder command(String... command) {
            command(Arrays.asList(command));
            return this;
        }

        public final String getJobRoleArn() {
            return jobRoleArn;
        }

        public final void setJobRoleArn(String jobRoleArn) {
            this.jobRoleArn = jobRoleArn;
        }

        @Override
        public final Builder jobRoleArn(String jobRoleArn) {
            this.jobRoleArn = jobRoleArn;
            return this;
        }

        public final String getExecutionRoleArn() {
            return executionRoleArn;
        }

        public final void setExecutionRoleArn(String executionRoleArn) {
            this.executionRoleArn = executionRoleArn;
        }

        @Override
        public final Builder executionRoleArn(String executionRoleArn) {
            this.executionRoleArn = executionRoleArn;
            return this;
        }

        public final List<Volume.Builder> getVolumes() {
            List<Volume.Builder> result = VolumesCopier.copyToBuilder(this.volumes);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setVolumes(Collection<Volume.BuilderImpl> volumes) {
            this.volumes = VolumesCopier.copyFromBuilder(volumes);
        }

        @Override
        public final Builder volumes(Collection<Volume> volumes) {
            this.volumes = VolumesCopier.copy(volumes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder volumes(Volume... volumes) {
            volumes(Arrays.asList(volumes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder volumes(Consumer<Volume.Builder>... volumes) {
            volumes(Stream.of(volumes).map(c -> Volume.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<KeyValuePair.Builder> getEnvironment() {
            List<KeyValuePair.Builder> result = EnvironmentVariablesCopier.copyToBuilder(this.environment);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setEnvironment(Collection<KeyValuePair.BuilderImpl> environment) {
            this.environment = EnvironmentVariablesCopier.copyFromBuilder(environment);
        }

        @Override
        public final Builder environment(Collection<KeyValuePair> environment) {
            this.environment = EnvironmentVariablesCopier.copy(environment);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder environment(KeyValuePair... environment) {
            environment(Arrays.asList(environment));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder environment(Consumer<KeyValuePair.Builder>... environment) {
            environment(Stream.of(environment).map(c -> KeyValuePair.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<MountPoint.Builder> getMountPoints() {
            List<MountPoint.Builder> result = MountPointsCopier.copyToBuilder(this.mountPoints);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setMountPoints(Collection<MountPoint.BuilderImpl> mountPoints) {
            this.mountPoints = MountPointsCopier.copyFromBuilder(mountPoints);
        }

        @Override
        public final Builder mountPoints(Collection<MountPoint> mountPoints) {
            this.mountPoints = MountPointsCopier.copy(mountPoints);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder mountPoints(MountPoint... mountPoints) {
            mountPoints(Arrays.asList(mountPoints));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder mountPoints(Consumer<MountPoint.Builder>... mountPoints) {
            mountPoints(Stream.of(mountPoints).map(c -> MountPoint.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Boolean getReadonlyRootFilesystem() {
            return readonlyRootFilesystem;
        }

        public final void setReadonlyRootFilesystem(Boolean readonlyRootFilesystem) {
            this.readonlyRootFilesystem = readonlyRootFilesystem;
        }

        @Override
        public final Builder readonlyRootFilesystem(Boolean readonlyRootFilesystem) {
            this.readonlyRootFilesystem = readonlyRootFilesystem;
            return this;
        }

        public final List<Ulimit.Builder> getUlimits() {
            List<Ulimit.Builder> result = UlimitsCopier.copyToBuilder(this.ulimits);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setUlimits(Collection<Ulimit.BuilderImpl> ulimits) {
            this.ulimits = UlimitsCopier.copyFromBuilder(ulimits);
        }

        @Override
        public final Builder ulimits(Collection<Ulimit> ulimits) {
            this.ulimits = UlimitsCopier.copy(ulimits);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ulimits(Ulimit... ulimits) {
            ulimits(Arrays.asList(ulimits));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ulimits(Consumer<Ulimit.Builder>... ulimits) {
            ulimits(Stream.of(ulimits).map(c -> Ulimit.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Boolean getPrivileged() {
            return privileged;
        }

        public final void setPrivileged(Boolean privileged) {
            this.privileged = privileged;
        }

        @Override
        public final Builder privileged(Boolean privileged) {
            this.privileged = privileged;
            return this;
        }

        public final String getUser() {
            return user;
        }

        public final void setUser(String user) {
            this.user = user;
        }

        @Override
        public final Builder user(String user) {
            this.user = user;
            return this;
        }

        public final Integer getExitCode() {
            return exitCode;
        }

        public final void setExitCode(Integer exitCode) {
            this.exitCode = exitCode;
        }

        @Override
        public final Builder exitCode(Integer exitCode) {
            this.exitCode = exitCode;
            return this;
        }

        public final String getReason() {
            return reason;
        }

        public final void setReason(String reason) {
            this.reason = reason;
        }

        @Override
        public final Builder reason(String reason) {
            this.reason = reason;
            return this;
        }

        public final String getContainerInstanceArn() {
            return containerInstanceArn;
        }

        public final void setContainerInstanceArn(String containerInstanceArn) {
            this.containerInstanceArn = containerInstanceArn;
        }

        @Override
        public final Builder containerInstanceArn(String containerInstanceArn) {
            this.containerInstanceArn = containerInstanceArn;
            return this;
        }

        public final String getTaskArn() {
            return taskArn;
        }

        public final void setTaskArn(String taskArn) {
            this.taskArn = taskArn;
        }

        @Override
        public final Builder taskArn(String taskArn) {
            this.taskArn = taskArn;
            return this;
        }

        public final String getLogStreamName() {
            return logStreamName;
        }

        public final void setLogStreamName(String logStreamName) {
            this.logStreamName = logStreamName;
        }

        @Override
        public final Builder logStreamName(String logStreamName) {
            this.logStreamName = logStreamName;
            return this;
        }

        public final String getInstanceType() {
            return instanceType;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        public final List<NetworkInterface.Builder> getNetworkInterfaces() {
            List<NetworkInterface.Builder> result = NetworkInterfaceListCopier.copyToBuilder(this.networkInterfaces);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setNetworkInterfaces(Collection<NetworkInterface.BuilderImpl> networkInterfaces) {
            this.networkInterfaces = NetworkInterfaceListCopier.copyFromBuilder(networkInterfaces);
        }

        @Override
        public final Builder networkInterfaces(Collection<NetworkInterface> networkInterfaces) {
            this.networkInterfaces = NetworkInterfaceListCopier.copy(networkInterfaces);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder networkInterfaces(NetworkInterface... networkInterfaces) {
            networkInterfaces(Arrays.asList(networkInterfaces));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder networkInterfaces(Consumer<NetworkInterface.Builder>... networkInterfaces) {
            networkInterfaces(Stream.of(networkInterfaces).map(c -> NetworkInterface.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<ResourceRequirement.Builder> getResourceRequirements() {
            List<ResourceRequirement.Builder> result = ResourceRequirementsCopier.copyToBuilder(this.resourceRequirements);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setResourceRequirements(Collection<ResourceRequirement.BuilderImpl> resourceRequirements) {
            this.resourceRequirements = ResourceRequirementsCopier.copyFromBuilder(resourceRequirements);
        }

        @Override
        public final Builder resourceRequirements(Collection<ResourceRequirement> resourceRequirements) {
            this.resourceRequirements = ResourceRequirementsCopier.copy(resourceRequirements);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceRequirements(ResourceRequirement... resourceRequirements) {
            resourceRequirements(Arrays.asList(resourceRequirements));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceRequirements(Consumer<ResourceRequirement.Builder>... resourceRequirements) {
            resourceRequirements(Stream.of(resourceRequirements).map(c -> ResourceRequirement.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final LinuxParameters.Builder getLinuxParameters() {
            return linuxParameters != null ? linuxParameters.toBuilder() : null;
        }

        public final void setLinuxParameters(LinuxParameters.BuilderImpl linuxParameters) {
            this.linuxParameters = linuxParameters != null ? linuxParameters.build() : null;
        }

        @Override
        public final Builder linuxParameters(LinuxParameters linuxParameters) {
            this.linuxParameters = linuxParameters;
            return this;
        }

        public final LogConfiguration.Builder getLogConfiguration() {
            return logConfiguration != null ? logConfiguration.toBuilder() : null;
        }

        public final void setLogConfiguration(LogConfiguration.BuilderImpl logConfiguration) {
            this.logConfiguration = logConfiguration != null ? logConfiguration.build() : null;
        }

        @Override
        public final Builder logConfiguration(LogConfiguration logConfiguration) {
            this.logConfiguration = logConfiguration;
            return this;
        }

        public final List<Secret.Builder> getSecrets() {
            List<Secret.Builder> result = SecretListCopier.copyToBuilder(this.secrets);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSecrets(Collection<Secret.BuilderImpl> secrets) {
            this.secrets = SecretListCopier.copyFromBuilder(secrets);
        }

        @Override
        public final Builder secrets(Collection<Secret> secrets) {
            this.secrets = SecretListCopier.copy(secrets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder secrets(Secret... secrets) {
            secrets(Arrays.asList(secrets));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder secrets(Consumer<Secret.Builder>... secrets) {
            secrets(Stream.of(secrets).map(c -> Secret.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final NetworkConfiguration.Builder getNetworkConfiguration() {
            return networkConfiguration != null ? networkConfiguration.toBuilder() : null;
        }

        public final void setNetworkConfiguration(NetworkConfiguration.BuilderImpl networkConfiguration) {
            this.networkConfiguration = networkConfiguration != null ? networkConfiguration.build() : null;
        }

        @Override
        public final Builder networkConfiguration(NetworkConfiguration networkConfiguration) {
            this.networkConfiguration = networkConfiguration;
            return this;
        }

        public final FargatePlatformConfiguration.Builder getFargatePlatformConfiguration() {
            return fargatePlatformConfiguration != null ? fargatePlatformConfiguration.toBuilder() : null;
        }

        public final void setFargatePlatformConfiguration(FargatePlatformConfiguration.BuilderImpl fargatePlatformConfiguration) {
            this.fargatePlatformConfiguration = fargatePlatformConfiguration != null ? fargatePlatformConfiguration.build()
                    : null;
        }

        @Override
        public final Builder fargatePlatformConfiguration(FargatePlatformConfiguration fargatePlatformConfiguration) {
            this.fargatePlatformConfiguration = fargatePlatformConfiguration;
            return this;
        }

        @Override
        public ContainerDetail build() {
            return new ContainerDetail(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
