/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.batch;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.batch.model.BatchException;
import software.amazon.awssdk.services.batch.model.BatchRequest;
import software.amazon.awssdk.services.batch.model.CancelJobRequest;
import software.amazon.awssdk.services.batch.model.CancelJobResponse;
import software.amazon.awssdk.services.batch.model.ClientException;
import software.amazon.awssdk.services.batch.model.CreateComputeEnvironmentRequest;
import software.amazon.awssdk.services.batch.model.CreateComputeEnvironmentResponse;
import software.amazon.awssdk.services.batch.model.CreateJobQueueRequest;
import software.amazon.awssdk.services.batch.model.CreateJobQueueResponse;
import software.amazon.awssdk.services.batch.model.CreateSchedulingPolicyRequest;
import software.amazon.awssdk.services.batch.model.CreateSchedulingPolicyResponse;
import software.amazon.awssdk.services.batch.model.DeleteComputeEnvironmentRequest;
import software.amazon.awssdk.services.batch.model.DeleteComputeEnvironmentResponse;
import software.amazon.awssdk.services.batch.model.DeleteJobQueueRequest;
import software.amazon.awssdk.services.batch.model.DeleteJobQueueResponse;
import software.amazon.awssdk.services.batch.model.DeleteSchedulingPolicyRequest;
import software.amazon.awssdk.services.batch.model.DeleteSchedulingPolicyResponse;
import software.amazon.awssdk.services.batch.model.DeregisterJobDefinitionRequest;
import software.amazon.awssdk.services.batch.model.DeregisterJobDefinitionResponse;
import software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsRequest;
import software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsResponse;
import software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsRequest;
import software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsResponse;
import software.amazon.awssdk.services.batch.model.DescribeJobQueuesRequest;
import software.amazon.awssdk.services.batch.model.DescribeJobQueuesResponse;
import software.amazon.awssdk.services.batch.model.DescribeJobsRequest;
import software.amazon.awssdk.services.batch.model.DescribeJobsResponse;
import software.amazon.awssdk.services.batch.model.DescribeSchedulingPoliciesRequest;
import software.amazon.awssdk.services.batch.model.DescribeSchedulingPoliciesResponse;
import software.amazon.awssdk.services.batch.model.ListJobsRequest;
import software.amazon.awssdk.services.batch.model.ListJobsResponse;
import software.amazon.awssdk.services.batch.model.ListSchedulingPoliciesRequest;
import software.amazon.awssdk.services.batch.model.ListSchedulingPoliciesResponse;
import software.amazon.awssdk.services.batch.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.batch.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.batch.model.RegisterJobDefinitionRequest;
import software.amazon.awssdk.services.batch.model.RegisterJobDefinitionResponse;
import software.amazon.awssdk.services.batch.model.ServerException;
import software.amazon.awssdk.services.batch.model.SubmitJobRequest;
import software.amazon.awssdk.services.batch.model.SubmitJobResponse;
import software.amazon.awssdk.services.batch.model.TagResourceRequest;
import software.amazon.awssdk.services.batch.model.TagResourceResponse;
import software.amazon.awssdk.services.batch.model.TerminateJobRequest;
import software.amazon.awssdk.services.batch.model.TerminateJobResponse;
import software.amazon.awssdk.services.batch.model.UntagResourceRequest;
import software.amazon.awssdk.services.batch.model.UntagResourceResponse;
import software.amazon.awssdk.services.batch.model.UpdateComputeEnvironmentRequest;
import software.amazon.awssdk.services.batch.model.UpdateComputeEnvironmentResponse;
import software.amazon.awssdk.services.batch.model.UpdateJobQueueRequest;
import software.amazon.awssdk.services.batch.model.UpdateJobQueueResponse;
import software.amazon.awssdk.services.batch.model.UpdateSchedulingPolicyRequest;
import software.amazon.awssdk.services.batch.model.UpdateSchedulingPolicyResponse;
import software.amazon.awssdk.services.batch.paginators.DescribeComputeEnvironmentsPublisher;
import software.amazon.awssdk.services.batch.paginators.DescribeJobDefinitionsPublisher;
import software.amazon.awssdk.services.batch.paginators.DescribeJobQueuesPublisher;
import software.amazon.awssdk.services.batch.paginators.ListJobsPublisher;
import software.amazon.awssdk.services.batch.paginators.ListSchedulingPoliciesPublisher;
import software.amazon.awssdk.services.batch.transform.CancelJobRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.CreateComputeEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.CreateJobQueueRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.CreateSchedulingPolicyRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.DeleteComputeEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.DeleteJobQueueRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.DeleteSchedulingPolicyRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.DeregisterJobDefinitionRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.DescribeComputeEnvironmentsRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.DescribeJobDefinitionsRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.DescribeJobQueuesRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.DescribeJobsRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.DescribeSchedulingPoliciesRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.ListJobsRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.ListSchedulingPoliciesRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.RegisterJobDefinitionRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.SubmitJobRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.TerminateJobRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.UpdateComputeEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.UpdateJobQueueRequestMarshaller;
import software.amazon.awssdk.services.batch.transform.UpdateSchedulingPolicyRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link BatchAsyncClient}.
 *
 * @see BatchAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultBatchAsyncClient implements BatchAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultBatchAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultBatchAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Cancels a job in an Batch job queue. Jobs that are in the <code>SUBMITTED</code>, <code>PENDING</code>, or
     * <code>RUNNABLE</code> state are canceled. Jobs that progressed to the <code>STARTING</code> or
     * <code>RUNNING</code> state aren't canceled. However, the API operation still succeeds, even if no job is
     * canceled. These jobs must be terminated with the <a>TerminateJob</a> operation.
     * </p>
     *
     * @param cancelJobRequest
     *        Contains the parameters for <code>CancelJob</code>.
     * @return A Java Future containing the result of the CancelJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action. One example cause is using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource.
     *         Another cause is specifying an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.CancelJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/CancelJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CancelJobResponse> cancelJob(CancelJobRequest cancelJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CancelJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CancelJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelJobRequest, CancelJobResponse>().withOperationName("CancelJob")
                            .withMarshaller(new CancelJobRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(cancelJobRequest));
            CompletableFuture<CancelJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Batch compute environment. You can create <code>MANAGED</code> or <code>UNMANAGED</code> compute
     * environments. <code>MANAGED</code> compute environments can use Amazon EC2 or Fargate resources.
     * <code>UNMANAGED</code> compute environments can only use EC2 resources.
     * </p>
     * <p>
     * In a managed compute environment, Batch manages the capacity and instance types of the compute resources within
     * the environment. This is based on the compute resource specification that you define or the <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html">launch template</a> that you
     * specify when you create the compute environment. Either, you can choose to use EC2 On-Demand Instances and EC2
     * Spot Instances. Or, you can use Fargate and Fargate Spot capacity in your managed compute environment. You can
     * optionally set a maximum price so that Spot Instances only launch when the Spot Instance price is less than a
     * specified percentage of the On-Demand price.
     * </p>
     * <note>
     * <p>
     * Multi-node parallel jobs aren't supported on Spot Instances.
     * </p>
     * </note>
     * <p>
     * In an unmanaged compute environment, you can manage your own EC2 compute resources and have flexibility with how
     * you configure your compute resources. For example, you can use custom AMIs. However, you must verify that each of
     * your AMIs meet the Amazon ECS container instance AMI specification. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container_instance_AMIs.html">container
     * instance AMIs</a> in the <i>Amazon Elastic Container Service Developer Guide</i>. After you created your
     * unmanaged compute environment, you can use the <a>DescribeComputeEnvironments</a> operation to find the Amazon
     * ECS cluster that's associated with it. Then, launch your container instances into that Amazon ECS cluster. For
     * more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_container_instance.html">Launching an
     * Amazon ECS container instance</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <note>
     * <p>
     * To create a compute environment that uses EKS resources, the caller must have permissions to call
     * <code>eks:DescribeCluster</code>.
     * </p>
     * </note> <note>
     * <p>
     * Batch doesn't automatically upgrade the AMIs in a compute environment after it's created. For example, it also
     * doesn't update the AMIs in your compute environment when a newer version of the Amazon ECS optimized AMI is
     * available. You're responsible for the management of the guest operating system. This includes any updates and
     * security patches. You're also responsible for any additional application software or utilities that you install
     * on the compute resources. There are two ways to use a new AMI for your Batch jobs. The original method is to
     * complete these steps:
     * </p>
     * <ol>
     * <li>
     * <p>
     * Create a new compute environment with the new AMI.
     * </p>
     * </li>
     * <li>
     * <p>
     * Add the compute environment to an existing job queue.
     * </p>
     * </li>
     * <li>
     * <p>
     * Remove the earlier compute environment from your job queue.
     * </p>
     * </li>
     * <li>
     * <p>
     * Delete the earlier compute environment.
     * </p>
     * </li>
     * </ol>
     * <p>
     * In April 2022, Batch added enhanced support for updating compute environments. For more information, see <a
     * href="https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html">Updating compute
     * environments</a>. To use the enhanced updating of compute environments to update AMIs, follow these rules:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Either don't set the service role (<code>serviceRole</code>) parameter or set it to the
     * <b>AWSBatchServiceRole</b> service-linked role.
     * </p>
     * </li>
     * <li>
     * <p>
     * Set the allocation strategy (<code>allocationStrategy</code>) parameter to <code>BEST_FIT_PROGRESSIVE</code> or
     * <code>SPOT_CAPACITY_OPTIMIZED</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Set the update to latest image version (<code>updateToLatestImageVersion</code>) parameter to <code>true</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Don't specify an AMI ID in <code>imageId</code>, <code>imageIdOverride</code> (in <a
     * href="https://docs.aws.amazon.com/batch/latest/APIReference/API_Ec2Configuration.html">
     * <code>ec2Configuration</code> </a>), or in the launch template (<code>launchTemplate</code>). In that case, Batch
     * selects the latest Amazon ECS optimized AMI that's supported by Batch at the time the infrastructure update is
     * initiated. Alternatively, you can specify the AMI ID in the <code>imageId</code> or <code>imageIdOverride</code>
     * parameters, or the launch template identified by the <code>LaunchTemplate</code> properties. Changing any of
     * these properties starts an infrastructure update. If the AMI ID is specified in the launch template, it can't be
     * replaced by specifying an AMI ID in either the <code>imageId</code> or <code>imageIdOverride</code> parameters.
     * It can only be replaced by specifying a different launch template, or if the launch template version is set to
     * <code>&#36Default</code> or <code>&#36Latest</code>, by setting either a new default version for the launch
     * template (if <code>&#36Default</code>) or by adding a new version to the launch template (if
     * <code>&#36Latest</code>).
     * </p>
     * </li>
     * </ul>
     * <p>
     * If these rules are followed, any update that starts an infrastructure update causes the AMI ID to be re-selected.
     * If the <code>version</code> setting in the launch template (<code>launchTemplate</code>) is set to
     * <code>&#36Latest</code> or <code>&#36Default</code>, the latest or default version of the launch template is
     * evaluated up at the time of the infrastructure update, even if the <code>launchTemplate</code> wasn't updated.
     * </p>
     * </note>
     *
     * @param createComputeEnvironmentRequest
     *        Contains the parameters for <code>CreateComputeEnvironment</code>.
     * @return A Java Future containing the result of the CreateComputeEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action. One example cause is using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource.
     *         Another cause is specifying an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.CreateComputeEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/CreateComputeEnvironment"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateComputeEnvironmentResponse> createComputeEnvironment(
            CreateComputeEnvironmentRequest createComputeEnvironmentRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createComputeEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateComputeEnvironment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateComputeEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateComputeEnvironmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateComputeEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateComputeEnvironmentRequest, CreateComputeEnvironmentResponse>()
                            .withOperationName("CreateComputeEnvironment")
                            .withMarshaller(new CreateComputeEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createComputeEnvironmentRequest));
            CompletableFuture<CreateComputeEnvironmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Batch job queue. When you create a job queue, you associate one or more compute environments to the
     * queue and assign an order of preference for the compute environments.
     * </p>
     * <p>
     * You also set a priority to the job queue that determines the order that the Batch scheduler places jobs onto its
     * associated compute environments. For example, if a compute environment is associated with more than one job
     * queue, the job queue with a higher priority is given preference for scheduling jobs to that compute environment.
     * </p>
     *
     * @param createJobQueueRequest
     *        Contains the parameters for <code>CreateJobQueue</code>.
     * @return A Java Future containing the result of the CreateJobQueue operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action. One example cause is using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource.
     *         Another cause is specifying an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.CreateJobQueue
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/CreateJobQueue" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateJobQueueResponse> createJobQueue(CreateJobQueueRequest createJobQueueRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createJobQueueRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateJobQueue");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateJobQueueResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateJobQueueResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateJobQueueResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateJobQueueRequest, CreateJobQueueResponse>()
                            .withOperationName("CreateJobQueue")
                            .withMarshaller(new CreateJobQueueRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createJobQueueRequest));
            CompletableFuture<CreateJobQueueResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Batch scheduling policy.
     * </p>
     *
     * @param createSchedulingPolicyRequest
     *        Contains the parameters for <code>CreateSchedulingPolicy</code>.
     * @return A Java Future containing the result of the CreateSchedulingPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action. One example cause is using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource.
     *         Another cause is specifying an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.CreateSchedulingPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/CreateSchedulingPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSchedulingPolicyResponse> createSchedulingPolicy(
            CreateSchedulingPolicyRequest createSchedulingPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSchedulingPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSchedulingPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSchedulingPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateSchedulingPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateSchedulingPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSchedulingPolicyRequest, CreateSchedulingPolicyResponse>()
                            .withOperationName("CreateSchedulingPolicy")
                            .withMarshaller(new CreateSchedulingPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createSchedulingPolicyRequest));
            CompletableFuture<CreateSchedulingPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Batch compute environment.
     * </p>
     * <p>
     * Before you can delete a compute environment, you must set its state to <code>DISABLED</code> with the
     * <a>UpdateComputeEnvironment</a> API operation and disassociate it from any job queues with the
     * <a>UpdateJobQueue</a> API operation. Compute environments that use Fargate resources must terminate all active
     * jobs on that compute environment before deleting the compute environment. If this isn't done, the compute
     * environment enters an invalid state.
     * </p>
     *
     * @param deleteComputeEnvironmentRequest
     *        Contains the parameters for <code>DeleteComputeEnvironment</code>.
     * @return A Java Future containing the result of the DeleteComputeEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action. One example cause is using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource.
     *         Another cause is specifying an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DeleteComputeEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DeleteComputeEnvironment"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteComputeEnvironmentResponse> deleteComputeEnvironment(
            DeleteComputeEnvironmentRequest deleteComputeEnvironmentRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteComputeEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteComputeEnvironment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteComputeEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteComputeEnvironmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteComputeEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteComputeEnvironmentRequest, DeleteComputeEnvironmentResponse>()
                            .withOperationName("DeleteComputeEnvironment")
                            .withMarshaller(new DeleteComputeEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteComputeEnvironmentRequest));
            CompletableFuture<DeleteComputeEnvironmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified job queue. You must first disable submissions for a queue with the <a>UpdateJobQueue</a>
     * operation. All jobs in the queue are eventually terminated when you delete a job queue. The jobs are terminated
     * at a rate of about 16 jobs each second.
     * </p>
     * <p>
     * It's not necessary to disassociate compute environments from a queue before submitting a
     * <code>DeleteJobQueue</code> request.
     * </p>
     *
     * @param deleteJobQueueRequest
     *        Contains the parameters for <code>DeleteJobQueue</code>.
     * @return A Java Future containing the result of the DeleteJobQueue operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action. One example cause is using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource.
     *         Another cause is specifying an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DeleteJobQueue
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DeleteJobQueue" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteJobQueueResponse> deleteJobQueue(DeleteJobQueueRequest deleteJobQueueRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteJobQueueRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteJobQueue");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteJobQueueResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteJobQueueResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteJobQueueResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteJobQueueRequest, DeleteJobQueueResponse>()
                            .withOperationName("DeleteJobQueue")
                            .withMarshaller(new DeleteJobQueueRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteJobQueueRequest));
            CompletableFuture<DeleteJobQueueResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified scheduling policy.
     * </p>
     * <p>
     * You can't delete a scheduling policy that's used in any job queues.
     * </p>
     *
     * @param deleteSchedulingPolicyRequest
     *        Contains the parameters for <code>DeleteSchedulingPolicy</code>.
     * @return A Java Future containing the result of the DeleteSchedulingPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action. One example cause is using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource.
     *         Another cause is specifying an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DeleteSchedulingPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DeleteSchedulingPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSchedulingPolicyResponse> deleteSchedulingPolicy(
            DeleteSchedulingPolicyRequest deleteSchedulingPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSchedulingPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSchedulingPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSchedulingPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteSchedulingPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSchedulingPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSchedulingPolicyRequest, DeleteSchedulingPolicyResponse>()
                            .withOperationName("DeleteSchedulingPolicy")
                            .withMarshaller(new DeleteSchedulingPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteSchedulingPolicyRequest));
            CompletableFuture<DeleteSchedulingPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deregisters an Batch job definition. Job definitions are permanently deleted after 180 days.
     * </p>
     *
     * @param deregisterJobDefinitionRequest
     * @return A Java Future containing the result of the DeregisterJobDefinition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action. One example cause is using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource.
     *         Another cause is specifying an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DeregisterJobDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DeregisterJobDefinition" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeregisterJobDefinitionResponse> deregisterJobDefinition(
            DeregisterJobDefinitionRequest deregisterJobDefinitionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deregisterJobDefinitionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeregisterJobDefinition");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeregisterJobDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeregisterJobDefinitionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeregisterJobDefinitionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeregisterJobDefinitionRequest, DeregisterJobDefinitionResponse>()
                            .withOperationName("DeregisterJobDefinition")
                            .withMarshaller(new DeregisterJobDefinitionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deregisterJobDefinitionRequest));
            CompletableFuture<DeregisterJobDefinitionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes one or more of your compute environments.
     * </p>
     * <p>
     * If you're using an unmanaged compute environment, you can use the <code>DescribeComputeEnvironment</code>
     * operation to determine the <code>ecsClusterArn</code> that you launch your Amazon ECS container instances into.
     * </p>
     *
     * @param describeComputeEnvironmentsRequest
     *        Contains the parameters for <code>DescribeComputeEnvironments</code>.
     * @return A Java Future containing the result of the DescribeComputeEnvironments operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action. One example cause is using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource.
     *         Another cause is specifying an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeComputeEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeComputeEnvironments"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeComputeEnvironmentsResponse> describeComputeEnvironments(
            DescribeComputeEnvironmentsRequest describeComputeEnvironmentsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeComputeEnvironmentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeComputeEnvironments");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeComputeEnvironmentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeComputeEnvironmentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeComputeEnvironmentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeComputeEnvironmentsRequest, DescribeComputeEnvironmentsResponse>()
                            .withOperationName("DescribeComputeEnvironments")
                            .withMarshaller(new DescribeComputeEnvironmentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeComputeEnvironmentsRequest));
            CompletableFuture<DescribeComputeEnvironmentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes one or more of your compute environments.
     * </p>
     * <p>
     * If you're using an unmanaged compute environment, you can use the <code>DescribeComputeEnvironment</code>
     * operation to determine the <code>ecsClusterArn</code> that you launch your Amazon ECS container instances into.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeComputeEnvironments(software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeComputeEnvironmentsPublisher publisher = client.describeComputeEnvironmentsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeComputeEnvironmentsPublisher publisher = client.describeComputeEnvironmentsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeComputeEnvironments(software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeComputeEnvironmentsRequest
     *        Contains the parameters for <code>DescribeComputeEnvironments</code>.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action. One example cause is using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource.
     *         Another cause is specifying an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeComputeEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeComputeEnvironments"
     *      target="_top">AWS API Documentation</a>
     */
    public DescribeComputeEnvironmentsPublisher describeComputeEnvironmentsPaginator(
            DescribeComputeEnvironmentsRequest describeComputeEnvironmentsRequest) {
        return new DescribeComputeEnvironmentsPublisher(this, applyPaginatorUserAgent(describeComputeEnvironmentsRequest));
    }

    /**
     * <p>
     * Describes a list of job definitions. You can specify a <code>status</code> (such as <code>ACTIVE</code>) to only
     * return job definitions that match that status.
     * </p>
     *
     * @param describeJobDefinitionsRequest
     *        Contains the parameters for <code>DescribeJobDefinitions</code>.
     * @return A Java Future containing the result of the DescribeJobDefinitions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action. One example cause is using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource.
     *         Another cause is specifying an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeJobDefinitions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobDefinitions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeJobDefinitionsResponse> describeJobDefinitions(
            DescribeJobDefinitionsRequest describeJobDefinitionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeJobDefinitionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeJobDefinitions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeJobDefinitionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeJobDefinitionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeJobDefinitionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeJobDefinitionsRequest, DescribeJobDefinitionsResponse>()
                            .withOperationName("DescribeJobDefinitions")
                            .withMarshaller(new DescribeJobDefinitionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeJobDefinitionsRequest));
            CompletableFuture<DescribeJobDefinitionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes a list of job definitions. You can specify a <code>status</code> (such as <code>ACTIVE</code>) to only
     * return job definitions that match that status.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeJobDefinitions(software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeJobDefinitionsPublisher publisher = client.describeJobDefinitionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeJobDefinitionsPublisher publisher = client.describeJobDefinitionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeJobDefinitions(software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeJobDefinitionsRequest
     *        Contains the parameters for <code>DescribeJobDefinitions</code>.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action. One example cause is using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource.
     *         Another cause is specifying an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeJobDefinitions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobDefinitions" target="_top">AWS
     *      API Documentation</a>
     */
    public DescribeJobDefinitionsPublisher describeJobDefinitionsPaginator(
            DescribeJobDefinitionsRequest describeJobDefinitionsRequest) {
        return new DescribeJobDefinitionsPublisher(this, applyPaginatorUserAgent(describeJobDefinitionsRequest));
    }

    /**
     * <p>
     * Describes one or more of your job queues.
     * </p>
     *
     * @param describeJobQueuesRequest
     *        Contains the parameters for <code>DescribeJobQueues</code>.
     * @return A Java Future containing the result of the DescribeJobQueues operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action. One example cause is using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource.
     *         Another cause is specifying an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeJobQueues
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobQueues" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeJobQueuesResponse> describeJobQueues(DescribeJobQueuesRequest describeJobQueuesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeJobQueuesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeJobQueues");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeJobQueuesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeJobQueuesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeJobQueuesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeJobQueuesRequest, DescribeJobQueuesResponse>()
                            .withOperationName("DescribeJobQueues")
                            .withMarshaller(new DescribeJobQueuesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeJobQueuesRequest));
            CompletableFuture<DescribeJobQueuesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes one or more of your job queues.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeJobQueues(software.amazon.awssdk.services.batch.model.DescribeJobQueuesRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeJobQueuesPublisher publisher = client.describeJobQueuesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeJobQueuesPublisher publisher = client.describeJobQueuesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.batch.model.DescribeJobQueuesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.batch.model.DescribeJobQueuesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeJobQueues(software.amazon.awssdk.services.batch.model.DescribeJobQueuesRequest)} operation.</b>
     * </p>
     *
     * @param describeJobQueuesRequest
     *        Contains the parameters for <code>DescribeJobQueues</code>.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action. One example cause is using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource.
     *         Another cause is specifying an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeJobQueues
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobQueues" target="_top">AWS API
     *      Documentation</a>
     */
    public DescribeJobQueuesPublisher describeJobQueuesPaginator(DescribeJobQueuesRequest describeJobQueuesRequest) {
        return new DescribeJobQueuesPublisher(this, applyPaginatorUserAgent(describeJobQueuesRequest));
    }

    /**
     * <p>
     * Describes a list of Batch jobs.
     * </p>
     *
     * @param describeJobsRequest
     *        Contains the parameters for <code>DescribeJobs</code>.
     * @return A Java Future containing the result of the DescribeJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action. One example cause is using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource.
     *         Another cause is specifying an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeJobsResponse> describeJobs(DescribeJobsRequest describeJobsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeJobsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeJobsRequest, DescribeJobsResponse>()
                            .withOperationName("DescribeJobs").withMarshaller(new DescribeJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeJobsRequest));
            CompletableFuture<DescribeJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes one or more of your scheduling policies.
     * </p>
     *
     * @param describeSchedulingPoliciesRequest
     *        Contains the parameters for <code>DescribeSchedulingPolicies</code>.
     * @return A Java Future containing the result of the DescribeSchedulingPolicies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action. One example cause is using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource.
     *         Another cause is specifying an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.DescribeSchedulingPolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeSchedulingPolicies"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeSchedulingPoliciesResponse> describeSchedulingPolicies(
            DescribeSchedulingPoliciesRequest describeSchedulingPoliciesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeSchedulingPoliciesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSchedulingPolicies");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeSchedulingPoliciesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeSchedulingPoliciesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeSchedulingPoliciesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeSchedulingPoliciesRequest, DescribeSchedulingPoliciesResponse>()
                            .withOperationName("DescribeSchedulingPolicies")
                            .withMarshaller(new DescribeSchedulingPoliciesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeSchedulingPoliciesRequest));
            CompletableFuture<DescribeSchedulingPoliciesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of Batch jobs.
     * </p>
     * <p>
     * You must specify only one of the following items:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A job queue ID to return a list of jobs in that job queue
     * </p>
     * </li>
     * <li>
     * <p>
     * A multi-node parallel job ID to return a list of nodes for that job
     * </p>
     * </li>
     * <li>
     * <p>
     * An array job ID to return a list of the children for that job
     * </p>
     * </li>
     * </ul>
     * <p>
     * You can filter the results by job status with the <code>jobStatus</code> parameter. If you don't specify a
     * status, only <code>RUNNING</code> jobs are returned.
     * </p>
     *
     * @param listJobsRequest
     *        Contains the parameters for <code>ListJobs</code>.
     * @return A Java Future containing the result of the ListJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action. One example cause is using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource.
     *         Another cause is specifying an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.ListJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListJobsResponse> listJobs(ListJobsRequest listJobsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListJobsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListJobsRequest, ListJobsResponse>().withOperationName("ListJobs")
                            .withMarshaller(new ListJobsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(listJobsRequest));
            CompletableFuture<ListJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of Batch jobs.
     * </p>
     * <p>
     * You must specify only one of the following items:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A job queue ID to return a list of jobs in that job queue
     * </p>
     * </li>
     * <li>
     * <p>
     * A multi-node parallel job ID to return a list of nodes for that job
     * </p>
     * </li>
     * <li>
     * <p>
     * An array job ID to return a list of the children for that job
     * </p>
     * </li>
     * </ul>
     * <p>
     * You can filter the results by job status with the <code>jobStatus</code> parameter. If you don't specify a
     * status, only <code>RUNNING</code> jobs are returned.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listJobs(software.amazon.awssdk.services.batch.model.ListJobsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.ListJobsPublisher publisher = client.listJobsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.ListJobsPublisher publisher = client.listJobsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.batch.model.ListJobsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.batch.model.ListJobsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listJobs(software.amazon.awssdk.services.batch.model.ListJobsRequest)} operation.</b>
     * </p>
     *
     * @param listJobsRequest
     *        Contains the parameters for <code>ListJobs</code>.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action. One example cause is using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource.
     *         Another cause is specifying an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.ListJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    public ListJobsPublisher listJobsPaginator(ListJobsRequest listJobsRequest) {
        return new ListJobsPublisher(this, applyPaginatorUserAgent(listJobsRequest));
    }

    /**
     * <p>
     * Returns a list of Batch scheduling policies.
     * </p>
     *
     * @param listSchedulingPoliciesRequest
     *        Contains the parameters for <code>ListSchedulingPolicies</code>.
     * @return A Java Future containing the result of the ListSchedulingPolicies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action. One example cause is using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource.
     *         Another cause is specifying an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.ListSchedulingPolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/ListSchedulingPolicies" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSchedulingPoliciesResponse> listSchedulingPolicies(
            ListSchedulingPoliciesRequest listSchedulingPoliciesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSchedulingPoliciesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSchedulingPolicies");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSchedulingPoliciesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSchedulingPoliciesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSchedulingPoliciesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSchedulingPoliciesRequest, ListSchedulingPoliciesResponse>()
                            .withOperationName("ListSchedulingPolicies")
                            .withMarshaller(new ListSchedulingPoliciesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listSchedulingPoliciesRequest));
            CompletableFuture<ListSchedulingPoliciesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of Batch scheduling policies.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSchedulingPolicies(software.amazon.awssdk.services.batch.model.ListSchedulingPoliciesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.ListSchedulingPoliciesPublisher publisher = client.listSchedulingPoliciesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.ListSchedulingPoliciesPublisher publisher = client.listSchedulingPoliciesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.batch.model.ListSchedulingPoliciesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.batch.model.ListSchedulingPoliciesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSchedulingPolicies(software.amazon.awssdk.services.batch.model.ListSchedulingPoliciesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSchedulingPoliciesRequest
     *        Contains the parameters for <code>ListSchedulingPolicies</code>.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action. One example cause is using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource.
     *         Another cause is specifying an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.ListSchedulingPolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/ListSchedulingPolicies" target="_top">AWS
     *      API Documentation</a>
     */
    public ListSchedulingPoliciesPublisher listSchedulingPoliciesPaginator(
            ListSchedulingPoliciesRequest listSchedulingPoliciesRequest) {
        return new ListSchedulingPoliciesPublisher(this, applyPaginatorUserAgent(listSchedulingPoliciesRequest));
    }

    /**
     * <p>
     * Lists the tags for an Batch resource. Batch resources that support tags are compute environments, jobs, job
     * definitions, job queues, and scheduling policies. ARNs for child jobs of array and multi-node parallel (MNP) jobs
     * aren't supported.
     * </p>
     *
     * @param listTagsForResourceRequest
     *        Contains the parameters for <code>ListTagsForResource</code>.
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action. One example cause is using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource.
     *         Another cause is specifying an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Registers an Batch job definition.
     * </p>
     *
     * @param registerJobDefinitionRequest
     *        Contains the parameters for <code>RegisterJobDefinition</code>.
     * @return A Java Future containing the result of the RegisterJobDefinition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action. One example cause is using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource.
     *         Another cause is specifying an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.RegisterJobDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/RegisterJobDefinition" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<RegisterJobDefinitionResponse> registerJobDefinition(
            RegisterJobDefinitionRequest registerJobDefinitionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, registerJobDefinitionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterJobDefinition");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RegisterJobDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RegisterJobDefinitionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RegisterJobDefinitionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RegisterJobDefinitionRequest, RegisterJobDefinitionResponse>()
                            .withOperationName("RegisterJobDefinition")
                            .withMarshaller(new RegisterJobDefinitionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(registerJobDefinitionRequest));
            CompletableFuture<RegisterJobDefinitionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Submits an Batch job from a job definition. Parameters that are specified during <a>SubmitJob</a> override
     * parameters defined in the job definition. vCPU and memory requirements that are specified in the
     * <code>resourceRequirements</code> objects in the job definition are the exception. They can't be overridden this
     * way using the <code>memory</code> and <code>vcpus</code> parameters. Rather, you must specify updates to job
     * definition parameters in a <code>resourceRequirements</code> object that's included in the
     * <code>containerOverrides</code> parameter.
     * </p>
     * <note>
     * <p>
     * Job queues with a scheduling policy are limited to 500 active fair share identifiers at a time.
     * </p>
     * </note> <important>
     * <p>
     * Jobs that run on Fargate resources can't be guaranteed to run for more than 14 days. This is because, after 14
     * days, Fargate resources might become unavailable and job might be terminated.
     * </p>
     * </important>
     *
     * @param submitJobRequest
     *        Contains the parameters for <code>SubmitJob</code>.
     * @return A Java Future containing the result of the SubmitJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action. One example cause is using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource.
     *         Another cause is specifying an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.SubmitJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/SubmitJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<SubmitJobResponse> submitJob(SubmitJobRequest submitJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, submitJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SubmitJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SubmitJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    SubmitJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SubmitJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SubmitJobRequest, SubmitJobResponse>().withOperationName("SubmitJob")
                            .withMarshaller(new SubmitJobRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(submitJobRequest));
            CompletableFuture<SubmitJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associates the specified tags to a resource with the specified <code>resourceArn</code>. If existing tags on a
     * resource aren't specified in the request parameters, they aren't changed. When a resource is deleted, the tags
     * that are associated with that resource are deleted as well. Batch resources that support tags are compute
     * environments, jobs, job definitions, job queues, and scheduling policies. ARNs for child jobs of array and
     * multi-node parallel (MNP) jobs aren't supported.
     * </p>
     *
     * @param tagResourceRequest
     *        Contains the parameters for <code>TagResource</code>.
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action. One example cause is using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource.
     *         Another cause is specifying an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Terminates a job in a job queue. Jobs that are in the <code>STARTING</code> or <code>RUNNING</code> state are
     * terminated, which causes them to transition to <code>FAILED</code>. Jobs that have not progressed to the
     * <code>STARTING</code> state are cancelled.
     * </p>
     *
     * @param terminateJobRequest
     *        Contains the parameters for <code>TerminateJob</code>.
     * @return A Java Future containing the result of the TerminateJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action. One example cause is using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource.
     *         Another cause is specifying an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.TerminateJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/TerminateJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TerminateJobResponse> terminateJob(TerminateJobRequest terminateJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, terminateJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TerminateJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TerminateJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TerminateJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TerminateJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TerminateJobRequest, TerminateJobResponse>()
                            .withOperationName("TerminateJob").withMarshaller(new TerminateJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(terminateJobRequest));
            CompletableFuture<TerminateJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes specified tags from an Batch resource.
     * </p>
     *
     * @param untagResourceRequest
     *        Contains the parameters for <code>UntagResource</code>.
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action. One example cause is using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource.
     *         Another cause is specifying an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an Batch compute environment.
     * </p>
     *
     * @param updateComputeEnvironmentRequest
     *        Contains the parameters for <code>UpdateComputeEnvironment</code>.
     * @return A Java Future containing the result of the UpdateComputeEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action. One example cause is using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource.
     *         Another cause is specifying an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.UpdateComputeEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/UpdateComputeEnvironment"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateComputeEnvironmentResponse> updateComputeEnvironment(
            UpdateComputeEnvironmentRequest updateComputeEnvironmentRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateComputeEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateComputeEnvironment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateComputeEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateComputeEnvironmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateComputeEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateComputeEnvironmentRequest, UpdateComputeEnvironmentResponse>()
                            .withOperationName("UpdateComputeEnvironment")
                            .withMarshaller(new UpdateComputeEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateComputeEnvironmentRequest));
            CompletableFuture<UpdateComputeEnvironmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a job queue.
     * </p>
     *
     * @param updateJobQueueRequest
     *        Contains the parameters for <code>UpdateJobQueue</code>.
     * @return A Java Future containing the result of the UpdateJobQueue operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action. One example cause is using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource.
     *         Another cause is specifying an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.UpdateJobQueue
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/UpdateJobQueue" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateJobQueueResponse> updateJobQueue(UpdateJobQueueRequest updateJobQueueRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateJobQueueRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateJobQueue");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateJobQueueResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateJobQueueResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateJobQueueResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateJobQueueRequest, UpdateJobQueueResponse>()
                            .withOperationName("UpdateJobQueue")
                            .withMarshaller(new UpdateJobQueueRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateJobQueueRequest));
            CompletableFuture<UpdateJobQueueResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a scheduling policy.
     * </p>
     *
     * @param updateSchedulingPolicyRequest
     *        Contains the parameters for <code>UpdateSchedulingPolicy</code>.
     * @return A Java Future containing the result of the UpdateSchedulingPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClientException These errors are usually caused by a client action. One example cause is using an
     *         action or resource on behalf of a user that doesn't have permissions to use the action or resource.
     *         Another cause is specifying an identifier that's not valid.</li>
     *         <li>ServerException These errors are usually caused by a server issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BatchException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BatchAsyncClient.UpdateSchedulingPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/UpdateSchedulingPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSchedulingPolicyResponse> updateSchedulingPolicy(
            UpdateSchedulingPolicyRequest updateSchedulingPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSchedulingPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Batch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSchedulingPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSchedulingPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateSchedulingPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateSchedulingPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSchedulingPolicyRequest, UpdateSchedulingPolicyResponse>()
                            .withOperationName("UpdateSchedulingPolicy")
                            .withMarshaller(new UpdateSchedulingPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateSchedulingPolicyRequest));
            CompletableFuture<UpdateSchedulingPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(BatchException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServerException")
                                .exceptionBuilderSupplier(ServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ClientException")
                                .exceptionBuilderSupplier(ClientException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends BatchRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
