/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.batch;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.SdkClient;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.batch.model.BatchException;
import software.amazon.awssdk.services.batch.model.CancelJobRequest;
import software.amazon.awssdk.services.batch.model.CancelJobResponse;
import software.amazon.awssdk.services.batch.model.ClientException;
import software.amazon.awssdk.services.batch.model.CreateComputeEnvironmentRequest;
import software.amazon.awssdk.services.batch.model.CreateComputeEnvironmentResponse;
import software.amazon.awssdk.services.batch.model.CreateJobQueueRequest;
import software.amazon.awssdk.services.batch.model.CreateJobQueueResponse;
import software.amazon.awssdk.services.batch.model.CreateSchedulingPolicyRequest;
import software.amazon.awssdk.services.batch.model.CreateSchedulingPolicyResponse;
import software.amazon.awssdk.services.batch.model.DeleteComputeEnvironmentRequest;
import software.amazon.awssdk.services.batch.model.DeleteComputeEnvironmentResponse;
import software.amazon.awssdk.services.batch.model.DeleteJobQueueRequest;
import software.amazon.awssdk.services.batch.model.DeleteJobQueueResponse;
import software.amazon.awssdk.services.batch.model.DeleteSchedulingPolicyRequest;
import software.amazon.awssdk.services.batch.model.DeleteSchedulingPolicyResponse;
import software.amazon.awssdk.services.batch.model.DeregisterJobDefinitionRequest;
import software.amazon.awssdk.services.batch.model.DeregisterJobDefinitionResponse;
import software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsRequest;
import software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsResponse;
import software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsRequest;
import software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsResponse;
import software.amazon.awssdk.services.batch.model.DescribeJobQueuesRequest;
import software.amazon.awssdk.services.batch.model.DescribeJobQueuesResponse;
import software.amazon.awssdk.services.batch.model.DescribeJobsRequest;
import software.amazon.awssdk.services.batch.model.DescribeJobsResponse;
import software.amazon.awssdk.services.batch.model.DescribeSchedulingPoliciesRequest;
import software.amazon.awssdk.services.batch.model.DescribeSchedulingPoliciesResponse;
import software.amazon.awssdk.services.batch.model.ListJobsRequest;
import software.amazon.awssdk.services.batch.model.ListJobsResponse;
import software.amazon.awssdk.services.batch.model.ListSchedulingPoliciesRequest;
import software.amazon.awssdk.services.batch.model.ListSchedulingPoliciesResponse;
import software.amazon.awssdk.services.batch.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.batch.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.batch.model.RegisterJobDefinitionRequest;
import software.amazon.awssdk.services.batch.model.RegisterJobDefinitionResponse;
import software.amazon.awssdk.services.batch.model.ServerException;
import software.amazon.awssdk.services.batch.model.SubmitJobRequest;
import software.amazon.awssdk.services.batch.model.SubmitJobResponse;
import software.amazon.awssdk.services.batch.model.TagResourceRequest;
import software.amazon.awssdk.services.batch.model.TagResourceResponse;
import software.amazon.awssdk.services.batch.model.TerminateJobRequest;
import software.amazon.awssdk.services.batch.model.TerminateJobResponse;
import software.amazon.awssdk.services.batch.model.UntagResourceRequest;
import software.amazon.awssdk.services.batch.model.UntagResourceResponse;
import software.amazon.awssdk.services.batch.model.UpdateComputeEnvironmentRequest;
import software.amazon.awssdk.services.batch.model.UpdateComputeEnvironmentResponse;
import software.amazon.awssdk.services.batch.model.UpdateJobQueueRequest;
import software.amazon.awssdk.services.batch.model.UpdateJobQueueResponse;
import software.amazon.awssdk.services.batch.model.UpdateSchedulingPolicyRequest;
import software.amazon.awssdk.services.batch.model.UpdateSchedulingPolicyResponse;
import software.amazon.awssdk.services.batch.paginators.DescribeComputeEnvironmentsIterable;
import software.amazon.awssdk.services.batch.paginators.DescribeJobDefinitionsIterable;
import software.amazon.awssdk.services.batch.paginators.DescribeJobQueuesIterable;
import software.amazon.awssdk.services.batch.paginators.ListJobsIterable;
import software.amazon.awssdk.services.batch.paginators.ListSchedulingPoliciesIterable;

/**
 * Service client for accessing AWS Batch. This can be created using the static {@link #builder()} method.
 *
 * <fullname>Batch</fullname>
 * <p>
 * Using Batch, you can run batch computing workloads on the Amazon Web Services Cloud. Batch computing is a common
 * means for developers, scientists, and engineers to access large amounts of compute resources. Batch uses the
 * advantages of the batch computing to remove the undifferentiated heavy lifting of configuring and managing required
 * infrastructure. At the same time, it also adopts a familiar batch computing software approach. You can use Batch to
 * efficiently provision resources d, and work toward eliminating capacity constraints, reducing your overall compute
 * costs, and delivering results more quickly.
 * </p>
 * <p>
 * As a fully managed service, Batch can run batch computing workloads of any scale. Batch automatically provisions
 * compute resources and optimizes workload distribution based on the quantity and scale of your specific workloads.
 * With Batch, there's no need to install or manage batch computing software. This means that you can focus on analyzing
 * results and solving your specific problems instead.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface BatchClient extends SdkClient {
    String SERVICE_NAME = "batch";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "batch";

    /**
     * <p>
     * Cancels a job in an Batch job queue. Jobs that are in the <code>SUBMITTED</code> or <code>PENDING</code> are
     * canceled. A job in<code>RUNNABLE</code> remains in <code>RUNNABLE</code> until it reaches the head of the job
     * queue. Then the job status is updated to <code>FAILED</code>.
     * </p>
     * <p>
     * Jobs that progressed to the <code>STARTING</code> or <code>RUNNING</code> state aren't canceled. However, the API
     * operation still succeeds, even if no job is canceled. These jobs must be terminated with the <a>TerminateJob</a>
     * operation.
     * </p>
     *
     * @param cancelJobRequest
     *        Contains the parameters for <code>CancelJob</code>.
     * @return Result of the CancelJob operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.CancelJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/CancelJob" target="_top">AWS API
     *      Documentation</a>
     */
    default CancelJobResponse cancelJob(CancelJobRequest cancelJobRequest) throws ClientException, ServerException,
            AwsServiceException, SdkClientException, BatchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Cancels a job in an Batch job queue. Jobs that are in the <code>SUBMITTED</code> or <code>PENDING</code> are
     * canceled. A job in<code>RUNNABLE</code> remains in <code>RUNNABLE</code> until it reaches the head of the job
     * queue. Then the job status is updated to <code>FAILED</code>.
     * </p>
     * <p>
     * Jobs that progressed to the <code>STARTING</code> or <code>RUNNING</code> state aren't canceled. However, the API
     * operation still succeeds, even if no job is canceled. These jobs must be terminated with the <a>TerminateJob</a>
     * operation.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CancelJobRequest.Builder} avoiding the need to
     * create one manually via {@link CancelJobRequest#builder()}
     * </p>
     *
     * @param cancelJobRequest
     *        A {@link Consumer} that will call methods on {@link CancelJobRequest.Builder} to create a request.
     *        Contains the parameters for <code>CancelJob</code>.
     * @return Result of the CancelJob operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.CancelJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/CancelJob" target="_top">AWS API
     *      Documentation</a>
     */
    default CancelJobResponse cancelJob(Consumer<CancelJobRequest.Builder> cancelJobRequest) throws ClientException,
            ServerException, AwsServiceException, SdkClientException, BatchException {
        return cancelJob(CancelJobRequest.builder().applyMutation(cancelJobRequest).build());
    }

    /**
     * <p>
     * Creates an Batch compute environment. You can create <code>MANAGED</code> or <code>UNMANAGED</code> compute
     * environments. <code>MANAGED</code> compute environments can use Amazon EC2 or Fargate resources.
     * <code>UNMANAGED</code> compute environments can only use EC2 resources.
     * </p>
     * <p>
     * In a managed compute environment, Batch manages the capacity and instance types of the compute resources within
     * the environment. This is based on the compute resource specification that you define or the <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html">launch template</a> that you
     * specify when you create the compute environment. Either, you can choose to use EC2 On-Demand Instances and EC2
     * Spot Instances. Or, you can use Fargate and Fargate Spot capacity in your managed compute environment. You can
     * optionally set a maximum price so that Spot Instances only launch when the Spot Instance price is less than a
     * specified percentage of the On-Demand price.
     * </p>
     * <note>
     * <p>
     * Multi-node parallel jobs aren't supported on Spot Instances.
     * </p>
     * </note>
     * <p>
     * In an unmanaged compute environment, you can manage your own EC2 compute resources and have flexibility with how
     * you configure your compute resources. For example, you can use custom AMIs. However, you must verify that each of
     * your AMIs meet the Amazon ECS container instance AMI specification. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container_instance_AMIs.html">container
     * instance AMIs</a> in the <i>Amazon Elastic Container Service Developer Guide</i>. After you created your
     * unmanaged compute environment, you can use the <a>DescribeComputeEnvironments</a> operation to find the Amazon
     * ECS cluster that's associated with it. Then, launch your container instances into that Amazon ECS cluster. For
     * more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_container_instance.html">Launching an
     * Amazon ECS container instance</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <note>
     * <p>
     * To create a compute environment that uses EKS resources, the caller must have permissions to call
     * <code>eks:DescribeCluster</code>.
     * </p>
     * </note> <note>
     * <p>
     * Batch doesn't automatically upgrade the AMIs in a compute environment after it's created. For example, it also
     * doesn't update the AMIs in your compute environment when a newer version of the Amazon ECS optimized AMI is
     * available. You're responsible for the management of the guest operating system. This includes any updates and
     * security patches. You're also responsible for any additional application software or utilities that you install
     * on the compute resources. There are two ways to use a new AMI for your Batch jobs. The original method is to
     * complete these steps:
     * </p>
     * <ol>
     * <li>
     * <p>
     * Create a new compute environment with the new AMI.
     * </p>
     * </li>
     * <li>
     * <p>
     * Add the compute environment to an existing job queue.
     * </p>
     * </li>
     * <li>
     * <p>
     * Remove the earlier compute environment from your job queue.
     * </p>
     * </li>
     * <li>
     * <p>
     * Delete the earlier compute environment.
     * </p>
     * </li>
     * </ol>
     * <p>
     * In April 2022, Batch added enhanced support for updating compute environments. For more information, see <a
     * href="https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html">Updating compute
     * environments</a>. To use the enhanced updating of compute environments to update AMIs, follow these rules:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Either don't set the service role (<code>serviceRole</code>) parameter or set it to the
     * <b>AWSBatchServiceRole</b> service-linked role.
     * </p>
     * </li>
     * <li>
     * <p>
     * Set the allocation strategy (<code>allocationStrategy</code>) parameter to <code>BEST_FIT_PROGRESSIVE</code> or
     * <code>SPOT_CAPACITY_OPTIMIZED</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Set the update to latest image version (<code>updateToLatestImageVersion</code>) parameter to <code>true</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Don't specify an AMI ID in <code>imageId</code>, <code>imageIdOverride</code> (in <a
     * href="https://docs.aws.amazon.com/batch/latest/APIReference/API_Ec2Configuration.html">
     * <code>ec2Configuration</code> </a>), or in the launch template (<code>launchTemplate</code>). In that case, Batch
     * selects the latest Amazon ECS optimized AMI that's supported by Batch at the time the infrastructure update is
     * initiated. Alternatively, you can specify the AMI ID in the <code>imageId</code> or <code>imageIdOverride</code>
     * parameters, or the launch template identified by the <code>LaunchTemplate</code> properties. Changing any of
     * these properties starts an infrastructure update. If the AMI ID is specified in the launch template, it can't be
     * replaced by specifying an AMI ID in either the <code>imageId</code> or <code>imageIdOverride</code> parameters.
     * It can only be replaced by specifying a different launch template, or if the launch template version is set to
     * <code>&#36Default</code> or <code>&#36Latest</code>, by setting either a new default version for the launch
     * template (if <code>&#36Default</code>) or by adding a new version to the launch template (if
     * <code>&#36Latest</code>).
     * </p>
     * </li>
     * </ul>
     * <p>
     * If these rules are followed, any update that starts an infrastructure update causes the AMI ID to be re-selected.
     * If the <code>version</code> setting in the launch template (<code>launchTemplate</code>) is set to
     * <code>&#36Latest</code> or <code>&#36Default</code>, the latest or default version of the launch template is
     * evaluated up at the time of the infrastructure update, even if the <code>launchTemplate</code> wasn't updated.
     * </p>
     * </note>
     *
     * @param createComputeEnvironmentRequest
     *        Contains the parameters for <code>CreateComputeEnvironment</code>.
     * @return Result of the CreateComputeEnvironment operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.CreateComputeEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/CreateComputeEnvironment"
     *      target="_top">AWS API Documentation</a>
     */
    default CreateComputeEnvironmentResponse createComputeEnvironment(
            CreateComputeEnvironmentRequest createComputeEnvironmentRequest) throws ClientException, ServerException,
            AwsServiceException, SdkClientException, BatchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates an Batch compute environment. You can create <code>MANAGED</code> or <code>UNMANAGED</code> compute
     * environments. <code>MANAGED</code> compute environments can use Amazon EC2 or Fargate resources.
     * <code>UNMANAGED</code> compute environments can only use EC2 resources.
     * </p>
     * <p>
     * In a managed compute environment, Batch manages the capacity and instance types of the compute resources within
     * the environment. This is based on the compute resource specification that you define or the <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html">launch template</a> that you
     * specify when you create the compute environment. Either, you can choose to use EC2 On-Demand Instances and EC2
     * Spot Instances. Or, you can use Fargate and Fargate Spot capacity in your managed compute environment. You can
     * optionally set a maximum price so that Spot Instances only launch when the Spot Instance price is less than a
     * specified percentage of the On-Demand price.
     * </p>
     * <note>
     * <p>
     * Multi-node parallel jobs aren't supported on Spot Instances.
     * </p>
     * </note>
     * <p>
     * In an unmanaged compute environment, you can manage your own EC2 compute resources and have flexibility with how
     * you configure your compute resources. For example, you can use custom AMIs. However, you must verify that each of
     * your AMIs meet the Amazon ECS container instance AMI specification. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container_instance_AMIs.html">container
     * instance AMIs</a> in the <i>Amazon Elastic Container Service Developer Guide</i>. After you created your
     * unmanaged compute environment, you can use the <a>DescribeComputeEnvironments</a> operation to find the Amazon
     * ECS cluster that's associated with it. Then, launch your container instances into that Amazon ECS cluster. For
     * more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_container_instance.html">Launching an
     * Amazon ECS container instance</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * <note>
     * <p>
     * To create a compute environment that uses EKS resources, the caller must have permissions to call
     * <code>eks:DescribeCluster</code>.
     * </p>
     * </note> <note>
     * <p>
     * Batch doesn't automatically upgrade the AMIs in a compute environment after it's created. For example, it also
     * doesn't update the AMIs in your compute environment when a newer version of the Amazon ECS optimized AMI is
     * available. You're responsible for the management of the guest operating system. This includes any updates and
     * security patches. You're also responsible for any additional application software or utilities that you install
     * on the compute resources. There are two ways to use a new AMI for your Batch jobs. The original method is to
     * complete these steps:
     * </p>
     * <ol>
     * <li>
     * <p>
     * Create a new compute environment with the new AMI.
     * </p>
     * </li>
     * <li>
     * <p>
     * Add the compute environment to an existing job queue.
     * </p>
     * </li>
     * <li>
     * <p>
     * Remove the earlier compute environment from your job queue.
     * </p>
     * </li>
     * <li>
     * <p>
     * Delete the earlier compute environment.
     * </p>
     * </li>
     * </ol>
     * <p>
     * In April 2022, Batch added enhanced support for updating compute environments. For more information, see <a
     * href="https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html">Updating compute
     * environments</a>. To use the enhanced updating of compute environments to update AMIs, follow these rules:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Either don't set the service role (<code>serviceRole</code>) parameter or set it to the
     * <b>AWSBatchServiceRole</b> service-linked role.
     * </p>
     * </li>
     * <li>
     * <p>
     * Set the allocation strategy (<code>allocationStrategy</code>) parameter to <code>BEST_FIT_PROGRESSIVE</code> or
     * <code>SPOT_CAPACITY_OPTIMIZED</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Set the update to latest image version (<code>updateToLatestImageVersion</code>) parameter to <code>true</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Don't specify an AMI ID in <code>imageId</code>, <code>imageIdOverride</code> (in <a
     * href="https://docs.aws.amazon.com/batch/latest/APIReference/API_Ec2Configuration.html">
     * <code>ec2Configuration</code> </a>), or in the launch template (<code>launchTemplate</code>). In that case, Batch
     * selects the latest Amazon ECS optimized AMI that's supported by Batch at the time the infrastructure update is
     * initiated. Alternatively, you can specify the AMI ID in the <code>imageId</code> or <code>imageIdOverride</code>
     * parameters, or the launch template identified by the <code>LaunchTemplate</code> properties. Changing any of
     * these properties starts an infrastructure update. If the AMI ID is specified in the launch template, it can't be
     * replaced by specifying an AMI ID in either the <code>imageId</code> or <code>imageIdOverride</code> parameters.
     * It can only be replaced by specifying a different launch template, or if the launch template version is set to
     * <code>&#36Default</code> or <code>&#36Latest</code>, by setting either a new default version for the launch
     * template (if <code>&#36Default</code>) or by adding a new version to the launch template (if
     * <code>&#36Latest</code>).
     * </p>
     * </li>
     * </ul>
     * <p>
     * If these rules are followed, any update that starts an infrastructure update causes the AMI ID to be re-selected.
     * If the <code>version</code> setting in the launch template (<code>launchTemplate</code>) is set to
     * <code>&#36Latest</code> or <code>&#36Default</code>, the latest or default version of the launch template is
     * evaluated up at the time of the infrastructure update, even if the <code>launchTemplate</code> wasn't updated.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateComputeEnvironmentRequest.Builder} avoiding
     * the need to create one manually via {@link CreateComputeEnvironmentRequest#builder()}
     * </p>
     *
     * @param createComputeEnvironmentRequest
     *        A {@link Consumer} that will call methods on {@link CreateComputeEnvironmentRequest.Builder} to create a
     *        request. Contains the parameters for <code>CreateComputeEnvironment</code>.
     * @return Result of the CreateComputeEnvironment operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.CreateComputeEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/CreateComputeEnvironment"
     *      target="_top">AWS API Documentation</a>
     */
    default CreateComputeEnvironmentResponse createComputeEnvironment(
            Consumer<CreateComputeEnvironmentRequest.Builder> createComputeEnvironmentRequest) throws ClientException,
            ServerException, AwsServiceException, SdkClientException, BatchException {
        return createComputeEnvironment(CreateComputeEnvironmentRequest.builder().applyMutation(createComputeEnvironmentRequest)
                .build());
    }

    /**
     * <p>
     * Creates an Batch job queue. When you create a job queue, you associate one or more compute environments to the
     * queue and assign an order of preference for the compute environments.
     * </p>
     * <p>
     * You also set a priority to the job queue that determines the order that the Batch scheduler places jobs onto its
     * associated compute environments. For example, if a compute environment is associated with more than one job
     * queue, the job queue with a higher priority is given preference for scheduling jobs to that compute environment.
     * </p>
     *
     * @param createJobQueueRequest
     *        Contains the parameters for <code>CreateJobQueue</code>.
     * @return Result of the CreateJobQueue operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.CreateJobQueue
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/CreateJobQueue" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateJobQueueResponse createJobQueue(CreateJobQueueRequest createJobQueueRequest) throws ClientException,
            ServerException, AwsServiceException, SdkClientException, BatchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates an Batch job queue. When you create a job queue, you associate one or more compute environments to the
     * queue and assign an order of preference for the compute environments.
     * </p>
     * <p>
     * You also set a priority to the job queue that determines the order that the Batch scheduler places jobs onto its
     * associated compute environments. For example, if a compute environment is associated with more than one job
     * queue, the job queue with a higher priority is given preference for scheduling jobs to that compute environment.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateJobQueueRequest.Builder} avoiding the need to
     * create one manually via {@link CreateJobQueueRequest#builder()}
     * </p>
     *
     * @param createJobQueueRequest
     *        A {@link Consumer} that will call methods on {@link CreateJobQueueRequest.Builder} to create a request.
     *        Contains the parameters for <code>CreateJobQueue</code>.
     * @return Result of the CreateJobQueue operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.CreateJobQueue
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/CreateJobQueue" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateJobQueueResponse createJobQueue(Consumer<CreateJobQueueRequest.Builder> createJobQueueRequest)
            throws ClientException, ServerException, AwsServiceException, SdkClientException, BatchException {
        return createJobQueue(CreateJobQueueRequest.builder().applyMutation(createJobQueueRequest).build());
    }

    /**
     * <p>
     * Creates an Batch scheduling policy.
     * </p>
     *
     * @param createSchedulingPolicyRequest
     *        Contains the parameters for <code>CreateSchedulingPolicy</code>.
     * @return Result of the CreateSchedulingPolicy operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.CreateSchedulingPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/CreateSchedulingPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    default CreateSchedulingPolicyResponse createSchedulingPolicy(CreateSchedulingPolicyRequest createSchedulingPolicyRequest)
            throws ClientException, ServerException, AwsServiceException, SdkClientException, BatchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates an Batch scheduling policy.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateSchedulingPolicyRequest.Builder} avoiding the
     * need to create one manually via {@link CreateSchedulingPolicyRequest#builder()}
     * </p>
     *
     * @param createSchedulingPolicyRequest
     *        A {@link Consumer} that will call methods on {@link CreateSchedulingPolicyRequest.Builder} to create a
     *        request. Contains the parameters for <code>CreateSchedulingPolicy</code>.
     * @return Result of the CreateSchedulingPolicy operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.CreateSchedulingPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/CreateSchedulingPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    default CreateSchedulingPolicyResponse createSchedulingPolicy(
            Consumer<CreateSchedulingPolicyRequest.Builder> createSchedulingPolicyRequest) throws ClientException,
            ServerException, AwsServiceException, SdkClientException, BatchException {
        return createSchedulingPolicy(CreateSchedulingPolicyRequest.builder().applyMutation(createSchedulingPolicyRequest)
                .build());
    }

    /**
     * <p>
     * Deletes an Batch compute environment.
     * </p>
     * <p>
     * Before you can delete a compute environment, you must set its state to <code>DISABLED</code> with the
     * <a>UpdateComputeEnvironment</a> API operation and disassociate it from any job queues with the
     * <a>UpdateJobQueue</a> API operation. Compute environments that use Fargate resources must terminate all active
     * jobs on that compute environment before deleting the compute environment. If this isn't done, the compute
     * environment enters an invalid state.
     * </p>
     *
     * @param deleteComputeEnvironmentRequest
     *        Contains the parameters for <code>DeleteComputeEnvironment</code>.
     * @return Result of the DeleteComputeEnvironment operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DeleteComputeEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DeleteComputeEnvironment"
     *      target="_top">AWS API Documentation</a>
     */
    default DeleteComputeEnvironmentResponse deleteComputeEnvironment(
            DeleteComputeEnvironmentRequest deleteComputeEnvironmentRequest) throws ClientException, ServerException,
            AwsServiceException, SdkClientException, BatchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes an Batch compute environment.
     * </p>
     * <p>
     * Before you can delete a compute environment, you must set its state to <code>DISABLED</code> with the
     * <a>UpdateComputeEnvironment</a> API operation and disassociate it from any job queues with the
     * <a>UpdateJobQueue</a> API operation. Compute environments that use Fargate resources must terminate all active
     * jobs on that compute environment before deleting the compute environment. If this isn't done, the compute
     * environment enters an invalid state.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteComputeEnvironmentRequest.Builder} avoiding
     * the need to create one manually via {@link DeleteComputeEnvironmentRequest#builder()}
     * </p>
     *
     * @param deleteComputeEnvironmentRequest
     *        A {@link Consumer} that will call methods on {@link DeleteComputeEnvironmentRequest.Builder} to create a
     *        request. Contains the parameters for <code>DeleteComputeEnvironment</code>.
     * @return Result of the DeleteComputeEnvironment operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DeleteComputeEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DeleteComputeEnvironment"
     *      target="_top">AWS API Documentation</a>
     */
    default DeleteComputeEnvironmentResponse deleteComputeEnvironment(
            Consumer<DeleteComputeEnvironmentRequest.Builder> deleteComputeEnvironmentRequest) throws ClientException,
            ServerException, AwsServiceException, SdkClientException, BatchException {
        return deleteComputeEnvironment(DeleteComputeEnvironmentRequest.builder().applyMutation(deleteComputeEnvironmentRequest)
                .build());
    }

    /**
     * <p>
     * Deletes the specified job queue. You must first disable submissions for a queue with the <a>UpdateJobQueue</a>
     * operation. All jobs in the queue are eventually terminated when you delete a job queue. The jobs are terminated
     * at a rate of about 16 jobs each second.
     * </p>
     * <p>
     * It's not necessary to disassociate compute environments from a queue before submitting a
     * <code>DeleteJobQueue</code> request.
     * </p>
     *
     * @param deleteJobQueueRequest
     *        Contains the parameters for <code>DeleteJobQueue</code>.
     * @return Result of the DeleteJobQueue operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DeleteJobQueue
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DeleteJobQueue" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteJobQueueResponse deleteJobQueue(DeleteJobQueueRequest deleteJobQueueRequest) throws ClientException,
            ServerException, AwsServiceException, SdkClientException, BatchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the specified job queue. You must first disable submissions for a queue with the <a>UpdateJobQueue</a>
     * operation. All jobs in the queue are eventually terminated when you delete a job queue. The jobs are terminated
     * at a rate of about 16 jobs each second.
     * </p>
     * <p>
     * It's not necessary to disassociate compute environments from a queue before submitting a
     * <code>DeleteJobQueue</code> request.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteJobQueueRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteJobQueueRequest#builder()}
     * </p>
     *
     * @param deleteJobQueueRequest
     *        A {@link Consumer} that will call methods on {@link DeleteJobQueueRequest.Builder} to create a request.
     *        Contains the parameters for <code>DeleteJobQueue</code>.
     * @return Result of the DeleteJobQueue operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DeleteJobQueue
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DeleteJobQueue" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteJobQueueResponse deleteJobQueue(Consumer<DeleteJobQueueRequest.Builder> deleteJobQueueRequest)
            throws ClientException, ServerException, AwsServiceException, SdkClientException, BatchException {
        return deleteJobQueue(DeleteJobQueueRequest.builder().applyMutation(deleteJobQueueRequest).build());
    }

    /**
     * <p>
     * Deletes the specified scheduling policy.
     * </p>
     * <p>
     * You can't delete a scheduling policy that's used in any job queues.
     * </p>
     *
     * @param deleteSchedulingPolicyRequest
     *        Contains the parameters for <code>DeleteSchedulingPolicy</code>.
     * @return Result of the DeleteSchedulingPolicy operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DeleteSchedulingPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DeleteSchedulingPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    default DeleteSchedulingPolicyResponse deleteSchedulingPolicy(DeleteSchedulingPolicyRequest deleteSchedulingPolicyRequest)
            throws ClientException, ServerException, AwsServiceException, SdkClientException, BatchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the specified scheduling policy.
     * </p>
     * <p>
     * You can't delete a scheduling policy that's used in any job queues.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteSchedulingPolicyRequest.Builder} avoiding the
     * need to create one manually via {@link DeleteSchedulingPolicyRequest#builder()}
     * </p>
     *
     * @param deleteSchedulingPolicyRequest
     *        A {@link Consumer} that will call methods on {@link DeleteSchedulingPolicyRequest.Builder} to create a
     *        request. Contains the parameters for <code>DeleteSchedulingPolicy</code>.
     * @return Result of the DeleteSchedulingPolicy operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DeleteSchedulingPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DeleteSchedulingPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    default DeleteSchedulingPolicyResponse deleteSchedulingPolicy(
            Consumer<DeleteSchedulingPolicyRequest.Builder> deleteSchedulingPolicyRequest) throws ClientException,
            ServerException, AwsServiceException, SdkClientException, BatchException {
        return deleteSchedulingPolicy(DeleteSchedulingPolicyRequest.builder().applyMutation(deleteSchedulingPolicyRequest)
                .build());
    }

    /**
     * <p>
     * Deregisters an Batch job definition. Job definitions are permanently deleted after 180 days.
     * </p>
     *
     * @param deregisterJobDefinitionRequest
     * @return Result of the DeregisterJobDefinition operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DeregisterJobDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DeregisterJobDefinition" target="_top">AWS
     *      API Documentation</a>
     */
    default DeregisterJobDefinitionResponse deregisterJobDefinition(DeregisterJobDefinitionRequest deregisterJobDefinitionRequest)
            throws ClientException, ServerException, AwsServiceException, SdkClientException, BatchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deregisters an Batch job definition. Job definitions are permanently deleted after 180 days.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeregisterJobDefinitionRequest.Builder} avoiding
     * the need to create one manually via {@link DeregisterJobDefinitionRequest#builder()}
     * </p>
     *
     * @param deregisterJobDefinitionRequest
     *        A {@link Consumer} that will call methods on {@link DeregisterJobDefinitionRequest.Builder} to create a
     *        request.
     * @return Result of the DeregisterJobDefinition operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DeregisterJobDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DeregisterJobDefinition" target="_top">AWS
     *      API Documentation</a>
     */
    default DeregisterJobDefinitionResponse deregisterJobDefinition(
            Consumer<DeregisterJobDefinitionRequest.Builder> deregisterJobDefinitionRequest) throws ClientException,
            ServerException, AwsServiceException, SdkClientException, BatchException {
        return deregisterJobDefinition(DeregisterJobDefinitionRequest.builder().applyMutation(deregisterJobDefinitionRequest)
                .build());
    }

    /**
     * <p>
     * Describes one or more of your compute environments.
     * </p>
     * <p>
     * If you're using an unmanaged compute environment, you can use the <code>DescribeComputeEnvironment</code>
     * operation to determine the <code>ecsClusterArn</code> that you launch your Amazon ECS container instances into.
     * </p>
     *
     * @return Result of the DescribeComputeEnvironments operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DescribeComputeEnvironments
     * @see #describeComputeEnvironments(DescribeComputeEnvironmentsRequest)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeComputeEnvironments"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeComputeEnvironmentsResponse describeComputeEnvironments() throws ClientException, ServerException,
            AwsServiceException, SdkClientException, BatchException {
        return describeComputeEnvironments(DescribeComputeEnvironmentsRequest.builder().build());
    }

    /**
     * <p>
     * Describes one or more of your compute environments.
     * </p>
     * <p>
     * If you're using an unmanaged compute environment, you can use the <code>DescribeComputeEnvironment</code>
     * operation to determine the <code>ecsClusterArn</code> that you launch your Amazon ECS container instances into.
     * </p>
     *
     * @param describeComputeEnvironmentsRequest
     *        Contains the parameters for <code>DescribeComputeEnvironments</code>.
     * @return Result of the DescribeComputeEnvironments operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DescribeComputeEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeComputeEnvironments"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeComputeEnvironmentsResponse describeComputeEnvironments(
            DescribeComputeEnvironmentsRequest describeComputeEnvironmentsRequest) throws ClientException, ServerException,
            AwsServiceException, SdkClientException, BatchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes one or more of your compute environments.
     * </p>
     * <p>
     * If you're using an unmanaged compute environment, you can use the <code>DescribeComputeEnvironment</code>
     * operation to determine the <code>ecsClusterArn</code> that you launch your Amazon ECS container instances into.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeComputeEnvironmentsRequest.Builder}
     * avoiding the need to create one manually via {@link DescribeComputeEnvironmentsRequest#builder()}
     * </p>
     *
     * @param describeComputeEnvironmentsRequest
     *        A {@link Consumer} that will call methods on {@link DescribeComputeEnvironmentsRequest.Builder} to create
     *        a request. Contains the parameters for <code>DescribeComputeEnvironments</code>.
     * @return Result of the DescribeComputeEnvironments operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DescribeComputeEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeComputeEnvironments"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeComputeEnvironmentsResponse describeComputeEnvironments(
            Consumer<DescribeComputeEnvironmentsRequest.Builder> describeComputeEnvironmentsRequest) throws ClientException,
            ServerException, AwsServiceException, SdkClientException, BatchException {
        return describeComputeEnvironments(DescribeComputeEnvironmentsRequest.builder()
                .applyMutation(describeComputeEnvironmentsRequest).build());
    }

    /**
     * <p>
     * Describes one or more of your compute environments.
     * </p>
     * <p>
     * If you're using an unmanaged compute environment, you can use the <code>DescribeComputeEnvironment</code>
     * operation to determine the <code>ecsClusterArn</code> that you launch your Amazon ECS container instances into.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeComputeEnvironments(software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeComputeEnvironmentsIterable responses = client.describeComputeEnvironmentsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.batch.paginators.DescribeComputeEnvironmentsIterable responses = client
     *             .describeComputeEnvironmentsPaginator(request);
     *     for (software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeComputeEnvironmentsIterable responses = client.describeComputeEnvironmentsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeComputeEnvironments(software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsRequest)}
     * operation.</b>
     * </p>
     *
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DescribeComputeEnvironments
     * @see #describeComputeEnvironmentsPaginator(DescribeComputeEnvironmentsRequest)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeComputeEnvironments"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeComputeEnvironmentsIterable describeComputeEnvironmentsPaginator() throws ClientException, ServerException,
            AwsServiceException, SdkClientException, BatchException {
        return describeComputeEnvironmentsPaginator(DescribeComputeEnvironmentsRequest.builder().build());
    }

    /**
     * <p>
     * Describes one or more of your compute environments.
     * </p>
     * <p>
     * If you're using an unmanaged compute environment, you can use the <code>DescribeComputeEnvironment</code>
     * operation to determine the <code>ecsClusterArn</code> that you launch your Amazon ECS container instances into.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeComputeEnvironments(software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeComputeEnvironmentsIterable responses = client.describeComputeEnvironmentsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.batch.paginators.DescribeComputeEnvironmentsIterable responses = client
     *             .describeComputeEnvironmentsPaginator(request);
     *     for (software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeComputeEnvironmentsIterable responses = client.describeComputeEnvironmentsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeComputeEnvironments(software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeComputeEnvironmentsRequest
     *        Contains the parameters for <code>DescribeComputeEnvironments</code>.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DescribeComputeEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeComputeEnvironments"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeComputeEnvironmentsIterable describeComputeEnvironmentsPaginator(
            DescribeComputeEnvironmentsRequest describeComputeEnvironmentsRequest) throws ClientException, ServerException,
            AwsServiceException, SdkClientException, BatchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes one or more of your compute environments.
     * </p>
     * <p>
     * If you're using an unmanaged compute environment, you can use the <code>DescribeComputeEnvironment</code>
     * operation to determine the <code>ecsClusterArn</code> that you launch your Amazon ECS container instances into.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeComputeEnvironments(software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeComputeEnvironmentsIterable responses = client.describeComputeEnvironmentsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.batch.paginators.DescribeComputeEnvironmentsIterable responses = client
     *             .describeComputeEnvironmentsPaginator(request);
     *     for (software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeComputeEnvironmentsIterable responses = client.describeComputeEnvironmentsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeComputeEnvironments(software.amazon.awssdk.services.batch.model.DescribeComputeEnvironmentsRequest)}
     * operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeComputeEnvironmentsRequest.Builder}
     * avoiding the need to create one manually via {@link DescribeComputeEnvironmentsRequest#builder()}
     * </p>
     *
     * @param describeComputeEnvironmentsRequest
     *        A {@link Consumer} that will call methods on {@link DescribeComputeEnvironmentsRequest.Builder} to create
     *        a request. Contains the parameters for <code>DescribeComputeEnvironments</code>.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DescribeComputeEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeComputeEnvironments"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeComputeEnvironmentsIterable describeComputeEnvironmentsPaginator(
            Consumer<DescribeComputeEnvironmentsRequest.Builder> describeComputeEnvironmentsRequest) throws ClientException,
            ServerException, AwsServiceException, SdkClientException, BatchException {
        return describeComputeEnvironmentsPaginator(DescribeComputeEnvironmentsRequest.builder()
                .applyMutation(describeComputeEnvironmentsRequest).build());
    }

    /**
     * <p>
     * Describes a list of job definitions. You can specify a <code>status</code> (such as <code>ACTIVE</code>) to only
     * return job definitions that match that status.
     * </p>
     *
     * @return Result of the DescribeJobDefinitions operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DescribeJobDefinitions
     * @see #describeJobDefinitions(DescribeJobDefinitionsRequest)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobDefinitions" target="_top">AWS
     *      API Documentation</a>
     */
    default DescribeJobDefinitionsResponse describeJobDefinitions() throws ClientException, ServerException, AwsServiceException,
            SdkClientException, BatchException {
        return describeJobDefinitions(DescribeJobDefinitionsRequest.builder().build());
    }

    /**
     * <p>
     * Describes a list of job definitions. You can specify a <code>status</code> (such as <code>ACTIVE</code>) to only
     * return job definitions that match that status.
     * </p>
     *
     * @param describeJobDefinitionsRequest
     *        Contains the parameters for <code>DescribeJobDefinitions</code>.
     * @return Result of the DescribeJobDefinitions operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DescribeJobDefinitions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobDefinitions" target="_top">AWS
     *      API Documentation</a>
     */
    default DescribeJobDefinitionsResponse describeJobDefinitions(DescribeJobDefinitionsRequest describeJobDefinitionsRequest)
            throws ClientException, ServerException, AwsServiceException, SdkClientException, BatchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes a list of job definitions. You can specify a <code>status</code> (such as <code>ACTIVE</code>) to only
     * return job definitions that match that status.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeJobDefinitionsRequest.Builder} avoiding the
     * need to create one manually via {@link DescribeJobDefinitionsRequest#builder()}
     * </p>
     *
     * @param describeJobDefinitionsRequest
     *        A {@link Consumer} that will call methods on {@link DescribeJobDefinitionsRequest.Builder} to create a
     *        request. Contains the parameters for <code>DescribeJobDefinitions</code>.
     * @return Result of the DescribeJobDefinitions operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DescribeJobDefinitions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobDefinitions" target="_top">AWS
     *      API Documentation</a>
     */
    default DescribeJobDefinitionsResponse describeJobDefinitions(
            Consumer<DescribeJobDefinitionsRequest.Builder> describeJobDefinitionsRequest) throws ClientException,
            ServerException, AwsServiceException, SdkClientException, BatchException {
        return describeJobDefinitions(DescribeJobDefinitionsRequest.builder().applyMutation(describeJobDefinitionsRequest)
                .build());
    }

    /**
     * <p>
     * Describes a list of job definitions. You can specify a <code>status</code> (such as <code>ACTIVE</code>) to only
     * return job definitions that match that status.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeJobDefinitions(software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeJobDefinitionsIterable responses = client.describeJobDefinitionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.batch.paginators.DescribeJobDefinitionsIterable responses = client
     *             .describeJobDefinitionsPaginator(request);
     *     for (software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeJobDefinitionsIterable responses = client.describeJobDefinitionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeJobDefinitions(software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsRequest)}
     * operation.</b>
     * </p>
     *
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DescribeJobDefinitions
     * @see #describeJobDefinitionsPaginator(DescribeJobDefinitionsRequest)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobDefinitions" target="_top">AWS
     *      API Documentation</a>
     */
    default DescribeJobDefinitionsIterable describeJobDefinitionsPaginator() throws ClientException, ServerException,
            AwsServiceException, SdkClientException, BatchException {
        return describeJobDefinitionsPaginator(DescribeJobDefinitionsRequest.builder().build());
    }

    /**
     * <p>
     * Describes a list of job definitions. You can specify a <code>status</code> (such as <code>ACTIVE</code>) to only
     * return job definitions that match that status.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeJobDefinitions(software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeJobDefinitionsIterable responses = client.describeJobDefinitionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.batch.paginators.DescribeJobDefinitionsIterable responses = client
     *             .describeJobDefinitionsPaginator(request);
     *     for (software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeJobDefinitionsIterable responses = client.describeJobDefinitionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeJobDefinitions(software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeJobDefinitionsRequest
     *        Contains the parameters for <code>DescribeJobDefinitions</code>.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DescribeJobDefinitions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobDefinitions" target="_top">AWS
     *      API Documentation</a>
     */
    default DescribeJobDefinitionsIterable describeJobDefinitionsPaginator(
            DescribeJobDefinitionsRequest describeJobDefinitionsRequest) throws ClientException, ServerException,
            AwsServiceException, SdkClientException, BatchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes a list of job definitions. You can specify a <code>status</code> (such as <code>ACTIVE</code>) to only
     * return job definitions that match that status.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeJobDefinitions(software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeJobDefinitionsIterable responses = client.describeJobDefinitionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.batch.paginators.DescribeJobDefinitionsIterable responses = client
     *             .describeJobDefinitionsPaginator(request);
     *     for (software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeJobDefinitionsIterable responses = client.describeJobDefinitionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeJobDefinitions(software.amazon.awssdk.services.batch.model.DescribeJobDefinitionsRequest)}
     * operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeJobDefinitionsRequest.Builder} avoiding the
     * need to create one manually via {@link DescribeJobDefinitionsRequest#builder()}
     * </p>
     *
     * @param describeJobDefinitionsRequest
     *        A {@link Consumer} that will call methods on {@link DescribeJobDefinitionsRequest.Builder} to create a
     *        request. Contains the parameters for <code>DescribeJobDefinitions</code>.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DescribeJobDefinitions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobDefinitions" target="_top">AWS
     *      API Documentation</a>
     */
    default DescribeJobDefinitionsIterable describeJobDefinitionsPaginator(
            Consumer<DescribeJobDefinitionsRequest.Builder> describeJobDefinitionsRequest) throws ClientException,
            ServerException, AwsServiceException, SdkClientException, BatchException {
        return describeJobDefinitionsPaginator(DescribeJobDefinitionsRequest.builder()
                .applyMutation(describeJobDefinitionsRequest).build());
    }

    /**
     * <p>
     * Describes one or more of your job queues.
     * </p>
     *
     * @return Result of the DescribeJobQueues operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DescribeJobQueues
     * @see #describeJobQueues(DescribeJobQueuesRequest)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobQueues" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeJobQueuesResponse describeJobQueues() throws ClientException, ServerException, AwsServiceException,
            SdkClientException, BatchException {
        return describeJobQueues(DescribeJobQueuesRequest.builder().build());
    }

    /**
     * <p>
     * Describes one or more of your job queues.
     * </p>
     *
     * @param describeJobQueuesRequest
     *        Contains the parameters for <code>DescribeJobQueues</code>.
     * @return Result of the DescribeJobQueues operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DescribeJobQueues
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobQueues" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeJobQueuesResponse describeJobQueues(DescribeJobQueuesRequest describeJobQueuesRequest)
            throws ClientException, ServerException, AwsServiceException, SdkClientException, BatchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes one or more of your job queues.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeJobQueuesRequest.Builder} avoiding the need
     * to create one manually via {@link DescribeJobQueuesRequest#builder()}
     * </p>
     *
     * @param describeJobQueuesRequest
     *        A {@link Consumer} that will call methods on {@link DescribeJobQueuesRequest.Builder} to create a request.
     *        Contains the parameters for <code>DescribeJobQueues</code>.
     * @return Result of the DescribeJobQueues operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DescribeJobQueues
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobQueues" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeJobQueuesResponse describeJobQueues(Consumer<DescribeJobQueuesRequest.Builder> describeJobQueuesRequest)
            throws ClientException, ServerException, AwsServiceException, SdkClientException, BatchException {
        return describeJobQueues(DescribeJobQueuesRequest.builder().applyMutation(describeJobQueuesRequest).build());
    }

    /**
     * <p>
     * Describes one or more of your job queues.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeJobQueues(software.amazon.awssdk.services.batch.model.DescribeJobQueuesRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeJobQueuesIterable responses = client.describeJobQueuesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.batch.paginators.DescribeJobQueuesIterable responses = client
     *             .describeJobQueuesPaginator(request);
     *     for (software.amazon.awssdk.services.batch.model.DescribeJobQueuesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeJobQueuesIterable responses = client.describeJobQueuesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeJobQueues(software.amazon.awssdk.services.batch.model.DescribeJobQueuesRequest)} operation.</b>
     * </p>
     *
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DescribeJobQueues
     * @see #describeJobQueuesPaginator(DescribeJobQueuesRequest)
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobQueues" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeJobQueuesIterable describeJobQueuesPaginator() throws ClientException, ServerException, AwsServiceException,
            SdkClientException, BatchException {
        return describeJobQueuesPaginator(DescribeJobQueuesRequest.builder().build());
    }

    /**
     * <p>
     * Describes one or more of your job queues.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeJobQueues(software.amazon.awssdk.services.batch.model.DescribeJobQueuesRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeJobQueuesIterable responses = client.describeJobQueuesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.batch.paginators.DescribeJobQueuesIterable responses = client
     *             .describeJobQueuesPaginator(request);
     *     for (software.amazon.awssdk.services.batch.model.DescribeJobQueuesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeJobQueuesIterable responses = client.describeJobQueuesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeJobQueues(software.amazon.awssdk.services.batch.model.DescribeJobQueuesRequest)} operation.</b>
     * </p>
     *
     * @param describeJobQueuesRequest
     *        Contains the parameters for <code>DescribeJobQueues</code>.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DescribeJobQueues
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobQueues" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeJobQueuesIterable describeJobQueuesPaginator(DescribeJobQueuesRequest describeJobQueuesRequest)
            throws ClientException, ServerException, AwsServiceException, SdkClientException, BatchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes one or more of your job queues.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeJobQueues(software.amazon.awssdk.services.batch.model.DescribeJobQueuesRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeJobQueuesIterable responses = client.describeJobQueuesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.batch.paginators.DescribeJobQueuesIterable responses = client
     *             .describeJobQueuesPaginator(request);
     *     for (software.amazon.awssdk.services.batch.model.DescribeJobQueuesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.DescribeJobQueuesIterable responses = client.describeJobQueuesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeJobQueues(software.amazon.awssdk.services.batch.model.DescribeJobQueuesRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeJobQueuesRequest.Builder} avoiding the need
     * to create one manually via {@link DescribeJobQueuesRequest#builder()}
     * </p>
     *
     * @param describeJobQueuesRequest
     *        A {@link Consumer} that will call methods on {@link DescribeJobQueuesRequest.Builder} to create a request.
     *        Contains the parameters for <code>DescribeJobQueues</code>.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DescribeJobQueues
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobQueues" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeJobQueuesIterable describeJobQueuesPaginator(
            Consumer<DescribeJobQueuesRequest.Builder> describeJobQueuesRequest) throws ClientException, ServerException,
            AwsServiceException, SdkClientException, BatchException {
        return describeJobQueuesPaginator(DescribeJobQueuesRequest.builder().applyMutation(describeJobQueuesRequest).build());
    }

    /**
     * <p>
     * Describes a list of Batch jobs.
     * </p>
     *
     * @param describeJobsRequest
     *        Contains the parameters for <code>DescribeJobs</code>.
     * @return Result of the DescribeJobs operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DescribeJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobs" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeJobsResponse describeJobs(DescribeJobsRequest describeJobsRequest) throws ClientException, ServerException,
            AwsServiceException, SdkClientException, BatchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes a list of Batch jobs.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeJobsRequest.Builder} avoiding the need to
     * create one manually via {@link DescribeJobsRequest#builder()}
     * </p>
     *
     * @param describeJobsRequest
     *        A {@link Consumer} that will call methods on {@link DescribeJobsRequest.Builder} to create a request.
     *        Contains the parameters for <code>DescribeJobs</code>.
     * @return Result of the DescribeJobs operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DescribeJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeJobs" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeJobsResponse describeJobs(Consumer<DescribeJobsRequest.Builder> describeJobsRequest) throws ClientException,
            ServerException, AwsServiceException, SdkClientException, BatchException {
        return describeJobs(DescribeJobsRequest.builder().applyMutation(describeJobsRequest).build());
    }

    /**
     * <p>
     * Describes one or more of your scheduling policies.
     * </p>
     *
     * @param describeSchedulingPoliciesRequest
     *        Contains the parameters for <code>DescribeSchedulingPolicies</code>.
     * @return Result of the DescribeSchedulingPolicies operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DescribeSchedulingPolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeSchedulingPolicies"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeSchedulingPoliciesResponse describeSchedulingPolicies(
            DescribeSchedulingPoliciesRequest describeSchedulingPoliciesRequest) throws ClientException, ServerException,
            AwsServiceException, SdkClientException, BatchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes one or more of your scheduling policies.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeSchedulingPoliciesRequest.Builder} avoiding
     * the need to create one manually via {@link DescribeSchedulingPoliciesRequest#builder()}
     * </p>
     *
     * @param describeSchedulingPoliciesRequest
     *        A {@link Consumer} that will call methods on {@link DescribeSchedulingPoliciesRequest.Builder} to create a
     *        request. Contains the parameters for <code>DescribeSchedulingPolicies</code>.
     * @return Result of the DescribeSchedulingPolicies operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.DescribeSchedulingPolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/DescribeSchedulingPolicies"
     *      target="_top">AWS API Documentation</a>
     */
    default DescribeSchedulingPoliciesResponse describeSchedulingPolicies(
            Consumer<DescribeSchedulingPoliciesRequest.Builder> describeSchedulingPoliciesRequest) throws ClientException,
            ServerException, AwsServiceException, SdkClientException, BatchException {
        return describeSchedulingPolicies(DescribeSchedulingPoliciesRequest.builder()
                .applyMutation(describeSchedulingPoliciesRequest).build());
    }

    /**
     * <p>
     * Returns a list of Batch jobs.
     * </p>
     * <p>
     * You must specify only one of the following items:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A job queue ID to return a list of jobs in that job queue
     * </p>
     * </li>
     * <li>
     * <p>
     * A multi-node parallel job ID to return a list of nodes for that job
     * </p>
     * </li>
     * <li>
     * <p>
     * An array job ID to return a list of the children for that job
     * </p>
     * </li>
     * </ul>
     * <p>
     * You can filter the results by job status with the <code>jobStatus</code> parameter. If you don't specify a
     * status, only <code>RUNNING</code> jobs are returned.
     * </p>
     *
     * @param listJobsRequest
     *        Contains the parameters for <code>ListJobs</code>.
     * @return Result of the ListJobs operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.ListJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    default ListJobsResponse listJobs(ListJobsRequest listJobsRequest) throws ClientException, ServerException,
            AwsServiceException, SdkClientException, BatchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of Batch jobs.
     * </p>
     * <p>
     * You must specify only one of the following items:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A job queue ID to return a list of jobs in that job queue
     * </p>
     * </li>
     * <li>
     * <p>
     * A multi-node parallel job ID to return a list of nodes for that job
     * </p>
     * </li>
     * <li>
     * <p>
     * An array job ID to return a list of the children for that job
     * </p>
     * </li>
     * </ul>
     * <p>
     * You can filter the results by job status with the <code>jobStatus</code> parameter. If you don't specify a
     * status, only <code>RUNNING</code> jobs are returned.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListJobsRequest.Builder} avoiding the need to
     * create one manually via {@link ListJobsRequest#builder()}
     * </p>
     *
     * @param listJobsRequest
     *        A {@link Consumer} that will call methods on {@link ListJobsRequest.Builder} to create a request. Contains
     *        the parameters for <code>ListJobs</code>.
     * @return Result of the ListJobs operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.ListJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    default ListJobsResponse listJobs(Consumer<ListJobsRequest.Builder> listJobsRequest) throws ClientException, ServerException,
            AwsServiceException, SdkClientException, BatchException {
        return listJobs(ListJobsRequest.builder().applyMutation(listJobsRequest).build());
    }

    /**
     * <p>
     * Returns a list of Batch jobs.
     * </p>
     * <p>
     * You must specify only one of the following items:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A job queue ID to return a list of jobs in that job queue
     * </p>
     * </li>
     * <li>
     * <p>
     * A multi-node parallel job ID to return a list of nodes for that job
     * </p>
     * </li>
     * <li>
     * <p>
     * An array job ID to return a list of the children for that job
     * </p>
     * </li>
     * </ul>
     * <p>
     * You can filter the results by job status with the <code>jobStatus</code> parameter. If you don't specify a
     * status, only <code>RUNNING</code> jobs are returned.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listJobs(software.amazon.awssdk.services.batch.model.ListJobsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.ListJobsIterable responses = client.listJobsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.batch.paginators.ListJobsIterable responses = client.listJobsPaginator(request);
     *     for (software.amazon.awssdk.services.batch.model.ListJobsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.ListJobsIterable responses = client.listJobsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listJobs(software.amazon.awssdk.services.batch.model.ListJobsRequest)} operation.</b>
     * </p>
     *
     * @param listJobsRequest
     *        Contains the parameters for <code>ListJobs</code>.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.ListJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    default ListJobsIterable listJobsPaginator(ListJobsRequest listJobsRequest) throws ClientException, ServerException,
            AwsServiceException, SdkClientException, BatchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of Batch jobs.
     * </p>
     * <p>
     * You must specify only one of the following items:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A job queue ID to return a list of jobs in that job queue
     * </p>
     * </li>
     * <li>
     * <p>
     * A multi-node parallel job ID to return a list of nodes for that job
     * </p>
     * </li>
     * <li>
     * <p>
     * An array job ID to return a list of the children for that job
     * </p>
     * </li>
     * </ul>
     * <p>
     * You can filter the results by job status with the <code>jobStatus</code> parameter. If you don't specify a
     * status, only <code>RUNNING</code> jobs are returned.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listJobs(software.amazon.awssdk.services.batch.model.ListJobsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.ListJobsIterable responses = client.listJobsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.batch.paginators.ListJobsIterable responses = client.listJobsPaginator(request);
     *     for (software.amazon.awssdk.services.batch.model.ListJobsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.ListJobsIterable responses = client.listJobsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listJobs(software.amazon.awssdk.services.batch.model.ListJobsRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListJobsRequest.Builder} avoiding the need to
     * create one manually via {@link ListJobsRequest#builder()}
     * </p>
     *
     * @param listJobsRequest
     *        A {@link Consumer} that will call methods on {@link ListJobsRequest.Builder} to create a request. Contains
     *        the parameters for <code>ListJobs</code>.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.ListJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    default ListJobsIterable listJobsPaginator(Consumer<ListJobsRequest.Builder> listJobsRequest) throws ClientException,
            ServerException, AwsServiceException, SdkClientException, BatchException {
        return listJobsPaginator(ListJobsRequest.builder().applyMutation(listJobsRequest).build());
    }

    /**
     * <p>
     * Returns a list of Batch scheduling policies.
     * </p>
     *
     * @param listSchedulingPoliciesRequest
     *        Contains the parameters for <code>ListSchedulingPolicies</code>.
     * @return Result of the ListSchedulingPolicies operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.ListSchedulingPolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/ListSchedulingPolicies" target="_top">AWS
     *      API Documentation</a>
     */
    default ListSchedulingPoliciesResponse listSchedulingPolicies(ListSchedulingPoliciesRequest listSchedulingPoliciesRequest)
            throws ClientException, ServerException, AwsServiceException, SdkClientException, BatchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of Batch scheduling policies.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListSchedulingPoliciesRequest.Builder} avoiding the
     * need to create one manually via {@link ListSchedulingPoliciesRequest#builder()}
     * </p>
     *
     * @param listSchedulingPoliciesRequest
     *        A {@link Consumer} that will call methods on {@link ListSchedulingPoliciesRequest.Builder} to create a
     *        request. Contains the parameters for <code>ListSchedulingPolicies</code>.
     * @return Result of the ListSchedulingPolicies operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.ListSchedulingPolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/ListSchedulingPolicies" target="_top">AWS
     *      API Documentation</a>
     */
    default ListSchedulingPoliciesResponse listSchedulingPolicies(
            Consumer<ListSchedulingPoliciesRequest.Builder> listSchedulingPoliciesRequest) throws ClientException,
            ServerException, AwsServiceException, SdkClientException, BatchException {
        return listSchedulingPolicies(ListSchedulingPoliciesRequest.builder().applyMutation(listSchedulingPoliciesRequest)
                .build());
    }

    /**
     * <p>
     * Returns a list of Batch scheduling policies.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSchedulingPolicies(software.amazon.awssdk.services.batch.model.ListSchedulingPoliciesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.ListSchedulingPoliciesIterable responses = client.listSchedulingPoliciesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.batch.paginators.ListSchedulingPoliciesIterable responses = client
     *             .listSchedulingPoliciesPaginator(request);
     *     for (software.amazon.awssdk.services.batch.model.ListSchedulingPoliciesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.ListSchedulingPoliciesIterable responses = client.listSchedulingPoliciesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSchedulingPolicies(software.amazon.awssdk.services.batch.model.ListSchedulingPoliciesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSchedulingPoliciesRequest
     *        Contains the parameters for <code>ListSchedulingPolicies</code>.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.ListSchedulingPolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/ListSchedulingPolicies" target="_top">AWS
     *      API Documentation</a>
     */
    default ListSchedulingPoliciesIterable listSchedulingPoliciesPaginator(
            ListSchedulingPoliciesRequest listSchedulingPoliciesRequest) throws ClientException, ServerException,
            AwsServiceException, SdkClientException, BatchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of Batch scheduling policies.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSchedulingPolicies(software.amazon.awssdk.services.batch.model.ListSchedulingPoliciesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.ListSchedulingPoliciesIterable responses = client.listSchedulingPoliciesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.batch.paginators.ListSchedulingPoliciesIterable responses = client
     *             .listSchedulingPoliciesPaginator(request);
     *     for (software.amazon.awssdk.services.batch.model.ListSchedulingPoliciesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.batch.paginators.ListSchedulingPoliciesIterable responses = client.listSchedulingPoliciesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSchedulingPolicies(software.amazon.awssdk.services.batch.model.ListSchedulingPoliciesRequest)}
     * operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListSchedulingPoliciesRequest.Builder} avoiding the
     * need to create one manually via {@link ListSchedulingPoliciesRequest#builder()}
     * </p>
     *
     * @param listSchedulingPoliciesRequest
     *        A {@link Consumer} that will call methods on {@link ListSchedulingPoliciesRequest.Builder} to create a
     *        request. Contains the parameters for <code>ListSchedulingPolicies</code>.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.ListSchedulingPolicies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/ListSchedulingPolicies" target="_top">AWS
     *      API Documentation</a>
     */
    default ListSchedulingPoliciesIterable listSchedulingPoliciesPaginator(
            Consumer<ListSchedulingPoliciesRequest.Builder> listSchedulingPoliciesRequest) throws ClientException,
            ServerException, AwsServiceException, SdkClientException, BatchException {
        return listSchedulingPoliciesPaginator(ListSchedulingPoliciesRequest.builder()
                .applyMutation(listSchedulingPoliciesRequest).build());
    }

    /**
     * <p>
     * Lists the tags for an Batch resource. Batch resources that support tags are compute environments, jobs, job
     * definitions, job queues, and scheduling policies. ARNs for child jobs of array and multi-node parallel (MNP) jobs
     * aren't supported.
     * </p>
     *
     * @param listTagsForResourceRequest
     *        Contains the parameters for <code>ListTagsForResource</code>.
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ClientException, ServerException, AwsServiceException, SdkClientException, BatchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the tags for an Batch resource. Batch resources that support tags are compute environments, jobs, job
     * definitions, job queues, and scheduling policies. ARNs for child jobs of array and multi-node parallel (MNP) jobs
     * aren't supported.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on {@link ListTagsForResourceRequest.Builder} to create a
     *        request. Contains the parameters for <code>ListTagsForResource</code>.
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) throws ClientException, ServerException,
            AwsServiceException, SdkClientException, BatchException {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * Registers an Batch job definition.
     * </p>
     *
     * @param registerJobDefinitionRequest
     *        Contains the parameters for <code>RegisterJobDefinition</code>.
     * @return Result of the RegisterJobDefinition operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.RegisterJobDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/RegisterJobDefinition" target="_top">AWS
     *      API Documentation</a>
     */
    default RegisterJobDefinitionResponse registerJobDefinition(RegisterJobDefinitionRequest registerJobDefinitionRequest)
            throws ClientException, ServerException, AwsServiceException, SdkClientException, BatchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Registers an Batch job definition.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link RegisterJobDefinitionRequest.Builder} avoiding the
     * need to create one manually via {@link RegisterJobDefinitionRequest#builder()}
     * </p>
     *
     * @param registerJobDefinitionRequest
     *        A {@link Consumer} that will call methods on {@link RegisterJobDefinitionRequest.Builder} to create a
     *        request. Contains the parameters for <code>RegisterJobDefinition</code>.
     * @return Result of the RegisterJobDefinition operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.RegisterJobDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/RegisterJobDefinition" target="_top">AWS
     *      API Documentation</a>
     */
    default RegisterJobDefinitionResponse registerJobDefinition(
            Consumer<RegisterJobDefinitionRequest.Builder> registerJobDefinitionRequest) throws ClientException, ServerException,
            AwsServiceException, SdkClientException, BatchException {
        return registerJobDefinition(RegisterJobDefinitionRequest.builder().applyMutation(registerJobDefinitionRequest).build());
    }

    /**
     * <p>
     * Submits an Batch job from a job definition. Parameters that are specified during <a>SubmitJob</a> override
     * parameters defined in the job definition. vCPU and memory requirements that are specified in the
     * <code>resourceRequirements</code> objects in the job definition are the exception. They can't be overridden this
     * way using the <code>memory</code> and <code>vcpus</code> parameters. Rather, you must specify updates to job
     * definition parameters in a <code>resourceRequirements</code> object that's included in the
     * <code>containerOverrides</code> parameter.
     * </p>
     * <note>
     * <p>
     * Job queues with a scheduling policy are limited to 500 active fair share identifiers at a time.
     * </p>
     * </note> <important>
     * <p>
     * Jobs that run on Fargate resources can't be guaranteed to run for more than 14 days. This is because, after 14
     * days, Fargate resources might become unavailable and job might be terminated.
     * </p>
     * </important>
     *
     * @param submitJobRequest
     *        Contains the parameters for <code>SubmitJob</code>.
     * @return Result of the SubmitJob operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.SubmitJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/SubmitJob" target="_top">AWS API
     *      Documentation</a>
     */
    default SubmitJobResponse submitJob(SubmitJobRequest submitJobRequest) throws ClientException, ServerException,
            AwsServiceException, SdkClientException, BatchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Submits an Batch job from a job definition. Parameters that are specified during <a>SubmitJob</a> override
     * parameters defined in the job definition. vCPU and memory requirements that are specified in the
     * <code>resourceRequirements</code> objects in the job definition are the exception. They can't be overridden this
     * way using the <code>memory</code> and <code>vcpus</code> parameters. Rather, you must specify updates to job
     * definition parameters in a <code>resourceRequirements</code> object that's included in the
     * <code>containerOverrides</code> parameter.
     * </p>
     * <note>
     * <p>
     * Job queues with a scheduling policy are limited to 500 active fair share identifiers at a time.
     * </p>
     * </note> <important>
     * <p>
     * Jobs that run on Fargate resources can't be guaranteed to run for more than 14 days. This is because, after 14
     * days, Fargate resources might become unavailable and job might be terminated.
     * </p>
     * </important><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link SubmitJobRequest.Builder} avoiding the need to
     * create one manually via {@link SubmitJobRequest#builder()}
     * </p>
     *
     * @param submitJobRequest
     *        A {@link Consumer} that will call methods on {@link SubmitJobRequest.Builder} to create a request.
     *        Contains the parameters for <code>SubmitJob</code>.
     * @return Result of the SubmitJob operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.SubmitJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/SubmitJob" target="_top">AWS API
     *      Documentation</a>
     */
    default SubmitJobResponse submitJob(Consumer<SubmitJobRequest.Builder> submitJobRequest) throws ClientException,
            ServerException, AwsServiceException, SdkClientException, BatchException {
        return submitJob(SubmitJobRequest.builder().applyMutation(submitJobRequest).build());
    }

    /**
     * <p>
     * Associates the specified tags to a resource with the specified <code>resourceArn</code>. If existing tags on a
     * resource aren't specified in the request parameters, they aren't changed. When a resource is deleted, the tags
     * that are associated with that resource are deleted as well. Batch resources that support tags are compute
     * environments, jobs, job definitions, job queues, and scheduling policies. ARNs for child jobs of array and
     * multi-node parallel (MNP) jobs aren't supported.
     * </p>
     *
     * @param tagResourceRequest
     *        Contains the parameters for <code>TagResource</code>.
     * @return Result of the TagResource operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ClientException, ServerException,
            AwsServiceException, SdkClientException, BatchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Associates the specified tags to a resource with the specified <code>resourceArn</code>. If existing tags on a
     * resource aren't specified in the request parameters, they aren't changed. When a resource is deleted, the tags
     * that are associated with that resource are deleted as well. Batch resources that support tags are compute
     * environments, jobs, job definitions, job queues, and scheduling policies. ARNs for child jobs of array and
     * multi-node parallel (MNP) jobs aren't supported.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on {@link TagResourceRequest.Builder} to create a request.
     *        Contains the parameters for <code>TagResource</code>.
     * @return Result of the TagResource operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default TagResourceResponse tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest) throws ClientException,
            ServerException, AwsServiceException, SdkClientException, BatchException {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * Terminates a job in a job queue. Jobs that are in the <code>STARTING</code> or <code>RUNNING</code> state are
     * terminated, which causes them to transition to <code>FAILED</code>. Jobs that have not progressed to the
     * <code>STARTING</code> state are cancelled.
     * </p>
     *
     * @param terminateJobRequest
     *        Contains the parameters for <code>TerminateJob</code>.
     * @return Result of the TerminateJob operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.TerminateJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/TerminateJob" target="_top">AWS API
     *      Documentation</a>
     */
    default TerminateJobResponse terminateJob(TerminateJobRequest terminateJobRequest) throws ClientException, ServerException,
            AwsServiceException, SdkClientException, BatchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Terminates a job in a job queue. Jobs that are in the <code>STARTING</code> or <code>RUNNING</code> state are
     * terminated, which causes them to transition to <code>FAILED</code>. Jobs that have not progressed to the
     * <code>STARTING</code> state are cancelled.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TerminateJobRequest.Builder} avoiding the need to
     * create one manually via {@link TerminateJobRequest#builder()}
     * </p>
     *
     * @param terminateJobRequest
     *        A {@link Consumer} that will call methods on {@link TerminateJobRequest.Builder} to create a request.
     *        Contains the parameters for <code>TerminateJob</code>.
     * @return Result of the TerminateJob operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.TerminateJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/TerminateJob" target="_top">AWS API
     *      Documentation</a>
     */
    default TerminateJobResponse terminateJob(Consumer<TerminateJobRequest.Builder> terminateJobRequest) throws ClientException,
            ServerException, AwsServiceException, SdkClientException, BatchException {
        return terminateJob(TerminateJobRequest.builder().applyMutation(terminateJobRequest).build());
    }

    /**
     * <p>
     * Deletes specified tags from an Batch resource.
     * </p>
     *
     * @param untagResourceRequest
     *        Contains the parameters for <code>UntagResource</code>.
     * @return Result of the UntagResource operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ClientException,
            ServerException, AwsServiceException, SdkClientException, BatchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes specified tags from an Batch resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on {@link UntagResourceRequest.Builder} to create a request.
     *        Contains the parameters for <code>UntagResource</code>.
     * @return Result of the UntagResource operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default UntagResourceResponse untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest)
            throws ClientException, ServerException, AwsServiceException, SdkClientException, BatchException {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    /**
     * <p>
     * Updates an Batch compute environment.
     * </p>
     *
     * @param updateComputeEnvironmentRequest
     *        Contains the parameters for <code>UpdateComputeEnvironment</code>.
     * @return Result of the UpdateComputeEnvironment operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.UpdateComputeEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/UpdateComputeEnvironment"
     *      target="_top">AWS API Documentation</a>
     */
    default UpdateComputeEnvironmentResponse updateComputeEnvironment(
            UpdateComputeEnvironmentRequest updateComputeEnvironmentRequest) throws ClientException, ServerException,
            AwsServiceException, SdkClientException, BatchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates an Batch compute environment.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateComputeEnvironmentRequest.Builder} avoiding
     * the need to create one manually via {@link UpdateComputeEnvironmentRequest#builder()}
     * </p>
     *
     * @param updateComputeEnvironmentRequest
     *        A {@link Consumer} that will call methods on {@link UpdateComputeEnvironmentRequest.Builder} to create a
     *        request. Contains the parameters for <code>UpdateComputeEnvironment</code>.
     * @return Result of the UpdateComputeEnvironment operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.UpdateComputeEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/UpdateComputeEnvironment"
     *      target="_top">AWS API Documentation</a>
     */
    default UpdateComputeEnvironmentResponse updateComputeEnvironment(
            Consumer<UpdateComputeEnvironmentRequest.Builder> updateComputeEnvironmentRequest) throws ClientException,
            ServerException, AwsServiceException, SdkClientException, BatchException {
        return updateComputeEnvironment(UpdateComputeEnvironmentRequest.builder().applyMutation(updateComputeEnvironmentRequest)
                .build());
    }

    /**
     * <p>
     * Updates a job queue.
     * </p>
     *
     * @param updateJobQueueRequest
     *        Contains the parameters for <code>UpdateJobQueue</code>.
     * @return Result of the UpdateJobQueue operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.UpdateJobQueue
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/UpdateJobQueue" target="_top">AWS API
     *      Documentation</a>
     */
    default UpdateJobQueueResponse updateJobQueue(UpdateJobQueueRequest updateJobQueueRequest) throws ClientException,
            ServerException, AwsServiceException, SdkClientException, BatchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates a job queue.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateJobQueueRequest.Builder} avoiding the need to
     * create one manually via {@link UpdateJobQueueRequest#builder()}
     * </p>
     *
     * @param updateJobQueueRequest
     *        A {@link Consumer} that will call methods on {@link UpdateJobQueueRequest.Builder} to create a request.
     *        Contains the parameters for <code>UpdateJobQueue</code>.
     * @return Result of the UpdateJobQueue operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.UpdateJobQueue
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/UpdateJobQueue" target="_top">AWS API
     *      Documentation</a>
     */
    default UpdateJobQueueResponse updateJobQueue(Consumer<UpdateJobQueueRequest.Builder> updateJobQueueRequest)
            throws ClientException, ServerException, AwsServiceException, SdkClientException, BatchException {
        return updateJobQueue(UpdateJobQueueRequest.builder().applyMutation(updateJobQueueRequest).build());
    }

    /**
     * <p>
     * Updates a scheduling policy.
     * </p>
     *
     * @param updateSchedulingPolicyRequest
     *        Contains the parameters for <code>UpdateSchedulingPolicy</code>.
     * @return Result of the UpdateSchedulingPolicy operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.UpdateSchedulingPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/UpdateSchedulingPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    default UpdateSchedulingPolicyResponse updateSchedulingPolicy(UpdateSchedulingPolicyRequest updateSchedulingPolicyRequest)
            throws ClientException, ServerException, AwsServiceException, SdkClientException, BatchException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates a scheduling policy.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateSchedulingPolicyRequest.Builder} avoiding the
     * need to create one manually via {@link UpdateSchedulingPolicyRequest#builder()}
     * </p>
     *
     * @param updateSchedulingPolicyRequest
     *        A {@link Consumer} that will call methods on {@link UpdateSchedulingPolicyRequest.Builder} to create a
     *        request. Contains the parameters for <code>UpdateSchedulingPolicy</code>.
     * @return Result of the UpdateSchedulingPolicy operation returned by the service.
     * @throws ClientException
     *         These errors are usually caused by a client action. One example cause is using an action or resource on
     *         behalf of a user that doesn't have permissions to use the action or resource. Another cause is specifying
     *         an identifier that's not valid.
     * @throws ServerException
     *         These errors are usually caused by a server issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BatchClient.UpdateSchedulingPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/batch-2016-08-10/UpdateSchedulingPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    default UpdateSchedulingPolicyResponse updateSchedulingPolicy(
            Consumer<UpdateSchedulingPolicyRequest.Builder> updateSchedulingPolicyRequest) throws ClientException,
            ServerException, AwsServiceException, SdkClientException, BatchException {
        return updateSchedulingPolicy(UpdateSchedulingPolicyRequest.builder().applyMutation(updateSchedulingPolicyRequest)
                .build());
    }

    /**
     * Create a {@link BatchClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static BatchClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link BatchClient}.
     */
    static BatchClientBuilder builder() {
        return new DefaultBatchClientBuilder();
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }
}
