/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.batch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object that represents an Batch job.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class JobDetail implements SdkPojo, Serializable, ToCopyableBuilder<JobDetail.Builder, JobDetail> {
    private static final SdkField<String> JOB_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("jobArn")
            .getter(getter(JobDetail::jobArn)).setter(setter(Builder::jobArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("jobArn").build()).build();

    private static final SdkField<String> JOB_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("jobName").getter(getter(JobDetail::jobName)).setter(setter(Builder::jobName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("jobName").build()).build();

    private static final SdkField<String> JOB_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("jobId")
            .getter(getter(JobDetail::jobId)).setter(setter(Builder::jobId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("jobId").build()).build();

    private static final SdkField<String> JOB_QUEUE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("jobQueue").getter(getter(JobDetail::jobQueue)).setter(setter(Builder::jobQueue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("jobQueue").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(JobDetail::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<String> SHARE_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("shareIdentifier").getter(getter(JobDetail::shareIdentifier)).setter(setter(Builder::shareIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("shareIdentifier").build()).build();

    private static final SdkField<Integer> SCHEDULING_PRIORITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("schedulingPriority").getter(getter(JobDetail::schedulingPriority))
            .setter(setter(Builder::schedulingPriority))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("schedulingPriority").build())
            .build();

    private static final SdkField<List<AttemptDetail>> ATTEMPTS_FIELD = SdkField
            .<List<AttemptDetail>> builder(MarshallingType.LIST)
            .memberName("attempts")
            .getter(getter(JobDetail::attempts))
            .setter(setter(Builder::attempts))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("attempts").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AttemptDetail> builder(MarshallingType.SDK_POJO)
                                            .constructor(AttemptDetail::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STATUS_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("statusReason").getter(getter(JobDetail::statusReason)).setter(setter(Builder::statusReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statusReason").build()).build();

    private static final SdkField<Long> CREATED_AT_FIELD = SdkField.<Long> builder(MarshallingType.LONG).memberName("createdAt")
            .getter(getter(JobDetail::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<RetryStrategy> RETRY_STRATEGY_FIELD = SdkField
            .<RetryStrategy> builder(MarshallingType.SDK_POJO).memberName("retryStrategy")
            .getter(getter(JobDetail::retryStrategy)).setter(setter(Builder::retryStrategy)).constructor(RetryStrategy::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("retryStrategy").build()).build();

    private static final SdkField<Long> STARTED_AT_FIELD = SdkField.<Long> builder(MarshallingType.LONG).memberName("startedAt")
            .getter(getter(JobDetail::startedAt)).setter(setter(Builder::startedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startedAt").build()).build();

    private static final SdkField<Long> STOPPED_AT_FIELD = SdkField.<Long> builder(MarshallingType.LONG).memberName("stoppedAt")
            .getter(getter(JobDetail::stoppedAt)).setter(setter(Builder::stoppedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("stoppedAt").build()).build();

    private static final SdkField<List<JobDependency>> DEPENDS_ON_FIELD = SdkField
            .<List<JobDependency>> builder(MarshallingType.LIST)
            .memberName("dependsOn")
            .getter(getter(JobDetail::dependsOn))
            .setter(setter(Builder::dependsOn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dependsOn").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<JobDependency> builder(MarshallingType.SDK_POJO)
                                            .constructor(JobDependency::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> JOB_DEFINITION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("jobDefinition").getter(getter(JobDetail::jobDefinition)).setter(setter(Builder::jobDefinition))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("jobDefinition").build()).build();

    private static final SdkField<Map<String, String>> PARAMETERS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("parameters")
            .getter(getter(JobDetail::parameters))
            .setter(setter(Builder::parameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("parameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<ContainerDetail> CONTAINER_FIELD = SdkField.<ContainerDetail> builder(MarshallingType.SDK_POJO)
            .memberName("container").getter(getter(JobDetail::container)).setter(setter(Builder::container))
            .constructor(ContainerDetail::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("container").build()).build();

    private static final SdkField<NodeDetails> NODE_DETAILS_FIELD = SdkField.<NodeDetails> builder(MarshallingType.SDK_POJO)
            .memberName("nodeDetails").getter(getter(JobDetail::nodeDetails)).setter(setter(Builder::nodeDetails))
            .constructor(NodeDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("nodeDetails").build()).build();

    private static final SdkField<NodeProperties> NODE_PROPERTIES_FIELD = SdkField
            .<NodeProperties> builder(MarshallingType.SDK_POJO).memberName("nodeProperties")
            .getter(getter(JobDetail::nodeProperties)).setter(setter(Builder::nodeProperties))
            .constructor(NodeProperties::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("nodeProperties").build()).build();

    private static final SdkField<ArrayPropertiesDetail> ARRAY_PROPERTIES_FIELD = SdkField
            .<ArrayPropertiesDetail> builder(MarshallingType.SDK_POJO).memberName("arrayProperties")
            .getter(getter(JobDetail::arrayProperties)).setter(setter(Builder::arrayProperties))
            .constructor(ArrayPropertiesDetail::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arrayProperties").build()).build();

    private static final SdkField<JobTimeout> TIMEOUT_FIELD = SdkField.<JobTimeout> builder(MarshallingType.SDK_POJO)
            .memberName("timeout").getter(getter(JobDetail::timeout)).setter(setter(Builder::timeout))
            .constructor(JobTimeout::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("timeout").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(JobDetail::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Boolean> PROPAGATE_TAGS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("propagateTags").getter(getter(JobDetail::propagateTags)).setter(setter(Builder::propagateTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("propagateTags").build()).build();

    private static final SdkField<List<String>> PLATFORM_CAPABILITIES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("platformCapabilities")
            .getter(getter(JobDetail::platformCapabilitiesAsStrings))
            .setter(setter(Builder::platformCapabilitiesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("platformCapabilities").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<EksPropertiesDetail> EKS_PROPERTIES_FIELD = SdkField
            .<EksPropertiesDetail> builder(MarshallingType.SDK_POJO).memberName("eksProperties")
            .getter(getter(JobDetail::eksProperties)).setter(setter(Builder::eksProperties))
            .constructor(EksPropertiesDetail::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eksProperties").build()).build();

    private static final SdkField<List<EksAttemptDetail>> EKS_ATTEMPTS_FIELD = SdkField
            .<List<EksAttemptDetail>> builder(MarshallingType.LIST)
            .memberName("eksAttempts")
            .getter(getter(JobDetail::eksAttempts))
            .setter(setter(Builder::eksAttempts))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eksAttempts").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<EksAttemptDetail> builder(MarshallingType.SDK_POJO)
                                            .constructor(EksAttemptDetail::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<EcsPropertiesDetail> ECS_PROPERTIES_FIELD = SdkField
            .<EcsPropertiesDetail> builder(MarshallingType.SDK_POJO).memberName("ecsProperties")
            .getter(getter(JobDetail::ecsProperties)).setter(setter(Builder::ecsProperties))
            .constructor(EcsPropertiesDetail::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ecsProperties").build()).build();

    private static final SdkField<Boolean> IS_CANCELLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("isCancelled").getter(getter(JobDetail::isCancelled)).setter(setter(Builder::isCancelled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("isCancelled").build()).build();

    private static final SdkField<Boolean> IS_TERMINATED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("isTerminated").getter(getter(JobDetail::isTerminated)).setter(setter(Builder::isTerminated))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("isTerminated").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(JOB_ARN_FIELD, JOB_NAME_FIELD,
            JOB_ID_FIELD, JOB_QUEUE_FIELD, STATUS_FIELD, SHARE_IDENTIFIER_FIELD, SCHEDULING_PRIORITY_FIELD, ATTEMPTS_FIELD,
            STATUS_REASON_FIELD, CREATED_AT_FIELD, RETRY_STRATEGY_FIELD, STARTED_AT_FIELD, STOPPED_AT_FIELD, DEPENDS_ON_FIELD,
            JOB_DEFINITION_FIELD, PARAMETERS_FIELD, CONTAINER_FIELD, NODE_DETAILS_FIELD, NODE_PROPERTIES_FIELD,
            ARRAY_PROPERTIES_FIELD, TIMEOUT_FIELD, TAGS_FIELD, PROPAGATE_TAGS_FIELD, PLATFORM_CAPABILITIES_FIELD,
            EKS_PROPERTIES_FIELD, EKS_ATTEMPTS_FIELD, ECS_PROPERTIES_FIELD, IS_CANCELLED_FIELD, IS_TERMINATED_FIELD));

    private static final long serialVersionUID = 1L;

    private final String jobArn;

    private final String jobName;

    private final String jobId;

    private final String jobQueue;

    private final String status;

    private final String shareIdentifier;

    private final Integer schedulingPriority;

    private final List<AttemptDetail> attempts;

    private final String statusReason;

    private final Long createdAt;

    private final RetryStrategy retryStrategy;

    private final Long startedAt;

    private final Long stoppedAt;

    private final List<JobDependency> dependsOn;

    private final String jobDefinition;

    private final Map<String, String> parameters;

    private final ContainerDetail container;

    private final NodeDetails nodeDetails;

    private final NodeProperties nodeProperties;

    private final ArrayPropertiesDetail arrayProperties;

    private final JobTimeout timeout;

    private final Map<String, String> tags;

    private final Boolean propagateTags;

    private final List<String> platformCapabilities;

    private final EksPropertiesDetail eksProperties;

    private final List<EksAttemptDetail> eksAttempts;

    private final EcsPropertiesDetail ecsProperties;

    private final Boolean isCancelled;

    private final Boolean isTerminated;

    private JobDetail(BuilderImpl builder) {
        this.jobArn = builder.jobArn;
        this.jobName = builder.jobName;
        this.jobId = builder.jobId;
        this.jobQueue = builder.jobQueue;
        this.status = builder.status;
        this.shareIdentifier = builder.shareIdentifier;
        this.schedulingPriority = builder.schedulingPriority;
        this.attempts = builder.attempts;
        this.statusReason = builder.statusReason;
        this.createdAt = builder.createdAt;
        this.retryStrategy = builder.retryStrategy;
        this.startedAt = builder.startedAt;
        this.stoppedAt = builder.stoppedAt;
        this.dependsOn = builder.dependsOn;
        this.jobDefinition = builder.jobDefinition;
        this.parameters = builder.parameters;
        this.container = builder.container;
        this.nodeDetails = builder.nodeDetails;
        this.nodeProperties = builder.nodeProperties;
        this.arrayProperties = builder.arrayProperties;
        this.timeout = builder.timeout;
        this.tags = builder.tags;
        this.propagateTags = builder.propagateTags;
        this.platformCapabilities = builder.platformCapabilities;
        this.eksProperties = builder.eksProperties;
        this.eksAttempts = builder.eksAttempts;
        this.ecsProperties = builder.ecsProperties;
        this.isCancelled = builder.isCancelled;
        this.isTerminated = builder.isTerminated;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the job.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the job.
     */
    public final String jobArn() {
        return jobArn;
    }

    /**
     * <p>
     * The job name.
     * </p>
     * 
     * @return The job name.
     */
    public final String jobName() {
        return jobName;
    }

    /**
     * <p>
     * The job ID.
     * </p>
     * 
     * @return The job ID.
     */
    public final String jobId() {
        return jobId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the job queue that the job is associated with.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the job queue that the job is associated with.
     */
    public final String jobQueue() {
        return jobQueue;
    }

    /**
     * <p>
     * The current status for the job.
     * </p>
     * <note>
     * <p>
     * If your jobs don't progress to <code>STARTING</code>, see <a
     * href="https://docs.aws.amazon.com/batch/latest/userguide/troubleshooting.html#job_stuck_in_runnable">Jobs stuck
     * in RUNNABLE status</a> in the troubleshooting section of the <i>Batch User Guide</i>.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link JobStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status for the job.</p> <note>
     *         <p>
     *         If your jobs don't progress to <code>STARTING</code>, see <a
     *         href="https://docs.aws.amazon.com/batch/latest/userguide/troubleshooting.html#job_stuck_in_runnable">Jobs
     *         stuck in RUNNABLE status</a> in the troubleshooting section of the <i>Batch User Guide</i>.
     *         </p>
     * @see JobStatus
     */
    public final JobStatus status() {
        return JobStatus.fromValue(status);
    }

    /**
     * <p>
     * The current status for the job.
     * </p>
     * <note>
     * <p>
     * If your jobs don't progress to <code>STARTING</code>, see <a
     * href="https://docs.aws.amazon.com/batch/latest/userguide/troubleshooting.html#job_stuck_in_runnable">Jobs stuck
     * in RUNNABLE status</a> in the troubleshooting section of the <i>Batch User Guide</i>.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link JobStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status for the job.</p> <note>
     *         <p>
     *         If your jobs don't progress to <code>STARTING</code>, see <a
     *         href="https://docs.aws.amazon.com/batch/latest/userguide/troubleshooting.html#job_stuck_in_runnable">Jobs
     *         stuck in RUNNABLE status</a> in the troubleshooting section of the <i>Batch User Guide</i>.
     *         </p>
     * @see JobStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The share identifier for the job.
     * </p>
     * 
     * @return The share identifier for the job.
     */
    public final String shareIdentifier() {
        return shareIdentifier;
    }

    /**
     * <p>
     * The scheduling policy of the job definition. This only affects jobs in job queues with a fair share policy. Jobs
     * with a higher scheduling priority are scheduled before jobs with a lower scheduling priority.
     * </p>
     * 
     * @return The scheduling policy of the job definition. This only affects jobs in job queues with a fair share
     *         policy. Jobs with a higher scheduling priority are scheduled before jobs with a lower scheduling
     *         priority.
     */
    public final Integer schedulingPriority() {
        return schedulingPriority;
    }

    /**
     * For responses, this returns true if the service returned a value for the Attempts property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAttempts() {
        return attempts != null && !(attempts instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of job attempts that are associated with this job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAttempts} method.
     * </p>
     * 
     * @return A list of job attempts that are associated with this job.
     */
    public final List<AttemptDetail> attempts() {
        return attempts;
    }

    /**
     * <p>
     * A short, human-readable string to provide more details for the current status of the job.
     * </p>
     * 
     * @return A short, human-readable string to provide more details for the current status of the job.
     */
    public final String statusReason() {
        return statusReason;
    }

    /**
     * <p>
     * The Unix timestamp (in milliseconds) for when the job was created. For non-array jobs and parent array jobs, this
     * is when the job entered the <code>SUBMITTED</code> state. This is specifically at the time <a
     * href="https://docs.aws.amazon.com/batch/latest/APIReference/API_SubmitJob.html">SubmitJob</a> was called. For
     * array child jobs, this is when the child job was spawned by its parent and entered the <code>PENDING</code>
     * state.
     * </p>
     * 
     * @return The Unix timestamp (in milliseconds) for when the job was created. For non-array jobs and parent array
     *         jobs, this is when the job entered the <code>SUBMITTED</code> state. This is specifically at the time <a
     *         href="https://docs.aws.amazon.com/batch/latest/APIReference/API_SubmitJob.html">SubmitJob</a> was called.
     *         For array child jobs, this is when the child job was spawned by its parent and entered the
     *         <code>PENDING</code> state.
     */
    public final Long createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The retry strategy to use for this job if an attempt fails.
     * </p>
     * 
     * @return The retry strategy to use for this job if an attempt fails.
     */
    public final RetryStrategy retryStrategy() {
        return retryStrategy;
    }

    /**
     * <p>
     * The Unix timestamp (in milliseconds) for when the job was started. More specifically, it's when the job
     * transitioned from the <code>STARTING</code> state to the <code>RUNNING</code> state.
     * </p>
     * 
     * @return The Unix timestamp (in milliseconds) for when the job was started. More specifically, it's when the job
     *         transitioned from the <code>STARTING</code> state to the <code>RUNNING</code> state.
     */
    public final Long startedAt() {
        return startedAt;
    }

    /**
     * <p>
     * The Unix timestamp (in milliseconds) for when the job was stopped. More specifically, it's when the job
     * transitioned from the <code>RUNNING</code> state to a terminal state, such as <code>SUCCEEDED</code> or
     * <code>FAILED</code>.
     * </p>
     * 
     * @return The Unix timestamp (in milliseconds) for when the job was stopped. More specifically, it's when the job
     *         transitioned from the <code>RUNNING</code> state to a terminal state, such as <code>SUCCEEDED</code> or
     *         <code>FAILED</code>.
     */
    public final Long stoppedAt() {
        return stoppedAt;
    }

    /**
     * For responses, this returns true if the service returned a value for the DependsOn property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDependsOn() {
        return dependsOn != null && !(dependsOn instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of job IDs that this job depends on.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDependsOn} method.
     * </p>
     * 
     * @return A list of job IDs that this job depends on.
     */
    public final List<JobDependency> dependsOn() {
        return dependsOn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the job definition that this job uses.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the job definition that this job uses.
     */
    public final String jobDefinition() {
        return jobDefinition;
    }

    /**
     * For responses, this returns true if the service returned a value for the Parameters property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasParameters() {
        return parameters != null && !(parameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Additional parameters that are passed to the job that replace parameter substitution placeholders or override any
     * corresponding parameter defaults from the job definition.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasParameters} method.
     * </p>
     * 
     * @return Additional parameters that are passed to the job that replace parameter substitution placeholders or
     *         override any corresponding parameter defaults from the job definition.
     */
    public final Map<String, String> parameters() {
        return parameters;
    }

    /**
     * <p>
     * An object that represents the details for the container that's associated with the job. If the details are for a
     * multiple-container job, this object will be empty.
     * </p>
     * 
     * @return An object that represents the details for the container that's associated with the job. If the details
     *         are for a multiple-container job, this object will be empty.
     */
    public final ContainerDetail container() {
        return container;
    }

    /**
     * <p>
     * An object that represents the details of a node that's associated with a multi-node parallel job.
     * </p>
     * 
     * @return An object that represents the details of a node that's associated with a multi-node parallel job.
     */
    public final NodeDetails nodeDetails() {
        return nodeDetails;
    }

    /**
     * <p>
     * An object that represents the node properties of a multi-node parallel job.
     * </p>
     * <note>
     * <p>
     * This isn't applicable to jobs that are running on Fargate resources.
     * </p>
     * </note>
     * 
     * @return An object that represents the node properties of a multi-node parallel job.</p> <note>
     *         <p>
     *         This isn't applicable to jobs that are running on Fargate resources.
     *         </p>
     */
    public final NodeProperties nodeProperties() {
        return nodeProperties;
    }

    /**
     * <p>
     * The array properties of the job, if it's an array job.
     * </p>
     * 
     * @return The array properties of the job, if it's an array job.
     */
    public final ArrayPropertiesDetail arrayProperties() {
        return arrayProperties;
    }

    /**
     * <p>
     * The timeout configuration for the job.
     * </p>
     * 
     * @return The timeout configuration for the job.
     */
    public final JobTimeout timeout() {
        return timeout;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The tags that are applied to the job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags that are applied to the job.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * Specifies whether to propagate the tags from the job or job definition to the corresponding Amazon ECS task. If
     * no value is specified, the tags aren't propagated. Tags can only be propagated to the tasks when the tasks are
     * created. For tags with the same name, job tags are given priority over job definitions tags. If the total number
     * of combined tags from the job and job definition is over 50, the job is moved to the <code>FAILED</code> state.
     * </p>
     * 
     * @return Specifies whether to propagate the tags from the job or job definition to the corresponding Amazon ECS
     *         task. If no value is specified, the tags aren't propagated. Tags can only be propagated to the tasks when
     *         the tasks are created. For tags with the same name, job tags are given priority over job definitions
     *         tags. If the total number of combined tags from the job and job definition is over 50, the job is moved
     *         to the <code>FAILED</code> state.
     */
    public final Boolean propagateTags() {
        return propagateTags;
    }

    /**
     * <p>
     * The platform capabilities required by the job definition. If no value is specified, it defaults to
     * <code>EC2</code>. Jobs run on Fargate resources specify <code>FARGATE</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPlatformCapabilities} method.
     * </p>
     * 
     * @return The platform capabilities required by the job definition. If no value is specified, it defaults to
     *         <code>EC2</code>. Jobs run on Fargate resources specify <code>FARGATE</code>.
     */
    public final List<PlatformCapability> platformCapabilities() {
        return PlatformCapabilityListCopier.copyStringToEnum(platformCapabilities);
    }

    /**
     * For responses, this returns true if the service returned a value for the PlatformCapabilities property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasPlatformCapabilities() {
        return platformCapabilities != null && !(platformCapabilities instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The platform capabilities required by the job definition. If no value is specified, it defaults to
     * <code>EC2</code>. Jobs run on Fargate resources specify <code>FARGATE</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPlatformCapabilities} method.
     * </p>
     * 
     * @return The platform capabilities required by the job definition. If no value is specified, it defaults to
     *         <code>EC2</code>. Jobs run on Fargate resources specify <code>FARGATE</code>.
     */
    public final List<String> platformCapabilitiesAsStrings() {
        return platformCapabilities;
    }

    /**
     * <p>
     * An object with various properties that are specific to Amazon EKS based jobs.
     * </p>
     * 
     * @return An object with various properties that are specific to Amazon EKS based jobs.
     */
    public final EksPropertiesDetail eksProperties() {
        return eksProperties;
    }

    /**
     * For responses, this returns true if the service returned a value for the EksAttempts property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEksAttempts() {
        return eksAttempts != null && !(eksAttempts instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of job attempts that are associated with this job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEksAttempts} method.
     * </p>
     * 
     * @return A list of job attempts that are associated with this job.
     */
    public final List<EksAttemptDetail> eksAttempts() {
        return eksAttempts;
    }

    /**
     * <p>
     * An object with properties that are specific to Amazon ECS-based jobs.
     * </p>
     * 
     * @return An object with properties that are specific to Amazon ECS-based jobs.
     */
    public final EcsPropertiesDetail ecsProperties() {
        return ecsProperties;
    }

    /**
     * <p>
     * Indicates whether the job is canceled.
     * </p>
     * 
     * @return Indicates whether the job is canceled.
     */
    public final Boolean isCancelled() {
        return isCancelled;
    }

    /**
     * <p>
     * Indicates whether the job is terminated.
     * </p>
     * 
     * @return Indicates whether the job is terminated.
     */
    public final Boolean isTerminated() {
        return isTerminated;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(jobArn());
        hashCode = 31 * hashCode + Objects.hashCode(jobName());
        hashCode = 31 * hashCode + Objects.hashCode(jobId());
        hashCode = 31 * hashCode + Objects.hashCode(jobQueue());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(shareIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(schedulingPriority());
        hashCode = 31 * hashCode + Objects.hashCode(hasAttempts() ? attempts() : null);
        hashCode = 31 * hashCode + Objects.hashCode(statusReason());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(retryStrategy());
        hashCode = 31 * hashCode + Objects.hashCode(startedAt());
        hashCode = 31 * hashCode + Objects.hashCode(stoppedAt());
        hashCode = 31 * hashCode + Objects.hashCode(hasDependsOn() ? dependsOn() : null);
        hashCode = 31 * hashCode + Objects.hashCode(jobDefinition());
        hashCode = 31 * hashCode + Objects.hashCode(hasParameters() ? parameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(container());
        hashCode = 31 * hashCode + Objects.hashCode(nodeDetails());
        hashCode = 31 * hashCode + Objects.hashCode(nodeProperties());
        hashCode = 31 * hashCode + Objects.hashCode(arrayProperties());
        hashCode = 31 * hashCode + Objects.hashCode(timeout());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(propagateTags());
        hashCode = 31 * hashCode + Objects.hashCode(hasPlatformCapabilities() ? platformCapabilitiesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(eksProperties());
        hashCode = 31 * hashCode + Objects.hashCode(hasEksAttempts() ? eksAttempts() : null);
        hashCode = 31 * hashCode + Objects.hashCode(ecsProperties());
        hashCode = 31 * hashCode + Objects.hashCode(isCancelled());
        hashCode = 31 * hashCode + Objects.hashCode(isTerminated());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof JobDetail)) {
            return false;
        }
        JobDetail other = (JobDetail) obj;
        return Objects.equals(jobArn(), other.jobArn()) && Objects.equals(jobName(), other.jobName())
                && Objects.equals(jobId(), other.jobId()) && Objects.equals(jobQueue(), other.jobQueue())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(shareIdentifier(), other.shareIdentifier())
                && Objects.equals(schedulingPriority(), other.schedulingPriority()) && hasAttempts() == other.hasAttempts()
                && Objects.equals(attempts(), other.attempts()) && Objects.equals(statusReason(), other.statusReason())
                && Objects.equals(createdAt(), other.createdAt()) && Objects.equals(retryStrategy(), other.retryStrategy())
                && Objects.equals(startedAt(), other.startedAt()) && Objects.equals(stoppedAt(), other.stoppedAt())
                && hasDependsOn() == other.hasDependsOn() && Objects.equals(dependsOn(), other.dependsOn())
                && Objects.equals(jobDefinition(), other.jobDefinition()) && hasParameters() == other.hasParameters()
                && Objects.equals(parameters(), other.parameters()) && Objects.equals(container(), other.container())
                && Objects.equals(nodeDetails(), other.nodeDetails()) && Objects.equals(nodeProperties(), other.nodeProperties())
                && Objects.equals(arrayProperties(), other.arrayProperties()) && Objects.equals(timeout(), other.timeout())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && Objects.equals(propagateTags(), other.propagateTags())
                && hasPlatformCapabilities() == other.hasPlatformCapabilities()
                && Objects.equals(platformCapabilitiesAsStrings(), other.platformCapabilitiesAsStrings())
                && Objects.equals(eksProperties(), other.eksProperties()) && hasEksAttempts() == other.hasEksAttempts()
                && Objects.equals(eksAttempts(), other.eksAttempts()) && Objects.equals(ecsProperties(), other.ecsProperties())
                && Objects.equals(isCancelled(), other.isCancelled()) && Objects.equals(isTerminated(), other.isTerminated());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("JobDetail").add("JobArn", jobArn()).add("JobName", jobName()).add("JobId", jobId())
                .add("JobQueue", jobQueue()).add("Status", statusAsString()).add("ShareIdentifier", shareIdentifier())
                .add("SchedulingPriority", schedulingPriority()).add("Attempts", hasAttempts() ? attempts() : null)
                .add("StatusReason", statusReason()).add("CreatedAt", createdAt()).add("RetryStrategy", retryStrategy())
                .add("StartedAt", startedAt()).add("StoppedAt", stoppedAt())
                .add("DependsOn", hasDependsOn() ? dependsOn() : null).add("JobDefinition", jobDefinition())
                .add("Parameters", hasParameters() ? parameters() : null).add("Container", container())
                .add("NodeDetails", nodeDetails()).add("NodeProperties", nodeProperties())
                .add("ArrayProperties", arrayProperties()).add("Timeout", timeout()).add("Tags", hasTags() ? tags() : null)
                .add("PropagateTags", propagateTags())
                .add("PlatformCapabilities", hasPlatformCapabilities() ? platformCapabilitiesAsStrings() : null)
                .add("EksProperties", eksProperties()).add("EksAttempts", hasEksAttempts() ? eksAttempts() : null)
                .add("EcsProperties", ecsProperties()).add("IsCancelled", isCancelled()).add("IsTerminated", isTerminated())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "jobArn":
            return Optional.ofNullable(clazz.cast(jobArn()));
        case "jobName":
            return Optional.ofNullable(clazz.cast(jobName()));
        case "jobId":
            return Optional.ofNullable(clazz.cast(jobId()));
        case "jobQueue":
            return Optional.ofNullable(clazz.cast(jobQueue()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "shareIdentifier":
            return Optional.ofNullable(clazz.cast(shareIdentifier()));
        case "schedulingPriority":
            return Optional.ofNullable(clazz.cast(schedulingPriority()));
        case "attempts":
            return Optional.ofNullable(clazz.cast(attempts()));
        case "statusReason":
            return Optional.ofNullable(clazz.cast(statusReason()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "retryStrategy":
            return Optional.ofNullable(clazz.cast(retryStrategy()));
        case "startedAt":
            return Optional.ofNullable(clazz.cast(startedAt()));
        case "stoppedAt":
            return Optional.ofNullable(clazz.cast(stoppedAt()));
        case "dependsOn":
            return Optional.ofNullable(clazz.cast(dependsOn()));
        case "jobDefinition":
            return Optional.ofNullable(clazz.cast(jobDefinition()));
        case "parameters":
            return Optional.ofNullable(clazz.cast(parameters()));
        case "container":
            return Optional.ofNullable(clazz.cast(container()));
        case "nodeDetails":
            return Optional.ofNullable(clazz.cast(nodeDetails()));
        case "nodeProperties":
            return Optional.ofNullable(clazz.cast(nodeProperties()));
        case "arrayProperties":
            return Optional.ofNullable(clazz.cast(arrayProperties()));
        case "timeout":
            return Optional.ofNullable(clazz.cast(timeout()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "propagateTags":
            return Optional.ofNullable(clazz.cast(propagateTags()));
        case "platformCapabilities":
            return Optional.ofNullable(clazz.cast(platformCapabilitiesAsStrings()));
        case "eksProperties":
            return Optional.ofNullable(clazz.cast(eksProperties()));
        case "eksAttempts":
            return Optional.ofNullable(clazz.cast(eksAttempts()));
        case "ecsProperties":
            return Optional.ofNullable(clazz.cast(ecsProperties()));
        case "isCancelled":
            return Optional.ofNullable(clazz.cast(isCancelled()));
        case "isTerminated":
            return Optional.ofNullable(clazz.cast(isTerminated()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<JobDetail, T> g) {
        return obj -> g.apply((JobDetail) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, JobDetail> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the job.
         * </p>
         * 
         * @param jobArn
         *        The Amazon Resource Name (ARN) of the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobArn(String jobArn);

        /**
         * <p>
         * The job name.
         * </p>
         * 
         * @param jobName
         *        The job name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobName(String jobName);

        /**
         * <p>
         * The job ID.
         * </p>
         * 
         * @param jobId
         *        The job ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobId(String jobId);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the job queue that the job is associated with.
         * </p>
         * 
         * @param jobQueue
         *        The Amazon Resource Name (ARN) of the job queue that the job is associated with.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobQueue(String jobQueue);

        /**
         * <p>
         * The current status for the job.
         * </p>
         * <note>
         * <p>
         * If your jobs don't progress to <code>STARTING</code>, see <a
         * href="https://docs.aws.amazon.com/batch/latest/userguide/troubleshooting.html#job_stuck_in_runnable">Jobs
         * stuck in RUNNABLE status</a> in the troubleshooting section of the <i>Batch User Guide</i>.
         * </p>
         * </note>
         * 
         * @param status
         *        The current status for the job.</p> <note>
         *        <p>
         *        If your jobs don't progress to <code>STARTING</code>, see <a
         *        href="https://docs.aws.amazon.com/batch/latest/userguide/troubleshooting.html#job_stuck_in_runnable"
         *        >Jobs stuck in RUNNABLE status</a> in the troubleshooting section of the <i>Batch User Guide</i>.
         *        </p>
         * @see JobStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current status for the job.
         * </p>
         * <note>
         * <p>
         * If your jobs don't progress to <code>STARTING</code>, see <a
         * href="https://docs.aws.amazon.com/batch/latest/userguide/troubleshooting.html#job_stuck_in_runnable">Jobs
         * stuck in RUNNABLE status</a> in the troubleshooting section of the <i>Batch User Guide</i>.
         * </p>
         * </note>
         * 
         * @param status
         *        The current status for the job.</p> <note>
         *        <p>
         *        If your jobs don't progress to <code>STARTING</code>, see <a
         *        href="https://docs.aws.amazon.com/batch/latest/userguide/troubleshooting.html#job_stuck_in_runnable"
         *        >Jobs stuck in RUNNABLE status</a> in the troubleshooting section of the <i>Batch User Guide</i>.
         *        </p>
         * @see JobStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobStatus
         */
        Builder status(JobStatus status);

        /**
         * <p>
         * The share identifier for the job.
         * </p>
         * 
         * @param shareIdentifier
         *        The share identifier for the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder shareIdentifier(String shareIdentifier);

        /**
         * <p>
         * The scheduling policy of the job definition. This only affects jobs in job queues with a fair share policy.
         * Jobs with a higher scheduling priority are scheduled before jobs with a lower scheduling priority.
         * </p>
         * 
         * @param schedulingPriority
         *        The scheduling policy of the job definition. This only affects jobs in job queues with a fair share
         *        policy. Jobs with a higher scheduling priority are scheduled before jobs with a lower scheduling
         *        priority.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schedulingPriority(Integer schedulingPriority);

        /**
         * <p>
         * A list of job attempts that are associated with this job.
         * </p>
         * 
         * @param attempts
         *        A list of job attempts that are associated with this job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attempts(Collection<AttemptDetail> attempts);

        /**
         * <p>
         * A list of job attempts that are associated with this job.
         * </p>
         * 
         * @param attempts
         *        A list of job attempts that are associated with this job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attempts(AttemptDetail... attempts);

        /**
         * <p>
         * A list of job attempts that are associated with this job.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.batch.model.AttemptDetail.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.batch.model.AttemptDetail#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.batch.model.AttemptDetail.Builder#build()} is called immediately and
         * its result is passed to {@link #attempts(List<AttemptDetail>)}.
         * 
         * @param attempts
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.batch.model.AttemptDetail.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #attempts(java.util.Collection<AttemptDetail>)
         */
        Builder attempts(Consumer<AttemptDetail.Builder>... attempts);

        /**
         * <p>
         * A short, human-readable string to provide more details for the current status of the job.
         * </p>
         * 
         * @param statusReason
         *        A short, human-readable string to provide more details for the current status of the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusReason(String statusReason);

        /**
         * <p>
         * The Unix timestamp (in milliseconds) for when the job was created. For non-array jobs and parent array jobs,
         * this is when the job entered the <code>SUBMITTED</code> state. This is specifically at the time <a
         * href="https://docs.aws.amazon.com/batch/latest/APIReference/API_SubmitJob.html">SubmitJob</a> was called. For
         * array child jobs, this is when the child job was spawned by its parent and entered the <code>PENDING</code>
         * state.
         * </p>
         * 
         * @param createdAt
         *        The Unix timestamp (in milliseconds) for when the job was created. For non-array jobs and parent array
         *        jobs, this is when the job entered the <code>SUBMITTED</code> state. This is specifically at the time
         *        <a href="https://docs.aws.amazon.com/batch/latest/APIReference/API_SubmitJob.html">SubmitJob</a> was
         *        called. For array child jobs, this is when the child job was spawned by its parent and entered the
         *        <code>PENDING</code> state.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Long createdAt);

        /**
         * <p>
         * The retry strategy to use for this job if an attempt fails.
         * </p>
         * 
         * @param retryStrategy
         *        The retry strategy to use for this job if an attempt fails.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder retryStrategy(RetryStrategy retryStrategy);

        /**
         * <p>
         * The retry strategy to use for this job if an attempt fails.
         * </p>
         * This is a convenience method that creates an instance of the {@link RetryStrategy.Builder} avoiding the need
         * to create one manually via {@link RetryStrategy#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RetryStrategy.Builder#build()} is called immediately and its
         * result is passed to {@link #retryStrategy(RetryStrategy)}.
         * 
         * @param retryStrategy
         *        a consumer that will call methods on {@link RetryStrategy.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #retryStrategy(RetryStrategy)
         */
        default Builder retryStrategy(Consumer<RetryStrategy.Builder> retryStrategy) {
            return retryStrategy(RetryStrategy.builder().applyMutation(retryStrategy).build());
        }

        /**
         * <p>
         * The Unix timestamp (in milliseconds) for when the job was started. More specifically, it's when the job
         * transitioned from the <code>STARTING</code> state to the <code>RUNNING</code> state.
         * </p>
         * 
         * @param startedAt
         *        The Unix timestamp (in milliseconds) for when the job was started. More specifically, it's when the
         *        job transitioned from the <code>STARTING</code> state to the <code>RUNNING</code> state.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startedAt(Long startedAt);

        /**
         * <p>
         * The Unix timestamp (in milliseconds) for when the job was stopped. More specifically, it's when the job
         * transitioned from the <code>RUNNING</code> state to a terminal state, such as <code>SUCCEEDED</code> or
         * <code>FAILED</code>.
         * </p>
         * 
         * @param stoppedAt
         *        The Unix timestamp (in milliseconds) for when the job was stopped. More specifically, it's when the
         *        job transitioned from the <code>RUNNING</code> state to a terminal state, such as
         *        <code>SUCCEEDED</code> or <code>FAILED</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stoppedAt(Long stoppedAt);

        /**
         * <p>
         * A list of job IDs that this job depends on.
         * </p>
         * 
         * @param dependsOn
         *        A list of job IDs that this job depends on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dependsOn(Collection<JobDependency> dependsOn);

        /**
         * <p>
         * A list of job IDs that this job depends on.
         * </p>
         * 
         * @param dependsOn
         *        A list of job IDs that this job depends on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dependsOn(JobDependency... dependsOn);

        /**
         * <p>
         * A list of job IDs that this job depends on.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.batch.model.JobDependency.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.batch.model.JobDependency#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.batch.model.JobDependency.Builder#build()} is called immediately and
         * its result is passed to {@link #dependsOn(List<JobDependency>)}.
         * 
         * @param dependsOn
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.batch.model.JobDependency.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dependsOn(java.util.Collection<JobDependency>)
         */
        Builder dependsOn(Consumer<JobDependency.Builder>... dependsOn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the job definition that this job uses.
         * </p>
         * 
         * @param jobDefinition
         *        The Amazon Resource Name (ARN) of the job definition that this job uses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobDefinition(String jobDefinition);

        /**
         * <p>
         * Additional parameters that are passed to the job that replace parameter substitution placeholders or override
         * any corresponding parameter defaults from the job definition.
         * </p>
         * 
         * @param parameters
         *        Additional parameters that are passed to the job that replace parameter substitution placeholders or
         *        override any corresponding parameter defaults from the job definition.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Map<String, String> parameters);

        /**
         * <p>
         * An object that represents the details for the container that's associated with the job. If the details are
         * for a multiple-container job, this object will be empty.
         * </p>
         * 
         * @param container
         *        An object that represents the details for the container that's associated with the job. If the details
         *        are for a multiple-container job, this object will be empty.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder container(ContainerDetail container);

        /**
         * <p>
         * An object that represents the details for the container that's associated with the job. If the details are
         * for a multiple-container job, this object will be empty.
         * </p>
         * This is a convenience method that creates an instance of the {@link ContainerDetail.Builder} avoiding the
         * need to create one manually via {@link ContainerDetail#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ContainerDetail.Builder#build()} is called immediately and its
         * result is passed to {@link #container(ContainerDetail)}.
         * 
         * @param container
         *        a consumer that will call methods on {@link ContainerDetail.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #container(ContainerDetail)
         */
        default Builder container(Consumer<ContainerDetail.Builder> container) {
            return container(ContainerDetail.builder().applyMutation(container).build());
        }

        /**
         * <p>
         * An object that represents the details of a node that's associated with a multi-node parallel job.
         * </p>
         * 
         * @param nodeDetails
         *        An object that represents the details of a node that's associated with a multi-node parallel job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodeDetails(NodeDetails nodeDetails);

        /**
         * <p>
         * An object that represents the details of a node that's associated with a multi-node parallel job.
         * </p>
         * This is a convenience method that creates an instance of the {@link NodeDetails.Builder} avoiding the need to
         * create one manually via {@link NodeDetails#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link NodeDetails.Builder#build()} is called immediately and its result
         * is passed to {@link #nodeDetails(NodeDetails)}.
         * 
         * @param nodeDetails
         *        a consumer that will call methods on {@link NodeDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #nodeDetails(NodeDetails)
         */
        default Builder nodeDetails(Consumer<NodeDetails.Builder> nodeDetails) {
            return nodeDetails(NodeDetails.builder().applyMutation(nodeDetails).build());
        }

        /**
         * <p>
         * An object that represents the node properties of a multi-node parallel job.
         * </p>
         * <note>
         * <p>
         * This isn't applicable to jobs that are running on Fargate resources.
         * </p>
         * </note>
         * 
         * @param nodeProperties
         *        An object that represents the node properties of a multi-node parallel job.</p> <note>
         *        <p>
         *        This isn't applicable to jobs that are running on Fargate resources.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodeProperties(NodeProperties nodeProperties);

        /**
         * <p>
         * An object that represents the node properties of a multi-node parallel job.
         * </p>
         * <note>
         * <p>
         * This isn't applicable to jobs that are running on Fargate resources.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link NodeProperties.Builder} avoiding
         * the need to create one manually via {@link NodeProperties#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link NodeProperties.Builder#build()} is called immediately and its
         * result is passed to {@link #nodeProperties(NodeProperties)}.
         * 
         * @param nodeProperties
         *        a consumer that will call methods on {@link NodeProperties.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #nodeProperties(NodeProperties)
         */
        default Builder nodeProperties(Consumer<NodeProperties.Builder> nodeProperties) {
            return nodeProperties(NodeProperties.builder().applyMutation(nodeProperties).build());
        }

        /**
         * <p>
         * The array properties of the job, if it's an array job.
         * </p>
         * 
         * @param arrayProperties
         *        The array properties of the job, if it's an array job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arrayProperties(ArrayPropertiesDetail arrayProperties);

        /**
         * <p>
         * The array properties of the job, if it's an array job.
         * </p>
         * This is a convenience method that creates an instance of the {@link ArrayPropertiesDetail.Builder} avoiding
         * the need to create one manually via {@link ArrayPropertiesDetail#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ArrayPropertiesDetail.Builder#build()} is called immediately and
         * its result is passed to {@link #arrayProperties(ArrayPropertiesDetail)}.
         * 
         * @param arrayProperties
         *        a consumer that will call methods on {@link ArrayPropertiesDetail.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #arrayProperties(ArrayPropertiesDetail)
         */
        default Builder arrayProperties(Consumer<ArrayPropertiesDetail.Builder> arrayProperties) {
            return arrayProperties(ArrayPropertiesDetail.builder().applyMutation(arrayProperties).build());
        }

        /**
         * <p>
         * The timeout configuration for the job.
         * </p>
         * 
         * @param timeout
         *        The timeout configuration for the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timeout(JobTimeout timeout);

        /**
         * <p>
         * The timeout configuration for the job.
         * </p>
         * This is a convenience method that creates an instance of the {@link JobTimeout.Builder} avoiding the need to
         * create one manually via {@link JobTimeout#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link JobTimeout.Builder#build()} is called immediately and its result
         * is passed to {@link #timeout(JobTimeout)}.
         * 
         * @param timeout
         *        a consumer that will call methods on {@link JobTimeout.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #timeout(JobTimeout)
         */
        default Builder timeout(Consumer<JobTimeout.Builder> timeout) {
            return timeout(JobTimeout.builder().applyMutation(timeout).build());
        }

        /**
         * <p>
         * The tags that are applied to the job.
         * </p>
         * 
         * @param tags
         *        The tags that are applied to the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * Specifies whether to propagate the tags from the job or job definition to the corresponding Amazon ECS task.
         * If no value is specified, the tags aren't propagated. Tags can only be propagated to the tasks when the tasks
         * are created. For tags with the same name, job tags are given priority over job definitions tags. If the total
         * number of combined tags from the job and job definition is over 50, the job is moved to the
         * <code>FAILED</code> state.
         * </p>
         * 
         * @param propagateTags
         *        Specifies whether to propagate the tags from the job or job definition to the corresponding Amazon ECS
         *        task. If no value is specified, the tags aren't propagated. Tags can only be propagated to the tasks
         *        when the tasks are created. For tags with the same name, job tags are given priority over job
         *        definitions tags. If the total number of combined tags from the job and job definition is over 50, the
         *        job is moved to the <code>FAILED</code> state.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder propagateTags(Boolean propagateTags);

        /**
         * <p>
         * The platform capabilities required by the job definition. If no value is specified, it defaults to
         * <code>EC2</code>. Jobs run on Fargate resources specify <code>FARGATE</code>.
         * </p>
         * 
         * @param platformCapabilities
         *        The platform capabilities required by the job definition. If no value is specified, it defaults to
         *        <code>EC2</code>. Jobs run on Fargate resources specify <code>FARGATE</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformCapabilitiesWithStrings(Collection<String> platformCapabilities);

        /**
         * <p>
         * The platform capabilities required by the job definition. If no value is specified, it defaults to
         * <code>EC2</code>. Jobs run on Fargate resources specify <code>FARGATE</code>.
         * </p>
         * 
         * @param platformCapabilities
         *        The platform capabilities required by the job definition. If no value is specified, it defaults to
         *        <code>EC2</code>. Jobs run on Fargate resources specify <code>FARGATE</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformCapabilitiesWithStrings(String... platformCapabilities);

        /**
         * <p>
         * The platform capabilities required by the job definition. If no value is specified, it defaults to
         * <code>EC2</code>. Jobs run on Fargate resources specify <code>FARGATE</code>.
         * </p>
         * 
         * @param platformCapabilities
         *        The platform capabilities required by the job definition. If no value is specified, it defaults to
         *        <code>EC2</code>. Jobs run on Fargate resources specify <code>FARGATE</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformCapabilities(Collection<PlatformCapability> platformCapabilities);

        /**
         * <p>
         * The platform capabilities required by the job definition. If no value is specified, it defaults to
         * <code>EC2</code>. Jobs run on Fargate resources specify <code>FARGATE</code>.
         * </p>
         * 
         * @param platformCapabilities
         *        The platform capabilities required by the job definition. If no value is specified, it defaults to
         *        <code>EC2</code>. Jobs run on Fargate resources specify <code>FARGATE</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformCapabilities(PlatformCapability... platformCapabilities);

        /**
         * <p>
         * An object with various properties that are specific to Amazon EKS based jobs.
         * </p>
         * 
         * @param eksProperties
         *        An object with various properties that are specific to Amazon EKS based jobs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eksProperties(EksPropertiesDetail eksProperties);

        /**
         * <p>
         * An object with various properties that are specific to Amazon EKS based jobs.
         * </p>
         * This is a convenience method that creates an instance of the {@link EksPropertiesDetail.Builder} avoiding the
         * need to create one manually via {@link EksPropertiesDetail#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EksPropertiesDetail.Builder#build()} is called immediately and
         * its result is passed to {@link #eksProperties(EksPropertiesDetail)}.
         * 
         * @param eksProperties
         *        a consumer that will call methods on {@link EksPropertiesDetail.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #eksProperties(EksPropertiesDetail)
         */
        default Builder eksProperties(Consumer<EksPropertiesDetail.Builder> eksProperties) {
            return eksProperties(EksPropertiesDetail.builder().applyMutation(eksProperties).build());
        }

        /**
         * <p>
         * A list of job attempts that are associated with this job.
         * </p>
         * 
         * @param eksAttempts
         *        A list of job attempts that are associated with this job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eksAttempts(Collection<EksAttemptDetail> eksAttempts);

        /**
         * <p>
         * A list of job attempts that are associated with this job.
         * </p>
         * 
         * @param eksAttempts
         *        A list of job attempts that are associated with this job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eksAttempts(EksAttemptDetail... eksAttempts);

        /**
         * <p>
         * A list of job attempts that are associated with this job.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.batch.model.EksAttemptDetail.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.batch.model.EksAttemptDetail#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.batch.model.EksAttemptDetail.Builder#build()} is called immediately
         * and its result is passed to {@link #eksAttempts(List<EksAttemptDetail>)}.
         * 
         * @param eksAttempts
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.batch.model.EksAttemptDetail.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #eksAttempts(java.util.Collection<EksAttemptDetail>)
         */
        Builder eksAttempts(Consumer<EksAttemptDetail.Builder>... eksAttempts);

        /**
         * <p>
         * An object with properties that are specific to Amazon ECS-based jobs.
         * </p>
         * 
         * @param ecsProperties
         *        An object with properties that are specific to Amazon ECS-based jobs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ecsProperties(EcsPropertiesDetail ecsProperties);

        /**
         * <p>
         * An object with properties that are specific to Amazon ECS-based jobs.
         * </p>
         * This is a convenience method that creates an instance of the {@link EcsPropertiesDetail.Builder} avoiding the
         * need to create one manually via {@link EcsPropertiesDetail#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EcsPropertiesDetail.Builder#build()} is called immediately and
         * its result is passed to {@link #ecsProperties(EcsPropertiesDetail)}.
         * 
         * @param ecsProperties
         *        a consumer that will call methods on {@link EcsPropertiesDetail.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ecsProperties(EcsPropertiesDetail)
         */
        default Builder ecsProperties(Consumer<EcsPropertiesDetail.Builder> ecsProperties) {
            return ecsProperties(EcsPropertiesDetail.builder().applyMutation(ecsProperties).build());
        }

        /**
         * <p>
         * Indicates whether the job is canceled.
         * </p>
         * 
         * @param isCancelled
         *        Indicates whether the job is canceled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isCancelled(Boolean isCancelled);

        /**
         * <p>
         * Indicates whether the job is terminated.
         * </p>
         * 
         * @param isTerminated
         *        Indicates whether the job is terminated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isTerminated(Boolean isTerminated);
    }

    static final class BuilderImpl implements Builder {
        private String jobArn;

        private String jobName;

        private String jobId;

        private String jobQueue;

        private String status;

        private String shareIdentifier;

        private Integer schedulingPriority;

        private List<AttemptDetail> attempts = DefaultSdkAutoConstructList.getInstance();

        private String statusReason;

        private Long createdAt;

        private RetryStrategy retryStrategy;

        private Long startedAt;

        private Long stoppedAt;

        private List<JobDependency> dependsOn = DefaultSdkAutoConstructList.getInstance();

        private String jobDefinition;

        private Map<String, String> parameters = DefaultSdkAutoConstructMap.getInstance();

        private ContainerDetail container;

        private NodeDetails nodeDetails;

        private NodeProperties nodeProperties;

        private ArrayPropertiesDetail arrayProperties;

        private JobTimeout timeout;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private Boolean propagateTags;

        private List<String> platformCapabilities = DefaultSdkAutoConstructList.getInstance();

        private EksPropertiesDetail eksProperties;

        private List<EksAttemptDetail> eksAttempts = DefaultSdkAutoConstructList.getInstance();

        private EcsPropertiesDetail ecsProperties;

        private Boolean isCancelled;

        private Boolean isTerminated;

        private BuilderImpl() {
        }

        private BuilderImpl(JobDetail model) {
            jobArn(model.jobArn);
            jobName(model.jobName);
            jobId(model.jobId);
            jobQueue(model.jobQueue);
            status(model.status);
            shareIdentifier(model.shareIdentifier);
            schedulingPriority(model.schedulingPriority);
            attempts(model.attempts);
            statusReason(model.statusReason);
            createdAt(model.createdAt);
            retryStrategy(model.retryStrategy);
            startedAt(model.startedAt);
            stoppedAt(model.stoppedAt);
            dependsOn(model.dependsOn);
            jobDefinition(model.jobDefinition);
            parameters(model.parameters);
            container(model.container);
            nodeDetails(model.nodeDetails);
            nodeProperties(model.nodeProperties);
            arrayProperties(model.arrayProperties);
            timeout(model.timeout);
            tags(model.tags);
            propagateTags(model.propagateTags);
            platformCapabilitiesWithStrings(model.platformCapabilities);
            eksProperties(model.eksProperties);
            eksAttempts(model.eksAttempts);
            ecsProperties(model.ecsProperties);
            isCancelled(model.isCancelled);
            isTerminated(model.isTerminated);
        }

        public final String getJobArn() {
            return jobArn;
        }

        public final void setJobArn(String jobArn) {
            this.jobArn = jobArn;
        }

        @Override
        public final Builder jobArn(String jobArn) {
            this.jobArn = jobArn;
            return this;
        }

        public final String getJobName() {
            return jobName;
        }

        public final void setJobName(String jobName) {
            this.jobName = jobName;
        }

        @Override
        public final Builder jobName(String jobName) {
            this.jobName = jobName;
            return this;
        }

        public final String getJobId() {
            return jobId;
        }

        public final void setJobId(String jobId) {
            this.jobId = jobId;
        }

        @Override
        public final Builder jobId(String jobId) {
            this.jobId = jobId;
            return this;
        }

        public final String getJobQueue() {
            return jobQueue;
        }

        public final void setJobQueue(String jobQueue) {
            this.jobQueue = jobQueue;
        }

        @Override
        public final Builder jobQueue(String jobQueue) {
            this.jobQueue = jobQueue;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(JobStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getShareIdentifier() {
            return shareIdentifier;
        }

        public final void setShareIdentifier(String shareIdentifier) {
            this.shareIdentifier = shareIdentifier;
        }

        @Override
        public final Builder shareIdentifier(String shareIdentifier) {
            this.shareIdentifier = shareIdentifier;
            return this;
        }

        public final Integer getSchedulingPriority() {
            return schedulingPriority;
        }

        public final void setSchedulingPriority(Integer schedulingPriority) {
            this.schedulingPriority = schedulingPriority;
        }

        @Override
        public final Builder schedulingPriority(Integer schedulingPriority) {
            this.schedulingPriority = schedulingPriority;
            return this;
        }

        public final List<AttemptDetail.Builder> getAttempts() {
            List<AttemptDetail.Builder> result = AttemptDetailsCopier.copyToBuilder(this.attempts);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAttempts(Collection<AttemptDetail.BuilderImpl> attempts) {
            this.attempts = AttemptDetailsCopier.copyFromBuilder(attempts);
        }

        @Override
        public final Builder attempts(Collection<AttemptDetail> attempts) {
            this.attempts = AttemptDetailsCopier.copy(attempts);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attempts(AttemptDetail... attempts) {
            attempts(Arrays.asList(attempts));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attempts(Consumer<AttemptDetail.Builder>... attempts) {
            attempts(Stream.of(attempts).map(c -> AttemptDetail.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getStatusReason() {
            return statusReason;
        }

        public final void setStatusReason(String statusReason) {
            this.statusReason = statusReason;
        }

        @Override
        public final Builder statusReason(String statusReason) {
            this.statusReason = statusReason;
            return this;
        }

        public final Long getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Long createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Long createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final RetryStrategy.Builder getRetryStrategy() {
            return retryStrategy != null ? retryStrategy.toBuilder() : null;
        }

        public final void setRetryStrategy(RetryStrategy.BuilderImpl retryStrategy) {
            this.retryStrategy = retryStrategy != null ? retryStrategy.build() : null;
        }

        @Override
        public final Builder retryStrategy(RetryStrategy retryStrategy) {
            this.retryStrategy = retryStrategy;
            return this;
        }

        public final Long getStartedAt() {
            return startedAt;
        }

        public final void setStartedAt(Long startedAt) {
            this.startedAt = startedAt;
        }

        @Override
        public final Builder startedAt(Long startedAt) {
            this.startedAt = startedAt;
            return this;
        }

        public final Long getStoppedAt() {
            return stoppedAt;
        }

        public final void setStoppedAt(Long stoppedAt) {
            this.stoppedAt = stoppedAt;
        }

        @Override
        public final Builder stoppedAt(Long stoppedAt) {
            this.stoppedAt = stoppedAt;
            return this;
        }

        public final List<JobDependency.Builder> getDependsOn() {
            List<JobDependency.Builder> result = JobDependencyListCopier.copyToBuilder(this.dependsOn);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDependsOn(Collection<JobDependency.BuilderImpl> dependsOn) {
            this.dependsOn = JobDependencyListCopier.copyFromBuilder(dependsOn);
        }

        @Override
        public final Builder dependsOn(Collection<JobDependency> dependsOn) {
            this.dependsOn = JobDependencyListCopier.copy(dependsOn);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dependsOn(JobDependency... dependsOn) {
            dependsOn(Arrays.asList(dependsOn));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dependsOn(Consumer<JobDependency.Builder>... dependsOn) {
            dependsOn(Stream.of(dependsOn).map(c -> JobDependency.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getJobDefinition() {
            return jobDefinition;
        }

        public final void setJobDefinition(String jobDefinition) {
            this.jobDefinition = jobDefinition;
        }

        @Override
        public final Builder jobDefinition(String jobDefinition) {
            this.jobDefinition = jobDefinition;
            return this;
        }

        public final Map<String, String> getParameters() {
            if (parameters instanceof SdkAutoConstructMap) {
                return null;
            }
            return parameters;
        }

        public final void setParameters(Map<String, String> parameters) {
            this.parameters = ParametersMapCopier.copy(parameters);
        }

        @Override
        public final Builder parameters(Map<String, String> parameters) {
            this.parameters = ParametersMapCopier.copy(parameters);
            return this;
        }

        public final ContainerDetail.Builder getContainer() {
            return container != null ? container.toBuilder() : null;
        }

        public final void setContainer(ContainerDetail.BuilderImpl container) {
            this.container = container != null ? container.build() : null;
        }

        @Override
        public final Builder container(ContainerDetail container) {
            this.container = container;
            return this;
        }

        public final NodeDetails.Builder getNodeDetails() {
            return nodeDetails != null ? nodeDetails.toBuilder() : null;
        }

        public final void setNodeDetails(NodeDetails.BuilderImpl nodeDetails) {
            this.nodeDetails = nodeDetails != null ? nodeDetails.build() : null;
        }

        @Override
        public final Builder nodeDetails(NodeDetails nodeDetails) {
            this.nodeDetails = nodeDetails;
            return this;
        }

        public final NodeProperties.Builder getNodeProperties() {
            return nodeProperties != null ? nodeProperties.toBuilder() : null;
        }

        public final void setNodeProperties(NodeProperties.BuilderImpl nodeProperties) {
            this.nodeProperties = nodeProperties != null ? nodeProperties.build() : null;
        }

        @Override
        public final Builder nodeProperties(NodeProperties nodeProperties) {
            this.nodeProperties = nodeProperties;
            return this;
        }

        public final ArrayPropertiesDetail.Builder getArrayProperties() {
            return arrayProperties != null ? arrayProperties.toBuilder() : null;
        }

        public final void setArrayProperties(ArrayPropertiesDetail.BuilderImpl arrayProperties) {
            this.arrayProperties = arrayProperties != null ? arrayProperties.build() : null;
        }

        @Override
        public final Builder arrayProperties(ArrayPropertiesDetail arrayProperties) {
            this.arrayProperties = arrayProperties;
            return this;
        }

        public final JobTimeout.Builder getTimeout() {
            return timeout != null ? timeout.toBuilder() : null;
        }

        public final void setTimeout(JobTimeout.BuilderImpl timeout) {
            this.timeout = timeout != null ? timeout.build() : null;
        }

        @Override
        public final Builder timeout(JobTimeout timeout) {
            this.timeout = timeout;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagrisTagsMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagrisTagsMapCopier.copy(tags);
            return this;
        }

        public final Boolean getPropagateTags() {
            return propagateTags;
        }

        public final void setPropagateTags(Boolean propagateTags) {
            this.propagateTags = propagateTags;
        }

        @Override
        public final Builder propagateTags(Boolean propagateTags) {
            this.propagateTags = propagateTags;
            return this;
        }

        public final Collection<String> getPlatformCapabilities() {
            if (platformCapabilities instanceof SdkAutoConstructList) {
                return null;
            }
            return platformCapabilities;
        }

        public final void setPlatformCapabilities(Collection<String> platformCapabilities) {
            this.platformCapabilities = PlatformCapabilityListCopier.copy(platformCapabilities);
        }

        @Override
        public final Builder platformCapabilitiesWithStrings(Collection<String> platformCapabilities) {
            this.platformCapabilities = PlatformCapabilityListCopier.copy(platformCapabilities);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder platformCapabilitiesWithStrings(String... platformCapabilities) {
            platformCapabilitiesWithStrings(Arrays.asList(platformCapabilities));
            return this;
        }

        @Override
        public final Builder platformCapabilities(Collection<PlatformCapability> platformCapabilities) {
            this.platformCapabilities = PlatformCapabilityListCopier.copyEnumToString(platformCapabilities);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder platformCapabilities(PlatformCapability... platformCapabilities) {
            platformCapabilities(Arrays.asList(platformCapabilities));
            return this;
        }

        public final EksPropertiesDetail.Builder getEksProperties() {
            return eksProperties != null ? eksProperties.toBuilder() : null;
        }

        public final void setEksProperties(EksPropertiesDetail.BuilderImpl eksProperties) {
            this.eksProperties = eksProperties != null ? eksProperties.build() : null;
        }

        @Override
        public final Builder eksProperties(EksPropertiesDetail eksProperties) {
            this.eksProperties = eksProperties;
            return this;
        }

        public final List<EksAttemptDetail.Builder> getEksAttempts() {
            List<EksAttemptDetail.Builder> result = EksAttemptDetailsCopier.copyToBuilder(this.eksAttempts);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setEksAttempts(Collection<EksAttemptDetail.BuilderImpl> eksAttempts) {
            this.eksAttempts = EksAttemptDetailsCopier.copyFromBuilder(eksAttempts);
        }

        @Override
        public final Builder eksAttempts(Collection<EksAttemptDetail> eksAttempts) {
            this.eksAttempts = EksAttemptDetailsCopier.copy(eksAttempts);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder eksAttempts(EksAttemptDetail... eksAttempts) {
            eksAttempts(Arrays.asList(eksAttempts));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder eksAttempts(Consumer<EksAttemptDetail.Builder>... eksAttempts) {
            eksAttempts(Stream.of(eksAttempts).map(c -> EksAttemptDetail.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final EcsPropertiesDetail.Builder getEcsProperties() {
            return ecsProperties != null ? ecsProperties.toBuilder() : null;
        }

        public final void setEcsProperties(EcsPropertiesDetail.BuilderImpl ecsProperties) {
            this.ecsProperties = ecsProperties != null ? ecsProperties.build() : null;
        }

        @Override
        public final Builder ecsProperties(EcsPropertiesDetail ecsProperties) {
            this.ecsProperties = ecsProperties;
            return this;
        }

        public final Boolean getIsCancelled() {
            return isCancelled;
        }

        public final void setIsCancelled(Boolean isCancelled) {
            this.isCancelled = isCancelled;
        }

        @Override
        public final Builder isCancelled(Boolean isCancelled) {
            this.isCancelled = isCancelled;
            return this;
        }

        public final Boolean getIsTerminated() {
            return isTerminated;
        }

        public final void setIsTerminated(Boolean isTerminated) {
            this.isTerminated = isTerminated;
        }

        @Override
        public final Builder isTerminated(Boolean isTerminated) {
            this.isTerminated = isTerminated;
            return this;
        }

        @Override
        public JobDetail build() {
            return new JobDetail(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
