/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.batch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The object that represents any node overrides to a job definition that's used in a <a
 * href="https://docs.aws.amazon.com/batch/latest/APIReference/API_SubmitJob.html">SubmitJob</a> API operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class NodePropertyOverride implements SdkPojo, Serializable,
        ToCopyableBuilder<NodePropertyOverride.Builder, NodePropertyOverride> {
    private static final SdkField<String> TARGET_NODES_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("targetNodes").getter(getter(NodePropertyOverride::targetNodes)).setter(setter(Builder::targetNodes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("targetNodes").build()).build();

    private static final SdkField<ContainerOverrides> CONTAINER_OVERRIDES_FIELD = SdkField
            .<ContainerOverrides> builder(MarshallingType.SDK_POJO).memberName("containerOverrides")
            .getter(getter(NodePropertyOverride::containerOverrides)).setter(setter(Builder::containerOverrides))
            .constructor(ContainerOverrides::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("containerOverrides").build())
            .build();

    private static final SdkField<EcsPropertiesOverride> ECS_PROPERTIES_OVERRIDE_FIELD = SdkField
            .<EcsPropertiesOverride> builder(MarshallingType.SDK_POJO).memberName("ecsPropertiesOverride")
            .getter(getter(NodePropertyOverride::ecsPropertiesOverride)).setter(setter(Builder::ecsPropertiesOverride))
            .constructor(EcsPropertiesOverride::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ecsPropertiesOverride").build())
            .build();

    private static final SdkField<List<String>> INSTANCE_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("instanceTypes")
            .getter(getter(NodePropertyOverride::instanceTypes))
            .setter(setter(Builder::instanceTypes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instanceTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<EksPropertiesOverride> EKS_PROPERTIES_OVERRIDE_FIELD = SdkField
            .<EksPropertiesOverride> builder(MarshallingType.SDK_POJO).memberName("eksPropertiesOverride")
            .getter(getter(NodePropertyOverride::eksPropertiesOverride)).setter(setter(Builder::eksPropertiesOverride))
            .constructor(EksPropertiesOverride::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eksPropertiesOverride").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TARGET_NODES_FIELD,
            CONTAINER_OVERRIDES_FIELD, ECS_PROPERTIES_OVERRIDE_FIELD, INSTANCE_TYPES_FIELD, EKS_PROPERTIES_OVERRIDE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String targetNodes;

    private final ContainerOverrides containerOverrides;

    private final EcsPropertiesOverride ecsPropertiesOverride;

    private final List<String> instanceTypes;

    private final EksPropertiesOverride eksPropertiesOverride;

    private NodePropertyOverride(BuilderImpl builder) {
        this.targetNodes = builder.targetNodes;
        this.containerOverrides = builder.containerOverrides;
        this.ecsPropertiesOverride = builder.ecsPropertiesOverride;
        this.instanceTypes = builder.instanceTypes;
        this.eksPropertiesOverride = builder.eksPropertiesOverride;
    }

    /**
     * <p>
     * The range of nodes, using node index values, that's used to override. A range of <code>0:3</code> indicates nodes
     * with index values of <code>0</code> through <code>3</code>. If the starting range value is omitted (
     * <code>:n</code>), then <code>0</code> is used to start the range. If the ending range value is omitted (
     * <code>n:</code>), then the highest possible node index is used to end the range.
     * </p>
     * 
     * @return The range of nodes, using node index values, that's used to override. A range of <code>0:3</code>
     *         indicates nodes with index values of <code>0</code> through <code>3</code>. If the starting range value
     *         is omitted (<code>:n</code>), then <code>0</code> is used to start the range. If the ending range value
     *         is omitted (<code>n:</code>), then the highest possible node index is used to end the range.
     */
    public final String targetNodes() {
        return targetNodes;
    }

    /**
     * <p>
     * The overrides that are sent to a node range.
     * </p>
     * 
     * @return The overrides that are sent to a node range.
     */
    public final ContainerOverrides containerOverrides() {
        return containerOverrides;
    }

    /**
     * <p>
     * An object that contains the properties that you want to replace for the existing Amazon ECS resources of a job.
     * </p>
     * 
     * @return An object that contains the properties that you want to replace for the existing Amazon ECS resources of
     *         a job.
     */
    public final EcsPropertiesOverride ecsPropertiesOverride() {
        return ecsPropertiesOverride;
    }

    /**
     * For responses, this returns true if the service returned a value for the InstanceTypes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasInstanceTypes() {
        return instanceTypes != null && !(instanceTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An object that contains the instance types that you want to replace for the existing resources of a job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInstanceTypes} method.
     * </p>
     * 
     * @return An object that contains the instance types that you want to replace for the existing resources of a job.
     */
    public final List<String> instanceTypes() {
        return instanceTypes;
    }

    /**
     * <p>
     * An object that contains the properties that you want to replace for the existing Amazon EKS resources of a job.
     * </p>
     * 
     * @return An object that contains the properties that you want to replace for the existing Amazon EKS resources of
     *         a job.
     */
    public final EksPropertiesOverride eksPropertiesOverride() {
        return eksPropertiesOverride;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(targetNodes());
        hashCode = 31 * hashCode + Objects.hashCode(containerOverrides());
        hashCode = 31 * hashCode + Objects.hashCode(ecsPropertiesOverride());
        hashCode = 31 * hashCode + Objects.hashCode(hasInstanceTypes() ? instanceTypes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(eksPropertiesOverride());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof NodePropertyOverride)) {
            return false;
        }
        NodePropertyOverride other = (NodePropertyOverride) obj;
        return Objects.equals(targetNodes(), other.targetNodes())
                && Objects.equals(containerOverrides(), other.containerOverrides())
                && Objects.equals(ecsPropertiesOverride(), other.ecsPropertiesOverride())
                && hasInstanceTypes() == other.hasInstanceTypes() && Objects.equals(instanceTypes(), other.instanceTypes())
                && Objects.equals(eksPropertiesOverride(), other.eksPropertiesOverride());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("NodePropertyOverride").add("TargetNodes", targetNodes())
                .add("ContainerOverrides", containerOverrides()).add("EcsPropertiesOverride", ecsPropertiesOverride())
                .add("InstanceTypes", hasInstanceTypes() ? instanceTypes() : null)
                .add("EksPropertiesOverride", eksPropertiesOverride()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "targetNodes":
            return Optional.ofNullable(clazz.cast(targetNodes()));
        case "containerOverrides":
            return Optional.ofNullable(clazz.cast(containerOverrides()));
        case "ecsPropertiesOverride":
            return Optional.ofNullable(clazz.cast(ecsPropertiesOverride()));
        case "instanceTypes":
            return Optional.ofNullable(clazz.cast(instanceTypes()));
        case "eksPropertiesOverride":
            return Optional.ofNullable(clazz.cast(eksPropertiesOverride()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<NodePropertyOverride, T> g) {
        return obj -> g.apply((NodePropertyOverride) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, NodePropertyOverride> {
        /**
         * <p>
         * The range of nodes, using node index values, that's used to override. A range of <code>0:3</code> indicates
         * nodes with index values of <code>0</code> through <code>3</code>. If the starting range value is omitted (
         * <code>:n</code>), then <code>0</code> is used to start the range. If the ending range value is omitted (
         * <code>n:</code>), then the highest possible node index is used to end the range.
         * </p>
         * 
         * @param targetNodes
         *        The range of nodes, using node index values, that's used to override. A range of <code>0:3</code>
         *        indicates nodes with index values of <code>0</code> through <code>3</code>. If the starting range
         *        value is omitted (<code>:n</code>), then <code>0</code> is used to start the range. If the ending
         *        range value is omitted (<code>n:</code>), then the highest possible node index is used to end the
         *        range.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetNodes(String targetNodes);

        /**
         * <p>
         * The overrides that are sent to a node range.
         * </p>
         * 
         * @param containerOverrides
         *        The overrides that are sent to a node range.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerOverrides(ContainerOverrides containerOverrides);

        /**
         * <p>
         * The overrides that are sent to a node range.
         * </p>
         * This is a convenience method that creates an instance of the {@link ContainerOverrides.Builder} avoiding the
         * need to create one manually via {@link ContainerOverrides#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ContainerOverrides.Builder#build()} is called immediately and its
         * result is passed to {@link #containerOverrides(ContainerOverrides)}.
         * 
         * @param containerOverrides
         *        a consumer that will call methods on {@link ContainerOverrides.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #containerOverrides(ContainerOverrides)
         */
        default Builder containerOverrides(Consumer<ContainerOverrides.Builder> containerOverrides) {
            return containerOverrides(ContainerOverrides.builder().applyMutation(containerOverrides).build());
        }

        /**
         * <p>
         * An object that contains the properties that you want to replace for the existing Amazon ECS resources of a
         * job.
         * </p>
         * 
         * @param ecsPropertiesOverride
         *        An object that contains the properties that you want to replace for the existing Amazon ECS resources
         *        of a job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ecsPropertiesOverride(EcsPropertiesOverride ecsPropertiesOverride);

        /**
         * <p>
         * An object that contains the properties that you want to replace for the existing Amazon ECS resources of a
         * job.
         * </p>
         * This is a convenience method that creates an instance of the {@link EcsPropertiesOverride.Builder} avoiding
         * the need to create one manually via {@link EcsPropertiesOverride#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EcsPropertiesOverride.Builder#build()} is called immediately and
         * its result is passed to {@link #ecsPropertiesOverride(EcsPropertiesOverride)}.
         * 
         * @param ecsPropertiesOverride
         *        a consumer that will call methods on {@link EcsPropertiesOverride.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ecsPropertiesOverride(EcsPropertiesOverride)
         */
        default Builder ecsPropertiesOverride(Consumer<EcsPropertiesOverride.Builder> ecsPropertiesOverride) {
            return ecsPropertiesOverride(EcsPropertiesOverride.builder().applyMutation(ecsPropertiesOverride).build());
        }

        /**
         * <p>
         * An object that contains the instance types that you want to replace for the existing resources of a job.
         * </p>
         * 
         * @param instanceTypes
         *        An object that contains the instance types that you want to replace for the existing resources of a
         *        job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceTypes(Collection<String> instanceTypes);

        /**
         * <p>
         * An object that contains the instance types that you want to replace for the existing resources of a job.
         * </p>
         * 
         * @param instanceTypes
         *        An object that contains the instance types that you want to replace for the existing resources of a
         *        job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceTypes(String... instanceTypes);

        /**
         * <p>
         * An object that contains the properties that you want to replace for the existing Amazon EKS resources of a
         * job.
         * </p>
         * 
         * @param eksPropertiesOverride
         *        An object that contains the properties that you want to replace for the existing Amazon EKS resources
         *        of a job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eksPropertiesOverride(EksPropertiesOverride eksPropertiesOverride);

        /**
         * <p>
         * An object that contains the properties that you want to replace for the existing Amazon EKS resources of a
         * job.
         * </p>
         * This is a convenience method that creates an instance of the {@link EksPropertiesOverride.Builder} avoiding
         * the need to create one manually via {@link EksPropertiesOverride#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EksPropertiesOverride.Builder#build()} is called immediately and
         * its result is passed to {@link #eksPropertiesOverride(EksPropertiesOverride)}.
         * 
         * @param eksPropertiesOverride
         *        a consumer that will call methods on {@link EksPropertiesOverride.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #eksPropertiesOverride(EksPropertiesOverride)
         */
        default Builder eksPropertiesOverride(Consumer<EksPropertiesOverride.Builder> eksPropertiesOverride) {
            return eksPropertiesOverride(EksPropertiesOverride.builder().applyMutation(eksPropertiesOverride).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String targetNodes;

        private ContainerOverrides containerOverrides;

        private EcsPropertiesOverride ecsPropertiesOverride;

        private List<String> instanceTypes = DefaultSdkAutoConstructList.getInstance();

        private EksPropertiesOverride eksPropertiesOverride;

        private BuilderImpl() {
        }

        private BuilderImpl(NodePropertyOverride model) {
            targetNodes(model.targetNodes);
            containerOverrides(model.containerOverrides);
            ecsPropertiesOverride(model.ecsPropertiesOverride);
            instanceTypes(model.instanceTypes);
            eksPropertiesOverride(model.eksPropertiesOverride);
        }

        public final String getTargetNodes() {
            return targetNodes;
        }

        public final void setTargetNodes(String targetNodes) {
            this.targetNodes = targetNodes;
        }

        @Override
        public final Builder targetNodes(String targetNodes) {
            this.targetNodes = targetNodes;
            return this;
        }

        public final ContainerOverrides.Builder getContainerOverrides() {
            return containerOverrides != null ? containerOverrides.toBuilder() : null;
        }

        public final void setContainerOverrides(ContainerOverrides.BuilderImpl containerOverrides) {
            this.containerOverrides = containerOverrides != null ? containerOverrides.build() : null;
        }

        @Override
        public final Builder containerOverrides(ContainerOverrides containerOverrides) {
            this.containerOverrides = containerOverrides;
            return this;
        }

        public final EcsPropertiesOverride.Builder getEcsPropertiesOverride() {
            return ecsPropertiesOverride != null ? ecsPropertiesOverride.toBuilder() : null;
        }

        public final void setEcsPropertiesOverride(EcsPropertiesOverride.BuilderImpl ecsPropertiesOverride) {
            this.ecsPropertiesOverride = ecsPropertiesOverride != null ? ecsPropertiesOverride.build() : null;
        }

        @Override
        public final Builder ecsPropertiesOverride(EcsPropertiesOverride ecsPropertiesOverride) {
            this.ecsPropertiesOverride = ecsPropertiesOverride;
            return this;
        }

        public final Collection<String> getInstanceTypes() {
            if (instanceTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return instanceTypes;
        }

        public final void setInstanceTypes(Collection<String> instanceTypes) {
            this.instanceTypes = StringListCopier.copy(instanceTypes);
        }

        @Override
        public final Builder instanceTypes(Collection<String> instanceTypes) {
            this.instanceTypes = StringListCopier.copy(instanceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceTypes(String... instanceTypes) {
            instanceTypes(Arrays.asList(instanceTypes));
            return this;
        }

        public final EksPropertiesOverride.Builder getEksPropertiesOverride() {
            return eksPropertiesOverride != null ? eksPropertiesOverride.toBuilder() : null;
        }

        public final void setEksPropertiesOverride(EksPropertiesOverride.BuilderImpl eksPropertiesOverride) {
            this.eksPropertiesOverride = eksPropertiesOverride != null ? eksPropertiesOverride.build() : null;
        }

        @Override
        public final Builder eksPropertiesOverride(EksPropertiesOverride eksPropertiesOverride) {
            this.eksPropertiesOverride = eksPropertiesOverride;
            return this;
        }

        @Override
        public NodePropertyOverride build() {
            return new NodePropertyOverride(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
