/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.batch.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the parameters for <code>CreateComputeEnvironment</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateComputeEnvironmentRequest extends BatchRequest implements
        ToCopyableBuilder<CreateComputeEnvironmentRequest.Builder, CreateComputeEnvironmentRequest> {
    private static final SdkField<String> COMPUTE_ENVIRONMENT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("computeEnvironmentName").getter(getter(CreateComputeEnvironmentRequest::computeEnvironmentName))
            .setter(setter(Builder::computeEnvironmentName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("computeEnvironmentName").build())
            .build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(CreateComputeEnvironmentRequest::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("state")
            .getter(getter(CreateComputeEnvironmentRequest::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("state").build()).build();

    private static final SdkField<Integer> UNMANAGEDV_CPUS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("unmanagedvCpus").getter(getter(CreateComputeEnvironmentRequest::unmanagedvCpus))
            .setter(setter(Builder::unmanagedvCpus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("unmanagedvCpus").build()).build();

    private static final SdkField<ComputeResource> COMPUTE_RESOURCES_FIELD = SdkField
            .<ComputeResource> builder(MarshallingType.SDK_POJO).memberName("computeResources")
            .getter(getter(CreateComputeEnvironmentRequest::computeResources)).setter(setter(Builder::computeResources))
            .constructor(ComputeResource::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("computeResources").build()).build();

    private static final SdkField<String> SERVICE_ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("serviceRole").getter(getter(CreateComputeEnvironmentRequest::serviceRole))
            .setter(setter(Builder::serviceRole))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("serviceRole").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(CreateComputeEnvironmentRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<EksConfiguration> EKS_CONFIGURATION_FIELD = SdkField
            .<EksConfiguration> builder(MarshallingType.SDK_POJO).memberName("eksConfiguration")
            .getter(getter(CreateComputeEnvironmentRequest::eksConfiguration)).setter(setter(Builder::eksConfiguration))
            .constructor(EksConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eksConfiguration").build()).build();

    private static final SdkField<String> CONTEXT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("context")
            .getter(getter(CreateComputeEnvironmentRequest::context)).setter(setter(Builder::context))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("context").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            COMPUTE_ENVIRONMENT_NAME_FIELD, TYPE_FIELD, STATE_FIELD, UNMANAGEDV_CPUS_FIELD, COMPUTE_RESOURCES_FIELD,
            SERVICE_ROLE_FIELD, TAGS_FIELD, EKS_CONFIGURATION_FIELD, CONTEXT_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("computeEnvironmentName", COMPUTE_ENVIRONMENT_NAME_FIELD);
                    put("type", TYPE_FIELD);
                    put("state", STATE_FIELD);
                    put("unmanagedvCpus", UNMANAGEDV_CPUS_FIELD);
                    put("computeResources", COMPUTE_RESOURCES_FIELD);
                    put("serviceRole", SERVICE_ROLE_FIELD);
                    put("tags", TAGS_FIELD);
                    put("eksConfiguration", EKS_CONFIGURATION_FIELD);
                    put("context", CONTEXT_FIELD);
                }
            });

    private final String computeEnvironmentName;

    private final String type;

    private final String state;

    private final Integer unmanagedvCpus;

    private final ComputeResource computeResources;

    private final String serviceRole;

    private final Map<String, String> tags;

    private final EksConfiguration eksConfiguration;

    private final String context;

    private CreateComputeEnvironmentRequest(BuilderImpl builder) {
        super(builder);
        this.computeEnvironmentName = builder.computeEnvironmentName;
        this.type = builder.type;
        this.state = builder.state;
        this.unmanagedvCpus = builder.unmanagedvCpus;
        this.computeResources = builder.computeResources;
        this.serviceRole = builder.serviceRole;
        this.tags = builder.tags;
        this.eksConfiguration = builder.eksConfiguration;
        this.context = builder.context;
    }

    /**
     * <p>
     * The name for your compute environment. It can be up to 128 characters long. It can contain uppercase and
     * lowercase letters, numbers, hyphens (-), and underscores (_).
     * </p>
     * 
     * @return The name for your compute environment. It can be up to 128 characters long. It can contain uppercase and
     *         lowercase letters, numbers, hyphens (-), and underscores (_).
     */
    public final String computeEnvironmentName() {
        return computeEnvironmentName;
    }

    /**
     * <p>
     * The type of the compute environment: <code>MANAGED</code> or <code>UNMANAGED</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html">Compute Environments</a> in
     * the <i>Batch User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link CEType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of the compute environment: <code>MANAGED</code> or <code>UNMANAGED</code>. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html">Compute
     *         Environments</a> in the <i>Batch User Guide</i>.
     * @see CEType
     */
    public final CEType type() {
        return CEType.fromValue(type);
    }

    /**
     * <p>
     * The type of the compute environment: <code>MANAGED</code> or <code>UNMANAGED</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html">Compute Environments</a> in
     * the <i>Batch User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link CEType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of the compute environment: <code>MANAGED</code> or <code>UNMANAGED</code>. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html">Compute
     *         Environments</a> in the <i>Batch User Guide</i>.
     * @see CEType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The state of the compute environment. If the state is <code>ENABLED</code>, then the compute environment accepts
     * jobs from a queue and can scale out automatically based on queues.
     * </p>
     * <p>
     * If the state is <code>ENABLED</code>, then the Batch scheduler can attempt to place jobs from an associated job
     * queue on the compute resources within the environment. If the compute environment is managed, then it can scale
     * its instances out or in automatically, based on the job queue demand.
     * </p>
     * <p>
     * If the state is <code>DISABLED</code>, then the Batch scheduler doesn't attempt to place jobs within the
     * environment. Jobs in a <code>STARTING</code> or <code>RUNNING</code> state continue to progress normally. Managed
     * compute environments in the <code>DISABLED</code> state don't scale out.
     * </p>
     * <note>
     * <p>
     * Compute environments in a <code>DISABLED</code> state may continue to incur billing charges. To prevent
     * additional charges, turn off and then delete the compute environment. For more information, see <a href=
     * "https://docs.aws.amazon.com/batch/latest/userguide/compute_environment_parameters.html#compute_environment_state"
     * >State</a> in the <i>Batch User Guide</i>.
     * </p>
     * </note>
     * <p>
     * When an instance is idle, the instance scales down to the <code>minvCpus</code> value. However, the instance size
     * doesn't change. For example, consider a <code>c5.8xlarge</code> instance with a <code>minvCpus</code> value of
     * <code>4</code> and a <code>desiredvCpus</code> value of <code>36</code>. This instance doesn't scale down to a
     * <code>c5.large</code> instance.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link CEState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the compute environment. If the state is <code>ENABLED</code>, then the compute environment
     *         accepts jobs from a queue and can scale out automatically based on queues.</p>
     *         <p>
     *         If the state is <code>ENABLED</code>, then the Batch scheduler can attempt to place jobs from an
     *         associated job queue on the compute resources within the environment. If the compute environment is
     *         managed, then it can scale its instances out or in automatically, based on the job queue demand.
     *         </p>
     *         <p>
     *         If the state is <code>DISABLED</code>, then the Batch scheduler doesn't attempt to place jobs within the
     *         environment. Jobs in a <code>STARTING</code> or <code>RUNNING</code> state continue to progress normally.
     *         Managed compute environments in the <code>DISABLED</code> state don't scale out.
     *         </p>
     *         <note>
     *         <p>
     *         Compute environments in a <code>DISABLED</code> state may continue to incur billing charges. To prevent
     *         additional charges, turn off and then delete the compute environment. For more information, see <a href=
     *         "https://docs.aws.amazon.com/batch/latest/userguide/compute_environment_parameters.html#compute_environment_state"
     *         >State</a> in the <i>Batch User Guide</i>.
     *         </p>
     *         </note>
     *         <p>
     *         When an instance is idle, the instance scales down to the <code>minvCpus</code> value. However, the
     *         instance size doesn't change. For example, consider a <code>c5.8xlarge</code> instance with a
     *         <code>minvCpus</code> value of <code>4</code> and a <code>desiredvCpus</code> value of <code>36</code>.
     *         This instance doesn't scale down to a <code>c5.large</code> instance.
     * @see CEState
     */
    public final CEState state() {
        return CEState.fromValue(state);
    }

    /**
     * <p>
     * The state of the compute environment. If the state is <code>ENABLED</code>, then the compute environment accepts
     * jobs from a queue and can scale out automatically based on queues.
     * </p>
     * <p>
     * If the state is <code>ENABLED</code>, then the Batch scheduler can attempt to place jobs from an associated job
     * queue on the compute resources within the environment. If the compute environment is managed, then it can scale
     * its instances out or in automatically, based on the job queue demand.
     * </p>
     * <p>
     * If the state is <code>DISABLED</code>, then the Batch scheduler doesn't attempt to place jobs within the
     * environment. Jobs in a <code>STARTING</code> or <code>RUNNING</code> state continue to progress normally. Managed
     * compute environments in the <code>DISABLED</code> state don't scale out.
     * </p>
     * <note>
     * <p>
     * Compute environments in a <code>DISABLED</code> state may continue to incur billing charges. To prevent
     * additional charges, turn off and then delete the compute environment. For more information, see <a href=
     * "https://docs.aws.amazon.com/batch/latest/userguide/compute_environment_parameters.html#compute_environment_state"
     * >State</a> in the <i>Batch User Guide</i>.
     * </p>
     * </note>
     * <p>
     * When an instance is idle, the instance scales down to the <code>minvCpus</code> value. However, the instance size
     * doesn't change. For example, consider a <code>c5.8xlarge</code> instance with a <code>minvCpus</code> value of
     * <code>4</code> and a <code>desiredvCpus</code> value of <code>36</code>. This instance doesn't scale down to a
     * <code>c5.large</code> instance.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link CEState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the compute environment. If the state is <code>ENABLED</code>, then the compute environment
     *         accepts jobs from a queue and can scale out automatically based on queues.</p>
     *         <p>
     *         If the state is <code>ENABLED</code>, then the Batch scheduler can attempt to place jobs from an
     *         associated job queue on the compute resources within the environment. If the compute environment is
     *         managed, then it can scale its instances out or in automatically, based on the job queue demand.
     *         </p>
     *         <p>
     *         If the state is <code>DISABLED</code>, then the Batch scheduler doesn't attempt to place jobs within the
     *         environment. Jobs in a <code>STARTING</code> or <code>RUNNING</code> state continue to progress normally.
     *         Managed compute environments in the <code>DISABLED</code> state don't scale out.
     *         </p>
     *         <note>
     *         <p>
     *         Compute environments in a <code>DISABLED</code> state may continue to incur billing charges. To prevent
     *         additional charges, turn off and then delete the compute environment. For more information, see <a href=
     *         "https://docs.aws.amazon.com/batch/latest/userguide/compute_environment_parameters.html#compute_environment_state"
     *         >State</a> in the <i>Batch User Guide</i>.
     *         </p>
     *         </note>
     *         <p>
     *         When an instance is idle, the instance scales down to the <code>minvCpus</code> value. However, the
     *         instance size doesn't change. For example, consider a <code>c5.8xlarge</code> instance with a
     *         <code>minvCpus</code> value of <code>4</code> and a <code>desiredvCpus</code> value of <code>36</code>.
     *         This instance doesn't scale down to a <code>c5.large</code> instance.
     * @see CEState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The maximum number of vCPUs for an unmanaged compute environment. This parameter is only used for fair share
     * scheduling to reserve vCPU capacity for new share identifiers. If this parameter isn't provided for a fair share
     * job queue, no vCPU capacity is reserved.
     * </p>
     * <note>
     * <p>
     * This parameter is only supported when the <code>type</code> parameter is set to <code>UNMANAGED</code>.
     * </p>
     * </note>
     * 
     * @return The maximum number of vCPUs for an unmanaged compute environment. This parameter is only used for fair
     *         share scheduling to reserve vCPU capacity for new share identifiers. If this parameter isn't provided for
     *         a fair share job queue, no vCPU capacity is reserved.</p> <note>
     *         <p>
     *         This parameter is only supported when the <code>type</code> parameter is set to <code>UNMANAGED</code>.
     *         </p>
     */
    public final Integer unmanagedvCpus() {
        return unmanagedvCpus;
    }

    /**
     * <p>
     * Details about the compute resources managed by the compute environment. This parameter is required for managed
     * compute environments. For more information, see <a
     * href="https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html">Compute Environments</a> in
     * the <i>Batch User Guide</i>.
     * </p>
     * 
     * @return Details about the compute resources managed by the compute environment. This parameter is required for
     *         managed compute environments. For more information, see <a
     *         href="https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html">Compute
     *         Environments</a> in the <i>Batch User Guide</i>.
     */
    public final ComputeResource computeResources() {
        return computeResources;
    }

    /**
     * <p>
     * The full Amazon Resource Name (ARN) of the IAM role that allows Batch to make calls to other Amazon Web Services
     * services on your behalf. For more information, see <a
     * href="https://docs.aws.amazon.com/batch/latest/userguide/service_IAM_role.html">Batch service IAM role</a> in the
     * <i>Batch User Guide</i>.
     * </p>
     * <important>
     * <p>
     * If your account already created the Batch service-linked role, that role is used by default for your compute
     * environment unless you specify a different role here. If the Batch service-linked role doesn't exist in your
     * account, and no role is specified here, the service attempts to create the Batch service-linked role in your
     * account.
     * </p>
     * </important>
     * <p>
     * If your specified role has a path other than <code>/</code>, then you must specify either the full role ARN
     * (recommended) or prefix the role name with the path. For example, if a role with the name <code>bar</code> has a
     * path of <code>/foo/</code>, specify <code>/foo/bar</code> as the role name. For more information, see <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-friendly-names"
     * >Friendly names and paths</a> in the <i>IAM User Guide</i>.
     * </p>
     * <note>
     * <p>
     * Depending on how you created your Batch service role, its ARN might contain the <code>service-role</code> path
     * prefix. When you only specify the name of the service role, Batch assumes that your ARN doesn't use the
     * <code>service-role</code> path prefix. Because of this, we recommend that you specify the full ARN of your
     * service role when you create compute environments.
     * </p>
     * </note>
     * 
     * @return The full Amazon Resource Name (ARN) of the IAM role that allows Batch to make calls to other Amazon Web
     *         Services services on your behalf. For more information, see <a
     *         href="https://docs.aws.amazon.com/batch/latest/userguide/service_IAM_role.html">Batch service IAM
     *         role</a> in the <i>Batch User Guide</i>.</p> <important>
     *         <p>
     *         If your account already created the Batch service-linked role, that role is used by default for your
     *         compute environment unless you specify a different role here. If the Batch service-linked role doesn't
     *         exist in your account, and no role is specified here, the service attempts to create the Batch
     *         service-linked role in your account.
     *         </p>
     *         </important>
     *         <p>
     *         If your specified role has a path other than <code>/</code>, then you must specify either the full role
     *         ARN (recommended) or prefix the role name with the path. For example, if a role with the name
     *         <code>bar</code> has a path of <code>/foo/</code>, specify <code>/foo/bar</code> as the role name. For
     *         more information, see <a href=
     *         "https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-friendly-names"
     *         >Friendly names and paths</a> in the <i>IAM User Guide</i>.
     *         </p>
     *         <note>
     *         <p>
     *         Depending on how you created your Batch service role, its ARN might contain the <code>service-role</code>
     *         path prefix. When you only specify the name of the service role, Batch assumes that your ARN doesn't use
     *         the <code>service-role</code> path prefix. Because of this, we recommend that you specify the full ARN of
     *         your service role when you create compute environments.
     *         </p>
     */
    public final String serviceRole() {
        return serviceRole;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The tags that you apply to the compute environment to help you categorize and organize your resources. Each tag
     * consists of a key and an optional value. For more information, see <a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging Amazon Web Services Resources</a>
     * in <i>Amazon Web Services General Reference</i>.
     * </p>
     * <p>
     * These tags can be updated or removed using the <a
     * href="https://docs.aws.amazon.com/batch/latest/APIReference/API_TagResource.html">TagResource</a> and <a
     * href="https://docs.aws.amazon.com/batch/latest/APIReference/API_UntagResource.html">UntagResource</a> API
     * operations. These tags don't propagate to the underlying compute resources.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags that you apply to the compute environment to help you categorize and organize your resources.
     *         Each tag consists of a key and an optional value. For more information, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging Amazon Web Services
     *         Resources</a> in <i>Amazon Web Services General Reference</i>.</p>
     *         <p>
     *         These tags can be updated or removed using the <a
     *         href="https://docs.aws.amazon.com/batch/latest/APIReference/API_TagResource.html">TagResource</a> and <a
     *         href="https://docs.aws.amazon.com/batch/latest/APIReference/API_UntagResource.html">UntagResource</a> API
     *         operations. These tags don't propagate to the underlying compute resources.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * The details for the Amazon EKS cluster that supports the compute environment.
     * </p>
     * 
     * @return The details for the Amazon EKS cluster that supports the compute environment.
     */
    public final EksConfiguration eksConfiguration() {
        return eksConfiguration;
    }

    /**
     * <p>
     * Reserved.
     * </p>
     * 
     * @return Reserved.
     */
    public final String context() {
        return context;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(computeEnvironmentName());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(unmanagedvCpus());
        hashCode = 31 * hashCode + Objects.hashCode(computeResources());
        hashCode = 31 * hashCode + Objects.hashCode(serviceRole());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(eksConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(context());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateComputeEnvironmentRequest)) {
            return false;
        }
        CreateComputeEnvironmentRequest other = (CreateComputeEnvironmentRequest) obj;
        return Objects.equals(computeEnvironmentName(), other.computeEnvironmentName())
                && Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(unmanagedvCpus(), other.unmanagedvCpus())
                && Objects.equals(computeResources(), other.computeResources())
                && Objects.equals(serviceRole(), other.serviceRole()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(eksConfiguration(), other.eksConfiguration())
                && Objects.equals(context(), other.context());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateComputeEnvironmentRequest").add("ComputeEnvironmentName", computeEnvironmentName())
                .add("Type", typeAsString()).add("State", stateAsString()).add("UnmanagedvCpus", unmanagedvCpus())
                .add("ComputeResources", computeResources()).add("ServiceRole", serviceRole())
                .add("Tags", hasTags() ? tags() : null).add("EksConfiguration", eksConfiguration()).add("Context", context())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "computeEnvironmentName":
            return Optional.ofNullable(clazz.cast(computeEnvironmentName()));
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "state":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "unmanagedvCpus":
            return Optional.ofNullable(clazz.cast(unmanagedvCpus()));
        case "computeResources":
            return Optional.ofNullable(clazz.cast(computeResources()));
        case "serviceRole":
            return Optional.ofNullable(clazz.cast(serviceRole()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "eksConfiguration":
            return Optional.ofNullable(clazz.cast(eksConfiguration()));
        case "context":
            return Optional.ofNullable(clazz.cast(context()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<CreateComputeEnvironmentRequest, T> g) {
        return obj -> g.apply((CreateComputeEnvironmentRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends BatchRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateComputeEnvironmentRequest> {
        /**
         * <p>
         * The name for your compute environment. It can be up to 128 characters long. It can contain uppercase and
         * lowercase letters, numbers, hyphens (-), and underscores (_).
         * </p>
         * 
         * @param computeEnvironmentName
         *        The name for your compute environment. It can be up to 128 characters long. It can contain uppercase
         *        and lowercase letters, numbers, hyphens (-), and underscores (_).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder computeEnvironmentName(String computeEnvironmentName);

        /**
         * <p>
         * The type of the compute environment: <code>MANAGED</code> or <code>UNMANAGED</code>. For more information,
         * see <a href="https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html">Compute
         * Environments</a> in the <i>Batch User Guide</i>.
         * </p>
         * 
         * @param type
         *        The type of the compute environment: <code>MANAGED</code> or <code>UNMANAGED</code>. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html">Compute
         *        Environments</a> in the <i>Batch User Guide</i>.
         * @see CEType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CEType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of the compute environment: <code>MANAGED</code> or <code>UNMANAGED</code>. For more information,
         * see <a href="https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html">Compute
         * Environments</a> in the <i>Batch User Guide</i>.
         * </p>
         * 
         * @param type
         *        The type of the compute environment: <code>MANAGED</code> or <code>UNMANAGED</code>. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html">Compute
         *        Environments</a> in the <i>Batch User Guide</i>.
         * @see CEType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CEType
         */
        Builder type(CEType type);

        /**
         * <p>
         * The state of the compute environment. If the state is <code>ENABLED</code>, then the compute environment
         * accepts jobs from a queue and can scale out automatically based on queues.
         * </p>
         * <p>
         * If the state is <code>ENABLED</code>, then the Batch scheduler can attempt to place jobs from an associated
         * job queue on the compute resources within the environment. If the compute environment is managed, then it can
         * scale its instances out or in automatically, based on the job queue demand.
         * </p>
         * <p>
         * If the state is <code>DISABLED</code>, then the Batch scheduler doesn't attempt to place jobs within the
         * environment. Jobs in a <code>STARTING</code> or <code>RUNNING</code> state continue to progress normally.
         * Managed compute environments in the <code>DISABLED</code> state don't scale out.
         * </p>
         * <note>
         * <p>
         * Compute environments in a <code>DISABLED</code> state may continue to incur billing charges. To prevent
         * additional charges, turn off and then delete the compute environment. For more information, see <a href=
         * "https://docs.aws.amazon.com/batch/latest/userguide/compute_environment_parameters.html#compute_environment_state"
         * >State</a> in the <i>Batch User Guide</i>.
         * </p>
         * </note>
         * <p>
         * When an instance is idle, the instance scales down to the <code>minvCpus</code> value. However, the instance
         * size doesn't change. For example, consider a <code>c5.8xlarge</code> instance with a <code>minvCpus</code>
         * value of <code>4</code> and a <code>desiredvCpus</code> value of <code>36</code>. This instance doesn't scale
         * down to a <code>c5.large</code> instance.
         * </p>
         * 
         * @param state
         *        The state of the compute environment. If the state is <code>ENABLED</code>, then the compute
         *        environment accepts jobs from a queue and can scale out automatically based on queues.</p>
         *        <p>
         *        If the state is <code>ENABLED</code>, then the Batch scheduler can attempt to place jobs from an
         *        associated job queue on the compute resources within the environment. If the compute environment is
         *        managed, then it can scale its instances out or in automatically, based on the job queue demand.
         *        </p>
         *        <p>
         *        If the state is <code>DISABLED</code>, then the Batch scheduler doesn't attempt to place jobs within
         *        the environment. Jobs in a <code>STARTING</code> or <code>RUNNING</code> state continue to progress
         *        normally. Managed compute environments in the <code>DISABLED</code> state don't scale out.
         *        </p>
         *        <note>
         *        <p>
         *        Compute environments in a <code>DISABLED</code> state may continue to incur billing charges. To
         *        prevent additional charges, turn off and then delete the compute environment. For more information,
         *        see <a href=
         *        "https://docs.aws.amazon.com/batch/latest/userguide/compute_environment_parameters.html#compute_environment_state"
         *        >State</a> in the <i>Batch User Guide</i>.
         *        </p>
         *        </note>
         *        <p>
         *        When an instance is idle, the instance scales down to the <code>minvCpus</code> value. However, the
         *        instance size doesn't change. For example, consider a <code>c5.8xlarge</code> instance with a
         *        <code>minvCpus</code> value of <code>4</code> and a <code>desiredvCpus</code> value of <code>36</code>
         *        . This instance doesn't scale down to a <code>c5.large</code> instance.
         * @see CEState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CEState
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the compute environment. If the state is <code>ENABLED</code>, then the compute environment
         * accepts jobs from a queue and can scale out automatically based on queues.
         * </p>
         * <p>
         * If the state is <code>ENABLED</code>, then the Batch scheduler can attempt to place jobs from an associated
         * job queue on the compute resources within the environment. If the compute environment is managed, then it can
         * scale its instances out or in automatically, based on the job queue demand.
         * </p>
         * <p>
         * If the state is <code>DISABLED</code>, then the Batch scheduler doesn't attempt to place jobs within the
         * environment. Jobs in a <code>STARTING</code> or <code>RUNNING</code> state continue to progress normally.
         * Managed compute environments in the <code>DISABLED</code> state don't scale out.
         * </p>
         * <note>
         * <p>
         * Compute environments in a <code>DISABLED</code> state may continue to incur billing charges. To prevent
         * additional charges, turn off and then delete the compute environment. For more information, see <a href=
         * "https://docs.aws.amazon.com/batch/latest/userguide/compute_environment_parameters.html#compute_environment_state"
         * >State</a> in the <i>Batch User Guide</i>.
         * </p>
         * </note>
         * <p>
         * When an instance is idle, the instance scales down to the <code>minvCpus</code> value. However, the instance
         * size doesn't change. For example, consider a <code>c5.8xlarge</code> instance with a <code>minvCpus</code>
         * value of <code>4</code> and a <code>desiredvCpus</code> value of <code>36</code>. This instance doesn't scale
         * down to a <code>c5.large</code> instance.
         * </p>
         * 
         * @param state
         *        The state of the compute environment. If the state is <code>ENABLED</code>, then the compute
         *        environment accepts jobs from a queue and can scale out automatically based on queues.</p>
         *        <p>
         *        If the state is <code>ENABLED</code>, then the Batch scheduler can attempt to place jobs from an
         *        associated job queue on the compute resources within the environment. If the compute environment is
         *        managed, then it can scale its instances out or in automatically, based on the job queue demand.
         *        </p>
         *        <p>
         *        If the state is <code>DISABLED</code>, then the Batch scheduler doesn't attempt to place jobs within
         *        the environment. Jobs in a <code>STARTING</code> or <code>RUNNING</code> state continue to progress
         *        normally. Managed compute environments in the <code>DISABLED</code> state don't scale out.
         *        </p>
         *        <note>
         *        <p>
         *        Compute environments in a <code>DISABLED</code> state may continue to incur billing charges. To
         *        prevent additional charges, turn off and then delete the compute environment. For more information,
         *        see <a href=
         *        "https://docs.aws.amazon.com/batch/latest/userguide/compute_environment_parameters.html#compute_environment_state"
         *        >State</a> in the <i>Batch User Guide</i>.
         *        </p>
         *        </note>
         *        <p>
         *        When an instance is idle, the instance scales down to the <code>minvCpus</code> value. However, the
         *        instance size doesn't change. For example, consider a <code>c5.8xlarge</code> instance with a
         *        <code>minvCpus</code> value of <code>4</code> and a <code>desiredvCpus</code> value of <code>36</code>
         *        . This instance doesn't scale down to a <code>c5.large</code> instance.
         * @see CEState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CEState
         */
        Builder state(CEState state);

        /**
         * <p>
         * The maximum number of vCPUs for an unmanaged compute environment. This parameter is only used for fair share
         * scheduling to reserve vCPU capacity for new share identifiers. If this parameter isn't provided for a fair
         * share job queue, no vCPU capacity is reserved.
         * </p>
         * <note>
         * <p>
         * This parameter is only supported when the <code>type</code> parameter is set to <code>UNMANAGED</code>.
         * </p>
         * </note>
         * 
         * @param unmanagedvCpus
         *        The maximum number of vCPUs for an unmanaged compute environment. This parameter is only used for fair
         *        share scheduling to reserve vCPU capacity for new share identifiers. If this parameter isn't provided
         *        for a fair share job queue, no vCPU capacity is reserved.</p> <note>
         *        <p>
         *        This parameter is only supported when the <code>type</code> parameter is set to <code>UNMANAGED</code>
         *        .
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder unmanagedvCpus(Integer unmanagedvCpus);

        /**
         * <p>
         * Details about the compute resources managed by the compute environment. This parameter is required for
         * managed compute environments. For more information, see <a
         * href="https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html">Compute Environments</a>
         * in the <i>Batch User Guide</i>.
         * </p>
         * 
         * @param computeResources
         *        Details about the compute resources managed by the compute environment. This parameter is required for
         *        managed compute environments. For more information, see <a
         *        href="https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html">Compute
         *        Environments</a> in the <i>Batch User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder computeResources(ComputeResource computeResources);

        /**
         * <p>
         * Details about the compute resources managed by the compute environment. This parameter is required for
         * managed compute environments. For more information, see <a
         * href="https://docs.aws.amazon.com/batch/latest/userguide/compute_environments.html">Compute Environments</a>
         * in the <i>Batch User Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the {@link ComputeResource.Builder} avoiding the
         * need to create one manually via {@link ComputeResource#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ComputeResource.Builder#build()} is called immediately and its
         * result is passed to {@link #computeResources(ComputeResource)}.
         * 
         * @param computeResources
         *        a consumer that will call methods on {@link ComputeResource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #computeResources(ComputeResource)
         */
        default Builder computeResources(Consumer<ComputeResource.Builder> computeResources) {
            return computeResources(ComputeResource.builder().applyMutation(computeResources).build());
        }

        /**
         * <p>
         * The full Amazon Resource Name (ARN) of the IAM role that allows Batch to make calls to other Amazon Web
         * Services services on your behalf. For more information, see <a
         * href="https://docs.aws.amazon.com/batch/latest/userguide/service_IAM_role.html">Batch service IAM role</a> in
         * the <i>Batch User Guide</i>.
         * </p>
         * <important>
         * <p>
         * If your account already created the Batch service-linked role, that role is used by default for your compute
         * environment unless you specify a different role here. If the Batch service-linked role doesn't exist in your
         * account, and no role is specified here, the service attempts to create the Batch service-linked role in your
         * account.
         * </p>
         * </important>
         * <p>
         * If your specified role has a path other than <code>/</code>, then you must specify either the full role ARN
         * (recommended) or prefix the role name with the path. For example, if a role with the name <code>bar</code>
         * has a path of <code>/foo/</code>, specify <code>/foo/bar</code> as the role name. For more information, see
         * <a href=
         * "https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-friendly-names"
         * >Friendly names and paths</a> in the <i>IAM User Guide</i>.
         * </p>
         * <note>
         * <p>
         * Depending on how you created your Batch service role, its ARN might contain the <code>service-role</code>
         * path prefix. When you only specify the name of the service role, Batch assumes that your ARN doesn't use the
         * <code>service-role</code> path prefix. Because of this, we recommend that you specify the full ARN of your
         * service role when you create compute environments.
         * </p>
         * </note>
         * 
         * @param serviceRole
         *        The full Amazon Resource Name (ARN) of the IAM role that allows Batch to make calls to other Amazon
         *        Web Services services on your behalf. For more information, see <a
         *        href="https://docs.aws.amazon.com/batch/latest/userguide/service_IAM_role.html">Batch service IAM
         *        role</a> in the <i>Batch User Guide</i>.</p> <important>
         *        <p>
         *        If your account already created the Batch service-linked role, that role is used by default for your
         *        compute environment unless you specify a different role here. If the Batch service-linked role doesn't
         *        exist in your account, and no role is specified here, the service attempts to create the Batch
         *        service-linked role in your account.
         *        </p>
         *        </important>
         *        <p>
         *        If your specified role has a path other than <code>/</code>, then you must specify either the full
         *        role ARN (recommended) or prefix the role name with the path. For example, if a role with the name
         *        <code>bar</code> has a path of <code>/foo/</code>, specify <code>/foo/bar</code> as the role name. For
         *        more information, see <a href=
         *        "https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-friendly-names"
         *        >Friendly names and paths</a> in the <i>IAM User Guide</i>.
         *        </p>
         *        <note>
         *        <p>
         *        Depending on how you created your Batch service role, its ARN might contain the
         *        <code>service-role</code> path prefix. When you only specify the name of the service role, Batch
         *        assumes that your ARN doesn't use the <code>service-role</code> path prefix. Because of this, we
         *        recommend that you specify the full ARN of your service role when you create compute environments.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceRole(String serviceRole);

        /**
         * <p>
         * The tags that you apply to the compute environment to help you categorize and organize your resources. Each
         * tag consists of a key and an optional value. For more information, see <a
         * href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging Amazon Web Services
         * Resources</a> in <i>Amazon Web Services General Reference</i>.
         * </p>
         * <p>
         * These tags can be updated or removed using the <a
         * href="https://docs.aws.amazon.com/batch/latest/APIReference/API_TagResource.html">TagResource</a> and <a
         * href="https://docs.aws.amazon.com/batch/latest/APIReference/API_UntagResource.html">UntagResource</a> API
         * operations. These tags don't propagate to the underlying compute resources.
         * </p>
         * 
         * @param tags
         *        The tags that you apply to the compute environment to help you categorize and organize your resources.
         *        Each tag consists of a key and an optional value. For more information, see <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging Amazon Web Services
         *        Resources</a> in <i>Amazon Web Services General Reference</i>.</p>
         *        <p>
         *        These tags can be updated or removed using the <a
         *        href="https://docs.aws.amazon.com/batch/latest/APIReference/API_TagResource.html">TagResource</a> and
         *        <a
         *        href="https://docs.aws.amazon.com/batch/latest/APIReference/API_UntagResource.html">UntagResource</a>
         *        API operations. These tags don't propagate to the underlying compute resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * The details for the Amazon EKS cluster that supports the compute environment.
         * </p>
         * 
         * @param eksConfiguration
         *        The details for the Amazon EKS cluster that supports the compute environment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eksConfiguration(EksConfiguration eksConfiguration);

        /**
         * <p>
         * The details for the Amazon EKS cluster that supports the compute environment.
         * </p>
         * This is a convenience method that creates an instance of the {@link EksConfiguration.Builder} avoiding the
         * need to create one manually via {@link EksConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EksConfiguration.Builder#build()} is called immediately and its
         * result is passed to {@link #eksConfiguration(EksConfiguration)}.
         * 
         * @param eksConfiguration
         *        a consumer that will call methods on {@link EksConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #eksConfiguration(EksConfiguration)
         */
        default Builder eksConfiguration(Consumer<EksConfiguration.Builder> eksConfiguration) {
            return eksConfiguration(EksConfiguration.builder().applyMutation(eksConfiguration).build());
        }

        /**
         * <p>
         * Reserved.
         * </p>
         * 
         * @param context
         *        Reserved.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder context(String context);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends BatchRequest.BuilderImpl implements Builder {
        private String computeEnvironmentName;

        private String type;

        private String state;

        private Integer unmanagedvCpus;

        private ComputeResource computeResources;

        private String serviceRole;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private EksConfiguration eksConfiguration;

        private String context;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateComputeEnvironmentRequest model) {
            super(model);
            computeEnvironmentName(model.computeEnvironmentName);
            type(model.type);
            state(model.state);
            unmanagedvCpus(model.unmanagedvCpus);
            computeResources(model.computeResources);
            serviceRole(model.serviceRole);
            tags(model.tags);
            eksConfiguration(model.eksConfiguration);
            context(model.context);
        }

        public final String getComputeEnvironmentName() {
            return computeEnvironmentName;
        }

        public final void setComputeEnvironmentName(String computeEnvironmentName) {
            this.computeEnvironmentName = computeEnvironmentName;
        }

        @Override
        public final Builder computeEnvironmentName(String computeEnvironmentName) {
            this.computeEnvironmentName = computeEnvironmentName;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(CEType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(CEState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final Integer getUnmanagedvCpus() {
            return unmanagedvCpus;
        }

        public final void setUnmanagedvCpus(Integer unmanagedvCpus) {
            this.unmanagedvCpus = unmanagedvCpus;
        }

        @Override
        public final Builder unmanagedvCpus(Integer unmanagedvCpus) {
            this.unmanagedvCpus = unmanagedvCpus;
            return this;
        }

        public final ComputeResource.Builder getComputeResources() {
            return computeResources != null ? computeResources.toBuilder() : null;
        }

        public final void setComputeResources(ComputeResource.BuilderImpl computeResources) {
            this.computeResources = computeResources != null ? computeResources.build() : null;
        }

        @Override
        public final Builder computeResources(ComputeResource computeResources) {
            this.computeResources = computeResources;
            return this;
        }

        public final String getServiceRole() {
            return serviceRole;
        }

        public final void setServiceRole(String serviceRole) {
            this.serviceRole = serviceRole;
        }

        @Override
        public final Builder serviceRole(String serviceRole) {
            this.serviceRole = serviceRole;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagrisTagsMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagrisTagsMapCopier.copy(tags);
            return this;
        }

        public final EksConfiguration.Builder getEksConfiguration() {
            return eksConfiguration != null ? eksConfiguration.toBuilder() : null;
        }

        public final void setEksConfiguration(EksConfiguration.BuilderImpl eksConfiguration) {
            this.eksConfiguration = eksConfiguration != null ? eksConfiguration.build() : null;
        }

        @Override
        public final Builder eksConfiguration(EksConfiguration eksConfiguration) {
            this.eksConfiguration = eksConfiguration;
            return this;
        }

        public final String getContext() {
            return context;
        }

        public final void setContext(String context) {
            this.context = context;
        }

        @Override
        public final Builder context(String context) {
            this.context = context;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateComputeEnvironmentRequest build() {
            return new CreateComputeEnvironmentRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
