/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.batch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object that represents a launch template to use in place of the default launch template. You must specify either
 * the launch template ID or launch template name in the request, but not both.
 * </p>
 * <p>
 * If security groups are specified using both the <code>securityGroupIds</code> parameter of
 * <code>CreateComputeEnvironment</code> and the launch template, the values in the <code>securityGroupIds</code>
 * parameter of <code>CreateComputeEnvironment</code> will be used.
 * </p>
 * <p>
 * You can define up to ten (10) overrides for each compute environment.
 * </p>
 * <note>
 * <p>
 * This object isn't applicable to jobs that are running on Fargate resources.
 * </p>
 * </note> <note>
 * <p>
 * To unset all override templates for a compute environment, you can pass an empty array to the <a
 * href="https://docs.aws.amazon.com/batch/latest/APIReference/API_UpdateComputeEnvironment.html"
 * >UpdateComputeEnvironment.overrides</a> parameter, or not include the <code>overrides</code> parameter when
 * submitting the <code>UpdateComputeEnvironment</code> API operation.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LaunchTemplateSpecificationOverride implements SdkPojo, Serializable,
        ToCopyableBuilder<LaunchTemplateSpecificationOverride.Builder, LaunchTemplateSpecificationOverride> {
    private static final SdkField<String> LAUNCH_TEMPLATE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("launchTemplateId").getter(getter(LaunchTemplateSpecificationOverride::launchTemplateId))
            .setter(setter(Builder::launchTemplateId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("launchTemplateId").build()).build();

    private static final SdkField<String> LAUNCH_TEMPLATE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("launchTemplateName").getter(getter(LaunchTemplateSpecificationOverride::launchTemplateName))
            .setter(setter(Builder::launchTemplateName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("launchTemplateName").build())
            .build();

    private static final SdkField<String> VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("version")
            .getter(getter(LaunchTemplateSpecificationOverride::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("version").build()).build();

    private static final SdkField<List<String>> TARGET_INSTANCE_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("targetInstanceTypes")
            .getter(getter(LaunchTemplateSpecificationOverride::targetInstanceTypes))
            .setter(setter(Builder::targetInstanceTypes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("targetInstanceTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(LAUNCH_TEMPLATE_ID_FIELD,
            LAUNCH_TEMPLATE_NAME_FIELD, VERSION_FIELD, TARGET_INSTANCE_TYPES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String launchTemplateId;

    private final String launchTemplateName;

    private final String version;

    private final List<String> targetInstanceTypes;

    private LaunchTemplateSpecificationOverride(BuilderImpl builder) {
        this.launchTemplateId = builder.launchTemplateId;
        this.launchTemplateName = builder.launchTemplateName;
        this.version = builder.version;
        this.targetInstanceTypes = builder.targetInstanceTypes;
    }

    /**
     * <p>
     * The ID of the launch template.
     * </p>
     * <p>
     * <b>Note:</b> If you specify the <code>launchTemplateId</code> you can't specify the
     * <code>launchTemplateName</code> as well.
     * </p>
     * 
     * @return The ID of the launch template.</p>
     *         <p>
     *         <b>Note:</b> If you specify the <code>launchTemplateId</code> you can't specify the
     *         <code>launchTemplateName</code> as well.
     */
    public final String launchTemplateId() {
        return launchTemplateId;
    }

    /**
     * <p>
     * The name of the launch template.
     * </p>
     * <p>
     * <b>Note:</b> If you specify the <code>launchTemplateName</code> you can't specify the
     * <code>launchTemplateId</code> as well.
     * </p>
     * 
     * @return The name of the launch template.</p>
     *         <p>
     *         <b>Note:</b> If you specify the <code>launchTemplateName</code> you can't specify the
     *         <code>launchTemplateId</code> as well.
     */
    public final String launchTemplateName() {
        return launchTemplateName;
    }

    /**
     * <p>
     * The version number of the launch template, <code>$Default</code>, or <code>$Latest</code>.
     * </p>
     * <p>
     * If the value is <code>$Default</code>, the default version of the launch template is used. If the value is
     * <code>$Latest</code>, the latest version of the launch template is used.
     * </p>
     * <important>
     * <p>
     * If the AMI ID that's used in a compute environment is from the launch template, the AMI isn't changed when the
     * compute environment is updated. It's only changed if the <code>updateToLatestImageVersion</code> parameter for
     * the compute environment is set to <code>true</code>. During an infrastructure update, if either
     * <code>$Default</code> or <code>$Latest</code> is specified, Batch re-evaluates the launch template version, and
     * it might use a different version of the launch template. This is the case even if the launch template isn't
     * specified in the update. When updating a compute environment, changing the launch template requires an
     * infrastructure update of the compute environment. For more information, see <a
     * href="https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html">Updating compute
     * environments</a> in the <i>Batch User Guide</i>.
     * </p>
     * </important>
     * <p>
     * Default: <code>$Default</code>
     * </p>
     * <p>
     * Latest: <code>$Latest</code>
     * </p>
     * 
     * @return The version number of the launch template, <code>$Default</code>, or <code>$Latest</code>.</p>
     *         <p>
     *         If the value is <code>$Default</code>, the default version of the launch template is used. If the value
     *         is <code>$Latest</code>, the latest version of the launch template is used.
     *         </p>
     *         <important>
     *         <p>
     *         If the AMI ID that's used in a compute environment is from the launch template, the AMI isn't changed
     *         when the compute environment is updated. It's only changed if the <code>updateToLatestImageVersion</code>
     *         parameter for the compute environment is set to <code>true</code>. During an infrastructure update, if
     *         either <code>$Default</code> or <code>$Latest</code> is specified, Batch re-evaluates the launch template
     *         version, and it might use a different version of the launch template. This is the case even if the launch
     *         template isn't specified in the update. When updating a compute environment, changing the launch template
     *         requires an infrastructure update of the compute environment. For more information, see <a
     *         href="https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html">Updating
     *         compute environments</a> in the <i>Batch User Guide</i>.
     *         </p>
     *         </important>
     *         <p>
     *         Default: <code>$Default</code>
     *         </p>
     *         <p>
     *         Latest: <code>$Latest</code>
     */
    public final String version() {
        return version;
    }

    /**
     * For responses, this returns true if the service returned a value for the TargetInstanceTypes property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTargetInstanceTypes() {
        return targetInstanceTypes != null && !(targetInstanceTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The instance type or family that this this override launch template should be applied to.
     * </p>
     * <p>
     * This parameter is required when defining a launch template override.
     * </p>
     * <p>
     * Information included in this parameter must meet the following requirements:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Must be a valid Amazon EC2 instance type or family.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>optimal</code> isn't allowed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>targetInstanceTypes</code> can target only instance types and families that are included within the <a
     * href=
     * "https://docs.aws.amazon.com/batch/latest/APIReference/API_ComputeResource.html#Batch-Type-ComputeResource-instanceTypes"
     * > <code>ComputeResource.instanceTypes</code> </a> set. <code>targetInstanceTypes</code> doesn't need to include
     * all of the instances from the <code>instanceType</code> set, but at least a subset. For example, if
     * <code>ComputeResource.instanceTypes</code> includes <code>[m5, g5]</code>, <code>targetInstanceTypes</code> can
     * include <code>[m5.2xlarge]</code> and <code>[m5.large]</code> but not <code>[c5.large]</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>targetInstanceTypes</code> included within the same launch template override or across launch template
     * overrides can't overlap for the same compute environment. For example, you can't define one launch template
     * override to target an instance family and another define an instance type within this same family.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTargetInstanceTypes} method.
     * </p>
     * 
     * @return The instance type or family that this this override launch template should be applied to.</p>
     *         <p>
     *         This parameter is required when defining a launch template override.
     *         </p>
     *         <p>
     *         Information included in this parameter must meet the following requirements:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Must be a valid Amazon EC2 instance type or family.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>optimal</code> isn't allowed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>targetInstanceTypes</code> can target only instance types and families that are included within the
     *         <a href=
     *         "https://docs.aws.amazon.com/batch/latest/APIReference/API_ComputeResource.html#Batch-Type-ComputeResource-instanceTypes"
     *         > <code>ComputeResource.instanceTypes</code> </a> set. <code>targetInstanceTypes</code> doesn't need to
     *         include all of the instances from the <code>instanceType</code> set, but at least a subset. For example,
     *         if <code>ComputeResource.instanceTypes</code> includes <code>[m5, g5]</code>,
     *         <code>targetInstanceTypes</code> can include <code>[m5.2xlarge]</code> and <code>[m5.large]</code> but
     *         not <code>[c5.large]</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>targetInstanceTypes</code> included within the same launch template override or across launch
     *         template overrides can't overlap for the same compute environment. For example, you can't define one
     *         launch template override to target an instance family and another define an instance type within this
     *         same family.
     *         </p>
     *         </li>
     */
    public final List<String> targetInstanceTypes() {
        return targetInstanceTypes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(launchTemplateId());
        hashCode = 31 * hashCode + Objects.hashCode(launchTemplateName());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(hasTargetInstanceTypes() ? targetInstanceTypes() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LaunchTemplateSpecificationOverride)) {
            return false;
        }
        LaunchTemplateSpecificationOverride other = (LaunchTemplateSpecificationOverride) obj;
        return Objects.equals(launchTemplateId(), other.launchTemplateId())
                && Objects.equals(launchTemplateName(), other.launchTemplateName()) && Objects.equals(version(), other.version())
                && hasTargetInstanceTypes() == other.hasTargetInstanceTypes()
                && Objects.equals(targetInstanceTypes(), other.targetInstanceTypes());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("LaunchTemplateSpecificationOverride").add("LaunchTemplateId", launchTemplateId())
                .add("LaunchTemplateName", launchTemplateName()).add("Version", version())
                .add("TargetInstanceTypes", hasTargetInstanceTypes() ? targetInstanceTypes() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "launchTemplateId":
            return Optional.ofNullable(clazz.cast(launchTemplateId()));
        case "launchTemplateName":
            return Optional.ofNullable(clazz.cast(launchTemplateName()));
        case "version":
            return Optional.ofNullable(clazz.cast(version()));
        case "targetInstanceTypes":
            return Optional.ofNullable(clazz.cast(targetInstanceTypes()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("launchTemplateId", LAUNCH_TEMPLATE_ID_FIELD);
        map.put("launchTemplateName", LAUNCH_TEMPLATE_NAME_FIELD);
        map.put("version", VERSION_FIELD);
        map.put("targetInstanceTypes", TARGET_INSTANCE_TYPES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<LaunchTemplateSpecificationOverride, T> g) {
        return obj -> g.apply((LaunchTemplateSpecificationOverride) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LaunchTemplateSpecificationOverride> {
        /**
         * <p>
         * The ID of the launch template.
         * </p>
         * <p>
         * <b>Note:</b> If you specify the <code>launchTemplateId</code> you can't specify the
         * <code>launchTemplateName</code> as well.
         * </p>
         * 
         * @param launchTemplateId
         *        The ID of the launch template.</p>
         *        <p>
         *        <b>Note:</b> If you specify the <code>launchTemplateId</code> you can't specify the
         *        <code>launchTemplateName</code> as well.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder launchTemplateId(String launchTemplateId);

        /**
         * <p>
         * The name of the launch template.
         * </p>
         * <p>
         * <b>Note:</b> If you specify the <code>launchTemplateName</code> you can't specify the
         * <code>launchTemplateId</code> as well.
         * </p>
         * 
         * @param launchTemplateName
         *        The name of the launch template.</p>
         *        <p>
         *        <b>Note:</b> If you specify the <code>launchTemplateName</code> you can't specify the
         *        <code>launchTemplateId</code> as well.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder launchTemplateName(String launchTemplateName);

        /**
         * <p>
         * The version number of the launch template, <code>$Default</code>, or <code>$Latest</code>.
         * </p>
         * <p>
         * If the value is <code>$Default</code>, the default version of the launch template is used. If the value is
         * <code>$Latest</code>, the latest version of the launch template is used.
         * </p>
         * <important>
         * <p>
         * If the AMI ID that's used in a compute environment is from the launch template, the AMI isn't changed when
         * the compute environment is updated. It's only changed if the <code>updateToLatestImageVersion</code>
         * parameter for the compute environment is set to <code>true</code>. During an infrastructure update, if either
         * <code>$Default</code> or <code>$Latest</code> is specified, Batch re-evaluates the launch template version,
         * and it might use a different version of the launch template. This is the case even if the launch template
         * isn't specified in the update. When updating a compute environment, changing the launch template requires an
         * infrastructure update of the compute environment. For more information, see <a
         * href="https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html">Updating compute
         * environments</a> in the <i>Batch User Guide</i>.
         * </p>
         * </important>
         * <p>
         * Default: <code>$Default</code>
         * </p>
         * <p>
         * Latest: <code>$Latest</code>
         * </p>
         * 
         * @param version
         *        The version number of the launch template, <code>$Default</code>, or <code>$Latest</code>.</p>
         *        <p>
         *        If the value is <code>$Default</code>, the default version of the launch template is used. If the
         *        value is <code>$Latest</code>, the latest version of the launch template is used.
         *        </p>
         *        <important>
         *        <p>
         *        If the AMI ID that's used in a compute environment is from the launch template, the AMI isn't changed
         *        when the compute environment is updated. It's only changed if the
         *        <code>updateToLatestImageVersion</code> parameter for the compute environment is set to
         *        <code>true</code>. During an infrastructure update, if either <code>$Default</code> or
         *        <code>$Latest</code> is specified, Batch re-evaluates the launch template version, and it might use a
         *        different version of the launch template. This is the case even if the launch template isn't specified
         *        in the update. When updating a compute environment, changing the launch template requires an
         *        infrastructure update of the compute environment. For more information, see <a
         *        href="https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html">Updating
         *        compute environments</a> in the <i>Batch User Guide</i>.
         *        </p>
         *        </important>
         *        <p>
         *        Default: <code>$Default</code>
         *        </p>
         *        <p>
         *        Latest: <code>$Latest</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);

        /**
         * <p>
         * The instance type or family that this this override launch template should be applied to.
         * </p>
         * <p>
         * This parameter is required when defining a launch template override.
         * </p>
         * <p>
         * Information included in this parameter must meet the following requirements:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Must be a valid Amazon EC2 instance type or family.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>optimal</code> isn't allowed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>targetInstanceTypes</code> can target only instance types and families that are included within the <a
         * href=
         * "https://docs.aws.amazon.com/batch/latest/APIReference/API_ComputeResource.html#Batch-Type-ComputeResource-instanceTypes"
         * > <code>ComputeResource.instanceTypes</code> </a> set. <code>targetInstanceTypes</code> doesn't need to
         * include all of the instances from the <code>instanceType</code> set, but at least a subset. For example, if
         * <code>ComputeResource.instanceTypes</code> includes <code>[m5, g5]</code>, <code>targetInstanceTypes</code>
         * can include <code>[m5.2xlarge]</code> and <code>[m5.large]</code> but not <code>[c5.large]</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>targetInstanceTypes</code> included within the same launch template override or across launch template
         * overrides can't overlap for the same compute environment. For example, you can't define one launch template
         * override to target an instance family and another define an instance type within this same family.
         * </p>
         * </li>
         * </ul>
         * 
         * @param targetInstanceTypes
         *        The instance type or family that this this override launch template should be applied to.</p>
         *        <p>
         *        This parameter is required when defining a launch template override.
         *        </p>
         *        <p>
         *        Information included in this parameter must meet the following requirements:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Must be a valid Amazon EC2 instance type or family.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>optimal</code> isn't allowed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>targetInstanceTypes</code> can target only instance types and families that are included within
         *        the <a href=
         *        "https://docs.aws.amazon.com/batch/latest/APIReference/API_ComputeResource.html#Batch-Type-ComputeResource-instanceTypes"
         *        > <code>ComputeResource.instanceTypes</code> </a> set. <code>targetInstanceTypes</code> doesn't need
         *        to include all of the instances from the <code>instanceType</code> set, but at least a subset. For
         *        example, if <code>ComputeResource.instanceTypes</code> includes <code>[m5, g5]</code>,
         *        <code>targetInstanceTypes</code> can include <code>[m5.2xlarge]</code> and <code>[m5.large]</code> but
         *        not <code>[c5.large]</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>targetInstanceTypes</code> included within the same launch template override or across launch
         *        template overrides can't overlap for the same compute environment. For example, you can't define one
         *        launch template override to target an instance family and another define an instance type within this
         *        same family.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetInstanceTypes(Collection<String> targetInstanceTypes);

        /**
         * <p>
         * The instance type or family that this this override launch template should be applied to.
         * </p>
         * <p>
         * This parameter is required when defining a launch template override.
         * </p>
         * <p>
         * Information included in this parameter must meet the following requirements:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Must be a valid Amazon EC2 instance type or family.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>optimal</code> isn't allowed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>targetInstanceTypes</code> can target only instance types and families that are included within the <a
         * href=
         * "https://docs.aws.amazon.com/batch/latest/APIReference/API_ComputeResource.html#Batch-Type-ComputeResource-instanceTypes"
         * > <code>ComputeResource.instanceTypes</code> </a> set. <code>targetInstanceTypes</code> doesn't need to
         * include all of the instances from the <code>instanceType</code> set, but at least a subset. For example, if
         * <code>ComputeResource.instanceTypes</code> includes <code>[m5, g5]</code>, <code>targetInstanceTypes</code>
         * can include <code>[m5.2xlarge]</code> and <code>[m5.large]</code> but not <code>[c5.large]</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>targetInstanceTypes</code> included within the same launch template override or across launch template
         * overrides can't overlap for the same compute environment. For example, you can't define one launch template
         * override to target an instance family and another define an instance type within this same family.
         * </p>
         * </li>
         * </ul>
         * 
         * @param targetInstanceTypes
         *        The instance type or family that this this override launch template should be applied to.</p>
         *        <p>
         *        This parameter is required when defining a launch template override.
         *        </p>
         *        <p>
         *        Information included in this parameter must meet the following requirements:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Must be a valid Amazon EC2 instance type or family.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>optimal</code> isn't allowed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>targetInstanceTypes</code> can target only instance types and families that are included within
         *        the <a href=
         *        "https://docs.aws.amazon.com/batch/latest/APIReference/API_ComputeResource.html#Batch-Type-ComputeResource-instanceTypes"
         *        > <code>ComputeResource.instanceTypes</code> </a> set. <code>targetInstanceTypes</code> doesn't need
         *        to include all of the instances from the <code>instanceType</code> set, but at least a subset. For
         *        example, if <code>ComputeResource.instanceTypes</code> includes <code>[m5, g5]</code>,
         *        <code>targetInstanceTypes</code> can include <code>[m5.2xlarge]</code> and <code>[m5.large]</code> but
         *        not <code>[c5.large]</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>targetInstanceTypes</code> included within the same launch template override or across launch
         *        template overrides can't overlap for the same compute environment. For example, you can't define one
         *        launch template override to target an instance family and another define an instance type within this
         *        same family.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetInstanceTypes(String... targetInstanceTypes);
    }

    static final class BuilderImpl implements Builder {
        private String launchTemplateId;

        private String launchTemplateName;

        private String version;

        private List<String> targetInstanceTypes = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(LaunchTemplateSpecificationOverride model) {
            launchTemplateId(model.launchTemplateId);
            launchTemplateName(model.launchTemplateName);
            version(model.version);
            targetInstanceTypes(model.targetInstanceTypes);
        }

        public final String getLaunchTemplateId() {
            return launchTemplateId;
        }

        public final void setLaunchTemplateId(String launchTemplateId) {
            this.launchTemplateId = launchTemplateId;
        }

        @Override
        public final Builder launchTemplateId(String launchTemplateId) {
            this.launchTemplateId = launchTemplateId;
            return this;
        }

        public final String getLaunchTemplateName() {
            return launchTemplateName;
        }

        public final void setLaunchTemplateName(String launchTemplateName) {
            this.launchTemplateName = launchTemplateName;
        }

        @Override
        public final Builder launchTemplateName(String launchTemplateName) {
            this.launchTemplateName = launchTemplateName;
            return this;
        }

        public final String getVersion() {
            return version;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        public final Collection<String> getTargetInstanceTypes() {
            if (targetInstanceTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return targetInstanceTypes;
        }

        public final void setTargetInstanceTypes(Collection<String> targetInstanceTypes) {
            this.targetInstanceTypes = StringListCopier.copy(targetInstanceTypes);
        }

        @Override
        public final Builder targetInstanceTypes(Collection<String> targetInstanceTypes) {
            this.targetInstanceTypes = StringListCopier.copy(targetInstanceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targetInstanceTypes(String... targetInstanceTypes) {
            targetInstanceTypes(Arrays.asList(targetInstanceTypes));
            return this;
        }

        @Override
        public LaunchTemplateSpecificationOverride build() {
            return new LaunchTemplateSpecificationOverride(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
