/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.batch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The overrides that should be sent to a container.
 * </p>
 * <p>
 * For information about using Batch overrides when you connect event sources to targets, see <a
 * href="https://docs.aws.amazon.com/eventbridge/latest/pipes-reference/API_BatchContainerOverrides.html"
 * >BatchContainerOverrides</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TaskContainerOverrides implements SdkPojo, Serializable,
        ToCopyableBuilder<TaskContainerOverrides.Builder, TaskContainerOverrides> {
    private static final SdkField<List<String>> COMMAND_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("command")
            .getter(getter(TaskContainerOverrides::command))
            .setter(setter(Builder::command))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("command").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<KeyValuePair>> ENVIRONMENT_FIELD = SdkField
            .<List<KeyValuePair>> builder(MarshallingType.LIST)
            .memberName("environment")
            .getter(getter(TaskContainerOverrides::environment))
            .setter(setter(Builder::environment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("environment").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<KeyValuePair> builder(MarshallingType.SDK_POJO)
                                            .constructor(KeyValuePair::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(TaskContainerOverrides::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<List<ResourceRequirement>> RESOURCE_REQUIREMENTS_FIELD = SdkField
            .<List<ResourceRequirement>> builder(MarshallingType.LIST)
            .memberName("resourceRequirements")
            .getter(getter(TaskContainerOverrides::resourceRequirements))
            .setter(setter(Builder::resourceRequirements))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceRequirements").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ResourceRequirement> builder(MarshallingType.SDK_POJO)
                                            .constructor(ResourceRequirement::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(COMMAND_FIELD,
            ENVIRONMENT_FIELD, NAME_FIELD, RESOURCE_REQUIREMENTS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<String> command;

    private final List<KeyValuePair> environment;

    private final String name;

    private final List<ResourceRequirement> resourceRequirements;

    private TaskContainerOverrides(BuilderImpl builder) {
        this.command = builder.command;
        this.environment = builder.environment;
        this.name = builder.name;
        this.resourceRequirements = builder.resourceRequirements;
    }

    /**
     * For responses, this returns true if the service returned a value for the Command property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasCommand() {
        return command != null && !(command instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The command to send to the container that overrides the default command from the Docker image or the job
     * definition.
     * </p>
     * <note>
     * <p>
     * This parameter can't contain an empty string.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCommand} method.
     * </p>
     * 
     * @return The command to send to the container that overrides the default command from the Docker image or the job
     *         definition.</p> <note>
     *         <p>
     *         This parameter can't contain an empty string.
     *         </p>
     */
    public final List<String> command() {
        return command;
    }

    /**
     * For responses, this returns true if the service returned a value for the Environment property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEnvironment() {
        return environment != null && !(environment instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The environment variables to send to the container. You can add new environment variables, which are added to the
     * container at launch, or you can override the existing environment variables from the Docker image or the job
     * definition.
     * </p>
     * <note>
     * <p>
     * Environment variables cannot start with <code>AWS_BATCH</code>. This naming convention is reserved for variables
     * that Batch sets.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEnvironment} method.
     * </p>
     * 
     * @return The environment variables to send to the container. You can add new environment variables, which are
     *         added to the container at launch, or you can override the existing environment variables from the Docker
     *         image or the job definition.</p> <note>
     *         <p>
     *         Environment variables cannot start with <code>AWS_BATCH</code>. This naming convention is reserved for
     *         variables that Batch sets.
     *         </p>
     */
    public final List<KeyValuePair> environment() {
        return environment;
    }

    /**
     * <p>
     * A pointer to the container that you want to override. The container's name provides a unique identifier for the
     * container being used.
     * </p>
     * 
     * @return A pointer to the container that you want to override. The container's name provides a unique identifier
     *         for the container being used.
     */
    public final String name() {
        return name;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourceRequirements property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourceRequirements() {
        return resourceRequirements != null && !(resourceRequirements instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The type and amount of resources to assign to a container. This overrides the settings in the job definition. The
     * supported resources include <code>GPU</code>, <code>MEMORY</code>, and <code>VCPU</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceRequirements} method.
     * </p>
     * 
     * @return The type and amount of resources to assign to a container. This overrides the settings in the job
     *         definition. The supported resources include <code>GPU</code>, <code>MEMORY</code>, and <code>VCPU</code>.
     */
    public final List<ResourceRequirement> resourceRequirements() {
        return resourceRequirements;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasCommand() ? command() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasEnvironment() ? environment() : null);
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceRequirements() ? resourceRequirements() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TaskContainerOverrides)) {
            return false;
        }
        TaskContainerOverrides other = (TaskContainerOverrides) obj;
        return hasCommand() == other.hasCommand() && Objects.equals(command(), other.command())
                && hasEnvironment() == other.hasEnvironment() && Objects.equals(environment(), other.environment())
                && Objects.equals(name(), other.name()) && hasResourceRequirements() == other.hasResourceRequirements()
                && Objects.equals(resourceRequirements(), other.resourceRequirements());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TaskContainerOverrides").add("Command", hasCommand() ? command() : null)
                .add("Environment", hasEnvironment() ? environment() : null).add("Name", name())
                .add("ResourceRequirements", hasResourceRequirements() ? resourceRequirements() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "command":
            return Optional.ofNullable(clazz.cast(command()));
        case "environment":
            return Optional.ofNullable(clazz.cast(environment()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "resourceRequirements":
            return Optional.ofNullable(clazz.cast(resourceRequirements()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("command", COMMAND_FIELD);
        map.put("environment", ENVIRONMENT_FIELD);
        map.put("name", NAME_FIELD);
        map.put("resourceRequirements", RESOURCE_REQUIREMENTS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<TaskContainerOverrides, T> g) {
        return obj -> g.apply((TaskContainerOverrides) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TaskContainerOverrides> {
        /**
         * <p>
         * The command to send to the container that overrides the default command from the Docker image or the job
         * definition.
         * </p>
         * <note>
         * <p>
         * This parameter can't contain an empty string.
         * </p>
         * </note>
         * 
         * @param command
         *        The command to send to the container that overrides the default command from the Docker image or the
         *        job definition.</p> <note>
         *        <p>
         *        This parameter can't contain an empty string.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(Collection<String> command);

        /**
         * <p>
         * The command to send to the container that overrides the default command from the Docker image or the job
         * definition.
         * </p>
         * <note>
         * <p>
         * This parameter can't contain an empty string.
         * </p>
         * </note>
         * 
         * @param command
         *        The command to send to the container that overrides the default command from the Docker image or the
         *        job definition.</p> <note>
         *        <p>
         *        This parameter can't contain an empty string.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(String... command);

        /**
         * <p>
         * The environment variables to send to the container. You can add new environment variables, which are added to
         * the container at launch, or you can override the existing environment variables from the Docker image or the
         * job definition.
         * </p>
         * <note>
         * <p>
         * Environment variables cannot start with <code>AWS_BATCH</code>. This naming convention is reserved for
         * variables that Batch sets.
         * </p>
         * </note>
         * 
         * @param environment
         *        The environment variables to send to the container. You can add new environment variables, which are
         *        added to the container at launch, or you can override the existing environment variables from the
         *        Docker image or the job definition.</p> <note>
         *        <p>
         *        Environment variables cannot start with <code>AWS_BATCH</code>. This naming convention is reserved for
         *        variables that Batch sets.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environment(Collection<KeyValuePair> environment);

        /**
         * <p>
         * The environment variables to send to the container. You can add new environment variables, which are added to
         * the container at launch, or you can override the existing environment variables from the Docker image or the
         * job definition.
         * </p>
         * <note>
         * <p>
         * Environment variables cannot start with <code>AWS_BATCH</code>. This naming convention is reserved for
         * variables that Batch sets.
         * </p>
         * </note>
         * 
         * @param environment
         *        The environment variables to send to the container. You can add new environment variables, which are
         *        added to the container at launch, or you can override the existing environment variables from the
         *        Docker image or the job definition.</p> <note>
         *        <p>
         *        Environment variables cannot start with <code>AWS_BATCH</code>. This naming convention is reserved for
         *        variables that Batch sets.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environment(KeyValuePair... environment);

        /**
         * <p>
         * The environment variables to send to the container. You can add new environment variables, which are added to
         * the container at launch, or you can override the existing environment variables from the Docker image or the
         * job definition.
         * </p>
         * <note>
         * <p>
         * Environment variables cannot start with <code>AWS_BATCH</code>. This naming convention is reserved for
         * variables that Batch sets.
         * </p>
         * </note> This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.batch.model.KeyValuePair.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.batch.model.KeyValuePair#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.batch.model.KeyValuePair.Builder#build()} is called immediately and
         * its result is passed to {@link #environment(List<KeyValuePair>)}.
         * 
         * @param environment
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.batch.model.KeyValuePair.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #environment(java.util.Collection<KeyValuePair>)
         */
        Builder environment(Consumer<KeyValuePair.Builder>... environment);

        /**
         * <p>
         * A pointer to the container that you want to override. The container's name provides a unique identifier for
         * the container being used.
         * </p>
         * 
         * @param name
         *        A pointer to the container that you want to override. The container's name provides a unique
         *        identifier for the container being used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The type and amount of resources to assign to a container. This overrides the settings in the job definition.
         * The supported resources include <code>GPU</code>, <code>MEMORY</code>, and <code>VCPU</code>.
         * </p>
         * 
         * @param resourceRequirements
         *        The type and amount of resources to assign to a container. This overrides the settings in the job
         *        definition. The supported resources include <code>GPU</code>, <code>MEMORY</code>, and
         *        <code>VCPU</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceRequirements(Collection<ResourceRequirement> resourceRequirements);

        /**
         * <p>
         * The type and amount of resources to assign to a container. This overrides the settings in the job definition.
         * The supported resources include <code>GPU</code>, <code>MEMORY</code>, and <code>VCPU</code>.
         * </p>
         * 
         * @param resourceRequirements
         *        The type and amount of resources to assign to a container. This overrides the settings in the job
         *        definition. The supported resources include <code>GPU</code>, <code>MEMORY</code>, and
         *        <code>VCPU</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceRequirements(ResourceRequirement... resourceRequirements);

        /**
         * <p>
         * The type and amount of resources to assign to a container. This overrides the settings in the job definition.
         * The supported resources include <code>GPU</code>, <code>MEMORY</code>, and <code>VCPU</code>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.batch.model.ResourceRequirement.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.batch.model.ResourceRequirement#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.batch.model.ResourceRequirement.Builder#build()} is called immediately
         * and its result is passed to {@link #resourceRequirements(List<ResourceRequirement>)}.
         * 
         * @param resourceRequirements
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.batch.model.ResourceRequirement.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resourceRequirements(java.util.Collection<ResourceRequirement>)
         */
        Builder resourceRequirements(Consumer<ResourceRequirement.Builder>... resourceRequirements);
    }

    static final class BuilderImpl implements Builder {
        private List<String> command = DefaultSdkAutoConstructList.getInstance();

        private List<KeyValuePair> environment = DefaultSdkAutoConstructList.getInstance();

        private String name;

        private List<ResourceRequirement> resourceRequirements = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(TaskContainerOverrides model) {
            command(model.command);
            environment(model.environment);
            name(model.name);
            resourceRequirements(model.resourceRequirements);
        }

        public final Collection<String> getCommand() {
            if (command instanceof SdkAutoConstructList) {
                return null;
            }
            return command;
        }

        public final void setCommand(Collection<String> command) {
            this.command = StringListCopier.copy(command);
        }

        @Override
        public final Builder command(Collection<String> command) {
            this.command = StringListCopier.copy(command);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder command(String... command) {
            command(Arrays.asList(command));
            return this;
        }

        public final List<KeyValuePair.Builder> getEnvironment() {
            List<KeyValuePair.Builder> result = EnvironmentVariablesCopier.copyToBuilder(this.environment);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setEnvironment(Collection<KeyValuePair.BuilderImpl> environment) {
            this.environment = EnvironmentVariablesCopier.copyFromBuilder(environment);
        }

        @Override
        public final Builder environment(Collection<KeyValuePair> environment) {
            this.environment = EnvironmentVariablesCopier.copy(environment);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder environment(KeyValuePair... environment) {
            environment(Arrays.asList(environment));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder environment(Consumer<KeyValuePair.Builder>... environment) {
            environment(Stream.of(environment).map(c -> KeyValuePair.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final List<ResourceRequirement.Builder> getResourceRequirements() {
            List<ResourceRequirement.Builder> result = ResourceRequirementsCopier.copyToBuilder(this.resourceRequirements);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setResourceRequirements(Collection<ResourceRequirement.BuilderImpl> resourceRequirements) {
            this.resourceRequirements = ResourceRequirementsCopier.copyFromBuilder(resourceRequirements);
        }

        @Override
        public final Builder resourceRequirements(Collection<ResourceRequirement> resourceRequirements) {
            this.resourceRequirements = ResourceRequirementsCopier.copy(resourceRequirements);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceRequirements(ResourceRequirement... resourceRequirements) {
            resourceRequirements(Arrays.asList(resourceRequirements));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceRequirements(Consumer<ResourceRequirement.Builder>... resourceRequirements) {
            resourceRequirements(Stream.of(resourceRequirements).map(c -> ResourceRequirement.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public TaskContainerOverrides build() {
            return new TaskContainerOverrides(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
