/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.batch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Container properties are used for Amazon ECS-based job definitions. These properties to describe the container that's
 * launched as part of a job.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TaskContainerProperties implements SdkPojo, Serializable,
        ToCopyableBuilder<TaskContainerProperties.Builder, TaskContainerProperties> {
    private static final SdkField<List<String>> COMMAND_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("command")
            .getter(getter(TaskContainerProperties::command))
            .setter(setter(Builder::command))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("command").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<TaskContainerDependency>> DEPENDS_ON_FIELD = SdkField
            .<List<TaskContainerDependency>> builder(MarshallingType.LIST)
            .memberName("dependsOn")
            .getter(getter(TaskContainerProperties::dependsOn))
            .setter(setter(Builder::dependsOn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dependsOn").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<TaskContainerDependency> builder(MarshallingType.SDK_POJO)
                                            .constructor(TaskContainerDependency::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<KeyValuePair>> ENVIRONMENT_FIELD = SdkField
            .<List<KeyValuePair>> builder(MarshallingType.LIST)
            .memberName("environment")
            .getter(getter(TaskContainerProperties::environment))
            .setter(setter(Builder::environment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("environment").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<KeyValuePair> builder(MarshallingType.SDK_POJO)
                                            .constructor(KeyValuePair::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> ESSENTIAL_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("essential").getter(getter(TaskContainerProperties::essential)).setter(setter(Builder::essential))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("essential").build()).build();

    private static final SdkField<String> IMAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("image")
            .getter(getter(TaskContainerProperties::image)).setter(setter(Builder::image))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("image").build()).build();

    private static final SdkField<LinuxParameters> LINUX_PARAMETERS_FIELD = SdkField
            .<LinuxParameters> builder(MarshallingType.SDK_POJO).memberName("linuxParameters")
            .getter(getter(TaskContainerProperties::linuxParameters)).setter(setter(Builder::linuxParameters))
            .constructor(LinuxParameters::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("linuxParameters").build()).build();

    private static final SdkField<LogConfiguration> LOG_CONFIGURATION_FIELD = SdkField
            .<LogConfiguration> builder(MarshallingType.SDK_POJO).memberName("logConfiguration")
            .getter(getter(TaskContainerProperties::logConfiguration)).setter(setter(Builder::logConfiguration))
            .constructor(LogConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("logConfiguration").build()).build();

    private static final SdkField<List<MountPoint>> MOUNT_POINTS_FIELD = SdkField
            .<List<MountPoint>> builder(MarshallingType.LIST)
            .memberName("mountPoints")
            .getter(getter(TaskContainerProperties::mountPoints))
            .setter(setter(Builder::mountPoints))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("mountPoints").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<MountPoint> builder(MarshallingType.SDK_POJO)
                                            .constructor(MountPoint::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(TaskContainerProperties::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<Boolean> PRIVILEGED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("privileged").getter(getter(TaskContainerProperties::privileged)).setter(setter(Builder::privileged))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("privileged").build()).build();

    private static final SdkField<Boolean> READONLY_ROOT_FILESYSTEM_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("readonlyRootFilesystem").getter(getter(TaskContainerProperties::readonlyRootFilesystem))
            .setter(setter(Builder::readonlyRootFilesystem))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("readonlyRootFilesystem").build())
            .build();

    private static final SdkField<RepositoryCredentials> REPOSITORY_CREDENTIALS_FIELD = SdkField
            .<RepositoryCredentials> builder(MarshallingType.SDK_POJO).memberName("repositoryCredentials")
            .getter(getter(TaskContainerProperties::repositoryCredentials)).setter(setter(Builder::repositoryCredentials))
            .constructor(RepositoryCredentials::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("repositoryCredentials").build())
            .build();

    private static final SdkField<List<ResourceRequirement>> RESOURCE_REQUIREMENTS_FIELD = SdkField
            .<List<ResourceRequirement>> builder(MarshallingType.LIST)
            .memberName("resourceRequirements")
            .getter(getter(TaskContainerProperties::resourceRequirements))
            .setter(setter(Builder::resourceRequirements))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceRequirements").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ResourceRequirement> builder(MarshallingType.SDK_POJO)
                                            .constructor(ResourceRequirement::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Secret>> SECRETS_FIELD = SdkField
            .<List<Secret>> builder(MarshallingType.LIST)
            .memberName("secrets")
            .getter(getter(TaskContainerProperties::secrets))
            .setter(setter(Builder::secrets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("secrets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Secret> builder(MarshallingType.SDK_POJO)
                                            .constructor(Secret::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Ulimit>> ULIMITS_FIELD = SdkField
            .<List<Ulimit>> builder(MarshallingType.LIST)
            .memberName("ulimits")
            .getter(getter(TaskContainerProperties::ulimits))
            .setter(setter(Builder::ulimits))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ulimits").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Ulimit> builder(MarshallingType.SDK_POJO)
                                            .constructor(Ulimit::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> USER_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("user")
            .getter(getter(TaskContainerProperties::user)).setter(setter(Builder::user))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("user").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(COMMAND_FIELD,
            DEPENDS_ON_FIELD, ENVIRONMENT_FIELD, ESSENTIAL_FIELD, IMAGE_FIELD, LINUX_PARAMETERS_FIELD, LOG_CONFIGURATION_FIELD,
            MOUNT_POINTS_FIELD, NAME_FIELD, PRIVILEGED_FIELD, READONLY_ROOT_FILESYSTEM_FIELD, REPOSITORY_CREDENTIALS_FIELD,
            RESOURCE_REQUIREMENTS_FIELD, SECRETS_FIELD, ULIMITS_FIELD, USER_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<String> command;

    private final List<TaskContainerDependency> dependsOn;

    private final List<KeyValuePair> environment;

    private final Boolean essential;

    private final String image;

    private final LinuxParameters linuxParameters;

    private final LogConfiguration logConfiguration;

    private final List<MountPoint> mountPoints;

    private final String name;

    private final Boolean privileged;

    private final Boolean readonlyRootFilesystem;

    private final RepositoryCredentials repositoryCredentials;

    private final List<ResourceRequirement> resourceRequirements;

    private final List<Secret> secrets;

    private final List<Ulimit> ulimits;

    private final String user;

    private TaskContainerProperties(BuilderImpl builder) {
        this.command = builder.command;
        this.dependsOn = builder.dependsOn;
        this.environment = builder.environment;
        this.essential = builder.essential;
        this.image = builder.image;
        this.linuxParameters = builder.linuxParameters;
        this.logConfiguration = builder.logConfiguration;
        this.mountPoints = builder.mountPoints;
        this.name = builder.name;
        this.privileged = builder.privileged;
        this.readonlyRootFilesystem = builder.readonlyRootFilesystem;
        this.repositoryCredentials = builder.repositoryCredentials;
        this.resourceRequirements = builder.resourceRequirements;
        this.secrets = builder.secrets;
        this.ulimits = builder.ulimits;
        this.user = builder.user;
    }

    /**
     * For responses, this returns true if the service returned a value for the Command property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasCommand() {
        return command != null && !(command instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The command that's passed to the container. This parameter maps to <code>Cmd</code> in the <a
     * href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the <a
     * href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>COMMAND</code> parameter to
     * <a href="https://docs.docker.com/engine/reference/run/">docker run</a>. For more information, see <a
     * href="https://docs.docker.com/engine/reference/builder/#cmd">Dockerfile reference: CMD</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCommand} method.
     * </p>
     * 
     * @return The command that's passed to the container. This parameter maps to <code>Cmd</code> in the <a
     *         href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the
     *         <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>COMMAND</code>
     *         parameter to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>. For more
     *         information, see <a href="https://docs.docker.com/engine/reference/builder/#cmd">Dockerfile reference:
     *         CMD</a>.
     */
    public final List<String> command() {
        return command;
    }

    /**
     * For responses, this returns true if the service returned a value for the DependsOn property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDependsOn() {
        return dependsOn != null && !(dependsOn instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of containers that this container depends on.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDependsOn} method.
     * </p>
     * 
     * @return A list of containers that this container depends on.
     */
    public final List<TaskContainerDependency> dependsOn() {
        return dependsOn;
    }

    /**
     * For responses, this returns true if the service returned a value for the Environment property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEnvironment() {
        return environment != null && !(environment instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The environment variables to pass to a container. This parameter maps to Env inthe <a
     * href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the <a
     * href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--env</code> parameter to <a
     * href="https://docs.docker.com/engine/reference/run/">docker run</a>.
     * </p>
     * <important>
     * <p>
     * We don't recommend using plaintext environment variables for sensitive information, such as credential data.
     * </p>
     * </important> <note>
     * <p>
     * Environment variables cannot start with <code>AWS_BATCH</code>. This naming convention is reserved for variables
     * that Batch sets.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEnvironment} method.
     * </p>
     * 
     * @return The environment variables to pass to a container. This parameter maps to Env inthe <a
     *         href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the
     *         <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--env</code>
     *         parameter to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>. </p> <important>
     *         <p>
     *         We don't recommend using plaintext environment variables for sensitive information, such as credential
     *         data.
     *         </p>
     *         </important> <note>
     *         <p>
     *         Environment variables cannot start with <code>AWS_BATCH</code>. This naming convention is reserved for
     *         variables that Batch sets.
     *         </p>
     */
    public final List<KeyValuePair> environment() {
        return environment;
    }

    /**
     * <p>
     * If the essential parameter of a container is marked as <code>true</code>, and that container fails or stops for
     * any reason, all other containers that are part of the task are stopped. If the <code>essential</code> parameter
     * of a container is marked as false, its failure doesn't affect the rest of the containers in a task. If this
     * parameter is omitted, a container is assumed to be essential.
     * </p>
     * <p>
     * All jobs must have at least one essential container. If you have an application that's composed of multiple
     * containers, group containers that are used for a common purpose into components, and separate the different
     * components into multiple task definitions. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/application_architecture.html">Application
     * Architecture</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * 
     * @return If the essential parameter of a container is marked as <code>true</code>, and that container fails or
     *         stops for any reason, all other containers that are part of the task are stopped. If the
     *         <code>essential</code> parameter of a container is marked as false, its failure doesn't affect the rest
     *         of the containers in a task. If this parameter is omitted, a container is assumed to be essential.</p>
     *         <p>
     *         All jobs must have at least one essential container. If you have an application that's composed of
     *         multiple containers, group containers that are used for a common purpose into components, and separate
     *         the different components into multiple task definitions. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/application_architecture.html"
     *         >Application Architecture</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     */
    public final Boolean essential() {
        return essential;
    }

    /**
     * <p>
     * The image used to start a container. This string is passed directly to the Docker daemon. By default, images in
     * the Docker Hub registry are available. Other repositories are specified with either
     * <code>repository-url/image:tag</code> or <code>repository-url/image@digest</code>. Up to 255 letters (uppercase
     * and lowercase), numbers, hyphens, underscores, colons, periods, forward slashes, and number signs are allowed.
     * This parameter maps to <code>Image</code> in the <a
     * href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a> section of the
     * <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a> and the <code>IMAGE</code> parameter of
     * the <a href="https://docs.docker.com/engine/reference/run/#security-configuration"> <i>docker run</i> </a>.
     * </p>
     * 
     * @return The image used to start a container. This string is passed directly to the Docker daemon. By default,
     *         images in the Docker Hub registry are available. Other repositories are specified with either
     *         <code>repository-url/image:tag</code> or <code>repository-url/image@digest</code>. Up to 255 letters
     *         (uppercase and lowercase), numbers, hyphens, underscores, colons, periods, forward slashes, and number
     *         signs are allowed. This parameter maps to <code>Image</code> in the <a
     *         href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a> section
     *         of the <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a> and the
     *         <code>IMAGE</code> parameter of the <a
     *         href="https://docs.docker.com/engine/reference/run/#security-configuration"> <i>docker run</i> </a>.
     */
    public final String image() {
        return image;
    }

    /**
     * <p>
     * Linux-specific modifications that are applied to the container, such as Linux kernel capabilities. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_KernelCapabilities.html">KernelCapabilities
     * </a>.
     * </p>
     * 
     * @return Linux-specific modifications that are applied to the container, such as Linux kernel capabilities. For
     *         more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_KernelCapabilities.html"
     *         >KernelCapabilities</a>.
     */
    public final LinuxParameters linuxParameters() {
        return linuxParameters;
    }

    /**
     * <p>
     * The log configuration specification for the container.
     * </p>
     * <p>
     * This parameter maps to <code>LogConfig</code> in the <a
     * href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a> section of the
     * <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a> and the <code>--log-driver</code>
     * option to <a href="https://docs.docker.com/engine/reference/run/#security-configuration">docker run</a>.
     * </p>
     * <p>
     * By default, containers use the same logging driver that the Docker daemon uses. However the container can use a
     * different logging driver than the Docker daemon by specifying a log driver with this parameter in the container
     * definition. To use a different logging driver for a container, the log system must be configured properly on the
     * container instance (or on a different log server for remote logging options). For more information about the
     * options for different supported log drivers, see <a
     * href="https://docs.docker.com/engine/admin/logging/overview/">Configure logging drivers </a> in the <i>Docker
     * documentation</i>.
     * </p>
     * <note>
     * <p>
     * Amazon ECS currently supports a subset of the logging drivers available to the Docker daemon (shown in the
     * <code>LogConfiguration</code> data type). Additional log drivers may be available in future releases of the
     * Amazon ECS container agent.
     * </p>
     * </note>
     * <p>
     * This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check the
     * Docker Remote API version on your container instance, log in to your container instance and run the following
     * command: sudo docker version <code>--format '{{.Server.APIVersion}}'</code>
     * </p>
     * <note>
     * <p>
     * The Amazon ECS container agent running on a container instance must register the logging drivers available on
     * that instance with the <code>ECS_AVAILABLE_LOGGING_DRIVERS</code> environment variable before containers placed
     * on that instance can use these log configuration options. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html">Amazon ECS container
     * agent configuration</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     * </p>
     * </note>
     * 
     * @return The log configuration specification for the container.</p>
     *         <p>
     *         This parameter maps to <code>LogConfig</code> in the <a
     *         href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a> section
     *         of the <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a> and the
     *         <code>--log-driver</code> option to <a
     *         href="https://docs.docker.com/engine/reference/run/#security-configuration">docker run</a>.
     *         </p>
     *         <p>
     *         By default, containers use the same logging driver that the Docker daemon uses. However the container can
     *         use a different logging driver than the Docker daemon by specifying a log driver with this parameter in
     *         the container definition. To use a different logging driver for a container, the log system must be
     *         configured properly on the container instance (or on a different log server for remote logging options).
     *         For more information about the options for different supported log drivers, see <a
     *         href="https://docs.docker.com/engine/admin/logging/overview/">Configure logging drivers </a> in the
     *         <i>Docker documentation</i>.
     *         </p>
     *         <note>
     *         <p>
     *         Amazon ECS currently supports a subset of the logging drivers available to the Docker daemon (shown in
     *         the <code>LogConfiguration</code> data type). Additional log drivers may be available in future releases
     *         of the Amazon ECS container agent.
     *         </p>
     *         </note>
     *         <p>
     *         This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To
     *         check the Docker Remote API version on your container instance, log in to your container instance and run
     *         the following command: sudo docker version <code>--format '{{.Server.APIVersion}}'</code>
     *         </p>
     *         <note>
     *         <p>
     *         The Amazon ECS container agent running on a container instance must register the logging drivers
     *         available on that instance with the <code>ECS_AVAILABLE_LOGGING_DRIVERS</code> environment variable
     *         before containers placed on that instance can use these log configuration options. For more information,
     *         see <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html">Amazon
     *         ECS container agent configuration</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
     *         </p>
     */
    public final LogConfiguration logConfiguration() {
        return logConfiguration;
    }

    /**
     * For responses, this returns true if the service returned a value for the MountPoints property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasMountPoints() {
        return mountPoints != null && !(mountPoints instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The mount points for data volumes in your container.
     * </p>
     * <p>
     * This parameter maps to <code>Volumes</code> in the <a
     * href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a> section of the
     * <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a> and the <a href="">--volume</a> option
     * to <a href="https://docs.docker.com/engine/reference/run/#security-configuration">docker run</a>.
     * </p>
     * <p>
     * Windows containers can mount whole directories on the same drive as <code>$env:ProgramData</code>. Windows
     * containers can't mount directories on a different drive, and mount point can't be across drives.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMountPoints} method.
     * </p>
     * 
     * @return The mount points for data volumes in your container.</p>
     *         <p>
     *         This parameter maps to <code>Volumes</code> in the <a
     *         href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a> section
     *         of the <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a> and the <a
     *         href="">--volume</a> option to <a
     *         href="https://docs.docker.com/engine/reference/run/#security-configuration">docker run</a>.
     *         </p>
     *         <p>
     *         Windows containers can mount whole directories on the same drive as <code>$env:ProgramData</code>.
     *         Windows containers can't mount directories on a different drive, and mount point can't be across drives.
     */
    public final List<MountPoint> mountPoints() {
        return mountPoints;
    }

    /**
     * <p>
     * The name of a container. The name can be used as a unique identifier to target your <code>dependsOn</code> and
     * <code>Overrides</code> objects.
     * </p>
     * 
     * @return The name of a container. The name can be used as a unique identifier to target your
     *         <code>dependsOn</code> and <code>Overrides</code> objects.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * When this parameter is <code>true</code>, the container is given elevated privileges on the host container
     * instance (similar to the <code>root</code> user). This parameter maps to <code>Privileged</code> in the <a
     * href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a> section of the
     * <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a> and the <code>--privileged</code>
     * option to <a href="https://docs.docker.com/engine/reference/run/#security-configuration">docker run</a>.
     * </p>
     * <note>
     * <p>
     * This parameter is not supported for Windows containers or tasks run on Fargate.
     * </p>
     * </note>
     * 
     * @return When this parameter is <code>true</code>, the container is given elevated privileges on the host
     *         container instance (similar to the <code>root</code> user). This parameter maps to
     *         <code>Privileged</code> in the <a
     *         href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a> section
     *         of the <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a> and the
     *         <code>--privileged</code> option to <a
     *         href="https://docs.docker.com/engine/reference/run/#security-configuration">docker run</a>.</p> <note>
     *         <p>
     *         This parameter is not supported for Windows containers or tasks run on Fargate.
     *         </p>
     */
    public final Boolean privileged() {
        return privileged;
    }

    /**
     * <p>
     * When this parameter is true, the container is given read-only access to its root file system. This parameter maps
     * to <code>ReadonlyRootfs</code> in the <a
     * href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a> section of the
     * <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a> and the <code>--read-only</code> option
     * to <a href="https://docs.docker.com/engine/reference/run/#security-configuration">docker run</a>.
     * </p>
     * <note>
     * <p>
     * This parameter is not supported for Windows containers.
     * </p>
     * </note>
     * 
     * @return When this parameter is true, the container is given read-only access to its root file system. This
     *         parameter maps to <code>ReadonlyRootfs</code> in the <a
     *         href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a> section
     *         of the <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a> and the
     *         <code>--read-only</code> option to <a
     *         href="https://docs.docker.com/engine/reference/run/#security-configuration">docker run</a>.</p> <note>
     *         <p>
     *         This parameter is not supported for Windows containers.
     *         </p>
     */
    public final Boolean readonlyRootFilesystem() {
        return readonlyRootFilesystem;
    }

    /**
     * <p>
     * The private repository authentication credentials to use.
     * </p>
     * 
     * @return The private repository authentication credentials to use.
     */
    public final RepositoryCredentials repositoryCredentials() {
        return repositoryCredentials;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourceRequirements property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourceRequirements() {
        return resourceRequirements != null && !(resourceRequirements instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The type and amount of a resource to assign to a container. The only supported resource is a GPU.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceRequirements} method.
     * </p>
     * 
     * @return The type and amount of a resource to assign to a container. The only supported resource is a GPU.
     */
    public final List<ResourceRequirement> resourceRequirements() {
        return resourceRequirements;
    }

    /**
     * For responses, this returns true if the service returned a value for the Secrets property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasSecrets() {
        return secrets != null && !(secrets instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The secrets to pass to the container. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html">Specifying
     * Sensitive Data</a> in the Amazon Elastic Container Service Developer Guide.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSecrets} method.
     * </p>
     * 
     * @return The secrets to pass to the container. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html"
     *         >Specifying Sensitive Data</a> in the Amazon Elastic Container Service Developer Guide.
     */
    public final List<Secret> secrets() {
        return secrets;
    }

    /**
     * For responses, this returns true if the service returned a value for the Ulimits property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasUlimits() {
        return ulimits != null && !(ulimits instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of <code>ulimits</code> to set in the container. If a <code>ulimit</code> value is specified in a task
     * definition, it overrides the default values set by Docker. This parameter maps to <code>Ulimits</code> in the <a
     * href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a> section of the
     * <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a> and the <code>--ulimit</code> option to
     * <a href="https://docs.docker.com/engine/reference/run/#security-configuration">docker run</a>.
     * </p>
     * <p>
     * Amazon ECS tasks hosted on Fargate use the default resource limit values set by the operating system with the
     * exception of the nofile resource limit parameter which Fargate overrides. The <code>nofile</code> resource limit
     * sets a restriction on the number of open files that a container can use. The default <code>nofile</code> soft
     * limit is <code>1024</code> and the default hard limit is <code>65535</code>.
     * </p>
     * <p>
     * This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check the
     * Docker Remote API version on your container instance, log in to your container instance and run the following
     * command: sudo docker version <code>--format '{{.Server.APIVersion}}'</code>
     * </p>
     * <note>
     * <p>
     * This parameter is not supported for Windows containers.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasUlimits} method.
     * </p>
     * 
     * @return A list of <code>ulimits</code> to set in the container. If a <code>ulimit</code> value is specified in a
     *         task definition, it overrides the default values set by Docker. This parameter maps to
     *         <code>Ulimits</code> in the <a
     *         href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a> section
     *         of the <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a> and the
     *         <code>--ulimit</code> option to <a
     *         href="https://docs.docker.com/engine/reference/run/#security-configuration">docker run</a>.</p>
     *         <p>
     *         Amazon ECS tasks hosted on Fargate use the default resource limit values set by the operating system with
     *         the exception of the nofile resource limit parameter which Fargate overrides. The <code>nofile</code>
     *         resource limit sets a restriction on the number of open files that a container can use. The default
     *         <code>nofile</code> soft limit is <code>1024</code> and the default hard limit is <code>65535</code>.
     *         </p>
     *         <p>
     *         This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To
     *         check the Docker Remote API version on your container instance, log in to your container instance and run
     *         the following command: sudo docker version <code>--format '{{.Server.APIVersion}}'</code>
     *         </p>
     *         <note>
     *         <p>
     *         This parameter is not supported for Windows containers.
     *         </p>
     */
    public final List<Ulimit> ulimits() {
        return ulimits;
    }

    /**
     * <p>
     * The user to use inside the container. This parameter maps to User in the Create a container section of the Docker
     * Remote API and the --user option to docker run.
     * </p>
     * <note>
     * <p>
     * When running tasks using the <code>host</code> network mode, don't run containers using the
     * <code>root user (UID 0)</code>. We recommend using a non-root user for better security.
     * </p>
     * </note>
     * <p>
     * You can specify the <code>user</code> using the following formats. If specifying a UID or GID, you must specify
     * it as a positive integer.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>user</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>user:group</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>uid</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>uid:gid</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>user:gi</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>uid:group</code>
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * This parameter is not supported for Windows containers.
     * </p>
     * </note>
     * 
     * @return The user to use inside the container. This parameter maps to User in the Create a container section of
     *         the Docker Remote API and the --user option to docker run.</p> <note>
     *         <p>
     *         When running tasks using the <code>host</code> network mode, don't run containers using the
     *         <code>root user (UID 0)</code>. We recommend using a non-root user for better security.
     *         </p>
     *         </note>
     *         <p>
     *         You can specify the <code>user</code> using the following formats. If specifying a UID or GID, you must
     *         specify it as a positive integer.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>user</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>user:group</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>uid</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>uid:gid</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>user:gi</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>uid:group</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         This parameter is not supported for Windows containers.
     *         </p>
     */
    public final String user() {
        return user;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasCommand() ? command() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasDependsOn() ? dependsOn() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasEnvironment() ? environment() : null);
        hashCode = 31 * hashCode + Objects.hashCode(essential());
        hashCode = 31 * hashCode + Objects.hashCode(image());
        hashCode = 31 * hashCode + Objects.hashCode(linuxParameters());
        hashCode = 31 * hashCode + Objects.hashCode(logConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(hasMountPoints() ? mountPoints() : null);
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(privileged());
        hashCode = 31 * hashCode + Objects.hashCode(readonlyRootFilesystem());
        hashCode = 31 * hashCode + Objects.hashCode(repositoryCredentials());
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceRequirements() ? resourceRequirements() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSecrets() ? secrets() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasUlimits() ? ulimits() : null);
        hashCode = 31 * hashCode + Objects.hashCode(user());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TaskContainerProperties)) {
            return false;
        }
        TaskContainerProperties other = (TaskContainerProperties) obj;
        return hasCommand() == other.hasCommand() && Objects.equals(command(), other.command())
                && hasDependsOn() == other.hasDependsOn() && Objects.equals(dependsOn(), other.dependsOn())
                && hasEnvironment() == other.hasEnvironment() && Objects.equals(environment(), other.environment())
                && Objects.equals(essential(), other.essential()) && Objects.equals(image(), other.image())
                && Objects.equals(linuxParameters(), other.linuxParameters())
                && Objects.equals(logConfiguration(), other.logConfiguration()) && hasMountPoints() == other.hasMountPoints()
                && Objects.equals(mountPoints(), other.mountPoints()) && Objects.equals(name(), other.name())
                && Objects.equals(privileged(), other.privileged())
                && Objects.equals(readonlyRootFilesystem(), other.readonlyRootFilesystem())
                && Objects.equals(repositoryCredentials(), other.repositoryCredentials())
                && hasResourceRequirements() == other.hasResourceRequirements()
                && Objects.equals(resourceRequirements(), other.resourceRequirements()) && hasSecrets() == other.hasSecrets()
                && Objects.equals(secrets(), other.secrets()) && hasUlimits() == other.hasUlimits()
                && Objects.equals(ulimits(), other.ulimits()) && Objects.equals(user(), other.user());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TaskContainerProperties").add("Command", hasCommand() ? command() : null)
                .add("DependsOn", hasDependsOn() ? dependsOn() : null)
                .add("Environment", hasEnvironment() ? environment() : null).add("Essential", essential()).add("Image", image())
                .add("LinuxParameters", linuxParameters()).add("LogConfiguration", logConfiguration())
                .add("MountPoints", hasMountPoints() ? mountPoints() : null).add("Name", name()).add("Privileged", privileged())
                .add("ReadonlyRootFilesystem", readonlyRootFilesystem()).add("RepositoryCredentials", repositoryCredentials())
                .add("ResourceRequirements", hasResourceRequirements() ? resourceRequirements() : null)
                .add("Secrets", hasSecrets() ? secrets() : null).add("Ulimits", hasUlimits() ? ulimits() : null)
                .add("User", user()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "command":
            return Optional.ofNullable(clazz.cast(command()));
        case "dependsOn":
            return Optional.ofNullable(clazz.cast(dependsOn()));
        case "environment":
            return Optional.ofNullable(clazz.cast(environment()));
        case "essential":
            return Optional.ofNullable(clazz.cast(essential()));
        case "image":
            return Optional.ofNullable(clazz.cast(image()));
        case "linuxParameters":
            return Optional.ofNullable(clazz.cast(linuxParameters()));
        case "logConfiguration":
            return Optional.ofNullable(clazz.cast(logConfiguration()));
        case "mountPoints":
            return Optional.ofNullable(clazz.cast(mountPoints()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "privileged":
            return Optional.ofNullable(clazz.cast(privileged()));
        case "readonlyRootFilesystem":
            return Optional.ofNullable(clazz.cast(readonlyRootFilesystem()));
        case "repositoryCredentials":
            return Optional.ofNullable(clazz.cast(repositoryCredentials()));
        case "resourceRequirements":
            return Optional.ofNullable(clazz.cast(resourceRequirements()));
        case "secrets":
            return Optional.ofNullable(clazz.cast(secrets()));
        case "ulimits":
            return Optional.ofNullable(clazz.cast(ulimits()));
        case "user":
            return Optional.ofNullable(clazz.cast(user()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("command", COMMAND_FIELD);
        map.put("dependsOn", DEPENDS_ON_FIELD);
        map.put("environment", ENVIRONMENT_FIELD);
        map.put("essential", ESSENTIAL_FIELD);
        map.put("image", IMAGE_FIELD);
        map.put("linuxParameters", LINUX_PARAMETERS_FIELD);
        map.put("logConfiguration", LOG_CONFIGURATION_FIELD);
        map.put("mountPoints", MOUNT_POINTS_FIELD);
        map.put("name", NAME_FIELD);
        map.put("privileged", PRIVILEGED_FIELD);
        map.put("readonlyRootFilesystem", READONLY_ROOT_FILESYSTEM_FIELD);
        map.put("repositoryCredentials", REPOSITORY_CREDENTIALS_FIELD);
        map.put("resourceRequirements", RESOURCE_REQUIREMENTS_FIELD);
        map.put("secrets", SECRETS_FIELD);
        map.put("ulimits", ULIMITS_FIELD);
        map.put("user", USER_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<TaskContainerProperties, T> g) {
        return obj -> g.apply((TaskContainerProperties) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TaskContainerProperties> {
        /**
         * <p>
         * The command that's passed to the container. This parameter maps to <code>Cmd</code> in the <a
         * href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the <a
         * href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>COMMAND</code> parameter
         * to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>. For more information, see <a
         * href="https://docs.docker.com/engine/reference/builder/#cmd">Dockerfile reference: CMD</a>.
         * </p>
         * 
         * @param command
         *        The command that's passed to the container. This parameter maps to <code>Cmd</code> in the <a
         *        href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of
         *        the <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the
         *        <code>COMMAND</code> parameter to <a href="https://docs.docker.com/engine/reference/run/">docker
         *        run</a>. For more information, see <a
         *        href="https://docs.docker.com/engine/reference/builder/#cmd">Dockerfile reference: CMD</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(Collection<String> command);

        /**
         * <p>
         * The command that's passed to the container. This parameter maps to <code>Cmd</code> in the <a
         * href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the <a
         * href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>COMMAND</code> parameter
         * to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>. For more information, see <a
         * href="https://docs.docker.com/engine/reference/builder/#cmd">Dockerfile reference: CMD</a>.
         * </p>
         * 
         * @param command
         *        The command that's passed to the container. This parameter maps to <code>Cmd</code> in the <a
         *        href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of
         *        the <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the
         *        <code>COMMAND</code> parameter to <a href="https://docs.docker.com/engine/reference/run/">docker
         *        run</a>. For more information, see <a
         *        href="https://docs.docker.com/engine/reference/builder/#cmd">Dockerfile reference: CMD</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(String... command);

        /**
         * <p>
         * A list of containers that this container depends on.
         * </p>
         * 
         * @param dependsOn
         *        A list of containers that this container depends on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dependsOn(Collection<TaskContainerDependency> dependsOn);

        /**
         * <p>
         * A list of containers that this container depends on.
         * </p>
         * 
         * @param dependsOn
         *        A list of containers that this container depends on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dependsOn(TaskContainerDependency... dependsOn);

        /**
         * <p>
         * A list of containers that this container depends on.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.batch.model.TaskContainerDependency.Builder} avoiding the need to
         * create one manually via {@link software.amazon.awssdk.services.batch.model.TaskContainerDependency#builder()}
         * .
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.batch.model.TaskContainerDependency.Builder#build()} is called
         * immediately and its result is passed to {@link #dependsOn(List<TaskContainerDependency>)}.
         * 
         * @param dependsOn
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.batch.model.TaskContainerDependency.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dependsOn(java.util.Collection<TaskContainerDependency>)
         */
        Builder dependsOn(Consumer<TaskContainerDependency.Builder>... dependsOn);

        /**
         * <p>
         * The environment variables to pass to a container. This parameter maps to Env inthe <a
         * href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the <a
         * href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--env</code> parameter
         * to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
         * </p>
         * <important>
         * <p>
         * We don't recommend using plaintext environment variables for sensitive information, such as credential data.
         * </p>
         * </important> <note>
         * <p>
         * Environment variables cannot start with <code>AWS_BATCH</code>. This naming convention is reserved for
         * variables that Batch sets.
         * </p>
         * </note>
         * 
         * @param environment
         *        The environment variables to pass to a container. This parameter maps to Env inthe <a
         *        href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of
         *        the <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the
         *        <code>--env</code> parameter to <a href="https://docs.docker.com/engine/reference/run/">docker
         *        run</a>. </p> <important>
         *        <p>
         *        We don't recommend using plaintext environment variables for sensitive information, such as credential
         *        data.
         *        </p>
         *        </important> <note>
         *        <p>
         *        Environment variables cannot start with <code>AWS_BATCH</code>. This naming convention is reserved for
         *        variables that Batch sets.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environment(Collection<KeyValuePair> environment);

        /**
         * <p>
         * The environment variables to pass to a container. This parameter maps to Env inthe <a
         * href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the <a
         * href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--env</code> parameter
         * to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
         * </p>
         * <important>
         * <p>
         * We don't recommend using plaintext environment variables for sensitive information, such as credential data.
         * </p>
         * </important> <note>
         * <p>
         * Environment variables cannot start with <code>AWS_BATCH</code>. This naming convention is reserved for
         * variables that Batch sets.
         * </p>
         * </note>
         * 
         * @param environment
         *        The environment variables to pass to a container. This parameter maps to Env inthe <a
         *        href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of
         *        the <a href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the
         *        <code>--env</code> parameter to <a href="https://docs.docker.com/engine/reference/run/">docker
         *        run</a>. </p> <important>
         *        <p>
         *        We don't recommend using plaintext environment variables for sensitive information, such as credential
         *        data.
         *        </p>
         *        </important> <note>
         *        <p>
         *        Environment variables cannot start with <code>AWS_BATCH</code>. This naming convention is reserved for
         *        variables that Batch sets.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environment(KeyValuePair... environment);

        /**
         * <p>
         * The environment variables to pass to a container. This parameter maps to Env inthe <a
         * href="https://docs.docker.com/engine/api/v1.23/#create-a-container">Create a container</a> section of the <a
         * href="https://docs.docker.com/engine/api/v1.23/">Docker Remote API</a> and the <code>--env</code> parameter
         * to <a href="https://docs.docker.com/engine/reference/run/">docker run</a>.
         * </p>
         * <important>
         * <p>
         * We don't recommend using plaintext environment variables for sensitive information, such as credential data.
         * </p>
         * </important> <note>
         * <p>
         * Environment variables cannot start with <code>AWS_BATCH</code>. This naming convention is reserved for
         * variables that Batch sets.
         * </p>
         * </note> This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.batch.model.KeyValuePair.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.batch.model.KeyValuePair#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.batch.model.KeyValuePair.Builder#build()} is called immediately and
         * its result is passed to {@link #environment(List<KeyValuePair>)}.
         * 
         * @param environment
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.batch.model.KeyValuePair.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #environment(java.util.Collection<KeyValuePair>)
         */
        Builder environment(Consumer<KeyValuePair.Builder>... environment);

        /**
         * <p>
         * If the essential parameter of a container is marked as <code>true</code>, and that container fails or stops
         * for any reason, all other containers that are part of the task are stopped. If the <code>essential</code>
         * parameter of a container is marked as false, its failure doesn't affect the rest of the containers in a task.
         * If this parameter is omitted, a container is assumed to be essential.
         * </p>
         * <p>
         * All jobs must have at least one essential container. If you have an application that's composed of multiple
         * containers, group containers that are used for a common purpose into components, and separate the different
         * components into multiple task definitions. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/application_architecture.html">Application
         * Architecture</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * 
         * @param essential
         *        If the essential parameter of a container is marked as <code>true</code>, and that container fails or
         *        stops for any reason, all other containers that are part of the task are stopped. If the
         *        <code>essential</code> parameter of a container is marked as false, its failure doesn't affect the
         *        rest of the containers in a task. If this parameter is omitted, a container is assumed to be
         *        essential.</p>
         *        <p>
         *        All jobs must have at least one essential container. If you have an application that's composed of
         *        multiple containers, group containers that are used for a common purpose into components, and separate
         *        the different components into multiple task definitions. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/application_architecture.html"
         *        >Application Architecture</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder essential(Boolean essential);

        /**
         * <p>
         * The image used to start a container. This string is passed directly to the Docker daemon. By default, images
         * in the Docker Hub registry are available. Other repositories are specified with either
         * <code>repository-url/image:tag</code> or <code>repository-url/image@digest</code>. Up to 255 letters
         * (uppercase and lowercase), numbers, hyphens, underscores, colons, periods, forward slashes, and number signs
         * are allowed. This parameter maps to <code>Image</code> in the <a
         * href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a> section of
         * the <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a> and the <code>IMAGE</code>
         * parameter of the <a href="https://docs.docker.com/engine/reference/run/#security-configuration"> <i>docker
         * run</i> </a>.
         * </p>
         * 
         * @param image
         *        The image used to start a container. This string is passed directly to the Docker daemon. By default,
         *        images in the Docker Hub registry are available. Other repositories are specified with either
         *        <code>repository-url/image:tag</code> or <code>repository-url/image@digest</code>. Up to 255 letters
         *        (uppercase and lowercase), numbers, hyphens, underscores, colons, periods, forward slashes, and number
         *        signs are allowed. This parameter maps to <code>Image</code> in the <a
         *        href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a>
         *        section of the <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a> and the
         *        <code>IMAGE</code> parameter of the <a
         *        href="https://docs.docker.com/engine/reference/run/#security-configuration"> <i>docker run</i> </a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder image(String image);

        /**
         * <p>
         * Linux-specific modifications that are applied to the container, such as Linux kernel capabilities. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_KernelCapabilities.html"
         * >KernelCapabilities</a>.
         * </p>
         * 
         * @param linuxParameters
         *        Linux-specific modifications that are applied to the container, such as Linux kernel capabilities. For
         *        more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_KernelCapabilities.html"
         *        >KernelCapabilities</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder linuxParameters(LinuxParameters linuxParameters);

        /**
         * <p>
         * Linux-specific modifications that are applied to the container, such as Linux kernel capabilities. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_KernelCapabilities.html"
         * >KernelCapabilities</a>.
         * </p>
         * This is a convenience method that creates an instance of the {@link LinuxParameters.Builder} avoiding the
         * need to create one manually via {@link LinuxParameters#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link LinuxParameters.Builder#build()} is called immediately and its
         * result is passed to {@link #linuxParameters(LinuxParameters)}.
         * 
         * @param linuxParameters
         *        a consumer that will call methods on {@link LinuxParameters.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #linuxParameters(LinuxParameters)
         */
        default Builder linuxParameters(Consumer<LinuxParameters.Builder> linuxParameters) {
            return linuxParameters(LinuxParameters.builder().applyMutation(linuxParameters).build());
        }

        /**
         * <p>
         * The log configuration specification for the container.
         * </p>
         * <p>
         * This parameter maps to <code>LogConfig</code> in the <a
         * href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a> section of
         * the <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a> and the
         * <code>--log-driver</code> option to <a
         * href="https://docs.docker.com/engine/reference/run/#security-configuration">docker run</a>.
         * </p>
         * <p>
         * By default, containers use the same logging driver that the Docker daemon uses. However the container can use
         * a different logging driver than the Docker daemon by specifying a log driver with this parameter in the
         * container definition. To use a different logging driver for a container, the log system must be configured
         * properly on the container instance (or on a different log server for remote logging options). For more
         * information about the options for different supported log drivers, see <a
         * href="https://docs.docker.com/engine/admin/logging/overview/">Configure logging drivers </a> in the <i>Docker
         * documentation</i>.
         * </p>
         * <note>
         * <p>
         * Amazon ECS currently supports a subset of the logging drivers available to the Docker daemon (shown in the
         * <code>LogConfiguration</code> data type). Additional log drivers may be available in future releases of the
         * Amazon ECS container agent.
         * </p>
         * </note>
         * <p>
         * This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check
         * the Docker Remote API version on your container instance, log in to your container instance and run the
         * following command: sudo docker version <code>--format '{{.Server.APIVersion}}'</code>
         * </p>
         * <note>
         * <p>
         * The Amazon ECS container agent running on a container instance must register the logging drivers available on
         * that instance with the <code>ECS_AVAILABLE_LOGGING_DRIVERS</code> environment variable before containers
         * placed on that instance can use these log configuration options. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html">Amazon ECS container
         * agent configuration</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * </note>
         * 
         * @param logConfiguration
         *        The log configuration specification for the container.</p>
         *        <p>
         *        This parameter maps to <code>LogConfig</code> in the <a
         *        href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a>
         *        section of the <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a> and the
         *        <code>--log-driver</code> option to <a
         *        href="https://docs.docker.com/engine/reference/run/#security-configuration">docker run</a>.
         *        </p>
         *        <p>
         *        By default, containers use the same logging driver that the Docker daemon uses. However the container
         *        can use a different logging driver than the Docker daemon by specifying a log driver with this
         *        parameter in the container definition. To use a different logging driver for a container, the log
         *        system must be configured properly on the container instance (or on a different log server for remote
         *        logging options). For more information about the options for different supported log drivers, see <a
         *        href="https://docs.docker.com/engine/admin/logging/overview/">Configure logging drivers </a> in the
         *        <i>Docker documentation</i>.
         *        </p>
         *        <note>
         *        <p>
         *        Amazon ECS currently supports a subset of the logging drivers available to the Docker daemon (shown in
         *        the <code>LogConfiguration</code> data type). Additional log drivers may be available in future
         *        releases of the Amazon ECS container agent.
         *        </p>
         *        </note>
         *        <p>
         *        This parameter requires version 1.18 of the Docker Remote API or greater on your container instance.
         *        To check the Docker Remote API version on your container instance, log in to your container instance
         *        and run the following command: sudo docker version <code>--format '{{.Server.APIVersion}}'</code>
         *        </p>
         *        <note>
         *        <p>
         *        The Amazon ECS container agent running on a container instance must register the logging drivers
         *        available on that instance with the <code>ECS_AVAILABLE_LOGGING_DRIVERS</code> environment variable
         *        before containers placed on that instance can use these log configuration options. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html">Amazon ECS
         *        container agent configuration</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logConfiguration(LogConfiguration logConfiguration);

        /**
         * <p>
         * The log configuration specification for the container.
         * </p>
         * <p>
         * This parameter maps to <code>LogConfig</code> in the <a
         * href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a> section of
         * the <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a> and the
         * <code>--log-driver</code> option to <a
         * href="https://docs.docker.com/engine/reference/run/#security-configuration">docker run</a>.
         * </p>
         * <p>
         * By default, containers use the same logging driver that the Docker daemon uses. However the container can use
         * a different logging driver than the Docker daemon by specifying a log driver with this parameter in the
         * container definition. To use a different logging driver for a container, the log system must be configured
         * properly on the container instance (or on a different log server for remote logging options). For more
         * information about the options for different supported log drivers, see <a
         * href="https://docs.docker.com/engine/admin/logging/overview/">Configure logging drivers </a> in the <i>Docker
         * documentation</i>.
         * </p>
         * <note>
         * <p>
         * Amazon ECS currently supports a subset of the logging drivers available to the Docker daemon (shown in the
         * <code>LogConfiguration</code> data type). Additional log drivers may be available in future releases of the
         * Amazon ECS container agent.
         * </p>
         * </note>
         * <p>
         * This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check
         * the Docker Remote API version on your container instance, log in to your container instance and run the
         * following command: sudo docker version <code>--format '{{.Server.APIVersion}}'</code>
         * </p>
         * <note>
         * <p>
         * The Amazon ECS container agent running on a container instance must register the logging drivers available on
         * that instance with the <code>ECS_AVAILABLE_LOGGING_DRIVERS</code> environment variable before containers
         * placed on that instance can use these log configuration options. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html">Amazon ECS container
         * agent configuration</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link LogConfiguration.Builder}
         * avoiding the need to create one manually via {@link LogConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link LogConfiguration.Builder#build()} is called immediately and its
         * result is passed to {@link #logConfiguration(LogConfiguration)}.
         * 
         * @param logConfiguration
         *        a consumer that will call methods on {@link LogConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #logConfiguration(LogConfiguration)
         */
        default Builder logConfiguration(Consumer<LogConfiguration.Builder> logConfiguration) {
            return logConfiguration(LogConfiguration.builder().applyMutation(logConfiguration).build());
        }

        /**
         * <p>
         * The mount points for data volumes in your container.
         * </p>
         * <p>
         * This parameter maps to <code>Volumes</code> in the <a
         * href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a> section of
         * the <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a> and the <a href="">--volume</a>
         * option to <a href="https://docs.docker.com/engine/reference/run/#security-configuration">docker run</a>.
         * </p>
         * <p>
         * Windows containers can mount whole directories on the same drive as <code>$env:ProgramData</code>. Windows
         * containers can't mount directories on a different drive, and mount point can't be across drives.
         * </p>
         * 
         * @param mountPoints
         *        The mount points for data volumes in your container.</p>
         *        <p>
         *        This parameter maps to <code>Volumes</code> in the <a
         *        href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a>
         *        section of the <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a> and the <a
         *        href="">--volume</a> option to <a
         *        href="https://docs.docker.com/engine/reference/run/#security-configuration">docker run</a>.
         *        </p>
         *        <p>
         *        Windows containers can mount whole directories on the same drive as <code>$env:ProgramData</code>.
         *        Windows containers can't mount directories on a different drive, and mount point can't be across
         *        drives.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mountPoints(Collection<MountPoint> mountPoints);

        /**
         * <p>
         * The mount points for data volumes in your container.
         * </p>
         * <p>
         * This parameter maps to <code>Volumes</code> in the <a
         * href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a> section of
         * the <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a> and the <a href="">--volume</a>
         * option to <a href="https://docs.docker.com/engine/reference/run/#security-configuration">docker run</a>.
         * </p>
         * <p>
         * Windows containers can mount whole directories on the same drive as <code>$env:ProgramData</code>. Windows
         * containers can't mount directories on a different drive, and mount point can't be across drives.
         * </p>
         * 
         * @param mountPoints
         *        The mount points for data volumes in your container.</p>
         *        <p>
         *        This parameter maps to <code>Volumes</code> in the <a
         *        href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a>
         *        section of the <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a> and the <a
         *        href="">--volume</a> option to <a
         *        href="https://docs.docker.com/engine/reference/run/#security-configuration">docker run</a>.
         *        </p>
         *        <p>
         *        Windows containers can mount whole directories on the same drive as <code>$env:ProgramData</code>.
         *        Windows containers can't mount directories on a different drive, and mount point can't be across
         *        drives.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mountPoints(MountPoint... mountPoints);

        /**
         * <p>
         * The mount points for data volumes in your container.
         * </p>
         * <p>
         * This parameter maps to <code>Volumes</code> in the <a
         * href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a> section of
         * the <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a> and the <a href="">--volume</a>
         * option to <a href="https://docs.docker.com/engine/reference/run/#security-configuration">docker run</a>.
         * </p>
         * <p>
         * Windows containers can mount whole directories on the same drive as <code>$env:ProgramData</code>. Windows
         * containers can't mount directories on a different drive, and mount point can't be across drives.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.batch.model.MountPoint.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.batch.model.MountPoint#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.batch.model.MountPoint.Builder#build()} is called immediately and its
         * result is passed to {@link #mountPoints(List<MountPoint>)}.
         * 
         * @param mountPoints
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.batch.model.MountPoint.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #mountPoints(java.util.Collection<MountPoint>)
         */
        Builder mountPoints(Consumer<MountPoint.Builder>... mountPoints);

        /**
         * <p>
         * The name of a container. The name can be used as a unique identifier to target your <code>dependsOn</code>
         * and <code>Overrides</code> objects.
         * </p>
         * 
         * @param name
         *        The name of a container. The name can be used as a unique identifier to target your
         *        <code>dependsOn</code> and <code>Overrides</code> objects.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * When this parameter is <code>true</code>, the container is given elevated privileges on the host container
         * instance (similar to the <code>root</code> user). This parameter maps to <code>Privileged</code> in the <a
         * href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a> section of
         * the <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a> and the
         * <code>--privileged</code> option to <a
         * href="https://docs.docker.com/engine/reference/run/#security-configuration">docker run</a>.
         * </p>
         * <note>
         * <p>
         * This parameter is not supported for Windows containers or tasks run on Fargate.
         * </p>
         * </note>
         * 
         * @param privileged
         *        When this parameter is <code>true</code>, the container is given elevated privileges on the host
         *        container instance (similar to the <code>root</code> user). This parameter maps to
         *        <code>Privileged</code> in the <a
         *        href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a>
         *        section of the <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a> and the
         *        <code>--privileged</code> option to <a
         *        href="https://docs.docker.com/engine/reference/run/#security-configuration">docker run</a>.</p> <note>
         *        <p>
         *        This parameter is not supported for Windows containers or tasks run on Fargate.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder privileged(Boolean privileged);

        /**
         * <p>
         * When this parameter is true, the container is given read-only access to its root file system. This parameter
         * maps to <code>ReadonlyRootfs</code> in the <a
         * href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a> section of
         * the <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a> and the
         * <code>--read-only</code> option to <a
         * href="https://docs.docker.com/engine/reference/run/#security-configuration">docker run</a>.
         * </p>
         * <note>
         * <p>
         * This parameter is not supported for Windows containers.
         * </p>
         * </note>
         * 
         * @param readonlyRootFilesystem
         *        When this parameter is true, the container is given read-only access to its root file system. This
         *        parameter maps to <code>ReadonlyRootfs</code> in the <a
         *        href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a>
         *        section of the <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a> and the
         *        <code>--read-only</code> option to <a
         *        href="https://docs.docker.com/engine/reference/run/#security-configuration">docker run</a>.</p> <note>
         *        <p>
         *        This parameter is not supported for Windows containers.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder readonlyRootFilesystem(Boolean readonlyRootFilesystem);

        /**
         * <p>
         * The private repository authentication credentials to use.
         * </p>
         * 
         * @param repositoryCredentials
         *        The private repository authentication credentials to use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder repositoryCredentials(RepositoryCredentials repositoryCredentials);

        /**
         * <p>
         * The private repository authentication credentials to use.
         * </p>
         * This is a convenience method that creates an instance of the {@link RepositoryCredentials.Builder} avoiding
         * the need to create one manually via {@link RepositoryCredentials#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RepositoryCredentials.Builder#build()} is called immediately and
         * its result is passed to {@link #repositoryCredentials(RepositoryCredentials)}.
         * 
         * @param repositoryCredentials
         *        a consumer that will call methods on {@link RepositoryCredentials.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #repositoryCredentials(RepositoryCredentials)
         */
        default Builder repositoryCredentials(Consumer<RepositoryCredentials.Builder> repositoryCredentials) {
            return repositoryCredentials(RepositoryCredentials.builder().applyMutation(repositoryCredentials).build());
        }

        /**
         * <p>
         * The type and amount of a resource to assign to a container. The only supported resource is a GPU.
         * </p>
         * 
         * @param resourceRequirements
         *        The type and amount of a resource to assign to a container. The only supported resource is a GPU.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceRequirements(Collection<ResourceRequirement> resourceRequirements);

        /**
         * <p>
         * The type and amount of a resource to assign to a container. The only supported resource is a GPU.
         * </p>
         * 
         * @param resourceRequirements
         *        The type and amount of a resource to assign to a container. The only supported resource is a GPU.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceRequirements(ResourceRequirement... resourceRequirements);

        /**
         * <p>
         * The type and amount of a resource to assign to a container. The only supported resource is a GPU.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.batch.model.ResourceRequirement.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.batch.model.ResourceRequirement#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.batch.model.ResourceRequirement.Builder#build()} is called immediately
         * and its result is passed to {@link #resourceRequirements(List<ResourceRequirement>)}.
         * 
         * @param resourceRequirements
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.batch.model.ResourceRequirement.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resourceRequirements(java.util.Collection<ResourceRequirement>)
         */
        Builder resourceRequirements(Consumer<ResourceRequirement.Builder>... resourceRequirements);

        /**
         * <p>
         * The secrets to pass to the container. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html">Specifying
         * Sensitive Data</a> in the Amazon Elastic Container Service Developer Guide.
         * </p>
         * 
         * @param secrets
         *        The secrets to pass to the container. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html"
         *        >Specifying Sensitive Data</a> in the Amazon Elastic Container Service Developer Guide.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secrets(Collection<Secret> secrets);

        /**
         * <p>
         * The secrets to pass to the container. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html">Specifying
         * Sensitive Data</a> in the Amazon Elastic Container Service Developer Guide.
         * </p>
         * 
         * @param secrets
         *        The secrets to pass to the container. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html"
         *        >Specifying Sensitive Data</a> in the Amazon Elastic Container Service Developer Guide.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secrets(Secret... secrets);

        /**
         * <p>
         * The secrets to pass to the container. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html">Specifying
         * Sensitive Data</a> in the Amazon Elastic Container Service Developer Guide.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.batch.model.Secret.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.batch.model.Secret#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.batch.model.Secret.Builder#build()} is called immediately and its
         * result is passed to {@link #secrets(List<Secret>)}.
         * 
         * @param secrets
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.batch.model.Secret.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #secrets(java.util.Collection<Secret>)
         */
        Builder secrets(Consumer<Secret.Builder>... secrets);

        /**
         * <p>
         * A list of <code>ulimits</code> to set in the container. If a <code>ulimit</code> value is specified in a task
         * definition, it overrides the default values set by Docker. This parameter maps to <code>Ulimits</code> in the
         * <a href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a> section
         * of the <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a> and the
         * <code>--ulimit</code> option to <a
         * href="https://docs.docker.com/engine/reference/run/#security-configuration">docker run</a>.
         * </p>
         * <p>
         * Amazon ECS tasks hosted on Fargate use the default resource limit values set by the operating system with the
         * exception of the nofile resource limit parameter which Fargate overrides. The <code>nofile</code> resource
         * limit sets a restriction on the number of open files that a container can use. The default
         * <code>nofile</code> soft limit is <code>1024</code> and the default hard limit is <code>65535</code>.
         * </p>
         * <p>
         * This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check
         * the Docker Remote API version on your container instance, log in to your container instance and run the
         * following command: sudo docker version <code>--format '{{.Server.APIVersion}}'</code>
         * </p>
         * <note>
         * <p>
         * This parameter is not supported for Windows containers.
         * </p>
         * </note>
         * 
         * @param ulimits
         *        A list of <code>ulimits</code> to set in the container. If a <code>ulimit</code> value is specified in
         *        a task definition, it overrides the default values set by Docker. This parameter maps to
         *        <code>Ulimits</code> in the <a
         *        href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a>
         *        section of the <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a> and the
         *        <code>--ulimit</code> option to <a
         *        href="https://docs.docker.com/engine/reference/run/#security-configuration">docker run</a>.</p>
         *        <p>
         *        Amazon ECS tasks hosted on Fargate use the default resource limit values set by the operating system
         *        with the exception of the nofile resource limit parameter which Fargate overrides. The
         *        <code>nofile</code> resource limit sets a restriction on the number of open files that a container can
         *        use. The default <code>nofile</code> soft limit is <code>1024</code> and the default hard limit is
         *        <code>65535</code>.
         *        </p>
         *        <p>
         *        This parameter requires version 1.18 of the Docker Remote API or greater on your container instance.
         *        To check the Docker Remote API version on your container instance, log in to your container instance
         *        and run the following command: sudo docker version <code>--format '{{.Server.APIVersion}}'</code>
         *        </p>
         *        <note>
         *        <p>
         *        This parameter is not supported for Windows containers.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ulimits(Collection<Ulimit> ulimits);

        /**
         * <p>
         * A list of <code>ulimits</code> to set in the container. If a <code>ulimit</code> value is specified in a task
         * definition, it overrides the default values set by Docker. This parameter maps to <code>Ulimits</code> in the
         * <a href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a> section
         * of the <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a> and the
         * <code>--ulimit</code> option to <a
         * href="https://docs.docker.com/engine/reference/run/#security-configuration">docker run</a>.
         * </p>
         * <p>
         * Amazon ECS tasks hosted on Fargate use the default resource limit values set by the operating system with the
         * exception of the nofile resource limit parameter which Fargate overrides. The <code>nofile</code> resource
         * limit sets a restriction on the number of open files that a container can use. The default
         * <code>nofile</code> soft limit is <code>1024</code> and the default hard limit is <code>65535</code>.
         * </p>
         * <p>
         * This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check
         * the Docker Remote API version on your container instance, log in to your container instance and run the
         * following command: sudo docker version <code>--format '{{.Server.APIVersion}}'</code>
         * </p>
         * <note>
         * <p>
         * This parameter is not supported for Windows containers.
         * </p>
         * </note>
         * 
         * @param ulimits
         *        A list of <code>ulimits</code> to set in the container. If a <code>ulimit</code> value is specified in
         *        a task definition, it overrides the default values set by Docker. This parameter maps to
         *        <code>Ulimits</code> in the <a
         *        href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a>
         *        section of the <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a> and the
         *        <code>--ulimit</code> option to <a
         *        href="https://docs.docker.com/engine/reference/run/#security-configuration">docker run</a>.</p>
         *        <p>
         *        Amazon ECS tasks hosted on Fargate use the default resource limit values set by the operating system
         *        with the exception of the nofile resource limit parameter which Fargate overrides. The
         *        <code>nofile</code> resource limit sets a restriction on the number of open files that a container can
         *        use. The default <code>nofile</code> soft limit is <code>1024</code> and the default hard limit is
         *        <code>65535</code>.
         *        </p>
         *        <p>
         *        This parameter requires version 1.18 of the Docker Remote API or greater on your container instance.
         *        To check the Docker Remote API version on your container instance, log in to your container instance
         *        and run the following command: sudo docker version <code>--format '{{.Server.APIVersion}}'</code>
         *        </p>
         *        <note>
         *        <p>
         *        This parameter is not supported for Windows containers.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ulimits(Ulimit... ulimits);

        /**
         * <p>
         * A list of <code>ulimits</code> to set in the container. If a <code>ulimit</code> value is specified in a task
         * definition, it overrides the default values set by Docker. This parameter maps to <code>Ulimits</code> in the
         * <a href="https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate">Create a container</a> section
         * of the <a href="https://docs.docker.com/engine/api/v1.35/">Docker Remote API</a> and the
         * <code>--ulimit</code> option to <a
         * href="https://docs.docker.com/engine/reference/run/#security-configuration">docker run</a>.
         * </p>
         * <p>
         * Amazon ECS tasks hosted on Fargate use the default resource limit values set by the operating system with the
         * exception of the nofile resource limit parameter which Fargate overrides. The <code>nofile</code> resource
         * limit sets a restriction on the number of open files that a container can use. The default
         * <code>nofile</code> soft limit is <code>1024</code> and the default hard limit is <code>65535</code>.
         * </p>
         * <p>
         * This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check
         * the Docker Remote API version on your container instance, log in to your container instance and run the
         * following command: sudo docker version <code>--format '{{.Server.APIVersion}}'</code>
         * </p>
         * <note>
         * <p>
         * This parameter is not supported for Windows containers.
         * </p>
         * </note> This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.batch.model.Ulimit.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.batch.model.Ulimit#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.batch.model.Ulimit.Builder#build()} is called immediately and its
         * result is passed to {@link #ulimits(List<Ulimit>)}.
         * 
         * @param ulimits
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.batch.model.Ulimit.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ulimits(java.util.Collection<Ulimit>)
         */
        Builder ulimits(Consumer<Ulimit.Builder>... ulimits);

        /**
         * <p>
         * The user to use inside the container. This parameter maps to User in the Create a container section of the
         * Docker Remote API and the --user option to docker run.
         * </p>
         * <note>
         * <p>
         * When running tasks using the <code>host</code> network mode, don't run containers using the
         * <code>root user (UID 0)</code>. We recommend using a non-root user for better security.
         * </p>
         * </note>
         * <p>
         * You can specify the <code>user</code> using the following formats. If specifying a UID or GID, you must
         * specify it as a positive integer.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>user</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>user:group</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>uid</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>uid:gid</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>user:gi</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>uid:group</code>
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * This parameter is not supported for Windows containers.
         * </p>
         * </note>
         * 
         * @param user
         *        The user to use inside the container. This parameter maps to User in the Create a container section of
         *        the Docker Remote API and the --user option to docker run.</p> <note>
         *        <p>
         *        When running tasks using the <code>host</code> network mode, don't run containers using the
         *        <code>root user (UID 0)</code>. We recommend using a non-root user for better security.
         *        </p>
         *        </note>
         *        <p>
         *        You can specify the <code>user</code> using the following formats. If specifying a UID or GID, you
         *        must specify it as a positive integer.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>user</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>user:group</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>uid</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>uid:gid</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>user:gi</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>uid:group</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        This parameter is not supported for Windows containers.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder user(String user);
    }

    static final class BuilderImpl implements Builder {
        private List<String> command = DefaultSdkAutoConstructList.getInstance();

        private List<TaskContainerDependency> dependsOn = DefaultSdkAutoConstructList.getInstance();

        private List<KeyValuePair> environment = DefaultSdkAutoConstructList.getInstance();

        private Boolean essential;

        private String image;

        private LinuxParameters linuxParameters;

        private LogConfiguration logConfiguration;

        private List<MountPoint> mountPoints = DefaultSdkAutoConstructList.getInstance();

        private String name;

        private Boolean privileged;

        private Boolean readonlyRootFilesystem;

        private RepositoryCredentials repositoryCredentials;

        private List<ResourceRequirement> resourceRequirements = DefaultSdkAutoConstructList.getInstance();

        private List<Secret> secrets = DefaultSdkAutoConstructList.getInstance();

        private List<Ulimit> ulimits = DefaultSdkAutoConstructList.getInstance();

        private String user;

        private BuilderImpl() {
        }

        private BuilderImpl(TaskContainerProperties model) {
            command(model.command);
            dependsOn(model.dependsOn);
            environment(model.environment);
            essential(model.essential);
            image(model.image);
            linuxParameters(model.linuxParameters);
            logConfiguration(model.logConfiguration);
            mountPoints(model.mountPoints);
            name(model.name);
            privileged(model.privileged);
            readonlyRootFilesystem(model.readonlyRootFilesystem);
            repositoryCredentials(model.repositoryCredentials);
            resourceRequirements(model.resourceRequirements);
            secrets(model.secrets);
            ulimits(model.ulimits);
            user(model.user);
        }

        public final Collection<String> getCommand() {
            if (command instanceof SdkAutoConstructList) {
                return null;
            }
            return command;
        }

        public final void setCommand(Collection<String> command) {
            this.command = StringListCopier.copy(command);
        }

        @Override
        public final Builder command(Collection<String> command) {
            this.command = StringListCopier.copy(command);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder command(String... command) {
            command(Arrays.asList(command));
            return this;
        }

        public final List<TaskContainerDependency.Builder> getDependsOn() {
            List<TaskContainerDependency.Builder> result = TaskContainerDependencyListCopier.copyToBuilder(this.dependsOn);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDependsOn(Collection<TaskContainerDependency.BuilderImpl> dependsOn) {
            this.dependsOn = TaskContainerDependencyListCopier.copyFromBuilder(dependsOn);
        }

        @Override
        public final Builder dependsOn(Collection<TaskContainerDependency> dependsOn) {
            this.dependsOn = TaskContainerDependencyListCopier.copy(dependsOn);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dependsOn(TaskContainerDependency... dependsOn) {
            dependsOn(Arrays.asList(dependsOn));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dependsOn(Consumer<TaskContainerDependency.Builder>... dependsOn) {
            dependsOn(Stream.of(dependsOn).map(c -> TaskContainerDependency.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<KeyValuePair.Builder> getEnvironment() {
            List<KeyValuePair.Builder> result = EnvironmentVariablesCopier.copyToBuilder(this.environment);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setEnvironment(Collection<KeyValuePair.BuilderImpl> environment) {
            this.environment = EnvironmentVariablesCopier.copyFromBuilder(environment);
        }

        @Override
        public final Builder environment(Collection<KeyValuePair> environment) {
            this.environment = EnvironmentVariablesCopier.copy(environment);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder environment(KeyValuePair... environment) {
            environment(Arrays.asList(environment));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder environment(Consumer<KeyValuePair.Builder>... environment) {
            environment(Stream.of(environment).map(c -> KeyValuePair.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Boolean getEssential() {
            return essential;
        }

        public final void setEssential(Boolean essential) {
            this.essential = essential;
        }

        @Override
        public final Builder essential(Boolean essential) {
            this.essential = essential;
            return this;
        }

        public final String getImage() {
            return image;
        }

        public final void setImage(String image) {
            this.image = image;
        }

        @Override
        public final Builder image(String image) {
            this.image = image;
            return this;
        }

        public final LinuxParameters.Builder getLinuxParameters() {
            return linuxParameters != null ? linuxParameters.toBuilder() : null;
        }

        public final void setLinuxParameters(LinuxParameters.BuilderImpl linuxParameters) {
            this.linuxParameters = linuxParameters != null ? linuxParameters.build() : null;
        }

        @Override
        public final Builder linuxParameters(LinuxParameters linuxParameters) {
            this.linuxParameters = linuxParameters;
            return this;
        }

        public final LogConfiguration.Builder getLogConfiguration() {
            return logConfiguration != null ? logConfiguration.toBuilder() : null;
        }

        public final void setLogConfiguration(LogConfiguration.BuilderImpl logConfiguration) {
            this.logConfiguration = logConfiguration != null ? logConfiguration.build() : null;
        }

        @Override
        public final Builder logConfiguration(LogConfiguration logConfiguration) {
            this.logConfiguration = logConfiguration;
            return this;
        }

        public final List<MountPoint.Builder> getMountPoints() {
            List<MountPoint.Builder> result = MountPointsCopier.copyToBuilder(this.mountPoints);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setMountPoints(Collection<MountPoint.BuilderImpl> mountPoints) {
            this.mountPoints = MountPointsCopier.copyFromBuilder(mountPoints);
        }

        @Override
        public final Builder mountPoints(Collection<MountPoint> mountPoints) {
            this.mountPoints = MountPointsCopier.copy(mountPoints);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder mountPoints(MountPoint... mountPoints) {
            mountPoints(Arrays.asList(mountPoints));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder mountPoints(Consumer<MountPoint.Builder>... mountPoints) {
            mountPoints(Stream.of(mountPoints).map(c -> MountPoint.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Boolean getPrivileged() {
            return privileged;
        }

        public final void setPrivileged(Boolean privileged) {
            this.privileged = privileged;
        }

        @Override
        public final Builder privileged(Boolean privileged) {
            this.privileged = privileged;
            return this;
        }

        public final Boolean getReadonlyRootFilesystem() {
            return readonlyRootFilesystem;
        }

        public final void setReadonlyRootFilesystem(Boolean readonlyRootFilesystem) {
            this.readonlyRootFilesystem = readonlyRootFilesystem;
        }

        @Override
        public final Builder readonlyRootFilesystem(Boolean readonlyRootFilesystem) {
            this.readonlyRootFilesystem = readonlyRootFilesystem;
            return this;
        }

        public final RepositoryCredentials.Builder getRepositoryCredentials() {
            return repositoryCredentials != null ? repositoryCredentials.toBuilder() : null;
        }

        public final void setRepositoryCredentials(RepositoryCredentials.BuilderImpl repositoryCredentials) {
            this.repositoryCredentials = repositoryCredentials != null ? repositoryCredentials.build() : null;
        }

        @Override
        public final Builder repositoryCredentials(RepositoryCredentials repositoryCredentials) {
            this.repositoryCredentials = repositoryCredentials;
            return this;
        }

        public final List<ResourceRequirement.Builder> getResourceRequirements() {
            List<ResourceRequirement.Builder> result = ResourceRequirementsCopier.copyToBuilder(this.resourceRequirements);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setResourceRequirements(Collection<ResourceRequirement.BuilderImpl> resourceRequirements) {
            this.resourceRequirements = ResourceRequirementsCopier.copyFromBuilder(resourceRequirements);
        }

        @Override
        public final Builder resourceRequirements(Collection<ResourceRequirement> resourceRequirements) {
            this.resourceRequirements = ResourceRequirementsCopier.copy(resourceRequirements);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceRequirements(ResourceRequirement... resourceRequirements) {
            resourceRequirements(Arrays.asList(resourceRequirements));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceRequirements(Consumer<ResourceRequirement.Builder>... resourceRequirements) {
            resourceRequirements(Stream.of(resourceRequirements).map(c -> ResourceRequirement.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<Secret.Builder> getSecrets() {
            List<Secret.Builder> result = SecretListCopier.copyToBuilder(this.secrets);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSecrets(Collection<Secret.BuilderImpl> secrets) {
            this.secrets = SecretListCopier.copyFromBuilder(secrets);
        }

        @Override
        public final Builder secrets(Collection<Secret> secrets) {
            this.secrets = SecretListCopier.copy(secrets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder secrets(Secret... secrets) {
            secrets(Arrays.asList(secrets));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder secrets(Consumer<Secret.Builder>... secrets) {
            secrets(Stream.of(secrets).map(c -> Secret.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<Ulimit.Builder> getUlimits() {
            List<Ulimit.Builder> result = UlimitsCopier.copyToBuilder(this.ulimits);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setUlimits(Collection<Ulimit.BuilderImpl> ulimits) {
            this.ulimits = UlimitsCopier.copyFromBuilder(ulimits);
        }

        @Override
        public final Builder ulimits(Collection<Ulimit> ulimits) {
            this.ulimits = UlimitsCopier.copy(ulimits);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ulimits(Ulimit... ulimits) {
            ulimits(Arrays.asList(ulimits));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ulimits(Consumer<Ulimit.Builder>... ulimits) {
            ulimits(Stream.of(ulimits).map(c -> Ulimit.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getUser() {
            return user;
        }

        public final void setUser(String user) {
            this.user = user;
        }

        @Override
        public final Builder user(String user) {
            this.user = user;
            return this;
        }

        @Override
        public TaskContainerProperties build() {
            return new TaskContainerProperties(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
