/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.batch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The overrides that should be sent to a container.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ContainerOverrides implements SdkPojo, Serializable,
        ToCopyableBuilder<ContainerOverrides.Builder, ContainerOverrides> {
    private static final SdkField<Integer> VCPUS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(ContainerOverrides::vcpus)).setter(setter(Builder::vcpus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("vcpus").build()).build();

    private static final SdkField<Integer> MEMORY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(ContainerOverrides::memory)).setter(setter(Builder::memory))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("memory").build()).build();

    private static final SdkField<List<String>> COMMAND_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(ContainerOverrides::command))
            .setter(setter(Builder::command))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("command").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ContainerOverrides::instanceType)).setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instanceType").build()).build();

    private static final SdkField<List<KeyValuePair>> ENVIRONMENT_FIELD = SdkField
            .<List<KeyValuePair>> builder(MarshallingType.LIST)
            .getter(getter(ContainerOverrides::environment))
            .setter(setter(Builder::environment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("environment").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<KeyValuePair> builder(MarshallingType.SDK_POJO)
                                            .constructor(KeyValuePair::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VCPUS_FIELD, MEMORY_FIELD,
            COMMAND_FIELD, INSTANCE_TYPE_FIELD, ENVIRONMENT_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer vcpus;

    private final Integer memory;

    private final List<String> command;

    private final String instanceType;

    private final List<KeyValuePair> environment;

    private ContainerOverrides(BuilderImpl builder) {
        this.vcpus = builder.vcpus;
        this.memory = builder.memory;
        this.command = builder.command;
        this.instanceType = builder.instanceType;
        this.environment = builder.environment;
    }

    /**
     * <p>
     * The number of vCPUs to reserve for the container. This value overrides the value set in the job definition.
     * </p>
     * 
     * @return The number of vCPUs to reserve for the container. This value overrides the value set in the job
     *         definition.
     */
    public Integer vcpus() {
        return vcpus;
    }

    /**
     * <p>
     * The number of MiB of memory reserved for the job. This value overrides the value set in the job definition.
     * </p>
     * 
     * @return The number of MiB of memory reserved for the job. This value overrides the value set in the job
     *         definition.
     */
    public Integer memory() {
        return memory;
    }

    /**
     * <p>
     * The command to send to the container that overrides the default command from the Docker image or the job
     * definition.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The command to send to the container that overrides the default command from the Docker image or the job
     *         definition.
     */
    public List<String> command() {
        return command;
    }

    /**
     * <p>
     * The instance type to use for a multi-node parallel job. This parameter is not valid for single-node container
     * jobs.
     * </p>
     * 
     * @return The instance type to use for a multi-node parallel job. This parameter is not valid for single-node
     *         container jobs.
     */
    public String instanceType() {
        return instanceType;
    }

    /**
     * <p>
     * The environment variables to send to the container. You can add new environment variables, which are added to the
     * container at launch, or you can override the existing environment variables from the Docker image or the job
     * definition.
     * </p>
     * <note>
     * <p>
     * Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved for
     * variables that are set by the AWS Batch service.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The environment variables to send to the container. You can add new environment variables, which are
     *         added to the container at launch, or you can override the existing environment variables from the Docker
     *         image or the job definition.</p> <note>
     *         <p>
     *         Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved for
     *         variables that are set by the AWS Batch service.
     *         </p>
     */
    public List<KeyValuePair> environment() {
        return environment;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(vcpus());
        hashCode = 31 * hashCode + Objects.hashCode(memory());
        hashCode = 31 * hashCode + Objects.hashCode(command());
        hashCode = 31 * hashCode + Objects.hashCode(instanceType());
        hashCode = 31 * hashCode + Objects.hashCode(environment());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ContainerOverrides)) {
            return false;
        }
        ContainerOverrides other = (ContainerOverrides) obj;
        return Objects.equals(vcpus(), other.vcpus()) && Objects.equals(memory(), other.memory())
                && Objects.equals(command(), other.command()) && Objects.equals(instanceType(), other.instanceType())
                && Objects.equals(environment(), other.environment());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ContainerOverrides").add("Vcpus", vcpus()).add("Memory", memory()).add("Command", command())
                .add("InstanceType", instanceType()).add("Environment", environment()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "vcpus":
            return Optional.ofNullable(clazz.cast(vcpus()));
        case "memory":
            return Optional.ofNullable(clazz.cast(memory()));
        case "command":
            return Optional.ofNullable(clazz.cast(command()));
        case "instanceType":
            return Optional.ofNullable(clazz.cast(instanceType()));
        case "environment":
            return Optional.ofNullable(clazz.cast(environment()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ContainerOverrides, T> g) {
        return obj -> g.apply((ContainerOverrides) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ContainerOverrides> {
        /**
         * <p>
         * The number of vCPUs to reserve for the container. This value overrides the value set in the job definition.
         * </p>
         * 
         * @param vcpus
         *        The number of vCPUs to reserve for the container. This value overrides the value set in the job
         *        definition.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vcpus(Integer vcpus);

        /**
         * <p>
         * The number of MiB of memory reserved for the job. This value overrides the value set in the job definition.
         * </p>
         * 
         * @param memory
         *        The number of MiB of memory reserved for the job. This value overrides the value set in the job
         *        definition.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder memory(Integer memory);

        /**
         * <p>
         * The command to send to the container that overrides the default command from the Docker image or the job
         * definition.
         * </p>
         * 
         * @param command
         *        The command to send to the container that overrides the default command from the Docker image or the
         *        job definition.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(Collection<String> command);

        /**
         * <p>
         * The command to send to the container that overrides the default command from the Docker image or the job
         * definition.
         * </p>
         * 
         * @param command
         *        The command to send to the container that overrides the default command from the Docker image or the
         *        job definition.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(String... command);

        /**
         * <p>
         * The instance type to use for a multi-node parallel job. This parameter is not valid for single-node container
         * jobs.
         * </p>
         * 
         * @param instanceType
         *        The instance type to use for a multi-node parallel job. This parameter is not valid for single-node
         *        container jobs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * The environment variables to send to the container. You can add new environment variables, which are added to
         * the container at launch, or you can override the existing environment variables from the Docker image or the
         * job definition.
         * </p>
         * <note>
         * <p>
         * Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved for
         * variables that are set by the AWS Batch service.
         * </p>
         * </note>
         * 
         * @param environment
         *        The environment variables to send to the container. You can add new environment variables, which are
         *        added to the container at launch, or you can override the existing environment variables from the
         *        Docker image or the job definition.</p> <note>
         *        <p>
         *        Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved
         *        for variables that are set by the AWS Batch service.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environment(Collection<KeyValuePair> environment);

        /**
         * <p>
         * The environment variables to send to the container. You can add new environment variables, which are added to
         * the container at launch, or you can override the existing environment variables from the Docker image or the
         * job definition.
         * </p>
         * <note>
         * <p>
         * Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved for
         * variables that are set by the AWS Batch service.
         * </p>
         * </note>
         * 
         * @param environment
         *        The environment variables to send to the container. You can add new environment variables, which are
         *        added to the container at launch, or you can override the existing environment variables from the
         *        Docker image or the job definition.</p> <note>
         *        <p>
         *        Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved
         *        for variables that are set by the AWS Batch service.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environment(KeyValuePair... environment);

        /**
         * <p>
         * The environment variables to send to the container. You can add new environment variables, which are added to
         * the container at launch, or you can override the existing environment variables from the Docker image or the
         * job definition.
         * </p>
         * <note>
         * <p>
         * Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved for
         * variables that are set by the AWS Batch service.
         * </p>
         * </note> This is a convenience that creates an instance of the {@link List<KeyValuePair>.Builder} avoiding the
         * need to create one manually via {@link List<KeyValuePair>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<KeyValuePair>.Builder#build()} is called immediately and its
         * result is passed to {@link #environment(List<KeyValuePair>)}.
         * 
         * @param environment
         *        a consumer that will call methods on {@link List<KeyValuePair>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #environment(List<KeyValuePair>)
         */
        Builder environment(Consumer<KeyValuePair.Builder>... environment);
    }

    static final class BuilderImpl implements Builder {
        private Integer vcpus;

        private Integer memory;

        private List<String> command = DefaultSdkAutoConstructList.getInstance();

        private String instanceType;

        private List<KeyValuePair> environment = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ContainerOverrides model) {
            vcpus(model.vcpus);
            memory(model.memory);
            command(model.command);
            instanceType(model.instanceType);
            environment(model.environment);
        }

        public final Integer getVcpus() {
            return vcpus;
        }

        @Override
        public final Builder vcpus(Integer vcpus) {
            this.vcpus = vcpus;
            return this;
        }

        public final void setVcpus(Integer vcpus) {
            this.vcpus = vcpus;
        }

        public final Integer getMemory() {
            return memory;
        }

        @Override
        public final Builder memory(Integer memory) {
            this.memory = memory;
            return this;
        }

        public final void setMemory(Integer memory) {
            this.memory = memory;
        }

        public final Collection<String> getCommand() {
            return command;
        }

        @Override
        public final Builder command(Collection<String> command) {
            this.command = StringListCopier.copy(command);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder command(String... command) {
            command(Arrays.asList(command));
            return this;
        }

        public final void setCommand(Collection<String> command) {
            this.command = StringListCopier.copy(command);
        }

        public final String getInstanceType() {
            return instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        public final Collection<KeyValuePair.Builder> getEnvironment() {
            return environment != null ? environment.stream().map(KeyValuePair::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder environment(Collection<KeyValuePair> environment) {
            this.environment = EnvironmentVariablesCopier.copy(environment);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder environment(KeyValuePair... environment) {
            environment(Arrays.asList(environment));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder environment(Consumer<KeyValuePair.Builder>... environment) {
            environment(Stream.of(environment).map(c -> KeyValuePair.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setEnvironment(Collection<KeyValuePair.BuilderImpl> environment) {
            this.environment = EnvironmentVariablesCopier.copyFromBuilder(environment);
        }

        @Override
        public ContainerOverrides build() {
            return new ContainerOverrides(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
