/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.batch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object representing the details of a container that is part of a job.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ContainerDetail implements SdkPojo, Serializable, ToCopyableBuilder<ContainerDetail.Builder, ContainerDetail> {
    private static final SdkField<String> IMAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ContainerDetail::image)).setter(setter(Builder::image))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("image").build()).build();

    private static final SdkField<Integer> VCPUS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(ContainerDetail::vcpus)).setter(setter(Builder::vcpus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("vcpus").build()).build();

    private static final SdkField<Integer> MEMORY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(ContainerDetail::memory)).setter(setter(Builder::memory))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("memory").build()).build();

    private static final SdkField<List<String>> COMMAND_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(ContainerDetail::command))
            .setter(setter(Builder::command))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("command").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> JOB_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ContainerDetail::jobRoleArn)).setter(setter(Builder::jobRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("jobRoleArn").build()).build();

    private static final SdkField<List<Volume>> VOLUMES_FIELD = SdkField
            .<List<Volume>> builder(MarshallingType.LIST)
            .getter(getter(ContainerDetail::volumes))
            .setter(setter(Builder::volumes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("volumes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Volume> builder(MarshallingType.SDK_POJO)
                                            .constructor(Volume::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<KeyValuePair>> ENVIRONMENT_FIELD = SdkField
            .<List<KeyValuePair>> builder(MarshallingType.LIST)
            .getter(getter(ContainerDetail::environment))
            .setter(setter(Builder::environment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("environment").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<KeyValuePair> builder(MarshallingType.SDK_POJO)
                                            .constructor(KeyValuePair::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<MountPoint>> MOUNT_POINTS_FIELD = SdkField
            .<List<MountPoint>> builder(MarshallingType.LIST)
            .getter(getter(ContainerDetail::mountPoints))
            .setter(setter(Builder::mountPoints))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("mountPoints").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<MountPoint> builder(MarshallingType.SDK_POJO)
                                            .constructor(MountPoint::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> READONLY_ROOT_FILESYSTEM_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(ContainerDetail::readonlyRootFilesystem)).setter(setter(Builder::readonlyRootFilesystem))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("readonlyRootFilesystem").build())
            .build();

    private static final SdkField<List<Ulimit>> ULIMITS_FIELD = SdkField
            .<List<Ulimit>> builder(MarshallingType.LIST)
            .getter(getter(ContainerDetail::ulimits))
            .setter(setter(Builder::ulimits))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ulimits").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Ulimit> builder(MarshallingType.SDK_POJO)
                                            .constructor(Ulimit::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> PRIVILEGED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(ContainerDetail::privileged)).setter(setter(Builder::privileged))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("privileged").build()).build();

    private static final SdkField<String> USER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ContainerDetail::user)).setter(setter(Builder::user))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("user").build()).build();

    private static final SdkField<Integer> EXIT_CODE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(ContainerDetail::exitCode)).setter(setter(Builder::exitCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("exitCode").build()).build();

    private static final SdkField<String> REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ContainerDetail::reason)).setter(setter(Builder::reason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("reason").build()).build();

    private static final SdkField<String> CONTAINER_INSTANCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ContainerDetail::containerInstanceArn)).setter(setter(Builder::containerInstanceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("containerInstanceArn").build())
            .build();

    private static final SdkField<String> TASK_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ContainerDetail::taskArn)).setter(setter(Builder::taskArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("taskArn").build()).build();

    private static final SdkField<String> LOG_STREAM_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ContainerDetail::logStreamName)).setter(setter(Builder::logStreamName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("logStreamName").build()).build();

    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ContainerDetail::instanceType)).setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instanceType").build()).build();

    private static final SdkField<List<NetworkInterface>> NETWORK_INTERFACES_FIELD = SdkField
            .<List<NetworkInterface>> builder(MarshallingType.LIST)
            .getter(getter(ContainerDetail::networkInterfaces))
            .setter(setter(Builder::networkInterfaces))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("networkInterfaces").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<NetworkInterface> builder(MarshallingType.SDK_POJO)
                                            .constructor(NetworkInterface::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<ResourceRequirement>> RESOURCE_REQUIREMENTS_FIELD = SdkField
            .<List<ResourceRequirement>> builder(MarshallingType.LIST)
            .getter(getter(ContainerDetail::resourceRequirements))
            .setter(setter(Builder::resourceRequirements))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceRequirements").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ResourceRequirement> builder(MarshallingType.SDK_POJO)
                                            .constructor(ResourceRequirement::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(IMAGE_FIELD, VCPUS_FIELD,
            MEMORY_FIELD, COMMAND_FIELD, JOB_ROLE_ARN_FIELD, VOLUMES_FIELD, ENVIRONMENT_FIELD, MOUNT_POINTS_FIELD,
            READONLY_ROOT_FILESYSTEM_FIELD, ULIMITS_FIELD, PRIVILEGED_FIELD, USER_FIELD, EXIT_CODE_FIELD, REASON_FIELD,
            CONTAINER_INSTANCE_ARN_FIELD, TASK_ARN_FIELD, LOG_STREAM_NAME_FIELD, INSTANCE_TYPE_FIELD, NETWORK_INTERFACES_FIELD,
            RESOURCE_REQUIREMENTS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String image;

    private final Integer vcpus;

    private final Integer memory;

    private final List<String> command;

    private final String jobRoleArn;

    private final List<Volume> volumes;

    private final List<KeyValuePair> environment;

    private final List<MountPoint> mountPoints;

    private final Boolean readonlyRootFilesystem;

    private final List<Ulimit> ulimits;

    private final Boolean privileged;

    private final String user;

    private final Integer exitCode;

    private final String reason;

    private final String containerInstanceArn;

    private final String taskArn;

    private final String logStreamName;

    private final String instanceType;

    private final List<NetworkInterface> networkInterfaces;

    private final List<ResourceRequirement> resourceRequirements;

    private ContainerDetail(BuilderImpl builder) {
        this.image = builder.image;
        this.vcpus = builder.vcpus;
        this.memory = builder.memory;
        this.command = builder.command;
        this.jobRoleArn = builder.jobRoleArn;
        this.volumes = builder.volumes;
        this.environment = builder.environment;
        this.mountPoints = builder.mountPoints;
        this.readonlyRootFilesystem = builder.readonlyRootFilesystem;
        this.ulimits = builder.ulimits;
        this.privileged = builder.privileged;
        this.user = builder.user;
        this.exitCode = builder.exitCode;
        this.reason = builder.reason;
        this.containerInstanceArn = builder.containerInstanceArn;
        this.taskArn = builder.taskArn;
        this.logStreamName = builder.logStreamName;
        this.instanceType = builder.instanceType;
        this.networkInterfaces = builder.networkInterfaces;
        this.resourceRequirements = builder.resourceRequirements;
    }

    /**
     * <p>
     * The image used to start the container.
     * </p>
     * 
     * @return The image used to start the container.
     */
    public String image() {
        return image;
    }

    /**
     * <p>
     * The number of VCPUs allocated for the job.
     * </p>
     * 
     * @return The number of VCPUs allocated for the job.
     */
    public Integer vcpus() {
        return vcpus;
    }

    /**
     * <p>
     * The number of MiB of memory reserved for the job.
     * </p>
     * 
     * @return The number of MiB of memory reserved for the job.
     */
    public Integer memory() {
        return memory;
    }

    /**
     * <p>
     * The command that is passed to the container.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The command that is passed to the container.
     */
    public List<String> command() {
        return command;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) associated with the job upon execution.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) associated with the job upon execution.
     */
    public String jobRoleArn() {
        return jobRoleArn;
    }

    /**
     * <p>
     * A list of volumes associated with the job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of volumes associated with the job.
     */
    public List<Volume> volumes() {
        return volumes;
    }

    /**
     * <p>
     * The environment variables to pass to a container.
     * </p>
     * <note>
     * <p>
     * Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved for
     * variables that are set by the AWS Batch service.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The environment variables to pass to a container.</p> <note>
     *         <p>
     *         Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved for
     *         variables that are set by the AWS Batch service.
     *         </p>
     */
    public List<KeyValuePair> environment() {
        return environment;
    }

    /**
     * <p>
     * The mount points for data volumes in your container.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The mount points for data volumes in your container.
     */
    public List<MountPoint> mountPoints() {
        return mountPoints;
    }

    /**
     * <p>
     * When this parameter is true, the container is given read-only access to its root file system.
     * </p>
     * 
     * @return When this parameter is true, the container is given read-only access to its root file system.
     */
    public Boolean readonlyRootFilesystem() {
        return readonlyRootFilesystem;
    }

    /**
     * <p>
     * A list of <code>ulimit</code> values to set in the container.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of <code>ulimit</code> values to set in the container.
     */
    public List<Ulimit> ulimits() {
        return ulimits;
    }

    /**
     * <p>
     * When this parameter is true, the container is given elevated privileges on the host container instance (similar
     * to the <code>root</code> user).
     * </p>
     * 
     * @return When this parameter is true, the container is given elevated privileges on the host container instance
     *         (similar to the <code>root</code> user).
     */
    public Boolean privileged() {
        return privileged;
    }

    /**
     * <p>
     * The user name to use inside the container.
     * </p>
     * 
     * @return The user name to use inside the container.
     */
    public String user() {
        return user;
    }

    /**
     * <p>
     * The exit code to return upon completion.
     * </p>
     * 
     * @return The exit code to return upon completion.
     */
    public Integer exitCode() {
        return exitCode;
    }

    /**
     * <p>
     * A short (255 max characters) human-readable string to provide additional details about a running or stopped
     * container.
     * </p>
     * 
     * @return A short (255 max characters) human-readable string to provide additional details about a running or
     *         stopped container.
     */
    public String reason() {
        return reason;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the container instance on which the container is running.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the container instance on which the container is running.
     */
    public String containerInstanceArn() {
        return containerInstanceArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the Amazon ECS task that is associated with the container job. Each container
     * attempt receives a task ARN when they reach the <code>STARTING</code> status.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the Amazon ECS task that is associated with the container job. Each
     *         container attempt receives a task ARN when they reach the <code>STARTING</code> status.
     */
    public String taskArn() {
        return taskArn;
    }

    /**
     * <p>
     * The name of the CloudWatch Logs log stream associated with the container. The log group for AWS Batch jobs is
     * <code>/aws/batch/job</code>. Each container attempt receives a log stream name when they reach the
     * <code>RUNNING</code> status.
     * </p>
     * 
     * @return The name of the CloudWatch Logs log stream associated with the container. The log group for AWS Batch
     *         jobs is <code>/aws/batch/job</code>. Each container attempt receives a log stream name when they reach
     *         the <code>RUNNING</code> status.
     */
    public String logStreamName() {
        return logStreamName;
    }

    /**
     * <p>
     * The instance type of the underlying host infrastructure of a multi-node parallel job.
     * </p>
     * 
     * @return The instance type of the underlying host infrastructure of a multi-node parallel job.
     */
    public String instanceType() {
        return instanceType;
    }

    /**
     * <p>
     * The network interfaces associated with the job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The network interfaces associated with the job.
     */
    public List<NetworkInterface> networkInterfaces() {
        return networkInterfaces;
    }

    /**
     * <p>
     * The type and amount of a resource to assign to a container. Currently, the only supported resource is
     * <code>GPU</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The type and amount of a resource to assign to a container. Currently, the only supported resource is
     *         <code>GPU</code>.
     */
    public List<ResourceRequirement> resourceRequirements() {
        return resourceRequirements;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(image());
        hashCode = 31 * hashCode + Objects.hashCode(vcpus());
        hashCode = 31 * hashCode + Objects.hashCode(memory());
        hashCode = 31 * hashCode + Objects.hashCode(command());
        hashCode = 31 * hashCode + Objects.hashCode(jobRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(volumes());
        hashCode = 31 * hashCode + Objects.hashCode(environment());
        hashCode = 31 * hashCode + Objects.hashCode(mountPoints());
        hashCode = 31 * hashCode + Objects.hashCode(readonlyRootFilesystem());
        hashCode = 31 * hashCode + Objects.hashCode(ulimits());
        hashCode = 31 * hashCode + Objects.hashCode(privileged());
        hashCode = 31 * hashCode + Objects.hashCode(user());
        hashCode = 31 * hashCode + Objects.hashCode(exitCode());
        hashCode = 31 * hashCode + Objects.hashCode(reason());
        hashCode = 31 * hashCode + Objects.hashCode(containerInstanceArn());
        hashCode = 31 * hashCode + Objects.hashCode(taskArn());
        hashCode = 31 * hashCode + Objects.hashCode(logStreamName());
        hashCode = 31 * hashCode + Objects.hashCode(instanceType());
        hashCode = 31 * hashCode + Objects.hashCode(networkInterfaces());
        hashCode = 31 * hashCode + Objects.hashCode(resourceRequirements());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ContainerDetail)) {
            return false;
        }
        ContainerDetail other = (ContainerDetail) obj;
        return Objects.equals(image(), other.image()) && Objects.equals(vcpus(), other.vcpus())
                && Objects.equals(memory(), other.memory()) && Objects.equals(command(), other.command())
                && Objects.equals(jobRoleArn(), other.jobRoleArn()) && Objects.equals(volumes(), other.volumes())
                && Objects.equals(environment(), other.environment()) && Objects.equals(mountPoints(), other.mountPoints())
                && Objects.equals(readonlyRootFilesystem(), other.readonlyRootFilesystem())
                && Objects.equals(ulimits(), other.ulimits()) && Objects.equals(privileged(), other.privileged())
                && Objects.equals(user(), other.user()) && Objects.equals(exitCode(), other.exitCode())
                && Objects.equals(reason(), other.reason())
                && Objects.equals(containerInstanceArn(), other.containerInstanceArn())
                && Objects.equals(taskArn(), other.taskArn()) && Objects.equals(logStreamName(), other.logStreamName())
                && Objects.equals(instanceType(), other.instanceType())
                && Objects.equals(networkInterfaces(), other.networkInterfaces())
                && Objects.equals(resourceRequirements(), other.resourceRequirements());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ContainerDetail").add("Image", image()).add("Vcpus", vcpus()).add("Memory", memory())
                .add("Command", command()).add("JobRoleArn", jobRoleArn()).add("Volumes", volumes())
                .add("Environment", environment()).add("MountPoints", mountPoints())
                .add("ReadonlyRootFilesystem", readonlyRootFilesystem()).add("Ulimits", ulimits())
                .add("Privileged", privileged()).add("User", user()).add("ExitCode", exitCode()).add("Reason", reason())
                .add("ContainerInstanceArn", containerInstanceArn()).add("TaskArn", taskArn())
                .add("LogStreamName", logStreamName()).add("InstanceType", instanceType())
                .add("NetworkInterfaces", networkInterfaces()).add("ResourceRequirements", resourceRequirements()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "image":
            return Optional.ofNullable(clazz.cast(image()));
        case "vcpus":
            return Optional.ofNullable(clazz.cast(vcpus()));
        case "memory":
            return Optional.ofNullable(clazz.cast(memory()));
        case "command":
            return Optional.ofNullable(clazz.cast(command()));
        case "jobRoleArn":
            return Optional.ofNullable(clazz.cast(jobRoleArn()));
        case "volumes":
            return Optional.ofNullable(clazz.cast(volumes()));
        case "environment":
            return Optional.ofNullable(clazz.cast(environment()));
        case "mountPoints":
            return Optional.ofNullable(clazz.cast(mountPoints()));
        case "readonlyRootFilesystem":
            return Optional.ofNullable(clazz.cast(readonlyRootFilesystem()));
        case "ulimits":
            return Optional.ofNullable(clazz.cast(ulimits()));
        case "privileged":
            return Optional.ofNullable(clazz.cast(privileged()));
        case "user":
            return Optional.ofNullable(clazz.cast(user()));
        case "exitCode":
            return Optional.ofNullable(clazz.cast(exitCode()));
        case "reason":
            return Optional.ofNullable(clazz.cast(reason()));
        case "containerInstanceArn":
            return Optional.ofNullable(clazz.cast(containerInstanceArn()));
        case "taskArn":
            return Optional.ofNullable(clazz.cast(taskArn()));
        case "logStreamName":
            return Optional.ofNullable(clazz.cast(logStreamName()));
        case "instanceType":
            return Optional.ofNullable(clazz.cast(instanceType()));
        case "networkInterfaces":
            return Optional.ofNullable(clazz.cast(networkInterfaces()));
        case "resourceRequirements":
            return Optional.ofNullable(clazz.cast(resourceRequirements()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ContainerDetail, T> g) {
        return obj -> g.apply((ContainerDetail) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ContainerDetail> {
        /**
         * <p>
         * The image used to start the container.
         * </p>
         * 
         * @param image
         *        The image used to start the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder image(String image);

        /**
         * <p>
         * The number of VCPUs allocated for the job.
         * </p>
         * 
         * @param vcpus
         *        The number of VCPUs allocated for the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vcpus(Integer vcpus);

        /**
         * <p>
         * The number of MiB of memory reserved for the job.
         * </p>
         * 
         * @param memory
         *        The number of MiB of memory reserved for the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder memory(Integer memory);

        /**
         * <p>
         * The command that is passed to the container.
         * </p>
         * 
         * @param command
         *        The command that is passed to the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(Collection<String> command);

        /**
         * <p>
         * The command that is passed to the container.
         * </p>
         * 
         * @param command
         *        The command that is passed to the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(String... command);

        /**
         * <p>
         * The Amazon Resource Name (ARN) associated with the job upon execution.
         * </p>
         * 
         * @param jobRoleArn
         *        The Amazon Resource Name (ARN) associated with the job upon execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobRoleArn(String jobRoleArn);

        /**
         * <p>
         * A list of volumes associated with the job.
         * </p>
         * 
         * @param volumes
         *        A list of volumes associated with the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumes(Collection<Volume> volumes);

        /**
         * <p>
         * A list of volumes associated with the job.
         * </p>
         * 
         * @param volumes
         *        A list of volumes associated with the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumes(Volume... volumes);

        /**
         * <p>
         * A list of volumes associated with the job.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Volume>.Builder} avoiding the need to
         * create one manually via {@link List<Volume>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Volume>.Builder#build()} is called immediately and its
         * result is passed to {@link #volumes(List<Volume>)}.
         * 
         * @param volumes
         *        a consumer that will call methods on {@link List<Volume>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #volumes(List<Volume>)
         */
        Builder volumes(Consumer<Volume.Builder>... volumes);

        /**
         * <p>
         * The environment variables to pass to a container.
         * </p>
         * <note>
         * <p>
         * Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved for
         * variables that are set by the AWS Batch service.
         * </p>
         * </note>
         * 
         * @param environment
         *        The environment variables to pass to a container.</p> <note>
         *        <p>
         *        Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved
         *        for variables that are set by the AWS Batch service.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environment(Collection<KeyValuePair> environment);

        /**
         * <p>
         * The environment variables to pass to a container.
         * </p>
         * <note>
         * <p>
         * Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved for
         * variables that are set by the AWS Batch service.
         * </p>
         * </note>
         * 
         * @param environment
         *        The environment variables to pass to a container.</p> <note>
         *        <p>
         *        Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved
         *        for variables that are set by the AWS Batch service.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environment(KeyValuePair... environment);

        /**
         * <p>
         * The environment variables to pass to a container.
         * </p>
         * <note>
         * <p>
         * Environment variables must not start with <code>AWS_BATCH</code>; this naming convention is reserved for
         * variables that are set by the AWS Batch service.
         * </p>
         * </note> This is a convenience that creates an instance of the {@link List<KeyValuePair>.Builder} avoiding the
         * need to create one manually via {@link List<KeyValuePair>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<KeyValuePair>.Builder#build()} is called immediately and its
         * result is passed to {@link #environment(List<KeyValuePair>)}.
         * 
         * @param environment
         *        a consumer that will call methods on {@link List<KeyValuePair>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #environment(List<KeyValuePair>)
         */
        Builder environment(Consumer<KeyValuePair.Builder>... environment);

        /**
         * <p>
         * The mount points for data volumes in your container.
         * </p>
         * 
         * @param mountPoints
         *        The mount points for data volumes in your container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mountPoints(Collection<MountPoint> mountPoints);

        /**
         * <p>
         * The mount points for data volumes in your container.
         * </p>
         * 
         * @param mountPoints
         *        The mount points for data volumes in your container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mountPoints(MountPoint... mountPoints);

        /**
         * <p>
         * The mount points for data volumes in your container.
         * </p>
         * This is a convenience that creates an instance of the {@link List<MountPoint>.Builder} avoiding the need to
         * create one manually via {@link List<MountPoint>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<MountPoint>.Builder#build()} is called immediately and its
         * result is passed to {@link #mountPoints(List<MountPoint>)}.
         * 
         * @param mountPoints
         *        a consumer that will call methods on {@link List<MountPoint>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #mountPoints(List<MountPoint>)
         */
        Builder mountPoints(Consumer<MountPoint.Builder>... mountPoints);

        /**
         * <p>
         * When this parameter is true, the container is given read-only access to its root file system.
         * </p>
         * 
         * @param readonlyRootFilesystem
         *        When this parameter is true, the container is given read-only access to its root file system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder readonlyRootFilesystem(Boolean readonlyRootFilesystem);

        /**
         * <p>
         * A list of <code>ulimit</code> values to set in the container.
         * </p>
         * 
         * @param ulimits
         *        A list of <code>ulimit</code> values to set in the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ulimits(Collection<Ulimit> ulimits);

        /**
         * <p>
         * A list of <code>ulimit</code> values to set in the container.
         * </p>
         * 
         * @param ulimits
         *        A list of <code>ulimit</code> values to set in the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ulimits(Ulimit... ulimits);

        /**
         * <p>
         * A list of <code>ulimit</code> values to set in the container.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Ulimit>.Builder} avoiding the need to
         * create one manually via {@link List<Ulimit>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Ulimit>.Builder#build()} is called immediately and its
         * result is passed to {@link #ulimits(List<Ulimit>)}.
         * 
         * @param ulimits
         *        a consumer that will call methods on {@link List<Ulimit>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ulimits(List<Ulimit>)
         */
        Builder ulimits(Consumer<Ulimit.Builder>... ulimits);

        /**
         * <p>
         * When this parameter is true, the container is given elevated privileges on the host container instance
         * (similar to the <code>root</code> user).
         * </p>
         * 
         * @param privileged
         *        When this parameter is true, the container is given elevated privileges on the host container instance
         *        (similar to the <code>root</code> user).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder privileged(Boolean privileged);

        /**
         * <p>
         * The user name to use inside the container.
         * </p>
         * 
         * @param user
         *        The user name to use inside the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder user(String user);

        /**
         * <p>
         * The exit code to return upon completion.
         * </p>
         * 
         * @param exitCode
         *        The exit code to return upon completion.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exitCode(Integer exitCode);

        /**
         * <p>
         * A short (255 max characters) human-readable string to provide additional details about a running or stopped
         * container.
         * </p>
         * 
         * @param reason
         *        A short (255 max characters) human-readable string to provide additional details about a running or
         *        stopped container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reason(String reason);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the container instance on which the container is running.
         * </p>
         * 
         * @param containerInstanceArn
         *        The Amazon Resource Name (ARN) of the container instance on which the container is running.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerInstanceArn(String containerInstanceArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the Amazon ECS task that is associated with the container job. Each
         * container attempt receives a task ARN when they reach the <code>STARTING</code> status.
         * </p>
         * 
         * @param taskArn
         *        The Amazon Resource Name (ARN) of the Amazon ECS task that is associated with the container job. Each
         *        container attempt receives a task ARN when they reach the <code>STARTING</code> status.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskArn(String taskArn);

        /**
         * <p>
         * The name of the CloudWatch Logs log stream associated with the container. The log group for AWS Batch jobs is
         * <code>/aws/batch/job</code>. Each container attempt receives a log stream name when they reach the
         * <code>RUNNING</code> status.
         * </p>
         * 
         * @param logStreamName
         *        The name of the CloudWatch Logs log stream associated with the container. The log group for AWS Batch
         *        jobs is <code>/aws/batch/job</code>. Each container attempt receives a log stream name when they reach
         *        the <code>RUNNING</code> status.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logStreamName(String logStreamName);

        /**
         * <p>
         * The instance type of the underlying host infrastructure of a multi-node parallel job.
         * </p>
         * 
         * @param instanceType
         *        The instance type of the underlying host infrastructure of a multi-node parallel job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * The network interfaces associated with the job.
         * </p>
         * 
         * @param networkInterfaces
         *        The network interfaces associated with the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaces(Collection<NetworkInterface> networkInterfaces);

        /**
         * <p>
         * The network interfaces associated with the job.
         * </p>
         * 
         * @param networkInterfaces
         *        The network interfaces associated with the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaces(NetworkInterface... networkInterfaces);

        /**
         * <p>
         * The network interfaces associated with the job.
         * </p>
         * This is a convenience that creates an instance of the {@link List<NetworkInterface>.Builder} avoiding the
         * need to create one manually via {@link List<NetworkInterface>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<NetworkInterface>.Builder#build()} is called immediately and
         * its result is passed to {@link #networkInterfaces(List<NetworkInterface>)}.
         * 
         * @param networkInterfaces
         *        a consumer that will call methods on {@link List<NetworkInterface>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #networkInterfaces(List<NetworkInterface>)
         */
        Builder networkInterfaces(Consumer<NetworkInterface.Builder>... networkInterfaces);

        /**
         * <p>
         * The type and amount of a resource to assign to a container. Currently, the only supported resource is
         * <code>GPU</code>.
         * </p>
         * 
         * @param resourceRequirements
         *        The type and amount of a resource to assign to a container. Currently, the only supported resource is
         *        <code>GPU</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceRequirements(Collection<ResourceRequirement> resourceRequirements);

        /**
         * <p>
         * The type and amount of a resource to assign to a container. Currently, the only supported resource is
         * <code>GPU</code>.
         * </p>
         * 
         * @param resourceRequirements
         *        The type and amount of a resource to assign to a container. Currently, the only supported resource is
         *        <code>GPU</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceRequirements(ResourceRequirement... resourceRequirements);

        /**
         * <p>
         * The type and amount of a resource to assign to a container. Currently, the only supported resource is
         * <code>GPU</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ResourceRequirement>.Builder} avoiding the
         * need to create one manually via {@link List<ResourceRequirement>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ResourceRequirement>.Builder#build()} is called immediately
         * and its result is passed to {@link #resourceRequirements(List<ResourceRequirement>)}.
         * 
         * @param resourceRequirements
         *        a consumer that will call methods on {@link List<ResourceRequirement>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resourceRequirements(List<ResourceRequirement>)
         */
        Builder resourceRequirements(Consumer<ResourceRequirement.Builder>... resourceRequirements);
    }

    static final class BuilderImpl implements Builder {
        private String image;

        private Integer vcpus;

        private Integer memory;

        private List<String> command = DefaultSdkAutoConstructList.getInstance();

        private String jobRoleArn;

        private List<Volume> volumes = DefaultSdkAutoConstructList.getInstance();

        private List<KeyValuePair> environment = DefaultSdkAutoConstructList.getInstance();

        private List<MountPoint> mountPoints = DefaultSdkAutoConstructList.getInstance();

        private Boolean readonlyRootFilesystem;

        private List<Ulimit> ulimits = DefaultSdkAutoConstructList.getInstance();

        private Boolean privileged;

        private String user;

        private Integer exitCode;

        private String reason;

        private String containerInstanceArn;

        private String taskArn;

        private String logStreamName;

        private String instanceType;

        private List<NetworkInterface> networkInterfaces = DefaultSdkAutoConstructList.getInstance();

        private List<ResourceRequirement> resourceRequirements = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ContainerDetail model) {
            image(model.image);
            vcpus(model.vcpus);
            memory(model.memory);
            command(model.command);
            jobRoleArn(model.jobRoleArn);
            volumes(model.volumes);
            environment(model.environment);
            mountPoints(model.mountPoints);
            readonlyRootFilesystem(model.readonlyRootFilesystem);
            ulimits(model.ulimits);
            privileged(model.privileged);
            user(model.user);
            exitCode(model.exitCode);
            reason(model.reason);
            containerInstanceArn(model.containerInstanceArn);
            taskArn(model.taskArn);
            logStreamName(model.logStreamName);
            instanceType(model.instanceType);
            networkInterfaces(model.networkInterfaces);
            resourceRequirements(model.resourceRequirements);
        }

        public final String getImage() {
            return image;
        }

        @Override
        public final Builder image(String image) {
            this.image = image;
            return this;
        }

        public final void setImage(String image) {
            this.image = image;
        }

        public final Integer getVcpus() {
            return vcpus;
        }

        @Override
        public final Builder vcpus(Integer vcpus) {
            this.vcpus = vcpus;
            return this;
        }

        public final void setVcpus(Integer vcpus) {
            this.vcpus = vcpus;
        }

        public final Integer getMemory() {
            return memory;
        }

        @Override
        public final Builder memory(Integer memory) {
            this.memory = memory;
            return this;
        }

        public final void setMemory(Integer memory) {
            this.memory = memory;
        }

        public final Collection<String> getCommand() {
            return command;
        }

        @Override
        public final Builder command(Collection<String> command) {
            this.command = StringListCopier.copy(command);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder command(String... command) {
            command(Arrays.asList(command));
            return this;
        }

        public final void setCommand(Collection<String> command) {
            this.command = StringListCopier.copy(command);
        }

        public final String getJobRoleArn() {
            return jobRoleArn;
        }

        @Override
        public final Builder jobRoleArn(String jobRoleArn) {
            this.jobRoleArn = jobRoleArn;
            return this;
        }

        public final void setJobRoleArn(String jobRoleArn) {
            this.jobRoleArn = jobRoleArn;
        }

        public final Collection<Volume.Builder> getVolumes() {
            return volumes != null ? volumes.stream().map(Volume::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder volumes(Collection<Volume> volumes) {
            this.volumes = VolumesCopier.copy(volumes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder volumes(Volume... volumes) {
            volumes(Arrays.asList(volumes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder volumes(Consumer<Volume.Builder>... volumes) {
            volumes(Stream.of(volumes).map(c -> Volume.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setVolumes(Collection<Volume.BuilderImpl> volumes) {
            this.volumes = VolumesCopier.copyFromBuilder(volumes);
        }

        public final Collection<KeyValuePair.Builder> getEnvironment() {
            return environment != null ? environment.stream().map(KeyValuePair::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder environment(Collection<KeyValuePair> environment) {
            this.environment = EnvironmentVariablesCopier.copy(environment);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder environment(KeyValuePair... environment) {
            environment(Arrays.asList(environment));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder environment(Consumer<KeyValuePair.Builder>... environment) {
            environment(Stream.of(environment).map(c -> KeyValuePair.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setEnvironment(Collection<KeyValuePair.BuilderImpl> environment) {
            this.environment = EnvironmentVariablesCopier.copyFromBuilder(environment);
        }

        public final Collection<MountPoint.Builder> getMountPoints() {
            return mountPoints != null ? mountPoints.stream().map(MountPoint::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder mountPoints(Collection<MountPoint> mountPoints) {
            this.mountPoints = MountPointsCopier.copy(mountPoints);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder mountPoints(MountPoint... mountPoints) {
            mountPoints(Arrays.asList(mountPoints));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder mountPoints(Consumer<MountPoint.Builder>... mountPoints) {
            mountPoints(Stream.of(mountPoints).map(c -> MountPoint.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setMountPoints(Collection<MountPoint.BuilderImpl> mountPoints) {
            this.mountPoints = MountPointsCopier.copyFromBuilder(mountPoints);
        }

        public final Boolean getReadonlyRootFilesystem() {
            return readonlyRootFilesystem;
        }

        @Override
        public final Builder readonlyRootFilesystem(Boolean readonlyRootFilesystem) {
            this.readonlyRootFilesystem = readonlyRootFilesystem;
            return this;
        }

        public final void setReadonlyRootFilesystem(Boolean readonlyRootFilesystem) {
            this.readonlyRootFilesystem = readonlyRootFilesystem;
        }

        public final Collection<Ulimit.Builder> getUlimits() {
            return ulimits != null ? ulimits.stream().map(Ulimit::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder ulimits(Collection<Ulimit> ulimits) {
            this.ulimits = UlimitsCopier.copy(ulimits);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ulimits(Ulimit... ulimits) {
            ulimits(Arrays.asList(ulimits));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ulimits(Consumer<Ulimit.Builder>... ulimits) {
            ulimits(Stream.of(ulimits).map(c -> Ulimit.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setUlimits(Collection<Ulimit.BuilderImpl> ulimits) {
            this.ulimits = UlimitsCopier.copyFromBuilder(ulimits);
        }

        public final Boolean getPrivileged() {
            return privileged;
        }

        @Override
        public final Builder privileged(Boolean privileged) {
            this.privileged = privileged;
            return this;
        }

        public final void setPrivileged(Boolean privileged) {
            this.privileged = privileged;
        }

        public final String getUser() {
            return user;
        }

        @Override
        public final Builder user(String user) {
            this.user = user;
            return this;
        }

        public final void setUser(String user) {
            this.user = user;
        }

        public final Integer getExitCode() {
            return exitCode;
        }

        @Override
        public final Builder exitCode(Integer exitCode) {
            this.exitCode = exitCode;
            return this;
        }

        public final void setExitCode(Integer exitCode) {
            this.exitCode = exitCode;
        }

        public final String getReason() {
            return reason;
        }

        @Override
        public final Builder reason(String reason) {
            this.reason = reason;
            return this;
        }

        public final void setReason(String reason) {
            this.reason = reason;
        }

        public final String getContainerInstanceArn() {
            return containerInstanceArn;
        }

        @Override
        public final Builder containerInstanceArn(String containerInstanceArn) {
            this.containerInstanceArn = containerInstanceArn;
            return this;
        }

        public final void setContainerInstanceArn(String containerInstanceArn) {
            this.containerInstanceArn = containerInstanceArn;
        }

        public final String getTaskArn() {
            return taskArn;
        }

        @Override
        public final Builder taskArn(String taskArn) {
            this.taskArn = taskArn;
            return this;
        }

        public final void setTaskArn(String taskArn) {
            this.taskArn = taskArn;
        }

        public final String getLogStreamName() {
            return logStreamName;
        }

        @Override
        public final Builder logStreamName(String logStreamName) {
            this.logStreamName = logStreamName;
            return this;
        }

        public final void setLogStreamName(String logStreamName) {
            this.logStreamName = logStreamName;
        }

        public final String getInstanceType() {
            return instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        public final Collection<NetworkInterface.Builder> getNetworkInterfaces() {
            return networkInterfaces != null ? networkInterfaces.stream().map(NetworkInterface::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder networkInterfaces(Collection<NetworkInterface> networkInterfaces) {
            this.networkInterfaces = NetworkInterfaceListCopier.copy(networkInterfaces);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder networkInterfaces(NetworkInterface... networkInterfaces) {
            networkInterfaces(Arrays.asList(networkInterfaces));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder networkInterfaces(Consumer<NetworkInterface.Builder>... networkInterfaces) {
            networkInterfaces(Stream.of(networkInterfaces).map(c -> NetworkInterface.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setNetworkInterfaces(Collection<NetworkInterface.BuilderImpl> networkInterfaces) {
            this.networkInterfaces = NetworkInterfaceListCopier.copyFromBuilder(networkInterfaces);
        }

        public final Collection<ResourceRequirement.Builder> getResourceRequirements() {
            return resourceRequirements != null ? resourceRequirements.stream().map(ResourceRequirement::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder resourceRequirements(Collection<ResourceRequirement> resourceRequirements) {
            this.resourceRequirements = ResourceRequirementsCopier.copy(resourceRequirements);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceRequirements(ResourceRequirement... resourceRequirements) {
            resourceRequirements(Arrays.asList(resourceRequirements));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceRequirements(Consumer<ResourceRequirement.Builder>... resourceRequirements) {
            resourceRequirements(Stream.of(resourceRequirements).map(c -> ResourceRequirement.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setResourceRequirements(Collection<ResourceRequirement.BuilderImpl> resourceRequirements) {
            this.resourceRequirements = ResourceRequirementsCopier.copyFromBuilder(resourceRequirements);
        }

        @Override
        public ContainerDetail build() {
            return new ContainerDetail(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
