/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bcmdataexports;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.bcmdataexports.internal.BcmDataExportsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.bcmdataexports.model.BcmDataExportsException;
import software.amazon.awssdk.services.bcmdataexports.model.CreateExportRequest;
import software.amazon.awssdk.services.bcmdataexports.model.CreateExportResponse;
import software.amazon.awssdk.services.bcmdataexports.model.DeleteExportRequest;
import software.amazon.awssdk.services.bcmdataexports.model.DeleteExportResponse;
import software.amazon.awssdk.services.bcmdataexports.model.GetExecutionRequest;
import software.amazon.awssdk.services.bcmdataexports.model.GetExecutionResponse;
import software.amazon.awssdk.services.bcmdataexports.model.GetExportRequest;
import software.amazon.awssdk.services.bcmdataexports.model.GetExportResponse;
import software.amazon.awssdk.services.bcmdataexports.model.GetTableRequest;
import software.amazon.awssdk.services.bcmdataexports.model.GetTableResponse;
import software.amazon.awssdk.services.bcmdataexports.model.InternalServerException;
import software.amazon.awssdk.services.bcmdataexports.model.ListExecutionsRequest;
import software.amazon.awssdk.services.bcmdataexports.model.ListExecutionsResponse;
import software.amazon.awssdk.services.bcmdataexports.model.ListExportsRequest;
import software.amazon.awssdk.services.bcmdataexports.model.ListExportsResponse;
import software.amazon.awssdk.services.bcmdataexports.model.ListTablesRequest;
import software.amazon.awssdk.services.bcmdataexports.model.ListTablesResponse;
import software.amazon.awssdk.services.bcmdataexports.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.bcmdataexports.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.bcmdataexports.model.ResourceNotFoundException;
import software.amazon.awssdk.services.bcmdataexports.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.bcmdataexports.model.TagResourceRequest;
import software.amazon.awssdk.services.bcmdataexports.model.TagResourceResponse;
import software.amazon.awssdk.services.bcmdataexports.model.ThrottlingException;
import software.amazon.awssdk.services.bcmdataexports.model.UntagResourceRequest;
import software.amazon.awssdk.services.bcmdataexports.model.UntagResourceResponse;
import software.amazon.awssdk.services.bcmdataexports.model.UpdateExportRequest;
import software.amazon.awssdk.services.bcmdataexports.model.UpdateExportResponse;
import software.amazon.awssdk.services.bcmdataexports.model.ValidationException;
import software.amazon.awssdk.services.bcmdataexports.transform.CreateExportRequestMarshaller;
import software.amazon.awssdk.services.bcmdataexports.transform.DeleteExportRequestMarshaller;
import software.amazon.awssdk.services.bcmdataexports.transform.GetExecutionRequestMarshaller;
import software.amazon.awssdk.services.bcmdataexports.transform.GetExportRequestMarshaller;
import software.amazon.awssdk.services.bcmdataexports.transform.GetTableRequestMarshaller;
import software.amazon.awssdk.services.bcmdataexports.transform.ListExecutionsRequestMarshaller;
import software.amazon.awssdk.services.bcmdataexports.transform.ListExportsRequestMarshaller;
import software.amazon.awssdk.services.bcmdataexports.transform.ListTablesRequestMarshaller;
import software.amazon.awssdk.services.bcmdataexports.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.bcmdataexports.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.bcmdataexports.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.bcmdataexports.transform.UpdateExportRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link BcmDataExportsClient}.
 *
 * @see BcmDataExportsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultBcmDataExportsClient implements BcmDataExportsClient {
    private static final Logger log = Logger.loggerFor(DefaultBcmDataExportsClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultBcmDataExportsClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a data export and specifies the data query, the delivery preference, and any optional resource tags.
     * </p>
     * <p>
     * A <code>DataQuery</code> consists of both a <code>QueryStatement</code> and <code>TableConfigurations</code>.
     * </p>
     * <p>
     * The <code>QueryStatement</code> is an SQL statement. Data Exports only supports a limited subset of the SQL
     * syntax. For more information on the SQL syntax that is supported, see <a
     * href="https://docs.aws.amazon.com/cur/latest/userguide/de-data-query.html">Data query</a>. To view the available
     * tables and columns, see the <a
     * href="https://docs.aws.amazon.com/cur/latest/userguide/de-table-dictionary.html">Data Exports table
     * dictionary</a>.
     * </p>
     * <p>
     * The <code>TableConfigurations</code> is a collection of specified <code>TableProperties</code> for the table
     * being queried in the <code>QueryStatement</code>. TableProperties are additional configurations you can provide
     * to change the data and schema of a table. Each table can have different TableProperties. However, tables are not
     * required to have any TableProperties. Each table property has a default value that it assumes if not specified.
     * For more information on table configurations, see <a
     * href="https://docs.aws.amazon.com/cur/latest/userguide/de-data-query.html">Data query</a>. To view the table
     * properties available for each table, see the <a
     * href="https://docs.aws.amazon.com/cur/latest/userguide/de-table-dictionary.html">Data Exports table
     * dictionary</a> or use the <code>ListTables</code> API to get a response of all tables and their available
     * properties.
     * </p>
     *
     * @param createExportRequest
     * @return Result of the CreateExport operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         You've reached the limit on the number of resources you can create, or exceeded the size of an individual
     *         resource.
     * @throws InternalServerException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDataExportsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDataExportsClient.CreateExport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-data-exports-2023-11-26/CreateExport" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateExportResponse createExport(CreateExportRequest createExportRequest) throws ServiceQuotaExceededException,
            InternalServerException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            BcmDataExportsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateExportResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateExportResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createExportRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createExportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BCM Data Exports");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateExport");

            return clientHandler.execute(new ClientExecutionParams<CreateExportRequest, CreateExportResponse>()
                    .withOperationName("CreateExport").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createExportRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateExportRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an existing data export.
     * </p>
     *
     * @param deleteExportRequest
     * @return Result of the DeleteExport operation returned by the service.
     * @throws InternalServerException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws ResourceNotFoundException
     *         The specified Amazon Resource Name (ARN) in the request doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDataExportsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDataExportsClient.DeleteExport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-data-exports-2023-11-26/DeleteExport" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteExportResponse deleteExport(DeleteExportRequest deleteExportRequest) throws InternalServerException,
            ResourceNotFoundException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            BcmDataExportsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteExportResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteExportResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteExportRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteExportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BCM Data Exports");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteExport");

            return clientHandler.execute(new ClientExecutionParams<DeleteExportRequest, DeleteExportResponse>()
                    .withOperationName("DeleteExport").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteExportRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteExportRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Exports data based on the source data update.
     * </p>
     *
     * @param getExecutionRequest
     * @return Result of the GetExecution operation returned by the service.
     * @throws InternalServerException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws ResourceNotFoundException
     *         The specified Amazon Resource Name (ARN) in the request doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDataExportsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDataExportsClient.GetExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-data-exports-2023-11-26/GetExecution" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetExecutionResponse getExecution(GetExecutionRequest getExecutionRequest) throws InternalServerException,
            ResourceNotFoundException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            BcmDataExportsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetExecutionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetExecutionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getExecutionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BCM Data Exports");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetExecution");

            return clientHandler.execute(new ClientExecutionParams<GetExecutionRequest, GetExecutionResponse>()
                    .withOperationName("GetExecution").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getExecutionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetExecutionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Views the definition of an existing data export.
     * </p>
     *
     * @param getExportRequest
     * @return Result of the GetExport operation returned by the service.
     * @throws InternalServerException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws ResourceNotFoundException
     *         The specified Amazon Resource Name (ARN) in the request doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDataExportsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDataExportsClient.GetExport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-data-exports-2023-11-26/GetExport" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetExportResponse getExport(GetExportRequest getExportRequest) throws InternalServerException,
            ResourceNotFoundException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            BcmDataExportsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetExportResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetExportResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getExportRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getExportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BCM Data Exports");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetExport");

            return clientHandler.execute(new ClientExecutionParams<GetExportRequest, GetExportResponse>()
                    .withOperationName("GetExport").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getExportRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetExportRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the metadata for the specified table and table properties. This includes the list of columns in the table
     * schema, their data types, and column descriptions.
     * </p>
     *
     * @param getTableRequest
     * @return Result of the GetTable operation returned by the service.
     * @throws InternalServerException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDataExportsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDataExportsClient.GetTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-data-exports-2023-11-26/GetTable" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetTableResponse getTable(GetTableRequest getTableRequest) throws InternalServerException, ThrottlingException,
            ValidationException, AwsServiceException, SdkClientException, BcmDataExportsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetTableResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTableRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BCM Data Exports");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTable");

            return clientHandler.execute(new ClientExecutionParams<GetTableRequest, GetTableResponse>()
                    .withOperationName("GetTable").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getTableRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetTableRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the historical executions for the export.
     * </p>
     *
     * @param listExecutionsRequest
     * @return Result of the ListExecutions operation returned by the service.
     * @throws InternalServerException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws ResourceNotFoundException
     *         The specified Amazon Resource Name (ARN) in the request doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDataExportsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDataExportsClient.ListExecutions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-data-exports-2023-11-26/ListExecutions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListExecutionsResponse listExecutions(ListExecutionsRequest listExecutionsRequest) throws InternalServerException,
            ResourceNotFoundException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            BcmDataExportsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListExecutionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListExecutionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listExecutionsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listExecutionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BCM Data Exports");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListExecutions");

            return clientHandler.execute(new ClientExecutionParams<ListExecutionsRequest, ListExecutionsResponse>()
                    .withOperationName("ListExecutions").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listExecutionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListExecutionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all data export definitions.
     * </p>
     *
     * @param listExportsRequest
     * @return Result of the ListExports operation returned by the service.
     * @throws InternalServerException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDataExportsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDataExportsClient.ListExports
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-data-exports-2023-11-26/ListExports" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListExportsResponse listExports(ListExportsRequest listExportsRequest) throws InternalServerException,
            ThrottlingException, ValidationException, AwsServiceException, SdkClientException, BcmDataExportsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListExportsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListExportsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listExportsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listExportsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BCM Data Exports");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListExports");

            return clientHandler.execute(new ClientExecutionParams<ListExportsRequest, ListExportsResponse>()
                    .withOperationName("ListExports").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listExportsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListExportsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all available tables in data exports.
     * </p>
     *
     * @param listTablesRequest
     * @return Result of the ListTables operation returned by the service.
     * @throws InternalServerException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDataExportsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDataExportsClient.ListTables
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-data-exports-2023-11-26/ListTables" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTablesResponse listTables(ListTablesRequest listTablesRequest) throws InternalServerException,
            ThrottlingException, ValidationException, AwsServiceException, SdkClientException, BcmDataExportsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTablesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTablesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTablesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTablesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BCM Data Exports");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTables");

            return clientHandler.execute(new ClientExecutionParams<ListTablesRequest, ListTablesResponse>()
                    .withOperationName("ListTables").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listTablesRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTablesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List tags associated with an existing data export.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InternalServerException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws ResourceNotFoundException
     *         The specified Amazon Resource Name (ARN) in the request doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDataExportsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDataExportsClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-data-exports-2023-11-26/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InternalServerException, ResourceNotFoundException, ThrottlingException, ValidationException,
            AwsServiceException, SdkClientException, BcmDataExportsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BCM Data Exports");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds tags for an existing data export definition.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InternalServerException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws ResourceNotFoundException
     *         The specified Amazon Resource Name (ARN) in the request doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDataExportsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDataExportsClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-data-exports-2023-11-26/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InternalServerException,
            ResourceNotFoundException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            BcmDataExportsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BCM Data Exports");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes tags associated with an existing data export definition.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InternalServerException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws ResourceNotFoundException
     *         The specified Amazon Resource Name (ARN) in the request doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDataExportsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDataExportsClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-data-exports-2023-11-26/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InternalServerException,
            ResourceNotFoundException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            BcmDataExportsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BCM Data Exports");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an existing data export by overwriting all export parameters. All export parameters must be provided in
     * the UpdateExport request.
     * </p>
     *
     * @param updateExportRequest
     * @return Result of the UpdateExport operation returned by the service.
     * @throws InternalServerException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws ResourceNotFoundException
     *         The specified Amazon Resource Name (ARN) in the request doesn't exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDataExportsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDataExportsClient.UpdateExport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-data-exports-2023-11-26/UpdateExport" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateExportResponse updateExport(UpdateExportRequest updateExportRequest) throws InternalServerException,
            ResourceNotFoundException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            BcmDataExportsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateExportResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateExportResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateExportRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateExportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "BCM Data Exports");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateExport");

            return clientHandler.execute(new ClientExecutionParams<UpdateExportRequest, UpdateExportResponse>()
                    .withOperationName("UpdateExport").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateExportRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateExportRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        BcmDataExportsServiceClientConfigurationBuilder serviceConfigBuilder = new BcmDataExportsServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(BcmDataExportsException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    @Override
    public final BcmDataExportsServiceClientConfiguration serviceClientConfiguration() {
        return new BcmDataExportsServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
