/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrock.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The definition of a custom metric for use in an Amazon Bedrock evaluation job. A custom metric definition includes a
 * metric name, prompt (instructions) and optionally, a rating scale. Your prompt must include a task description and
 * input variables. The required input variables are different for model-as-a-judge and RAG evaluations.
 * </p>
 * <p>
 * For more information about how to define a custom metric in Amazon Bedrock, see <a
 * href="https://docs.aws.amazon.com/bedrock/latest/userguide/model-evaluation-custom-metrics-prompt-formats.html"
 * >Create a prompt for a custom metrics (LLM-as-a-judge model evaluations)</a> and <a
 * href="https://docs.aws.amazon.com/bedrock/latest/userguide/kb-evaluation-custom-metrics-prompt-formats.html">Create a
 * prompt for a custom metrics (RAG evaluations)</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CustomMetricDefinition implements SdkPojo, Serializable,
        ToCopyableBuilder<CustomMetricDefinition.Builder, CustomMetricDefinition> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(CustomMetricDefinition::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> INSTRUCTIONS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("instructions").getter(getter(CustomMetricDefinition::instructions))
            .setter(setter(Builder::instructions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instructions").build()).build();

    private static final SdkField<List<RatingScaleItem>> RATING_SCALE_FIELD = SdkField
            .<List<RatingScaleItem>> builder(MarshallingType.LIST)
            .memberName("ratingScale")
            .getter(getter(CustomMetricDefinition::ratingScale))
            .setter(setter(Builder::ratingScale))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ratingScale").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<RatingScaleItem> builder(MarshallingType.SDK_POJO)
                                            .constructor(RatingScaleItem::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            INSTRUCTIONS_FIELD, RATING_SCALE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String instructions;

    private final List<RatingScaleItem> ratingScale;

    private CustomMetricDefinition(BuilderImpl builder) {
        this.name = builder.name;
        this.instructions = builder.instructions;
        this.ratingScale = builder.ratingScale;
    }

    /**
     * <p>
     * The name for a custom metric. Names must be unique in your Amazon Web Services region.
     * </p>
     * 
     * @return The name for a custom metric. Names must be unique in your Amazon Web Services region.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The prompt for a custom metric that instructs the evaluator model how to rate the model or RAG source under
     * evaluation.
     * </p>
     * 
     * @return The prompt for a custom metric that instructs the evaluator model how to rate the model or RAG source
     *         under evaluation.
     */
    public final String instructions() {
        return instructions;
    }

    /**
     * For responses, this returns true if the service returned a value for the RatingScale property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRatingScale() {
        return ratingScale != null && !(ratingScale instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Defines the rating scale to be used for a custom metric. We recommend that you always define a ratings scale when
     * creating a custom metric. If you don't define a scale, Amazon Bedrock won't be able to visually display the
     * results of the evaluation in the console or calculate average values of numerical scores. For more information on
     * specifying a rating scale, see <a href=
     * "https://docs.aws.amazon.com/bedrock/latest/userguide/model-evaluation-custom-metrics-prompt-formats.html#model-evaluation-custom-metrics-prompt-formats-schema"
     * >Specifying an output schema (rating scale)</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRatingScale} method.
     * </p>
     * 
     * @return Defines the rating scale to be used for a custom metric. We recommend that you always define a ratings
     *         scale when creating a custom metric. If you don't define a scale, Amazon Bedrock won't be able to
     *         visually display the results of the evaluation in the console or calculate average values of numerical
     *         scores. For more information on specifying a rating scale, see <a href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/model-evaluation-custom-metrics-prompt-formats.html#model-evaluation-custom-metrics-prompt-formats-schema"
     *         >Specifying an output schema (rating scale)</a>.
     */
    public final List<RatingScaleItem> ratingScale() {
        return ratingScale;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(instructions());
        hashCode = 31 * hashCode + Objects.hashCode(hasRatingScale() ? ratingScale() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CustomMetricDefinition)) {
            return false;
        }
        CustomMetricDefinition other = (CustomMetricDefinition) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(instructions(), other.instructions())
                && hasRatingScale() == other.hasRatingScale() && Objects.equals(ratingScale(), other.ratingScale());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CustomMetricDefinition").add("Name", name() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Instructions", instructions()).add("RatingScale", hasRatingScale() ? ratingScale() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "instructions":
            return Optional.ofNullable(clazz.cast(instructions()));
        case "ratingScale":
            return Optional.ofNullable(clazz.cast(ratingScale()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("name", NAME_FIELD);
        map.put("instructions", INSTRUCTIONS_FIELD);
        map.put("ratingScale", RATING_SCALE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CustomMetricDefinition, T> g) {
        return obj -> g.apply((CustomMetricDefinition) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CustomMetricDefinition> {
        /**
         * <p>
         * The name for a custom metric. Names must be unique in your Amazon Web Services region.
         * </p>
         * 
         * @param name
         *        The name for a custom metric. Names must be unique in your Amazon Web Services region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The prompt for a custom metric that instructs the evaluator model how to rate the model or RAG source under
         * evaluation.
         * </p>
         * 
         * @param instructions
         *        The prompt for a custom metric that instructs the evaluator model how to rate the model or RAG source
         *        under evaluation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instructions(String instructions);

        /**
         * <p>
         * Defines the rating scale to be used for a custom metric. We recommend that you always define a ratings scale
         * when creating a custom metric. If you don't define a scale, Amazon Bedrock won't be able to visually display
         * the results of the evaluation in the console or calculate average values of numerical scores. For more
         * information on specifying a rating scale, see <a href=
         * "https://docs.aws.amazon.com/bedrock/latest/userguide/model-evaluation-custom-metrics-prompt-formats.html#model-evaluation-custom-metrics-prompt-formats-schema"
         * >Specifying an output schema (rating scale)</a>.
         * </p>
         * 
         * @param ratingScale
         *        Defines the rating scale to be used for a custom metric. We recommend that you always define a ratings
         *        scale when creating a custom metric. If you don't define a scale, Amazon Bedrock won't be able to
         *        visually display the results of the evaluation in the console or calculate average values of numerical
         *        scores. For more information on specifying a rating scale, see <a href=
         *        "https://docs.aws.amazon.com/bedrock/latest/userguide/model-evaluation-custom-metrics-prompt-formats.html#model-evaluation-custom-metrics-prompt-formats-schema"
         *        >Specifying an output schema (rating scale)</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ratingScale(Collection<RatingScaleItem> ratingScale);

        /**
         * <p>
         * Defines the rating scale to be used for a custom metric. We recommend that you always define a ratings scale
         * when creating a custom metric. If you don't define a scale, Amazon Bedrock won't be able to visually display
         * the results of the evaluation in the console or calculate average values of numerical scores. For more
         * information on specifying a rating scale, see <a href=
         * "https://docs.aws.amazon.com/bedrock/latest/userguide/model-evaluation-custom-metrics-prompt-formats.html#model-evaluation-custom-metrics-prompt-formats-schema"
         * >Specifying an output schema (rating scale)</a>.
         * </p>
         * 
         * @param ratingScale
         *        Defines the rating scale to be used for a custom metric. We recommend that you always define a ratings
         *        scale when creating a custom metric. If you don't define a scale, Amazon Bedrock won't be able to
         *        visually display the results of the evaluation in the console or calculate average values of numerical
         *        scores. For more information on specifying a rating scale, see <a href=
         *        "https://docs.aws.amazon.com/bedrock/latest/userguide/model-evaluation-custom-metrics-prompt-formats.html#model-evaluation-custom-metrics-prompt-formats-schema"
         *        >Specifying an output schema (rating scale)</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ratingScale(RatingScaleItem... ratingScale);

        /**
         * <p>
         * Defines the rating scale to be used for a custom metric. We recommend that you always define a ratings scale
         * when creating a custom metric. If you don't define a scale, Amazon Bedrock won't be able to visually display
         * the results of the evaluation in the console or calculate average values of numerical scores. For more
         * information on specifying a rating scale, see <a href=
         * "https://docs.aws.amazon.com/bedrock/latest/userguide/model-evaluation-custom-metrics-prompt-formats.html#model-evaluation-custom-metrics-prompt-formats-schema"
         * >Specifying an output schema (rating scale)</a>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.bedrock.model.RatingScaleItem.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.bedrock.model.RatingScaleItem#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.bedrock.model.RatingScaleItem.Builder#build()} is called immediately
         * and its result is passed to {@link #ratingScale(List<RatingScaleItem>)}.
         * 
         * @param ratingScale
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.bedrock.model.RatingScaleItem.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ratingScale(java.util.Collection<RatingScaleItem>)
         */
        Builder ratingScale(Consumer<RatingScaleItem.Builder>... ratingScale);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String instructions;

        private List<RatingScaleItem> ratingScale = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CustomMetricDefinition model) {
            name(model.name);
            instructions(model.instructions);
            ratingScale(model.ratingScale);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getInstructions() {
            return instructions;
        }

        public final void setInstructions(String instructions) {
            this.instructions = instructions;
        }

        @Override
        public final Builder instructions(String instructions) {
            this.instructions = instructions;
            return this;
        }

        public final List<RatingScaleItem.Builder> getRatingScale() {
            List<RatingScaleItem.Builder> result = RatingScaleCopier.copyToBuilder(this.ratingScale);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRatingScale(Collection<RatingScaleItem.BuilderImpl> ratingScale) {
            this.ratingScale = RatingScaleCopier.copyFromBuilder(ratingScale);
        }

        @Override
        public final Builder ratingScale(Collection<RatingScaleItem> ratingScale) {
            this.ratingScale = RatingScaleCopier.copy(ratingScale);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ratingScale(RatingScaleItem... ratingScale) {
            ratingScale(Arrays.asList(ratingScale));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ratingScale(Consumer<RatingScaleItem.Builder>... ratingScale) {
            ratingScale(Stream.of(ratingScale).map(c -> RatingScaleItem.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public CustomMetricDefinition build() {
            return new CustomMetricDefinition(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
