/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrock.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration details for text generation using a language model via the <code>RetrieveAndGenerate</code>
 * function.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TextInferenceConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<TextInferenceConfig.Builder, TextInferenceConfig> {
    private static final SdkField<Float> TEMPERATURE_FIELD = SdkField.<Float> builder(MarshallingType.FLOAT)
            .memberName("temperature").getter(getter(TextInferenceConfig::temperature)).setter(setter(Builder::temperature))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("temperature").build()).build();

    private static final SdkField<Float> TOPP_FIELD = SdkField.<Float> builder(MarshallingType.FLOAT).memberName("topP")
            .getter(getter(TextInferenceConfig::topP)).setter(setter(Builder::topP))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("topP").build()).build();

    private static final SdkField<Integer> MAX_TOKENS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("maxTokens").getter(getter(TextInferenceConfig::maxTokens)).setter(setter(Builder::maxTokens))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maxTokens").build()).build();

    private static final SdkField<List<String>> STOP_SEQUENCES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("stopSequences")
            .getter(getter(TextInferenceConfig::stopSequences))
            .setter(setter(Builder::stopSequences))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("stopSequences").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TEMPERATURE_FIELD, TOPP_FIELD,
            MAX_TOKENS_FIELD, STOP_SEQUENCES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Float temperature;

    private final Float topP;

    private final Integer maxTokens;

    private final List<String> stopSequences;

    private TextInferenceConfig(BuilderImpl builder) {
        this.temperature = builder.temperature;
        this.topP = builder.topP;
        this.maxTokens = builder.maxTokens;
        this.stopSequences = builder.stopSequences;
    }

    /**
     * <p>
     * Controls the random-ness of text generated by the language model, influencing how much the model sticks to the
     * most predictable next words versus exploring more surprising options. A lower temperature value (e.g. 0.2 or 0.3)
     * makes model outputs more deterministic or predictable, while a higher temperature (e.g. 0.8 or 0.9) makes the
     * outputs more creative or unpredictable.
     * </p>
     * 
     * @return Controls the random-ness of text generated by the language model, influencing how much the model sticks
     *         to the most predictable next words versus exploring more surprising options. A lower temperature value
     *         (e.g. 0.2 or 0.3) makes model outputs more deterministic or predictable, while a higher temperature (e.g.
     *         0.8 or 0.9) makes the outputs more creative or unpredictable.
     */
    public final Float temperature() {
        return temperature;
    }

    /**
     * <p>
     * A probability distribution threshold which controls what the model considers for the set of possible next tokens.
     * The model will only consider the top p% of the probability distribution when generating the next token.
     * </p>
     * 
     * @return A probability distribution threshold which controls what the model considers for the set of possible next
     *         tokens. The model will only consider the top p% of the probability distribution when generating the next
     *         token.
     */
    public final Float topP() {
        return topP;
    }

    /**
     * <p>
     * The maximum number of tokens to generate in the output text. Do not use the minimum of 0 or the maximum of 65536.
     * The limit values described here are arbitrary values, for actual values consult the limits defined by your
     * specific model.
     * </p>
     * 
     * @return The maximum number of tokens to generate in the output text. Do not use the minimum of 0 or the maximum
     *         of 65536. The limit values described here are arbitrary values, for actual values consult the limits
     *         defined by your specific model.
     */
    public final Integer maxTokens() {
        return maxTokens;
    }

    /**
     * For responses, this returns true if the service returned a value for the StopSequences property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasStopSequences() {
        return stopSequences != null && !(stopSequences instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of sequences of characters that, if generated, will cause the model to stop generating further tokens. Do
     * not use a minimum length of 1 or a maximum length of 1000. The limit values described here are arbitrary values,
     * for actual values consult the limits defined by your specific model.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStopSequences} method.
     * </p>
     * 
     * @return A list of sequences of characters that, if generated, will cause the model to stop generating further
     *         tokens. Do not use a minimum length of 1 or a maximum length of 1000. The limit values described here are
     *         arbitrary values, for actual values consult the limits defined by your specific model.
     */
    public final List<String> stopSequences() {
        return stopSequences;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(temperature());
        hashCode = 31 * hashCode + Objects.hashCode(topP());
        hashCode = 31 * hashCode + Objects.hashCode(maxTokens());
        hashCode = 31 * hashCode + Objects.hashCode(hasStopSequences() ? stopSequences() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TextInferenceConfig)) {
            return false;
        }
        TextInferenceConfig other = (TextInferenceConfig) obj;
        return Objects.equals(temperature(), other.temperature()) && Objects.equals(topP(), other.topP())
                && Objects.equals(maxTokens(), other.maxTokens()) && hasStopSequences() == other.hasStopSequences()
                && Objects.equals(stopSequences(), other.stopSequences());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TextInferenceConfig").add("Temperature", temperature()).add("TopP", topP())
                .add("MaxTokens", maxTokens()).add("StopSequences", hasStopSequences() ? stopSequences() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "temperature":
            return Optional.ofNullable(clazz.cast(temperature()));
        case "topP":
            return Optional.ofNullable(clazz.cast(topP()));
        case "maxTokens":
            return Optional.ofNullable(clazz.cast(maxTokens()));
        case "stopSequences":
            return Optional.ofNullable(clazz.cast(stopSequences()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("temperature", TEMPERATURE_FIELD);
        map.put("topP", TOPP_FIELD);
        map.put("maxTokens", MAX_TOKENS_FIELD);
        map.put("stopSequences", STOP_SEQUENCES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<TextInferenceConfig, T> g) {
        return obj -> g.apply((TextInferenceConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TextInferenceConfig> {
        /**
         * <p>
         * Controls the random-ness of text generated by the language model, influencing how much the model sticks to
         * the most predictable next words versus exploring more surprising options. A lower temperature value (e.g. 0.2
         * or 0.3) makes model outputs more deterministic or predictable, while a higher temperature (e.g. 0.8 or 0.9)
         * makes the outputs more creative or unpredictable.
         * </p>
         * 
         * @param temperature
         *        Controls the random-ness of text generated by the language model, influencing how much the model
         *        sticks to the most predictable next words versus exploring more surprising options. A lower
         *        temperature value (e.g. 0.2 or 0.3) makes model outputs more deterministic or predictable, while a
         *        higher temperature (e.g. 0.8 or 0.9) makes the outputs more creative or unpredictable.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder temperature(Float temperature);

        /**
         * <p>
         * A probability distribution threshold which controls what the model considers for the set of possible next
         * tokens. The model will only consider the top p% of the probability distribution when generating the next
         * token.
         * </p>
         * 
         * @param topP
         *        A probability distribution threshold which controls what the model considers for the set of possible
         *        next tokens. The model will only consider the top p% of the probability distribution when generating
         *        the next token.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder topP(Float topP);

        /**
         * <p>
         * The maximum number of tokens to generate in the output text. Do not use the minimum of 0 or the maximum of
         * 65536. The limit values described here are arbitrary values, for actual values consult the limits defined by
         * your specific model.
         * </p>
         * 
         * @param maxTokens
         *        The maximum number of tokens to generate in the output text. Do not use the minimum of 0 or the
         *        maximum of 65536. The limit values described here are arbitrary values, for actual values consult the
         *        limits defined by your specific model.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxTokens(Integer maxTokens);

        /**
         * <p>
         * A list of sequences of characters that, if generated, will cause the model to stop generating further tokens.
         * Do not use a minimum length of 1 or a maximum length of 1000. The limit values described here are arbitrary
         * values, for actual values consult the limits defined by your specific model.
         * </p>
         * 
         * @param stopSequences
         *        A list of sequences of characters that, if generated, will cause the model to stop generating further
         *        tokens. Do not use a minimum length of 1 or a maximum length of 1000. The limit values described here
         *        are arbitrary values, for actual values consult the limits defined by your specific model.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stopSequences(Collection<String> stopSequences);

        /**
         * <p>
         * A list of sequences of characters that, if generated, will cause the model to stop generating further tokens.
         * Do not use a minimum length of 1 or a maximum length of 1000. The limit values described here are arbitrary
         * values, for actual values consult the limits defined by your specific model.
         * </p>
         * 
         * @param stopSequences
         *        A list of sequences of characters that, if generated, will cause the model to stop generating further
         *        tokens. Do not use a minimum length of 1 or a maximum length of 1000. The limit values described here
         *        are arbitrary values, for actual values consult the limits defined by your specific model.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stopSequences(String... stopSequences);
    }

    static final class BuilderImpl implements Builder {
        private Float temperature;

        private Float topP;

        private Integer maxTokens;

        private List<String> stopSequences = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(TextInferenceConfig model) {
            temperature(model.temperature);
            topP(model.topP);
            maxTokens(model.maxTokens);
            stopSequences(model.stopSequences);
        }

        public final Float getTemperature() {
            return temperature;
        }

        public final void setTemperature(Float temperature) {
            this.temperature = temperature;
        }

        @Override
        public final Builder temperature(Float temperature) {
            this.temperature = temperature;
            return this;
        }

        public final Float getTopP() {
            return topP;
        }

        public final void setTopP(Float topP) {
            this.topP = topP;
        }

        @Override
        public final Builder topP(Float topP) {
            this.topP = topP;
            return this;
        }

        public final Integer getMaxTokens() {
            return maxTokens;
        }

        public final void setMaxTokens(Integer maxTokens) {
            this.maxTokens = maxTokens;
        }

        @Override
        public final Builder maxTokens(Integer maxTokens) {
            this.maxTokens = maxTokens;
            return this;
        }

        public final Collection<String> getStopSequences() {
            if (stopSequences instanceof SdkAutoConstructList) {
                return null;
            }
            return stopSequences;
        }

        public final void setStopSequences(Collection<String> stopSequences) {
            this.stopSequences = RAGStopSequencesCopier.copy(stopSequences);
        }

        @Override
        public final Builder stopSequences(Collection<String> stopSequences) {
            this.stopSequences = RAGStopSequencesCopier.copy(stopSequences);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder stopSequences(String... stopSequences) {
            stopSequences(Arrays.asList(stopSequences));
            return this;
        }

        @Override
        public TextInferenceConfig build() {
            return new TextInferenceConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
