/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrock.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a test scenario used to validate an Automated Reasoning policy, including the test conditions and expected
 * outcomes.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AutomatedReasoningPolicyScenario implements SdkPojo, Serializable,
        ToCopyableBuilder<AutomatedReasoningPolicyScenario.Builder, AutomatedReasoningPolicyScenario> {
    private static final SdkField<String> EXPRESSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("expression").getter(getter(AutomatedReasoningPolicyScenario::expression))
            .setter(setter(Builder::expression))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("expression").build()).build();

    private static final SdkField<String> ALTERNATE_EXPRESSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("alternateExpression").getter(getter(AutomatedReasoningPolicyScenario::alternateExpression))
            .setter(setter(Builder::alternateExpression))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("alternateExpression").build())
            .build();

    private static final SdkField<List<String>> RULE_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ruleIds")
            .getter(getter(AutomatedReasoningPolicyScenario::ruleIds))
            .setter(setter(Builder::ruleIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ruleIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> EXPECTED_RESULT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("expectedResult").getter(getter(AutomatedReasoningPolicyScenario::expectedResultAsString))
            .setter(setter(Builder::expectedResult))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("expectedResult").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EXPRESSION_FIELD,
            ALTERNATE_EXPRESSION_FIELD, RULE_IDS_FIELD, EXPECTED_RESULT_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String expression;

    private final String alternateExpression;

    private final List<String> ruleIds;

    private final String expectedResult;

    private AutomatedReasoningPolicyScenario(BuilderImpl builder) {
        this.expression = builder.expression;
        this.alternateExpression = builder.alternateExpression;
        this.ruleIds = builder.ruleIds;
        this.expectedResult = builder.expectedResult;
    }

    /**
     * <p>
     * The logical expression or condition that defines this test scenario.
     * </p>
     * 
     * @return The logical expression or condition that defines this test scenario.
     */
    public final String expression() {
        return expression;
    }

    /**
     * <p>
     * An alternative way to express the same test scenario, used for validation and comparison purposes.
     * </p>
     * 
     * @return An alternative way to express the same test scenario, used for validation and comparison purposes.
     */
    public final String alternateExpression() {
        return alternateExpression;
    }

    /**
     * For responses, this returns true if the service returned a value for the RuleIds property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasRuleIds() {
        return ruleIds != null && !(ruleIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of rule identifiers that are expected to be triggered or evaluated by this test scenario.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRuleIds} method.
     * </p>
     * 
     * @return The list of rule identifiers that are expected to be triggered or evaluated by this test scenario.
     */
    public final List<String> ruleIds() {
        return ruleIds;
    }

    /**
     * <p>
     * The expected outcome when this scenario is evaluated against the policy (e.g., PASS, FAIL, VIOLATION).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #expectedResult}
     * will return {@link AutomatedReasoningCheckResult#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service
     * is available from {@link #expectedResultAsString}.
     * </p>
     * 
     * @return The expected outcome when this scenario is evaluated against the policy (e.g., PASS, FAIL, VIOLATION).
     * @see AutomatedReasoningCheckResult
     */
    public final AutomatedReasoningCheckResult expectedResult() {
        return AutomatedReasoningCheckResult.fromValue(expectedResult);
    }

    /**
     * <p>
     * The expected outcome when this scenario is evaluated against the policy (e.g., PASS, FAIL, VIOLATION).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #expectedResult}
     * will return {@link AutomatedReasoningCheckResult#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service
     * is available from {@link #expectedResultAsString}.
     * </p>
     * 
     * @return The expected outcome when this scenario is evaluated against the policy (e.g., PASS, FAIL, VIOLATION).
     * @see AutomatedReasoningCheckResult
     */
    public final String expectedResultAsString() {
        return expectedResult;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(expression());
        hashCode = 31 * hashCode + Objects.hashCode(alternateExpression());
        hashCode = 31 * hashCode + Objects.hashCode(hasRuleIds() ? ruleIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(expectedResultAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AutomatedReasoningPolicyScenario)) {
            return false;
        }
        AutomatedReasoningPolicyScenario other = (AutomatedReasoningPolicyScenario) obj;
        return Objects.equals(expression(), other.expression())
                && Objects.equals(alternateExpression(), other.alternateExpression()) && hasRuleIds() == other.hasRuleIds()
                && Objects.equals(ruleIds(), other.ruleIds())
                && Objects.equals(expectedResultAsString(), other.expectedResultAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AutomatedReasoningPolicyScenario")
                .add("Expression", expression() == null ? null : "*** Sensitive Data Redacted ***")
                .add("AlternateExpression", alternateExpression() == null ? null : "*** Sensitive Data Redacted ***")
                .add("RuleIds", hasRuleIds() ? ruleIds() : null).add("ExpectedResult", expectedResultAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "expression":
            return Optional.ofNullable(clazz.cast(expression()));
        case "alternateExpression":
            return Optional.ofNullable(clazz.cast(alternateExpression()));
        case "ruleIds":
            return Optional.ofNullable(clazz.cast(ruleIds()));
        case "expectedResult":
            return Optional.ofNullable(clazz.cast(expectedResultAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("expression", EXPRESSION_FIELD);
        map.put("alternateExpression", ALTERNATE_EXPRESSION_FIELD);
        map.put("ruleIds", RULE_IDS_FIELD);
        map.put("expectedResult", EXPECTED_RESULT_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AutomatedReasoningPolicyScenario, T> g) {
        return obj -> g.apply((AutomatedReasoningPolicyScenario) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AutomatedReasoningPolicyScenario> {
        /**
         * <p>
         * The logical expression or condition that defines this test scenario.
         * </p>
         * 
         * @param expression
         *        The logical expression or condition that defines this test scenario.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expression(String expression);

        /**
         * <p>
         * An alternative way to express the same test scenario, used for validation and comparison purposes.
         * </p>
         * 
         * @param alternateExpression
         *        An alternative way to express the same test scenario, used for validation and comparison purposes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alternateExpression(String alternateExpression);

        /**
         * <p>
         * The list of rule identifiers that are expected to be triggered or evaluated by this test scenario.
         * </p>
         * 
         * @param ruleIds
         *        The list of rule identifiers that are expected to be triggered or evaluated by this test scenario.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleIds(Collection<String> ruleIds);

        /**
         * <p>
         * The list of rule identifiers that are expected to be triggered or evaluated by this test scenario.
         * </p>
         * 
         * @param ruleIds
         *        The list of rule identifiers that are expected to be triggered or evaluated by this test scenario.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleIds(String... ruleIds);

        /**
         * <p>
         * The expected outcome when this scenario is evaluated against the policy (e.g., PASS, FAIL, VIOLATION).
         * </p>
         * 
         * @param expectedResult
         *        The expected outcome when this scenario is evaluated against the policy (e.g., PASS, FAIL, VIOLATION).
         * @see AutomatedReasoningCheckResult
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AutomatedReasoningCheckResult
         */
        Builder expectedResult(String expectedResult);

        /**
         * <p>
         * The expected outcome when this scenario is evaluated against the policy (e.g., PASS, FAIL, VIOLATION).
         * </p>
         * 
         * @param expectedResult
         *        The expected outcome when this scenario is evaluated against the policy (e.g., PASS, FAIL, VIOLATION).
         * @see AutomatedReasoningCheckResult
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AutomatedReasoningCheckResult
         */
        Builder expectedResult(AutomatedReasoningCheckResult expectedResult);
    }

    static final class BuilderImpl implements Builder {
        private String expression;

        private String alternateExpression;

        private List<String> ruleIds = DefaultSdkAutoConstructList.getInstance();

        private String expectedResult;

        private BuilderImpl() {
        }

        private BuilderImpl(AutomatedReasoningPolicyScenario model) {
            expression(model.expression);
            alternateExpression(model.alternateExpression);
            ruleIds(model.ruleIds);
            expectedResult(model.expectedResult);
        }

        public final String getExpression() {
            return expression;
        }

        public final void setExpression(String expression) {
            this.expression = expression;
        }

        @Override
        public final Builder expression(String expression) {
            this.expression = expression;
            return this;
        }

        public final String getAlternateExpression() {
            return alternateExpression;
        }

        public final void setAlternateExpression(String alternateExpression) {
            this.alternateExpression = alternateExpression;
        }

        @Override
        public final Builder alternateExpression(String alternateExpression) {
            this.alternateExpression = alternateExpression;
            return this;
        }

        public final Collection<String> getRuleIds() {
            if (ruleIds instanceof SdkAutoConstructList) {
                return null;
            }
            return ruleIds;
        }

        public final void setRuleIds(Collection<String> ruleIds) {
            this.ruleIds = AutomatedReasoningPolicyDefinitionRuleIdListCopier.copy(ruleIds);
        }

        @Override
        public final Builder ruleIds(Collection<String> ruleIds) {
            this.ruleIds = AutomatedReasoningPolicyDefinitionRuleIdListCopier.copy(ruleIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ruleIds(String... ruleIds) {
            ruleIds(Arrays.asList(ruleIds));
            return this;
        }

        public final String getExpectedResult() {
            return expectedResult;
        }

        public final void setExpectedResult(String expectedResult) {
            this.expectedResult = expectedResult;
        }

        @Override
        public final Builder expectedResult(String expectedResult) {
            this.expectedResult = expectedResult;
            return this;
        }

        @Override
        public final Builder expectedResult(AutomatedReasoningCheckResult expectedResult) {
            this.expectedResult(expectedResult == null ? null : expectedResult.toString());
            return this;
        }

        @Override
        public AutomatedReasoningPolicyScenario build() {
            return new AutomatedReasoningPolicyScenario(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
