/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrock.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains filter strengths for harmful content. Guardrails support the following content filters to detect and filter
 * harmful user inputs and FM-generated outputs.
 * </p>
 * <ul>
 * <li>
 * <p>
 * <b>Hate</b> – Describes language or a statement that discriminates, criticizes, insults, denounces, or dehumanizes a
 * person or group on the basis of an identity (such as race, ethnicity, gender, religion, sexual orientation, ability,
 * and national origin).
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Insults</b> – Describes language or a statement that includes demeaning, humiliating, mocking, insulting, or
 * belittling language. This type of language is also labeled as bullying.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Sexual</b> – Describes language or a statement that indicates sexual interest, activity, or arousal using direct
 * or indirect references to body parts, physical traits, or sex.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Violence</b> – Describes language or a statement that includes glorification of or threats to inflict physical
 * pain, hurt, or injury toward a person, group or thing.
 * </p>
 * </li>
 * </ul>
 * <p>
 * Content filtering depends on the confidence classification of user inputs and FM responses across each of the four
 * harmful categories. All input and output statements are classified into one of four confidence levels (NONE, LOW,
 * MEDIUM, HIGH) for each harmful category. For example, if a statement is classified as <i>Hate</i> with HIGH
 * confidence, the likelihood of the statement representing hateful content is high. A single statement can be
 * classified across multiple categories with varying confidence levels. For example, a single statement can be
 * classified as <i>Hate</i> with HIGH confidence, <i>Insults</i> with LOW confidence, <i>Sexual</i> with NONE
 * confidence, and <i>Violence</i> with MEDIUM confidence.
 * </p>
 * <p>
 * For more information, see <a
 * href="https://docs.aws.amazon.com/bedrock/latest/userguide/guardrails-filters.html">Guardrails content filters</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GuardrailContentFilterConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<GuardrailContentFilterConfig.Builder, GuardrailContentFilterConfig> {
    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(GuardrailContentFilterConfig::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<String> INPUT_STRENGTH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("inputStrength").getter(getter(GuardrailContentFilterConfig::inputStrengthAsString))
            .setter(setter(Builder::inputStrength))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("inputStrength").build()).build();

    private static final SdkField<String> OUTPUT_STRENGTH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("outputStrength").getter(getter(GuardrailContentFilterConfig::outputStrengthAsString))
            .setter(setter(Builder::outputStrength))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("outputStrength").build()).build();

    private static final SdkField<List<String>> INPUT_MODALITIES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("inputModalities")
            .getter(getter(GuardrailContentFilterConfig::inputModalitiesAsStrings))
            .setter(setter(Builder::inputModalitiesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("inputModalities").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> OUTPUT_MODALITIES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("outputModalities")
            .getter(getter(GuardrailContentFilterConfig::outputModalitiesAsStrings))
            .setter(setter(Builder::outputModalitiesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("outputModalities").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> INPUT_ACTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("inputAction").getter(getter(GuardrailContentFilterConfig::inputActionAsString))
            .setter(setter(Builder::inputAction))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("inputAction").build()).build();

    private static final SdkField<String> OUTPUT_ACTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("outputAction").getter(getter(GuardrailContentFilterConfig::outputActionAsString))
            .setter(setter(Builder::outputAction))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("outputAction").build()).build();

    private static final SdkField<Boolean> INPUT_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("inputEnabled").getter(getter(GuardrailContentFilterConfig::inputEnabled))
            .setter(setter(Builder::inputEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("inputEnabled").build()).build();

    private static final SdkField<Boolean> OUTPUT_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("outputEnabled").getter(getter(GuardrailContentFilterConfig::outputEnabled))
            .setter(setter(Builder::outputEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("outputEnabled").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_FIELD,
            INPUT_STRENGTH_FIELD, OUTPUT_STRENGTH_FIELD, INPUT_MODALITIES_FIELD, OUTPUT_MODALITIES_FIELD, INPUT_ACTION_FIELD,
            OUTPUT_ACTION_FIELD, INPUT_ENABLED_FIELD, OUTPUT_ENABLED_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String type;

    private final String inputStrength;

    private final String outputStrength;

    private final List<String> inputModalities;

    private final List<String> outputModalities;

    private final String inputAction;

    private final String outputAction;

    private final Boolean inputEnabled;

    private final Boolean outputEnabled;

    private GuardrailContentFilterConfig(BuilderImpl builder) {
        this.type = builder.type;
        this.inputStrength = builder.inputStrength;
        this.outputStrength = builder.outputStrength;
        this.inputModalities = builder.inputModalities;
        this.outputModalities = builder.outputModalities;
        this.inputAction = builder.inputAction;
        this.outputAction = builder.outputAction;
        this.inputEnabled = builder.inputEnabled;
        this.outputEnabled = builder.outputEnabled;
    }

    /**
     * <p>
     * The harmful category that the content filter is applied to.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link GuardrailContentFilterType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #typeAsString}.
     * </p>
     * 
     * @return The harmful category that the content filter is applied to.
     * @see GuardrailContentFilterType
     */
    public final GuardrailContentFilterType type() {
        return GuardrailContentFilterType.fromValue(type);
    }

    /**
     * <p>
     * The harmful category that the content filter is applied to.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link GuardrailContentFilterType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #typeAsString}.
     * </p>
     * 
     * @return The harmful category that the content filter is applied to.
     * @see GuardrailContentFilterType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The strength of the content filter to apply to prompts. As you increase the filter strength, the likelihood of
     * filtering harmful content increases and the probability of seeing harmful content in your application reduces.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #inputStrength}
     * will return {@link GuardrailFilterStrength#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #inputStrengthAsString}.
     * </p>
     * 
     * @return The strength of the content filter to apply to prompts. As you increase the filter strength, the
     *         likelihood of filtering harmful content increases and the probability of seeing harmful content in your
     *         application reduces.
     * @see GuardrailFilterStrength
     */
    public final GuardrailFilterStrength inputStrength() {
        return GuardrailFilterStrength.fromValue(inputStrength);
    }

    /**
     * <p>
     * The strength of the content filter to apply to prompts. As you increase the filter strength, the likelihood of
     * filtering harmful content increases and the probability of seeing harmful content in your application reduces.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #inputStrength}
     * will return {@link GuardrailFilterStrength#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #inputStrengthAsString}.
     * </p>
     * 
     * @return The strength of the content filter to apply to prompts. As you increase the filter strength, the
     *         likelihood of filtering harmful content increases and the probability of seeing harmful content in your
     *         application reduces.
     * @see GuardrailFilterStrength
     */
    public final String inputStrengthAsString() {
        return inputStrength;
    }

    /**
     * <p>
     * The strength of the content filter to apply to model responses. As you increase the filter strength, the
     * likelihood of filtering harmful content increases and the probability of seeing harmful content in your
     * application reduces.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #outputStrength}
     * will return {@link GuardrailFilterStrength#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #outputStrengthAsString}.
     * </p>
     * 
     * @return The strength of the content filter to apply to model responses. As you increase the filter strength, the
     *         likelihood of filtering harmful content increases and the probability of seeing harmful content in your
     *         application reduces.
     * @see GuardrailFilterStrength
     */
    public final GuardrailFilterStrength outputStrength() {
        return GuardrailFilterStrength.fromValue(outputStrength);
    }

    /**
     * <p>
     * The strength of the content filter to apply to model responses. As you increase the filter strength, the
     * likelihood of filtering harmful content increases and the probability of seeing harmful content in your
     * application reduces.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #outputStrength}
     * will return {@link GuardrailFilterStrength#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #outputStrengthAsString}.
     * </p>
     * 
     * @return The strength of the content filter to apply to model responses. As you increase the filter strength, the
     *         likelihood of filtering harmful content increases and the probability of seeing harmful content in your
     *         application reduces.
     * @see GuardrailFilterStrength
     */
    public final String outputStrengthAsString() {
        return outputStrength;
    }

    /**
     * <p>
     * The input modalities selected for the guardrail content filter configuration.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInputModalities} method.
     * </p>
     * 
     * @return The input modalities selected for the guardrail content filter configuration.
     */
    public final List<GuardrailModality> inputModalities() {
        return GuardrailModalitiesCopier.copyStringToEnum(inputModalities);
    }

    /**
     * For responses, this returns true if the service returned a value for the InputModalities property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasInputModalities() {
        return inputModalities != null && !(inputModalities instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The input modalities selected for the guardrail content filter configuration.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInputModalities} method.
     * </p>
     * 
     * @return The input modalities selected for the guardrail content filter configuration.
     */
    public final List<String> inputModalitiesAsStrings() {
        return inputModalities;
    }

    /**
     * <p>
     * The output modalities selected for the guardrail content filter configuration.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasOutputModalities} method.
     * </p>
     * 
     * @return The output modalities selected for the guardrail content filter configuration.
     */
    public final List<GuardrailModality> outputModalities() {
        return GuardrailModalitiesCopier.copyStringToEnum(outputModalities);
    }

    /**
     * For responses, this returns true if the service returned a value for the OutputModalities property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasOutputModalities() {
        return outputModalities != null && !(outputModalities instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The output modalities selected for the guardrail content filter configuration.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasOutputModalities} method.
     * </p>
     * 
     * @return The output modalities selected for the guardrail content filter configuration.
     */
    public final List<String> outputModalitiesAsStrings() {
        return outputModalities;
    }

    /**
     * <p>
     * Specifies the action to take when harmful content is detected. Supported values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>BLOCK</code> – Block the content and replace it with blocked messaging.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NONE</code> – Take no action but return detection information in the trace response.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #inputAction} will
     * return {@link GuardrailContentFilterAction#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #inputActionAsString}.
     * </p>
     * 
     * @return Specifies the action to take when harmful content is detected. Supported values include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>BLOCK</code> – Block the content and replace it with blocked messaging.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NONE</code> – Take no action but return detection information in the trace response.
     *         </p>
     *         </li>
     * @see GuardrailContentFilterAction
     */
    public final GuardrailContentFilterAction inputAction() {
        return GuardrailContentFilterAction.fromValue(inputAction);
    }

    /**
     * <p>
     * Specifies the action to take when harmful content is detected. Supported values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>BLOCK</code> – Block the content and replace it with blocked messaging.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NONE</code> – Take no action but return detection information in the trace response.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #inputAction} will
     * return {@link GuardrailContentFilterAction#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #inputActionAsString}.
     * </p>
     * 
     * @return Specifies the action to take when harmful content is detected. Supported values include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>BLOCK</code> – Block the content and replace it with blocked messaging.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NONE</code> – Take no action but return detection information in the trace response.
     *         </p>
     *         </li>
     * @see GuardrailContentFilterAction
     */
    public final String inputActionAsString() {
        return inputAction;
    }

    /**
     * <p>
     * Specifies the action to take when harmful content is detected in the output. Supported values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>BLOCK</code> – Block the content and replace it with blocked messaging.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NONE</code> – Take no action but return detection information in the trace response.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #outputAction} will
     * return {@link GuardrailContentFilterAction#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #outputActionAsString}.
     * </p>
     * 
     * @return Specifies the action to take when harmful content is detected in the output. Supported values
     *         include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>BLOCK</code> – Block the content and replace it with blocked messaging.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NONE</code> – Take no action but return detection information in the trace response.
     *         </p>
     *         </li>
     * @see GuardrailContentFilterAction
     */
    public final GuardrailContentFilterAction outputAction() {
        return GuardrailContentFilterAction.fromValue(outputAction);
    }

    /**
     * <p>
     * Specifies the action to take when harmful content is detected in the output. Supported values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>BLOCK</code> – Block the content and replace it with blocked messaging.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NONE</code> – Take no action but return detection information in the trace response.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #outputAction} will
     * return {@link GuardrailContentFilterAction#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #outputActionAsString}.
     * </p>
     * 
     * @return Specifies the action to take when harmful content is detected in the output. Supported values
     *         include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>BLOCK</code> – Block the content and replace it with blocked messaging.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NONE</code> – Take no action but return detection information in the trace response.
     *         </p>
     *         </li>
     * @see GuardrailContentFilterAction
     */
    public final String outputActionAsString() {
        return outputAction;
    }

    /**
     * <p>
     * Specifies whether to enable guardrail evaluation on the input. When disabled, you aren't charged for the
     * evaluation. The evaluation doesn't appear in the response.
     * </p>
     * 
     * @return Specifies whether to enable guardrail evaluation on the input. When disabled, you aren't charged for the
     *         evaluation. The evaluation doesn't appear in the response.
     */
    public final Boolean inputEnabled() {
        return inputEnabled;
    }

    /**
     * <p>
     * Specifies whether to enable guardrail evaluation on the output. When disabled, you aren't charged for the
     * evaluation. The evaluation doesn't appear in the response.
     * </p>
     * 
     * @return Specifies whether to enable guardrail evaluation on the output. When disabled, you aren't charged for the
     *         evaluation. The evaluation doesn't appear in the response.
     */
    public final Boolean outputEnabled() {
        return outputEnabled;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(inputStrengthAsString());
        hashCode = 31 * hashCode + Objects.hashCode(outputStrengthAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasInputModalities() ? inputModalitiesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasOutputModalities() ? outputModalitiesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(inputActionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(outputActionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(inputEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(outputEnabled());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GuardrailContentFilterConfig)) {
            return false;
        }
        GuardrailContentFilterConfig other = (GuardrailContentFilterConfig) obj;
        return Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(inputStrengthAsString(), other.inputStrengthAsString())
                && Objects.equals(outputStrengthAsString(), other.outputStrengthAsString())
                && hasInputModalities() == other.hasInputModalities()
                && Objects.equals(inputModalitiesAsStrings(), other.inputModalitiesAsStrings())
                && hasOutputModalities() == other.hasOutputModalities()
                && Objects.equals(outputModalitiesAsStrings(), other.outputModalitiesAsStrings())
                && Objects.equals(inputActionAsString(), other.inputActionAsString())
                && Objects.equals(outputActionAsString(), other.outputActionAsString())
                && Objects.equals(inputEnabled(), other.inputEnabled()) && Objects.equals(outputEnabled(), other.outputEnabled());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GuardrailContentFilterConfig").add("Type", typeAsString())
                .add("InputStrength", inputStrengthAsString()).add("OutputStrength", outputStrengthAsString())
                .add("InputModalities", inputModalitiesAsStrings() == null ? null : "*** Sensitive Data Redacted ***")
                .add("OutputModalities", outputModalitiesAsStrings() == null ? null : "*** Sensitive Data Redacted ***")
                .add("InputAction", inputActionAsString() == null ? null : "*** Sensitive Data Redacted ***")
                .add("OutputAction", outputActionAsString() == null ? null : "*** Sensitive Data Redacted ***")
                .add("InputEnabled", inputEnabled()).add("OutputEnabled", outputEnabled()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "inputStrength":
            return Optional.ofNullable(clazz.cast(inputStrengthAsString()));
        case "outputStrength":
            return Optional.ofNullable(clazz.cast(outputStrengthAsString()));
        case "inputModalities":
            return Optional.ofNullable(clazz.cast(inputModalitiesAsStrings()));
        case "outputModalities":
            return Optional.ofNullable(clazz.cast(outputModalitiesAsStrings()));
        case "inputAction":
            return Optional.ofNullable(clazz.cast(inputActionAsString()));
        case "outputAction":
            return Optional.ofNullable(clazz.cast(outputActionAsString()));
        case "inputEnabled":
            return Optional.ofNullable(clazz.cast(inputEnabled()));
        case "outputEnabled":
            return Optional.ofNullable(clazz.cast(outputEnabled()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("type", TYPE_FIELD);
        map.put("inputStrength", INPUT_STRENGTH_FIELD);
        map.put("outputStrength", OUTPUT_STRENGTH_FIELD);
        map.put("inputModalities", INPUT_MODALITIES_FIELD);
        map.put("outputModalities", OUTPUT_MODALITIES_FIELD);
        map.put("inputAction", INPUT_ACTION_FIELD);
        map.put("outputAction", OUTPUT_ACTION_FIELD);
        map.put("inputEnabled", INPUT_ENABLED_FIELD);
        map.put("outputEnabled", OUTPUT_ENABLED_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<GuardrailContentFilterConfig, T> g) {
        return obj -> g.apply((GuardrailContentFilterConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, GuardrailContentFilterConfig> {
        /**
         * <p>
         * The harmful category that the content filter is applied to.
         * </p>
         * 
         * @param type
         *        The harmful category that the content filter is applied to.
         * @see GuardrailContentFilterType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GuardrailContentFilterType
         */
        Builder type(String type);

        /**
         * <p>
         * The harmful category that the content filter is applied to.
         * </p>
         * 
         * @param type
         *        The harmful category that the content filter is applied to.
         * @see GuardrailContentFilterType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GuardrailContentFilterType
         */
        Builder type(GuardrailContentFilterType type);

        /**
         * <p>
         * The strength of the content filter to apply to prompts. As you increase the filter strength, the likelihood
         * of filtering harmful content increases and the probability of seeing harmful content in your application
         * reduces.
         * </p>
         * 
         * @param inputStrength
         *        The strength of the content filter to apply to prompts. As you increase the filter strength, the
         *        likelihood of filtering harmful content increases and the probability of seeing harmful content in
         *        your application reduces.
         * @see GuardrailFilterStrength
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GuardrailFilterStrength
         */
        Builder inputStrength(String inputStrength);

        /**
         * <p>
         * The strength of the content filter to apply to prompts. As you increase the filter strength, the likelihood
         * of filtering harmful content increases and the probability of seeing harmful content in your application
         * reduces.
         * </p>
         * 
         * @param inputStrength
         *        The strength of the content filter to apply to prompts. As you increase the filter strength, the
         *        likelihood of filtering harmful content increases and the probability of seeing harmful content in
         *        your application reduces.
         * @see GuardrailFilterStrength
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GuardrailFilterStrength
         */
        Builder inputStrength(GuardrailFilterStrength inputStrength);

        /**
         * <p>
         * The strength of the content filter to apply to model responses. As you increase the filter strength, the
         * likelihood of filtering harmful content increases and the probability of seeing harmful content in your
         * application reduces.
         * </p>
         * 
         * @param outputStrength
         *        The strength of the content filter to apply to model responses. As you increase the filter strength,
         *        the likelihood of filtering harmful content increases and the probability of seeing harmful content in
         *        your application reduces.
         * @see GuardrailFilterStrength
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GuardrailFilterStrength
         */
        Builder outputStrength(String outputStrength);

        /**
         * <p>
         * The strength of the content filter to apply to model responses. As you increase the filter strength, the
         * likelihood of filtering harmful content increases and the probability of seeing harmful content in your
         * application reduces.
         * </p>
         * 
         * @param outputStrength
         *        The strength of the content filter to apply to model responses. As you increase the filter strength,
         *        the likelihood of filtering harmful content increases and the probability of seeing harmful content in
         *        your application reduces.
         * @see GuardrailFilterStrength
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GuardrailFilterStrength
         */
        Builder outputStrength(GuardrailFilterStrength outputStrength);

        /**
         * <p>
         * The input modalities selected for the guardrail content filter configuration.
         * </p>
         * 
         * @param inputModalities
         *        The input modalities selected for the guardrail content filter configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputModalitiesWithStrings(Collection<String> inputModalities);

        /**
         * <p>
         * The input modalities selected for the guardrail content filter configuration.
         * </p>
         * 
         * @param inputModalities
         *        The input modalities selected for the guardrail content filter configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputModalitiesWithStrings(String... inputModalities);

        /**
         * <p>
         * The input modalities selected for the guardrail content filter configuration.
         * </p>
         * 
         * @param inputModalities
         *        The input modalities selected for the guardrail content filter configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputModalities(Collection<GuardrailModality> inputModalities);

        /**
         * <p>
         * The input modalities selected for the guardrail content filter configuration.
         * </p>
         * 
         * @param inputModalities
         *        The input modalities selected for the guardrail content filter configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputModalities(GuardrailModality... inputModalities);

        /**
         * <p>
         * The output modalities selected for the guardrail content filter configuration.
         * </p>
         * 
         * @param outputModalities
         *        The output modalities selected for the guardrail content filter configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputModalitiesWithStrings(Collection<String> outputModalities);

        /**
         * <p>
         * The output modalities selected for the guardrail content filter configuration.
         * </p>
         * 
         * @param outputModalities
         *        The output modalities selected for the guardrail content filter configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputModalitiesWithStrings(String... outputModalities);

        /**
         * <p>
         * The output modalities selected for the guardrail content filter configuration.
         * </p>
         * 
         * @param outputModalities
         *        The output modalities selected for the guardrail content filter configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputModalities(Collection<GuardrailModality> outputModalities);

        /**
         * <p>
         * The output modalities selected for the guardrail content filter configuration.
         * </p>
         * 
         * @param outputModalities
         *        The output modalities selected for the guardrail content filter configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputModalities(GuardrailModality... outputModalities);

        /**
         * <p>
         * Specifies the action to take when harmful content is detected. Supported values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>BLOCK</code> – Block the content and replace it with blocked messaging.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NONE</code> – Take no action but return detection information in the trace response.
         * </p>
         * </li>
         * </ul>
         * 
         * @param inputAction
         *        Specifies the action to take when harmful content is detected. Supported values include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>BLOCK</code> – Block the content and replace it with blocked messaging.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NONE</code> – Take no action but return detection information in the trace response.
         *        </p>
         *        </li>
         * @see GuardrailContentFilterAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GuardrailContentFilterAction
         */
        Builder inputAction(String inputAction);

        /**
         * <p>
         * Specifies the action to take when harmful content is detected. Supported values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>BLOCK</code> – Block the content and replace it with blocked messaging.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NONE</code> – Take no action but return detection information in the trace response.
         * </p>
         * </li>
         * </ul>
         * 
         * @param inputAction
         *        Specifies the action to take when harmful content is detected. Supported values include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>BLOCK</code> – Block the content and replace it with blocked messaging.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NONE</code> – Take no action but return detection information in the trace response.
         *        </p>
         *        </li>
         * @see GuardrailContentFilterAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GuardrailContentFilterAction
         */
        Builder inputAction(GuardrailContentFilterAction inputAction);

        /**
         * <p>
         * Specifies the action to take when harmful content is detected in the output. Supported values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>BLOCK</code> – Block the content and replace it with blocked messaging.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NONE</code> – Take no action but return detection information in the trace response.
         * </p>
         * </li>
         * </ul>
         * 
         * @param outputAction
         *        Specifies the action to take when harmful content is detected in the output. Supported values
         *        include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>BLOCK</code> – Block the content and replace it with blocked messaging.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NONE</code> – Take no action but return detection information in the trace response.
         *        </p>
         *        </li>
         * @see GuardrailContentFilterAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GuardrailContentFilterAction
         */
        Builder outputAction(String outputAction);

        /**
         * <p>
         * Specifies the action to take when harmful content is detected in the output. Supported values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>BLOCK</code> – Block the content and replace it with blocked messaging.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NONE</code> – Take no action but return detection information in the trace response.
         * </p>
         * </li>
         * </ul>
         * 
         * @param outputAction
         *        Specifies the action to take when harmful content is detected in the output. Supported values
         *        include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>BLOCK</code> – Block the content and replace it with blocked messaging.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NONE</code> – Take no action but return detection information in the trace response.
         *        </p>
         *        </li>
         * @see GuardrailContentFilterAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GuardrailContentFilterAction
         */
        Builder outputAction(GuardrailContentFilterAction outputAction);

        /**
         * <p>
         * Specifies whether to enable guardrail evaluation on the input. When disabled, you aren't charged for the
         * evaluation. The evaluation doesn't appear in the response.
         * </p>
         * 
         * @param inputEnabled
         *        Specifies whether to enable guardrail evaluation on the input. When disabled, you aren't charged for
         *        the evaluation. The evaluation doesn't appear in the response.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputEnabled(Boolean inputEnabled);

        /**
         * <p>
         * Specifies whether to enable guardrail evaluation on the output. When disabled, you aren't charged for the
         * evaluation. The evaluation doesn't appear in the response.
         * </p>
         * 
         * @param outputEnabled
         *        Specifies whether to enable guardrail evaluation on the output. When disabled, you aren't charged for
         *        the evaluation. The evaluation doesn't appear in the response.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputEnabled(Boolean outputEnabled);
    }

    static final class BuilderImpl implements Builder {
        private String type;

        private String inputStrength;

        private String outputStrength;

        private List<String> inputModalities = DefaultSdkAutoConstructList.getInstance();

        private List<String> outputModalities = DefaultSdkAutoConstructList.getInstance();

        private String inputAction;

        private String outputAction;

        private Boolean inputEnabled;

        private Boolean outputEnabled;

        private BuilderImpl() {
        }

        private BuilderImpl(GuardrailContentFilterConfig model) {
            type(model.type);
            inputStrength(model.inputStrength);
            outputStrength(model.outputStrength);
            inputModalitiesWithStrings(model.inputModalities);
            outputModalitiesWithStrings(model.outputModalities);
            inputAction(model.inputAction);
            outputAction(model.outputAction);
            inputEnabled(model.inputEnabled);
            outputEnabled(model.outputEnabled);
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(GuardrailContentFilterType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getInputStrength() {
            return inputStrength;
        }

        public final void setInputStrength(String inputStrength) {
            this.inputStrength = inputStrength;
        }

        @Override
        public final Builder inputStrength(String inputStrength) {
            this.inputStrength = inputStrength;
            return this;
        }

        @Override
        public final Builder inputStrength(GuardrailFilterStrength inputStrength) {
            this.inputStrength(inputStrength == null ? null : inputStrength.toString());
            return this;
        }

        public final String getOutputStrength() {
            return outputStrength;
        }

        public final void setOutputStrength(String outputStrength) {
            this.outputStrength = outputStrength;
        }

        @Override
        public final Builder outputStrength(String outputStrength) {
            this.outputStrength = outputStrength;
            return this;
        }

        @Override
        public final Builder outputStrength(GuardrailFilterStrength outputStrength) {
            this.outputStrength(outputStrength == null ? null : outputStrength.toString());
            return this;
        }

        public final Collection<String> getInputModalities() {
            if (inputModalities instanceof SdkAutoConstructList) {
                return null;
            }
            return inputModalities;
        }

        public final void setInputModalities(Collection<String> inputModalities) {
            this.inputModalities = GuardrailModalitiesCopier.copy(inputModalities);
        }

        @Override
        public final Builder inputModalitiesWithStrings(Collection<String> inputModalities) {
            this.inputModalities = GuardrailModalitiesCopier.copy(inputModalities);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder inputModalitiesWithStrings(String... inputModalities) {
            inputModalitiesWithStrings(Arrays.asList(inputModalities));
            return this;
        }

        @Override
        public final Builder inputModalities(Collection<GuardrailModality> inputModalities) {
            this.inputModalities = GuardrailModalitiesCopier.copyEnumToString(inputModalities);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder inputModalities(GuardrailModality... inputModalities) {
            inputModalities(Arrays.asList(inputModalities));
            return this;
        }

        public final Collection<String> getOutputModalities() {
            if (outputModalities instanceof SdkAutoConstructList) {
                return null;
            }
            return outputModalities;
        }

        public final void setOutputModalities(Collection<String> outputModalities) {
            this.outputModalities = GuardrailModalitiesCopier.copy(outputModalities);
        }

        @Override
        public final Builder outputModalitiesWithStrings(Collection<String> outputModalities) {
            this.outputModalities = GuardrailModalitiesCopier.copy(outputModalities);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder outputModalitiesWithStrings(String... outputModalities) {
            outputModalitiesWithStrings(Arrays.asList(outputModalities));
            return this;
        }

        @Override
        public final Builder outputModalities(Collection<GuardrailModality> outputModalities) {
            this.outputModalities = GuardrailModalitiesCopier.copyEnumToString(outputModalities);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder outputModalities(GuardrailModality... outputModalities) {
            outputModalities(Arrays.asList(outputModalities));
            return this;
        }

        public final String getInputAction() {
            return inputAction;
        }

        public final void setInputAction(String inputAction) {
            this.inputAction = inputAction;
        }

        @Override
        public final Builder inputAction(String inputAction) {
            this.inputAction = inputAction;
            return this;
        }

        @Override
        public final Builder inputAction(GuardrailContentFilterAction inputAction) {
            this.inputAction(inputAction == null ? null : inputAction.toString());
            return this;
        }

        public final String getOutputAction() {
            return outputAction;
        }

        public final void setOutputAction(String outputAction) {
            this.outputAction = outputAction;
        }

        @Override
        public final Builder outputAction(String outputAction) {
            this.outputAction = outputAction;
            return this;
        }

        @Override
        public final Builder outputAction(GuardrailContentFilterAction outputAction) {
            this.outputAction(outputAction == null ? null : outputAction.toString());
            return this;
        }

        public final Boolean getInputEnabled() {
            return inputEnabled;
        }

        public final void setInputEnabled(Boolean inputEnabled) {
            this.inputEnabled = inputEnabled;
        }

        @Override
        public final Builder inputEnabled(Boolean inputEnabled) {
            this.inputEnabled = inputEnabled;
            return this;
        }

        public final Boolean getOutputEnabled() {
            return outputEnabled;
        }

        public final void setOutputEnabled(Boolean outputEnabled) {
            this.outputEnabled = outputEnabled;
        }

        @Override
        public final Builder outputEnabled(Boolean outputEnabled) {
            this.outputEnabled = outputEnabled;
            return this;
        }

        @Override
        public GuardrailContentFilterConfig build() {
            return new GuardrailContentFilterConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
